{$J+,Z4}
unit KeyPropTypes;
{$IFDEF CONDITIONALEXPRESSIONS} {$DEFINE FIXED} {$ENDIF} // Delphi 6 and later

{------------------------------------------------------------------------------}
{                                                                              }
{                                 This code is                                 }
{              Copyright (C) 2001-2006 by Michael in der Wiesche               }
{                                                                              }
{------------------------------------------------------------------------------}

interface

uses
  Windows,
  Classes,
  SysUtils,
  StrTools;

const
  ShortHexIDLen				= 10;
  LongHexIDLen				= 18;
  DefHexIDLen				= LongHexIDLen;

{ My tribute to Steve Heller's great work with creating SPGP & DPGP }
const
  spgpKeyPropFlag_None			= $00000000;
  spgpKeyPropFlag_All			= $FFFFFFFF;

  // "string" properties
  spgpKeyPropFlag_KeyID			= $00000001;
  spgpKeyPropFlag_UserID		= $00000002;
  spgpKeyPropFlag_Fingerprint		= $00000004;
  spgpKeyPropFlag_CreationTimeStr	= $00000008;
  spgpKeyPropFlag_ExpirationTimeStr	= $00000010;

  // "numeric" properties
  spgpKeyPropFlag_KeyBits		= $00000040;
  spgpKeyPropFlag_X509Cert		= $00000080;
  spgpKeyPropFlag_KeyAlg		= $00000100;
  spgpKeyPropFlag_HashAlg		= $00000200;
  spgpKeyPropFlag_Trust			= $00000400;
  spgpKeyPropFlag_Validity		= $00000800;
  spgpKeyPropFlag_CreationTime		= $00001000;
  spgpKeyPropFlag_ExpirationTime	= $00002000;

  // "boolean" properties
  spgpKeyPropFlag_IsCorrupt		= $00004000;
  spgpKeyPropFlag_IsSecret		= $00008000;
  spgpKeyPropFlag_IsAxiomatic		= $00010000;
  spgpKeyPropFlag_IsRevoked		= $00020000;
  spgpKeyPropFlag_IsDisabled		= $00040000;
  spgpKeyPropFlag_IsExpired		= $00080000;
  spgpKeyPropFlag_IsSecretShared	= $00100000;

  spgpKeyPropFlag_CanEncrypt		= $00200000;
  spgpKeyPropFlag_CanDecrypt		= $00400000;
  spgpKeyPropFlag_CanSign		= $00800000;
  spgpKeyPropFlag_CanVerify		= $01000000;

  spgpKeyPropFlag_HasRevoker		= $02000000;
  spgpKeyPropFlag_HasADK		= $04000000;
  spgpKeyPropFlag_HasSubKey		= $08000000;
  spgpKeyPropFlag_LegacyKey		= $10000000;

  // "list" flags
  spgpKeyPropFlag_IncludeUserIDs	= $20000000;
  spgpKeyPropFlag_IncludeSignerIDs	= $40000000;
  spgpKeyPropFlag_IncludeGroupsList	= $80000000;

  // "convenience" flags
  spgpKeyPropFlag_IDFlags		= spgpKeyPropFlag_KeyID or spgpKeyPropFlag_UserID;
  spgpKeyPropFlag_IDComplete		= spgpKeyPropFlag_IDFlags or spgpKeyPropFlag_KeyBits or spgpKeyPropFlag_KeyAlg;

const
  // include all keyring keys
  KeyFilterFlag_AllKeys			= $00000;

  // capabilities
  KeyFilterFlag_CanEncrypt		= $00001;
  KeyFilterFlag_CanDecrypt		= $00002;
  KeyFilterFlag_CanSign			= $00004;
  KeyFilterFlag_CanVerify		= $00008;

  // key status
  KeyFilterFlag_Enabled			= $00010;
  KeyFilterFlag_Disabled		= $00020;

  // algorithms
  KeyFilterFlag_DHDSS			= $00100;
  KeyFilterFlag_RSA			= $00200;

  // versions
  KeyFilterFlag_V4			= $00400;
  KeyFilterFlag_V3			= $00800;

  // X509
  KeyFilterFlag_X509Cert		= $01000;
  KeyFilterFlag_X509Root		= $02000;

  // the groups can be combined
  KeyFilterMask_Boolean			= $000FF;
  KeyFilterMask_Algorithm		= $0FF00;

  // count user IDs of all keys
  KeyFilterFlag_CountUserIDs		= $10000;

const
  // pre-selection flag for key import
  IgnoreFlag_None			= 0;
  IgnoreFlag_ByHexID			= 1;
  IgnoreFlag_ByUserID			= 2;

type
  TKeyProp = (
    KeyProp_HexID,
    KeyProp_UserID,
    KeyProp_Fingerprint,
    KeyProp_CreaTimeStr,
    KeyProp_ExpTimeStr,
    KeyProp_Size,
    KeyProp_X509Cert,
    KeyProp_KeyAlgorithm,
    KeyProp_HashAlgorithm,
    KeyProp_Trust,
    KeyProp_Validity,
    KeyProp_CreaTimeNum,
    KeyProp_ExpTimeNum,
    KeyProp_Corrupt,
    KeyProp_Secret,
    KeyProp_ImplicitTrust,
    KeyProp_Revoked,
    KeyProp_Disabled,
    KeyProp_Expired,
    KeyProp_SecretShared,
    KeyProp_CanEncrypt,
    KeyProp_CanDecrypt,
    KeyProp_CanSign,
    KeyProp_CanVerify,
    KeyProp_HasRevoker,
    KeyProp_HasADK,
    KeyProp_HasSubKey,
    KeyProp_LegacyKey,
    KeyProp_IncludeUserIDs,
    KeyProp_IncludeSignerIDs,
    KeyProp_IncludeGroupsList
  );
  TKeyProps = Set of TKeyProp;
  TPGPKeyOrdering = (
    Invalid_Ordering,
    Any_Ordering,
    UserID_Ordering,
    ReverseUserID_Ordering,
    KeyID_Ordering,
    ReverseKeyID_Ordering,
    Validity_Ordering,
    ReverseValidity_Ordering,
    Trust_Ordering,
    ReverseTrust_Ordering,
    EncryptKeySize_Ordering,
    ReverseEncryptKeySize_Ordering,
    SigKeySize_Ordering,
    ReverseSigKeySize_Ordering,
    Creation_Ordering,
    ReverseCreation_Ordering,
    Expiration_Ordering,
    ReverseExpiration_Ordering
  );
  TTrustLevel = (
    KeyTrust_Undefined,
    KeyTrust_Unknown,
    KeyTrust_Never,
    KeyTrust_Reserved1,
    KeyTrust_Reserved2,
    KeyTrust_Marginal,
    KeyTrust_Complete,
    KeyTrust_Ultimate
  );
  TValidityLevel = (
    Validity_Unknown,
    Validity_Invalid,
    Validity_Marginal,
    Validity_Complete
  );
  TKeyAlgorithm = (
    KeyAlgorithm_Invalid,
    KeyAlgorithm_RSA,
    KeyAlgorithm_RSAEncryptOnly,
    KeyAlgorithm_RSASignOnly,
    KeyAlgorithm_Reserved01,
    KeyAlgorithm_Reserved02,
    KeyAlgorithm_Reserved03,
    KeyAlgorithm_Reserved04,
    KeyAlgorithm_Reserved05,
    KeyAlgorithm_Reserved06,
    KeyAlgorithm_Reserved07,
    KeyAlgorithm_Reserved08,
    KeyAlgorithm_Reserved09,
    KeyAlgorithm_Reserved10,
    KeyAlgorithm_Reserved11,
    KeyAlgorithm_Reserved12,
    KeyAlgorithm_DH,
    KeyAlgorithm_DSS,
    KeyAlgorithm_DHDSS
  );
  TCipherAlgorithm = (
    CipherAlgorithm_None,
    CipherAlgorithm_IDEA,
    CipherAlgorithm_3DES,
    CipherAlgorithm_CAST5,
    CipherAlgorithm_AES128,
    CipherAlgorithm_AES192,
    CipherAlgorithm_AES256,
    CipherAlgorithm_Twofish256
  );
  THashAlgorithm = (
    HashAlgorithm_Invalid,
    HashAlgorithm_MD2,
    HashAlgorithm_MD5,
    HashAlgorithm_SHA,
    HashAlgorithm_RIPEMD160,
    HashAlgorithm_SHA256,
    HashAlgorithm_SHA384,
    HashAlgorithm_SHA512
  );
  TADKType = (
    NoADK,
    SimpleADK,
    EnforcedADK
  );
  TX509Type = (
    NoCert,
    BadCert,
    StdCert,
    RootCert
  );
  TX509ChainValidity = (
    CertChain_Invalid,
    CertChain_Valid,
    CertChain_Root
  );

  TUserIDs = class(TStringList)
  private
    function  GetChainValidity(Index: Longint): TX509ChainValidity;
    procedure SetChainValidity(Index: Longint; Value: TX509ChainValidity);
    function  GetKeyValidity(Index: Longint): TValidityLevel;
    procedure SetKeyValidity(Index: Longint; Value: TValidityLevel);
    function  GetX509Type(Index: Longint): TX509Type;
    procedure SetX509Type(Index: Longint; Value: TX509Type);
  protected
    function  Get(Index: Integer): String; override;
    function  GetRaw(Index: Longint): String;
  public
    property  CertOwners[Index: Longint]: String read GetRaw;
    property  ChainValidities[Index: Longint]: TX509ChainValidity read GetChainValidity write SetChainValidity;
    property  KeyValidities[Index: Longint]: TValidityLevel read GetKeyValidity write SetKeyValidity;
    property  X509Types[Index: Longint]: TX509Type read GetX509Type write SetX509Type;
  end;

  TSignerIDs = TStringList;
  TADKeyIDs = TStringList;
  TRevKeyIDs = TStringList;
  TGroupsList = TStringList;

  pKeyPropsRec = ^TKeyPropsRec;
  TKeyPropsRec = Record
    kHexID: String;
    kUserID: String;
    kFingerprint: String;
    kCreaTimeStr: String;
    kExpTimeStr: String;
    kSize: String;
    kX509Cert: TX509Type;
    kKeyAlgorithm: TKeyAlgorithm;
    kHashAlgorithm: THashAlgorithm;
    kTrust: TTrustLevel;
    kValidity: TValidityLevel;
    kCreaTimeNum: Longint;
    kExpTimeNum: Longint;
    kCorrupt: Longbool;
    kPrivate: Longbool;
    kImplicitTrust: Longbool;
    kRevoked: Longbool;
    kDisabled: Longbool;
    kExpired: Longbool;
    kSecShared: Longbool;
    kCanEncrypt: Longbool;
    kCanDecrypt: Longbool;
    kCanSign: Longbool;
    kCanVerify: Longbool;
    kHasRevoker: Longbool;
    kHasADK: TADKType;
    kHasSubKey: Longbool;
    kLegacyKey: Longbool;
    kUserIDList: TUserIDs;
    kSignerIDList: TSignerIDs;
    kADKeyIDList: TADKeyIDs;
    kRevKeyIDList: TRevKeyIDs;
  end;

  pKeyPropsList = ^TKeyPropsList;
  TKeyPropsList = class(TStringList)
  private
    FDuplicates: Longbool;
    FValidProps: TKeyProps;
    IncludeUserIDs: Longbool;
    IncludeSignerIDs: Longbool;
    IncludeADKeyIDs: Longbool;
    IncludeRevKeyIDs: Longbool;
    function	AllocKeyPropsRec: pKeyPropsRec;
    function	GetKeyProps(Index: Longint): TKeyPropsRec;
    function	FreeKeyPropsRec(Index: Longint): pKeyPropsRec;
    function	GetKeyPropsSet(KeyPropsFlag: DWord): TKeyProps;
  public
    GroupsList:	TGroupsList;
    constructor	Create(PropertyFlags: DWord);
    destructor	Destroy; override;
    procedure	Clear; override;
    procedure 	Changed; override;
    procedure 	Changing; override;
    procedure	Delete(Index: Integer); override;
    procedure	Move(CurIndex, NewIndex: Integer); override;
    function	Add(const S: String): Integer; override;
    function	AddObject(const S: String; AObject: TObject): Integer; override;
    procedure	Insert(Index: Integer; const S: String); override;
    // returns true if requested record item exists
    function	GetKeyPropsRec(var KeyPropsRec: TKeyPropsRec; Index: Integer): Longbool;
    // access indexed KeyProps record like Strings[] or Objects[]
    property	KeyProps[Index: Longint]: TKeyPropsRec read GetKeyProps;
    // indicates accessable key properties for the respective list
    property	ValidProps: TKeyProps read FValidProps;
    // just a dummy to prevent using the inherited property
    property	Duplicates: Longbool read FDuplicates;
  end;

function ExtractEmail(const sText: String): String;
function ExtractName(const sText: String): String;
// extracts standard UserID from the same values as below
function GetUserIDFromLongName(const LongName: String): String;
// extracts CommonName and Email or - if not available - OrganizationName and OrganizationalUnitName
function GetUserFromLongName(const LongName: String; IncludeEmail: Longbool): String;

implementation

// Utilities -------------------------------------------------------------------

function ExtractEmail(const sText: String): String;
var
  iATPos, iBegPos, iEndPos, iLen: Integer;
begin
  Result := E;
  iATPos := FirstPos(AT, sText);
  if iATPos > 0 then begin
    iBegPos := LastShortPos(LT, sText, pred(iATPos));
    if iBegPos > 0 then begin
      iEndPos := FirstShortPos(GT, sText, succ(iATPos));
      if iEndPos > 0 then begin
	repeat
	  inc(iBegPos)
	until sText[iBegPos] > SP;
	repeat
	  dec(iEndPos)
	until sText[iEndPos] > SP;
	Result := Copy(sText, iBegPos, succ(iEndPos - iBegPos));
      end;
    end;
    if Result = E then begin
      iLen := Length(sText);
      iBegPos := pred(iATPos);
      while (iBegPos > 0) and (sText[iBegPos] > SP) do dec(iBegPos);
      iEndPos := succ(iATPos);
      while (iEndPos <= iLen) and (sText[iEndPos] >= '-') do inc(iEndPos);
      Result := Copy(sText, succ(iBegPos), pred(iEndPos - iBegPos));
    end;
  end;
end;

function ExtractName(const sText: String): String;
var
  iLen, iBegPos, iEndPos, iQUBeg, iQUEnd, iBSPos: Integer;
begin
  Result := E;
  iLen := Length(sText);
  iEndPos := FirstPos(LT, sText);
  if iEndPos = 0 then begin
    iEndPos := FirstPos(AT, sText);
    while (iEndPos > 0) and (sText[iEndPos] > SP) do dec(iEndPos);
  end
  else dec(iEndPos);
  if iEndPos = 0 then begin
    iBegPos := LastPos(LB, sText);
    while (iBegPos > 1) and (sText[pred(iBegPos)] = BS) do iBegPos := LastShortPos(LB, sText, iBegPos - 2);
    if iBegPos > 0 then begin
      iEndPos := LastPos(RB, sText);
      while (iEndPos > iBegPos) and (sText[pred(iEndPos)] = BS) do iEndPos := LastShortPos(RB, sText, iEndPos - 2);
      inc(iBegPos);
      dec(iEndPos);
    end
    else begin
      iBegPos := LastPos(GT, sText);
      if iBegPos = 0 then begin
	iBegPos := FirstPos(AT, sText);
	while (iBegPos <= iLen) and (sText[iBegPos] > SP) do inc(iBegPos);
      end
      else inc(iBegPos);
      if iBegPos > 0 then iEndPos := iLen;
    end;
  end
  else iBegPos := 1;
  if (iBegPos = 0) and (iEndPos = 0) then begin
    iBegPos := 1;
    iEndPos := iLen;
  end;
  iQUBeg := FirstPos(QU, sText);
  if iQUBeg > 0 then begin
    iQUEnd := FirstShortPos(QU, sText, succ(iQUBeg));
    if iQUEnd > 1 then begin
      while (iQUEnd <= iLen) and (sText[pred(iQUEnd)] = BS) do iQUEnd := FirstShortPos(QU, sText, succ(iQUEnd));
      iBegPos := succ(iQUBeg);
      iEndPos := pred(iQUEnd);
    end;
  end;
  while (iBegPos <= iEndPos) and (sText[iBegPos] <= SP) do inc(iBegPos);
  while (iEndPos > 0) and (sText[iEndPos] <= SP) do dec(iEndPos);
  iLen := succ(iEndPos - iBegPos);
  if iLen > 0 then begin
    Result := Copy(sText, iBegPos, iLen);
    iBSPos := succ(iLen);
    repeat
      iBSPos := LastShortPos(BS, Result, pred(iBSPos));
      if iBSPos > 0 then Delete(Result, iBSPos, 1);
    until iBSPos = 0;
  end;
end;

function ExtractValues(const CertIDs, LongName: String; IncludeEmail, UserIDFormat: Longbool): String;
var
  IDBeg, ValBeg, IDsLen, ValLen, IDEnd, ValEnd: Integer;
  ID: String;
begin
  Result := E;
  IDBeg := 1;
  ValBeg := 1;
  IDsLen := Length(CertIDs);
  ValLen := Length(LongName);
  repeat
    IDEnd := FirstShortPos(CM, CertIDs, IDBeg);
    if IDEnd = 0 then IDEnd := succ(IDsLen);
    ID := Copy(CertIDs, IDBeg, IDEnd - IDBeg);
    ValBeg := ShortStrPos(ID, LongName, ValBeg);
    if ValBeg > 0 then begin
      if UserIDFormat then inc(ValBeg, Length(ID));
      ValEnd := FirstShortPos(CM, LongName, ValBeg + Length(ID));
      if ValEnd = 0 then ValEnd := succ(ValLen);
      if ValEnd > ValBeg then begin
	if UserIDFormat then begin
	  if Result = E then
	    Result := Copy(LongName, ValBeg, ValEnd - ValBeg)
	  else Result := Result + SP + LT + Copy(LongName, ValBeg, ValEnd - ValBeg) + GT;
	end
	else begin
	  if Result = E then
	    Result := Copy(LongName, ValBeg, ValEnd - ValBeg)
	  else Result := Result + CM + SP + Copy(LongName, ValBeg, ValEnd - ValBeg);
	end;
      end;
      ValBeg := succ(ValEnd);
    end;
    if IncludeEmail then
      IDBeg := succ(IDEnd)
    else Break;
  until IDBeg > IDsLen;
end;

function GetUserIDFromLongName(const LongName: String): String;
begin
  Result := ExtractValues('CN=,EMAIL=', LongName, true, true);
  if Result = E then Result := ExtractValues('O=,OU=', LongName, true, true);
end;

function GetUserFromLongName(const LongName: String; IncludeEmail: Longbool): String;
begin
  Result := ExtractValues('CN=,EMAIL=', LongName, IncludeEmail, false);
  if Result = E then Result := ExtractValues('O=,OU=', LongName, IncludeEmail, false);
end;

// TUserIDs --------------------------------------------------------------------

function TUserIDs.Get(Index: Integer): String;
begin
  if X509Types[Index] > BadCert then
    Result := GetUserIDFromLongName(inherited Get(Index))
  else Result := inherited Get(Index);
end;

function TUserIDs.GetRaw(Index: Longint): String;
begin
  Result := inherited Get(Index);
end;

function TUserIDs.GetChainValidity(Index: Longint): TX509ChainValidity;
begin
  Result := TX509ChainValidity(DWord(Objects[Index]) and $000000FF);
end;

procedure TUserIDs.SetChainValidity(Index: Longint; Value: TX509ChainValidity);
begin
  Objects[Index] := TObject((DWord(Objects[Index]) and $FFFFFF00) or DWord(Value));
end;

function TUserIDs.GetKeyValidity(Index: Longint): TValidityLevel;
begin
  Result := TValidityLevel((DWord(Objects[Index]) and $0000FF00) shr 8);
end;

procedure TUserIDs.SetKeyValidity(Index: Longint; Value: TValidityLevel);
begin
  Objects[Index] := TObject((DWord(Objects[Index]) and $FFFF00FF) or (DWord(Value) shl 8));
end;

function TUserIDs.GetX509Type(Index: Longint): TX509Type;
begin
  Result := TX509Type((DWord(Objects[Index]) and $00FF0000) shr 16);
end;

procedure TUserIDs.SetX509Type(Index: Longint; Value: TX509Type);
begin
  Objects[Index] := TObject((DWord(Objects[Index]) and $FF00FFFF) or (DWord(Value) shl 16));
end;

// TKeyPropsList ---------------------------------------------------------------

function TKeyPropsList.AllocKeyPropsRec: pKeyPropsRec;
begin
  New(Result);
  if Result <> nil then begin
    FillChar(Result^, SizeOf(TKeyPropsRec), 0);
    with Result^ do begin
      if IncludeUserIDs then begin
	kUserIDList := TUserIDs.Create;
	kUserIDList.BeginUpdate;
      end;
      if IncludeSignerIDs then begin
	kSignerIDList := TSignerIDs.Create;
	kSignerIDList.BeginUpdate;
      end;
      if IncludeADKeyIDs then begin
	kADKeyIDList := TADKeyIDs.Create;
	kADKeyIDList.BeginUpdate;
      end;
      if IncludeRevKeyIDs then begin
	kRevKeyIDList := TRevKeyIDs.Create;
	kRevKeyIDList.BeginUpdate;
      end;
    end;
  end;
end;

function TKeyPropsList.GetKeyProps(Index: Longint): TKeyPropsRec;
begin
  Result := pKeyPropsRec(Objects[Index])^;
end;

function TKeyPropsList.FreeKeyPropsRec(Index: Longint): pKeyPropsRec;
begin
  Result := pKeyPropsRec(Objects[Index]);
  if Result <> nil then begin
    Result^.kUserIDList.Free;
    Result^.kSignerIDList.Free;
    Result^.kADKeyIDList.Free;
    Result^.kRevKeyIDList.Free;
    Dispose(Result);
    Result := nil;
  end;
end;

function TKeyPropsList.GetKeyPropsSet(KeyPropsFlag: DWord): TKeyProps;
begin
  Result := [];
  if KeyPropsFlag and spgpKeyPropFlag_KeyID <> 0 then Include(Result, KeyProp_HexID);
  if KeyPropsFlag and spgpKeyPropFlag_UserID <> 0 then  Include(Result, KeyProp_UserID);
  if KeyPropsFlag and spgpKeyPropFlag_Fingerprint <> 0 then Include(Result, KeyProp_Fingerprint);
  if KeyPropsFlag and spgpKeyPropFlag_CreationTimeStr <> 0 then Include(Result, KeyProp_CreaTimeStr);
  if KeyPropsFlag and spgpKeyPropFlag_ExpirationTimeStr <> 0 then Include(Result, KeyProp_ExpTimeStr);
  if KeyPropsFlag and spgpKeyPropFlag_KeyBits <> 0 then Include(Result, KeyProp_Size);
  if KeyPropsFlag and spgpKeyPropFlag_X509Cert<> 0 then Include(Result, KeyProp_X509Cert);
  if KeyPropsFlag and spgpKeyPropFlag_KeyAlg <> 0 then Include(Result, KeyProp_KeyAlgorithm);
  if KeyPropsFlag and spgpKeyPropFlag_HashAlg <> 0 then Include(Result, KeyProp_HashAlgorithm);
  if KeyPropsFlag and spgpKeyPropFlag_Trust <> 0 then Include(Result, KeyProp_Trust);
  if KeyPropsFlag and spgpKeyPropFlag_Validity <> 0 then Include(Result, KeyProp_Validity);
  if KeyPropsFlag and spgpKeyPropFlag_CreationTime <> 0 then Include(Result, KeyProp_CreaTimeNum);
  if KeyPropsFlag and spgpKeyPropFlag_ExpirationTime <> 0 then Include(Result, KeyProp_ExpTimeNum);
  if KeyPropsFlag and spgpKeyPropFlag_IsCorrupt <> 0 then Include(Result, KeyProp_Corrupt);
  if KeyPropsFlag and spgpKeyPropFlag_IsSecret <> 0 then Include(Result, KeyProp_Secret);
  if KeyPropsFlag and spgpKeyPropFlag_IsAxiomatic <> 0 then Include(Result, KeyProp_ImplicitTrust);
  if KeyPropsFlag and spgpKeyPropFlag_IsRevoked <> 0 then Include(Result, KeyProp_Revoked);
  if KeyPropsFlag and spgpKeyPropFlag_IsDisabled <> 0 then Include(Result, KeyProp_Disabled);
  if KeyPropsFlag and spgpKeyPropFlag_IsExpired <> 0 then Include(Result, KeyProp_Expired);
  if KeyPropsFlag and spgpKeyPropFlag_IsSecretShared <> 0 then Include(Result, KeyProp_SecretShared);
  if KeyPropsFlag and spgpKeyPropFlag_CanEncrypt <> 0 then Include(Result, KeyProp_CanEncrypt);
  if KeyPropsFlag and spgpKeyPropFlag_CanDecrypt <> 0 then Include(Result, KeyProp_CanDecrypt);
  if KeyPropsFlag and spgpKeyPropFlag_CanSign <> 0 then Include(Result, KeyProp_CanSign);
  if KeyPropsFlag and spgpKeyPropFlag_CanVerify <> 0 then Include(Result, KeyProp_CanVerify);
  if KeyPropsFlag and spgpKeyPropFlag_HasRevoker <> 0 then Include(Result, KeyProp_HasRevoker);
  if KeyPropsFlag and spgpKeyPropFlag_HasADK <> 0 then Include(Result, KeyProp_HasADK);
  if KeyPropsFlag and spgpKeyPropFlag_HasSubKey <> 0 then Include(Result, KeyProp_HasSubKey);
  if KeyPropsFlag and spgpKeyPropFlag_LegacyKey <> 0 then Include(Result, KeyProp_LegacyKey);
  if KeyPropsFlag and spgpKeyPropFlag_IncludeUserIDs <> 0 then Include(Result, KeyProp_IncludeUserIDs );
  if KeyPropsFlag and spgpKeyPropFlag_IncludeSignerIDs <> 0 then Include(Result, KeyProp_IncludeSignerIDs);
  if KeyPropsFlag and spgpKeyPropFlag_IncludeGroupsList <> 0 then Include(Result, KeyProp_IncludeGroupsList);
end;

constructor TKeyPropsList.Create(PropertyFlags: DWord);
begin
  inherited Create;
  BeginUpdate;
  GroupsList := TGroupsList.Create;
  GroupsList.BeginUpdate;
  IncludeUserIDs := ((PropertyFlags and spgpKeyPropFlag_IncludeUserIDs) <> 0);
  IncludeSignerIDs := ((PropertyFlags and spgpKeyPropFlag_IncludeSignerIDs) <> 0);
  IncludeADKeyIDs := ((PropertyFlags and spgpKeyPropFlag_HasADK) <> 0);
  IncludeRevKeyIDs := ((PropertyFlags and spgpKeyPropFlag_HasRevoker) <> 0);
  if IncludeSignerIDs then IncludeUserIDs := true;
  FValidProps := GetKeyPropsSet(PropertyFlags);
end;

destructor TKeyPropsList.Destroy;
var Index: Integer;
begin
  try
    for Index := 0 to pred(Count) do FreeKeyPropsRec(Index);
  finally
    try
      GroupsList.Free;
    finally
      inherited Destroy;
    end;
  end;
end;

procedure TKeyPropsList.Clear;
var Index: Integer;
begin
  try
    for Index := 0 to pred(Count) do FreeKeyPropsRec(Index);
  finally
    try
      GroupsList.Clear;
    finally
      inherited Clear;
    end;
  end;
end;

procedure TKeyPropsList.Changed;
begin
  // save clock cycles
end;

procedure TKeyPropsList.Changing;
begin
  // save clock cycles
end;

procedure TKeyPropsList.Delete(Index: Integer);
begin
  FreeKeyPropsRec(Index);
  inherited Delete(Index);
end;

procedure TKeyPropsList.Move(CurIndex, NewIndex: Integer);
var TempStr: String; TempObj: TObject;
begin
  if CurIndex <> NewIndex then begin
    TempStr := Strings[CurIndex];
    TempObj := Objects[CurIndex];
    inherited Delete(CurIndex);
    inherited Insert(NewIndex, TempStr);
    Objects[NewIndex] := TempObj;
  end;
end;

function TKeyPropsList.Add(const S: String): Integer;
begin
  {$IFDEF FIXED}
  Result := inherited AddObject(S, TObject(AllocKeyPropsRec));
  {$ELSE}
  if not (Sorted and (inherited Duplicates = dupIgnore) and Find(S, Result)) then begin
    Result := inherited Add(S);
    Objects[Result] := TObject(AllocKeyPropsRec);
  end;
  {$ENDIF}
end;

function TKeyPropsList.AddObject(const S: String; AObject: TObject): Integer;
begin
  Result := -1;
end;

procedure TKeyPropsList.Insert(Index: Integer; const S: String);
begin
  inherited Insert(Index, S);
  Objects[Index] := TObject(AllocKeyPropsRec);
end;

function TKeyPropsList.GetKeyPropsRec(var KeyPropsRec: TKeyPropsRec; Index: Integer): Longbool;
begin
  Result := false;
  if (Index < Count) and (Objects[Index] <> nil) then begin
    KeyPropsRec := pKeyPropsRec(Objects[Index])^;
    Result := true;
  end;
end;

end.

