unit ASN1;

interface

uses
  Windows,
  SysUtils,
  KeyPropTypes;

function GetSerialNumber(const ASNData: PChar; Size: DWord): String;
function GetCertificateID(const ASNData: PChar; Size: DWord): String;
function GetAlgorithms(const ASNData: PChar; Size: DWord; var KeyAlg: TKeyAlgorithm; var HashAlg: THashAlgorithm): Integer;

implementation

function OIDCheck(const OID, ASNData: PChar; Pos: DWord): Longbool; register; assembler;
asm	// EAX = @OID, EDX = @ASNData, ECX = Pos
  OR	EAX,EAX
  JE	@ERROR
  OR	EDX,EDX
  JE	@ERROR
  CMP	BYTE PTR [EAX],0
  JE	@ERROR
  ADD	EDX,ECX
  CMP	BYTE PTR [EDX],0
  JE	@ERROR
  PUSH	EBX
  DEC	EAX
  DEC	EDX
  XOR	EBX,EBX
  @LOOP:
  INC	EBX
  JO	@ERROR
  MOV	CL,[EAX + EBX]
  OR	CL,CL
  JE	@END
  CMP	[EDX + EBX],CL
  JE	@LOOP
  POP	EBX
  @ERROR:
  XOR	EAX,EAX
  RET
  @END:
  POP	EBX
  MOV	EAX,TRUE
end;

function GetTypeLength(TaggedData: PChar; Size: DWord; var Pos: DWord): DWord;
var
  SizeLen, Index: DWord;
begin
  Result := 0;
  inc(Pos);
  if Pos < Size then begin
    if ord(TaggedData[Pos]) < $80 then
      Result := ord(TaggedData[Pos])
    else begin
      SizeLen := ord(TaggedData[Pos]) and $7F;
      if (SizeLen <= 4) and (Pos + SizeLen < Size) then begin
	for Index := 1 to SizeLen do begin
	  inc(Pos);
	  Result := Result shl 8 + ord(TaggedData[Pos])
	end;
      end;
    end;
    inc(Pos);
  end;
end;

function FindObjectID(const OID, ASNData: PChar; Size: DWord; var Pos: DWord): DWord;
var
  OIDLen: DWord;
begin
  Result := 0;
  OIDLen := StrLen(OID);
  while Pos + OIDLen <= Size do begin
    if OIDCheck(OID, ASNData, Pos) then begin
      Result := GetTypeLength(ASNData, Size, Pos);
      Break;
    end
    else if not (ASNData[Pos] in [#$30..#$31]) then begin
      Result := GetTypeLength(ASNData, Size, Pos);
      inc(Pos, Result);
      Continue;
    end;
    Result := GetTypeLength(ASNData, Size, Pos);
  end;
end;

function GetSerialNumber(const ASNData: PChar; Size: DWord): String;
const
  oidSN	= #$06#$03#$55#$04#$05#$13#0;
var
  Pos, Len: DWord;
begin
  Pos := 0;
  Result := '';
  Len := FindObjectID(oidSN, ASNData, Size, Pos);
  if Len <> 0 then begin
    inc(Pos, Len);
    Len := GetTypeLength(ASNData, Size, Pos);
    if Len <> 0 then begin
      SetLength(Result, Len);
      Move(ASNData[Pos], Pointer(Result)^, Len);
    end;
  end;
end;

function BytesToString(const Bytes: PChar; Size: DWord): String; register; assembler;
asm	// EAX = @Bytes, EDX = Size, ECX = @Result
  PUSH	EBX
  PUSH	EDI
  PUSH	ESI
  MOV	ESI,EAX
  MOV	EDI,ECX
  MOV	EAX,EDX
  MOV	EBX,EDX
  SHL	EAX,1
  DEC	EDX
  ADD	EBX,EAX
  PUSH	EDX
  DEC	EBX
  MOV	EAX,EDI
  CALL	SYSTEM.@LSTRCLR
  MOV	EAX,EBX
  CALL	SYSTEM.@NEWANSISTRING
  MOV	[EDI],EAX
  MOV	EDI,EAX
  SUB	EBX,2
  POP	ECX
  @LOOP:
  XOR	EAX,EAX
  MOV	AL,[ESI + ECX]
  MOV	EDX,EAX
  AND	EAX,0Fh
  SHR	EDX,4
  MOV	AH,BYTE PTR[@HEXCHARS + EAX]
  MOV	AL,BYTE PTR[@HEXCHARS + EDX]
  MOV	[EDI + EBX],AX
  SUB	EBX,3
  DEC	ECX
  JS	@END
  MOV	BYTE PTR[EDI + EBX + 2],' '
  JMP	@LOOP
  @END:
  POP	ESI
  POP	EDI
  POP	EBX
  RET
  @HEXCHARS:
  DB	'0123456789ABCDEF';
end;

function GetCertificateID(const ASNData: PChar; Size: DWord): String;
const
  oidID = #$02#0;
var
  Pos, Len: DWord;
begin
  Pos := 0;
  Result := '';
  Len := FindObjectID(oidID, ASNData, Size, Pos);
  if Len <> 0 then Result := BytesToString(@ASNData[Pos], Len);
end;

function GetAlgorithms(const ASNData: PChar; Size: DWord; var KeyAlg: TKeyAlgorithm; var HashAlg: THashAlgorithm): Integer;
const
  oidRSAFilter = #$06#$09#$2A#$86#$48#$86#$F7#$0D#$01#$01#0;
  oidDSAFilter = #$06#$07#$2A#$86#$48#$CE#$38#$04#0;
  idRSA = 1;
  idMD2WithRSA = 2;
  idMD5WithRSA = 4;
  idSHA1WithRSA = 5;
  idSHA256WithRSA = 11;
  idSHA384WithRSA = 12;
  idSHA512WithRSA = 13;
  idDSA = 2;
  idSHA1WithDSA = 3;
var
  Pos, Len: DWord;
begin
  Pos := 0;
  Result := 0;
  KeyAlg := KeyAlgorithm_Invalid;
  HashAlg := HashAlgorithm_Invalid;
  GetTypeLength(ASNData, Size, Pos);
  Pos := Pos + GetTypeLength(ASNData, Size, Pos);
  GetTypeLength(ASNData, Size, Pos);
  Len := GetTypeLength(ASNData, Size, Pos);
  if Len in [7, 9] then begin
    Result := ord(ASNData[Pos + pred(Len)]);
    if Len = 9 then begin
      KeyAlg := KeyAlgorithm_RSA;
      case Result of
	idMD2WithRSA: HashAlg := HashAlgorithm_MD2;
	idMD5WithRSA: HashAlg := HashAlgorithm_MD5;
	idSHA1WithRSA: HashAlg := HashAlgorithm_SHA;
	idSHA256WithRSA: HashAlg := HashAlgorithm_SHA256;
	idSHA384WithRSA: HashAlg := HashAlgorithm_SHA384;
	idSHA512WithRSA: HashAlg := HashAlgorithm_SHA512;
      end;
    end
    else if Len = 7 then begin
      KeyAlg := KeyAlgorithm_DSS;
      case Result of
	idSHA1WithDSA: HashAlg := HashAlgorithm_SHA;
      end;
    end;
  end;
end;

end.
