{$J+,Z4}
unit PGPDecode;

{------------------------------------------------------------------------------}
{                                                                              }
{                 This unit is partly based on Steve Heller's                  }
{         SPGP sources available from http://www.oz.net/~srheller/spgp/        }
{                                                                              }
{                Portions created by Michael in der Wiesche are                }
{              Copyright (C) 2001-2005 by Michael in der Wiesche               }
{                                                                              }
{------------------------------------------------------------------------------}

interface

uses
  Windows,
  Classes,
  SysUtils,
  KeyPropTypes,
  UTF8,
  pgpBase,
  pgpErrors,
  pgpPubTypes,
  pgpUtilities,
  pgpOptionList,
  pgpMemoryMgr,
  pgpEvents,
  pgpKeys,
  pgpTLS,
  pgpSC,
  KeyFuncs,
  PrefFuncs,
  PGPDialogs;

type
  TSigStatus = (
    SIGNED_NOT,
    SIGNED_BAD,
    SIGNED_GOOD,
    SIGNED_NO_KEY,
    SIGNED_UNKNOWN_ALG,
    SIGNED_SIG_EXPIRED
  );
  TSigPropsRec = Record
    sStatus: TSigStatus;
    sHexID: String;
    sUserID: String;
    sCreaTimeStr: String;
    sCreaTimeNum: DWord;
    sChecked: Longbool;
    sVerified: Longbool;
    sValidity: TValidityLevel;
    sRevoked: Longbool;
    sDisabled: Longbool;
    sExpired: Longbool;
    sDetached: Longbool;
    sAxiomatic: Longbool;
    sExpTimeNum: Longint;
    sExpTimeStr: String;
  end;
  TAnalysis = (
    ENCRYPTED,
    SIGNED,
    DETACHED_SIGNATURE,
    KEY,
    UNKNOWN,
    X509_CERTIFICATE
  );
  TPassThroughOption = (
    PassThrough_Keys,
    PassThrough_ClearSigned,
    PassThrough_Unrecognized
  );
  TPassOption = (
    AnalyseOnly,
    AnalysePass,
    DecodePass
  );
  TX509Mode = (
    X509Off,
    X509Binary,
    X509Armored,
    X509Processed
  );
  TPassThroughOptions = Set of TPassThroughOption;
  TOnAnalysis = procedure(SectionNumber: Longint; SectionType: TAnalysis; var SkipSection: Longbool) of Object;
  TOnGetInputFileName = procedure(var SuggestedName: String) of Object;
  TOnGetOutputFileName = procedure(var SuggestedName: String) of Object;
  TOnGetSignedFileName = procedure(var SuggestedName: String) of Object;
  TOnEnterPassphrase = procedure(const Passphrase: PChar;
				 const DecryptionKeyList: TKeyPropsList;
				 Conventional, BadPassphrase: Longbool;
				 var Cancel: Longbool) of Object;
  TOnGetSignature = procedure(const SigProps: TSigPropsRec) of Object;
  TOnShowProgress = procedure(BytesProcessed, BytesTotal: Longint) of Object;
  TOnWipePassphrase = procedure(const Passphrase: PChar) of Object;
  TPGPDecodeCustom = class(TComponent)
  private
    // internal
    FContext: pPGPContext;
    FKeySetMain: pPGPKeySet;
    FAllocatedOutputBuffer: PChar;
    FActualOutputSize: PGPSize;
    FOutputOptions: pPGPOptionList;
    FInputSize: PGPSize;
    FInputBuffer: PChar;
    FInputFileName: String;
    FFileNameBuffer: String;
    FAnalysis: TAnalysis;
    FClearSigned: Longbool;
    FEyesOnly: Longbool;
    FFileToFile: Longbool;
    FOutputExists: Longbool;
    FRecognized: Longbool;
    FRecurring: Longbool;
    FPassOption: TPassOption;
    FPassphrase: PChar;
    FPassSection: PGPUInt32;
    FRecipientsData: TPGPEventRecipientsData;
    FSectionNumber: PGPUInt32;
    FDecryptionKey: pPGPKey;
    FSecureBuffer: PChar;
    FSecureBufferSize: PGPSize;
    // protected properties
    FOutputBuffer: String;
    FDetachedSigVerify: Longbool;
    FIgnoreKnownFlag: Longint;
    FSigPropsRec: TSigPropsRec;
    FKeyPropsList: TKeyPropsList;
    // public properties
    FParentHandle: THandle;
    // published properties
    FFileOutput: Longbool;
    FGetKeyFromServer: Longbool;
    FLogBadSignature: String;
    FLogExpSignature: String;
    FLogKeyDisabled: String;
    FLogKeyExpired: String;
    FLogKeyRevoked: String;
    FLogKeyUnknown: String;
    FLogUnknownAlg: String;
    FPassCacheSeconds: TPassCacheSeconds;
    FPassCacheShare : Longbool;
    FPassThroughOptions: TPassThroughOptions;
    FProgressInterval: Cardinal;
    FQueryAddKeys: Longbool;
    FRecursivelyDecode: Longbool;
    FShowSigLog: Longbool;
    FX509Mode: TX509Mode;
    FKeyDlgPrompt: String;
    FPassDlgPrompt: String;
    FOutputFileName: String;
    FSignedFileName: String;
    // events
    FOnAnalysis: TOnAnalysis;
    FOnGetInputFileName: TOnGetInputFileName;
    FOnGetOutputFileName: TOnGetOutputFileName;
    FOnGetSignedFileName: TOnGetSignedFileName;
    FOnEnterPassphrase: TOnEnterPassphrase;
    FOnGetSignature: TOnGetSignature;
    FOnShowProgress: TOnShowProgress;
    FOnWipePassphrase: TOnWipePassphrase;
    procedure SetRecursive(Value: Longbool);
    procedure SetX509Mode(Value: TX509Mode);
    procedure ClearSigPropsRec;
    function  InitDecode: PGPError;
    procedure FinitDecode;
    function  IsPEMData: Longbool;
    function  SecureMove: PGPError;
    function  ShowSigInfo: PGPError;
    function  MaxFileSize: DWord;
    function  FileSize(const FileName: String; var Bytes: DWord): PGPError;
    function  ReadInputFile(const Input: PChar; InputSize: DWord): PGPError;
    function  WriteOutputFile(const Output: PChar; OutputSize: DWord): PGPError;
    function  GetOutputFileName: String;
    function  Analyze(InFileSpec: pPGPFileSpec): PGPError;
    function  GetOptionList(var OptionList: pPGPOptionList): PGPError;
    function  SetOutputOption(var OptionList: pPGPOptionList): PGPError;
    function  Decode(IsFile: Longbool): Longint;
    function  AnalyzeHandler(Event: pPGPEvent): PGPError;
    function  RecipientsHandler(Event: pPGPEvent): PGPError;
    function  PassphraseHandler(Event: pPGPEvent): PGPError;
    function  KeyFoundHandler(Event: pPGPEvent): PGPError;
    function  OutputHandler(Event: pPGPEvent): PGPError;
    function  SignatureHandler(Event: pPGPEvent): PGPError;
    function  EndLexHandler(Event: pPGPEvent): PGPError;
  protected
    property OutputBuffer: String
      read FOutputBuffer;
    property DetachedSignature: Longbool
      read FDetachedSigVerify;
    property IgnoreKnownFlag: Longint
      write FIgnoreKnownFlag;
    property SigPropsRec: TSigPropsRec
      read FSigPropsRec;
    property KeyPropsList: TKeyPropsList
      read FKeyPropsList;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    function AnalyseBuffer(const DataBuffer: String): Longint; virtual;
    function AnalyseFile(const FileName: String): Longint; virtual;
    function DecodeBuffer(const DataBuffer: String): Longint; virtual;
    function DecodeFile(const FileName: String): Longint; virtual;
    property ParentHandle: THandle
      read FParentHandle
      write FParentHandle;
  published
    property FileOutput: Longbool
      read FFileOutput
      write FFileOutput;
    property GetKeyFromServer: Longbool
      read FGetKeyFromServer
      write FGetKeyFromServer;
    property LogBadSignature: String
      read FLogBadSignature
      write FLogBadSignature;
    property LogExpSignature: String
      read FLogExpSignature
      write FLogExpSignature;
    property LogKeyDisabled: String
      read FLogKeyDisabled
      write FLogKeyDisabled;
    property LogKeyExpired: String
      read FLogKeyExpired
      write FLogKeyExpired;
    property LogKeyRevoked: String
      read FLogKeyRevoked
      write FLogKeyRevoked;
    property LogKeyUnknown: String
      read FLogKeyUnknown
      write FLogKeyUnknown;
    property LogUnknownAlg: String
      read FLogUnknownAlg
      write FLogUnknownAlg;
    property PassCacheSeconds: TPassCacheSeconds
      read FPassCacheSeconds
      write FPassCacheSeconds;
    property PassCacheShare : Longbool
      read FPassCacheShare
      write FPassCacheShare;
    property PassThroughOptions: TPassThroughOptions
      read FPassThroughOptions
      write FPassThroughOptions;
    property ProgressInterval: Cardinal
      read FProgressInterval
      write FProgressInterval;
    property RecursivelyDecode: Longbool
      read FRecursivelyDecode
      write SetRecursive;
    property QueryAddKeys: Longbool
      read FQueryAddKeys
      write FQueryAddKeys;
    property ShowSigLog: Longbool
      read FShowSigLog
      write FShowSigLog;
    property X509Mode: TX509Mode
      read FX509Mode
      write SetX509Mode;
    property KeyDlgPrompt: String
      read FKeyDlgPrompt
      write FKeyDlgPrompt;
    property PassDlgPrompt: String
      read FPassDlgPrompt
      write FPassDlgPrompt;
    property OutputFileName: String
      read FOutputFileName
      write FOutputFileName;
    property SignedFileName: String
      read FSignedFileName
      write FSignedFileName;
    property OnAnalysis: TOnAnalysis
      read FOnAnalysis
      write FOnAnalysis;
    property OnGetInputFileName: TOnGetInputFileName
      read FOnGetInputFileName
      write FOnGetInputFileName;
    property OnGetSignedFileName: TOnGetSignedFileName
      read FOnGetSignedFileName
      write FOnGetSignedFileName;
    property OnGetOutputFileName: TOnGetOutputFileName
      read FOnGetOutputFileName
      write FOnGetOutputFileName;
    property OnEnterPassphrase: TOnEnterPassphrase
      read FOnEnterPassphrase
      write FOnEnterPassphrase;
    property OnGetSignature: TOnGetSignature
      read FOnGetSignature
      write FOnGetSignature;
    property OnShowProgress: TOnShowProgress
      read FOnShowProgress
      write FOnShowProgress;
    property OnWipePassphrase: TOnWipePassphrase
      read FOnWipePassphrase
      write FOnWipePassphrase;
  end;

implementation

function EventHandler(Context: pPGPContext; Event: pPGPEvent; UserValue: PGPUserValue): PGPError; cdecl;
begin
  Result := 0;
  with TPGPDecodeCustom(UserValue) do begin
    case Event^.EType of
      kPGPEvent_NullEvent:		if (FPassOption = DecodePass)
					and not (FDetachedSigVerify or FClearSigned) then begin
					  if TMethod(FOnShowProgress).Code <> nil then begin
					    // BytesTotal always stays 0 => use FInputSize (at least in 6.5.X)
					    with Event^.EData.NullData do FOnShowProgress(BytesWritten, FInputSize);
					  end;
					  ProcessMessages;
					end;
      kPGPEvent_InitialEvent:		;
      kPGPEvent_FinalEvent:		;
      kPGPEvent_ErrorEvent:	  	Result := Event^.EData.ErrorData.Error;
      kPGPEvent_WarningEvent:		;
      kPGPEvent_PassphraseEvent:	Result := PassphraseHandler(Event);
      kPGPEvent_AnalyzeEvent:		Result := AnalyzeHandler(Event);
      kPGPEvent_RecipientsEvent:  	if FPassOption = DecodePass then Result := RecipientsHandler(Event);
      kPGPEvent_KeyFoundEvent:		if FPassOption = DecodePass then Result := KeyFoundHandler(Event);
      kPGPEvent_OutputEvent:		if FPassOption = DecodePass then OutputHandler(Event);
      kPGPEvent_SignatureEvent:		if FPassOption = DecodePass then Result := SignatureHandler(Event);
      kPGPEvent_BeginLexEvent:	  	FSectionNumber := succ(Event^.EData.BeginLexData.SectionNumber);
      kPGPEvent_EndLexEvent:      	Result := EndLexHandler(Event);
      kPGPEvent_DetachedSignatureEvent:	;
      kPGPEvent_DecryptionEvent:	;
    end;
  end;
end;

constructor TPGPDecodeCustom.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FKeyPropsList := TKeyPropsList.Create(spgpKeyPropFlag_IDFlags);
  FLogBadSignature := 'bad signature';
  FLogExpSignature := 'expired signature';
  FLogKeyDisabled := 'disabled key';
  FLogKeyExpired := 'expired key';
  FLogKeyRevoked := 'revoked key';
  FLogKeyUnknown := 'unknown key';
  FProgressInterval := 1000;
end;

destructor TPGPDecodeCustom.Destroy;
begin
  FKeyPropsList.Free;
  inherited Destroy;
end;

procedure TPGPDecodeCustom.SetRecursive(Value: Longbool);
begin
  if Value <> FRecursivelyDecode then FRecursivelyDecode := Value;
  if FRecursivelyDecode then Include(FPassThroughOptions, PassThrough_ClearSigned);
end;

procedure TPGPDecodeCustom.SetX509Mode(Value: TX509Mode);
begin
  if PGP7X and (FPassOption <> DecodePass) then FX509Mode := Value;
end;

procedure TPGPDecodeCustom.ClearSigPropsRec;
begin
  with FSigPropsRec do begin
    sStatus:=SIGNED_NOT;
    sHexID:='';
    sUserID:='';
    sCreaTimeStr:='';
    sCreaTimeNum:=0;
    sChecked:=false;
    sVerified:=false;
    sValidity:=Validity_Unknown;
    sRevoked:=false;
    sDisabled:=false;
    sExpired:=false;
    sDetached:=false;
    sAxiomatic:=false;
    sExpTimeNum:=0;
    sExpTimeStr:='';
  end;
end;

function TPGPDecodeCustom.InitDecode: PGPError;
begin
  FAnalysis := UNKNOWN;
  FSectionNumber := 0;
  FOutputBuffer := '';
  FSecureBufferSize := 0;
  FSecureBuffer := nil;
  FActualOutputSize := 0;
  FAllocatedOutputBuffer := nil;
  FDetachedSigVerify := false;
  FClearSigned := false;
  FRecognized := false;
  if not FRecurring then FEyesOnly := false;
  FKeyPropsList.Clear;
  FPassSection := 0;
  FDecryptionKey := nil;
  FPassphrase := nil;
  ClearSigPropsRec;
  FillChar(FRecipientsData, SizeOf(TPGPEventRecipientsData), 0);
  Result := KeyRings.InitKeyRings(FContext, FKeySetMain);
end;

procedure TPGPDecodeCustom.FinitDecode;
begin
  FX509Mode := X509Off;
  if (FKeyPropsList.Count <> 0) or (FAnalysis = X509_CERTIFICATE) then KeyRings.UpdateKeyRings;
  with FRecipientsData do begin
    PGPFreeData(KeyIDArray);
    PGPFreeKeySet(RecipientSet);
  end;
  KeyRings.FreeKeyRings;
end;

function TPGPDecodeCustom.IsPEMData: Longbool;
var
  Pos	: DWord;
begin
  Pos := DWord(StrPos(FInputBuffer, '-----BEGIN CERTIFICATE')) - DWord(FInputBuffer);
  Result := (Pos = 0) or ((Pos > 0) and (FInputBuffer[pred(Pos)] = #10));
end;

function TPGPDecodeCustom.SecureMove: PGPError;
begin
  try
    if FEyesOnly and (FSecureBuffer = nil) then begin
      Result := kPGPError_OutOfMemory;
      FSecureBuffer := PGPNewSecureData(PGPGetDefaultMemoryMgr,
					succ(FSecureBufferSize + FActualOutputSize),
					kPGPMemoryMgrFlags_None);
      if FSecureBuffer <> nil then Result := 0;
    end
    else begin
      Result := PGPReallocData(PGPGetDefaultMemoryMgr,
			       pointer(FSecureBuffer),
			       succ(FSecureBufferSize + FActualOutputSize),
			       kPGPMemoryMgrFlags_None);
    end;
    if Result = 0 then begin
      Move(FAllocatedOutputBuffer[0], FSecureBuffer[FSecureBufferSize], FActualOutputSize);
      if FEyesOnly then FillChar(FAllocatedOutputBuffer^, FActualOutputSize, 0);
      inc(FSecureBufferSize, FActualOutputSize);
    end;
  finally
    PGPFreeData(FAllocatedOutputBuffer);
    FAllocatedOutputBuffer := nil;
    FActualOutputSize := 0;
  end;
end;

function TPGPDecodeCustom.ShowSigInfo: PGPError;
var
  Status	: String;
  Source	: String;
  ID		: String;
  Validity	: String;
  Info		: String;
begin
  Result := 0;
  with FSigPropsRec do begin
    if sStatus <> SIGNED_NOT then begin
      // don't change these (constant) PGP strings, see SigEvent.c
      if sStatus = SIGNED_GOOD then
	Status := 'Good Signature' + #10
      else Status := 'Bad Signature' + #10;
      // these strings may be customized
      if FInputFileName <> '' then begin
	if FDetachedSigVerify then
	  Source := ExtractFileName(FSignedFileName) + #10
	else if FRecurring then
	  Source := ExtractFileName(FFileNameBuffer) + #10
	else Source := ExtractFileName(FInputFileName) + #10;
      end
      else Source := '(...)' + #10;
      if (sStatus = SIGNED_NO_KEY) or (sUserID = '') then begin
	if PGP8X then
	  ID := '(' + sHexID + ')' + #10 + ShortHexID(sHexID) + #10
	else ID := '(' + ShortHexID(sHexID) + ')' + #10;
      end
      else begin
	if PGP8X then
	  ID := sUserID + #10 + ShortHexID(sHexID) + #10
	else ID := sUserID + #10;
      end;
      // don't change these (constant) PGP strings, see SigEvent.c
      if sAxiomatic then
	Validity := 'Implicit Trust' + #10 + sCreaTimeStr
      else begin
	case sValidity of
	  Validity_Unknown, Validity_Invalid: Validity := 'Invalid Key' + #10 + sCreaTimeStr;
	  Validity_Marginal: Validity := 'Marginal Key' + #10 + sCreaTimeStr;
	  Validity_Complete: Validity := 'Valid Key' + #10 + sCreaTimeStr;
	end;
      end;
      // these strings may be customized
      if sUserID = '' then
	Info := FLogKeyUnknown
      else begin
	if sStatus <> SIGNED_UNKNOWN_ALG then begin
	  if sVerified then begin
	    if sStatus = SIGNED_SIG_EXPIRED then
	      Info := FLogExpSignature
	    else begin
	      if sRevoked then Info := FLogKeyRevoked;
	      if sDisabled then Info := FLogKeyDisabled;
	      if sExpired then Info := FLogKeyExpired;
	    end;
	  end
	  else Info := FLogBadSignature;
	end
	else Info := FLogUnknownAlg;
      end;
      if Info <> '' then  Info := ' (' + Info + ')';
      if not SendPGPlogMsg(FParentHandle, PChar(Status + Source + ID + Validity + Info)) then begin
	Result := kPGPError_FeatureNotAvailable;
      end;
    end;
  end;
end;

function TPGPDecodeCustom.MaxFileSize: DWord;
var
  MemoryStatus	: TMemoryStatus;
begin
  MemoryStatus.dwLength := SizeOf(TMemoryStatus);
  GlobalMemoryStatus(MemoryStatus);
  Result := MemoryStatus.dwAvailPhys shr 2;
end;

function TPGPDecodeCustom.FileSize(const FileName: String; var Bytes: DWord): PGPError;
var
  FileHandle	: THandle;
  FileSizeHigh	: DWord;
begin
  Result := 0;
  FileHandle := FileOpen(FInputFileName, fmOpenRead or fmShareDenyNone);
  if FileHandle <> INVALID_HANDLE_VALUE then begin
    Bytes := GetFileSize(FileHandle, @FileSizeHigh);
    FileClose(FileHandle);
  end
  else Bytes := $FFFFFFFF;
  if (Bytes = 0) or (Bytes = $FFFFFFFF) or (FileSizeHigh <> 0) then Result := kPGPError_ReadFailed;
end;

function TPGPDecodeCustom.ReadInputFile(const Input: PChar; InputSize: DWord): PGPError;
var
  InputFile	: THandle;
  BytesRead	: DWord;
begin
  InputFile := CreateFile(PChar(FInputFileName), GENERIC_READ, FILE_SHARE_READ, nil, OPEN_EXISTING,
			   FILE_FLAG_SEQUENTIAL_SCAN, 0);
  if InputFile <> INVALID_HANDLE_VALUE then begin
    try
      if not ReadFile(InputFile, Input[0], InputSize, BytesRead, nil) or (BytesRead <> InputSize) then
	Result := kPGPError_ReadFailed
      else Result := kPGPError_NoErr;
    finally
      CloseHandle(InputFile);
    end;
  end
  else Result := kPGPError_CantOpenFile;
end;

function TPGPDecodeCustom.WriteOutputFile(const Output: PChar; OutputSize: DWord): PGPError;
var
  OutputFile	: THandle;
  BytesWritten	: DWord;
begin
  OutputFile := CreateFile(PChar(FOutputFileName), GENERIC_WRITE, 0, nil, CREATE_ALWAYS,
			   FILE_ATTRIBUTE_ARCHIVE or FILE_FLAG_WRITE_THROUGH, 0);
  if OutputFile <> INVALID_HANDLE_VALUE then begin
    try
      if not WriteFile(OutputFile, Output[0], OutputSize, BytesWritten, nil) or (BytesWritten <> OutputSize) then
	Result := kPGPError_WriteFailed
      else Result := kPGPError_NoErr;
    finally
      CloseHandle(OutputFile);
    end;
  end
  else Result := kPGPError_CantOpenFile;
end;

function TPGPDecodeCustom.GetOutputFileName: String;
var
  FileExtension	: String;
begin
  if FDetachedSigVerify then begin
    if FSignedFileName = '' then begin
      FSignedFileName := Trim(FInputFileName);
      FileExtension := LowerCase(ExtractFileExt(FSignedFileName));
      if (FileExtension = '.sig') or (FileExtension = '.asc') then begin
	Delete(FSignedFileName, Length(FSignedFileName) - 3, 4);
      end;
    end;
    if ((FSignedFileName = FInputFileName) or not FileExists(FSignedFileName))
    and Assigned(FOnGetSignedFileName) then begin
      FOnGetSignedFileName(FSignedFileName);
    end;
    Result := FSignedFileName;
  end
  else begin
    if FOutputFileName = '' then begin
      FOutputFileName := Trim(FInputFileName);
      FileExtension := LowerCase(ExtractFileExt(FOutputFileName));
      if (FileExtension = '.pgp') or (FileExtension = '.gpg') or (FileExtension = '.asc') then begin
	Delete(FOutputFileName, Length(FOutputFileName) - 3, 4);
      end;
    end;
    if (FOutputFileName = '') or (ExtractFileExt(FOutputFileName) = '') or FileExists(FOutputFileName) then begin
      if Assigned(FOnGetOutputFileName) then
	FOnGetOutputFileName(FOutputFileName)
      else FOutputFileName := '';
    end;
    Result := FOutputFileName;
    if Result <> '' then DeleteFile(Result);
  end;
end;

function TPGPDecodeCustom.Analyze(InFileSpec: pPGPFileSpec): PGPError;
var
  OptionList	: pPGPOptionList;
begin
  OptionList := nil;
  Result := PGPBuildOptionList(FContext, OptionList,
    [
      PGPODiscardOutput(FContext, PGPTrue),
      PGPOEventHandler(FContext, EventHandler, Self)
    ]);
  if Result <> 0 then Exit;
  if InFileSpec <> nil then begin
    Result := PGPAppendOptionList(OptionList, [PGPOInputFile(FContext, InFileSpec)]);
  end
  else if FInputSize <> 0 then begin
    Result := PGPAppendOptionList(OptionList, [PGPOInputBuffer(FContext, FInputBuffer, FInputSize)]);
  end;
  if Result <> 0 then Exit;
  try
    Result := pgpEvents.PGPDecode(FContext, OptionList, PGPOLastOption(FContext));
    if Result = kPGPError_UserAbort then Result := 0;
  finally
    PGPFreeOptionList(OptionList);
  end;
end;

function TPGPDecodeCustom.GetOptionList(var OptionList: pPGPOptionList): PGPError;
var
  X509Option: pPGPOptionList;
begin
  Result := PGPBuildOptionList(FContext, OptionList,
    [
      PGPOKeySetRef(FContext, FKeySetMain),
      PGPOSendNullEvents(FContext, FProgressInterval),
      PGPOPassThroughKeys(FContext, PGPBoolean(PassThrough_Keys in PassThroughOptions)),
      PGPOPassThroughClearSigned(FContext, PGPBoolean(PassThrough_ClearSigned in PassThroughOptions)),
      PGPOPassThroughIfUnrecognized(FContext, PGPBoolean(PassThrough_Unrecognized in PassThroughOptions)),
      PGPORecursivelyDecode(FContext, PGPBoolean(FRecursivelyDecode) and 1),
      PGPOSendEventIfKeyFound(FContext, PGPTrue),
      PGPOEventHandler(FContext, EventHandler, Self)
    ]);
  if Result <> 0 then Exit;
  case FX509Mode of
    X509Binary: X509Option := PGPOInputFormat(FContext, kPGPInputFormat_X509DataInPKCS7);
    X509Armored: X509Option := PGPOInputFormat(FContext, kPGPInputFormat_PEMEncodedX509Cert);
  else
    Exit;
  end;
  Result := PGPAppendOptionList(OptionList, [PGPOImportKeysTo(FContext, FKeySetMain), X509Option]);
end;

function TPGPDecodeCustom.SetOutputOption(var OptionList: pPGPOptionList): PGPError;
var
  FileSpec	: pPGPFileSpec;
begin
  if FDetachedSigVerify then begin
    FFileToFile := false;
    if GetOutputFileName <> '' then begin
      FileSpec := nil;
      Result := PGPNewFileSpecFromFullPath(FContext, PChar(FSignedFileName), FileSpec);
      if Result <> 0 then Exit;
      try
	Result := PGPAppendOptionList(OptionList,
	  [
	    PGPODetachedSig(FContext, PGPOInputFile(FContext, FileSpec), PGPOLastOption(FContext))
	  ]);
      finally
	PGPFreeFileSpec(FileSpec);
      end;
    end
    else Result := kPGPError_DetachedSignatureFound;
  end
  else begin
    if FFileOutput then begin
      if GetOutputFileName = '' then begin
	Result := kPGPError_CantOpenFile;
	Exit;
      end;
    end;
    if FFileToFile then begin
      FileSpec := nil;
      Result := PGPNewFileSpecFromFullPath(FContext, PChar(FOutputFileName), FileSpec);
      try
	if Result = 0 then begin
	  Result := PGPAppendOptionList(OptionList,
	    [
	      PGPOOutputFile(FContext, FileSpec),
	      PGPOAppendOutput(FContext, PGPTrue)
	    ]);
	end;
      finally
	PGPFreeFileSpec(FileSpec);
      end;
    end
    else begin
      Result := PGPAppendOptionList(OptionList,
	[
	  PGPOAllocatedOutputBuffer(FContext, FAllocatedOutputBuffer, $FFFFFFFF, FActualOutputSize)
	]);
    end;
  end;
end;

function TPGPDecodeCustom.Decode(IsFile: Longbool): Longint;
var
  BugFix	: Longbool;
  Buffered	: Longbool;
  OptionList	: pPGPOptionList;
  InFileSpec	: pPGPFileSpec;
  Output	: PChar;
  OutputSize	: DWord;
  Input		: PChar;
  OutputBuffer	: PChar;
  BufferSize	: DWord;
begin
  Result := 0;
  BugFix := false;
  Buffered := false;
  if IsFile then begin
    FInputFileName := FInputBuffer;
    if not FileExists(FInputFileName) then begin
      if Assigned(FOnGetInputFileName) then begin
	FOnGetInputFileName(FInputFileName);
	if FInputFileName = '' then Result := kPGPError_FileNotFound;
      end
      else Result := kPGPError_FileNotFound;
    end;
    if Result <> 0 then Exit;
    Result := FileSize(FInputFileName, FInputSize);
    if PGP7X and (FInputSize > MaxFileSize) then
      FFileToFile := FFileOutput
    else FFileToFile := false;
  end
  else if FInputSize = 0 then Result := kPGPError_ItemNotFound;
  Output := nil;
  OutputSize := 0;
  Input := FInputBuffer;
  try
    if Result = 0 then begin
      InFileSpec := nil;
      OptionList := nil;
      Result := InitDecode;
      if Result <> 0 then Exit;
      if IsFile then begin
	if PGP7X and not FFileToFile then begin
	  FInputBuffer := PGPNewSecureData(PGPGetDefaultMemoryMgr, succ(FInputSize), kPGPMemoryMgrFlags_None);
	  FInputBuffer[FInputSize] := #0;
	  if FInputBuffer <> nil then begin
	    Result := ReadInputFile(FInputBuffer, FInputSize);
	    Input := FInputBuffer;
	    Buffered := true;
	    IsFile := false;
	  end
	  else begin
	    Result := kPGPError_OutOfMemory;
	    Exit;
	  end;
	end
	else begin
	  Result := PGPNewFileSpecFromFullPath(FContext, PChar(FInputFileName), InFileSpec);
	  if Result <> 0 then Exit;
	end;
      end;
      try
	Result := Analyze(InFileSpec);
	if FPassOption <> AnalyseOnly then begin
	  if Result <> 0 then begin // try to circumvent errors with some non-ASCII data
	    if (FAnalysis = UNKNOWN) and not IsFile and (FInputSize > 0) and (FInputBuffer[0] > #127) then begin
	      BugFix := true;
	      inc(FInputSize);
	      if FEyesOnly then
		FInputBuffer := PGPNewSecureData(PGPGetDefaultMemoryMgr, FInputSize + 2, kPGPMemoryMgrFlags_None)
	      else FInputBuffer := PGPNewData(PGPGetDefaultMemoryMgr, FInputSize + 2, kPGPMemoryMgrFlags_None);
	      FInputBuffer[succ(FInputSize)] := #0;
	      if FInputBuffer <> nil then begin
		Move(Input[0], FInputBuffer[1], FInputSize);
		FInputBuffer[0]:='@';
		Result := 0;
	      end
	      else begin
		Result := kPGPError_OutOfMemory;
		Exit;
	      end;
	    end
	    else Exit;
	  end;
	  if PGP7X and not (FRecognized or IsFile) and (FX509Mode = X509Off) then begin
	    if IsPEMData then
	      FX509Mode := X509Armored
	    else FX509Mode := X509Binary;
	  end;
	  Result := GetOptionList(OptionList);
	  if Result <> 0 then Exit;
	  if IsFile then
	    Result := PGPAppendOptionList(OptionList, [PGPOInputFile(FContext, InFileSpec)])
	  else Result := PGPAppendOptionList(OptionList, [PGPOInputBuffer(FContext, FInputBuffer, FInputSize)]);
	  try
	    if Result <> 0 then Exit;
	    if FDetachedSigVerify or FClearSigned then begin
	      Result := SetOutputOption(OptionList);
	      if Result <> 0 then Exit;
	    end
	    else FOutputExists := false;
	    if not FRecurring or FClearSigned then begin
	      FPassOption := DecodePass;
	      Result := pgpEvents.PGPDecode(FContext, OptionList, PGPOLastOption(FContext));
	      if (FPassOption <> AnalyseOnly) and (Result = 0) then begin
		if not FFileToFile then begin
		  if not (FDetachedSigVerify or FClearSigned) then begin
		    Result := SecureMove;
		    FAllocatedOutputBuffer := FSecureBuffer;
		    FActualOutputSize := FSecureBufferSize;
		    if Result <> 0 then Exit;
		  end;
		  if BugFix and (PassThrough_Unrecognized in PassThroughOptions) then begin // remove leading AT
		    Output := FAllocatedOutputBuffer + 1;
		    OutputSize := pred(FActualOutputSize);
		  end
		  else begin // AT already removed
		    Output := FAllocatedOutputBuffer;
		    OutputSize := FActualOutputSize;
		  end;
		end;
	      end;
	    end;
	  finally
	    try
	      if Assigned(FOnWipePassphrase) then begin
		if PGP8X then Utf8ToAnsiPChar(FPassphrase, FPassphrase, MaxUTF8Length);
		FOnWipePassphrase(FPassphrase);
	      end;
	      if FPassCacheSeconds <> 0 then CachePassphrase(FContext, FDecryptionKey, FPassCacheSeconds, FPassCacheShare, FPassphrase);
	    finally
	      PGPFreeData(FPassphrase);
	      FPassphrase := nil;
	    end;
	  end;
	end
	else if Result <> kPGPError_AsciiParseIncomplete then Result := 0; // assume FAnalysis = UNKNOWN on other errors
      finally
	PGPFreeFileSpec(InFileSpec);
	PGPFreeOptionList(OptionList);
	if BugFix then begin
	  dec(FInputSize);
	  PGPFreeData(FInputBuffer);
	  FInputBuffer := Input;
	end;
      end;
    end;
    // PGP doesn't automatically recognize X509 data which forces us to recursively decode
    if (Result = kPGPError_InvalidPKCS7Encoding) or (Result = -1) and (FX509Mode = X509Armored) then begin
      FX509Mode := X509Processed;
      FPassOption := AnalysePass;
      if IsFile then
	FInputBuffer := PChar(FInputFileName)
      else FInputBuffer := Input;
      Result := Decode(IsFile);
      Exit;
    end
    else if Result <> 0 then begin
      if FAnalysis = X509_CERTIFICATE then FAnalysis := UNKNOWN;
      Exit;
    end;
    // workaround for verifying encrypted PGP/MIME signatures like PGP does internally with encrypted clear signed data
    if FRecursivelyDecode and not FRecurring and (FPassOption <> AnalyseOnly)
    and (FAnalysis = ENCRYPTED) and (FSigPropsRec.sStatus = SIGNED_NOT) then begin
      OutputBuffer := FAllocatedOutputBuffer;
      BufferSize := FActualOutputSize;
      FPassOption := AnalysePass;
      FRecurring := true;
      try
	if FFileToFile then begin
	  FFileNameBuffer := FInputFileName;
	  SetLength(FInputFileName, MAX_PATH);
	  if GetTempFileName(PChar(ExtractFilePath(FOutputFileName)), 'PGP', 0, PChar(FInputFileName)) <> 0 then begin
	    SetLength(FInputFileName, StrLen(pointer(FInputFileName)));
	    if MoveFileEx(PChar(FOutputFileName), PChar(FInputFileName), MOVEFILE_REPLACE_EXISTING) then begin
	      FInputBuffer := PChar(FInputFileName);
	      FInputSize := Length(FInputFileName);
	      Result := Decode(true);
	      if Result <> 0 then DeleteFile(FInputFileName);
	    end
	    else begin
	      Result := kPGPError_FileOpFailed;
	      DeleteFile(FOutputFileName);
	      DeleteFile(FInputFileName);
	      Exit;
	    end;
	  end
	  else begin
	    Result := kPGPError_FileOpFailed;
	    DeleteFile(FOutputFileName);
	    Exit;
	  end;
	end
	else begin
	  FInputBuffer := Output;
	  FInputSize := OutputSize;
	  Result := Decode(false);
	end;
      finally
	if FEyesOnly then begin
	  if PGPGetMemoryMgrDataInfo(OutputBuffer)
	  and (kPGPMemoryMgrBlockInfo_Valid or kPGPMemoryMgrBlockInfo_Secure) = kPGPMemoryMgrBlockInfo_Valid then begin
	    FillChar(OutputBuffer^, BufferSize, 0);
	  end;
	end;
	PGPFreeData(OutputBuffer);
	FRecurring := false;
      end;
    end
    else if FFileToFile then begin
      if FRecurring then begin
	if FAnalysis = SIGNED then
	  DeleteFile(FInputFileName)
	else if not RenameFile(FInputFileName, FOutputFileName) then begin
	  Result := kPGPError_FileOpFailed;
	  DeleteFile(FInputFileName);
	end;
      end;
    end
    else begin
      if FRecurring and (FAnalysis <> SIGNED) then begin
	Output := FInputBuffer;
	OutputSize := FInputSize;
      end;
      if FEyesOnly then begin
	Result := TempestViewer(FContext, FParentHandle, Output, OutputSize, true);
	ProcessMessages;
      end
      else if not FFileOutput then begin
	try
	  SetLength(FOutputBuffer, OutputSize);
	  Move(Output^, pointer(FOutputBuffer)^, OutputSize);
	except
	  on EOutOfMemory do Result := kPGPError_OutOfMemory;
	end;
      end
      else WriteOutputFile(Output, OutputSize);
    end;
  finally
    if Buffered then begin
      PGPFreeData(FInputBuffer);
      FInputBuffer := nil;
    end;
    try
      if FEyesOnly then begin
	if PGPGetMemoryMgrDataInfo(FAllocatedOutputBuffer)
	and (kPGPMemoryMgrBlockInfo_Valid or kPGPMemoryMgrBlockInfo_Secure) = kPGPMemoryMgrBlockInfo_Valid then begin
	  FillChar(FAllocatedOutputBuffer^, FActualOutputSize, 0);
	end;
      end;
    finally
      try
	PGPFreeData(FAllocatedOutputBuffer);
	FAllocatedOutputBuffer := nil;
      finally
	FinitDecode;
      end;
    end;
  end;
end;

function TPGPDecodeCustom.AnalyseBuffer(const DataBuffer: String): Longint;
begin
  FPassOption := AnalyseOnly;
  FInputBuffer := PChar(DataBuffer);
  FInputSize := Length(DataBuffer);
  FInputFileName := '';
  FOutputOptions := nil;
  try
    Result := Decode(false);
  finally
    PGPFreeOptionList(FOutputOptions);
  end;
end;

function TPGPDecodeCustom.AnalyseFile(const FileName: String): Longint;
begin
  FPassOption := AnalyseOnly;
  FInputBuffer := PChar(FileName);
  FInputSize := Length(FileName);
  FOutputOptions := nil;
  try
    Result := Decode(true);
  finally
    PGPFreeOptionList(FOutputOptions);
  end;
end;

function TPGPDecodeCustom.DecodeBuffer(const DataBuffer: String): Longint;
begin
  FPassOption := AnalysePass;
  FInputBuffer := PChar(DataBuffer);
  FInputSize := Length(DataBuffer);
  FInputFileName := '';
  FOutputOptions := nil;
  try
    Result := Decode(false);
  finally
    PGPFreeOptionList(FOutputOptions);
  end;
end;

function TPGPDecodeCustom.DecodeFile(const FileName: String): Longint;
begin
  FPassOption := AnalysePass;
  FInputBuffer := PChar(FileName);
  FInputSize := Length(FileName);
  FOutputOptions := nil;
  try
    Result := Decode(true);
  finally
    PGPFreeOptionList(FOutputOptions);
  end;
end;

function TPGPDecodeCustom.AnalyzeHandler(Event: pPGPEvent): PGPError;
var
  SkipSection	: Longbool;
begin
  Result := 0;
  SkipSection := false;
  with Event^.EData.AnalyzeData do begin
    case FPassOption of
      AnalyseOnly: begin
	if Assigned(FOnAnalysis) then FOnAnalysis(FSectionNumber, TAnalysis(SectionType), SkipSection);
	if SectionType <> kPGPAnalyze_Key then Result := kPGPError_SkipSection; // avoid key data bug
      end;
      AnalysePass: begin
	FRecognized := (SectionType <> kPGPAnalyze_Unknown);
	case SectionType of
	  kPGPAnalyze_Signed: FClearSigned := true;
	  kPGPAnalyze_DetachedSignature: FDetachedSigVerify := true;
	end;
	if FDetachedSigVerify or FClearSigned then
	  Result := kPGPError_UserAbort
	else if SectionType <> kPGPAnalyze_Key then Result := kPGPError_SkipSection; // avoid key data bug
      end;
      DecodePass: begin
	if Assigned(FOnAnalysis) then FOnAnalysis(FSectionNumber, TAnalysis(SectionType), SkipSection);
	if SectionType <> kPGPAnalyze_Unknown then FAnalysis := TAnalysis(SectionType);
	Result := (ord(SkipSection) and 1) * kPGPError_SkipSection;
      end;
    end;
  end;
end;

function TPGPDecodeCustom.RecipientsHandler(Event: pPGPEvent): PGPError;
var
  KeyIDArraySize: PGPSize;
begin
  Result := 0;
  with Event^.EData.RecipientsData do begin
    PGPIncKeySetRefCount(RecipientSet);
    FRecipientsData.RecipientSet := RecipientSet;
    FRecipientsData.KeyCount := KeyCount;
    if PGP7X then
      KeyIDArraySize := SizeOf(TPGPKeyID7) * KeyCount
    else KeyIDArraySize := SizeOf(TPGPKeyID6) * KeyCount;
    FRecipientsData.KeyIDArray := PGPNewData(PGPGetDefaultMemoryMgr, KeyIDArraySize, kPGPMemoryMgrFlags_None);
    if FRecipientsData.KeyIDArray <> nil then Move(KeyIDArray^, FRecipientsData.KeyIDArray^, KeyIDArraySize)
  end;
end;

function TPGPDecodeCustom.PassphraseHandler(Event: pPGPEvent): PGPError;
var
  Cancel	: Longbool;
  KeyPropsList	: TKeyPropsList;
  BadPassphrase	: Longbool;
begin
  Result := 0;
  Cancel := false;
  with Event^.EData.PassphraseData do begin
    KeyPropsList := TKeyPropsList.Create(spgpKeyPropFlag_IDComplete);
    try
      if (FPassphrase = nil) or (FSectionNumber = FPassSection) then begin
	FPassSection := FSectionNumber;
	BadPassphrase := false;
	if FPassphrase <> nil then begin
	  PGPFreeData(FPassphrase);
	  FPassphrase := nil;
	  BadPassphrase := true;
	end;
	if boolean(IsConventional) or (FPassphrase <> nil) or (FPassCacheSeconds = 0)
	or not GetCachedPassphrase(FContext, FDecryptionKey, KeySet, FPassphrase) then begin
	  if Assigned(FOnEnterPassphrase) then begin
	    if not boolean(IsConventional) then begin
	      Result := GetKeySetProps(FContext, KeySet, KeyPropsList, spgpKeyPropFlag_IDComplete,
				       KeyFilterFlag_CanDecrypt, UserID_Ordering);
	      if Result = 0 then Result := kPGPError_SecretKeyNotFound;
	    end;
	    if Result >= 0 then begin
	      Result := kPGPError_UserAbort;
	      FPassphrase := PGPNewSecureData(PGPGetDefaultMemoryMgr, MaxUTF8Length, kPGPMemoryMgrFlags_Clear);
	      if FPassphrase <> nil then begin
		FOnEnterPassphrase(FPassphrase, KeyPropsList, boolean(IsConventional), BadPassphrase, Cancel);
		if not Cancel then begin
		  if PGP8X then AnsiToUtf8PChar(FPassphrase, FPassphrase, MaxUTF8Length);
		  if KeyPropsList.Count = 1 then
		    GetKeyFromKeySet(KeySet, FDecryptionKey)
		  else FDecryptionKey := GetMatchingKey(FContext, KeySet, FPassphrase);
		end
		else Exit;
	      end
	      else begin
		Result := kPGPError_OutOfMemory;
		Exit;
	      end;
	    end
	    else Exit;
	  end
	  else begin
	    if boolean(IsConventional) then
	      Result := ConvDecPassphraseDialog(FContext, FPassphrase, FPassDlgPrompt, FParentHandle)
	    else with FRecipientsData do begin
	      Result := DecryptionPassphraseDialog(FContext, RecipientSet, KeyCount, KeyIDArray,
						   FDecryptionKey, FPassphrase, true, FPassDlgPrompt, FParentHandle);
	    end;
	    if Result <> 0 then Exit;
	  end;
	end;
      end
      else FPassSection := FSectionNumber;
    finally
      KeyPropsList.Free;
    end;
  end;
  Result := PGPAddJobOptions(Event^.Job, PGPOPassphrase(FContext, FPassphrase), PGPOLastOption(FContext));
end;

function TPGPDecodeCustom.KeyFoundHandler(Event: pPGPEvent): PGPError;
var
  KeySetSelected: pPGPKeySet;
begin
  Result := 0; // don't cancel decoding whatever happens here
  with Event^.EData.KeyFoundData do begin
    if PGP7X then begin
      try
	if FQueryAddKeys then begin
	  KeySetSelected := PGPPeekKeyDBRootKeySet(KeysFound);
	  Result := GetExclusiveKeySet(KeySetSelected, FKeySetMain, FContext, FIgnoreKnownFlag);
	  if Result <> 0 then Exit;
	  if FKeyDlgPrompt <> '' then begin
	    SelectAddKeysToKeyRing(FContext, PGPPeekKeyDBRootKeySet(KeysFound), FKeySetMain,
				   FKeyPropsList, PChar(FKeyDlgPrompt),
				   false, spgpKeyPropFlag_IDFlags,
				   FParentHandle);
	  end
	  else AddKeysToKeyRing(FContext, FKeySetMain, KeysFound, FKeyPropsList, spgpKeyPropFlag_IDFlags);
	end;
      finally
	PGPFreeKeyDB(PGPPeekKeySetKeyDB(KeysFound));
      end;
    end
    else begin
      try
	if FQueryAddKeys then begin
	  Result := GetExclusiveKeySet(KeysFound, FKeySetMain, FContext, FIgnoreKnownFlag);
	  if Result <> 0 then Exit;
	  if FKeyDlgPrompt <> '' then begin
	    SelectAddKeysToKeyRing(FContext, KeysFound, FKeySetMain,
				   FKeyPropsList, PChar(FKeyDlgPrompt),
				   false, spgpKeyPropFlag_IDFlags,
				   FParentHandle);
	  end
	  else AddKeysToKeyRing(FContext, FKeySetMain, KeysFound, FKeyPropsList, spgpKeyPropFlag_IDFlags);
	end;
      finally
	PGPFreeKeySet(KeysFound);
      end;
    end;
  end;
end;

// causes trouble when used with FPassThroughClearSigned on clear signed input
// causes trouble with FPassThroughUnrecognized and local output options
function TPGPDecodeCustom.OutputHandler(Event: pPGPEvent): PGPError;
var
  FileSpec	: pPGPFileSpec;
begin
  Result := 0;
  with Event^.EData.OutputData do begin
    if (ForYourEyesOnly = PGPTrue) and not FFileToFile then FEyesOnly := true;
    if FOutputExists then begin
      if not FFileToFile then Result := SecureMove;
    end
    else if FFileOutput and not FEyesOnly then begin
      if (FOutputFileName = '') and (SuggestedName <> '') then begin
	FOutputFileName := ExtractFilePath(FInputFileName) + SuggestedName;
      end;
      if GetOutputFileName = '' then begin
	Result := kPGPError_CantOpenFile;
	Exit;
      end;
      if FFileToFile then begin
	FileSpec := nil;
	Result := PGPNewFileSpecFromFullPath(FContext, PChar(FOutputFileName), FileSpec);
	try
	  if Result = 0 then begin
	    Result := PGPBuildOptionList(FContext, FOutputOptions,
	      [
		PGPOOutputFile(FContext, FileSpec),
		PGPOAppendOutput(FContext, PGPTrue)
	      ]);
	  end;
	finally
	  PGPFreeFileSpec(FileSpec);
	end;
      end
      else begin
	Result := PGPBuildOptionList(FContext, FOutputOptions,
	  [
	    PGPOAllocatedOutputBuffer(FContext, FAllocatedOutputBuffer, $FFFFFFFF, FActualOutputSize)
	  ]);
      end;
    end
    else begin
      FFileToFile := false;
      Result := PGPBuildOptionList(FContext, FOutputOptions,
	[
	  PGPOAllocatedOutputBuffer(FContext, FAllocatedOutputBuffer, $FFFFFFFF, FActualOutputSize)
	]);
    end;
  end;
  if Result = 0 then begin
    Result := PGPAddJobOptions(Event^.Job, FOutputOptions, PGPOLastOption(FContext));
    FOutputExists := true;
  end;
end;

function TPGPDecodeCustom.SignatureHandler(Event: pPGPEvent): PGPError;
var
  SignKeyType	: PGPInt32;
  PGPSignKeyID	: TPGPKeyID7;
  SignKeyID	: TKeyID;
  KeysFound	: Pointer;
begin
  Result := 0; // don't cancel decoding whatever happens here
  ClearSigPropsRec;
  if PGP7X then with Event^.EData.SignatureData.PGPEventSignatureData7, FSigPropsRec do begin
    if PGP7X and (PGPGetKeyDBObjNumericProperty(SigningKey, kPGPKeyDBObjProperty_ObjectType, SignKeyType) = 0) then begin
      if SignKeyType = kPGPKeyDBObjType_SubKey then SigningKey := PGPPeekKeyDBObjKey(SigningKey);
    end;
    if boolean(Checked) then begin
      sHexID := GetKeyPropKeyID(SigningKey);
      sUserID := GetKeyPropUserID(SigningKey);
      sCreaTimeNum := PGPGetStdTimeFromPGPTime(CreationTime);
      sCreaTimeStr := UnixTimeToLocalTimeStr(sCreaTimeNum);
      sChecked := boolean(Checked);
      sVerified := boolean(Verified);
      sValidity := TValidityLevel(KeyValidity);
      sRevoked := boolean(KeyRevoked);
      sDisabled := boolean(KeyDisabled);
      sExpired := boolean(KeyExpired);
      sAxiomatic := GetKeyPropIsAxiomatic(SigningKey);
      if PGP8X and (ExpirationPeriod <> 0) then begin
	sExpTimeNum := sCreaTimeNum + ExpirationPeriod;
	sExpTimeStr := UnixTimeToLocalTimeStr(sExpTimeNum);
      end;
      if GetKeyPropIsSigningKey(SigningKey) then begin
	sStatus := TSigStatus(Checked + Verified);
	if PGP8X and (sStatus = SIGNED_GOOD) and (ExpirationPeriod <> 0)
	and (sExpTimeNum < SystemTimeToUnixTimeNum) then begin
	  sStatus := SIGNED_SIG_EXPIRED;
	end;
      end
      else sStatus := SIGNED_UNKNOWN_ALG;
    end
    else begin
      Move(SigningKeyID, PGPSignKeyID, SizeOf(TPGPKeyID7));
      Result := PGPGetKeyIDString(PGPSignKeyID, kPGPKeyIDString_Full, SignKeyID);
      if Result = 0 then begin
	sCreaTimeNum := PGPGetStdTimeFromPGPTime(CreationTime);
	sCreaTimeStr := UnixTimeToLocalTimeStr(sCreaTimeNum);
	sHexID := SignKeyID;
      end;
      sStatus := SIGNED_NO_KEY;
      if (Result = 0) and FGetKeyFromServer then begin
	KeysFound := nil;
	Result := KeyServerDialog(FContext, FKeySetMain, SignKeyID, pPGPKeyDB(KeysFound), FParentHandle);
	if (Result = 0) and (KeysFound <> nil) then begin
	  try
	    if FKeyDlgPrompt <> '' then begin
	      SelectAddKeysToKeyRing(FContext, PGPPeekKeyDBRootKeySet(pPGPKeyDB(KeysFound)), FKeySetMain,
				     FKeyPropsList, PChar(FKeyDlgPrompt),
				     false, spgpKeyPropFlag_IDFlags,
				     FParentHandle);
	    end;
	    PGPAddJobOptions(Event^.Job,
			     PGPOKeySetRef(FContext, PGPPeekKeyDBRootKeySet(pPGPKeyDB(KeysFound))),
			     PGPOLastOption(FContext));
	  finally
	    PGPFreeKeyDB(pPGPKeyDB(KeysFound));
	  end;
	end;
      end;
      Result := 0;
    end;
    sDetached := FDetachedSigVerify;
  end
  else with Event^.EData.SignatureData.PGPEventSignatureData6, FSigPropsRec do begin
    if boolean(Checked) then begin
      sHexID := GetKeyPropKeyID(SigningKey);
      sUserID := GetKeyPropUserID(SigningKey);
      sCreaTimeNum := PGPGetStdTimeFromPGPTime(CreationTime);
      sCreaTimeStr := UnixTimeToLocalTimeStr(sCreaTimeNum);
      sChecked := boolean(Checked);
      sVerified := boolean(Verified);
      sValidity := TValidityLevel(KeyValidity);
      sRevoked := boolean(KeyRevoked);
      sDisabled := boolean(KeyDisabled);
      sExpired := boolean(KeyExpired);
      sAxiomatic := GetKeyPropIsAxiomatic(SigningKey);
      if GetKeyPropIsSigningKey(SigningKey) then
	sStatus := TSigStatus(Checked + Verified)
      else sStatus := SIGNED_UNKNOWN_ALG;
    end
    else begin
      Move(SigningKeyID, PGPSignKeyID, SizeOf(TPGPKeyID6));
      Result := PGPGetKeyIDString(PGPSignKeyID, kPGPKeyIDString_Full, SignKeyID);
      if Result = 0 then begin
	sCreaTimeNum := PGPGetStdTimeFromPGPTime(CreationTime);
	sCreaTimeStr := UnixTimeToLocalTimeStr(sCreaTimeNum);
	sHexID := SignKeyID;
      end;
      sStatus := SIGNED_NO_KEY;
      if (Result = 0) and FGetKeyFromServer then begin
	KeysFound := nil;
	Result := KeyServerDialog(FContext, FKeySetMain, SignKeyID, pPGPKeySet(KeysFound), FParentHandle);
	if (Result = 0) and (KeysFound <> nil) then begin
	  try
	    if FKeyDlgPrompt <> '' then begin
	      SelectAddKeysToKeyRing(FContext, pPGPKeySet(KeysFound), FKeySetMain,
				     FKeyPropsList, PChar(FKeyDlgPrompt),
				     false, spgpKeyPropFlag_IDFlags,
				     FParentHandle);
	    end;
	    PGPAddJobOptions(Event^.Job,
			     PGPOKeySetRef(FContext, pPGPKeySet(KeysFound)),
			     PGPOLastOption(FContext));
	  finally
	    PGPFreeKeySet(pPGPKeySet(KeysFound));
	  end;
	end;
      end;
      Result := 0;
    end;
    sDetached := FDetachedSigVerify;
  end;
  if FShowSigLog then ShowSigInfo;
  if Assigned(FOnGetSignature) then FOnGetSignature(SigPropsRec);
end;

function TPGPDecodeCustom.EndLexHandler(Event: pPGPEvent): PGPError;
begin
  Result := 0;
  with FRecipientsData do begin
    PGPFreeData(KeyIDArray);
    PGPFreeKeySet(RecipientSet);
  end;
  FillChar(FRecipientsData, SizeOf(TPGPEventRecipientsData), 0);
end;

end.

