{$J+,Z4}
unit KeyFuncs;

{------------------------------------------------------------------------------}
{                                                                              }
{     This unit is partly based on Steve Heller's spgpKeyUtil.pas from his     }
{         SPGP sources available from http://www.oz.net/~srheller/spgp/        }
{                                                                              }
{                Portions created by Michael in der Wiesche are                }
{              Copyright (C) 2001-2005 by Michael in der Wiesche               }
{                                                                              }
{------------------------------------------------------------------------------}

interface

uses
  Windows,
  Classes,
  SysUtils,
  TimeUtils,
  KeyPropTypes,
  X509Types,
  UTF8,
  pgpBase,
  pgpErrors,
  pgpPubTypes,
  pgpUtilities,
  pgpOptionList,
  pgpMemoryMgr,
  pgpGroups,
  pgpKeys,
  pgpCl;

type
  TKeyRings = Class
  private
    InitCount: Longint;
    Recursion: Longbool;
    RingKeyDB: pPGPKeyDB;
    RingKeySet: pPGPKeySet;
    RingContext: pPGPContext;
  public
    PubringFile: String;
    SecringFile: String;
    GroupsFile: String;
    function UpdateKeyRings: PGPError;
    function SetKeyRings(const Pubring, Secring: String): PGPError;
    function InitKeyRings(var Context: pPGPContext; var KeySetMain: pPGPKeySet): PGPError;
    procedure FreeKeyRings;
  end;

{ "public" functions }
function IsHexID(const ToCheck: String): Longbool;
function ShortHexID(const HexID: String): String;
function KeyRemove(KeyIDCommaText: PChar): Integer;
function KeyEnable(HexKeyID: PChar): Integer;
function KeyDisable(HexKeyID: PChar): Integer;
function KeyRevoke(HexKeyID, Passphrase: PChar): Integer;
function ChangePassphrase(HexKeyID, OldPassphrase, NewPassphrase: PChar): Integer;
// properties, key & user IDs for all keys matching the specified KeyIDCommaText
// if KeyIDCommaText = '' then all keys will be checked
// - returns number of keys found or error
function FindKeyProps(const KeyIDCommaText: String;
		      var KeyPropsList: TKeyPropsList;
		      PropertyFlags, FilterFlags: DWord;
		      KeyOrder: TPGPKeyOrdering): Integer;
{ "private" functions }
function SystemTimeToUnixTimeNum: Integer;
function UnixTimeToLocalTimeStr(UnixSeconds: Integer): String;
function CountUserIDs(KeySet: pPGPKeySet): Integer;
function GetSubKeyPropKeyID(SubKey: pPGPKey): String;
function GetSubKeyPropIsRevoked(SubKey: pPGPKey): Longbool;
function GetSubKeyPropIsExpired(SubKey: pPGPKey): Longbool;
function GetKeyPropKeyID(Key: pPGPKey): String;
function GetKeyPropUserID(Key: pPGPKey): String;
function GetKeyPropFingerprint(Key: pPGPKey): String;
function GetKeyPropKeyCreationTimeStr(Key: pPGPKey): String;
function GetKeyPropKeyExpirationTimeStr(Key: pPGPKey): String;
function GetKeyPropKeyBits(Key: pPGPKey): String;
function GetKeyPropX509Cert(KeySet: pPGPKeySet; Key: pPGPKey): TX509Type;
function GetKeyPropKeyAlg(Key: pPGPKey): TKeyAlgorithm;
function GetKeyPropHashAlg(Key: pPGPKey): THashAlgorithm;
function GetKeyPropLegacy(Key: pPGPKey): Longbool;
function GetKeyPropTrust(Key: pPGPKey): TTrustLevel;
function GetKeyPropValidity(Key: pPGPKey): TValidityLevel;
function GetKeyPropKeyCreationTime(Key: pPGPKey): Integer;
function GetKeyPropKeyExpirationTime(Key: pPGPKey): Integer;
function GetKeyPropIsCorrupt(Key: pPGPKey): Longbool;
function GetKeyPropIsSecret(Key: pPGPKey): Longbool;
function GetKeyPropIsAxiomatic(Key: pPGPKey): Longbool;
function GetKeyPropIsSigningKey(Key: pPGPKey): Longbool;
function GetKeyPropIsRevoked(Key: pPGPKey): Longbool;
function GetKeyPropIsDisabled(Key: pPGPKey): Longbool;
function GetKeyPropIsExpired(Key: pPGPKey): Longbool;
function GetKeyPropIsSecretShared(Key: pPGPKey): Longbool;
function GetKeyPropCanEncrypt(Key: pPGPKey): Longbool;
function GetKeyPropCanDecrypt(Key: pPGPKey): Longbool;
function GetKeyPropCanSign(Key: pPGPKey): Longbool;
function GetKeyPropCanVerify(Key: pPGPKey): Longbool;
function GetKeyPropCanCertify(KeySet: pPGPKeySet; Key: pPGPKey): Longbool;
function GetKeyPropHasRevoker(KeySet: pPGPKeySet; Key: pPGPKey; KeyPropsRec: TKeyPropsRec): Longbool;
function GetKeyPropHasARR(KeySet: pPGPKeySet; Key: pPGPKey; KeyPropsRec: TKeyPropsRec): TADKType;
function GetKeyPropHasSubKey(KeySet: pPGPKeySet; Key: pPGPKey): Longbool;
function GetKeyFromKeySet(KeySet: pPGPKeySet; var Key: pPGPKey): PGPError;
function HasKeyUserID(Key: pPGPKey; const UserID: String): Longbool;
function GetKeyByHexID(KeySet: pPGPKeySet; const HexID: String; var Key: pPGPKey): PGPError;
function GetKeyFromNewSet(Key: pPGPKey; NewSet: pPGPKeySet; var NewKey: pPGPKey): PGPError;
function RemoveKeyFromKeySet(Key: pPGPKey; TargetKeySet: pPGPKeySet): PGPError;
function AddKeyToKeySet(Key: pPGPKey; TargetKeySet: pPGPKeySet): PGPError;
function GetKeyIterAnyOrdering(KeySet: pPGPKeySet; var KeyList: pPGPKeyList; var KeyIter: pPGPKeyIter): PGPError;
function GetSingleKeyDBObjIter(KeyDBObj: pPGPKeyDBObj; var ObjSet: pPGPKeySet;
			       var ObjList: pPGPKeyList; var ObjIter: pPGPKeyIter): Integer;
function GetKeyFilter(Context: pPGPContext; FilterFlags: DWord; var KeyFilter: pPGPFilter): PGPError;
function GetHexIDFilter(Context: pPGPContext; const HexID: String;
			IncludeSubKeys: Longbool; var HexIDFilter: pPGPFilter): PGPError;
function GetUserIDFilter(Context: pPGPContext; const UserID: String;
			 IncludeUTF8: Longbool; var UserIDFilter: pPGPFilter): PGPError;
function GetKeyExcludeFilter(Context: pPGPContext; Key: pPGPKeyDBObj): pPGPFilter;
function GetKeySetWithoutSigner(Context: pPGPContext; const KeySetToFilter: pPGPKeySet; SignKey: pPGPKeyDBObj;
				var KeySetFiltered: pPGPKeySet): PGPError;
function GetKeyFilterByAnyID(Context: pPGPContext; const AnyID: String;
			     IncludeSubKeys, IncludeUTF8: Longbool;
			     var KeyFilter: pPGPFilter): PGPError;
function GetKeySetByAnyIDs(Context: pPGPContext;
			   KeySetMain: pPGPKeySet;
			   const KeyIDCommaText: String;
			   var KeySetFound: pPGPKeySet): PGPError;
function GetExclusiveKeySet(var KeySetToCheck: pPGPKeySet; KeySetMain: pPGPKeySet;
			    Context: pPGPContext; IgnoreKnownFlag: Integer): PGPError;
function GetKeyProps(KeySet: pPGPKeySet; Key: pPGPKey; Flags: DWord; var KeyPropsRec: TKeyPropsRec): PGPError;
function GetKeySetProps(Context: pPGPContext; KeySet: pPGPKeySet;
			var KeyPropsList: TKeyPropsList;
			PropertyFlags, FilterFlags: DWord;
			KeyOrder: TPGPKeyOrdering): Integer; // - returns number of keys found or error
function AddKeysToKeyRing(Context: pPGPContext; KeySetMain: pPGPKeySet;
			  KeysToImport: Pointer; var KeyPropsList: TKeyPropsList;
			  PropertyFlags: DWord): Integer; // - returns number of keys added or error
function GetHexIDByAnyID(Context: pPGPContext; KeySetMain: pPGPKeySet;
			 const AnyID: String; var HexID: String): Integer; // - returns number of keys found or error
function GetMatchingKey(Context: pPGPContext; KeySet: pPGPKeySet; Passphrase: PChar): pPGPKey;			 
function PassphraseIsValid(Context: pPGPContext; KeySetMain: pPGPKeySet; AnyID, Passphrase: PChar): Longbool;
function GetCachedPassphrase(Context: pPGPContext; var Key: pPGPKey; KeySet: pPGPKeySet; var Passphrase: PChar): Longbool;
function CachePassphrase(Context: pPGPContext; Key: pPGPKey; TimeInterval: PGPTime;
			 ShareCache: Longbool; var Passphrase: PChar): Longbool;
function IsPassphraseCacheEmpty(Context: pPGPContext): Longbool;
function PurgePassphraseCache(Context: pPGPContext): Longbool;

const KeyRings: TKeyRings = nil;

implementation

uses PrefFuncs;

function IsHexID(const ToCheck: String): Longbool; register; assembler;
asm	// EAX = @ToCheck
  OR	EAX,EAX
  JE	@FALSE
  MOV	ECX,[EAX - 4]
  CMP	ECX,LongHexIDLen
  JE	@START
  CMP	ECX,ShortHexIDLen
  JNE	@FALSE
  @START:
  MOV	DX,[EAX]
  AND	DH,0DFh
  CMP	DX,'X0'
  JNE	@FALSE
  SUB	ECX,2
  ADD	EAX,2
  @LOOP:
  DEC	ECX
  JS	@TRUE
  MOV	DL,[EAX + ECX]
  CMP	DL,'0'
  JB	@FALSE
  CMP	DL,'9'
  JBE	@LOOP
  AND	DL,0DFh
  CMP	DL,'A'
  JB	@FALSE
  CMP	DL,'F'
  JBE	@LOOP
  @FALSE:
  XOR	EAX,EAX
  RET
  @TRUE:
  MOV	EAX,true
end;

function ShortHexID(const HexID: String): String;
begin
  Result := '';
  if IsHexID(HexID) then begin
    case Length(HexID) of
      ShortHexIDLen: Result := HexID;
      LongHexIDLen: Result := '0x' + Copy(HexID, ShortHexIDLen + 1, ShortHexIDLen - 2);
    end;
  end;
end;

function KeyRemove(KeyIDCommaText: PChar): Integer;
var
  Context	: pPGPContext;
  KeySetMain	: pPGPKeySet;
  KeySetFound	: pPGPKeySet;
  KeyCount	: PGPUInt32;
begin
  KeySetFound := nil;
  Result := KeyRings.InitKeyRings(Context, KeySetMain);
  if Result <> 0 then Exit;
  try
    Result := GetKeySetByAnyIDs(Context, KeySetMain, KeyIDCommaText, KeySetFound);
    if Result <> 0 then Exit;
    try
      Result := PGPCountKeys(KeySetFound, KeyCount);
      if Result <> 0 then Exit;
      if KeyCount <> 0 then begin
	Result := PGPRemoveKeys(KeySetFound, KeySetMain);
	if Result <> 0 then Exit;
      end;
    finally
      PGPFreeKeySet(KeySetFound);
    end;
    Result := KeyRings.UpdateKeyRings;
  finally
    KeyRings.FreeKeyRings;
  end;
end;

function KeyEnable(HexKeyID: PChar): Integer;
var
  Context	: pPGPContext;
  KeySetMain	: pPGPKeySet;
  Key		: pPGPKey;
begin
  Result := KeyRings.InitKeyRings(Context, KeySetMain);
  if Result <> 0 then Exit;
  try
    Result := GetKeyByHexID(KeySetMain, HexKeyID, Key);
    if Result <> 0 then Exit;
    Result := PGPEnableKey(Key);
    if Result <> 0 then Exit;
    Result := KeyRings.UpdateKeyRings;
  finally
    KeyRings.FreeKeyRings;
  end;
end;

function KeyDisable(HexKeyID: PChar): Integer;
var
  Context	: pPGPContext;
  KeySetMain	: pPGPKeySet;
  Key		: pPGPKey;
begin
  Result := KeyRings.InitKeyRings(Context, KeySetMain);
  if Result <> 0 then Exit;
  try
    Result := GetKeyByHexID(KeySetMain, HexKeyID, Key);
    if Result <> 0 then Exit;
    Result := PGPDisableKey(Key);
    if Result <> 0 then Exit;
    Result := KeyRings.UpdateKeyRings;
  finally
    KeyRings.FreeKeyRings;
  end;
end;

function KeyRevoke(HexKeyID, Passphrase: PChar): Integer;
var
  Context	: pPGPContext;
  KeySetMain	: pPGPKeySet;
  Key		: pPGPKey;
begin
  Result := KeyRings.InitKeyRings(Context, KeySetMain);
  if Result <> 0 then Exit;
  try
    Result := GetKeyByHexID(KeySetMain, HexKeyID, Key);
    if Result <> 0 then Exit;
    Result := PGPRevokeKey(Key, PGPOPassphrase(Context, Passphrase), PGPOLastOption(Context));
    if Result <> 0 then Exit;
    Result := KeyRings.UpdateKeyRings;
  finally
    KeyRings.FreeKeyRings;
  end;
end;

function ChangePassphrase(HexKeyID, OldPassphrase, NewPassphrase: PChar): Integer;
var
  Context	: pPGPContext;
  KeySetMain	: pPGPKeySet;
  OptionList	: pPGPOptionList;
  Key		: pPGPKey;
  KeyList	: pPGPKeyList;
  KeyIter	: pPGPKeyIter;
  SubKey	: pPGPSubKey;
begin
  Result := KeyRings.InitKeyRings(Context, KeySetMain);
  if Result <> 0 then Exit;
  OptionList := nil;
  try
    Result := PGPBuildOptionList(Context, OptionList,
      [
	PGPOPassphrase(Context, OldPassphrase),
	PGPOPassphrase(Context, NewPassphrase)
      ]);
    if Result <> 0 then Exit;
    try
      Result := GetKeyByHexID(KeySetMain, HexKeyID, Key);
      if Result <> 0 then Exit;
      Result := PGPChangePassphrase(Key, OptionList, PGPOLastOption(Context));
      if Result <> 0 then Exit;
      if not GetKeyPropLegacy(Key) then begin
	Result := GetKeyIterAnyOrdering(KeySetMain, KeyList, KeyIter);
	if Result <> 0 then Exit;
	try
	  PGPKeyIterSeek(KeyIter, Key);
	  while PGPKeyIterNextKeyDBObj(KeyIter, kPGPKeyDBObjType_SubKey, SubKey) = 0 do begin
	    Result := PGPChangeSubKeyPassphrase(SubKey, OptionList, PGPOLastOption(Context));
	    if Result <> 0 then Exit;
	  end;
	finally
	  PGPFreeKeyIter(KeyIter);
	  PGPFreeKeyList(KeyList);
	end;
      end;
      Result := KeyRings.UpdateKeyRings;
    finally
      PGPFreeOptionList(OptionList);
    end;
  finally
    KeyRings.FreeKeyRings;
  end;
end;

function FindKeyProps(const KeyIDCommaText: String;
		      var KeyPropsList: TKeyPropsList;
		      PropertyFlags, FilterFlags: DWord;
		      KeyOrder: TPGPKeyOrdering): Integer;
var
  Context	: pPGPContext;
  KeySetMain	: pPGPKeySet;
  KeySetFound	: pPGPKeySet;
  KeyFilter	: pPGPFilter;
  KeySetFiltered: pPGPKeySet;
  KeyCount	: PGPUInt32;
begin
  KeySetFound := nil;
  KeyFilter := nil;
  KeySetFiltered := nil;
  Result := KeyRings.InitKeyRings(Context, KeySetMain);
  if Result <> 0 then Exit;
  try
    if KeyIDCommaText = '' then
      KeySetFound := KeySetMain // all keys on key rings
    else Result := GetKeySetByAnyIDs(Context, KeySetMain, KeyIDCommaText, KeySetFound); // key(s) matching KeyData
    if Result <> 0 then Exit;
    try
      if PropertyFlags <> 0 then begin
	Result := GetKeySetProps(Context, KeySetFound,
				 KeyPropsList, PropertyFlags,
				 FilterFlags, KeyOrder);
      end
      else begin
	if FilterFlags <> KeyFilterFlag_AllKeys then begin
	  if FilterFlags and KeyFilterFlag_CountUserIDs <> 0 then
	    Result := CountUserIDs(KeySetFound)
	  else begin
	    Result := GetKeyFilter(Context, FilterFlags, KeyFilter);
	    try
	      if Result <> 0 then Exit;
	      Result := PGPFilterKeySet(KeySetFound, KeyFilter, KeySetFiltered);
	    finally
	      PGPFreeFilter(KeyFilter);
	    end;
	  end;
	end
	else KeySetFiltered := KeySetFound;
	if Result <> 0 then Exit;
	try
	  Result := PGPCountKeys(KeySetFiltered, KeyCount);
	  if Result = 0 then Result := KeyCount;
	finally
	  if KeySetFiltered <> KeySetFound then PGPFreeKeySet(KeySetFiltered);
	end;
      end;
    finally
      if KeySetFound <> KeySetMain then PGPFreeKeySet(KeySetFound);
    end;
  finally
    KeyRings.FreeKeyRings;
  end;
end;

function GetKeyFromKeySet(KeySet: pPGPKeySet; var Key: pPGPKey): PGPError;
var
  KeyList	: pPGPKeyList;
  KeyIter	: pPGPKeyIter;
begin
  Key := nil;
  Result := GetKeyIterAnyOrdering(KeySet, KeyList, KeyIter);
  if Result <> 0 then Exit;
  try
    Result := PGPKeyIterNextKeyDBObj(KeyIter, kPGPKeyDBObjType_Key, Key);
  finally
    PGPFreeKeyIter(KeyIter);
    PGPFreeKeyList(KeyList);
  end;
end;

function HasKeyUserID(Key: pPGPKey; const UserID: String): Longbool;
var
  KeySet	: pPGPKeySet;
  KeyFilter	: pPGPFilter;
  Context	: pPGPContext;
  KeySetFiltered: pPGPKeySet;
  KeyCount	: PGPUInt32;
begin
  Result := false;
  KeySet := nil;
  KeyFilter := nil;
  Context := PGPPeekKeyDBObjContext(Key);
  if PGPNewSingletonKeySet(Key, KeySet) <> 0 then Exit;
  try
    if GetUserIDFilter(Context, UserID, true, KeyFilter) <> 0 then Exit;
    try
      if PGPFilterKeySet(KeySet, KeyFilter, KeySetFiltered) <> 0 then Exit;
      try
	Result := (PGPCountKeys(KeySetFiltered, KeyCount) = 0) and (KeyCount <> 0);
      finally
	PGPFreeKeySet(KeySetFiltered);
      end;
    finally
      PGPFreeFilter(KeyFilter);
    end;
  finally
    PGPFreeKeySet(KeySet);
  end;
end;

function GetKeyByHexID(KeySet: pPGPKeySet; const HexID: String; var Key: pPGPKey): PGPError;
var
  PGPKeyID	: TPGPKeyID7;
begin
  Key := nil;
  Result := PGPGetKeyIDFromString(PChar(HexID), kPGPPublicKeyAlgorithm_Invalid, PGPKeyID);
  if Result = 0 then Result := PGPGetKeyByKeyID(KeySet, PGPKeyID, kPGPPublicKeyAlgorithm_Invalid, Key);
end;

function GetKeyFromNewSet(Key: pPGPKey; NewSet: pPGPKeySet; var NewKey: pPGPKey): PGPError;
var
  PGPKeyID	: TPGPKeyID7;
begin
  NewKey := nil;
  Result := PGPGetKeyIDFromKey(Key, PGPKeyID);
  if Result = 0 then Result := PGPGetKeyByKeyID(NewSet, PGPKeyID, kPGPPublicKeyAlgorithm_Invalid, NewKey);
end;

function RemoveKeyFromKeySet(Key: pPGPKey; TargetKeySet: pPGPKeySet): PGPError;
var
  SingleKeySet	: pPGPKeySet;
begin
  Result := PGPNewSingletonKeySet(Key, SingleKeySet);
  try
    if Result = 0 then Result := PGPRemoveKeys(SingleKeySet, TargetKeySet);
  finally
    PGPFreeKeySet(SingleKeySet);
  end;
end;

function AddKeyToKeySet(Key: pPGPKey; TargetKeySet: pPGPKeySet): PGPError;
var
  SingleKeySet	: pPGPKeySet;
begin
  Result := PGPNewSingletonKeySet(Key, SingleKeySet);
  try
    if Result = 0 then begin
      if PGP7X then
	Result := PGPCopyKeys(SingleKeySet, PGPPeekKeySetKeyDB(TargetKeySet), TargetKeySet)
      else Result := PGPAddKeys(SingleKeySet, TargetKeySet);
    end;
  finally
    PGPFreeKeySet(SingleKeySet);
  end;
end;

function GetKeyIterAnyOrdering(KeySet: pPGPKeySet; var KeyList: pPGPKeyList; var KeyIter: pPGPKeyIter): PGPError;
begin
  KeyList := nil;
  KeyIter := nil;
  if PGP7X then
    Result := PGPNewKeyIterFromKeySet(KeySet, KeyIter)
  else begin
    Result := PGPOrderKeySet(KeySet, kPGPKeyOrdering_Any, PGPFalse, KeyList);
    if Result = 0 then begin
      Result := PGPNewKeyIter(KeyList, KeyIter);
      if Result <> 0 then PGPFreeKeyList(KeyList);
    end;
  end;
end;

function GetSingleKeyDBObjIter(KeyDBObj: pPGPKeyDBObj; var ObjSet: pPGPKeySet;
			       var ObjList: pPGPKeyList; var ObjIter: pPGPKeyIter): Integer;
begin
  ObjSet := nil;
  ObjList := nil;
  ObjIter := nil;
  Result := PGPNewSingletonKeySet(KeyDBObj, ObjSet);
  if Result = 0 then Result := GetKeyIterAnyOrdering(ObjSet, ObjList, ObjIter);
end;

function GetKeyFilter(Context: pPGPContext; FilterFlags: DWord; var KeyFilter: pPGPFilter): PGPError;
var
  BoolFilter	: pPGPFilter;
  AlgoFilter	: pPGPFilter;
begin
  Result := 0;
  KeyFilter := nil;
  BoolFilter := nil;
  AlgoFilter := nil;
  try
    if FilterFlags and KeyFilterMask_Boolean <> 0 then begin
      if FilterFlags and KeyFilterFlag_CanEncrypt <> 0 then begin
	Result := PGPNewKeyDBObjBooleanFilter(Context, kPGPKeyProperty_CanEncrypt, PGPTrue, BoolFilter);
      end
      else if FilterFlags and KeyFilterFlag_CanDecrypt <> 0 then begin
	Result := PGPNewKeyDBObjBooleanFilter(Context, kPGPKeyProperty_CanDecrypt, PGPTrue, BoolFilter);
      end
      else if FilterFlags and KeyFilterFlag_CanSign <> 0 then begin
	Result := PGPNewKeyDBObjBooleanFilter(Context, kPGPKeyProperty_CanSign, PGPTrue, BoolFilter);
      end
      else if FilterFlags and KeyFilterFlag_CanVerify <> 0 then begin
	Result := PGPNewKeyDBObjBooleanFilter(Context, kPGPKeyProperty_CanVerify, PGPTrue, BoolFilter);
      end
      else if FilterFlags and KeyFilterFlag_Enabled <> 0 then begin
	Result := PGPNewKeyDBObjBooleanFilter(Context, kPGPKeyProperty_IsDisabled, PGPFalse, BoolFilter);
      end
      else if FilterFlags and KeyFilterFlag_Disabled <> 0 then begin
	Result := PGPNewKeyDBObjBooleanFilter(Context, kPGPKeyProperty_IsDisabled, PGPTrue, BoolFilter);
      end;
    end;
    if Result <> 0 then Exit;
    if FilterFlags and KeyFilterMask_Algorithm <> 0 then begin
      if FilterFlags and KeyFilterFlag_DHDSS <> 0 then begin
	Result := PGPNewKeyDBObjNumericFilter(Context, kPGPKeyProperty_AlgorithmID,
					      kPGPPublicKeyAlgorithm_DSA, kPGPMatchEqual, AlgoFilter);
      end else if FilterFlags and KeyFilterFlag_RSA <> 0 then begin
	Result := PGPNewKeyDBObjNumericFilter(Context, kPGPKeyProperty_AlgorithmID,
					      kPGPPublicKeyAlgorithm_RSA, kPGPMatchEqual, AlgoFilter);
      end else if FilterFlags and KeyFilterFlag_V4 <> 0 then begin
	if PGP7X then begin
	  Result := PGPNewKeyDBObjNumericFilter(Context, kPGPKeyProperty_Version,
						kPGPKeyVersion_V4, kPGPMatchEqual, AlgoFilter);
	end
	else begin
	  Result := PGPNewKeyDBObjNumericFilter(Context, kPGPKeyProperty_AlgorithmID,
						kPGPPublicKeyAlgorithm_DSA, kPGPMatchEqual, AlgoFilter);
	end;
      end else if FilterFlags and KeyFilterFlag_V3 <> 0 then begin
	if PGP7X then begin
	  Result := PGPNewKeyDBObjNumericFilter(Context, kPGPKeyProperty_Version,
						kPGPKeyVersion_V3, kPGPMatchLessOrEqual, AlgoFilter);
	end
	else begin
	  Result := PGPNewKeyDBObjNumericFilter(Context, kPGPKeyProperty_AlgorithmID,
						kPGPPublicKeyAlgorithm_RSA, kPGPMatchEqual, AlgoFilter);
	end;
      end
      else if (FilterFlags and KeyFilterFlag_X509Cert <> 0) or (FilterFlags and KeyFilterFlag_X509Root <> 0) then begin
	Result := PGPNewKeyDBObjBooleanFilter(Context, kPGPSigProperty_IsX509, PGPTrue, AlgoFilter);
      end;
    end;
    if Result <> 0 then Exit;
    if (BoolFilter <> nil) and (AlgoFilter <> nil) then begin
      Result := PGPIntersectFilters(BoolFilter, AlgoFilter, KeyFilter);
      BoolFilter := nil;
      AlgoFilter := nil;
    end
    else if BoolFilter <> nil then
      KeyFilter := BoolFilter
    else if AlgoFilter <> nil then
      KeyFilter := AlgoFilter;
  finally
    if (KeyFilter <> BoolFilter) and (BoolFilter <> nil) then PGPFreeFilter(BoolFilter);
    if (KeyFilter <> AlgoFilter) and (AlgoFilter <> nil) then PGPFreeFilter(AlgoFilter);
  end;
end;

function GetHexIDFilter(Context: pPGPContext; const HexID: String;
			IncludeSubKeys: Longbool; var HexIDFilter: pPGPFilter): PGPError;
var
  PGPKeyID	: TPGPKeyID7;
  KeyFilter	: pPGPFilter;
  SubKeyFilter	: pPGPFilter;
begin
  HexIDFilter := nil;
  KeyFilter := nil;
  SubKeyFilter := nil;
  Result := PGPGetKeyIDFromString(PChar(HexID), kPGPPublicKeyAlgorithm_Invalid, PGPKeyID);
  if Result <> 0 then Exit;
  Result := PGPNewKeyDBObjDataFilter(Context, kPGPKeyProperty_KeyID, @PGPKeyID,
				     SizeOf(TPGPKeyID7), kPGPMatchEqual, KeyFilter);
  if Result <> 0 then Exit;
  try
    if IncludeSubKeys then begin
      Result := PGPNewKeyDBObjDataFilter(Context, kPGPSubKeyProperty_KeyID, @PGPKeyID,
					 SizeOf(TPGPKeyID7), kPGPMatchEqual, SubKeyFilter);
      if Result <> 0 then Exit;
      try
	Result := PGPUnionFilters(KeyFilter, SubKeyFilter, HexIDFilter);
	KeyFilter := nil;
	SubKeyFilter := nil;
      finally
	if SubKeyFilter <> nil then PGPFreeFilter(SubKeyFilter);
      end;
    end
    else HexIDFilter := KeyFilter;
  finally
    if IncludeSubKeys and (KeyFilter <> nil) then PGPFreeFilter(KeyFilter);
  end;
end;

function GetUserIDFilter(Context: pPGPContext; const UserID: String;
			 IncludeUTF8: Longbool; var UserIDFilter: pPGPFilter): PGPError;
var
  AnsiFilter	: pPGPFilter;
  UTF8Filter	: pPGPFilter;
  UTF8ID	: UTF8String;
begin
  UserIDFilter :=nil;
  AnsiFilter := nil;
  UTF8Filter := nil;
  Result := PGPNewKeyDBObjDataFilter(Context, kPGPUserIDProperty_Name, PChar(UserID),
				     Length(UserID), kPGPMatchSubString, AnsiFilter);
  if Result <> 0 then Exit;
  try
    if PGP8X and IncludeUTF8 then begin
      UTF8ID := AnsiToUtf8(UserID);
      Result := PGPNewKeyDBObjDataFilter(Context, kPGPUserIDProperty_Name, PChar(UTF8ID),
					 Length(UTF8ID), kPGPMatchSubString, UTF8Filter);
      if Result <> 0 then Exit;
      try
	Result := PGPUnionFilters(AnsiFilter, UTF8Filter, UserIDFilter);
	AnsiFilter := nil;
	UTF8Filter := nil;
      finally
	if UTF8Filter <> nil then PGPFreeFilter(UTF8Filter);
      end;
    end
    else UserIDFilter := AnsiFilter;
  finally
    if (UTF8ID <> '') and (AnsiFilter <> nil) then PGPFreeFilter(AnsiFilter);
  end;
end;

function GetKeyExcludeFilter(Context: pPGPContext; Key: pPGPKeyDBObj): pPGPFilter;
var
  KeyID		: TPGPKeyID7;
  KeyFilter	: pPGPFilter;
begin
  Result := nil;
  if PGPGetKeyIDFromKey(Key, KeyID) = 0 then begin
    if PGPNewKeyDBObjDataFilter(Context, kPGPKeyProperty_KeyID, @KeyID,
				SizeOf(TPGPKeyID7), kPGPMatchEqual, KeyFilter) = 0 then begin
      try
	PGPNegateFilter(KeyFilter, Result);
	KeyFilter := nil;
      finally
	if KeyFilter <> nil then PGPFreeFilter(KeyFilter);
      end;
    end;
  end;
end;

function GetKeySetWithoutSigner(Context: pPGPContext; const KeySetToFilter: pPGPKeySet; SignKey: pPGPKeyDBObj;
				var KeySetFiltered: pPGPKeySet): PGPError;
var
  ExcludeFilter	: pPGPFilter;
begin
  KeySetFiltered := nil;
  ExcludeFilter := GetKeyExcludeFilter(Context, SignKey);
  try
    Result := PGPFilterKeySet(KeySetToFilter, ExcludeFilter, KeySetFiltered);
  finally
    PGPFreeFilter(ExcludeFilter);
  end;
  if Result <> 0 then KeySetFiltered := KeySetToFilter;
end;

function GetKeyFilterByAnyID(Context: pPGPContext; const AnyID: String;
			     IncludeSubKeys, IncludeUTF8: Longbool;
			     var KeyFilter: pPGPFilter): PGPError;
begin
  if IsHexID(AnyID) then
    Result := GetHexIDFilter(Context, AnyID, IncludeSubKeys, KeyFilter)
  else Result := GetUserIDFilter(Context, AnyID, IncludeUTF8, KeyFilter);
end;

function GetKeySetByAnyIDs(Context: pPGPContext;
			   KeySetMain: pPGPKeySet;
			   const KeyIDCommaText: String;
			   var KeySetFound: pPGPKeySet): PGPError;
var
  KeyDataList	: TStringList;
  KeyIndex	: PGPUInt32;
  KeyString	: String;
  KeyFilter	: pPGPFilter;
  KeySetFiltered: pPGPKeySet;
  KeyCount	: PGPUInt32;
begin
  KeySetFound := nil;
  if KeyIDCommaText <> '' then begin
    KeyDataList := TStringList.Create;
    KeyFilter := nil;
    KeySetFiltered := nil;
    try
      KeyDataList.CommaText := KeyIDCommaText;
      if PGP7X then
	Result := PGPNewEmptyKeySet(PGPPeekKeySetKeyDB(KeySetMain), KeySetFound)
      else Result := PGPNewKeySet(Context, KeySetFound);
      if Result <> 0 then Exit;
      try
	for KeyIndex := 0 to pred(KeyDataList.Count) do begin
	  KeyString := Trim(KeyDataList[KeyIndex]);
	  if KeyString <> '' then begin
	    Result := GetKeyFilterByAnyID(Context, KeyString, false, true, KeyFilter);
	    if Result <> 0 then Exit;
	    try
	      Result := PGPFilterKeySet(KeySetMain, KeyFilter, KeySetFiltered);
	      if Result <> 0 then Exit;
	      try
		Result := PGPCountKeys(KeySetFiltered, KeyCount);
		if Result <> 0 then Exit;
		if KeyCount <> 0 then begin
		  Result := PGPAddKeys(KeySetFiltered, KeySetFound);
		  if Result <> 0 then Exit;
		end
		else begin
		  Result := kPGPError_PublicKeyNotFound;
		  Exit;
		end;
	      finally
		PGPFreeKeySet(KeySetFiltered);
	      end;
	    finally
	      PGPFreeFilter(KeyFilter);
	    end;
	  end;
	end;
      finally
	if Result <> 0 then begin
	  PGPFreeKeySet(KeySetFound);
	  KeySetFound := nil;
	end;
      end;
    finally
      KeyDataList.Free;
    end;
  end
  else Result := kPGPError_PublicKeyNotFound;
end;

function GetExclusiveKeySet(var KeySetToCheck: pPGPKeySet; KeySetMain: pPGPKeySet;
			    Context: pPGPContext; IgnoreKnownFlag: Integer): PGPError;
var
  KeyListToCheck: pPGPKeyList;
  KeyIterToCheck: pPGPKeyIter;
  KeyToCheck	: pPGPKey;
  KeyIDToCheck	: TKeyID;
  UserIDToCheck	: TUserID;
  KeyCount	: PGPUInt32;
  IDSize	: PGPSize;
  UserIDFound	: pPGPUserID;
  UserIDProp	: PGPBoolean;
  KeyFilter	: pPGPFilter;
  KeySetFound	: pPGPKeySet;
  KeySetToRemove: pPGPKeySet;
begin
  KeyToCheck := nil;
  Result := GetKeyIterAnyOrdering(KeySetToCheck, KeyListToCheck, KeyIterToCheck);
  if Result <> 0 then Exit;
  try
    if (IgnoreKnownFlag and IgnoreFlag_ByHexID) <> 0 then begin
      while PGPKeyIterNextKeyDBObj(KeyIterToCheck, kPGPKeyDBObjType_Key, KeyToCheck) = 0 do begin
	KeyFilter := nil;
	KeySetFound := nil;
	KeySetToRemove := nil;
	try
	  if (PGPGetKeyDBObjDataProperty(KeyToCheck, kPGPKeyProperty_KeyID,
					 @KeyIDToCheck, kPGPMaxKeyIDStringSize, IDSize) = 0)
	  and (PGPNewKeyDBObjDataFilter(Context, kPGPKeyProperty_KeyID, @KeyIDToCheck,
					IDSize, kPGPMatchEqual, KeyFilter) = 0)
	  and (PGPFilterKeySet(KeySetMain, KeyFilter, KeySetFound) = 0)
	  and (PGPCountKeys(KeySetFound, KeyCount) = 0) and (KeyCount <> 0)
	  and (PGPNewSingletonKeySet(KeyToCheck, KeySetToRemove) = 0) then begin
	    PGPRemoveKeys(KeySetToRemove, KeySetToCheck);
	  end;
	finally
	  PGPFreeFilter(KeyFilter);
	  PGPFreeKeySet(KeySetFound);
	  PGPFreeKeySet(KeySetToRemove);
	end;
      end;
      PGPKeyIterRewindKeyDBObj(KeyIterToCheck, kPGPKeyDBObjType_Key);
    end;
    if (PGPCountKeys(KeySetToCheck, KeyCount) = 0) and (KeyCount = 0) then Exit;
    if (IgnoreKnownFlag and IgnoreFlag_ByUserID) <> 0 then begin
      while PGPKeyIterNextKeyDBObj(KeyIterToCheck, kPGPKeyDBObjType_Key, KeyToCheck) = 0 do begin
	while (PGPKeyIterNextKeyDBObj(KeyIterToCheck, kPGPKeyDBObjType_UserID, UserIDFound) = 0)
	and (PGPGetKeyDBObjBooleanProperty(UserIDFound, kPGPUserIDProperty_IsAttribute, UserIDProp) = 0)
	and (UserIDProp = PGPFalse) do begin
	  KeyFilter := nil;
	  KeySetFound := nil;
	  KeySetToRemove := nil;
	  try
	    if (PGPGetKeyDBObjDataProperty(UserIDFound, kPGPUserIDProperty_Name,
					   @UserIDToCheck, SizeOf(TUserID), IDSize) = 0)
	    and (PGPNewKeyDBObjDataFilter(Context, kPGPUserIDProperty_Name, @UserIDToCheck,
					  IDSize, kPGPMatchEqual, KeyFilter) = 0)
	    and (PGPFilterKeySet(KeySetMain, KeyFilter, KeySetFound) = 0)
	    and (PGPCountKeys(KeySetFound, KeyCount) = 0) and (KeyCount <> 0)
	    and (PGPNewSingletonKeySet(KeyToCheck, KeySetToRemove) = 0) then begin
	      PGPRemoveKeys(KeySetToRemove, KeySetToCheck);
	      Break;
	    end;
	  finally
	    PGPFreeFilter(KeyFilter);
	    PGPFreeKeySet(KeySetFound);
	    PGPFreeKeySet(KeySetToRemove);
	  end;
	end;
	if (PGPCountKeys(KeySetToCheck, KeyCount) = 0) and (KeyCount = 0) then Break;
      end;
    end;
  finally
    PGPFreeKeyIter(KeyIterToCheck);
    PGPFreeKeyList(KeyListToCheck);
  end;
end;

function GetFingerprintString(const Fingerprint: TFingerprint; FPSize: PGPSize): String; register; assembler;
asm	// EAX = @Fingerprint, EDX = FPSize, ECX = @Result
  PUSH	EBX
  PUSH	EDI
  PUSH	ESI
  PUSH	EBP
  MOV	ESI,EAX
  MOV	EDI,ECX
  MOV	EAX,EDX
  MOV	EBX,EDX
  SHL	EAX,1
  SHR	EBX,1
  MOV	EBP,EBX
  ADD	EBX,EAX
  INC	EBP
  PUSH	EDX
  MOV	EAX,EDI
  CALL	SYSTEM.@LSTRCLR
  MOV	EAX,EBX
  CALL	SYSTEM.@NEWANSISTRING
  MOV	[EDI],EAX
  MOV	EDI,EAX
  POP	ECX
  SUB	EBX,2
  SHR	ECX,1
  SHR	EBP,1
  DEC	ECX
  @LOOP:
  XOR	EAX,EAX
  MOV	AL,[ESI + ECX * 2 + 1]
  MOV	EDX,EAX
  AND	EAX,0Fh
  SHR	EDX,4
  MOV	AH,BYTE PTR[@HEXCHARS + EAX]
  MOV	AL,BYTE PTR[@HEXCHARS + EDX]
  MOV	[EDI + EBX],AX
  XOR	EAX,EAX
  MOV	AL,[ESI + ECX * 2]
  MOV	EDX,EAX
  AND	EAX,0Fh
  SHR	EDX,4
  MOV	AH,BYTE PTR[@HEXCHARS + EAX]
  MOV	AL,BYTE PTR[@HEXCHARS + EDX]
  MOV	[EDI + EBX - 2],AX
  CMP	EBP,ECX
  JE	@MIDDLE
  SUB	EBX,5
  DEC	ECX
  JS	@END
  MOV	BYTE PTR[EDI + EBX + 2],' '
  JMP	@LOOP
  @MIDDLE:
  SUB	EBX,6
  DEC	ECX
  MOV	WORD PTR[EDI + EBX + 2],'  '
  JMP	@LOOP
  @END:
  POP	EBP
  POP	ESI
  POP	EDI
  POP	EBX
  RET
  @HEXCHARS:
  DB	'0123456789ABCDEF';
end;

function SystemTimeToUnixTimeNum: Integer;
var SystemTime: TSystemTime;
begin
  GetSystemTime(SystemTime);
  Result := SystemTimeToUnixTime(SystemTime);
end;

function UnixTimeToLocalTimeStr(UnixSeconds: Integer): String;
begin
  Result := FormatDateTime('ddd, dd mmm yyyy, tt', UnixTimeToLocalTime(UnixSeconds));
end;

function CountUserIDs(KeySet: pPGPKeySet): Integer;
var
  KeyList	: pPGPKeyList;
  KeyIter	: pPGPKeyIter;
  Key		: pPGPKey;
  UserID	: pPGPUserID;
  UserIDProp	: PGPBoolean;
  UserIDCount	: PGPUInt32;
begin
  UserIDCount := 0;
  Result := GetKeyIterAnyOrdering(KeySet, KeyList, KeyIter);
  if Result <> 0 then Exit;
  try
    while PGPKeyIterNextKeyDBObj(KeyIter, kPGPKeyDBObjType_Key, Key) = 0 do begin
      while PGPKeyIterNextKeyDBObj(KeyIter, kPGPKeyDBObjType_UserID, UserID) = 0 do begin
	Result := PGPGetKeyDBObjBooleanProperty(UserID, kPGPUserIDProperty_IsAttribute, UserIDProp);
	if Result <> 0 then Exit;
	if UserIDProp = PGPFalse then inc(UserIDCount);
      end;
    end;
    Result := UserIDCount;
  finally
    PGPFreeKeyIter(KeyIter);
    PGPFreeKeyList(KeyList);
  end;
end;

function GetSubKeyPropKeyID(SubKey: pPGPKey): String;
var
  PGPKeyID	: TPGPKeyID7;
  KeyID		: TKeyID;
begin
  Result := '';
  if (PGPGetKeyIDFromSubKey(SubKey, PGPKeyID) = 0)
  and (PGPGetKeyIDString(PGPKeyID, kPGPKeyIDString_Full, KeyID) = 0) then begin
    Result := KeyID;
  end;
end;

function GetSubKeyPropIsRevoked(SubKey: pPGPKey): Longbool;
var
  Prop		: PGPBoolean;
begin
  Result := false;
  if PGPGetKeyDBObjBooleanProperty(SubKey, kPGPSubKeyProperty_IsRevoked, Prop) = 0 then begin
    Result := boolean(Prop);
  end;
end;

function GetSubKeyPropIsExpired(SubKey: pPGPKey): Longbool;
var
  Prop		: PGPBoolean;
begin
  Result := false;
  if PGPGetKeyDBObjBooleanProperty(SubKey, kPGPSubKeyProperty_IsExpired, Prop) = 0 then begin
    Result := boolean(Prop);
  end;
end;

function GetKeyPropKeyID(Key: pPGPKey): String;
var
  PGPKeyID	: TPGPKeyID7;
  KeyID		: TKeyID;
begin
  Result := '';
  if (PGPGetKeyIDFromKey(Key, PGPKeyID) = 0)
  and (PGPGetKeyIDString(PGPKeyID, kPGPKeyIDString_Full, KeyID) = 0) then begin
    Result := KeyID;
  end;
end;

function GetKeyPropUserID(Key: pPGPKey): String;
var
  UserIDBuf	: TUserID;
  IDSize	: PGPSize;
begin
  Result := '';
  if PGPGetPrimaryUserIDNameBuffer(Key, UserIDBuf, SizeOf(TUserID), IDSize) = 0 then begin
    if PGP8X then
      Result := Utf8OrAnsi(UserIDBuf)
    else Result := UserIDBuf;
  end;
end;

function GetKeyPropFingerprint(Key: pPGPKey): String;
var
  Fingerprint	: TFingerprint;
  FPSize	: PGPSize;
begin
  Result := '';
  if (PGPGetKeyDBObjDataProperty(Key, kPGPKeyProperty_Fingerprint, @Fingerprint, kPGPmaxFingerprintSize, FPSize) = 0)
  and (FPSize <> 0) then begin
    Result := GetFingerprintString(Fingerprint, FPSize);
  end;
end;

// local time
function GetKeyPropKeyCreationTimeStr(Key: pPGPKey): String;
var
  CreaTime	: PGPTime;
begin
  Result := '';
  if PGPGetKeyDBObjTimeProperty(Key, kPGPKeyProperty_Creation, CreaTime) = 0 then begin
    Result := UnixTimeToLocalTimeStr(PGPGetStdTimeFromPGPTime(CreaTime));
  end;
end;

// local time
function GetKeyPropKeyExpirationTimeStr(Key: pPGPKey): String;
var
  ExpTime	: PGPTime;
  CreaTime	: PGPTime;
begin
  Result := '';
  if (PGPGetKeyDBObjTimeProperty(Key, kPGPKeyProperty_Expiration, ExpTime) = 0)
  and (PGPGetKeyDBObjTimeProperty(Key, kPGPKeyProperty_Creation, CreaTime) = 0) then begin
    if ExpTime > CreaTime then Result := UnixTimeToLocalTimeStr(PGPGetStdTimeFromPGPTime(ExpTime));
  end;
end;

function GetKeyPropKeyBits(Key: pPGPKey): String;
var
  Error		: PGPError;
  KeySet	: pPGPKeySet;
  KeyList	: pPGPKeyList;
  KeyIter	: pPGPKeyIter;
  SubKey	: pPGPSubKey;
  KeyAlg	: PGPInt32;
  KeyBits	: PGPInt32;
  SubKeyBits	: PGPInt32;
begin
  Result := '';
  Error := 0;
  KeySet := nil;
  SubKey := nil;
  try
    if (PGPGetKeyDBObjNumericProperty(Key, kPGPKeyProperty_AlgorithmID, KeyAlg) = 0)
    and (PGPGetKeyDBObjNumericProperty(Key, kPGPKeyProperty_Bits, KeyBits) = 0)
    and (GetSingleKeyDBObjIter(Key, KeySet, KeyList, KeyIter) = 0) then begin
      SubKeyBits := 0;
      PGPKeyIterSeek(KeyIter, Key);
      while (PGPKeyIterNextKeyDBObj(KeyIter, kPGPKeyDBObjType_SubKey, SubKey) = 0) do begin
	if not (GetSubKeyPropIsRevoked(SubKey) or GetSubKeyPropIsExpired(SubKey)) then begin
	  Error := PGPGetKeyDBObjNumericProperty(SubKey, kPGPSubKeyProperty_Bits, SubKeyBits);
	  Break;
	end;
      end;
      if Error = 0 then begin
	if SubKeyBits > 0 then
	  Result := IntToStr(SubKeyBits) + '/' + IntToStr(KeyBits) + ' bits'
	else Result := IntToStr(KeyBits) + ' bits';
      end;
    end;
  finally
    PGPFreeKeyIter(KeyIter);
    PGPFreeKeyList(KeyList);
    PGPFreeKeySet(KeySet);
  end;
end;

function GetKeyPropX509Cert(KeySet: pPGPKeySet; Key: pPGPKey): TX509Type;
var
  SingleKeySet	: pPGPKeySet;
  SingleKeyList : pPGPKeyList;
  SingleKeyIter	: pPGPKeyIter;
  ObjFound	: pPGPKeyDBObj;
  X509Prop	: PGPBoolean;
  INData	: TX509CertData;
  INSize	: PGPSize;
  ONData	: TX509CertData;
  ONSize	: PGPSize;
  SignKey	: pPGPKeyDBObj;
begin
  Result := NoCert;
  if PGP7X then begin
    try
      if GetSingleKeyDBObjIter(Key, SingleKeySet, SingleKeyList, SingleKeyIter) = 0 then begin
	while PGPKeyIterNextKeyDBObj(SingleKeyIter, kPGPKeyDBObjType_Any, ObjFound) = 0 do begin
	  if (Result = NoCert)
	  and (PGPGetKeyDBObjBooleanProperty(ObjFound, kPGPSigProperty_IsX509, X509Prop) = 0)
	  and (X509Prop = PGPTrue) then begin
	    Result := StdCert;
	    if (PGPGetKeyDBObjDataProperty(ObjFound, kPGPSigProperty_X509IssuerLongName,
					   @INData, MAX_509_CERT_SIZE, INSize) = 0)
	    and (PGPGetKeyDBObjDataProperty(ObjFound, kPGPSigProperty_X509LongName,
					    @ONData, MAX_509_CERT_SIZE, ONSize) = 0)
	    and (PGPGetSigCertifierKey(ObjFound, PGPPeekKeyDBRootKeySet(PGPPeekKeyDBObjKeyDB(Key)), SignKey) = 0)
	    and (SignKey = Key) and (INData = ONData) then begin
	      Result := RootCert;
	      Break;
	    end;
	  end;
	end;
      end;
    finally
      PGPFreeKeyIter(SingleKeyIter);
      PGPFreeKeyList(SingleKeyList);
      PGPFreeKEySet(SingleKeySet);
    end;
  end;
end;

function GetKeyPropKeyAlg(Key: pPGPKey): TKeyAlgorithm;
var
  KeyAlg	: PGPInt32;
  KeySet	: pPGPKeySet;
  KeyList	: pPGPKeyList;
  KeyIter	: pPGPKeyIter;
  SubKey	: pPGPSubKey;
begin
  Result := KeyAlgorithm_Invalid;
  if PGPGetKeyDBObjNumericProperty(Key, kPGPKeyProperty_AlgorithmID, KeyAlg) = 0 then begin
    Result := TKeyAlgorithm(KeyAlg);
    if Result = KeyAlgorithm_DSS then begin
      try
	if GetSingleKeyDBObjIter(Key, KeySet, KeyList, KeyIter) = 0 then begin
	  PGPKeyIterSeek(KeyIter, Key);
	  if PGPKeyIterNextKeyDBObj(KeyIter, kPGPKeyDBObjType_SubKey, SubKey) = 0 then Result := KeyAlgorithm_DHDSS;
	end;
      finally
	PGPFreeKeyIter(KeyIter);
	PGPFreeKeyList(KeyList);
	PGPFreeKeySet(KeySet);
      end;
    end;
  end;
end;

function GetKeyPropHashAlg(Key: pPGPKey): THashAlgorithm;
var
  HashAlg	: PGPInt32;
begin
  Result := HashAlgorithm_Invalid;
  if PGP7X then begin
    if PGPGetKeyDBObjNumericProperty(Key, kPGPKeyProperty_HashAlgorithmID, HashAlg) <> 0 then Exit;
  end
  else if PGPGetHashAlgUsed(Key, HashAlg) <> 0 then Exit;
  Result := THashAlgorithm(
    HashAlg - ord(HashAlg > kPGPHashAlgorithm_RIPEMD160) * pred(kPGPHashAlgorithm_SHA256 - kPGPHashAlgorithm_RIPEMD160) + 1
  );
end;

function GetKeyPropLegacy(Key: pPGPKey): Longbool;
var
  KeyVersion	: PGPInt32;
begin
  Result := true;
  if PGP7X then begin
    if PGPGetKeyDBObjNumericProperty(Key, kPGPKeyProperty_Version, KeyVersion) = 0 then begin
      Result := (KeyVersion <= kPGPKeyVersion_V3);
    end;
  end
  else Result := (GetKeyPropKeyAlg(Key) = KeyAlgorithm_RSA);
end;

function GetKeyPropTrust(Key: pPGPKey): TTrustLevel;
var
  KeyTrust	: PGPInt32;
begin
  Result := KeyTrust_Undefined;
  if PGPGetKeyDBObjNumericProperty(Key, kPGPKeyProperty_Trust, KeyTrust) = 0 then begin
    Result := TTrustLevel(KeyTrust);
  end;
end;

function GetKeyPropValidity(Key: pPGPKey): TValidityLevel;
var
  KeyValidity	: PGPInt32;
begin
  Result := Validity_Unknown;
  if PGPGetKeyDBObjNumericProperty(Key, kPGPKeyProperty_Validity, KeyValidity) = 0 then begin
    Result := TValidityLevel(KeyValidity);
  end;
end;

// UTC/GMT Unix format seconds
function GetKeyPropKeyCreationTime(Key: pPGPKey): Integer;
var
  CreaTime	: PGPTime;
begin
  Result := 0;
  if PGPGetKeyDBObjTimeProperty(Key, kPGPKeyProperty_Creation, CreaTime) = 0 then begin
    Result := PGPGetStdTimeFromPGPTime(CreaTime);
  end;
end;

// UTC/GMT Unix format seconds
function GetKeyPropKeyExpirationTime(Key: pPGPKey): Integer;
var
  ExpTime	: PGPTime;
  CreaTime	: PGPTime;
begin
  Result := 0;
  if (PGPGetKeyDBObjTimeProperty(Key, kPGPKeyProperty_Expiration, ExpTime) = 0)
  and (PGPGetKeyDBObjTimeProperty(Key, kPGPKeyProperty_Creation, CreaTime) = 0) then begin
    if ExpTime > CreaTime then Result := PGPGetStdTimeFromPGPTime(ExpTime);
  end;
end;

function GetKeyPropIsCorrupt(Key: pPGPKey): Longbool;
var
  Prop		: PGPBoolean;
begin
  Result := false;
  if PGPGetKeyDBObjBooleanProperty(Key, kPGPKeyProperty_IsNotCorrupt, Prop) = 0 then begin
    Result := not boolean(Prop);
  end;
end;

function GetKeyPropIsSecret(Key: pPGPKey): Longbool;
var
  Prop		: PGPBoolean;
begin
  Result := false;
  if PGPGetKeyDBObjBooleanProperty(Key, kPGPKeyProperty_IsSecret, Prop) = 0 then begin
    Result := boolean(Prop);
  end;
end;

function GetKeyPropIsAxiomatic(Key: pPGPKey): Longbool;
var
  Prop		: PGPBoolean;
begin
  Result := false;
  if PGPGetKeyDBObjBooleanProperty(Key, kPGPKeyProperty_IsAxiomatic, Prop) = 0 then begin
    Result := boolean(Prop);
  end;
end;

function GetKeyPropIsSigningKey(Key: pPGPKey): Longbool;
var
  Prop		: PGPBoolean;
begin
  Result := false;
  if PGPGetKeyDBObjBooleanProperty(Key, kPGPKeyProperty_IsSigningKey, Prop) = 0 then begin
    Result := boolean(Prop);
  end;
end;

function GetKeyPropIsRevoked(Key: pPGPKey): Longbool;
var
  Prop		: PGPBoolean;
begin
  Result := false;
  if PGPGetKeyDBObjBooleanProperty(Key, kPGPKeyProperty_IsRevoked, Prop) = 0 then begin
    Result := boolean(Prop);
  end;
end;

function GetKeyPropIsDisabled(Key: pPGPKey): Longbool;
var
  Prop		: PGPBoolean;
begin
  Result := false;
  if PGPGetKeyDBObjBooleanProperty(Key, kPGPKeyProperty_IsDisabled, Prop) = 0 then begin
    Result := boolean(Prop);
  end;
end;

function GetKeyPropIsExpired(Key: pPGPKey): Longbool;
var
  Prop		: PGPBoolean;
begin
  Result := false;
  if PGPGetKeyDBObjBooleanProperty(Key, kPGPKeyProperty_IsExpired, Prop) = 0 then begin
    Result := boolean(Prop);
  end;
end;

function GetKeyPropIsSecretShared(Key: pPGPKey): Longbool;
var
  Prop		: PGPBoolean;
begin
  Result := false;
  if PGPGetKeyDBObjBooleanProperty(Key, kPGPKeyProperty_IsSecretShared, Prop) = 0 then begin
    Result := boolean(Prop);
  end;
end;

function GetKeyPropCanEncrypt(Key: pPGPKey): Longbool;
var
  Prop		: PGPBoolean;
begin
  Result := false;
  if PGPGetKeyDBObjBooleanProperty(Key, kPGPKeyProperty_CanEncrypt, Prop) = 0 then begin
    Result := boolean(Prop);
  end;
end;

function GetKeyPropCanDecrypt(Key: pPGPKey): Longbool;
var
  Prop		: PGPBoolean;
begin
  Result := false;
  if PGPGetKeyDBObjBooleanProperty(Key, kPGPKeyProperty_CanDecrypt, Prop) = 0 then begin
    Result := boolean(Prop);
  end;
end;

function GetKeyPropCanSign(Key: pPGPKey): Longbool;
var
  Prop		: PGPBoolean;
begin
  Result := false;
  if PGPGetKeyDBObjBooleanProperty(Key, kPGPKeyProperty_CanSign, Prop) = 0 then begin
    Result := boolean(Prop);
  end;
end;

function GetKeyPropCanVerify(Key: pPGPKey): Longbool;
var
  Prop		: PGPBoolean;
begin
  Result := false;
  if PGPGetKeyDBObjBooleanProperty(Key, kPGPKeyProperty_CanVerify, Prop) = 0 then begin
    Result := boolean(Prop);
  end;
end;

function GetKeyPropCanCertify(KeySet: pPGPKeySet; Key: pPGPKey): Longbool;
begin
  Result := GetKeyPropCanSign(Key) and (GetKeyPropX509Cert(KeySet, Key) <> NoCert);
end;

function GetKeyPropHasRevoker(KeySet: pPGPKeySet; Key: pPGPKey; KeyPropsRec: TKeyPropsRec): Longbool;
var
  RevKeyCount	: PGPUInt32;
  RevKeyIndex	: PGPUInt32;
  RevKey	: pPGPKey;
  RevKeyID	: TPGPKeyID7;
  KeyID		: TKeyID;
begin
  Result := false;
  if (PGPCountRevocationKeys(Key, RevKeyCount) = 0) and (RevKeyCount > 0) then begin
    Result := true;
    if KeyPropsRec.kRevKeyIDList <> nil then begin
      for RevKeyIndex := 0 to pred(RevKeyCount) do begin
	if PGPGetIndexedRevocationKey(Key, KeySet, RevKeyIndex, RevKey, RevKeyID) = 0 then begin
	  if PGPGetKeyIDString(RevKeyID, kPGPKeyIDString_Full, KeyID) = 0 then begin
	    KeyPropsRec.kRevKeyIDList.Add(KeyID);
	  end;
	end;
      end;
    end;
  end;
end;

function GetKeyPropHasARR(KeySet: pPGPKeySet; Key: pPGPKey; KeyPropsRec: TKeyPropsRec): TADKType;
var
  ARKeyCount	: PGPUInt32;
  ARKeyIndex	: PGPUInt32;
  ARKey		: pPGPKey;
  ARKeyID	: TPGPKeyID7;
  ARClass	: PGPByte;
  KeyID		: TKeyID;
begin
  Result := NoADK;
  ARKey := nil;
  if (PGPCountAdditionalRecipientRequests(Key, ARKeyCount) = 0) and (ARKeyCount > 0) then begin
    Result := SimpleADK;
    for ARKeyIndex := 0 to pred(ARKeyCount) do begin
      if PGPGetIndexedAdditionalRecipientRequestKey(Key, KeySet, ARKeyIndex, ARKey, ARKeyID, ARClass) = 0 then begin
	if ARClass > 0 then Result := EnforcedADK;
	if KeyPropsRec.kADKeyIDList <> nil then begin
	  if PGPGetKeyIDString(ARKeyID, kPGPKeyIDString_Full, KeyID) = 0 then begin
	    KeyPropsRec.kADKeyIDList.Add(KeyID);
	  end;
	end;
      end;
    end;
  end;
end;

function GetKeyPropHasSubKey(KeySet: pPGPKeySet; Key: pPGPKey): Longbool;
var
  KeyList	: pPGPKeyList;
  KeyIter	: pPGPKeyIter;
  SubKey	: pPGPSubKey;
begin
  Result := false;
  if not GetKeyPropLegacy(Key) then begin
    if GetKeyIterAnyOrdering(KeySet, KeyList, KeyIter) = 0 then begin
      try
	PGPKeyIterSeek(KeyIter, Key);
	Result := (PGPKeyIterNextKeyDBObj(KeyIter, kPGPKeyDBObjType_SubKey, SubKey) = 0);
      finally
	PGPFreeKeyIter(KeyIter);
	PGPFreeKeyList(KeyList);
      end;
    end;
  end;
end;

function GetKeyPropUserIDs(Key: pPGPKey; KeyPropsRec: TKeyPropsRec; IncludeSignerIDs: Longbool): Integer;
var
  KeySet	: pPGPKeySet;
  KeyList	: pPGPKeyList;
  KeyIter	: pPGPKeyIter;
  UserID	: pPGPUserID;
  UserIDProp	: PGPBoolean;
  UserIDBuf	: TUserID;
  UserIDNum	: PGPInt32;
  IDSize	: PGPSize;
  IDBuffer	: String;
  KeySig	: pPGPSig;
  SigProp	: PGPBoolean;
  SignKeyID	: TPGPKeyID7;
  KeyID		: TKeyID;
begin
  try
    Result := GetSingleKeyDBObjIter(Key, KeySet, KeyList, KeyIter);
    if Result <> 0 then Exit;
    PGPKeyIterSeek(KeyIter, Key);
    // add UserIDs one by one to UserIDList
    while PGPKeyIterNextKeyDBObj(KeyIter, kPGPKeyDBObjType_UserID, UserID) = 0 do begin
      Result := PGPGetKeyDBObjBooleanProperty(UserID, kPGPUserIDProperty_IsAttribute, UserIDProp);
      if Result <> 0 then Exit;
      // check for attributes (picture, e.g.)
      if UserIDProp = PGPFalse then begin
	// UserID validity
	Result := PGPGetKeyDBObjNumericProperty(UserID, kPGPUserIDProperty_Validity, UserIDNum);
	if Result <> 0 then Exit;
	// UserID string
	Result := PGPGetKeyDBObjDataProperty(UserID, kPGPUserIDProperty_Name, @UserIDBuf, SizeOf(TUserID), IDSize);
	if Result <> 0 then Exit;
	try
	  with KeyPropsRec do begin
	    if PGP8X then
	      kUserIDList.AddObject(Utf8OrAnsi(UserIDBuf), ptr(UserIDNum))
	    else kUserIDList.AddObject(UserIDBuf, ptr(UserIDNum));
	    if IncludeSignerIDs and (kSignerIDList <> nil) then kSignerIDList.Add('');
	  end;
	except
	  Result := kPGPError_OutOfMemory;
	  Exit;
	end;
      end;
      IDBuffer := '';
      if IncludeSignerIDs then begin
	// add comma separated HexIDs of signing keys to KeyPropsRec.SignerIDList
	while PGPKeyIterNextKeyDBObj(KeyIter, kPGPKeyDBObjType_Signature, KeySig) = 0 do begin
	  if (PGPGetKeyDBObjBooleanProperty(KeySig, kPGPSigProperty_IsX509, SigProp) = 0)
	  and (SigProp = PGPTrue) then begin
	    with KeyPropsRec.kSignerIDList do Objects[pred(Count)] := ptr(SigProp);
	  end;
	  if PGP7X then
	    IDBuffer := IDBuffer + ',' + GetKeyPropKeyID(PGPPeekKeyDBObjKey(KeySig))
	  else begin
	    Result := PGPGetKeyIDOfCertifier(KeySig, SignKeyID);
	    if Result <> 0 then Exit;
	    Result := PGPGetKeyIDString(SignKeyID, kPGPKeyIDString_Full, KeyID);
	    if Result <> 0 then Exit;
	    IDBuffer := IDBuffer + ',' + KeyID;
	  end;
	end;
	Delete(IDBuffer, 1, 1);
	try
	  with KeyPropsRec.kSignerIDList do Strings[pred(Count)] := IDBuffer;
	except
	  Result := kPGPError_OutOfMemory;
	  Exit;
	end;
      end;
    end;
  finally
    PGPFreeKeyIter(KeyIter);
    PGPFreeKeyList(KeyList);
    PGPFreeKeySet(KeySet);
  end;
end;

function GetGroupsList(Context: pPGPContext; KeySetMain: pPGPKeySet; var GroupsList: TGroupsList): Integer;
var
  GroupFileSpec	: pPGPFileSpec;
  GroupSet	: pPGPGroupSet;
  GroupCount	: PGPUInt32;
  GroupIndex	: PGPUInt32;
  GroupID	: PGPGroupID;
  GroupInfo	: TPGPGroupInfo;
  KeyCount	: PGPUInt32;
  ItemCount	: PGPUInt32;
  GroupItemIter	: pPGPGroupItemIter;
  GroupItem	: TPGPGroupItem;
  HexIDList	: String;
  Key		: pPGPKey;
  HexID		: String;
begin
  Result := kPGPError_OutOfMemory;
  if GroupsList <> nil then begin
    try
      GroupsList.Clear;
      GroupFileSpec := nil;
      GroupSet := nil;
      Result := PGPNewFileSpecFromFullPath(Context, PChar(KeyRings.GroupsFile), GroupFileSpec);
      if Result <> 0 then Exit;
      try
	Result := PGPNewGroupSetFromFile(Context, GroupFileSpec, GroupSet);
	if Result <> 0 then Exit;
	try
	  Result := PGPCountGroupsInSet(GroupSet, GroupCount);
	  if (Result <> 0) or (GroupCount = 0) then Exit;
	  for GroupIndex := 0 to pred(GroupCount) do begin
	    Result := PGPGetIndGroupID(GroupSet, GroupIndex, GroupID);
	    if Result <> 0 then Continue;
	    Result := PGPGetGroupInfo(GroupSet, GroupID, GroupInfo);
	    if Result <> 0 then Continue;
	    Result := PGPCountGroupItems(GroupSet, GroupID, PGPTrue, KeyCount, ItemCount);
	    if (Result <> 0) or (KeyCount = 0) then Continue;
	    with GroupInfo do begin
	      if Description <> '' then
		GroupsList.Add(Description + '<' + Name + '>' + #9)
	      else GroupsList.Add('<' + Name + '>' + #9);
	    end;
	    GroupItemIter := nil;
	    Result := PGPNewGroupItemIter(GroupSet, GroupID, kPGPGroupIterFlags_AllKeysRecursive, GroupItemIter);
	    if Result <> 0 then Continue;
	    try
	      HexIDList := '';
	      while PGPGroupItemIterNext(GroupItemIter, GroupItem) = 0 do begin
		with GroupItem.Item.Key do begin
		  if PGP7X then
		    Result := PGPGetKeyByKeyID(KeySetMain, KeyStruct7.KeyID, kPGPPublicKeyAlgorithm_Invalid, Key)
		  else Result := PGPGetKeyByKeyID(KeySetMain, KeyStruct6.KeyID, kPGPPublicKeyAlgorithm_Invalid, Key);
		end;
		if Result <> 0 then Continue;
		HexID := GetKeyPropKeyID(Key);
		if HexID <> '' then HexIDList := HexIDList + ',' + HexID;
	      end;
	      Delete(HexIDList, 1, 1);
	      if HexIDList <> '' then with GroupsList do Strings[GroupIndex] := Strings[GroupIndex] + HexIDList;
	    finally
	      PGPFreeGroupItemIter(GroupItemIter);
	    end;
	  end;
	  Result := GroupsList.Count;
	finally
	  if GroupSet <> nil then PGPFreeGroupSet(GroupSet);
	end;
      finally
	PGPFreeFileSpec(GroupFileSpec);
      end;
    except
      Result := 0;
    end;
    if (Result <= 0) and (GroupsList <> nil) then GroupsList.Clear;
  end;
end;

function GetKeyProps(KeySet: pPGPKeySet; Key: pPGPKey; Flags: DWord; var KeyPropsRec: TKeyPropsRec): PGPError;
begin
  Result := 0;
  try
    with KeyPropsRec do begin
      // "string" properties
      if Flags and spgpKeyPropFlag_KeyID <> 0 then begin
	kHexID := ShortHexID(GetKeyPropKeyID(Key));
      end;
      if Flags and spgpKeyPropFlag_UserID <> 0 then kUserID := GetKeyPropUserID(Key);
      if Flags and spgpKeyPropFlag_Fingerprint <> 0 then kFingerprint := GetKeyPropFingerprint(Key);
      if Flags and spgpKeyPropFlag_CreationTimeStr <> 0 then kCreaTimeStr := GetKeyPropKeyCreationTimeStr(Key);
      if Flags and spgpKeyPropFlag_ExpirationTimeStr <> 0 then kExpTimeStr := GetKeyPropKeyExpirationTimeStr(Key);
      // "number" properties
      if Flags and spgpKeyPropFlag_KeyBits <> 0 then kSize := GetKeyPropKeyBits(Key);
      if Flags and spgpKeyPropFlag_X509Cert <> 0 then kX509Cert := GetKeyPropX509Cert(KeySet, Key);
      if Flags and spgpKeyPropFlag_KeyAlg <> 0 then kKeyAlgorithm := GetKeyPropKeyAlg(Key);
      if Flags and spgpKeyPropFlag_HashAlg <> 0 then kHashAlgorithm := GetKeyPropHashAlg(Key);
      if Flags and spgpKeyPropFlag_Trust <> 0 then kTrust := GetKeyPropTrust(Key);
      if Flags and spgpKeyPropFlag_Validity <> 0 then kValidity := GetKeyPropValidity(Key);
      if Flags and spgpKeyPropFlag_CreationTime <> 0 then kCreaTimeNum := GetKeyPropKeyCreationTime(Key);
      if Flags and spgpKeyPropFlag_ExpirationTime <> 0 then kExpTimeNum := GetKeyPropKeyExpirationTime(Key);
      // "boolean" properties
      if Flags and spgpKeyPropFlag_IsCorrupt <> 0 then kCorrupt := GetKeyPropIsCorrupt(Key);
      if Flags and spgpKeyPropFlag_IsSecret <> 0 then kPrivate := GetKeyPropIsSecret(Key);
      if Flags and spgpKeyPropFlag_IsAxiomatic <> 0 then kImplicitTrust := GetKeyPropIsAxiomatic(Key);
      if Flags and spgpKeyPropFlag_IsRevoked <> 0 then kRevoked := GetKeyPropIsRevoked(Key);
      if Flags and spgpKeyPropFlag_IsDisabled <> 0 then kDisabled := GetKeyPropIsDisabled(Key);
      if Flags and spgpKeyPropFlag_IsExpired <> 0 then kExpired := GetKeyPropIsExpired(Key);
      if Flags and spgpKeyPropFlag_IsSecretShared <> 0 then kSecShared := GetKeyPropIsSecretShared(Key);
      if Flags and spgpKeyPropFlag_CanEncrypt <> 0 then kCanEncrypt := GetKeyPropCanEncrypt(Key);
      if Flags and spgpKeyPropFlag_CanDecrypt <> 0 then kCanDecrypt := GetKeyPropCanDecrypt(Key);
      if Flags and spgpKeyPropFlag_CanSign <> 0 then kCanSign := GetKeyPropCanSign(Key);
      if Flags and spgpKeyPropFlag_CanVerify <> 0 then kCanVerify := GetKeyPropCanVerify(Key);
      if Flags and spgpKeyPropFlag_HasRevoker <> 0 then kHasRevoker := GetKeyPropHasRevoker(KeySet, Key, KeyPropsRec);
      if Flags and spgpKeyPropFlag_HasADK <> 0 then kHasADK := GetKeyPropHasARR(KeySet, Key, KeyPropsRec);
      if Flags and spgpKeyPropFlag_HasSubKey <> 0 then kHasSubKey := GetKeyPropHasSubKey(KeySet, Key);
      if Flags and spgpKeyPropFlag_LegacyKey <> 0 then kLegacyKey := GetKeyPropLegacy(Key);
      // "list" properties
      if (Flags and spgpKeyPropFlag_IncludeUserIDs <> 0) or (Flags and spgpKeyPropFlag_IncludeSignerIDs <> 0) then begin
	if kUserIDList <> nil then GetKeyPropUserIDs(Key, KeyPropsRec, (Flags and spgpKeyPropFlag_IncludeSignerIDs <> 0));
      end;
    end;
  except
    Result := -1;
  end;
end;

function GetKeySetProps(Context: pPGPContext; KeySet: pPGPKeySet;
			var KeyPropsList: TKeyPropsList;
			PropertyFlags, FilterFlags: DWord;
			KeyOrder: TPGPKeyOrdering): Integer;
var
  KeyFilter	: pPGPFilter;
  KeySetFound	: pPGPKeySet;
  KeyCount	: PGPUInt32;
  KeyList	: pPGPKeyList;
  KeyIter	: pPGPKeyIter;
  KeyFound	: pPGPKey;
begin
  KeyFilter := nil;
  KeySetFound := nil;
  KeyList := nil;
  KeyIter := nil;
  if FilterFlags <> KeyFilterFlag_AllKeys then begin
    Result := GetKeyFilter(Context, FilterFlags, KeyFilter);
    try
      if Result <> 0 then Exit;
      Result := PGPFilterKeySet(KeySet, KeyFilter, KeySetFound);
      if Result <> 0 then Exit;
      KeySet := KeySetFound;
    finally
      PGPFreeFilter(KeyFilter);
    end;
  end
  else Result := 0;
  try
    if Result <> 0 then Exit;
    Result := PGPCountKeys(KeySet, KeyCount);
    if Result <> 0 then Exit;
    if PropertyFlags <> 0 then begin
      if KeyPropsList = nil then KeyPropsList := TKeyPropsList.Create(PropertyFlags);
      if KeyPropsList = nil then begin
	Result := kPGPError_OutOfMemory;
	Exit;
      end;
      if (KeyPropsList.Count = 0) and ((PropertyFlags and spgpKeyPropFlag_IncludeGroupsList) <> 0) then begin
	GetGroupsList(Context, KeySet, KeyPropsList.GroupsList);
      end;
      Result := PGPOrderKeySet(KeySet, PGPKeyOrdering(succ(ord(KeyOrder) shr 1)), PGPBoolean(odd(ord(KeyOrder))), KeyList);
      if Result <> 0 then Exit;
      try
	Result := PGPNewKeyIter(KeyList, KeyIter);
	try
	  while (Result = 0) and (PGPKeyIterNextKeyDBObj(KeyIter, kPGPKeyDBObjType_Key, KeyFound) = 0) do begin
	    with KeyPropsList do begin
	      Result := GetKeyProps(KeySet, KeyFound, PropertyFlags, pKeyPropsRec(Objects[Add(GetKeyPropKeyID(KeyFound))])^);
	    end;
	  end;
	  if Result = 0 then Result := KeyPropsList.Count;
	finally
	  PGPFreeKeyIter(KeyIter);
	end;
      finally
	PGPFreeKeyList(KeyList);
      end;
    end
    else Result := KeyCount;
  finally
    PGPFreeKeySet(KeySetFound);
  end;
end;

function AddKeysToKeyRing(Context: pPGPContext; KeySetMain: pPGPKeySet;
			  KeysToImport: Pointer; var KeyPropsList: TKeyPropsList;
			  PropertyFlags: DWord): Integer;
var
  KeySetToAdd	: pPGPKeySet;
begin
  if PGP7X then begin
    KeySetToAdd := nil;
    Result := PGPCopyKeys(PGPPeekKeyDBRootKeySet(pPGPKeyDB(KeysToImport)), PGPPeekKeySetKeyDB(KeySetMain), KeySetToAdd);
    if Result <> 0 then Exit;
    try
      Result := KeyRings.UpdateKeyRings;
      if Result <> 0 then Exit;
      Result := GetKeySetProps(Context, KeySetToAdd, KeyPropsList,
			       PropertyFlags, KeyFilterFlag_AllKeys, UserID_Ordering);
    finally
      PGPFreeKeySet(KeySetToAdd);
    end;
  end
  else begin
    Result := PGPAddKeys(pPGPKeySet(KeysToImport), KeySetMain);
    if Result <> 0 then Exit;
    Result := KeyRings.UpdateKeyRings;
    if Result <> 0 then Exit;
    Result := GetKeySetProps(Context, pPGPKeySet(KeysToImport), KeyPropsList,
			     PropertyFlags, KeyFilterFlag_AllKeys, UserID_Ordering);
  end;
end;

function GetHexIDByAnyID(Context: pPGPContext; KeySetMain: pPGPKeySet;
			 const AnyID: String; var HexID: String): Integer;
var
  KeySetFound	: pPGPKeySet;
  KeyCount	: PGPUInt32;
  Key		: pPGPKey;
begin
  HexID := '';
  Result := 0;
  KeySetFound := nil;
  if AnyID <> '' then begin
    try
      Result := GetKeySetByAnyIDs(Context, KeySetMain, AnyID, KeySetFound);
      if Result <> 0 then Exit;
      Result := PGPCountKeys(KeySetFound, KeyCount);
      if Result <> 0 then Exit;
      if KeyCount > 0 then begin
	Result := GetKeyFromKeySet(KeySetFound, Key);
	if Result <> 0 then Exit;
	HexID := GetKeyPropKeyID(Key);
      end;
      Result := KeyCount;
    finally
      PGPFreeKeySet(KeySetFound);
    end;
  end;
end;

function GetMatchingKey(Context: pPGPContext; KeySet: pPGPKeySet; Passphrase: PChar): pPGPKey;
var
  KeyList: pPGPKeyList;
  KeyIter: pPGPKeyIter;
  Key: pPGPKey;
begin
  Result := nil;
  if GetKeyIterAnyOrdering(KeySet, KeyList, KeyIter) = 0 then begin
    try
      while PGPKeyIterNextKeyDBObj(KeyIter, kPGPKeyDBObjType_Key, Key) = 0 do begin
	if PGPPassphraseIsValid(Key, PGPOPassphrase(Context, Passphrase), PGPOLastOption(Context)) = PGPTrue then begin
	  Result := Key;
	  Break;
	end;
      end;
    finally
      PGPFreeKeyIter(KeyIter);
      PGPFreeKeyList(KeyList);
    end;
  end;
end;

function PassphraseIsValid(Context: pPGPContext; KeySetMain: pPGPKeySet; AnyID, Passphrase: PChar): Longbool;
var
  KeySetFound	: pPGPKeySet;
  PhraseBuffer	: PChar;
  KeyCount	: PGPUInt32;
  Key		: pPGPKey;
begin
  Result := false;
  KeySetFound := nil;
  if GetKeySetByAnyIDs(Context, KeySetMain, AnyID, KeySetFound) = 0 then begin
    PhraseBuffer := Passphrase;
    if PGP8X then begin
      PhraseBuffer := PGPNewSecureData(PGPGetDefaultMemoryMgr, MaxUTF8Length, kPGPMemoryMgrFlags_Clear);
      if PhraseBuffer <> nil then AnsiToUtf8PChar(Passphrase, PhraseBuffer, MaxUTF8Length);
    end;
    try
      Result := (PGPCountKeys(KeySetFound, KeyCount) = 0) and
		(KeyCount = 1) and (GetKeyFromKeySet(KeySetFound, Key) = 0) and
		(PGPPassphraseIsValid(Key, PGPOPassphrase(Context, PhraseBuffer), PGPOLastOption(Context)) <> PGPFalse);
    finally
      PGPFreeKeySet(KeySetFound);
      if PGP8X then PGPFreeData(PhraseBuffer);
    end;
  end;
end;

function GetCachedPassphrase(Context: pPGPContext; var Key: pPGPKey; KeySet: pPGPKeySet; var Passphrase: PChar): Longbool;
var
  KeyIter: pPGPKeyIter;
begin
  Result := false;
  if PGP8X and ((Key <> nil) or (KeySet <> nil)) then begin
    Passphrase := PGPNewSecureData(PGPGetDefaultMemoryMgr, 1024, kPGPMemoryMgrFlags_Clear);
    if Passphrase <> nil then begin
      try
	if Key <> nil then
	  Result := (PGPPassphraseIsValid(Key, PGPOLastOption(Context), PGPOLastOption(Context)) = PGPTrue)
	else begin
	  KeyIter := nil;
	  try
	    if PGPNewKeyIterFromKeySet(KeySet, KeyIter) = 0 then begin
	      while PGPKeyIterNextKeyDBObj(KeyIter, kPGPKeyDBObjType_Key, Key) = 0 do begin
		Result := (PGPPassphraseIsValid(Key, PGPOLastOption(Context), PGPOLastOption(Context)) = PGPTrue);
		if Result then Break;
	      end;
	    end;
	  finally
	    PGPFreeKeyIter(KeyIter);
	  end;
	end;
      finally
	if not Result and (PGPFreeData(Passphrase) = kPGPError_NoErr) then Passphrase := nil;
      end;
    end;
  end;
end;

function CachePassphrase(Context: pPGPContext; Key: pPGPKey; TimeInterval: PGPTime;
			 ShareCache: Longbool; var Passphrase: PChar): Longbool;
var
  OptionList: pPGPOptionList;
begin
  Result := false;
  if PGP8X and (Key <> nil) and (Passphrase <> nil) then begin
    OptionList := nil;
    if PGPBuildOptionList(Context, OptionList,
      [
       PGPOPassphrase(Context, Passphrase),
       PGPOCachePassphrase(Context, TimeInterval, PGPBoolean(ord(ShareCache) and 1))
      ]
    ) = 0 then begin
      try
	Result := (PGPPassphraseIsValid(Key, OptionList, PGPOLastOption(Context)) = PGPTrue) and
		  (PGPCacheKeyDB(PGPPeekKeyDBObjKeyDB(Key), TimeInterval) = 0);
      finally
	if (PGPFreeData(Passphrase) = kPGPError_NoErr) then Passphrase := nil;
	PGPFreeOptionList(OptionList);
      end;
    end;
  end;
end;

function IsPassphraseCacheEmpty(Context: pPGPContext): Longbool;
var
  Global, Local, Other: PGPUInt32;
begin
  Result := PGP8X and (PGPCountCachedPassphrases(Context, Global, Local, Other) = 0) and (Global + Local + Other = 0);
end;

function PurgePassphraseCache(Context: pPGPContext): Longbool;
begin
  Result := (PGPPurgePassphraseCache(Context) = 0) and (PGPPurgeKeyDBCache(Context) = 0);
end;

function TKeyRings.UpdateKeyRings: PGPError;
begin
  Result := kPGPError_NoErr;
  if PGP7X then
    Result := PGPFlushKeyDB(RingKeyDB)
  else if PGPKeySetNeedsCommit(RingKeySet) = PGPTrue then Result := PGPCommitKeyringChanges(RingKeySet);
  if Result = 0 then PGPclNotifyKeyringChanges(GetCurrentProcessID);
end;

function TKeyRings.SetKeyRings(const Pubring, Secring: String): PGPError;
begin
  Result := 0;
  PubringFile := Pubring;
  SecringFile := Secring;
  if ((PubringFile = '') or (SecringFile = '')) then Result := kPGPError_ImproperInitialization;
end;

function TKeyRings.InitKeyRings(var Context: pPGPContext; var KeySetMain: pPGPKeySet): PGPError;
var
  Prefs: TPreferenceRec;
  PubFileSpec: pPGPFileSpec;
  SecFileSpec: pPGPFileSpec;
  OldInitCount: Longint;
begin
  Result := kPGPError_ImproperInitialization;
  if PGPInitErrorCode = ieNone then begin
    if InitCount <= 0 then begin
      InitCount := 0;
      RingKeyDB := nil;
      RingKeySet := nil;
      RingContext := nil;
      PubFileSpec := nil;
      SecFileSpec := nil;
      Result := PGPNewContext(kPGPsdkAPIVersion, RingContext);
      if Result <> 0 then Exit;
      if (PubRingFile = '') or (SecringFile = '') then begin
	GetPreferences(Prefs, PrefsFlag_PublicKeyring or PrefsFlag_PrivateKeyring or PrefsFlag_GroupsFile);
	SetKeyRings(Prefs.PublicKeyring, Prefs.PrivateKeyring);
	if GroupsFile = '' then GroupsFile := Prefs.GroupsFile;
      end
      else if (GroupsFile = '') and (GetPreferences(Prefs, PrefsFlag_GroupsFile) = 0) then begin
	GroupsFile := Prefs.GroupsFile;
      end;
      try
	Result := PGPNewFileSpecFromFullPath(RingContext, PChar(PubringFile), PubFileSpec);
	if Result <> 0 then Exit;
	Result := PGPNewFileSpecFromFullPath(RingContext, PChar(SecringFile), SecFileSpec);
	if Result <> 0 then Exit;
	if PGP7X then begin
	  Result := PGPOpenKeyDBFile(RingContext, kPGPOpenKeyDBFileOptions_Mutable, PubFileSpec, SecFileSpec, RingKeyDB);
	  if Result = 0 then RingKeySet := PGPPeekKeyDBRootKeySet(RingKeyDB);
	end
	else Result := PGPOpenKeyRingPair(RingContext, kPGPKeyRingOpenFlags_Mutable, PubFileSpec, SecFileSpec, RingKeySet);
      finally
	PGPFreeFileSpec(PubFileSpec);
	PGPFreeFileSpec(SecFileSpec);
      end;
      if Result <> 0 then begin
	PGPFreeContext(RingContext);
	RingContext := nil;
	if PGP7X then begin
	  PGPFreeKeyDB(RingKeyDB);
	  RingKeyDB := nil;
	end
	else begin
	  PGPFreeKeySet(RingKeySet);
	  RingKeySet := nil;
	end;
      end;
    end
    else if PGP7X then begin
      Result := PGPsdkReconnect;
      // prevents PGP 8.1 problems
      if Result > 0 then Result := 0;
    end
    else Result := 0;
    if Result = 0 then inc(InitCount);
    KeySetMain := RingKeySet;
    Context := RingContext;
    if PGP7X and not Recursion and (Result = kPGPError_RPCFailed) then begin
      if InitCount > 0 then begin
	OldInitCount := InitCount;
	InitCount := 1;
	FreeKeyRings;
      end
      else OldInitCount := 0;
      PGPsdkCleanup;
      Recursion := true;
      if StartPGPsdkServ then Result := PGPsdkInit(InitFlags);
      if Result = 0 then Result := InitKeyRings(Context, KeySetMain);
      if Result = 0 then InitCount := succ(OldInitCount);
    end
    else Recursion := false;
  end;
end;

procedure TKeyRings.FreeKeyRings;
begin
  try
    if InitCount = 1 then begin
      try
	if PGP7X then
	  PGPFreeKeyDB(RingKeyDB)
	else PGPFreeKeySet(RingKeySet);
      finally
	RingKeySet := nil;
	RingKeyDB := nil;
	try
	  PGPFreeContext(RingContext);
	finally
	  RingContext := nil;
	end;
      end;
    end;
  finally
    if InitCount > 0 then dec(InitCount);
  end;
end;

initialization
  KeyRings := TKeyRings.Create;

finalization
  KeyRings.Free;
  KeyRings := nil;

end.

