{$J+,Z4}
unit X509Funcs; // PGP 7.X and later only

{------------------------------------------------------------------------------}
{                                                                              }
{                 This unit is partly based on Steve Heller's                  }
{         SPGP sources available from http://www.oz.net/~srheller/spgp/        }
{                                                                              }
{                Portions created by Michael in der Wiesche are                }
{              Copyright (C) 2001-2004 by Michael in der Wiesche               }
{                                                                              }
{------------------------------------------------------------------------------}

interface

uses
  Windows,
  Classes,
  SysUtils,
  KeyPropTypes,
  X509Types,
  UTF8,
  pgpBase,
  pgpErrors,
  pgpPubTypes,
  pgpUtilities,
  pgpOptionList,
  pgpMemoryMgr,
  pgpKeys,
  KeyFuncs,
  PGPDialogs;

{ specifying ...Prompt strings will present the respective PGP dialogs regardless of specifying any ...ID(s) }

// if CertAttributes' CommonName or Email are undefined they are taken from the CertKey's primary user ID
function CreateX509CertificateRequest(const CertKeyAndPassPrompt, CertKeyHexID: String; Passphrase: PChar;
				      var CertAttributes: TX509CertAttributes; Armor: Longbool;
				      var Request: String; var ReqKeyProps: TKeyPropsRec;
				      KeyPropFlags, WinHandle: Integer): Integer;
// specifying a unique CertAttributes' SerialNumber is mandatory, CommonName and Email are taken from the request
// if the SignCertIASN string contains a SignKeyHexID the key's last valid certificate will be used for certifying
function CreateX509CertificateFromRequest(const SignKeyAndPassPrompt, SignCertIASN: String; Passphrase: PChar;
					  var CertAttributes: TX509CertAttributes; ValidDays: Integer;
					  Armor: Longbool; const Request: String; var Certificate: String;
					  var CertPropsRec: TX509CertPropsRec; WinHandle: Integer): Integer;
// if the RevokeCertIASN string contains valid CertKeyHexID(s) all certificates on the respective key(s) will be revoked
// if no RevokeCertIASN string is specified all (non-self) certificates issued by the respective SignKey will be revoked
// if the SignCertIASN string contains a SignKeyHexID it's last valid certificate will be used for revoking certificates
// if SignCertIASN isn't provided and RevokeCertIASN contains a unique IASN the certifying key can be found automatically
function CreateX509CertificateRevocationList(const CertKeyPrompt, SignKeyAndPassPrompt, RevokeCertIASN, SignCertIASN: String;
					     Passphrase: PChar; ValidDays: Integer; var CertPropsList: TX509CertPropsList;
					     WinHandle: Integer): Integer;
// specifying a unique CertAttributes.SerialNumber is mandatory
// splitting a user ID into CommonName and Email will certify it
// self-signed certificates cannot be created for an existing user ID
function CreateX509Certificate(const CertKeyPrompt, SignKeyAndPassPrompt, CertKeyHexID, SignCertIASN: String;
			       Passphrase: PChar; var CertAttributes: TX509CertAttributes; ValidDays: Integer;
			       SelfSigned: Longbool; var CertPropsList: TX509CertPropsList;
			       WinHandle: Longint): Integer;
// self-signed certificates can only be removed including the user ID they are attached to
// user IDs of certificates can only be removed if they don't have any other signatures assigned
function RemoveX509Certificate(const CertKeyPrompt, RemoveCertIASN: String; IncludeUserID: Longbool;
			       var CertPropsList: TX509CertPropsList; WinHandle: Integer): Integer;
// import binary or armored standard X509 certificates or PGP keys to keyring(s)
function ImportX509Certificate(const CertKeyPrompt, CertData: String; FromFile: Longbool;
			       var CertPropsList: TX509CertPropsList; WinHandle: Integer): Integer;
// either export a single certificate or a (binary) certificate chain with parent certificate(s) included
function ExportX509Certificate(const CertKeyPrompt, ExportCertIASN: String; Armor, Chain, ToFile: Longbool;
			       var CertData: String; var CertPropsList: TX509CertPropsList; WinHandle: Integer): Integer;
// the required certificate chain can be created using ExportX509Certificate() with the Chain parameter enabled
function VerifyX509CertificateChain(const CertChain: String; FromFile: Longbool): Integer;
// returns number of certificates found or error
function GetX509CertificateProperties(const CertData: String; FromFile: Longbool;
				      var CertPropsList: TX509CertPropsList): Integer;
// returns number of keyring certificates or error
// KeyFilterFlags will be applied to any kind of search
// SignKeyID can narrow the search with CertKey...IDs specified
// IncludeSigner is only valid without any CertKey...IDs specified
// in this case you will get the following results depending on SignKeyHexID and IncludeSigner:
// (SignKey = nil) and (IncludeSigner = false)  => all certified keys on keyring
// (SignKey = nil) and (IncludeSigner = true)   => all certifier keys on keyring (signing keys only)
// (SignKey <> nil) and (IncludeSigner = false) => all SignKey-certified keys on keyring without certifier
// (SignKey <> nil) and (IncludeSigner = true)  => all SignKey-certified keys on keyring including certifier
function FindX509CertProps(const CertKeyUserID, CertKeyHexID, SignKeyHexID: String; KeyFilterFlags: DWord;
			   IncludeSigner: Longbool; var CertPropsList: TX509CertPropsList): Integer;

implementation

function GetCertPropIASN(Cert: pPGPKeyDBObj): String; forward;
function GetCertPropIssuerName(Cert: pPGPKeyDBObj): String; forward;
function GetCertPropIssuerSerialNumber(Cert: pPGPKeyDBObj): String; forward;
function GetCertPropOwnerKeyID(Cert: pPGPKeyDBObj): String; forward;
function GetCertPropOwnerName(Cert: pPGPKeyDBObj): String; forward;
function GetCertPropOwnerSerialNumber(Cert: pPGPKeyDBObj): String; forward;
function GetCertPropCertCreationTimeStr(Cert: pPGPKeyDBObj): String; forward;
function GetCertPropCertExpirationTimeStr(Cert: pPGPKeyDBObj): String; forward;
function GetCertPropCertCreationTime(Cert: pPGPKeyDBObj): Integer; forward;
function GetCertPropCertExpirationTime(Cert: pPGPKeyDBObj): Integer; forward;
function GetCertPropIsVerified(Cert: pPGPKeyDBObj): Longbool; forward;
function GetCertPropIsCorrupt(Cert: pPGPKeyDBObj): Longbool; forward;
function GetCertPropIsRevoked(Cert: pPGPKeyDBObj): Longbool; forward;
function GetCertPropIsExpired(Cert: pPGPKeyDBObj): Longbool; forward;
function GetCertPropIsRoot(Cert: pPGPKeyDBObj): Longbool; forward;
function GetCertPropIsValid(Cert: pPGPKeyDBObj): Longbool; forward;
function IsSerialNumberUnique(Context: pPGPContext; KeySetToCheck: pPGPKeySet; SignKey: pPGPKeyDBObj;
			      SerialNumber: Longint; var Unique: Longbool): Integer; forward;
function GetX509CertPropsRec(Certificate: pPGPKeyDBObj; var CertPropsRec: TX509CertPropsRec): Integer; forward;
function GetX509CertPropsList(Certificate: pPGPKeyDBObj; var CertPropsList: TX509CertPropsList): Integer; forward;
function GetX509CertSetProps(CertSet, KeySetToSearch: pPGPKeySet; SignKey: pPGPKeyDBObj;
			     var CertPropsList: TX509CertPropsList): Integer; forward;

// convert X509 errors to PGP errors with corresponding messages
function TranslateX509ErrorToPGPError(X509Error: PGPError): PGPError;
begin
  case X509Error of
    k509Error_NeededCertNotAvailable: Result := kPGPError_X509NeededCertNotAvailable;
    k509Error_SelfSignedCert: Result := kPGPError_X509SelfSignedCert;
    k509Error_InvalidCertificateSignature: Result := kPGPError_X509InvalidCertificateSignature;
    k509Error_InvalidCertificateFormat: Result := kPGPError_X509InvalidCertificateFormat;
  else
    Result := X509Error;
  end;
end;

// init main keyset and filter for keys with X509 certificates
function InitX509CertKeySet(var Context: pPGPContext; var KeySetMain, KeySetX509: pPGPKeySet): Integer;
var
  X509Filter	: pPGPFilter;
begin
  Context := nil;
  KeySetMain := nil;
  KeySetX509 := nil;
  Result := KeyRings.InitKeyRings(Context, KeySetMain);
  if Result <> 0 then Exit;
  try
    Result := PGPNewKeyDBObjBooleanFilter(Context, kPGPSigProperty_IsX509, PGPTrue, X509Filter);
    if Result <> 0 then Exit;
    try
      Result := PGPFilterKeySet(KeySetMain, X509Filter, KeySetX509);
    finally
      PGPFreeFilter(X509Filter);
    end;
  finally
    if Result <> 0 then begin
      KeyRings.FreeKeyRings;
      KeySetMain := nil;
      Context := nil;
    end;
  end;
end;

// finit X509 keyset and free keyrings, convert X509 specific errors
procedure FinitX509CertKeySet(var KeySetX509: pPGPKeySet);
begin
  PGPFreeKeySet(KeySetX509);
  KeyRings.FreeKeyRings;
  KeySetX509 := nil;
end;

function LastPos(Find: Char; const Dest: String): Integer; register; assembler;
asm	// EAX=Find, EDX=@Dest
  OR	EDX,EDX
  JE	@EXIT
  MOV	ECX,[EDX-04h]
  @LOOP:
  DEC	ECX
  JL	@EXIT
  CMP	AL,[EDX+ECX]
  JNE	@LOOP
  MOV	EAX,ECX
  INC	EAX
  RET
  @EXIT:
  XOR	EAX,EAX
end;

// retrieve a key's primary user ID for using them as CommonName and Email on a certificate
function GetPrimaryUserIDAsAttributes(CertKey: pPGPKeyDBObj; var Name, Address: String): Integer;
var
  StrPos, StrLen: Integer;
begin
  try
    Name := GetKeyPropUserID(CertKey);
    StrPos := LastPos(' ', Name);
    if StrPos <> 0 then Address := Copy(Name, succ(StrPos), MAXINT);
    Delete(Name, StrPos, MAXINT);
    StrLen := Length(Address);
    if StrLen > 2 then begin
      if Address[StrLen] = '>' then Delete(Address, StrLen, 1);
      if Address[1] = '<' then Delete(Address, 1, 1);
    end;
  finally
    Result := ord((Name = '') and (Address = '')) * kPGPError_InvalidDistinguishedName;
  end;
end;

// convert attribute record to option list parameter using risky type cast ...
function GetAttributeListFromRecord(const CertAttributes: TX509CertAttributes;
				    const AttributeList: TX509CertAttrArray;
				    var AttributeCount: Longint): Integer;
var
  AttributeAddr	: ^PChar;
  AttributeIndex: PGPUInt32;
  SerialNumber	: String;
begin
  Result := 0;
  AttributeCount := 0;
  try
    AttributeAddr := @CertAttributes;
    for AttributeIndex := 0 to pred(SizeOf(CertAttributes) div SizeOf(String)) do begin
      with AttributeList[AttributeCount] do begin
	if PGPAVAttribute(AttributeIndex) <> kPGPAVAttribute_SerialNumber then begin
	  if AttributeAddr^ <> '' then begin
	    Attribute := PGPAVAttribute(AttributeIndex);
	    Value.PointerValue := AttributeAddr^;
	    Size := Length(Value.PointerValue);
	    inc(AttributeCount);
	  end;
	end
	else begin
	  Attribute := PGPAVAttribute(AttributeIndex);
	  SerialNumber := IntToStr(CertAttributes.SerialNumber);
	  Value.PointerValue := pointer(SerialNumber);
	  Size := Length(SerialNumber);
	  inc(AttributeCount);
	end;
	inc(AttributeAddr);
      end;
    end;
  except
    on EAccessViolation do
      Result :=kPGPError_BadMemAddress
    else Result := kPGPError_UnknownError;
  end;
end;

// search for certificates and their signing keys
function FindCertificate(KeyIter: pPGPKeyIter; KeySetToSearch: pPGPKeySet; var Certificate, SignKey: pPGPKeyDBObj): Integer;
var
  X509Prop	: PGPBoolean;
begin
  SignKey := nil;
  repeat
    Result := PGPKeyIterNextKeyDBObj(KeyIter, kPGPKeyDBObjType_Any, Certificate);
    if (Result = 0) and (PGPGetKeyDBObjBooleanProperty(Certificate, kPGPSigProperty_IsX509, X509Prop) = 0)
    and (X509Prop = PGPTrue) then begin
      PGPGetSigCertifierKey(Certificate, KeySetToSearch, SignKey);
      Break;
    end;
  until Result <> 0;
end;

// check whether a user ID already has a certificate by the signing key(s)
function HasUserIDCertifierCert(CertKey: pPGPKeyDBObj; const CertKeyUserID: String;
				KeySetToSearch: pPGPKeySet; SignKey: pPGPKeyDBObj): Longbool;
var
  KeySet	: pPGPKeySet;
  KeyList	: pPGPKeyList;
  KeyIter	: pPGPKeyIter;
  UTF8ID	: UTF8String;
  UIDFound	: pPGPKeyDBObj;
  UserIDBuf	: TUserID;
  IDSize	: PGPSize;
  SigFound	: pPGPKeyDBObj;
  X509Prop	: PGPBoolean;
  KeyFound	: pPGPKeyDBObj;
begin
  Result := false;
  try
    if GetSingleKeyDBObjIter(CertKey, KeySet, KeyList, KeyIter) = 0 then begin
      UTF8ID := AnsiToUtf8(CertKeyUserID);
      while PGPKeyIterNextKeyDBObj(KeyIter, kPGPKeyDBObjType_UserID, UIDFound) = 0 do begin
	if (PGPGetKeyDBObjDataProperty(UIDFound, kPGPUserIDProperty_Name, @UserIDBuf, SizeOf(TUserID), IDSize) = 0)
	and ((PGPCompareUserIDStrings(UserIDBuf, PChar(CertKeyUserID)) = 0) or
	     (PGPCompareUserIDStrings(UserIDBuf, PChar(UTF8ID)) = 0)) then begin
	  while PGPKeyIterNextKeyDBObj(KeyIter, kPGPKeyDBObjType_Signature, SigFound) = 0 do begin
	    if (PGPGetKeyDBObjBooleanProperty(SigFound, kPGPSigProperty_IsX509, X509Prop) = 0)
	    and (X509Prop = PGPTrue) and (PGPGetSigCertifierKey(SigFound, KeySetToSearch, KeyFound) = 0)
	    and ((KeyFound = SignKey) or ((SignKey = nil) and GetKeyPropCanCertify(KeySetToSearch, KeyFound))) then begin
	      Result := true;
	      Exit;
	    end;
	  end;
	end;
      end;
    end;
  finally
    PGPFreeKeyIter(KeyIter);
    PGPFreeKeyIter(KeyList);
    PGPFreeKeySet(KeySet);
  end;
end;

// returns the last valid certificate issued by SignKey (if specified)
function GetValidX509CertFromKey(CertKey, SignKey: pPGPKeyDBObj; KeySetToSearch: pPGPKeySet): pPGPKeyDBObj;
var
  KeySet	: pPGPKeySet;
  KeyList	: pPGPKeyList;
  KeyIter	: pPGPKeyIter;
  Certificate	: pPGPKeyDBObj;
  KeyFound	: pPGPKeyDBObj;
begin
  Result := nil;
  try
    if GetSingleKeyDBObjIter(CertKey, KeySet, KeyList, KeyIter) = 0 then begin
      while FindCertificate(KeyIter, KeySetToSearch, Certificate, KeyFound) = 0 do begin
	if ((KeyFound = SignKey) or (SignKey = nil)) and GetCertPropIsValid(Certificate) then begin
	  Result := Certificate;
	end;
      end;
    end;
  finally
    PGPFreeKeyIter(KeyIter);
    PGPFreeKeyIter(KeyList);
    PGPFreeKeySet(KeySet);
  end;
end;

// filter the specified keyset for signing keys with valid certificates
function GetValidX509CertifierSet(const KeySetToFilter: pPGPKeySet;
				  var KeySetValidCertifiers: pPGPKeySet;
				  RootOnly: Longbool): PGPError;
var
  KeyIter	: pPGPKeyIter;
  Certificate	: pPGPKeyDBObj;
  SignKey	: pPGPKeyDBObj;
  CertKey	: pPGPKeyDBObj;
begin
  KeySetValidCertifiers := nil;
  Result := PGPNewEmptyKeySet(PGPPeekKeySetKeyDB(KeySetToFilter), KeySetValidCertifiers);
  if Result <> 0 then Exit;
  Result := PGPNewKeyIterFromKeySet(KeySetToFilter, KeyIter);
  if Result <> 0 then Exit;
  try
    while FindCertificate(KeyIter, KeySetToFilter, Certificate, SignKey) = 0 do begin
      CertKey := PGPPeekKeyDBObjKey(Certificate);
      if GetKeyPropCanSign(CertKey) and GetCertPropIsValid(Certificate) then begin
	if not RootOnly or (SignKey = CertKey) then PGPAddKey(CertKey, KeySetValidCertifiers);
      end;
    end;
  finally
    PGPFreeKeyIter(KeyIter);
  end;
end;

// filter the specified keyset for signing keys with certificates
function GetX509CertifierSet(const KeySetToFilter: pPGPKeySet; var KeySetCertifiers: pPGPKeySet): PGPError;
var
  KeyIter	: pPGPKeyIter;
  Certificate	: pPGPKeyDBObj;
  SignKey	: pPGPKeyDBObj;
  CertKey	: pPGPKeyDBObj;
begin
  KeySetCertifiers := nil;
  Result := PGPNewEmptyKeySet(PGPPeekKeySetKeyDB(KeySetToFilter), KeySetCertifiers);
  if Result <> 0 then Exit;
  Result := PGPNewKeyIterFromKeySet(KeySetToFilter, KeyIter);
  if Result <> 0 then Exit;
  try
    while FindCertificate(KeyIter, KeySetToFilter, Certificate, SignKey) = 0 do begin
      CertKey := PGPPeekKeyDBObjKey(Certificate);
      if GetKeyPropCanSign(CertKey) then PGPAddKey(CertKey, KeySetCertifiers);
    end;
  finally
    PGPFreeKeyIter(KeyIter);
  end;
end;

// retrieve keys with certificates - options:
// (SignKey = nil) and (IncludeSigner = false)  => all certified keys on keyring
// (SignKey = nil) and (IncludeSigner = true)   => all certifier keys on keyring (signing keys only)
// (SignKey <> nil) and (IncludeSigner = false) => all SignKey-certified keys on keyring without certifier
// (SignKey <> nil) and (IncludeSigner = true)  => all SignKey-certified keys on keyring including certifier
function GetCertKeySetBySignKey(Context: pPGPContext; KeySetToFilter: pPGPKeySet; SignKey: pPGPKeyDBObj;
				IncludeSigner, ValidKeysOnly: Longbool; var KeySetFiltered: pPGPKeySet): Integer;
var
  SignKeyID	: TPGPKeyID7;
  SignKeyFilter	: pPGPFilter;
  ExcludeFilter	: pPGPFilter;
  CertKeyFilter	: pPGPFilter;
  KeyCount	: PGPUInt32;
  KeyIter	: pPGPKeyIter;
  Certificate	: pPGPKeyDBObj;
  KeyFound	: pPGPKeyDBObj;
  KeySetChecked	: pPGPKeySet;
begin
  KeySetFiltered := nil;
  if SignKey = nil then begin
    if IncludeSigner then begin
      // signing keys with certificates only
      Result := GetX509CertifierSet(KeySetToFilter, KeySetFiltered);
    end
    else begin
      // all certified keys in keyset (as is)
      Result := PGPIncKeySetRefCount(KeySetToFilter);
      if Result = 0 then KeySetFiltered := KeySetToFilter;
    end;
  end
  else begin
    SignKeyFilter := nil;
    ExcludeFilter := nil;
    CertKeyFilter := nil;
    try
      // filter by signing key
      Result := PGPGetKeyIDFromKey(SignKey, SignKeyID);
      if Result <> 0 then Exit;
      Result := PGPNewKeyDBObjDataFilter(Context, kPGPSigProperty_KeyID, @SignKeyID,
					 SizeOf(TPGPKeyID7), kPGPMatchEqual, SignKeyFilter);
      if Result <> 0 then Exit;
      // remove signing key?
      if not IncludeSigner then begin
	ExcludeFilter := GetKeyExcludeFilter(Context, SignKey);
	Result := PGPIntersectFilters(SignKeyFilter, ExcludeFilter, CertKeyFilter);
	SignKeyFilter := nil;
	ExcludeFilter := nil;
	if Result <> 0 then Exit;
      end
      else begin
	CertKeyFilter := SignKeyFilter;
	SignKeyFilter := nil;
      end;
      Result := PGPFilterKeySet(KeySetToFilter, CertKeyFilter, KeySetFiltered);
    finally
      PGPFreeFilter(SignKeyFilter);
      PGPFreeFilter(ExcludeFilter);
      PGPFreeFilter(CertKeyFilter);
    end;
  end;
  if Result <> 0 then Exit;
  Result := PGPCountKeys(KeySetFiltered, KeyCount);
  if Result <> 0 then Exit;
  if KeyCount <> 0 then begin
    // remove keys without (valid) SignKey certificates
    if SignKey <> nil then begin
      KeySetChecked := nil;
      KeyIter := nil;
      try
	Result := PGPNewEmptyKeySet(PGPPeekKeySetKeyDB(KeySetFiltered), KeySetChecked);
	if Result = 0 then begin
	  Result := PGPNewKeyIterFromKeySet(KeySetFiltered, KeyIter);
	  if Result = 0 then begin
	    while FindCertificate(KeyIter, KeySetFiltered, Certificate, KeyFound) = 0 do begin
	      if (KeyFound = SignKey) and (not ValidKeysOnly or GetCertPropIsValid(Certificate)) then begin
		PGPAddKey(PGPPeekKeyDBObjKey(Certificate), KeySetChecked);
	      end;
	    end;
	    if Result = 0 then begin
	      Result := PGPCountKeys(KeySetChecked, KeyCount);
	      if (Result = 0) and (KeyCount = 0) then Result := kPGPError_X509NeededCertNotAvailable;
	    end;
	  end;
	end;
      finally
	PGPFreeKeyIter(KeyIter);
	PGPFreeKeySet(KeySetFiltered);
	if Result = 0 then
	  KeySetFiltered := KeySetChecked
	else begin
	  PGPFreeKeySet(KeySetChecked);
	  KeySetFiltered := nil;
	end;
      end;
    end;
  end
  else begin
    Result := kPGPError_X509NeededCertNotAvailable;
    PGPFreeKeySet(KeySetFiltered);
    KeySetFiltered := nil;
  end;
end;

// search the specified keyset for a certificate with the specified CertificateIASN
function GetCertByCertIASN(KeySetToSearch: pPGPKeySet; const CertificateIASN: String;
			   var SignKey: pPGPKeyDBObj): pPGPKeyDBObj;
var
  KeyIter	: pPGPKeyIter;
  Certificate	: pPGPKeyDBObj;
begin
  Result := nil;
  SignKey := nil;
  KeyIter := nil;
  if PGPNewKeyIterFromKeySet(KeySetToSearch, KeyIter) <> 0 then Exit;
  try
    while FindCertificate(KeyIter, KeySetToSearch, Certificate, SignKey) = 0 do begin
      if GetCertPropIASN(Certificate) = CertificateIASN then begin
	Result := Certificate;
	Break;
      end;
    end;
  finally
    PGPFreeKeyIter(KeyIter);
  end;
end;

// retrieve a set containing a single certificate
function GetCertSetByCertIASN(Context: pPGPContext; KeySetToSearch: pPGPKeySet; const CertificateIASN: String;
			      var SingleCertSet: pPGPKeySet; var SignKey: pPGPKeyDBObj): Integer;
var
  Certificate	: pPGPKeyDBObj;
begin
  Certificate := GetCertByCertIASN(KeySetToSearch, CertificateIASN, SignKey);
  if Certificate <> nil then
    Result := PGPNewSingletonKeySet(Certificate, SingleCertSet)
  else Result := kPGPError_X509NeededCertNotAvailable;
end;

// a simple file read function
function ReadDataFromFile(const CertFile: String; var CertData: String): Integer;
var
  CertStream	: TMemoryStream;
begin
  Result := 0;
  CertData := '';
  try
    CertStream := TMemoryStream.Create;
    if CertStream <> nil then with CertStream do begin
      try
	LoadFromFile(CertFile);
	if Size <> 0 then begin
	  SetLength(CertData, Size);
	  ReadBuffer(pointer(CertData)^, Size);
	end;
      finally
	CertStream.Free;
      end;
    end;
  except
    Result := kPGPError_ReadFailed;
  end;
end;

// a simple file write function
function WriteDataToFile(const CertData, CertFile: String): Integer;
var
  CertStream	: TMemoryStream;
begin
  Result := 0;
  try
    CertStream := TMemoryStream.Create;
    if CertStream <> nil then with CertStream do begin
      try
	WriteBuffer(pointer(CertData)^, Length(CertData));
	SaveToFile(CertFile);
      finally
	CertStream.Free;
      end;
    end
    else Result := kPGPError_OutOfMemory;
  except
    Result := kPGPError_WriteFailed;
  end;
end;

// copy (binary) data to string
function CopyDataToString(const InData: PChar; var OutData: String; Size: DWord): Integer;
begin
  Result := 0;
  try
    SetLength(OutData, Size);
    Move(InData^, pointer(OutData)^, Size);
  except
    on EOutOfMemory do begin
      Result := kPGPError_OutOfMemory;
      OutData := '';
    end;
  end;
end;

// dialog for retrieving a signing key and it's passphrase
function X509GetKeyAndPassphraseDialog(Context: pPGPContext; KeySetMain: pPGPKeySet;
				       var SignKey, SignCert: pPGPKeyDBObj; var Passphrase: PChar;
				       const SignCertIASN, SignKeyAndPassPrompt: String; WinHandle: Integer): Integer;
var
  KeyDBTemp	: pPGPKeyDB;
  KeySetTemp	: pPGPKeySet;
  Dummy		: PGPUInt32;
begin
  SignKey := nil;
  SignCert := nil;
  if SignKeyAndPassPrompt <> '' then begin // prompt for signing key and passphrase
    Passphrase := nil;
    Result := PGPNewKeyDB(Context, KeyDBTemp);
    if Result <> 0 then Exit;
    try
      Result := PGPCopyKeys(KeySetMain, KeyDBTemp, KeySetTemp);
      if Result <> 0 then Exit;
      try
	Result := SigningPassphraseDialog(Context, KeySetTemp, SignKey, Passphrase, SignCertIASN,
					  true, soNo, Dummy, Dummy, Dummy, SignKeyAndPassPrompt, WinHandle);
	if Result <> 0 then begin
	  if Result = kPGPError_SecretKeyNotFound then Result := kPGPError_X509NeededCertNotAvailable;
	  Exit;
	end;
	Result := GetKeyFromNewSet(SignKey, KeySetMain, SignKey);
	if Result = 0 then SignCert := GetValidX509CertFromKey(SignKey, nil, KeySetMain);
      finally
	PGPFreeKeySet(KeySetTemp);
      end;
    finally
      PGPFreeKeyDB(KeyDBTemp);
    end;
  end
  else begin // do we have a hex ID or an IASN?
    Result := GetKeyByHexID(KeySetMain, SignCertIASN, SignKey);
    if Result <> 0 then begin
      SignCert := GetCertByCertIASN(KeySetMain, SignCertIASN, SignKey);
      if SignCert <> nil then Result := 0;
    end
    else begin
      if GetKeyPropCanSign(SignKey) then
	SignCert := GetValidX509CertFromKey(SignKey, nil, KeySetMain)
      else Result := kPGPError_KeyUnusableForSignature;
    end;
  end;
  if Result <> 0 then Exit;
  if SignCert = nil then Result := kPGPError_X509NeededCertNotAvailable;
end;

// if CertAttributes' CommonName or Email are undefined they are taken from the CertKey's primary user ID
function CreateX509CertificateRequest(const CertKeyAndPassPrompt, CertKeyHexID: String; Passphrase: PChar;
				      var CertAttributes: TX509CertAttributes; Armor: Longbool;
				      var Request: String; var ReqKeyProps: TKeyPropsRec;
				      KeyPropFlags, WinHandle: Integer): Integer;
var
  Context	: pPGPContext;
  KeySetMain	: pPGPKeySet;
  PassphraseBuf	: PChar;
  KeyToCertify	: pPGPKeyDBObj;
  Dummy		: pPGPKeyDBObj;
  AttributeCount: Longint;
  AttributeList	: TX509CertAttrArray;
  OptionList	: pPGPOptionList;
  CertOutput	: PChar;
  CertLength	: PGPSize;
begin
  if PGP7X then begin
    Result := KeyRings.InitKeyRings(Context, KeySetMain);
    if Result <> 0 then Exit;
    try
      Result := X509GetKeyAndPassphraseDialog(Context, KeySetMain, KeyToCertify, Dummy, PassphraseBuf,
					      CertKeyHexID, CertKeyAndPassPrompt, WinHandle);
      if (Result <> 0) and (Result <> kPGPError_X509NeededCertNotAvailable) then Exit;
      Result := GetKeyProps(KeySetMain, KeyToCertify, KeyPropFlags, ReqKeyProps);
      if CertKeyAndPassPrompt = '' then PassphraseBuf := Passphrase;
      try
	with CertAttributes do begin
	  if (CommonName = '') or (Email = '') then begin
	    Result := GetPrimaryUserIDAsAttributes(KeyToCertify, CommonName, Email);
	    if Result <> 0 then Exit;
	  end;
	end;
	Result := GetAttributeListFromRecord(CertAttributes, AttributeList, AttributeCount);
	if Result <> 0 then Exit;
	OptionList := nil;
	Result := PGPBuildOptionList(Context, OptionList,
	  [
	    PGPOExportKey(Context, KeyToCertify),
	    PGPOPassphrase(Context, PassphraseBuf),
	    PGPOArmorOutput(Context, PGPBoolean(Armor) and 1),
	    PGPOExportFormat(Context, kPGPExportFormat_X509CertReq),
	    PGPOAttributeValue(Context, @AttributeList, AttributeCount),
	    PGPOAllocatedOutputBuffer(Context, CertOutput, $FFFFFFFF, CertLength)
	  ]);
      finally
	if CertKeyAndPassPrompt <> '' then PGPFreeData(PassphraseBuf);
      end;
      if Result <> 0 then Exit;
      try
	Result := PGPExport(Context, OptionList, PGPOLastOption(Context));
	if (Result = 0) and (CertLength <> 0) then begin
	  try
	    Result := CopyDataToString(CertOutput, Request, CertLength);
	    if Result <> 0 then Exit;
	  finally
	    PGPFreeData(CertOutput);
	  end;
	end
	else begin
	  Request := '';
	  Exit;
	end;
      finally
	PGPFreeOptionList(OptionList);
      end;
    finally
      KeyRings.FreeKeyRings;
      Result := TranslateX509ErrorToPGPError(Result);
    end;
  end
  else Result := kPGPError_LazyProgrammer;
end;

// specifying a unique CertAttributes' SerialNumber is mandatory, CommonName and Email are taken from the request
// if the SignCertIASN string contains a SignKeyHexID the key's last valid certificate will be used for certifying
function CreateX509CertificateFromRequest(const SignKeyAndPassPrompt, SignCertIASN: String; Passphrase: PChar;
					  var CertAttributes: TX509CertAttributes; ValidDays: Integer;
					  Armor: Longbool; const Request: String; var Certificate: String;
					  var CertPropsRec: TX509CertPropsRec; WinHandle: Integer): Integer;
var
  Context	: pPGPContext;
  KeySetMain	: pPGPKeySet;
  KeySetX509	: pPGPKeySet;
  KeySetX509Sign: pPGPKeySet;
  SignKey	: pPGPKeyDBObj;
  SignCert	: pPGPKeyDBObj;
  PassphraseBuf	: PChar;
  SNUnique	: Longbool;
  AttributeCount: Longint;
  AttributeList	: TX509CertAttrArray;
  OptionList	: pPGPOptionList;
  KeyDBTemp	: pPGPKeyDB;
  KeySetTemp	: pPGPKeySet;
  RequestCert	: pPGPKeyDBObj;
  RequestKey	: pPGPKeyDBObj;
  UserIDFound	: pPGPKeyDBObj;
  KeyFound	: pPGPKeyDBObj;
  UserID	: TUserID;
  IDSize	: PGPSize;
  RequestCertSet: pPGPKeySet;
  CertOutput	: PChar;
  CertLength	: PGPSize;
begin
  if PGP7X then begin
    Result := InitX509CertKeySet(Context, KeySetMain, KeySetX509);
    if Result <> 0 then Exit;
    try
      Result := GetValidX509CertifierSet(KeySetX509, KeySetX509Sign, false);
      if Result <> 0 then Exit;
      try
	Result := X509GetKeyAndPassphraseDialog(Context, KeySetX509Sign, SignKey, SignCert, PassphraseBuf,
						SignCertIASN, SignKeyAndPassPrompt, WinHandle);
      finally
	PGPFreeKeySet(KeySetX509Sign);
      end;
      if Result <> 0 then Exit;
      if SignKeyAndPassPrompt = '' then PassphraseBuf := Passphrase;
      Result := IsSerialNumberUnique(Context, KeySetMain, SignKey, CertAttributes.SerialNumber, SNUnique);
      if Result <> 0 then Exit;
      if not SNUnique then begin
	Result := kPGPError_DuplicateCert;
	Exit;
      end;
      try
	Result := GetAttributeListFromRecord(CertAttributes, AttributeList, AttributeCount);
	if Result <> 0 then Exit;
	OptionList := nil;
	Result := PGPBuildOptionList(Context, OptionList,
	  [
	    PGPOInputBuffer(Context, PChar(Request), Length(Request)),
	    PGPOAttributeValue(Context, @AttributeList, AttributeCount),
	    PGPOPassphrase(Context, PassphraseBuf),
	    PGPOExpiration(Context, ValidDays)
	  ]);
      finally
	if SignKeyAndPassPrompt <> '' then PGPFreeData(PassphraseBuf);
      end;
      if Result <> 0 then Exit;
      try
	Result := PGPNewKeyDB(Context, KeyDBTemp);
	if Result <> 0 then Exit;
	try
	  Result := PGPCopyKeys(KeySetX509, KeyDBTemp, KeySetTemp);
	  if Result <> 0 then Exit;
	  try
	    Result := PGPCreateX509CertificateFromRequest(SignCert, RequestCert, OptionList, PGPOLastOption(Context));
	    if Result <> 0 then Exit;
	    Result := GetX509CertPropsRec(RequestCert, CertPropsRec);
	    if Result <> 0 then Exit;
	    RequestKey := PGPPeekKeyDBObjKey(RequestCert);
	    if RequestKey = SignKey then begin
	      Result := kPGPError_X509SelfSignedCert;
	      if GetKeyFromNewSet(SignKey, KeySetTemp, SignKey) = 0 then begin
		RemoveKeyFromKeySet(RequestKey, KeySetX509);
		AddKeyToKeySet(SignKey, KeySetX509);
	      end;
	    end
	    else begin
	      UserIDFound := PGPPeekKeyDBObjUserID(RequestCert);
	      if (GetKeyFromNewSet(SignKey, KeySetTemp, SignKey) = 0)
	      and (GetKeyFromNewSet(RequestKey, KeySetTemp, KeyFound) = 0)
	      and (PGPGetKeyDBObjDataProperty(UserIDFound, kPGPUserIDProperty_Name, @UserID, SizeOf(TUserID), IDSize) = 0)
	      and HasUserIDCertifierCert(KeyFound, UserID, KeySetTemp, SignKey) then begin
		Result := kPGPError_DuplicateUserID;
		RemoveKeyFromKeySet(RequestKey, KeySetX509);
		AddKeyToKeySet(KeyFound, KeySetX509);
	      end;
	    end;
	  finally
	    PGPFreeKeySet(KeySetTemp);
	  end;
	finally
	  PGPFreeKeyDB(KeyDBTemp);
	end;
	if Result <> 0 then Exit;
	Result := KeyRings.UpdateKeyRings;
	if Result <> 0 then Exit;
	Result := PGPNewSingletonKeySet(RequestCert, RequestCertSet);
	if Result <> 0 then Exit;
	try
	  Result := PGPBuildOptionList(Context, OptionList,
	    [
	      PGPOExportKeySet(Context, RequestCertSet),
	      PGPOExportFormat(Context, kPGPExportFormat_X509Cert),
	      PGPOArmorOutput(Context, PGPBoolean(Armor) and 1),
	      PGPOAllocatedOutputBuffer(Context, CertOutput, $FFFFFFFF, CertLength)
	    ]);
	  if Result <> 0 then Exit;
	  try
	    Result := PGPExport(Context, OptionList, PGPOLastOption(Context));
	    if Result <> 0 then Exit;
	    try
	      Result := CopyDataToString(CertOutput, Certificate, CertLength);
	    finally
	      PGPFreeData(CertOutput);
	    end;
	  finally
	    PGPFreeOptionList(OptionList);
	  end;
	finally
	  PGPFreeKeySet(RequestCertSet);
	end;
      finally
	PGPFreeOptionList(OptionList);
      end;
    finally
      FinitX509CertKeySet(KeySetX509);
      Result := TranslateX509ErrorToPGPError(Result);
    end;
  end
  else Result := kPGPError_LazyProgrammer;
end;

// if the RevokeCertIASN string contains valid CertKeyHexID(s) all certificates on the respective key(s) will be revoked
// if no RevokeCertIASN string is specified all (non-self) certificates issued by the respective SignKey will be revoked
// if the SignCertIASN string contains a SignKeyHexID it's last valid certificate will be used for revoking certificates
// if SignCertIASN isn't provided and RevokeCertIASN contains a unique IASN the certifying key can be found automatically
function CreateX509CertificateRevocationList(const CertKeyPrompt, SignKeyAndPassPrompt, RevokeCertIASN, SignCertIASN: String;
					     Passphrase: PChar; ValidDays: Integer; var CertPropsList: TX509CertPropsList;
					     WinHandle: Integer): Integer;
var
  Context	: pPGPContext;
  KeySetMain	: pPGPKeySet;
  KeySetX509	: pPGPKeySet;
  KeySetX509Sign: pPGPKeySet;
  SignKey	: pPGPKeyDBObj;
  SignCert	: pPGPKeyDBObj;
  PassphraseBuf	: PChar;
  SignKeyCertSet: pPGPKeySet;
  RevokeCertSet	: pPGPKeySet;
  SignKeyFound	: pPGPKeyDBObj;
  OptionList	: pPGPOptionList;
begin
  if PGP7X then begin
    Result := InitX509CertKeySet(Context, KeySetMain, KeySetX509);
    if Result <> 0 then Exit;
    try
      Result := GetValidX509CertifierSet(KeySetX509, KeySetX509Sign, true);
      if Result <> 0 then Exit;
      if (SignKeyAndPassPrompt <> '') or (SignCertIASN <> '') then begin
	try
	  Result := X509GetKeyAndPassphraseDialog(Context, KeySetX509Sign, SignKey, SignCert, PassphraseBuf,
						  SignCertIASN, SignKeyAndPassPrompt, WinHandle);
	finally
	  PGPFreeKeySet(KeySetX509Sign);
	end;
	if Result <> 0 then Exit;
	if not GetCertPropIsRoot(SignCert) then begin
	  Result := kPGPError_MissingX509Certificate;
	  Exit;
	end;
      end
      else SignCert := nil;
      if SignKeyAndPassPrompt = '' then PassphraseBuf := Passphrase;
      if CertKeyPrompt <> '' then begin // prompt for certified key(s)
	Result := GetCertKeySetBySignKey(Context, KeySetX509, SignKey, true, true, SignKeyCertSet);
	if Result <> 0 then Exit;
	try
	  Result := SelectX509CertifiedKeysDialog(Context, SignKeyCertSet, RevokeCertSet, CertKeyPrompt, false, WinHandle);
	finally
	  PGPFreeKeySet(SignKeyCertSet);
	end;
      end
      else begin
	if RevokeCertIASN <> '' then begin // do we have hex ID(s) or an IASN?
	  Result := GetKeySetByAnyIDs(Context, KeySetX509, RevokeCertIASN, RevokeCertSet);
	  if Result <> 0 then begin
	    Result := GetCertSetByCertIASN(Context, KeySetX509, RevokeCertIASN, RevokeCertSet, SignKeyFound);
	    if (Result = 0) and (SignCert = nil) then begin
	      SignCert := GetValidX509CertFromKey(SignKeyFound, SignKeyFound, KeySetX509);
	      if SignCert <> nil then SignKey := SignKeyFound;
	    end;
	  end;
	end
	else begin // get key(s) certified by signing key
	  Result := GetCertKeySetBySignKey(Context, KeySetX509, SignKey, false, true, RevokeCertSet);
	end;
      end;
      if Result <> 0 then Exit;
      try
	try
	  OptionList := nil;
	  Result := PGPBuildOptionList(Context, OptionList,
	    [
	      PGPOPassphrase(Context, PassphraseBuf),
	      PGPOExpiration(Context, ValidDays)
	    ]);
	  if Result <> 0 then Exit;
	  try
	    Result := PGPCreateX509CRL(SignCert, RevokeCertSet, OptionList, PGPOLastOption(Context));
	    if Result <> 0 then Exit;
	    Result := KeyRings.UpdateKeyRings;
	    if Result <> 0 then Exit;
	    Result := GetX509CertSetProps(RevokeCertSet, KeySetX509, SignKey, CertPropsList);
	    if Result > 0 then Result := 0;
	  finally
	    PGPFreeOptionList(OptionList);
	  end;
	finally
	  if SignKeyAndPassPrompt <> '' then PGPFreeData(PassphraseBuf);
	end;
      finally
	PGPFreeKeySet(RevokeCertSet);
      end;
    finally
      FinitX509CertKeySet(KeySetX509);
      Result := TranslateX509ErrorToPGPError(Result);
    end;
  end
  else Result := kPGPError_LazyProgrammer;
end;

// dialog for retrieving a single (to get) certified key
function X509GetKeyDialog(Context: pPGPContext; KeySetMain: pPGPKeySet; var CertKey: pPGPKeyDBObj;
			  const CertKeyPrompt: String; WinHandle: Integer): Integer;
var
  CertKeySet	: pPGPKeySet;
begin
  Result := SelectX509CertifiedKeysDialog(Context, KeySetMain, CertKeySet, CertKeyPrompt, true, WinHandle);
  if Result <> 0 then Exit;
  try
    Result := GetKeyFromKeySet(CertKeySet, CertKey);
  finally
    PGPFreeKeySet(CertKeySet);
  end;
end;

// specifying a unique CertAttributes.SerialNumber is mandatory
// splitting a user ID into CommonName and Email will certify it
// self-signed certificates cannot be created for an existing user ID
function CreateX509Certificate(const CertKeyPrompt, SignKeyAndPassPrompt, CertKeyHexID, SignCertIASN: String;
			       Passphrase: PChar; var CertAttributes: TX509CertAttributes; ValidDays: Integer;
			       SelfSigned: Longbool; var CertPropsList: TX509CertPropsList;
			       WinHandle: Longint): Integer;
var
  Context	: pPGPContext;
  KeySetMain	: pPGPKeySet;
  KeySetX509	: pPGPKeySet;
  KeySetX509Sign: pPGPKeySet;
  SignKey	: pPGPKeyDBObj;
  SignCert	: pPGPKeyDBObj;
  PassphraseBuf	: PChar;
  SNUnique	: Longbool;
  KeySetFiltered: pPGPKeySet;
  KeyToCertify	: pPGPKeyDBObj;
  AttributeCount: Longint;
  AttributeList	: TX509CertAttrArray;
  OptionList	: pPGPOptionList;
  Certificate	: pPGPKeyDBObj;
begin
  if PGP7X then begin
    Result := InitX509CertKeySet(Context, KeySetMain, KeySetX509);
    if Result <> 0 then Exit;
    try
      if SelfSigned then
        KeySetX509Sign := KeySetMain
      else begin
	Result := GetValidX509CertifierSet(KeySetX509, KeySetX509Sign, false);
	if Result <> 0 then Exit;
      end;
      try
	Result := X509GetKeyAndPassphraseDialog(Context, KeySetX509Sign, SignKey, SignCert, PassphraseBuf,
						SignCertIASN, SignKeyAndPassPrompt, WinHandle);
      finally
	if SelfSigned then begin
	  if Result = kPGPError_X509NeededCertNotAvailable then Result := 0;
	end
	else PGPFreeKeySet(KeySetX509Sign);
      end;
      if Result <> 0 then Exit;
      if SignKeyAndPassPrompt = '' then PassphraseBuf := Passphrase;
      Result := IsSerialNumberUnique(Context, KeySetMain, SignKey, CertAttributes.SerialNumber, SNUnique);
      if Result <> 0 then Exit;
      if not SNUnique then begin
	Result := kPGPError_DuplicateCert;
	Exit;
      end;
      try
	if not SelfSigned then begin
	  GetKeySetWithoutSigner(Context, KeySetMain, SignKey, KeySetFiltered);
	  try
	    if CertKeyPrompt <> '' then
	      Result := X509GetKeyDialog(Context, KeySetFiltered, KeyToCertify, CertKeyPrompt, WinHandle)
	    else Result := GetKeyByHexID(KeySetFiltered, CertKeyHexID, KeyToCertify);
	  finally
	    if KeySetFiltered <> KeySetMain then PGPFreeKeySet(KeySetFiltered);
	  end;
	end
	else KeyToCertify := SignKey;
	if Result <> 0 then Exit;
	with CertAttributes do begin
	  if KeyToCertify <> SignKey then begin
	    if (CommonName = '') or (Email = '') then begin
	      Result := GetPrimaryUserIDAsAttributes(KeyToCertify, CommonName, Email);
	      if Result <> 0 then Exit;
	    end;
	    // as we already know which certifier key will be used we only have to check with the selected SignKey
	    if HasUserIDCertifierCert(KeyToCertify, CommonName + ' <' + Email + '>', KeySetMain, SignKey) then begin
	      Result := kPGPError_DuplicateUserID;
	      Exit;
	    end;
	  end
	  else begin
	    if (CommonName = '') or (Email = '') then begin
	      Result := kPGPError_InvalidDistinguishedName;
	      Exit;
	    end
	    else if HasKeyUserID(KeyToCertify, CommonName + ' <' + Email + '>') then begin
	      Result := kPGPError_DuplicateUserID;
	      Exit;
	    end;
	  end;
	end;
	Result := GetAttributeListFromRecord(CertAttributes, AttributeList, AttributeCount);
	if Result <> 0 then Exit;
	OptionList := nil;
	Result := PGPBuildOptionList(Context, OptionList,
	  [
	    PGPOAttributeValue(Context, @AttributeList, AttributeCount),
	    PGPOPassphrase(Context, PassphraseBuf),
	    PGPOExpiration(Context, ValidDays)
	  ]);
      finally
	if SignKeyAndPassPrompt <> '' then PGPFreeData(PassphraseBuf);
      end;
      if Result <> 0 then Exit;
      try
	if KeyToCertify = SignKey then
	  Result := PGPCreateSelfSignedX509Certificate(SignKey, Certificate, OptionList, PGPOLastOption(Context))
	else Result := PGPCreateX509Certificate(SignCert, KeyToCertify, Certificate, OptionList, PGPOLastOption(Context));
	if Result <> 0 then Exit;
	Result := KeyRings.UpdateKeyRings;
	if Result <> 0 then Exit;
	Result := GetX509CertPropsList(Certificate, CertPropsList);
      finally
	PGPFreeOptionList(OptionList);
      end;
    finally
      FinitX509CertKeySet(KeySetX509);
      Result := TranslateX509ErrorToPGPError(Result);
    end;
  end
  else Result := kPGPError_LazyProgrammer;
end;

// self-signed certificates can only be removed including the user ID they are attached to
// user IDs of certificates can only be removed if they don't have any other signatures assigned
function RemoveX509Certificate(const CertKeyPrompt, RemoveCertIASN: String; IncludeUserID: Longbool;
			       var CertPropsList: TX509CertPropsList; WinHandle: Integer): Integer;
var
  Context	: pPGPContext;
  KeySetMain	: pPGPKeySet;
  KeySetX509	: pPGPKeySet;
  RemoveCertKey : pPGPKeyDBObj;
  RemoveCert	: pPGPKeyDBObj;
  SignKey	: pPGPKeyDBObj;
  RemoveUserID	: pPGPKeyDBObj;
  SigRemoved	: Longbool;
  KeySet	: pPGPKeySet;
  KeyList	: pPGPKeyList;
  KeyIter	: pPGPKeyIter;
  UserIDFound	: pPGPKeyDBObj;
  SignatureFound: pPGPKeyDBObj;
begin
  if PGP7X then begin
    Result := InitX509CertKeySet(Context, KeySetMain, KeySetX509);
    if Result <> 0 then Exit;
    try
      RemoveCert := nil;
      if CertKeyPrompt <> '' then begin // prompt for a certified key
	Result := X509GetKeyDialog(Context, KeySetX509, RemoveCertKey, CertKeyPrompt, WinHandle);
	if Result = 0 then RemoveCert := GetValidX509CertFromKey(RemoveCertKey, nil, KeySetX509);
      end
      else begin // do we have a hex ID or an IASN?
	Result := GetKeyByHexID(KeySetX509, RemoveCertIASN, RemoveCertKey);
	if Result = 0 then
	  RemoveCert := GetValidX509CertFromKey(RemoveCertKey, nil, KeySetX509)
	else RemoveCert := GetCertByCertIASN(KeySetX509, RemoveCertIASN, SignKey);
      end;
      RemoveUserID := nil;
      if RemoveCert <> nil then begin
	RemoveUserID := PGPPeekKeyDBObjUserID(RemoveCert);
	Result := GetX509CertPropsList(RemoveCert, CertPropsList);
      end
      else Result := kPGPError_X509NeededCertNotAvailable;
      if Result <> 0 then Exit;
      with CertPropsList do begin
	if not X509CertProps[pred(Count)].x509IsRoot then begin
	  Result := PGPRemoveSig(RemoveCert);
	  if Result <> 0 then Exit;
	  SigRemoved := true;
	end
	else SigRemoved := false;
      end;
      if IncludeUserID then begin
	try
	  Result := GetSingleKeyDBObjIter(RemoveUserID, KeySet, KeyList, KeyIter);
	  while PGPKeyIterNextKeyDBObj(KeyIter, kPGPKeyDBObjType_UserID, UserIDFound) = 0 do begin
	    if UserIDFound = RemoveUserID then begin
	      if (PGPKeyIterRewindKeyDBObj(KeyIter, kPGPKeyDBObjType_Signature) <> 0)
	      or (PGPKeyIterNextKeyDBObj(KeyIter, kPGPKeyDBObjType_Signature, SignatureFound) <> 0)
	      or (
		(SignatureFound = RemoveCert) and
		(PGPKeyIterNextKeyDBObj(KeyIter, kPGPKeyDBObjType_Signature, SignatureFound) <> 0)
	      ) then begin
		Result := PGPRemoveUserID(RemoveUserID);
		if Result = 0 then SigRemoved := true;
	      end;
	      Break;
	    end;
	  end;
	finally
	  PGPFreeKeyIter(KeyIter);
	  PGPFreeKeyList(KeyList);
	  PGPFreeKeySet(KeySet);
	end;
	if SigRemoved then Result := KeyRings.UpdateKeyRings;
      end;
    finally
      FinitX509CertKeySet(KeySetX509);
      Result := TranslateX509ErrorToPGPError(Result);
    end;
  end
  else Result := kPGPError_LazyProgrammer;
end;

// import binary or armored standard X509 certificates or PGP keys to keyring(s)
function ImportX509Certificate(const CertKeyPrompt, CertData: String; FromFile: Longbool;
			       var CertPropsList: TX509CertPropsList; WinHandle: Integer): Integer;
var
  Context	: pPGPContext;
  KeySetMain	: pPGPKeySet;
  KeyPropsList	: TKeyPropsList;
  ImportKeySet	: pPGPKeySet;
begin
  if PGP7X then begin
    if CertData <> '' then begin
      KeyPropsList := nil;
      try
	Result := KeyImportDialog(CertKeyPrompt, CertData, KeyPropsList, FromFile,
				  IgnoreFlag_None, spgpKeyPropFlag_KeyID, WinHandle);
	if Result <> 0 then Exit;
	Result := KeyRings.InitKeyRings(Context, KeySetMain);
	if Result <> 0 then Exit;
	try
	  Result := GetKeySetByAnyIDs(Context, KeySetMain, KeyPropsList.CommaText, ImportKeySet);
	  if Result <> 0 then Exit;
	  try
	    Result := GetX509CertSetProps(ImportKeySet, KeySetMain, nil, CertPropsList);
	    if Result > 0 then Result := 0;
	  finally
	    PGPFreeKeySet(ImportKeySet);
	  end;
	finally
	  KeyRings.FreeKeyRings;
	end;
      finally
	KeyPropsList.Free;
      end;
    end
    else Result := kPGPError_InvalidInputFormat;
    Result := TranslateX509ErrorToPGPError(Result);
  end
  else Result := kPGPError_LazyProgrammer;
end;

// write certificate(s) and size information to chain
function WriteDataToChain(const CertData: PChar; var Chain: String;
			  var ChainPos: PGPSize; CertLen: PGPSize): Integer;
var
  pChain	: PChar;
begin
  Result := 0;
  pChain := pointer(Chain);
  if ChainPos = 0 then begin
    pChain[0]:=chr(CertLen shr 16);
    pChain[1]:=chr(CertLen shr 8);
    pChain[2]:=chr(CertLen);
    ChainPos := 3;
  end
  else begin
    SetLength(Chain, ChainPos + CertLen + 3);
    pChain := pointer(Chain);
    pChain[ChainPos + 0]:=chr(CertLen shr 16);
    pChain[ChainPos + 1]:=chr(CertLen shr 8);
    pChain[ChainPos + 2]:=chr(CertLen);
    Move(CertData^, pChain[ChainPos + 3], CertLen);
    inc(ChainPos, CertLen + 3);
    if ChainPos > $FFFFFF then begin
      Result := kPGPError_LDAPSizelimitExceeded;
      Chain := '';
    end;
  end;
end;

// create a certificate chain from any keyset
function CreateCertChainFromKeySet(CertKeySet: pPGPKeySet; var CertChain: String): Integer;
var
  KeyCount	: PGPUInt32;
  KeyIter	: pPGPKeyIter;
  CertList	: TStringList;
  Certificate	: pPGPKeyDBObj;
  KeyFound	: pPGPKeyDBObj;
  CreaTime	: PGPTime;
  ChainPos	: PGPSize;
  CertData	: TX509CertData;
  CertIndex	: Integer;
  CertLength	: PGPSize;
begin
  Result := PGPCountKeys(CertKeySet, KeyCount);
  if Result <> 0 then Exit;
  if KeyCount = 0 then begin
    Result := kPGPError_X509NeededCertNotAvailable;
    Exit;
  end;
  Result := PGPNewKeyIterFromKeySet(CertKeySet, KeyIter);
  if Result <> 0 then Exit;
  try
    CertList := TStringList.Create;
    if CertList <> nil then begin
      try
	while FindCertificate(KeyIter, CertKeySet, Certificate, KeyFound) = 0 do begin
	  Result := PGPGetKeyDBObjTimeProperty(Certificate, kPGPSigProperty_Creation, CreaTime);
	  if Result <> 0 then Exit;
	  CertList.AddObject(IntToHex(CreaTime, 8), TObject(Certificate));
	end;
	CertList.Sort;
	ChainPos := 0;
	CertData[0] := #0;
	SetLength(CertChain, 3);
	Result := WriteDataToChain(CertData, CertChain, ChainPos, 0);
	if Result <> 0 then Exit;
	CertIndex := pred(CertList.Count);
	while CertIndex >= 0 do begin
	  Result := PGPGetKeyDBObjDataProperty(pPGPKeyDBObj(CertList.Objects[CertIndex]), kPGPSigProperty_X509Certificate,
					       @CertData, MAX_509_CERT_SIZE, CertLength);
	  if Result <> 0 then Exit;
	  try
	    Result := WriteDataToChain(CertData, CertChain, ChainPos, CertLength);
	    if Result <> 0 then Exit;
	  except
	    on EOutOfMemory do begin
	      Result := kPGPError_OutOfMemory;
	      CertChain := '';
	      Exit;
	    end;
	  end;
	  dec(CertIndex);
	end;
	if ChainPos > 3 then begin
	  CertLength := 0;
	  CertData[0] := #0;
	  WriteDataToChain(CertData, CertChain, CertLength, ChainPos - 3);
	end
	else begin
	  Result := kPGPError_X509NeededCertNotAvailable;
	  Exit;
	end;
      finally
	CertList.Free;
      end;
    end
    else Result := kPGPError_OutOfMemory;
  finally
    PGPFreeKeyIter(KeyIter);
  end;
end;

// either export a single certificate or a (binary) certificate chain with parent certificate(s) included
function ExportX509Certificate(const CertKeyPrompt, ExportCertIASN: String; Armor, Chain, ToFile: Longbool;
			       var CertData: String; var CertPropsList: TX509CertPropsList; WinHandle: Integer): Integer;
var
  Context	: pPGPContext;
  KeySetMain	: pPGPKeySet;
  KeySetX509	: pPGPKeySet;
  ExportCert	: pPGPKeyDBObj;
  ExportCertKey	: pPGPKeyDBObj;
  SignKey	: pPGPKeyDBObj;
  CertFile	: String;
  ExportCertSet	: pPGPKeySet;
  OptionList	: pPGPOptionList;
  CertOutput	: PChar;
  CertLength	: PGPSize;
  CertSigner	: pPGPKeyDBObj;
begin
  if PGP7X then begin
    Result := InitX509CertKeySet(Context, KeySetMain, KeySetX509);
    if Result <> 0 then Exit;
    try
      ExportCert := nil;
      if CertKeyPrompt <> '' then begin // prompt for a certified key
	Result := X509GetKeyDialog(Context, KeySetX509, ExportCertKey, CertKeyPrompt, WinHandle);
	if Result = 0 then ExportCert := GetValidX509CertFromKey(ExportCertKey, nil, KeySetX509);
      end
      else begin // do we have a hex ID or an IASN?
	Result := GetKeyByHexID(KeySetX509, ExportCertIASN, ExportCertKey);
	if Result = 0 then
	  ExportCert := GetValidX509CertFromKey(ExportCertKey, nil, KeySetX509)
	else ExportCert := GetCertByCertIASN(KeySetX509, ExportCertIASN, SignKey);
      end;
      if ExportCert <> nil then begin
	ExportCertKey := PGPPeekKeyDBObjKey(ExportCert);
	Result := 0;
      end
      else Result := kPGPError_X509NeededCertNotAvailable;
      if Result <> 0 then Exit;
      if ToFile then CertFile := CertData;
      ExportCertSet := nil;
      try
	if not Chain then begin
	  Result := PGPNewSingletonKeySet(ExportCert, ExportCertSet);
	  if Result <> 0 then Exit;
	  Result := PGPBuildOptionList(Context, OptionList,
	    [
	      PGPOExportKeySet(Context, ExportCertSet),
	      PGPOExportFormat(Context, kPGPExportFormat_X509Cert),
	      PGPOArmorOutput(Context, PGPBoolean(Armor) and 1),
	      PGPOAllocatedOutputBuffer(Context, CertOutput, $FFFFFFFF, CertLength)
	    ]);
	  if Result <> 0 then Exit;
	  try
	    Result := PGPExport(Context, OptionList, PGPOLastOption(Context));
	    if Result <> 0 then Exit;
	    try
	      Result := CopyDataToString(CertOutput, CertData, CertLength);
	      if Result <> 0 then Exit;
	      Result := GetX509CertPropsList(ExportCert, CertPropsList);
	    finally
	      PGPFreeData(CertOutput);
	    end;
	  finally
	    PGPFreeOptionList(OptionList);
	  end;
	end
	else begin
	  Result := PGPNewEmptyKeySet(PGPPeekKeySetKeyDB(KeySetX509), ExportCertSet);
	  if Result <> 0 then Exit;
	  repeat
	    ExportCert := GetValidX509CertFromKey(ExportCertKey, nil, KeySetX509);
	    if ExportCert <> nil then begin
	      Result := PGPAddKey(ExportCertKey, ExportCertSet);
	      if Result <> 0 then Exit;
	      Result := PGPGetSigCertifierKey(ExportCert, KeySetX509, CertSigner);
	      if Result <> 0 then Exit;
	      if CertSigner <> ExportCertKey then
		ExportCertKey := CertSigner
	      else Chain := false;
	    end
	    else begin
	      Result := kPGPError_X509NeededCertNotAvailable;
	      Exit;
	    end;
	  until not Chain;
	  Result := CreateCertChainFromKeySet(ExportCertSet, CertData);
	  if Result <> 0 then Exit;
	  Result := GetX509CertSetProps(ExportCertSet, KeySetX509, nil, CertPropsList);
	  if Result > 0 then Result := 0;
	end;
      finally
	PGPFreeKeySet(ExportCertSet);
      end;
      if ToFile then begin
	Result := WriteDataToFile(CertData, CertFile);
	CertData := CertFile;
      end;
    finally
      FinitX509CertKeySet(KeySetX509);
      if Result <> 0 then CertData := '';
      Result := TranslateX509ErrorToPGPError(Result);
    end;
  end
  else Result := kPGPError_LazyProgrammer;
end;

// create a temporary keyDB for checking certificates
function GetX509CertImportDB(Context: pPGPContext; const CertData: String;
			     var CertImportDB: pPGPKeyDB; FromFile: Longbool): Integer;
var
  CertInput	: String;
  LoopCount	: Integer;
  OptionList	: pPGPOptionList;
begin
  Result := 0;
  OptionList := nil;
  CertImportDB := nil;
  if FromFile then
    Result := ReadDataFromFile(CertData, CertInput)
  else CertInput := CertData;
  if Result <> 0 then Exit;
  LoopCount := 0;
  repeat
    inc(LoopCount);
    if Result <> kPGPError_InvalidPKCS7Encoding then begin
      Result := PGPBuildOptionList(Context, OptionList,
	[
	  PGPOInputFormat(Context, kPGPInputFormat_X509DataInPKCS7),
	  PGPOInputBuffer(Context, PChar(CertInput), Length(CertInput))
	]);
    end
    else begin
      Result := PGPBuildOptionList(Context, OptionList,
	[
	  PGPOInputFormat(Context, kPGPInputFormat_PEMEncodedX509Cert),
	  PGPOInputBuffer(Context, PChar(CertInput), Length(CertInput))
	]);
    end;
    if Result <> 0 then Exit;
    try
      Result := PGPImport(Context, CertImportDB, OptionList, PGPOLastOption(Context));
    finally
      PGPFreeOptionList(OptionList);
      OptionList := nil;
    end;
  until (Result = 0) or (LoopCount = 2);
  if Result < 0 then Result := kPGPError_X509CertificateParseError;
end;

// the required certificate chain can be created using ExportX509Certificate() with the Chain parameter enabled
function VerifyX509CertificateChain(const CertChain: String; FromFile: Longbool): Integer;
var
  Context	: pPGPContext;
  KeySetMain	: pPGPKeySet;
  KeySetX509	: pPGPKeySet;
  CertDataBuf	: String;
  RootCertsBuf	: String;
  CertDataDB	: pPGPKeyDB;
begin
  if PGP7X then begin
    Result := InitX509CertKeySet(Context, KeySetMain, KeySetX509);
    if Result <> 0 then Exit;
    try
      Result := CreateCertChainFromKeySet(KeySetX509, RootCertsBuf);
      if Result <> 0 then Exit;
      if FromFile then
	Result := ReadDataFromFile(CertChain, CertDataBuf)
      else CertDataBuf := CertChain;
      if Result <> 0 then Exit;
      Result := PGPVerifyX509CertificateChain(Context, PChar(CertDataBuf), PChar(RootCertsBuf));
      if Result <> 0 then begin
	if GetX509CertImportDB(Context, CertDataBuf, CertDataDB, false) = 0 then begin
	  Result := CreateCertChainFromKeySet(PGPPeekKeyDBRootKeySet(CertDataDB), CertDataBuf);
	  if Result = 0 then Result := PGPVerifyX509CertificateChain(Context, PChar(CertDataBuf), PChar(RootCertsBuf));
	end;
      end;
    finally
      FinitX509CertKeySet(KeySetX509);
      Result := TranslateX509ErrorToPGPError(Result);
    end;
  end
  else Result := kPGPError_LazyProgrammer;
end;

// eight routines for X509 specific ASN.1 decoding
function OIDCheck(const OID, ASNData: PChar; Pos: DWord): Longbool; register; assembler;
asm	// EAX = @OID, EDX = @ASNData, ECX = Pos
  OR	EAX,EAX
  JE	@ERROR
  OR	EDX,EDX
  JE	@ERROR
  CMP	BYTE PTR [EAX],0
  JE	@ERROR
  ADD	EDX,ECX
  CMP	BYTE PTR [EDX],0
  JE	@ERROR
  PUSH	EBX
  DEC	EAX
  DEC	EDX
  XOR	EBX,EBX
  @LOOP:
  INC	EBX
  JO	@ERROR
  MOV	CL,[EAX + EBX]
  OR	CL,CL
  JE	@END
  CMP	[EDX + EBX],CL
  JE	@LOOP
  POP	EBX
  @ERROR:
  XOR	EAX,EAX
  RET
  @END:
  POP	EBX
  MOV	EAX,TRUE
end;

function GetTypeLength(TaggedData: PChar; Size: DWord; var Pos: DWord): DWord;
var
  SizeLen, Index: DWord;
begin
  Result := 0;
  inc(Pos);
  if Pos < Size then begin
    if ord(TaggedData[Pos]) < $80 then
      Result := ord(TaggedData[Pos])
    else begin
      SizeLen := ord(TaggedData[Pos]) and $7F;
      if (SizeLen <= 4) and (Pos + SizeLen < Size) then begin
	for Index := 1 to SizeLen do begin
	  inc(Pos);
	  Result := Result shl 8 + ord(TaggedData[Pos])
	end;
      end;
    end;
    inc(Pos);
  end;
end;

function FindObjectID(const OID, ASNData: PChar; Size: DWord; var Pos: DWord): DWord;
var
  OIDLen: DWord;
begin
  Result := 0;
  OIDLen := StrLen(OID);
  while Pos + OIDLen <= Size do begin
    if OIDCheck(OID, ASNData, Pos) then begin
      Result := GetTypeLength(ASNData, Size, Pos);
      Break;
    end
    else if ASNData[Pos] < #$30 then begin
      Result := GetTypeLength(ASNData, Size, Pos);
      inc(Pos, Result);
      Continue;
    end
    else if ASNData[Pos] > #$31 then begin
      Result := GetTypeLength(ASNData, Size, Pos);
      inc(Pos, Result);
      Continue;
    end;
    Result := GetTypeLength(ASNData, Size, Pos);
  end;
end;

function GetSerialNumber(const ASNData: PChar; Size: DWord): String;
const
  oidSN	= #$06#$03#$55#$04#$05#$13#0;
var
  Pos, Len: DWord;
begin
  Pos := 0;
  Result := '';
  Len := FindObjectID(oidSN, ASNData, Size, Pos);
  if Len <> 0 then begin
    inc(Pos, Len);
    Len := GetTypeLength(ASNData, Size, Pos);
    if Len <> 0 then begin
      SetLength(Result, Len);
      Move(ASNData[Pos], pointer(Result)^, Len);
    end;
  end;
end;

function BytesToString(const Bytes: PChar; Size: PGPSize): String; register; assembler;
asm	// EAX = @Bytes, EDX = Size, ECX = @Result
  PUSH	EBX
  PUSH	EDI
  PUSH	ESI
  MOV	ESI,EAX
  MOV	EDI,ECX
  MOV	EAX,EDX
  MOV	EBX,EDX
  SHL	EAX,1
  DEC	EDX
  ADD	EBX,EAX
  PUSH	EDX
  DEC	EBX
  MOV	EAX,EDI
  CALL	SYSTEM.@LSTRCLR
  MOV	EAX,EBX
  CALL	SYSTEM.@NEWANSISTRING
  MOV	[EDI],EAX
  MOV	EDI,EAX
  SUB	EBX,2
  POP	ECX
  @LOOP:
  XOR	EAX,EAX
  MOV	AL,[ESI + ECX]
  MOV	EDX,EAX
  AND	EAX,0Fh
  SHR	EDX,4
  MOV	AH,BYTE PTR[@HEXCHARS + EAX]
  MOV	AL,BYTE PTR[@HEXCHARS + EDX]
  MOV	[EDI + EBX],AX
  SUB	EBX,3
  DEC	ECX
  JS	@END
  MOV	BYTE PTR[EDI + EBX + 2],' '
  JMP	@LOOP
  @END:
  POP	ESI
  POP	EDI
  POP	EBX
  RET
  @HEXCHARS:
  DB	'0123456789ABCDEF';
end;

function GetCertificateID(const ASNData: PChar; Size: DWord): String;
const
  oidID = #$02#0;
var
  Pos, Len: DWord;
begin
  Pos := 0;
  Result := '';
  Len := FindObjectID(oidID, ASNData, Size, Pos);
  if Len <> 0 then Result := BytesToString(@ASNData[Pos], Len);
end;

function GetAlgorithms(const ASNData: PChar; Size: DWord; var KeyAlg: TKeyAlgorithm; var HashAlg: THashAlgorithm): Integer;
const
  oidRSAFilter = #$06#$09#$2A#$86#$48#$86#$F7#$0D#$01#$01#0;
  oidDSAFilter = #$06#$07#$2A#$86#$48#$CE#$38#$04#0;
  idRSA = 1;
  idMD2WithRSA = 2;
  idMD5WithRSA = 4;
  idSHA1WithRSA = 5;
  idSHA256WithRSA = 11;
  idSHA384WithRSA = 12;
  idSHA512WithRSA = 13;
  idDSA = 2;
  idSHA1WithDSA = 3;
var
  Pos, Len: DWord;
begin
  Pos := 0;
  Result := 0;
  KeyAlg := KeyAlgorithm_Invalid;
  HashAlg := HashAlgorithm_Invalid;
  GetTypeLength(ASNData, Size, Pos);
  Pos := Pos + GetTypeLength(ASNData, Size, Pos);
  GetTypeLength(ASNData, Size, Pos);
  Len := GetTypeLength(ASNData, Size, Pos);
  if Len in [7, 9] then begin
    Result := ord(ASNData[Pos + pred(Len)]);
    if Len = 9 then begin
      KeyAlg := KeyAlgorithm_RSA;
      case Result of
	idMD2WithRSA: HashAlg := HashAlgorithm_MD2;
	idMD5WithRSA: HashAlg := HashAlgorithm_MD5;
	idSHA1WithRSA: HashAlg := HashAlgorithm_SHA;
	idSHA256WithRSA: HashAlg := HashAlgorithm_SHA256;
	idSHA384WithRSA: HashAlg := HashAlgorithm_SHA384;
	idSHA512WithRSA: HashAlg := HashAlgorithm_SHA512;
      end;
    end
    else if Len = 7 then begin
      KeyAlg := KeyAlgorithm_DSS;
      case Result of
	idSHA1WithDSA: HashAlg := HashAlgorithm_SHA;
      end;
    end;
  end;
end;

procedure GetCertIssuerInfos(Cert: pPGPKeyDBObj; var CertID: String; var KeyAlg: TKeyAlgorithm; var HashAlg: THashAlgorithm);
var
  CertData	: TX509CertData;
  CertSize	: PGPSize;
begin
  if PGPGetKeyDBObjDataProperty(Cert, kPGPSigProperty_X509Certificate, @CertData, MAX_509_CERT_SIZE, CertSize) = 0 then begin
    CertID := GetCertificateID(CertData, CertSize);
    GetAlgorithms(CertData, CertSize, KeyAlg, HashAlg);
  end;
end;

// the rest is about retrieving certificate properties
function GetCertPropIASN(Cert: pPGPKeyDBObj): String;
var
  IDData	: TX509CertData;
  IDSize	: PGPSize;
begin
  Result := '';
  if PGPGetKeyDBObjDataProperty(Cert, kPGPSigProperty_X509IASN, @IDData, MAX_509_CERT_SIZE, IDSize) = 0 then begin
    SetLength(Result, IDSize);
    Move(IDData, pointer(Result)^, IDSize);
  end;
end;

function GetCertPropIssuerName(Cert: pPGPKeyDBObj): String;
var
  INData	: TX509CertData;
  INSize	: PGPSize;
begin
  Result := '';
  if PGPGetKeyDBObjDataProperty(Cert, kPGPSigProperty_X509IssuerLongName, @INData, MAX_509_CERT_SIZE, INSize) = 0 then begin
    Result := INData;
  end;
end;

function GetCertPropIssuerSerialNumber(Cert: pPGPKeyDBObj): String;
var
  ISNData	: TX509CertData;
  ISNSize	: PGPSize;
begin
  Result := '';
  if PGPGetKeyDBObjDataProperty(Cert, kPGPSigProperty_X509IASN, @ISNData, MAX_509_CERT_SIZE, ISNSize) = 0 then begin
    Result := GetSerialNumber(ISNData, ISNSize);
  end;
end;

function GetCertPropOwnerKeyID(Cert: pPGPKeyDBObj): String;
var
  PGPKeyID	: TPGPKeyID7;
  KeyID		: TKeyID;
begin
  Result := '';
  if (PGPGetKeyIDFromKey(PGPPeekKeyDBObjKey(Cert), PGPKeyID) = 0)
  and (PGPGetKeyIDString(PGPKeyID, kPGPKeyIDString_Full, KeyID) = 0) then begin
    Result := KeyID;
  end;
end;

function GetCertPropOwnerName(Cert: pPGPKeyDBObj): String;
var
  ONData	: TX509CertData;
  ONSize	: PGPSize;
begin
  Result := '';
  if PGPGetKeyDBObjDataProperty(Cert, kPGPSigProperty_X509LongName, @ONData, MAX_509_CERT_SIZE, ONSize) = 0 then begin
    Result := ONData;
  end;
end;

function GetCertPropOwnerSerialNumber(Cert: pPGPKeyDBObj): String;
var
  OSNData	: TX509CertData;
  OSNSize	: PGPSize;
begin
  Result := '';
  if PGPGetKeyDBObjDataProperty(Cert, kPGPSigProperty_X509DERDName, @OSNData, MAX_509_CERT_SIZE, OSNSize) = 0 then begin
    Result := GetSerialNumber(OSNData, OSNSize);
  end;
end;

// local time
function GetCertPropCertCreationTimeStr(Cert: pPGPKeyDBObj): String;
var
  CreaTime	: PGPTime;
begin
  Result := '';
  if PGPGetKeyDBObjTimeProperty(Cert, kPGPSigProperty_Creation, CreaTime) = 0 then begin
    Result := UnixTimeToLocalTimeStr(PGPGetStdTimeFromPGPTime(CreaTime));
  end;
end;

// local time
function GetCertPropCertExpirationTimeStr(Cert: pPGPKeyDBObj): String;
var
  ExpTime	: PGPTime;
  CreaTime	: PGPTime;
begin
  Result := '';
  if (PGPGetKeyDBObjTimeProperty(Cert, kPGPSigProperty_Expiration, ExpTime) = 0)
  and (PGPGetKeyDBObjTimeProperty(Cert, kPGPSigProperty_Creation, CreaTime) = 0) then begin
    if ExpTime > CreaTime then Result := UnixTimeToLocalTimeStr(PGPGetStdTimeFromPGPTime(ExpTime));
  end;
end;

// UTC/GMT Unix format seconds
function GetCertPropCertCreationTime(Cert: pPGPKeyDBObj): Integer;
var
  CreaTime	: PGPTime;
begin
  Result := 0;
  if PGPGetKeyDBObjTimeProperty(Cert, kPGPSigProperty_Creation, CreaTime) = 0 then begin
    Result := PGPGetStdTimeFromPGPTime(CreaTime);
  end;
end;

// UTC/GMT Unix format seconds
function GetCertPropCertExpirationTime(Cert: pPGPKeyDBObj): Integer;
var
  ExpTime	: PGPTime;
  CreaTime	: PGPTime;
begin
  Result := 0;
  if (PGPGetKeyDBObjTimeProperty(Cert, kPGPSigProperty_Expiration, ExpTime) = 0)
  and (PGPGetKeyDBObjTimeProperty(Cert, kPGPSigProperty_Creation, CreaTime) = 0) then begin
    if ExpTime > CreaTime then Result := PGPGetStdTimeFromPGPTime(ExpTime);
  end;
end;

function GetCertPropIsVerified(Cert: pPGPKeyDBObj): Longbool;
var
  Prop		: PGPBoolean;
begin
  Result := false;
  if PGPGetKeyDBObjBooleanProperty(Cert, kPGPSigProperty_IsVerified, Prop) = 0 then begin
    Result := boolean(Prop);
  end;
end;

function GetCertPropIsCorrupt(Cert: pPGPKeyDBObj): Longbool;
var
  Prop		: PGPBoolean;
begin
  Result := false;
  if PGPGetKeyDBObjBooleanProperty(Cert, kPGPSigProperty_IsNotCorrupt, Prop) = 0 then begin
    Result := not boolean(Prop);
  end;
end;

function GetCertPropIsRevoked(Cert: pPGPKeyDBObj): Longbool;
var
  Prop		: PGPBoolean;
begin
  Result := false;
  if PGPGetKeyDBObjBooleanProperty(Cert, kPGPSigProperty_IsRevoked, Prop) = 0 then begin
    Result := boolean(Prop);
  end;
end;

function GetCertPropIsExpired(Cert: pPGPKeyDBObj): Longbool;
var
  Prop		: PGPBoolean;
begin
  Result := false;
  if PGPGetKeyDBObjBooleanProperty(Cert, kPGPSigProperty_IsExpired, Prop) = 0 then begin
    Result := boolean(Prop);
  end;
end;

function GetCertPropIsRoot(Cert: pPGPKeyDBObj): Longbool;
var
  INData	: TX509CertData;
  INSize	: PGPSize;
  ONData	: TX509CertData;
  ONSize	: PGPSize;
  SignKey	: pPGPKeyDBObj;
begin
  Result := false;
  if (PGPGetKeyDBObjDataProperty(Cert, kPGPSigProperty_X509IssuerLongName, @INData, MAX_509_CERT_SIZE, INSize) = 0)
  and (PGPGetKeyDBObjDataProperty(Cert, kPGPSigProperty_X509LongName, @ONData, MAX_509_CERT_SIZE, ONSize) = 0)
  and (PGPGetSigCertifierKey(Cert, PGPPeekKeyDBRootKeySet(PGPPeekKeyDBObjKeyDB(Cert)), SignKey) = 0)
  and (PGPPeekKeyDBObjKey(Cert) = SignKey) and (INData = ONData) then begin
    Result := true;
  end;
end;

function GetCertPropIsValid(Cert: pPGPKeyDBObj): Longbool;
var
  NotCorrupt	: PGPBoolean;
  Expired	: PGPBoolean;
  Revoked	: PGPBoolean;
begin
  Result := false;
  if (PGPGetKeyDBObjBooleanProperty(Cert, kPGPSigProperty_IsNotCorrupt, NotCorrupt) = 0)
  and (PGPGetKeyDBObjBooleanProperty(Cert, kPGPSigProperty_IsExpired, Expired) = 0)
  and (PGPGetKeyDBObjBooleanProperty(Cert, kPGPSigProperty_IsRevoked, Revoked) = 0) then begin
    Result := boolean(NotCorrupt) and not (boolean(Expired) or boolean(Revoked));
  end;
end;

// check whether a certificate with the given serial number already exists on the signer's keyring
function IsSerialNumberUnique(Context: pPGPContext; KeySetToCheck: pPGPKeySet; SignKey: pPGPKeyDBObj;
			      SerialNumber: Longint; var Unique: Longbool): Integer;
var
  SignKeyCertSet: pPGPKeySet;
  KeyIter	: pPGPKeyIter;
  Certificate	: pPGPKeyDBObj;
  KeyFound	: pPGPKeyDBObj;
  SerialNumStr	: String;
begin
  Unique := false;
  Result := GetCertKeySetBySignKey(Context, KeySetToCheck, SignKey, true, false, SignKeyCertSet);
  if Result = kPGPError_X509NeededCertNotAvailable then begin
    Unique := true;
    Result := 0;
    Exit;
  end
  else if Result <> 0 then Exit;
  try
    Result := PGPNewKeyIterFromKeySet(SignKeyCertSet, KeyIter);
    if Result <> 0 then Exit;
    try
      Unique := true;
      SerialNumStr := IntToStr(SerialNumber);
      while (FindCertificate(KeyIter, KeySetToCheck, Certificate, KeyFound) = 0) and Unique do begin
	if (KeyFound = SignKey) and (GetCertPropOwnerSerialNumber(Certificate) = SerialNumStr) then Unique := false;
      end;
    finally
      PGPFreeKeyIter(KeyIter);
    end;
  finally
    PGPFreeKeySet(SignKeyCertSet);
  end;
end;

// retrieve property values of certificate
function GetX509CertPropsRec(Certificate: pPGPKeyDBObj; var CertPropsRec: TX509CertPropsRec): Integer;
begin
  Result := 0;
  try
    with CertPropsRec do begin
      GetCertIssuerInfos(Certificate, x509CertificateID, x509IssuerKeyAlgorithm, x509IssuerHashAlgorithm);
      x509IssuerLongName := GetCertPropIssuerName(Certificate);
      x509IssuerSerialNumber := GetCertPropIssuerSerialNumber(Certificate);
      x509OwnerKeyHexID := GetCertPropOwnerKeyID(Certificate);
      x509OwnerLongName := GetCertPropOwnerName(Certificate);
      x509OwnerSerialNumber := GetCertPropOwnerSerialNumber(Certificate);
      x509CreaTimeStr := GetCertPropCertCreationTimeStr(Certificate);
      x509ExpTimeStr := GetCertPropCertExpirationTimeStr(Certificate);
      x509CreaTimeNum := GetCertPropCertCreationTime(Certificate);
      x509ExpTimeNum := GetCertPropCertExpirationTime(Certificate);
      x509IsVerified := GetCertPropIsVerified(Certificate);
      x509IsCorrupt := GetCertPropIsCorrupt(Certificate);
      x509IsRevoked := GetCertPropIsRevoked(Certificate);
      x509IsExpired := GetCertPropIsExpired(Certificate);
      x509IsRoot := GetCertPropIsRoot(Certificate);
    end;
  except
    Result := -1;
  end;
end;

// retrieve properties and IASN of single certificate
function GetX509CertPropsList(Certificate: pPGPKeyDBObj; var CertPropsList: TX509CertPropsList): Integer;
begin
  if CertPropsList = nil then CertPropsList := TX509CertPropsList.Create;
  if CertPropsList <> nil then with CertPropsList do begin
    Result := GetX509CertPropsRec(Certificate, pX509CertPropsRec(Objects[Add(GetCertPropIASN(Certificate))])^);
  end
  else Result := kPGPError_OutOfMemory;
end;

// returns number of certificates found or error
// strings contain the supposedly unique IssuerAndSerialNumber (binary) structure
function GetX509CertSetProps(CertSet, KeySetToSearch: pPGPKeySet; SignKey: pPGPKeyDBObj;
			     var CertPropsList: TX509CertPropsList): Integer;
var
  CertIter	: pPGPKeyIter;
  Certificate	: pPGPKeyDBObj;
  KeyFound	: pPGPKeyDBObj;
begin
  if PGP7X then begin
    if CertPropsList = nil then CertPropsList := TX509CertPropsList.Create;
    if CertPropsList = nil then begin
      Result := kPGPError_OutOfMemory;
      Exit;
    end;
    Result := PGPNewKeyIterFromKeySet(CertSet, CertIter);
    if Result <> 0 then Exit;
    try
      while FindCertificate(CertIter, KeySetToSearch, Certificate, KeyFound) = 0 do begin
	if (KeyFound = SignKey) or (SignKey = nil) then begin
	  with CertPropsList do begin
	    Result := GetX509CertPropsRec(Certificate, pX509CertPropsRec(Objects[Add(GetCertPropIASN(Certificate))])^);
	  end;
	end;
      end;
      if Result = 0 then Result := CertPropsList.Count;
    finally
      PGPFreeKeyIter(CertIter);
    end;
  end
  else Result := kPGPError_LazyProgrammer;
end;

// returns number of certificates found or error
function GetX509CertificateProperties(const CertData: String; FromFile: Longbool;
				      var CertPropsList: TX509CertPropsList): Integer;
var
  Context	: pPGPContext;
  KeySetMain	: pPGPKeySet;
  CertDBToCheck	: pPGPKeyDB;
  CertSetToCheck: pPGPKeySet;
begin
  if PGP7X then begin
    Result := KeyRings.InitKeyRings(Context, KeySetMain);
    if Result <> 0 then Exit;
    try
      Result := GetX509CertImportDB(Context, CertData, CertDBToCheck, FromFile);
      if Result <> 0 then Exit;
      try
	Result := PGPNewKeySet(CertDBToCheck, CertSetToCheck);
	if Result <> 0 then Exit;
	try
	  Result := GetX509CertSetProps(CertSetToCheck, KeySetMain, nil, CertPropsList);
	finally
	  PGPFreeKeySet(CertSetToCheck);
	end;
      finally
	PGPFreeKeyDB(CertDBToCheck);
      end;
    finally
      KeyRings.FreeKeyRings;
      Result := TranslateX509ErrorToPGPError(Result);
    end;
  end
  else Result := kPGPError_LazyProgrammer;
end;

// returns number of certificates found or error
// KeyFilterFlags is applied to any kind of search
// SignKeyID can narrow down the search with CertKey...IDs
// IncludeSigner is only valid without CertKey...IDs specified
// in this case you will get the following results depending on SignKeyHexID and IncludeSigner:
// (SignKey = nil) and (IncludeSigner = false)  => all certified keys on keyring
// (SignKey = nil) and (IncludeSigner = true)   => all certifier keys on keyring (signing keys only)
// (SignKey <> nil) and (IncludeSigner = false) => all SignKey-certified keys on keyring without certifier
// (SignKey <> nil) and (IncludeSigner = true)  => all SignKey-certified keys on keyring including certifier
function FindX509CertProps(const CertKeyUserID, CertKeyHexID, SignKeyHexID: String; KeyFilterFlags: DWord;
			   IncludeSigner: Longbool; var CertPropsList: TX509CertPropsList): Integer;
var
  Context	: pPGPContext;
  KeySetMain	: pPGPKeySet;
  KeyFilter	: pPGPFilter;
  KeySetToCheck	: pPGPKeySet;
  SignKey	: pPGPKeyDBObj;
  CertKey	: pPGPKeyDBObj;
  CertSet	: pPGPKeySet;
  CertSetBuf	: pPGPKeySet;
  UserIDFilter	: pPGPFilter;
  KeyCount	: PGPUInt32;
begin
  if PGP7X then begin
    Result := KeyRings.InitKeyRings(Context, KeySetMain);
    if Result <> 0 then Exit;
    CertSet := nil;
    try
      if KeyFilterFlags <> KeyFilterFlag_AllKeys then begin
	Result := GetKeyFilter(Context, KeyFilterFlags, KeyFilter);
	if Result <>0 then Exit;
	try
	  Result := PGPFilterKeySet(KeySetMain, KeyFilter, KeySetToCheck);
	  if Result <> 0 then Exit;
	finally
	  PGPFreeFilter(KeyFilter);
	end;
      end
      else KeySetToCheck := KeySetMain;
      try
	if SignKeyHexID <> '' then begin
	  Result := GetKeyByHexID(KeySetMain, SignKeyHexID, SignKey);
	  if (Result = 0) and not GetKeyPropCanSign(SignKey) then Result := kPGPError_KeyUnusableForSignature;
	  if Result <> 0 then Exit;
	end
	else SignKey := nil;
	if (CertKeyUserID <> '') or (CertKeyHexID <> '') then begin
	  if CertKeyUserID <> '' then begin
	    Result := GetUserIDFilter(Context, CertKeyUserID, true, UserIDFilter);
	    if Result <> 0 then Exit;
	  end
	  else UserIDFilter := nil;
	  try
	    if CertKeyHexID <> '' then begin
	      Result := GetKeyByHexID(KeySetToCheck, CertKeyHexID, CertKey);
	      if Result <> 0 then Exit;
	      Result := PGPNewSingletonKeySet(CertKey, CertSet);
	      if Result <> 0 then Exit;
	      if UserIDFilter <> nil then begin
		CertSetBuf := CertSet;
		CertSet := nil;
		try
		  Result := PGPFilterKeySet(CertSetBuf, UserIDFilter, CertSet);
		  if Result <> 0 then Exit;
		finally
		  PGPFreeKeySet(CertSetBuf);
		end;
	      end;
	    end
	    else begin
	      Result := PGPFilterKeySet(KeySetToCheck, UserIDFilter, CertSet);
	      if Result <> 0 then Exit;
	    end;
	  finally
	    PGPFreeFilter(UserIDFilter);
	  end;
	end
	else begin
	  Result := GetCertKeySetBySignKey(Context, KeySetToCheck, SignKey, IncludeSigner, false, CertSet);
	  if Result <> 0 then Exit;
	  Result := PGPCountKeys(CertSet, KeyCount);
	  if (Result <> 0) or (KeyCount = 0) then Exit;
	end;
	try
	  Result := GetX509CertSetProps(CertSet, KeySetMain, SignKey, CertPropsList);
	finally
	  PGPFreeKeySet(CertSet);
	end;
      finally
	if KeySetToCheck <> KeySetMain then PGPFreeKeySet(KeySetToCheck);
      end;
    finally
      KeyRings.FreeKeyRings;
      Result := TranslateX509ErrorToPGPError(Result);
    end;
  end
  else Result := kPGPError_LazyProgrammer;
end;

end.

