{$J+,Z4}
unit PrefFuncs;

{------------------------------------------------------------------------------}
{                                                                              }
{       This unit is based on Steve Heller's spgpPreferences.pas from his      }
{         SPGP sources available from http://www.oz.net/~srheller/spgp/        }
{                                                                              }
{                Portions created by Michael in der Wiesche are                }
{              Copyright (C) 2001-2003 by Michael in der Wiesche               }
{                                                                              }
{------------------------------------------------------------------------------}

interface

uses
  Windows,
  Classes,
  SysUtils,
  KeyPropTypes,
  pgpBase,
  pgpErrors,
  pgpPubTypes,
  pgpUtilities,
  pgpMemoryMgr,
  pgpKeyServer,
  pgpSDKPrefs,
  pgpKeys,
  pgpCL,
  KeyFuncs;

const
  E = '';
  BS = '\';
  CM = ',';
  QU = '"';
  SP = ' ';
  CRLF = #13#10;
  AppDataRegPath = 'Software\Microsoft\Windows\CurrentVersion\Explorer\Shell Folders';
  PrefsFileName = 'PGPprefs.txt';
  GroupsFileName = 'PGPgroup.pgr';
  PubFile = 'PublicKeyringFile';
  SecFile = 'PrivateKeyringFile';
  RNGFile = 'RNGSeedFile';
  DefKeyID ='DefaultKeyID';
  KeyServers = 'KeyServerList';
  WipeWarning = 'WarnOnWipe';
  FileWipeCount = 'FileWipePasses';
  PrefAlgorithm = 'PreferredAlgorithm';
  AllowedAlgList = 'AllowedAlgorithmsList';

  PrefsFlag_Default		= $00;
  PrefsFlag_PublicKeyring	= $01;
  PrefsFlag_PrivateKeyring	= $02;
  PrefsFlag_RandomSeedFile	= $04;
  PrefsFlag_DefaultKeyID	= $08;
  PrefsFlag_GroupsFile		= $10;

type
  pPreferenceRec = ^TPreferenceRec;
  TPreferenceRec = Record
    PublicKeyring: String;
    PrivateKeyring: String;
    RandomSeedFile: String;
    GroupsFile: String;
    DefaultKeyHexID: String;
  end;

type
  TPrefsFile = Class
    PrefsData: TStringList;
    PrefsFilePath: String;
    GroupsFilePath: String;
    CacheAge: Integer;
    constructor Create;
    destructor Destroy; override;
    function LoadPrefs: Longbool;
    function SavePrefs: Longbool;
    function GetPGPKeyID(const PGPKeyIDString: String): TPGPKeyID7;
    function GetPGPKeyIDString(const PGPKeyID: TPGPKeyID7): String;
    function GetAppPathRegEntry(const RegPath, RegKey: String): String;
  end;

const PrefsFile: TPrefsFile = nil;

function GetPreferences(var Prefs: TPreferenceRec; Flags: Longint): Longint;
function SetPreferences(const Prefs: TPreferenceRec; Flags: Longint): Longint;
function GetServerList(var ServerList: TStringList): Longint;
function GetPrefWarnOnWipe(var WarnOnWipe: Longbool): Longint;
function SetPrefWarnOnWipe(WarnOnWipe: Longbool): Longint;
function GetPrefWipeCount(var WipeCount: PGPUInt32): Longint;
function SetPrefWipeCount(WipeCount: PGPUInt32): Longint;
function GetPreferredCipherAlgorithm(var Algorithm: PGPCipherAlgorithm): Longint;
function GetAllowedCipherAlgorithms(var AlgorithmList: TPGPCipherAlgorithms; var Count: Longint): Longint;

implementation

function ShortStrPos(const Find, Dest: String; Pos: Integer): Integer; register; assembler;
asm	// EAX = @Find, EDX = @Dest, ECX = Pos
  OR	EAX,EAX
  JE	@EXIT
  OR	EDX,EDX
  JE	@EXIT
  OR	ECX,ECX
  JLE	@EXIT
  PUSH	EBX
  MOV	EBX,[EAX - 4]
  SUB	ECX,[EDX - 4]
  ADD	EBX,ECX
  CMP	EBX,1
  JG	@END
  CMP	DWORD PTR [EAX - 4],1
  JE	@CHAR
  PUSH	EBP
  PUSH	EDI
  PUSH	ESI
  MOV	EDI,EDX
  MOV	ESI,EAX
  ADD	EDI,[EDX - 4]
  MOV	AL,[ESI]
  DEC	EDI
  @LOOP:
  CMP	[EDI + ECX],AL
  JE	@MATCH
  INC	ECX
  JNE	@LOOP
  XOR	EAX,EAX
  POP	ESI
  POP	EDI
  POP	EBP
  POP	EBX
  RET
  @MATCH:
  PUSH	EDI
  MOV	EBX,[ESI - 4]
  MOV	EBP,ESI
  ADD	EDI,EBX
  ADD	EBP,EBX
  ADD	EDI,ECX
  NEG	EBX
  @CMP:
  INC	EBX
  JE	@YES
  MOV	AH,[EDI + EBX]
  CMP	[EBP + EBX],AH
  JE	@CMP
  POP	EDI
  INC	ECX
  JNE	@LOOP
  XOR	EAX,EAX
  POP	ESI
  POP	EDI
  POP	EBP
  POP	EBX
  RET
  @YES:
  ADD	ECX,[EDX - 4]
  POP	EDI
  MOV	EAX,ECX
  POP	ESI
  POP	EDI
  POP	EBP
  POP	EBX
  RET
  @END:
  POP	EBX
  @EXIT:
  XOR	EAX,EAX
  RET
  @CHAR:
  POP	EBX
  PUSH	EDX
  DEC	ECX
  ADD	EDX,[EDX - 4]
  MOV	AL,[EAX]
  @@LOOP:
  CMP	[EDX + ECX],AL
  JE	@@MATCH
  INC	ECX
  JNE	@@LOOP
  POP	EDX
  XOR	EAX,EAX
  RET
  @@MATCH:
  INC	ECX
  POP	EDX
  MOV	EAX,ECX
  ADD	EAX,[EDX - 4]
end;

function ExtractStr(const Str: String): String;
var iPos: Longint;
begin
  Result := Str;
  iPos := Length(Result);
  if iPos > 1 then begin
    if (Result[iPos] = BS) and (Result[pred(iPos)] = SP) then begin
      Delete(Result, pred(iPos), 2);
      dec(iPos, 2);
    end;
    if (iPos <> 0) and (Result[1] = QU) and (Result[iPos] = QU) then begin
      Delete(Result, iPos, 1);
      Delete(Result, 1, 1);
    end;
  end;
end;

constructor TPrefsFile.Create;
var FilePath: String;
begin
  inherited;
  PrefsData := TStringList.Create;
  PrefsData.BeginUpdate;
  if PGP8X then
    FilePath := GetAppPathRegEntry(AppDataRegPath, 'AppData') + PathPGP + '\'
  else FilePath := GetAppPathRegEntry(AppDataRegPath, 'AppData') + PathNAI + '\';
  PrefsFilePath := FilePath + PrefsFileName;
  GroupsFilePath := FilePath + GroupsFileName;
  LoadPrefs;
end;

destructor TPrefsFile.Destroy;
begin
  PrefsData.Free;
  inherited;
end;

function TPrefsFile.LoadPrefs: Longbool;
var Buffer: String; iIndex: Longint;
begin
  Result := false;
  if (PrefsData <> nil) and (PrefsFilePath <> E) then with PrefsData do begin
    if FileAge(PrefsFilePath) >= CacheAge then begin
      try
	LoadFromFile(PrefsFilePath);
	Buffer := Text;
	iIndex := 1;
	repeat
	  iIndex := ShortStrPos(SP + BS + CRLF, Buffer, iIndex);
	  if iIndex > 1 then begin
	    if (Buffer[pred(iIndex)] = QU) then
	      System.Delete(Buffer, pred(iIndex), Length(QU + SP + BS + CRLF + QU))
	    else System.Delete(Buffer, iIndex, Length(SP + BS + CRLF));
	  end;
	until iIndex = 0;
	Text := Buffer;
	Result := (Count <> 0);
	if Result then CacheAge := DateTimeToFileDate(Now);
      except
      end;
    end
    else Result := true;
  end;
end;

function TPrefsFile.SavePrefs: Longbool;
begin
  Result := false;
  if (PrefsData <> nil) and (PrefsFilePath <> E) then with PrefsData do begin
    try
      SaveToFile(PrefsFilePath);
      Result := true;
    except
    end;
  end;
end;

function TPrefsFile.GetPGPKeyID(const PGPKeyIDString: String): TPGPKeyID7; register; assembler;
asm	// EAX = @Self, EDX = @PGPKeyIDString, ECX = @Result
  PUSH	EBX
  OR	EDX,EDX
  JE	@ERROR
  CMP	DWORD PTR [EDX - 4],TYPE TPGPKeyID7 * 2 + 2
  JNE	@ERROR
  AND	BYTE PTR [EDX + 1],0DFh
  CMP	WORD PTR [EDX],'X0'
  JNE	@ERROR
  MOV	EBX,[EDX - 4]
  SHR	EBX,1
  DEC	EBX
  @START:
  MOV	AX,[EDX + EBX * 2]
  @FIRSTBYTE:
  SUB	AL,'0'
  JB	@ERROR
  CMP	AL,09h
  JA	@FIRSTCHAR
  JMP	@NEXTBYTE
  @FIRSTCHAR:
  AND	AL,0DFh
  SUB	AL,'A' - '0'
  JB	@ERROR
  CMP	AL,05h
  JA	@ERROR
  ADD	AL,0Ah
  @NEXTBYTE:
  SUB	AH,'0'
  JB	@ERROR
  CMP	AH,09h
  JA	@NEXTCHAR
  JMP	@DECODE
  @NEXTCHAR:
  AND	AH,0DFh
  SUB	AH,'A' - '0'
  JB	@ERROR
  CMP	AH,05h
  JA	@ERROR
  ADD	AH,0Ah
  @DECODE:
  SHL	AL,4
  ADD	AL,AH
  MOV	[ECX + EBX - 1],AL
  DEC	EBX
  JNZ	@START
  POP	EBX
  RET
  @ERROR:
  POP	EBX
  MOV	DWORD PTR [ECX],0
end;

function TPrefsFile.GetPGPKeyIDString(const PGPKeyID: TPGPKeyID7): String; register; assembler;
asm	// EAX = @Self, EDX = @PGPKeyID, ECX = @Result
  PUSH	EBX
  PUSH	EDI
  PUSH	ESI
  MOV	ESI,EDX
  MOV	EDI,ECX
  MOV	EAX,ECX
  CALL	SYSTEM.@LSTRCLR
  MOV	EAX,TYPE TPGPKeyID7 * 2
  CALL	SYSTEM.@NEWANSISTRING
  MOV	[EDI],EAX
  MOV	EDI,EAX
  MOV	ECX,TYPE TPGPKeyID7 - 1
  @LOOP:
  MOVZX	EAX,BYTE PTR[ESI + ECX]
  MOV	EDX,EAX
  AND	EAX,0Fh
  SHR	EDX,4
  MOV	AH,BYTE PTR[@HEXCHARS + EAX]
  MOV	AL,BYTE PTR[@HEXCHARS + EDX]
  MOV	[EDI + ECX * 2],AX
  DEC	ECX
  JNS	@LOOP
  @END:
  POP	ESI
  POP	EDI
  POP	EBX
  RET
  @HEXCHARS:
  DB	'0123456789ABCDEF';
end;

function TPrefsFile.GetAppPathRegEntry(const RegPath, RegKey: String): String;
var
  hRegKey: hKey;
  dwBufSize: DWord;
  aFilePath: Array[0..MAX_PATH] of Char;
begin
  dwBufSize := MAX_PATH;
  if RegOpenKeyEx(HKEY_CURRENT_USER, PChar(RegPath), 0, KEY_QUERY_VALUE, hRegKey) = ERROR_SUCCESS then begin
    try
      if RegQueryValueEx(hRegKey, PChar(RegKey), nil, nil, @aFilePath, @dwBufSize) = ERROR_SUCCESS then begin
	Result := aFilePath;
	Exit;
      end;
    finally
      RegCloseKey(hRegKey);
    end;
  end;
  Result := E;
end;

function GetPreferences(var Prefs: TPreferenceRec; Flags: Longint): Longint;
var
  Context	: pPGPContext;
  FileSpec	: pPGPFileSpec;
  FileOut	: PChar;
  KeyOut	: PChar;
  PGPKeyID	: TPGPKeyID7;
  IDSize	: PGPSize;
  KeyID		: TKeyID;
begin
  Result := kPGPError_PrefNotFound;
  if Flags <> 0 then begin
    FillChar(Prefs, SizeOf(Prefs), 0);
    if PGP7X then begin
      if PrefsFile <> nil then with PrefsFile do if LoadPrefs then begin
	try
	  if Flags and PrefsFlag_PublicKeyring <> 0 then Prefs.PublicKeyring := ExtractStr(PrefsData.Values[PubFile]);
	  if Flags and PrefsFlag_PrivateKeyring <> 0 then Prefs.PrivateKeyring := ExtractStr(PrefsData.Values[SecFile]);
	  if Flags and PrefsFlag_RandomSeedFile <> 0 then Prefs.RandomSeedFile := ExtractStr(PrefsData.Values[RNGFile]);
	  if Flags and PrefsFlag_GroupsFile <> 0 then Prefs.GroupsFile := PrefsFile.GroupsFilePath;
	  if Flags and PrefsFlag_DefaultKeyID <> 0 then begin
	    PGPKeyID := GetPGPKeyID(PrefsData.Values[DefKeyID]);
	    Result := PGPGetKeyIDString(PGPKeyID, kPGPKeyIDString_Full, KeyID);
	    if Result = 0 then Prefs.DefaultKeyHexID := KeyID;
	  end;
	  Result := 0;
	except
	end;
      end;
    end
    else begin
      Context := nil;
      FileSpec := nil;
      Result := PGPNewContext(kPGPsdkAPIVersion, Context);
      if Result <> 0 then Exit;
      try
	Result := PGPsdkLoadDefaultPrefs(Context);
	if Result <> 0 then Exit;

	// Pubring
	if Flags and PrefsFlag_PublicKeyring <> 0 then begin
	  Result := PGPsdkPrefGetFileSpec(Context, kPGPsdkPref_PublicKeyring, FileSpec);
	  try
	    if Result = 0 then begin
	      FileOut := nil;
	      Result := PGPGetFullPathFromFileSpec(FileSpec, FileOut);
	      try
		Prefs.PublicKeyring := FileOut;
	      finally
		PGPFreeData(FileOut);
	      end;
	    end;
	  finally
	    PGPFreeFileSpec(FileSpec);
	  end;
	end;
	if Result <> 0 then Exit;

	// Secring
	if Flags and PrefsFlag_PrivateKeyring <> 0 then begin
	  Result := PGPsdkPrefGetFileSpec(Context, kPGPsdkPref_PrivateKeyring, FileSpec);
	  try
	    if Result = 0 then begin
	      FileOut := nil;
	      Result := PGPGetFullPathFromFileSpec(FileSpec, FileOut);
	      try
		Prefs.PrivateKeyring := FileOut;
	      finally
		PGPFreeData(FileOut);
	      end;
	    end;
	  finally
	    PGPFreeFileSpec(FileSpec);
	  end;
	end;
	if Result <> 0 then Exit;

	// Randseed file
	if Flags and PrefsFlag_RandomSeedFile <> 0 then begin
	  Result := PGPsdkPrefGetFileSpec(Context, kPGPsdkPref_RandomSeedFile, FileSpec);
	  try
	    if Result = 0 then begin
	      FileOut := nil;
	      Result := PGPGetFullPathFromFileSpec(FileSpec, FileOut);
	      try
		Prefs.RandomSeedFile := FileOut;
	      finally
		PGPFreeData(FileOut);
	      end;
	    end;
	  finally
	    PGPFreeFileSpec(FileSpec);
	  end;
	end;
	if Result <> 0 then Exit;

	// Groups file
	if Flags and PrefsFlag_GroupsFile <> 0 then begin
	  Result := PGPsdkPrefGetFileSpec(Context, kPGPsdkPref_GroupsFile, FileSpec);
	  try
	    if Result = 0 then begin
	      FileOut := nil;
	      Result := PGPGetFullPathFromFileSpec(FileSpec, FileOut);
	      try
		Prefs.GroupsFile := FileOut;
	      finally
		PGPFreeData(FileOut);
	      end;
	    end;
	  finally
	    PGPFreeFileSpec(FileSpec);
	  end;
	end;
	if Result <> 0 then Exit;

	// Private key
	if Flags and PrefsFlag_DefaultKeyID <> 0 then begin
	  KeyOut := nil;
	  Result := PGPsdkPrefGetData(Context, kPGPsdkPref_DefaultKeyID, @KeyOut, IDSize);
	  if Result <> 0 then Exit;
	  try
	    Result := PGPImportKeyID(KeyOut, PGPKeyID);
	  finally
	    PGPFreeData(KeyOut);
	  end;
	  if Result <> 0 then Exit;
	  Result := PGPGetKeyIDString(PGPKeyID, kPGPKeyIDString_Full, KeyID);
	  if Result = 0 then Prefs.DefaultKeyHexID := KeyID;
	end;
      finally
	PGPFreeContext(Context);
      end;
    end;
  end;
end;

function SetPreferences(const Prefs: TPreferenceRec; Flags: Longint): Longint;
var
  Context	: pPGPContext;
  FileSpec	: pPGPFileSpec;
  KeyPropsList	: TKeyPropsList;
  RingContext	: pPGPContext;
  MainKeySet	: pPGPKeySet;
  Key		: pPGPKey;
  KeyIn		: TPGPKeyID7;
  PGPKeyID	: TPGPKeyID7;
  IDSize	: PGPSize;
begin
  Result := kPGPError_PrefNotFound;
  if Flags <> 0 then begin
    if PGP7X then begin
      if PrefsFile <> nil then with PrefsFile do if LoadPrefs then begin
	try
	  if Flags and PrefsFlag_PublicKeyring <> 0 then begin
	  if not FileExists(Prefs.PublicKeyring) then begin
	    Result := kPGPError_FileNotFound;
	    Exit;
	  end;
	  PrefsData.Values[PubFile] := QU + Prefs.PublicKeyring + QU;
	  end;
	  if Flags and PrefsFlag_PrivateKeyring <> 0 then begin
	    if not FileExists(Prefs.PrivateKeyring) then begin
	      Result := kPGPError_FileNotFound;
	      Exit;
	    end;
	    PrefsData.Values[SecFile] := QU + Prefs.PrivateKeyring + QU;
	  end;
	  if Flags and PrefsFlag_RandomSeedFile <> 0 then begin
	    if not FileExists(Prefs.RandomSeedFile) then begin
	      Result := kPGPError_FileNotFound;
	      Exit;
	    end;
	    PrefsData.Values[RNGFile] := QU + Prefs.RandomSeedFile + QU;
	  end;
	  if Flags and PrefsFlag_GroupsFile <> 0 then begin
	    MessageBox(0, 'PGP 7.X/8.X don''t support changing the default groups file name',
		       'PGP Components error:', MB_ICONERROR);
	  end;
	  if Flags and PrefsFlag_DefaultKeyID <> 0 then begin
	    KeyPropsList := nil;
	    try
	      Result := KeyRings.InitKeyRings(RingContext, MainKeySet);
	      if Result <> 0 then Exit;
	      try
		if FindKeyProps(Prefs.DefaultKeyHexID, KeyPropsList, 0, KeyFilterFlag_CanSign, Any_Ordering) = 1 then begin
		  Result := GetKeyByHexID(MainKeySet, Prefs.DefaultKeyHexID, Key);
		  if Result <> 0 then Exit;
		  Result := PGPGetKeyIDFromKey(Key, PGPKeyID);
		  if Result <> 0 then Exit;
		  PrefsData.Values[DefKeyID] := '0x' + GetPGPKeyIDString(PGPKeyID);
		end
		else Result := kPGPError_SecretKeyNotFound;
	      finally
		KeyRings.FreeKeyRings;
	      end;
	    finally
	      KeyPropsList.Free;
	    end;
	  end;
	  if (Result = 0) and not SavePrefs then Result := kPGPError_WriteFailed;
	except
	end;
      end;
    end
    else begin
      Context := nil;
      FileSpec := nil;
      Result := PGPNewContext(kPGPsdkAPIVersion, Context);
      if Result <> 0 then Exit;
      try
	Result := PGPsdkLoadDefaultPrefs(Context);
	if Result <> 0 then Exit;

	// Pubring
	if Flags and PrefsFlag_PublicKeyring <> 0 then begin
	  if not FileExists(Prefs.PublicKeyring) then begin
	    Result := kPGPError_FileNotFound;
	    Exit;
	  end;
	  Result := PGPNewFileSpecFromFullPath(Context, PChar(Prefs.PublicKeyring), FileSpec);
	  try
	    if Result = 0 then Result := PGPsdkPrefSetFileSpec(Context, kPGPsdkPref_PublicKeyring, FileSpec);
	    if Result <> 0 then Exit;
	  finally
	    PGPFreeFileSpec(FileSpec);
	  end;
	end;

	// Secring
	if Flags and PrefsFlag_PrivateKeyring <> 0 then begin
	  if not FileExists(Prefs.PrivateKeyring) then begin
	    Result := kPGPError_FileNotFound;
	    Exit;
	  end;
	  Result := PGPNewFileSpecFromFullPath(Context, PChar(Prefs.PrivateKeyring), FileSpec);
	  try
	    if Result = 0 then Result := PGPsdkPrefSetFileSpec(Context, kPGPsdkPref_PrivateKeyring, FileSpec);
	    if Result <> 0 then Exit;
	  finally
	    PGPFreeFileSpec(FileSpec);
	  end;
	end;

	// Randseed file
	if Flags and PrefsFlag_RandomSeedFile <> 0 then begin
	  if not FileExists(Prefs.RandomSeedFile) then begin
	    Result := kPGPError_FileNotFound;
	    Exit;
	  end;
	  Result := PGPNewFileSpecFromFullPath(Context, PChar(Prefs.RandomSeedFile), FileSpec);
	  try
	    if Result = 0 then Result := PGPsdkPrefSetFileSpec(Context, kPGPsdkPref_RandomSeedFile, FileSpec);
	    if Result <> 0 then Exit;
	  finally
	    PGPFreeFileSpec(FileSpec);
	  end;
	end;

	// Groups file
	if Flags and PrefsFlag_GroupsFile <> 0 then begin
	  if not FileExists(Prefs.GroupsFile) then begin
	    Result := kPGPError_FileNotFound;
	    Exit;
	  end;
	  Result := PGPNewFileSpecFromFullPath(Context, PChar(Prefs.GroupsFile), FileSpec);
	  try
	    if Result = 0 then Result := PGPsdkPrefSetFileSpec(Context, kPGPsdkPref_GroupsFile, FileSpec);
	    if Result <> 0 then Exit;
	  finally
	    PGPFreeFileSpec(FileSpec);
	  end;
	end;

	// Private key
	if Flags and PrefsFlag_DefaultKeyID <> 0 then begin
	  KeyPropsList := nil;
	  try
	    if FindKeyProps(Prefs.DefaultKeyHexID, KeyPropsList, 0, KeyFilterFlag_CanSign, Any_Ordering) = 1 then begin
	      Result := KeyRings.InitKeyRings(RingContext, MainKeySet);
	      if Result <> 0 then Exit;
	      try
		Result := GetKeyByHexID(MainKeySet, Prefs.DefaultKeyHexID, Key);
		if Result <> 0 then Exit;
		Result := PGPGetKeyIDFromString(PChar(Prefs.DefaultKeyHexID), kPGPPublicKeyAlgorithm_Invalid, KeyIn);
		if Result <> 0 then Exit;
		Result := PGPExportKeyID(@KeyIn, PGPKeyID, IDSize);
		if Result <> 0 then Exit;
	      finally
		KeyRings.FreeKeyRings;
	      end;
	      Result := PGPsdkPrefSetData(Context, kPGPsdkPref_DefaultKeyID, @PGPKeyID, IDSize);
	    end
	    else Result := kPGPError_SecretKeyNotFound;
	    if Result <> 0 then Exit;
	  finally
	    KeyPropsList.Free;
	  end;
	end;
	Result := PGPsdkSavePrefs(Context);
      finally
	PGPFreeContext(Context);
      end;
    end;
    if Result = 0 then begin
      PGPclNotifyKeyringChanges(GetCurrentProcessID);
      PGPclNotifyPrefsChanges(GetCurrentProcessID);
    end;
  end;
end;

function GetServerList(var ServerList: TStringList): Longint;
const
  Protocols: Array[kPGPKeyServerProtocol_Invalid..kPGPKeyServerProtocol_HTTPS] of String = (
    E, 'ldap', 'http', 'ldaps', 'https'
  );
var
  Entry		 : TStringList;
  PGPPref	 : pPGPPref;
  KeyServerList	 : pPGPKeyServerEntry;
  KeyServerEntry : pPGPKeyServerEntry;
  ServerCount	 : PGPUInt32;
  ServerURL	 : String;
begin
  if PGP7X then begin
    Result := kPGPError_PrefNotFound;
    if PrefsFile <> nil then with PrefsFile do if LoadPrefs then begin
      try
	Entry := TStringList.Create;
	ServerList := TStringList.Create;
	if (Entry <> nil) and (ServerList <> nil) then with Entry do begin
	  try
	    ServerCount := 0;
	    repeat
	      inc(ServerCount);
	      CommaText := PrefsData.Values[KeyServers + IntToStr(ServerCount)];
	      if Count >= 8 then begin
		if Strings[0] <> E then ServerURL := Protocols[StrToIntDef(Strings[0], 0)];
		if (ServerURL <> E) and (Strings[3] <> E) then begin
		  ServerURL := ServerURL + '://' + Strings[3];
		  if (Strings[4] <> E) and (Strings[4] <> '0') then ServerURL := ServerURL + ':' + Strings[4];
		  ServerList.Add(ServerURL);
		end;
	      end;
	    until Count = 0;
	  finally
	    Entry.Free;
	  end;
	  Result := 0;
	end;
      except
      end;
    end;
  end
  else begin
    PGPPref := nil;
    KeyServerList := nil;
    ServerList := TStringList.Create;
    if ServerList <> nil then begin
      Result := PGPclOpenClientPrefs(PGPGetDefaultMemoryMgr, PGPPref);
      if Result = 0 then begin
	try
	  Result := PGPGetKeyServerPrefs(PGPPref, KeyServerList, ServerCount);
	  if Result = 0 then begin
	    try
	      if ServerCount <> 0 then begin
		KeyServerEntry := KeyServerList;
		for ServerCount := ServerCount downto 1 do with KeyServerEntry^ do begin
		  if Protocol <> kPGPKeyServerProtocol_Invalid then ServerURL := Protocols[Protocol] + '://';
		  if ServerDNS <> E then begin
		    ServerURL := ServerURL + ServerDNS;
		    if ServerPort <> 0 then ServerURL := ServerURL + ':' + IntToStr(ServerPort);
		    ServerList.Add(ServerURL);
		  end;
		  inc(KeyServerEntry);
		end;
	      end
	      else Result := kPGPError_PrefNotFound;
	    finally
	      PGPFreeData(KeyServerList);
	    end;
	  end;
	finally
	  PGPclCloseClientPrefs(PGPPref, false);
	end;
      end;
    end
    else Result := kPGPError_OutOfMemory;
  end;
end;

function GetPrefWarnOnWipe(var WarnOnWipe: Longbool): Longint;
var
  PGPPref	 : pPGPPref;
  Pref		 : PGPBoolean;
begin
  WarnOnWipe := false;
  if PGP7X then begin
    Result := kPGPError_PrefNotFound;
    if PrefsFile <> nil then with PrefsFile do if LoadPrefs then begin
      try
	WarnOnWipe := (CompareText(PrefsData.Values[WipeWarning], 'TRUE') = 0);
	Result := 0;
      except
      end;
    end;
  end
  else begin
    PGPPref := nil;
    Result := PGPclOpenClientPrefs(PGPGetDefaultMemoryMgr, PGPPref);
    if Result = 0 then begin
      try
	Result := PGPclGetPrefBoolean(PGPPref, kPGPPrefWarnOnWipe, Pref);
	if Result = 0 then WarnOnWipe := Boolean(Pref);
      finally
	PGPclCloseClientPrefs(PGPPref, false);
      end;
    end;
  end;
end;

function SetPrefWarnOnWipe(WarnOnWipe: Longbool): Longint;
var
  PGPPref	 : pPGPPref;
begin
  if PGP7X then begin
    Result := kPGPError_PrefNotFound;
    if PrefsFile <> nil then with PrefsFile do if LoadPrefs then begin
      try
	if WarnOnWipe then
	  PrefsData.Values[WipeWarning] := 'TRUE'
	else PrefsData.Values[WipeWarning] := 'FALSE';
	if SavePrefs then Result := 0;
      except
      end;
    end;
  end
  else begin
    PGPPref := nil;
    Result := PGPclOpenClientPrefs(PGPGetDefaultMemoryMgr, PGPPref);
    if Result = 0 then begin
      try
	Result := PGPclSetPrefBoolean(PGPPref, kPGPPrefWarnOnWipe, PGPBoolean(WarnOnWipe) and 1);
      finally
	PGPclCloseClientPrefs(PGPPref, true);
      end;
    end;
  end;
  if Result = 0 then PGPclNotifyPrefsChanges(GetCurrentProcessID);
end;

function GetPrefWipeCount(var WipeCount: PGPUInt32): Longint;
var
  PGPPref	 : pPGPPref;
begin
  WipeCount := 0;
  if PGP7X then begin
    Result := kPGPError_PrefNotFound;
    if PrefsFile <> nil then with PrefsFile do if LoadPrefs then begin
      try
	WipeCount := StrToIntDef(PrefsData.Values[FileWipeCount], 0);
	Result := 0;
      except
      end;
    end;
  end
  else begin
    PGPPref := nil;
    Result := PGPclOpenClientPrefs(PGPGetDefaultMemoryMgr, PGPPref);
    if Result = 0 then begin
      try
	Result := PGPclGetPrefNumber(PGPPref, kPGPPrefFileWipePasses, WipeCount);
      finally
	PGPclCloseClientPrefs(PGPPref, false);
      end;
    end;
  end;
end;

function SetPrefWipeCount(WipeCount: PGPUInt32): Longint;
var
  PGPPref	 : pPGPPref;
begin
  if PGP7X then begin
    Result := kPGPError_PrefNotFound;
    if PrefsFile <> nil then with PrefsFile do if LoadPrefs then begin
      try
	PrefsData.Values[FileWipeCount] := IntToStr(WipeCount);
	if SavePrefs then Result := 0;
      except
      end;
    end;
  end
  else begin
    PGPPref := nil;
    Result := PGPclOpenClientPrefs(PGPGetDefaultMemoryMgr, PGPPref);
    if Result = 0 then begin
      try
	Result := PGPclSetPrefNumber(PGPPref, kPGPPrefFileWipePasses, WipeCount);
      finally
	PGPclCloseClientPrefs(PGPPref, true);
      end;
    end;
  end;
  if Result = 0 then PGPclNotifyPrefsChanges(GetCurrentProcessID);
end;

function GetPreferredCipherAlgorithm(var Algorithm: PGPCipherAlgorithm): Longint;
var
  PGPPref	 : pPGPPref;
  Pref		 : PGPUInt32;
begin
  // PGP's default
  Algorithm := kPGPCipherAlgorithm_CAST5;
  if PGP7X then begin
    Result := kPGPError_PrefNotFound;
    if PrefsFile <> nil then with PrefsFile do if LoadPrefs then begin
      try
	Algorithm := PGPCipherAlgorithm(StrToIntDef(PrefsData.Values[PrefAlgorithm], kPGPCipherAlgorithm_CAST5));
	Result := 0;
      except
      end;
    end;
  end
  else begin
    PGPPref := nil;
    Result := PGPclOpenClientPrefs(PGPGetDefaultMemoryMgr, PGPPref);
    if Result = 0 then begin
      try
	Result := PGPclGetPrefNumber(PGPPref, kPGPPrefPreferredAlgorithm, Pref);
	if Result = 0 then Algorithm := PGPCipherAlgorithm(Pref);
      finally
	PGPclCloseClientPrefs(PGPPref, false);
      end;
    end;
  end;
end;

function GetAllowedCipherAlgorithms(var AlgorithmList: TPGPCipherAlgorithms; var Count: Longint): Longint;
var
  Pref		 : String;
  PGPPref	 : pPGPPref;
  DataLength	 : PGPSize;
  PrefData	 : pPGPCipherAlgorithm;
  PrefIndex	 : pPGPCipherAlgorithm;
  AlgIndex	 : Longint;
begin
  Count := 0;
  FillChar(AlgorithmList, SizeOf(TPGPCipherAlgorithms), 0);
  if PGP7X then begin
    Result := kPGPError_PrefNotFound;
    if PrefsFile <> nil then with PrefsFile do if LoadPrefs then begin
      try
	for AlgIndex := 0 to High(AlgorithmList) do begin
	  Pref := PrefsData.Values[AllowedAlgList + IntToStr(succ(AlgIndex))];
	  if Pref <> E then
	    AlgorithmList[AlgIndex] := PGPCipherAlgorithm(StrToIntDef(Pref, 0))
	  else Break;
	end;
	Count := AlgIndex;
	Result := 0;
      except
      end;
    end;
  end
  else begin
    PGPPref := nil;
    Result := PGPclOpenClientPrefs(PGPGetDefaultMemoryMgr, PGPPref);
    if Result = 0 then begin
      try
	Result := PGPclGetPrefData(PGPPref, kPGPPrefAllowedAlgorithmsList, DataLength, Pointer(PrefData));
	if Result = 0 then begin
	  PrefIndex := PrefData;
	  try
	    Count := DataLength div SizeOf(PGPCipherAlgorithm);
	    if Count > succ(High(AlgorithmList)) then Count := succ(High(AlgorithmList));
	    for AlgIndex := 0 to pred(Count) do begin
	      AlgorithmList[AlgIndex] := PrefIndex^;
	      inc(PrefIndex);
	    end;
	  finally
	    PGPFreeData(PrefData);
	  end;
	end;
      finally
	PGPclCloseClientPrefs(PGPPref, false);
      end;
    end;
  end;
end;

initialization
  PrefsFile := TPrefsFile.Create;

finalization
  PrefsFile.Free;
  PrefsFile := nil;

end.

