{$J+,Z4}
unit KeyFuncs;

{------------------------------------------------------------------------------}
{                                                                              }
{     This unit is partly based on Steve Heller's spgpKeyUtil.pas from his     }
{         SPGP sources available from http://www.oz.net/~srheller/spgp/        }
{                                                                              }
{                Portions created by Michael in der Wiesche are                }
{              Copyright (C) 2001-2003 by Michael in der Wiesche               }
{                                                                              }
{------------------------------------------------------------------------------}

interface

uses
  Windows,
  Classes,
  SysUtils,
  TimeUtils,
  KeyPropTypes,
  UTF8,
  pgpBase,
  pgpErrors,
  pgpPubTypes,
  pgpUtilities,
  pgpOptionList,
  pgpMemoryMgr,
  pgpGroups,
  pgpKeys,
  pgpCl;

type
  TKeyRings = Class
  private
    InitCount: Longint;
    RingKeyDB: pPGPKeyDB;
    RingKeySet: pPGPKeySet;
    RingContext: pPGPContext;
  public
    PubringFile: String;
    SecringFile: String;
    GroupsFile: String;
    function UpdateKeyRings: PGPError;
    function SetKeyRings(const Pubring, Secring: String): Longbool;
    function InitKeyRings(var Context: pPGPContext; var KeySetMain: pPGPKeySet): PGPError;
    procedure FreeKeyRings;
  end;

{ "public" functions }
function IsHexID(const ToCheck: String): Longbool;
function ShortHexID(const HexID: String): String;
function KeyRemove(KeyIDCommaText: PChar): Integer;
function KeyEnable(HexKeyID: PChar): Integer;
function KeyDisable(HexKeyID: PChar): Integer;
function KeyRevoke(HexKeyID, Passphrase: PChar): Integer;
function ChangePassphrase(HexKeyID, OldPassphrase, NewPassphrase: PChar): Integer;
// properties, key & user IDs for all keys matching the specified KeyIDCommaText
// if KeyIDCommaText = '' then all keys will be checked
// - returns number of keys found or error
function FindKeyProps(const KeyIDCommaText: String;
		      var KeyPropsList: TKeyPropsList;
		      PropertyFlags, FilterFlags: DWord;
		      KeyOrder: TPGPKeyOrdering): Integer;
{ "private" functions }
function SystemTimeToUnixTimeNum: Integer;
function UnixTimeToLocalTimeStr(UnixSeconds: Integer): String;
function GetSubKeyPropKeyID(SubKey: pPGPKey): String;
function GetKeyPropKeyID(Key: pPGPKey): String;
function GetKeyPropUserID(Key: pPGPKey): String;
function GetKeyPropAlg(Key: pPGPKey): TKeyAlgorithm;
function GetKeyPropLegacy(Key: pPGPKey): Longbool;
function GetKeyPropIsAxiomatic(Key: pPGPKey): Longbool;
function GetKeyFromKeySet(KeySet: pPGPKeySet; var Key: pPGPKey): PGPError;
function GetKeyByHexID(KeySet: pPGPKeySet; const HexID: String; var Key: pPGPKey): PGPError;
function GetKeyFilter(Context: pPGPContext; FilterFlags: DWord; var KeyFilter: pPGPFilter): PGPError;
function GetKeyFilterByAnyID(Context: pPGPContext; const AnyID: String;
			     IncludeSubKeys: Longbool; var KeyFilter: pPGPFilter): PGPError;
function GetKeySetByAnyIDs(Context: pPGPContext;
			   KeySetMain: pPGPKeySet;
			   const KeyIDCommaText: String;
			   var KeySetFound: pPGPKeySet): PGPError;
function GetExclusiveKeySet(var KeySetToCheck: pPGPKeySet; KeySetMain: pPGPKeySet;
			    Context: pPGPContext; IgnoreKnownFlag: Integer): PGPError;
function GetKeySetProps(Context: pPGPContext; KeySet: pPGPKeySet;
			var KeyPropsList: TKeyPropsList;
			PropertyFlags, FilterFlags: DWord;
			KeyOrder: TPGPKeyOrdering): Integer; // - returns number of keys found or error
function AddKeysToKeyRing(Context: pPGPContext; KeySetMain: pPGPKeySet;
			  KeysToImport: Pointer; var KeyPropsList: TKeyPropsList;
			  PropertyFlags: DWord): Integer; // - returns number of keys added or error
function GetHexIDByAnyID(Context: pPGPContext; KeySetMain: pPGPKeySet;
			 const AnyID: String; var HexID: String): Integer; // - returns number of keys found or error
function PassphraseIsValid(Context: pPGPContext; KeySetMain: pPGPKeySet; AnyID, Passphrase: PChar): Longbool;

const KeyRings: TKeyRings = nil;

implementation

uses PrefFuncs;

function IsHexID(const ToCheck: String): Longbool; register; assembler;
asm	// EAX = @ToCheck
  OR	EAX,EAX
  JE	@FALSE
  MOV	ECX,[EAX - 4]
  CMP	ECX,LongHexIDLen
  JE	@START
  CMP	ECX,ShortHexIDLen
  JNE	@FALSE
  @START:
  MOV	DX,[EAX]
  AND	DH,0DFh
  CMP	DX,'X0'
  JNE	@FALSE
  SUB	ECX,2
  ADD	EAX,2
  @LOOP:
  DEC	ECX
  JS	@TRUE
  MOV	DL,[EAX + ECX]
  CMP	DL,'0'
  JB	@FALSE
  CMP	DL,'9'
  JBE	@LOOP
  AND	DL,0DFh
  CMP	DL,'A'
  JB	@FALSE
  CMP	DL,'F'
  JBE	@LOOP
  @FALSE:
  XOR	EAX,EAX
  RET
  @TRUE:
  MOV	EAX,true
end;

function ShortHexID(const HexID: String): String;
begin
  Result := '';
  if IsHexID(HexID) then begin
    case Length(HexID) of
      ShortHexIDLen: Result := HexID;
      LongHexIDLen: Result := '0x' + Copy(HexID, ShortHexIDLen + 1, ShortHexIDLen - 2);
    end;
  end;
end;

function KeyRemove(KeyIDCommaText: PChar): Integer;
var
  Context	: pPGPContext;
  KeySetMain	: pPGPKeySet;
  KeySetFound	: pPGPKeySet;
  KeyCount	: PGPUInt32;
begin
  KeySetFound := nil;
  Result := KeyRings.InitKeyRings(Context, KeySetMain);
  if Result <> 0 then Exit;
  try
    Result := GetKeySetByAnyIDs(Context, KeySetMain, KeyIDCommaText, KeySetFound);
    if Result <> 0 then Exit;
    try
      Result := PGPCountKeys(KeySetFound, KeyCount);
      if Result <> 0 then Exit;
      if KeyCount <> 0 then begin
	Result := PGPRemoveKeys(KeySetFound, KeySetMain);
	if Result <> 0 then Exit;
      end;
    finally
      PGPFreeKeySet(KeySetFound);
    end;
    Result := KeyRings.UpdateKeyRings;
  finally
    KeyRings.FreeKeyRings;
  end;
end;

function KeyEnable(HexKeyID: PChar): Integer;
var
  Context	: pPGPContext;
  KeySetMain	: pPGPKeySet;
  Key		: pPGPKey;
begin
  Result := KeyRings.InitKeyRings(Context, KeySetMain);
  if Result <> 0 then Exit;
  try
    Result := GetKeyByHexID(KeySetMain, HexKeyID, Key);
    if Result <> 0 then Exit;
    Result := PGPEnableKey(Key);
    if Result <> 0 then Exit;
    Result := KeyRings.UpdateKeyRings;
  finally
    KeyRings.FreeKeyRings;
  end;
end;

function KeyDisable(HexKeyID: PChar): Integer;
var
  Context	: pPGPContext;
  KeySetMain	: pPGPKeySet;
  Key		: pPGPKey;
begin
  Result := KeyRings.InitKeyRings(Context, KeySetMain);
  if Result <> 0 then Exit;
  try
    Result := GetKeyByHexID(KeySetMain, HexKeyID, Key);
    if Result <> 0 then Exit;
    Result := PGPDisableKey(Key);
    if Result <> 0 then Exit;
    Result := KeyRings.UpdateKeyRings;
  finally
    KeyRings.FreeKeyRings;
  end;
end;

function KeyRevoke(HexKeyID, Passphrase: PChar): Integer;
var
  Context	: pPGPContext;
  KeySetMain	: pPGPKeySet;
  Key		: pPGPKey;
begin
  Result := KeyRings.InitKeyRings(Context, KeySetMain);
  if Result <> 0 then Exit;
  try
    Result := GetKeyByHexID(KeySetMain, HexKeyID, Key);
    if Result <> 0 then Exit;
    Result := PGPRevokeKey(Key, PGPOPassphrase(Context, Passphrase), PGPOLastOption(Context));
    if Result <> 0 then Exit;
    Result := KeyRings.UpdateKeyRings;
  finally
    KeyRings.FreeKeyRings;
  end;
end;

function ChangePassphrase(HexKeyID, OldPassphrase, NewPassphrase: PChar): Integer;
var
  Context	: pPGPContext;
  KeySetMain	: pPGPKeySet;
  OptionList	: pPGPOptionList;
  Key		: pPGPKey;
  KeyList	: pPGPKeyList;
  KeyIter	: pPGPKeyIter;
  SubKey	: pPGPSubKey;
begin
  Result := KeyRings.InitKeyRings(Context, KeySetMain);
  if Result <> 0 then Exit;
  OptionList := nil;
  try
    Result := PGPBuildOptionList(Context, OptionList,
      [
	PGPOPassphrase(Context, OldPassphrase),
	PGPOPassphrase(Context, NewPassphrase)
      ]);
    if Result <> 0 then Exit;
    try
      Result := GetKeyByHexID(KeySetMain, HexKeyID, Key);
      if Result <> 0 then Exit;
      Result := PGPChangePassphrase(Key, OptionList, PGPOLastOption(Context));
      if Result <> 0 then Exit;
      if not GetKeyPropLegacy(Key) then begin
	KeyList := nil;
	KeyIter := nil;
	Result := PGPOrderKeySet(KeySetMain, kPGPKeyOrdering_Any, PGPFalse, KeyList);
	if Result <> 0 then Exit;
	try
	  Result := PGPNewKeyIter(KeyList, KeyIter);
	  if Result <> 0 then Exit;
	  try
	    PGPKeyIterSeek(KeyIter, Key);
	    while PGPKeyIterNextKeyDBObj(KeyIter, kPGPKeyDBObjType_SubKey, SubKey) = 0 do begin
	      Result := PGPChangeSubKeyPassphrase(SubKey, OptionList, PGPOLastOption(Context));
	      if Result <> 0 then Exit;
	    end;
	  finally
	    PGPFreeKeyIter(KeyIter);
	  end;
	finally
	  PGPFreeKeyList(KeyList);
	end;
      end;
      Result := KeyRings.UpdateKeyRings;
    finally
      PGPFreeOptionList(OptionList);
    end;
  finally
    KeyRings.FreeKeyRings;
  end;
end;

function FindKeyProps(const KeyIDCommaText: String;
		      var KeyPropsList: TKeyPropsList;
		      PropertyFlags, FilterFlags: DWord;
		      KeyOrder: TPGPKeyOrdering): Integer;
var
  Context	: pPGPContext;
  KeySetMain	: pPGPKeySet;
  KeySetFound	: pPGPKeySet;
  KeyFilter	: pPGPFilter;
  KeySetFiltered: pPGPKeySet;
  KeyCount	: PGPUInt32;
begin
  KeySetFound := nil;
  KeyFilter := nil;
  KeySetFiltered := nil;
  Result := KeyRings.InitKeyRings(Context, KeySetMain);
  if Result <> 0 then Exit;
  try
    if KeyIDCommaText = '' then
      KeySetFound := KeySetMain // all keys on key rings
    else Result := GetKeySetByAnyIDs(Context, KeySetMain, KeyIDCommaText, KeySetFound); // key(s) matching KeyData
    if Result <> 0 then Exit;
    try
      if PropertyFlags <> 0 then begin
	Result := GetKeySetProps(Context, KeySetFound,
				KeyPropsList, PropertyFlags,
				FilterFlags, KeyOrder);
      end
      else begin
	if FilterFlags <> KeyFilterFlag_AllKeys then begin
	  Result := GetKeyFilter(Context, FilterFlags, KeyFilter);
	  try
	    if Result <> 0 then Exit;
	    Result := PGPFilterKeySet(KeySetFound, KeyFilter, KeySetFiltered);
	  finally
	    PGPFreeFilter(KeyFilter);
	  end;
	end
	else KeySetFiltered := KeySetFound;
	if Result <> 0 then Exit;
	try
	  Result := PGPCountKeys(KeySetFiltered, KeyCount);
	  if Result = 0 then Result := KeyCount;
	finally
	  if KeySetFiltered <> KeySetFound then PGPFreeKeySet(KeySetFiltered);
	end;
      end;
    finally
      if KeySetFound <> KeySetMain then PGPFreeKeySet(KeySetFound);
    end;
  finally
    KeyRings.FreeKeyRings;
  end;
end;

function GetKeyFromKeySet(KeySet: pPGPKeySet; var Key: pPGPKey): PGPError;
var
  KeyList	: pPGPKeyList;
  KeyIter	: pPGPKeyIter;
begin
  Key := nil;
  KeyList := nil;
  KeyIter := nil;
  Result := PGPOrderKeySet(KeySet, kPGPKeyOrdering_UserID, PGPFalse, KeyList);
  if Result <> 0 then Exit;
  try
    Result := PGPNewKeyIter(KeyList, KeyIter);
    if Result <> 0 then Exit;
    try
      Result := PGPKeyIterNextKeyDBObj(KeyIter, kPGPKeyDBObjType_Key, Key);
    finally
      PGPFreeKeyIter(KeyIter);
    end;
  finally
    PGPFreeKeyList(KeyList);
  end;
end;

function GetKeyByHexID(KeySet: pPGPKeySet; const HexID: String; var Key: pPGPKey): PGPError;
var
  PGPKeyID	: TPGPKeyID7;
begin
  Key := nil;
  Result := PGPGetKeyIDFromString(PChar(HexID), kPGPPublicKeyAlgorithm_Invalid, PGPKeyID);
  if Result = 0 then Result := PGPGetKeyByKeyID(KeySet, PGPKeyID, kPGPPublicKeyAlgorithm_Invalid, Key);
end;

function GetKeyFilter(Context: pPGPContext; FilterFlags: DWord; var KeyFilter: pPGPFilter): PGPError;
var
  KeyBoolFilter	: pPGPFilter;
  KeyAlgFilter	: pPGPFilter;
begin
  Result := 0;
  KeyFilter := nil;
  KeyBoolFilter := nil;
  KeyAlgFilter := nil;
  try
    if FilterFlags and KeyFilterMask_Boolean <> 0 then begin
      if FilterFlags and KeyFilterFlag_CanEncrypt <> 0 then begin
	Result := PGPNewKeyDBObjBooleanFilter(Context, kPGPKeyProperty_CanEncrypt, PGPTrue, KeyBoolFilter);
      end
      else if FilterFlags and KeyFilterFlag_CanDecrypt <> 0 then begin
	Result := PGPNewKeyDBObjBooleanFilter(Context, kPGPKeyProperty_CanDecrypt, PGPTrue, KeyBoolFilter);
      end
      else if FilterFlags and KeyFilterFlag_CanSign <> 0 then begin
	Result := PGPNewKeyDBObjBooleanFilter(Context, kPGPKeyProperty_CanSign, PGPTrue, KeyBoolFilter);
      end
      else if FilterFlags and KeyFilterFlag_CanVerify <> 0 then begin
	Result := PGPNewKeyDBObjBooleanFilter(Context, kPGPKeyProperty_CanVerify, PGPTrue, KeyBoolFilter);
      end
      else if FilterFlags and KeyFilterFlag_Enabled <> 0 then begin
	Result := PGPNewKeyDBObjBooleanFilter(Context, kPGPKeyProperty_IsDisabled, PGPFalse, KeyBoolFilter);
      end
      else if FilterFlags and KeyFilterFlag_Disabled <> 0 then begin
	Result := PGPNewKeyDBObjBooleanFilter(Context, kPGPKeyProperty_IsDisabled, PGPTrue, KeyBoolFilter);
      end;
    end;
    if Result <> 0 then Exit;
    if FilterFlags and KeyFilterMask_Algorithm <> 0 then begin
      if FilterFlags and KeyFilterFlag_DHDSS <> 0 then begin
	Result := PGPNewKeyDBObjNumericFilter(Context, kPGPKeyProperty_AlgorithmID,
					      kPGPPublicKeyAlgorithm_DSA, kPGPMatchEqual, KeyAlgFilter);
      end else if FilterFlags and KeyFilterFlag_RSA <> 0 then begin
	Result := PGPNewKeyDBObjNumericFilter(Context, kPGPKeyProperty_AlgorithmID,
					      kPGPPublicKeyAlgorithm_RSA, kPGPMatchEqual, KeyAlgFilter);
      end else if FilterFlags and KeyFilterFlag_V4 <> 0 then begin
	if PGP7X then begin
	  Result := PGPNewKeyDBObjNumericFilter(Context, kPGPKeyProperty_Version,
						kPGPKeyVersion_V4, kPGPMatchEqual, KeyAlgFilter);
	end
	else begin
	  Result := PGPNewKeyDBObjNumericFilter(Context, kPGPKeyProperty_AlgorithmID,
						kPGPPublicKeyAlgorithm_DSA, kPGPMatchEqual, KeyAlgFilter);
	end;
      end else if FilterFlags and KeyFilterFlag_V3 <> 0 then begin
	if PGP7X then begin
	  Result := PGPNewKeyDBObjNumericFilter(Context, kPGPKeyProperty_Version,
						kPGPKeyVersion_V3, kPGPMatchLessOrEqual, KeyAlgFilter);
	end
	else begin
	  Result := PGPNewKeyDBObjNumericFilter(Context, kPGPKeyProperty_AlgorithmID,
						kPGPPublicKeyAlgorithm_RSA, kPGPMatchEqual, KeyAlgFilter);
	end;
      end;
    end;
    if Result <> 0 then Exit;
    if (KeyBoolFilter <> nil) and (KeyAlgFilter <> nil) then
      Result := PGPIntersectFilters(KeyBoolFilter, KeyAlgFilter, KeyFilter)
    else if KeyBoolFilter <> nil then
      KeyFilter := KeyBoolFilter
    else if KeyAlgFilter <> nil then
      KeyFilter := KeyAlgFilter;
  finally
    if Result <> 0 then begin
      if KeyBoolFilter <> nil then PGPFreeFilter(KeyBoolFilter);
      if KeyAlgFilter <> nil then PGPFreeFilter(KeyAlgFilter);
    end;
  end;
end;

function GetKeyFilterByAnyID(Context: pPGPContext; const AnyID: String;
			     IncludeSubKeys: Longbool; var KeyFilter: pPGPFilter): PGPError;
var
  PGPKeyID	: TPGPKeyID7;
  KeyIDFilter	: pPGPFilter;
  SubKeyIDFilter: pPGPFilter;
  UTF8Filter	: pPGPFilter;
  AnsiFilter	: pPGPFilter;
  UTF8ID	: UTF8String;
begin
  Result := 0;
  KeyFilter := nil;
  if IsHexID(AnyID) then begin
    KeyIDFilter := nil;
    SubKeyIDFilter := nil;
    try
      Result := PGPGetKeyIDFromString(PChar(AnyID), kPGPPublicKeyAlgorithm_Invalid, PGPKeyID);
      if Result <> 0 then Exit;
      Result := PGPNewKeyDBObjDataFilter(Context, kPGPKeyProperty_KeyID, @PGPKeyID,
					 SizeOf(TPGPKeyID7), kPGPMatchEqual, KeyIDFilter);
      if Result <> 0 then Exit;
      if IncludeSubKeys then begin
	Result := PGPNewKeyDBObjDataFilter(Context, kPGPSubKeyProperty_KeyID, @PGPKeyID,
					   SizeOf(TPGPKeyID7), kPGPMatchEqual, SubKeyIDFilter);
	if Result <> 0 then Exit;
	Result := PGPUnionFilters(KeyIDFilter, SubKeyIDFilter, KeyFilter);
      end
      else KeyFilter := KeyIDFilter;
    finally
      if Result <> 0 then begin
	if KeyIDFilter <> nil then PGPFreeFilter(KeyIDFilter);
	if SubKeyIDFilter <> nil then PGPFreeFilter(SubKeyIDFilter);
      end;
    end;
  end
  else begin
    UTF8Filter := nil;
    AnsiFilter := nil;
    try
      Result := PGPNewKeyDBObjDataFilter(Context, kPGPUserIDProperty_Name, PChar(AnyID),
					 Length(AnyID), kPGPMatchSubString, AnsiFilter);
      if Result <> 0 then Exit;
      if PGP8X then begin
	UTF8ID := AnsiToUtf8(AnyID);
	Result := PGPNewKeyDBObjDataFilter(Context, kPGPUserIDProperty_Name, PChar(UTF8ID),
					   Length(UTF8ID), kPGPMatchSubString, UTF8Filter);
	if Result <> 0 then Exit;
	Result := PGPUnionFilters(UTF8Filter, AnsiFilter, KeyFilter);
      end
      else KeyFilter := AnsiFilter;
    finally
      if Result <> 0 then begin
	if UTF8Filter <> nil then PGPFreeFilter(UTF8Filter);
	if AnsiFilter <> nil then PGPFreeFilter(AnsiFilter);
      end;
    end;
  end;
end;

function GetKeySetByAnyIDs(Context: pPGPContext;
			   KeySetMain: pPGPKeySet;
			   const KeyIDCommaText: String;
			   var KeySetFound: pPGPKeySet): PGPError;
var
  KeyDataList	: TStringList;
  KeyIndex	: PGPUInt32;
  KeyString	: String;
  KeyFilter	: pPGPFilter;
  KeySetFiltered: pPGPKeySet;
  KeyCount	: PGPUInt32;
begin
  KeySetFound := nil;
  if KeyIDCommaText <> '' then begin
    KeyDataList := TStringList.Create;
    KeyFilter := nil;
    KeySetFiltered := nil;
    try
      KeyDataList.CommaText := KeyIDCommaText;
      if PGP7X then
	Result := PGPNewEmptyKeySet(PGPPeekKeySetKeyDB(KeySetMain), KeySetFound)
      else Result := PGPNewKeySet(Context, KeySetFound);
      if Result <> 0 then Exit;
      for KeyIndex := 0 to pred(KeyDataList.Count) do begin
	KeyString := Trim(KeyDataList[KeyIndex]);
	if KeyString <> '' then begin
	  Result := GetKeyFilterByAnyID(Context, KeyString, false, KeyFilter);
	  if Result <> 0 then Exit;
	  try
	    Result := PGPFilterKeySet(KeySetMain, KeyFilter, KeySetFiltered);
	    if Result <> 0 then Exit;
	    try
	      Result := PGPCountKeys(KeySetFiltered, KeyCount);
	      if Result <> 0 then Exit;
	      if KeyCount <> 0 then begin
		Result := PGPAddKeys(KeySetFiltered, KeySetFound);
		if Result <> 0 then Exit;
	      end
	      else begin
		Result := kPGPError_PublicKeyNotFound;
		PGPFreeKeySet(KeySetFound);
		KeySetFound := nil;
		Exit;
	      end;
	    finally
	      PGPFreeKeySet(KeySetFiltered);
	    end;
	  finally
	    PGPFreeFilter(KeyFilter);
	  end;
	end;
      end;
    finally
      KeyDataList.Free;
    end;
  end
  else Result := kPGPError_PublicKeyNotFound;
end;

function GetExclusiveKeySet(var KeySetToCheck: pPGPKeySet; KeySetMain: pPGPKeySet;
			    Context: pPGPContext; IgnoreKnownFlag: Integer): PGPError;
var
  KeyListToCheck: pPGPKeyList;
  KeyIterToCheck: pPGPKeyIter;
  KeyToCheck	: pPGPKey;
  KeyIDToCheck	: TKeyID;
  UserIDToCheck	: TUserID;
  KeyCount	: PGPUInt32;
  IDSize	: PGPSize;
  UserIDFound	: pPGPUserID;
  UserIDProp	: PGPBoolean;
  KeyFilter	: pPGPFilter;
  KeySetFound	: pPGPKeySet;
  KeySetToRemove: pPGPKeySet;
begin
  KeyToCheck := nil;
  KeyListToCheck := nil;
  KeyIterToCheck := nil;
  try
    if PGP7X then
      Result := PGPNewKeyIterFromKeySet(KeySetToCheck, KeyIterToCheck)
    else begin
      Result := PGPOrderKeySet(KeySetToCheck, kPGPKeyOrdering_UserID, PGPFalse, KeyListToCheck);
      if Result <> 0 then Exit;
      Result := PGPNewKeyIter(KeyListToCheck, KeyIterToCheck);
    end;
    if Result <> 0 then Exit;
    try
      if (IgnoreKnownFlag and IgnoreFlag_ByHexID) <> 0 then begin
	while PGPKeyIterNextKeyDBObj(KeyIterToCheck, kPGPKeyDBObjType_Key, KeyToCheck) = 0 do begin
	  KeyFilter := nil;
	  KeySetFound := nil;
	  KeySetToRemove := nil;
	  try
	    if (PGPGetKeyDBObjDataProperty(KeyToCheck, kPGPKeyProperty_KeyID,
					   @KeyIDToCheck, kPGPMaxKeyIDStringSize, IDSize) = 0)
	    and (PGPNewKeyDBObjDataFilter(Context, kPGPKeyProperty_KeyID, @KeyIDToCheck,
					  IDSize, kPGPMatchEqual, KeyFilter) = 0)
	    and (PGPFilterKeySet(KeySetMain, KeyFilter, KeySetFound) = 0)
	    and (PGPCountKeys(KeySetFound, KeyCount) = 0) and (KeyCount <> 0)
	    and (PGPNewSingletonKeySet(KeyToCheck, KeySetToRemove) = 0) then begin
	      PGPRemoveKeys(KeySetToRemove, KeySetToCheck);
	    end;
	  finally
	    PGPFreeFilter(KeyFilter);
	    PGPFreeKeySet(KeySetFound);
	    PGPFreeKeySet(KeySetToRemove);
	  end;
	end;
	PGPKeyIterRewindKeyDBObj(KeyIterToCheck, kPGPKeyDBObjType_Key);
      end;
      if (PGPCountKeys(KeySetToCheck, KeyCount) = 0) and (KeyCount = 0) then Exit;
      if (IgnoreKnownFlag and IgnoreFlag_ByUserID) <> 0 then begin
	while (PGPKeyIterNextKeyDBObj(KeyIterToCheck, kPGPKeyDBObjType_Key, KeyToCheck) = 0)
	and (PGPKeyIterRewindKeyDBObj(KeyIterToCheck, kPGPKeyDBObjType_UserID) = 0) do begin
	  while (PGPKeyIterNextKeyDBObj(KeyIterToCheck, kPGPKeyDBObjType_UserID, UserIDFound) = 0)
	  and (PGPGetKeyDBObjBooleanProperty(UserIDFound, kPGPUserIDProperty_IsAttribute, UserIDProp) = 0)
	  and not boolean(UserIDProp) do begin
	    KeyFilter := nil;
	    KeySetFound := nil;
	    KeySetToRemove := nil;
	    try
	      if (PGPGetKeyDBObjDataProperty(UserIDFound, kPGPUserIDProperty_Name,
					     @UserIDToCheck, SizeOf(TUserID), IDSize) = 0)
	      and (PGPNewKeyDBObjDataFilter(Context, kPGPUserIDProperty_Name, @UserIDToCheck,
					    IDSize, kPGPMatchEqual, KeyFilter) = 0)
	      and (PGPFilterKeySet(KeySetMain, KeyFilter, KeySetFound) = 0)
	      and (PGPCountKeys(KeySetFound, KeyCount) = 0) and (KeyCount <> 0)
	      and (PGPNewSingletonKeySet(KeyToCheck, KeySetToRemove) = 0) then begin
		PGPRemoveKeys(KeySetToRemove, KeySetToCheck);
		Break;
	      end;
	    finally
	      PGPFreeFilter(KeyFilter);
	      PGPFreeKeySet(KeySetFound);
	      PGPFreeKeySet(KeySetToRemove);
	    end;
	  end;
	  if (PGPCountKeys(KeySetToCheck, KeyCount) = 0) and (KeyCount = 0) then Break;
	end;
      end;
    finally
      PGPFreeKeyIter(KeyIterToCheck);
    end;
  finally
    PGPFreeKeyList(KeyListToCheck);
  end;
end;

function GetFingerprintString(const Fingerprint: TFingerprint; FPSize: PGPSize): String; register; assembler;
asm	// EAX = @Fingerprint, EDX = FPSize, ECX = @Result
  PUSH	EBX
  PUSH	EDI
  PUSH	ESI
  PUSH	EBP
  MOV	ESI,EAX
  MOV	EDI,ECX
  MOV	EAX,EDX
  MOV	EBX,EDX
  SHL	EAX,1
  SHR	EBX,1
  MOV	EBP,EBX
  ADD	EAX,EBX
  INC	EBP
  PUSH	EDX
  PUSH	EAX
  CALL	SYSTEM.@NEWANSISTRING
  MOV	[EDI],EAX
  MOV	EDI,EAX
  POP	EBX
  POP	ECX
  SUB	EBX,2
  SHR	ECX,1
  SHR	EBP,1
  DEC	ECX
  @LOOP:
  MOVZX	EAX,BYTE PTR[ESI + ECX * 2 + 1]
  MOV	EDX,EAX
  AND	EAX,0Fh
  SHR	EDX,4
  MOV	AH,BYTE PTR[@HEXCHARS + EAX]
  MOV	AL,BYTE PTR[@HEXCHARS + EDX]
  MOV	[EDI + EBX],AX
  MOVZX	EAX,BYTE PTR[ESI + ECX * 2]
  MOV	EDX,EAX
  AND	EAX,0Fh
  SHR	EDX,4
  MOV	AH,BYTE PTR[@HEXCHARS + EAX]
  MOV	AL,BYTE PTR[@HEXCHARS + EDX]
  MOV	[EDI + EBX - 2],AX
  CMP	EBP,ECX
  JE	@MIDDLE
  SUB	EBX,5
  DEC	ECX
  JS	@END
  MOV	BYTE PTR[EDI + EBX + 2],' '
  JMP	@LOOP
  @MIDDLE:
  SUB	EBX,6
  DEC	ECX
  MOV	WORD PTR[EDI + EBX + 2],'  '
  JMP	@LOOP
  @END:
  POP	EBP
  POP	ESI
  POP	EDI
  POP	EBX
  RET
  @HEXCHARS:
  DB	'0123456789ABCDEF';
end;

function SystemTimeToUnixTimeNum: Integer;
var SystemTime: TSystemTime;
begin
  GetSystemTime(SystemTime);
  Result := SystemTimeToUnixTime(SystemTime);
end;

function UnixTimeToLocalTimeStr(UnixSeconds: Integer): String;
begin
  Result := FormatDateTime('ddd, dd mmm yyyy, tt', UnixTimeToLocalTime(UnixSeconds));
end;

function GetSubKeyPropKeyID(SubKey: pPGPKey): String;
var
  PGPKeyID	: TPGPKeyID7;
  KeyID		: TKeyID;
begin
  Result := '';
  if (PGPGetKeyIDFromSubKey(SubKey, PGPKeyID) = 0)
  and (PGPGetKeyIDString(PGPKeyID, kPGPKeyIDString_Full, KeyID) = 0) then begin
    Result := KeyID;
  end;
end;

function GetSubKeyPropIsRevoked(SubKey: pPGPKey): Longbool;
var
  Prop		: PGPBoolean;
begin
  Result := false;
  if PGPGetKeyDBObjBooleanProperty(SubKey, kPGPSubKeyProperty_IsRevoked, Prop) = 0 then begin
    Result := boolean(Prop);
  end;
end;

function GetSubKeyPropIsExpired(SubKey: pPGPKey): Longbool;
var
  Prop		: PGPBoolean;
begin
  Result := false;
  if PGPGetKeyDBObjBooleanProperty(SubKey, kPGPSubKeyProperty_IsExpired, Prop) = 0 then begin
    Result := boolean(Prop);
  end;
end;

function GetKeyPropKeyID(Key: pPGPKey): String;
var
  PGPKeyID	: TPGPKeyID7;
  KeyID		: TKeyID;
begin
  Result := '';
  if (PGPGetKeyIDFromKey(Key, PGPKeyID) = 0)
  and (PGPGetKeyIDString(PGPKeyID, kPGPKeyIDString_Full, KeyID) = 0) then begin
    Result := KeyID;
  end;
end;

function GetKeyPropUserID(Key: pPGPKey): String;
var
  UserIDBuf	: TUserID;
  IDSize	: PGPSize;
begin
  Result := '';
  if PGPGetPrimaryUserIDNameBuffer(Key, UserIDBuf, SizeOf(TUserID), IDSize) = 0 then begin
    if PGP8X then
      Result := Utf8OrAnsi(UserIDBuf)
    else Result := UserIDBuf;
  end;
end;

function GetKeyPropFingerprint(Key: pPGPKey): String;
var
  Fingerprint	: TFingerprint;
  FPSize	: PGPSize;
begin
  Result := '';
  if (PGPGetKeyDBObjDataProperty(Key, kPGPKeyProperty_Fingerprint, @Fingerprint, kPGPmaxFingerprintSize, FPSize) = 0)
  and (FPSize <> 0) then begin
    Result := GetFingerprintString(Fingerprint, FPSize);
  end;
end;

// local time
function GetKeyPropKeyCreationTimeStr(Key: pPGPKey): String;
var
  CreaTime	: PGPTime;
begin
  Result := '';
  if PGPGetKeyDBObjTimeProperty(Key, kPGPKeyProperty_Creation, CreaTime) = 0 then begin
    Result := UnixTimeToLocalTimeStr(PGPGetStdTimeFromPGPTime(CreaTime));
  end;
end;

// local time
function GetKeyPropKeyExpirationTimeStr(Key: pPGPKey): String;
var
  ExpTime	: PGPTime;
  CreaTime	: PGPTime;
begin
  Result := '';
  if (PGPGetKeyDBObjTimeProperty(Key, kPGPKeyProperty_Expiration, ExpTime) = 0)
  and (PGPGetKeyDBObjTimeProperty(Key, kPGPKeyProperty_Creation, CreaTime) = 0) then begin
    if ExpTime > CreaTime then Result := UnixTimeToLocalTimeStr(PGPGetStdTimeFromPGPTime(ExpTime));
  end;
end;

function GetKeyPropKeyBits(Key: pPGPKey): String;
var
  Error		: PGPError;
  KeySet	: pPGPKeySet;
  KeyList	: pPGPKeyList;
  KeyIter	: pPGPKeyIter;
  SubKey	: pPGPSubKey;
  KeyAlg	: PGPInt32;
  KeyBits	: PGPInt32;
  SubKeyBits	: PGPInt32;
begin
  Result := '';
  Error := 0;
  KeySet := nil;
  KeyList := nil;
  KeyIter := nil;
  SubKey := nil;
  if (PGPGetKeyDBObjNumericProperty(Key, kPGPKeyProperty_AlgorithmID, KeyAlg) = 0)
  and (PGPGetKeyDBObjNumericProperty(Key, kPGPKeyProperty_Bits, KeyBits) = 0)
  and (PGPNewSingletonKeySet(Key, KeySet) = 0) then begin
    try
      if PGPOrderKeySet(KeySet, kPGPKeyOrdering_Any, PGPFalse, KeyList) <> 0 then Exit;
      try
	if PGPNewKeyIter(KeyList, KeyIter) <> 0 then Exit;
	try
	  SubKeyBits := 0;
	  PGPKeyIterSeek(KeyIter, Key);
	  while (PGPKeyIterNextKeyDBObj(KeyIter, kPGPKeyDBObjType_SubKey, SubKey) = 0) do begin
	    if not (GetSubKeyPropIsRevoked(SubKey) or GetSubKeyPropIsExpired(SubKey)) then begin
	      Error := PGPGetKeyDBObjNumericProperty(SubKey, kPGPSubKeyProperty_Bits, SubKeyBits);
	      Break;
	    end;
	  end;
	  if Error = 0 then begin
	    if SubKeyBits > 0 then
	      Result := IntToStr(SubKeyBits) + '/' + IntToStr(KeyBits) + ' bits'
	    else Result := IntToStr(KeyBits) + ' bits';
	  end;
	finally
	  PGPFreeKeyIter(KeyIter);
	end;
      finally
	PGPFreeKeyList(KeyList);
      end;
    finally
      PGPFreeKeySet(KeySet);
    end;
  end;
end;

function GetKeyPropAlg(Key: pPGPKey): TKeyAlgorithm;
var
  KeyAlg	: PGPInt32;
  KeySet	: pPGPKeySet;
  KeyList	: pPGPKeyList;
  KeyIter	: pPGPKeyIter;
  SubKey	: pPGPSubKey;
begin
  Result := KeyAlgorithm_Invalid;
  if PGPGetKeyDBObjNumericProperty(Key, kPGPKeyProperty_AlgorithmID, KeyAlg) = 0 then begin
    Result := TKeyAlgorithm(KeyAlg);
    if Result = KeyAlgorithm_DSS then begin
      KeySet := nil;
      KeyList := nil;
      KeyIter := nil;
      if PGPNewSingletonKeySet(Key, KeySet) <> 0 then Exit;
      try
	if PGPOrderKeySet(KeySet, kPGPKeyOrdering_Any, PGPFalse, KeyList) <> 0 then Exit;
	try
	  if PGPNewKeyIter(KeyList, KeyIter) <> 0 then Exit;
	  try
	    PGPKeyIterSeek(KeyIter, Key);
	    if PGPKeyIterNextKeyDBObj(KeyIter, kPGPKeyDBObjType_SubKey, SubKey) = 0 then Result := KeyAlgorithm_DHDSS;
	  finally
	    PGPFreeKeyIter(KeyIter);
	  end;
	finally
	  PGPFreeKeyList(KeyList);
	end;
      finally
	PGPFreeKeySet(KeySet);
      end;
    end;
  end;
end;

function GetKeyPropLegacy(Key: pPGPKey): Longbool;
var
  KeyVersion	: PGPInt32;
begin
  Result := true;
  if PGP7X then begin
    if PGPGetKeyDBObjNumericProperty(Key, kPGPKeyProperty_Version, KeyVersion) = 0 then begin
      Result := (KeyVersion <= kPGPKeyVersion_V3);
    end;
  end
  else Result := (GetKeyPropAlg(Key) = KeyAlgorithm_RSA);
end;

function GetKeyPropTrust(Key: pPGPKey): TTrustLevel;
var
  KeyTrust	: PGPInt32;
begin
  Result := KeyTrust_Undefined;
  if PGPGetKeyDBObjNumericProperty(Key, kPGPKeyProperty_Trust, KeyTrust) = 0 then begin
    Result := TTrustLevel(KeyTrust);
  end;
end;

function GetKeyPropValidity(Key: pPGPKey): TValidityLevel;
var
  KeyValidity	: PGPInt32;
begin
  Result := Validity_Unknown;
  if PGPGetKeyDBObjNumericProperty(Key, kPGPKeyProperty_Validity, KeyValidity) = 0 then begin
    Result := TValidityLevel(KeyValidity);
  end;
end;

// UTC/GMT Unix format seconds
function GetKeyPropKeyCreationTime(Key: pPGPKey): Integer;
var
  CreaTime	: PGPTime;
begin
  Result := 0;
  if PGPGetKeyDBObjTimeProperty(Key, kPGPKeyProperty_Creation, CreaTime) = 0 then begin
    Result := PGPGetStdTimeFromPGPTime(CreaTime);
  end;
end;

// UTC/GMT Unix format seconds
function GetKeyPropKeyExpirationTime(Key: pPGPKey): Integer;
var
  ExpTime	: PGPTime;
  CreaTime	: PGPTime;
begin
  Result := 0;
  if (PGPGetKeyDBObjTimeProperty(Key, kPGPKeyProperty_Expiration, ExpTime) = 0)
  and (PGPGetKeyDBObjTimeProperty(Key, kPGPKeyProperty_Creation, CreaTime) = 0) then begin
    if ExpTime > CreaTime then Result := PGPGetStdTimeFromPGPTime(ExpTime);
  end;
end;

function GetKeyPropIsCorrupt(Key: pPGPKey): Longbool;
var
  Prop		: PGPBoolean;
begin
  Result := false;
  if PGPGetKeyDBObjBooleanProperty(Key, kPGPKeyProperty_IsNotCorrupt, Prop) = 0 then begin
    Result := not boolean(Prop);
  end;
end;

function GetKeyPropIsSecret(Key: pPGPKey): Longbool;
var
  Prop		: PGPBoolean;
begin
  Result := false;
  if PGPGetKeyDBObjBooleanProperty(Key, kPGPKeyProperty_IsSecret, Prop) = 0 then begin
    Result := boolean(Prop);
  end;
end;

function GetKeyPropIsAxiomatic(Key: pPGPKey): Longbool;
var
  Prop		: PGPBoolean;
begin
  Result := false;
  if PGPGetKeyDBObjBooleanProperty(Key, kPGPKeyProperty_IsAxiomatic, Prop) = 0 then begin
    Result := boolean(Prop);
  end;
end;

function GetKeyPropIsRevoked(Key: pPGPKey): Longbool;
var
  Prop		: PGPBoolean;
begin
  Result := false;
  if PGPGetKeyDBObjBooleanProperty(Key, kPGPKeyProperty_IsRevoked, Prop) = 0 then begin
    Result := boolean(Prop);
  end;
end;

function GetKeyPropIsDisabled(Key: pPGPKey): Longbool;
var
  Prop		: PGPBoolean;
begin
  Result := false;
  if PGPGetKeyDBObjBooleanProperty(Key, kPGPKeyProperty_IsDisabled, Prop) = 0 then begin
    Result := boolean(Prop);
  end;
end;

function GetKeyPropIsExpired(Key: pPGPKey): Longbool;
var
  Prop		: PGPBoolean;
begin
  Result := false;
  if PGPGetKeyDBObjBooleanProperty(Key, kPGPKeyProperty_IsExpired, Prop) = 0 then begin
    Result := boolean(Prop);
  end;
end;

function GetKeyPropIsSecretShared(Key: pPGPKey): Longbool;
var
  Prop		: PGPBoolean;
begin
  Result := false;
  if PGPGetKeyDBObjBooleanProperty(Key, kPGPKeyProperty_IsSecretShared, Prop) = 0 then begin
    Result := boolean(Prop);
  end;
end;

function GetKeyPropCanEncrypt(Key: pPGPKey): Longbool;
var
  Prop		: PGPBoolean;
begin
  Result := false;
  if PGPGetKeyDBObjBooleanProperty(Key, kPGPKeyProperty_CanEncrypt, Prop) = 0 then begin
    Result := boolean(Prop);
  end;
end;

function GetKeyPropCanDecrypt(Key: pPGPKey): Longbool;
var
  Prop		: PGPBoolean;
begin
  Result := false;
  if PGPGetKeyDBObjBooleanProperty(Key, kPGPKeyProperty_CanDecrypt, Prop) = 0 then begin
    Result := boolean(Prop);
  end;
end;

function GetKeyPropCanSign(Key: pPGPKey): Longbool;
var
  Prop		: PGPBoolean;
begin
  Result := false;
  if PGPGetKeyDBObjBooleanProperty(Key, kPGPKeyProperty_CanSign, Prop) = 0 then begin
    Result := boolean(Prop);
  end;
end;

function GetKeyPropCanVerify(Key: pPGPKey): Longbool;
var
  Prop		: PGPBoolean;
begin
  Result := false;
  if PGPGetKeyDBObjBooleanProperty(Key, kPGPKeyProperty_CanVerify, Prop) = 0 then begin
    Result := boolean(Prop);
  end;
end;

function GetKeyPropHasRevoker(KeySet: pPGPKeySet; Key: pPGPKey; KeyPropsRec: pKeyPropsRec): Longbool;
var
  RevKeyCount	: PGPUInt32;
  RevKeyIndex	: PGPUInt32;
  RevKey	: pPGPKey;
  RevKeyID	: TPGPKeyID7;
  KeyID		: TKeyID;
begin
  Result := false;
  if (PGPCountRevocationKeys(Key, RevKeyCount) = 0) and (RevKeyCount > 0) then begin
    Result := true;
    for RevKeyIndex := 0 to pred(RevKeyCount) do begin
      if PGPGetIndexedRevocationKey(Key, KeySet, RevKeyIndex, RevKey, RevKeyID) = 0 then begin
	if PGPGetKeyIDString(RevKeyID, kPGPKeyIDString_Full, KeyID) = 0 then begin
	  KeyPropsRec^.kRevKeyIDList.Add(KeyID);
	end;
      end;
    end;
  end;
end;

function GetKeyPropHasARR(KeySet: pPGPKeySet; Key: pPGPKey; KeyPropsRec: pKeyPropsRec): TADKType;
var
  ARKeyCount	: PGPUInt32;
  ARKeyIndex	: PGPUInt32;
  ARKey		: pPGPKey;
  ARKeyID	: TPGPKeyID7;
  ARClass	: PGPByte;
  KeyID		: TKeyID;
begin
  Result := NoADK;
  ARKey := nil;
  if (PGPCountAdditionalRecipientRequests(Key, ARKeyCount) = 0) and (ARKeyCount > 0) then begin
    Result := SimpleADK;
    for ARKeyIndex := 0 to pred(ARKeyCount) do begin
      if PGPGetIndexedAdditionalRecipientRequestKey(Key, KeySet, ARKeyIndex, ARKey, ARKeyID, ARClass) = 0 then begin
	if ARClass > 0 then Result := EnforcedADK;
	if PGPGetKeyIDString(ARKeyID, kPGPKeyIDString_Full, KeyID) = 0 then begin
	  KeyPropsRec^.kADKeyIDList.Add(KeyID);
	end;
      end;
    end;
  end;
end;

function GetKeyPropHasSubKey(KeySet: pPGPKeySet; Key: pPGPKey): Longbool;
var
  KeyList	: pPGPKeyList;
  KeyIter	: pPGPKeyIter;
  SubKey	: pPGPSubKey;
begin
  Result := false;
  if not GetKeyPropLegacy(Key) then begin
    KeyList := nil;
    KeyIter := nil;
    if PGPOrderKeySet(KeySet, kPGPKeyOrdering_Any, PGPFalse, KeyList) = 0 then begin
      try
	if PGPNewKeyIter(KeyList, KeyIter) = 0 then begin
	  try
	    PGPKeyIterSeek(KeyIter, Key);
	    Result := (PGPKeyIterNextKeyDBObj(KeyIter, kPGPKeyDBObjType_SubKey, SubKey) = 0);
	  finally
	    PGPFreeKeyIter(KeyIter);
	  end;
	end;
      finally
	PGPFreeKeyList(KeyList);
      end;
    end;
  end;
end;

function GetKeyPropUserIDs(Key: pPGPKey; KeyPropsRec: pKeyPropsRec; IncludeSignerIDs: Longbool): Integer;
var
  KeySet	: pPGPKeySet;
  KeyList	: pPGPKeyList;
  KeyIter	: pPGPKeyIter;
  UserID	: pPGPUserID;
  UserIDProp	: PGPBoolean;
  UserIDBuf	: TUserID;
  UserIDNum	: PGPInt32;
  IDSize	: PGPSize;
  IDBuffer	: String;
  KeySig	: pPGPSig;
  SignKey	: TPGPKeyID7;
  KeyID		: TKeyID;
begin
  KeySet := nil;
  KeyList := nil;
  KeyIter := nil;
  Result := PGPNewSingletonKeySet(Key, KeySet);
  if Result <> 0 then Exit;
  try
    Result := PGPOrderKeySet(KeySet, kPGPKeyOrdering_UserID, PGPFalse, KeyList);
    if Result <> 0 then Exit;
    try
      Result := PGPNewKeyIter(KeyList, KeyIter);
      if Result <> 0 then Exit;
      try
	Result := PGPKeyIterNextKeyDBObj(KeyIter, kPGPKeyDBObjType_Key, Key);
	if Result <> 0 then Exit;
	Result := PGPKeyIterRewindKeyDBObj(KeyIter, kPGPKeyDBObjType_UserID);
	if Result <> 0 then Exit;
	// add UserIDs one by one to UserIDList
	while PGPKeyIterNextKeyDBObj(KeyIter, kPGPKeyDBObjType_UserID, UserID) = 0 do begin
	  Result := PGPGetKeyDBObjBooleanProperty(UserID, kPGPUserIDProperty_IsAttribute, UserIDProp);
	  if Result <> 0 then Exit;
	  // check for attributes (picture, e.g.)
	  if not boolean(UserIDProp) then begin
	    // UserID validity
	    Result := PGPGetKeyDBObjNumericProperty(UserID, kPGPUserIDProperty_Validity, UserIDNum);
	    if Result <> 0 then Exit;
	    // UserID string
	    Result := PGPGetKeyDBObjDataProperty(UserID, kPGPUserIDProperty_Name, @UserIDBuf, SizeOf(TUserID), IDSize);
	    if Result <> 0 then Exit;
	    try
	      with KeyPropsRec^ do begin
		if PGP8X then
		  kUserIDList.AddObject(Utf8OrAnsi(UserIDBuf), ptr(UserIDNum))
		else kUserIDList.AddObject(UserIDBuf, ptr(UserIDNum));
		if IncludeSignerIDs then kSignerIDList.Add('');
	      end;
	    except
	      Result := kPGPError_OutOfMemory;
	      Exit;
	    end;
	  end;
	  IDBuffer := '';
	  if IncludeSignerIDs then begin
	    // add comma separated HexIDs of signing keys to KeyPropsRec.SignerIDList
	    while PGPKeyIterNextKeyDBObj(KeyIter, kPGPKeyDBObjType_Signature, KeySig) = 0 do begin
	      Result := PGPGetKeyIDOfCertifier(KeySig, SignKey);
	      if Result <> 0 then Exit;
	      Result := PGPGetKeyIDString(SignKey, kPGPKeyIDString_Full, KeyID);
	      if Result <> 0 then Exit;
	      IDBuffer := IDBuffer + ',' + KeyID;
	    end;
	    Delete(IDBuffer, 1, 1);
	    try
	      with KeyPropsRec^.kSignerIDList do Strings[pred(Count)] := IDBuffer;
	    except
	      Result := kPGPError_OutOfMemory;
	      Exit;
	    end;
	  end;
	end;
      finally
	PGPFreeKeyIter(KeyIter);
      end;
    finally
      PGPFreeKeyList(KeyList);
    end;
  finally
    PGPFreeKeySet(KeySet);
  end;
end;

function GetGroupsList(Context: pPGPContext; KeySetMain: pPGPKeySet; var GroupsList: TGroupsList): Integer;
var
  GroupFileSpec	: pPGPFileSpec;
  GroupSet	: pPGPGroupSet;
  GroupCount	: PGPUInt32;
  GroupIndex	: PGPUInt32;
  GroupID	: PGPGroupID;
  GroupInfo	: TPGPGroupInfo;
  KeyCount	: PGPUInt32;
  ItemCount	: PGPUInt32;
  GroupItemIter	: pPGPGroupItemIter;
  GroupItem	: TPGPGroupItem;
  HexIDList	: String;
  Key		: pPGPKey;
  HexID		: String;
begin
  Result := kPGPError_OutOfMemory;
  if GroupsList <> nil then begin
    try
      GroupsList.Clear;
      GroupFileSpec := nil;
      GroupSet := nil;
      Result := PGPNewFileSpecFromFullPath(Context, PChar(KeyRings.GroupsFile), GroupFileSpec);
      if Result <> 0 then Exit;
      try
	Result := PGPNewGroupSetFromFile(Context, GroupFileSpec, GroupSet);
	if Result <> 0 then Exit;
	try
	  Result := PGPCountGroupsInSet(GroupSet, GroupCount);
	  if (Result <> 0) or (GroupCount = 0) then Exit;
	  for GroupIndex := 0 to pred(GroupCount) do begin
	    Result := PGPGetIndGroupID(GroupSet, GroupIndex, GroupID);
	    if Result <> 0 then Continue;
	    Result := PGPGetGroupInfo(GroupSet, GroupID, GroupInfo);
	    if Result <> 0 then Continue;
	    Result := PGPCountGroupItems(GroupSet, GroupID, PGPTrue, KeyCount, ItemCount);
	    if (Result <> 0) or (KeyCount = 0) then Continue;
	    with GroupInfo do begin
	      if Description <> '' then
		GroupsList.Add(Description + '<' + Name + '>' + #9)
	      else GroupsList.Add('<' + Name + '>' + #9);
	    end;
	    GroupItemIter := nil;
	    Result := PGPNewGroupItemIter(GroupSet, GroupID, kPGPGroupIterFlags_AllKeysRecursive, GroupItemIter);
	    if Result <> 0 then Continue;
	    try
	      HexIDList := '';
	      while PGPGroupItemIterNext(GroupItemIter, GroupItem) = 0 do begin
		with GroupItem.Item.Key do begin
		  if PGP7X then
		    Result := PGPGetKeyByKeyID(KeySetMain, KeyStruct7.KeyID, kPGPPublicKeyAlgorithm_Invalid, Key)
		  else Result := PGPGetKeyByKeyID(KeySetMain, KeyStruct6.KeyID, kPGPPublicKeyAlgorithm_Invalid, Key);
		end;
		if Result <> 0 then Continue;
		HexID := GetKeyPropKeyID(Key);
		if HexID <> '' then HexIDList := HexIDList + ',' + HexID;
	      end;
	      Delete(HexIDList, 1, 1);
	      if HexIDList <> '' then with GroupsList do Strings[GroupIndex] := Strings[GroupIndex] + HexIDList;
	    finally
	      PGPFreeGroupItemIter(GroupItemIter);
	    end;
	  end;
	  Result := GroupsList.Count;
	finally
	  if GroupSet <> nil then PGPFreeGroupSet(GroupSet);
	end;
      finally
	PGPFreeFileSpec(GroupFileSpec);
      end;
    except
      Result := 0;
    end;
    if (Result <= 0) and (GroupsList <> nil) then GroupsList.Clear;
  end;
end;

function GetKeyPropsData(KeySet: pPGPKeySet; KeyFound: pPGPKey; Flags: DWord; KeyPropsList: TKeyPropsList): Integer;
var
  KeyIndex	: Integer;
  KeyPropsRec	: pKeyPropsRec;
begin
  Result := -1;
  try
    // "string" properties
    KeyIndex := KeyPropsList.Add(GetKeyPropKeyID(KeyFound));
    if KeyIndex <> -1 then begin
      KeyPropsRec := pKeyPropsRec(KeyPropsList.Objects[KeyIndex]);
      with KeyPropsRec^ do begin
	if Flags and spgpKeyPropFlag_KeyID <> 0 then begin
	  kHexID := '0x' + Copy(KeyPropsList[KeyIndex], Length(KeyPropsList.Strings[KeyIndex]) - 7, 8);
	end;
	if Flags and spgpKeyPropFlag_UserID <> 0 then kUserID := GetKeyPropUserID(KeyFound);
	if Flags and spgpKeyPropFlag_Fingerprint <> 0 then kFingerprint := GetKeyPropFingerprint(KeyFound);
	if Flags and spgpKeyPropFlag_CreationTimeStr <> 0 then kCreaTimeStr := GetKeyPropKeyCreationTimeStr(KeyFound);
	if Flags and spgpKeyPropFlag_ExpirationTimeStr <> 0 then kExpTimeStr := GetKeyPropKeyExpirationTimeStr(KeyFound);
	// "number" properties
	if Flags and spgpKeyPropFlag_KeyBits <> 0 then kSize := GetKeyPropKeyBits(KeyFound);
	if Flags and spgpKeyPropFlag_KeyAlg <> 0 then kAlgorithm := GetKeyPropAlg(KeyFound);
	if Flags and spgpKeyPropFlag_Trust <> 0 then kTrust := GetKeyPropTrust(KeyFound);
	if Flags and spgpKeyPropFlag_Validity <> 0 then kValidity := GetKeyPropValidity(KeyFound);
	if Flags and spgpKeyPropFlag_CreationTime <> 0 then kCreaTimeNum := GetKeyPropKeyCreationTime(KeyFound);
	if Flags and spgpKeyPropFlag_ExpirationTime <> 0 then kExpTimeNum := GetKeyPropKeyExpirationTime(KeyFound);
	// "boolean" properties
	if Flags and spgpKeyPropFlag_IsCorrupt <> 0 then kCorrupt := GetKeyPropIsCorrupt(KeyFound);
	if Flags and spgpKeyPropFlag_IsSecret <> 0 then kPrivate := GetKeyPropIsSecret(KeyFound);
	if Flags and spgpKeyPropFlag_IsAxiomatic <> 0 then kImplicitTrust := GetKeyPropIsAxiomatic(KeyFound);
	if Flags and spgpKeyPropFlag_IsRevoked <> 0 then kRevoked := GetKeyPropIsRevoked(KeyFound);
	if Flags and spgpKeyPropFlag_IsDisabled <> 0 then kDisabled := GetKeyPropIsDisabled(KeyFound);
	if Flags and spgpKeyPropFlag_IsExpired <> 0 then kExpired := GetKeyPropIsExpired(KeyFound);
	if Flags and spgpKeyPropFlag_IsSecretShared <> 0 then kSecShared := GetKeyPropIsSecretShared(KeyFound);
	if Flags and spgpKeyPropFlag_CanEncrypt <> 0 then kCanEncrypt := GetKeyPropCanEncrypt(KeyFound);
	if Flags and spgpKeyPropFlag_CanDecrypt <> 0 then kCanDecrypt := GetKeyPropCanDecrypt(KeyFound);
	if Flags and spgpKeyPropFlag_CanSign <> 0 then kCanSign := GetKeyPropCanSign(KeyFound);
	if Flags and spgpKeyPropFlag_CanVerify <> 0 then kCanVerify := GetKeyPropCanVerify(KeyFound);
	if Flags and spgpKeyPropFlag_HasRevoker <> 0 then kHasRevoker := GetKeyPropHasRevoker(KeySet, KeyFound, KeyPropsRec);
	if Flags and spgpKeyPropFlag_HasADK <> 0 then kHasADK := GetKeyPropHasARR(KeySet, KeyFound, KeyPropsRec);
	if Flags and spgpKeyPropFlag_HasSubKey <> 0 then kHasSubKey := GetKeyPropHasSubKey(KeySet, KeyFound);
	if Flags and spgpKeyPropFlag_LegacyKey <> 0 then kLegacyKey := GetKeyPropLegacy(KeyFound);
	// "list" properties
	if Flags and spgpKeyPropFlag_IncludeUserIDs <> 0 then begin
	  GetKeyPropUserIDs(KeyFound, KeyPropsRec, (Flags and spgpKeyPropFlag_IncludeSignerIDs <> 0));
	end;
	Result := 0;
      end;
    end;
  except
  end;
end;

function GetKeySetProps(Context: pPGPContext; KeySet: pPGPKeySet;
			var KeyPropsList: TKeyPropsList;
			PropertyFlags, FilterFlags: DWord;
			KeyOrder: TPGPKeyOrdering): Integer;
var
  KeyFilter	: pPGPFilter;
  KeySetFound	: pPGPKeySet;
  KeyCount	: PGPUInt32;
  KeyList	: pPGPKeyList;
  KeyIter	: pPGPKeyIter;
  KeyFound	: pPGPKey;
begin
  KeyFilter := nil;
  KeySetFound := nil;
  KeyList := nil;
  KeyIter := nil;
  if FilterFlags <> KeyFilterFlag_AllKeys then begin
    Result := GetKeyFilter(Context, FilterFlags, KeyFilter);
    try
      if Result <> 0 then Exit;
      Result := PGPFilterKeySet(KeySet, KeyFilter, KeySetFound);
      if Result <> 0 then Exit;
      KeySet := KeySetFound;
    finally
      PGPFreeFilter(KeyFilter);
    end;
  end
  else Result := 0;
  try
    if Result <> 0 then Exit;
    Result := PGPCountKeys(KeySet, KeyCount);
    if Result <> 0 then Exit;
    if PropertyFlags <> 0 then begin
      if KeyPropsList = nil then KeyPropsList := TKeyPropsList.Create(PropertyFlags);
      if KeyPropsList = nil then begin
	Result := kPGPError_OutOfMemory;
	Exit;
      end;
      if (KeyPropsList.Count = 0) and ((PropertyFlags and spgpKeyPropFlag_IncludeGroupsList) <> 0) then begin
	GetGroupsList(Context, KeySet, KeyPropsList.GroupsList);
      end;
      Result := PGPOrderKeySet(KeySet, PGPKeyOrdering(succ(ord(KeyOrder) shr 1)), PGPBoolean(odd(ord(KeyOrder))), KeyList);
      if Result <> 0 then Exit;
      try
	Result := PGPNewKeyIter(KeyList, KeyIter);
	try
	  while (Result = 0) and (PGPKeyIterNextKeyDBObj(KeyIter, kPGPKeyDBObjType_Key, KeyFound) = 0) do begin
	    Result := GetKeyPropsData(KeySet, KeyFound, PropertyFlags, KeyPropsList);
	  end;
	  Result := KeyPropsList.Count;
	finally
	  PGPFreeKeyIter(KeyIter);
	end;
      finally
	PGPFreeKeyList(KeyList);
      end;
    end
    else Result := KeyCount;
  finally
    PGPFreeKeySet(KeySetFound);
  end;
end;

function AddKeysToKeyRing(Context: pPGPContext; KeySetMain: pPGPKeySet;
			  KeysToImport: Pointer; var KeyPropsList: TKeyPropsList;
			  PropertyFlags: DWord): Integer;
var
  KeySetToAdd	: pPGPKeySet;
begin
  if PGP7X then begin
    KeySetToAdd := nil;
    Result := PGPCopyKeys(PGPPeekKeyDBRootKeySet(pPGPKeyDB(KeysToImport)), PGPPeekKeySetKeyDB(KeySetMain), KeySetToAdd);
    if Result <> 0 then Exit;
    try
      Result := KeyRings.UpdateKeyRings;
      if Result <> 0 then Exit;
      Result := GetKeySetProps(Context, KeySetToAdd, KeyPropsList,
			       PropertyFlags, KeyFilterFlag_AllKeys, UserID_Ordering);
    finally
      PGPFreeKeySet(KeySetToAdd);
    end;
  end
  else begin
    Result := PGPAddKeys(pPGPKeySet(KeysToImport), KeySetMain);
    if Result <> 0 then Exit;
    Result := KeyRings.UpdateKeyRings;
    if Result <> 0 then Exit;
    Result := GetKeySetProps(Context, pPGPKeySet(KeysToImport), KeyPropsList,
			     PropertyFlags, KeyFilterFlag_AllKeys, UserID_Ordering);
  end;
end;

function GetHexIDByAnyID(Context: pPGPContext; KeySetMain: pPGPKeySet;
			 const AnyID: String; var HexID: String): Integer;
var
  KeySetFound	: pPGPKeySet;
  KeyCount	: PGPUInt32;
  Key		: pPGPKey;
begin
  HexID := '';
  Result := 0;
  KeySetFound := nil;
  if AnyID <> '' then begin
    try
      Result := GetKeySetByAnyIDs(Context, KeySetMain, AnyID, KeySetFound);
      if Result <> 0 then Exit;
      Result := PGPCountKeys(KeySetFound, KeyCount);
      if Result <> 0 then Exit;
      if KeyCount > 0 then begin
	Result := GetKeyFromKeySet(KeySetFound, Key);
	if Result <> 0 then Exit;
	HexID := GetKeyPropKeyID(Key);
      end;
      Result := KeyCount;
    finally
      PGPFreeKeySet(KeySetFound);
    end;
  end;
end;

function PassphraseIsValid(Context: pPGPContext; KeySetMain: pPGPKeySet; AnyID, Passphrase: PChar): Longbool;
var
  KeySetFound	: pPGPKeySet;
  PhraseBuffer	: PChar;
  KeyCount	: PGPUInt32;
  Key		: pPGPKey;
begin
  Result := false;
  KeySetFound := nil;
  if GetKeySetByAnyIDs(Context, KeySetMain, AnyID, KeySetFound) = 0 then begin
    PhraseBuffer := Passphrase;
    if PGP8X then begin
      PhraseBuffer := PGPNewSecureData(PGPGetDefaultMemoryMgr, MaxUTF8Length, kPGPMemoryMgrFlags_Clear);
      if PhraseBuffer <> nil then AnsiToUtf8PChar(Passphrase, PhraseBuffer, MaxUTF8Length);
    end;
    try
      Result := (PGPCountKeys(KeySetFound, KeyCount) = 0) and
		(KeyCount = 1) and (GetKeyFromKeySet(KeySetFound, Key) = 0) and
		(PGPPassphraseIsValid(Key, PGPOPassphrase(Context, PhraseBuffer), PGPOLastOption(Context)) <> PGPFalse);
    finally
      PGPFreeKeySet(KeySetFound);
      if PGP8X then PGPFreeData(PhraseBuffer);
    end;
  end;
end;

function TKeyRings.UpdateKeyRings: PGPError;
begin
  Result := kPGPError_NoErr;
  if PGP7X then
    Result := PGPFlushKeyDB(RingKeyDB)
  else if boolean(PGPKeySetNeedsCommit(RingKeySet)) then Result := PGPCommitKeyringChanges(RingKeySet);
  if Result = 0 then PGPclNotifyKeyringChanges(GetCurrentProcessID);
end;

function TKeyRings.SetKeyRings(const Pubring, Secring: String): Longbool;
begin
  PubringFile := Pubring;
  SecringFile := Secring;
  Result := ((PubringFile <> '') and (SecringFile <> ''));
end;

function TKeyRings.InitKeyRings(var Context: pPGPContext; var KeySetMain: pPGPKeySet): PGPError;
var
  Prefs: TPreferenceRec;
  PubFileSpec: pPGPFileSpec;
  SecFileSpec: pPGPFileSpec;
begin
  Result := kPGPError_ImproperInitialization;
  if PGPInitErrorCode = ieNone then begin
    if InitCount <= 0 then begin
      InitCount := 0;
      RingKeyDB := nil;
      RingKeySet := nil;
      RingContext := nil;
      PubFileSpec := nil;
      SecFileSpec := nil;
      Result := PGPNewContext(kPGPsdkAPIVersion, RingContext);
      if Result <> 0 then Exit;
      if (PubRingFile = '') or (SecringFile = '') then begin
	GetPreferences(Prefs, PrefsFlag_PublicKeyring or PrefsFlag_PrivateKeyring or PrefsFlag_GroupsFile);
	SetKeyRings(Prefs.PublicKeyring, Prefs.PrivateKeyring);
	if GroupsFile = '' then GroupsFile := Prefs.GroupsFile;
      end
      else if (GroupsFile = '') and (GetPreferences(Prefs, PrefsFlag_GroupsFile) = 0) then begin
	GroupsFile := Prefs.GroupsFile;
      end;
      try
	Result := PGPNewFileSpecFromFullPath(RingContext, PChar(PubringFile), PubFileSpec);
	if Result <> 0 then Exit;
	Result := PGPNewFileSpecFromFullPath(RingContext, PChar(SecringFile), SecFileSpec);
	if Result <> 0 then Exit;
	if PGP7X then begin
	  Result := PGPOpenKeyDBFile(RingContext, kPGPOpenKeyDBFileOptions_Mutable, PubFileSpec, SecFileSpec, RingKeyDB);
	  if Result = 0 then RingKeySet := PGPPeekKeyDBRootKeySet(RingKeyDB);
	end
	else Result := PGPOpenKeyRingPair(RingContext, kPGPKeyRingOpenFlags_Mutable, PubFileSpec, SecFileSpec, RingKeySet);
      finally
	PGPFreeFileSpec(PubFileSpec);
	PGPFreeFileSpec(SecFileSpec);
      end;
      if Result <> 0 then begin
	PGPFreeContext(RingContext);
	RingContext := nil;
	if PGP7X then begin
	  PGPFreeKeyDB(RingKeyDB);
	  RingKeyDB := nil;
	end
	else begin
	  PGPFreeKeySet(RingKeySet);
	  RingKeySet := nil;
	end;
      end;
    end
    else Result := 0;
    if Result = 0 then inc(InitCount);
    KeySetMain := RingKeySet;
    Context := RingContext;
  end;
end;

procedure TKeyRings.FreeKeyRings;
begin
  try
    if InitCount = 1 then begin
      try
	if PGP7X then
	  PGPFreeKeyDB(RingKeyDB)
	else PGPFreeKeySet(RingKeySet);
      finally
	RingKeySet := nil;
	RingKeyDB := nil;
	try
	  PGPFreeContext(RingContext);
	finally
	  RingContext := nil;
	end;
      end;
    end;
  finally
    if InitCount > 0 then dec(InitCount);
  end;
end;

initialization
  KeyRings := TKeyRings.Create;

finalization
  KeyRings.Free;
  KeyRings := nil;

end.

