{$J+,Z4}
unit PGPKeyGenerate;

{------------------------------------------------------------------------------}
{                                                                              }
{                 This unit is partly based on Steve Heller's                  }
{         SPGP sources available from http://www.oz.net/~srheller/spgp/        }
{                                                                              }
{                Portions created by Michael in der Wiesche are                }
{              Copyright (C) 2001-2003 by Michael in der Wiesche               }
{                                                                              }
{------------------------------------------------------------------------------}

interface

uses
  Windows,
  Classes,
  SysUtils,
  KeyPropTypes,
  UTF8,
  pgpBase,
  pgpErrors,
  pgpPubTypes,
  pgpUtilities,
  pgpOptionList,
  pgpRandomPool,
  pgpMemoryMgr,
  pgpEvents,
  pgpKeys,
  pgpTLS,
  pgpUI,
  pgpSC,
  KeyFuncs,
  PrefFuncs,
  PGPDialogs;

type
  TKeySize = 512..2048;
  TSubKeySize = 512..4096;
  TMinPassLen = 0..255;
  TMinPassQual = 0..100;
  TCipherCount = 0..ord(high(TCipherAlgorithm));

  TOnEnterPassphrase = procedure(const Passphrase: PChar; const MasterKeyProps: TKeyPropsRec;
				 MinPassLen: TMinPassLen; MinPassQual: TMinPassQual;
				 var Cancel: Longbool) of Object;
  TOnGetUserNameAddress = procedure(var UserName, EmailAddress: String) of Object;
  TOnKeyGeneration = procedure(const NewHexID: String; MasterKeyProps: TKeyPropsRec; Aborted: Longbool) of Object;
  TOnShowState = procedure(State: Char; var Cancel: Longbool) of Object;
  TPGPKeysGenerateCustom = Class(TComponent)
  private
    FAborted: Longbool;
    FContext: pPGPContext;
    FKeySetMain: pPGPKeySet;
    FPubKeyAlgorithm: TKeyAlgorithm;
    FCipherAlgorithm: TCipherAlgorithm;
    FCipherAlgorithmList: Array[TCipherCount] of TCipherAlgorithm;
    FExpires: Longint;
    FFastGenerate: Longbool;
    FFailNoEntropy: Longbool;
    FLegacyKey: Longbool;
    FMinPassLen: TMinPassLen;
    FMinPassQual: TMinPassQual;
    FPassphrase: PChar;
    FUserID: String;
    FNewHexID: String;
    FUserName: String;
    FEmailAddress: String;
    FMasterKeyHexID: String;
    FNewKeyHexID: String;
    FNewSubKeyHexID: String;
    FKeySize: TKeySize;
    FSubKeySize: TSubKeySize;
    FParentHandle: THandle;
    FKeyDlgPrompt: String;
    FPassDlgPrompt: String;
    FOnEnterPassphrase: TOnEnterPassphrase;
    FOnGetUserNameAddress: TOnGetUserNameAddress;
    FOnKeyGeneration: TOnKeyGeneration;
    FOnShowState: TOnShowState;
    function InitKeyGen: PGPError;
    procedure FinitKeyGen(var Result: PGPError);
    function GetUserID: String;
    function GetMasterKeyHexID: Longint;
    function GetUserNameAddress: Longint;
    function GetPassphrase(OfMasterKey: Longbool): PGPError;
    function GetEntropy(IncludeSubKey: Longbool): PGPError;
    function KeyGenerate(IncludeSubKey: Longbool): PGPError;
    function SubKeyGenerate(ForOldMasterKey: Longbool): PGPError;
  protected
    procedure SetUserName(const Name: String);
    procedure SetEmailAddress(const Address: String);
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure SetCipherAlgPrefs(const CipherAlgList: Array of TCipherAlgorithm);
    function DHDSSKeyGenerate: Longint; virtual;
    function DHSubKeyGenerate: Longint; virtual;
    function DSAKeyGenerate: Longint; virtual;
    function RSASubKeyGenerate: Longint; virtual;
    function RSAKeyGenerate(Legacy: Longbool): Longint; virtual;
    property ParentHandle: THandle
      read FParentHandle
      write FParentHandle;
  published
    property CipherAlgorithm: TCipherAlgorithm
      read FCipherAlgorithm
      write FCipherAlgorithm;
    property Expires: Longint
      read FExpires
      write FExpires;
    property FastGenerate: Longbool
      read FFastGenerate
      write FFastGenerate;
    property FailNoEntropy: Longbool
      read FFailNoEntropy
      write FFailNoEntropy;
    property MinPassLen: TMinPassLen
      read FMinPassLen
      write FMinPassLen;
    property MinPassQual: TMinPassQual
      read FMinPassQual
      write FMinPassQual;
    property UserName: String
      read FUserName
      write SetUserName;
    property EmailAddress: String
      read FEmailAddress
      write SetEmailAddress;
    property MasterKeyHexID: String
      read FMasterKeyHexID
      write FMasterKeyHexID;
    property KeySize: TKeySize
      read FKeySize
      write FKeySize;
    property SubKeySize: TSubKeySize
      read FSubKeySize
      write FSubKeySize;
    property KeyDlgPrompt: String
      read FKeyDlgPrompt
      write FKeyDlgPrompt;
    property PassDlgPrompt: String
      read FPassDlgPrompt
      write FPassDlgPrompt;
    property OnEnterPassphrase: TOnEnterPassphrase
      read FOnEnterPassphrase
      write FOnEnterPassphrase;
    property OnGetUserNameAddress: TOnGetUserNameAddress
      read FOnGetUserNameAddress
      write FOnGetUserNameAddress;
    property OnKeyGeneration: TOnKeyGeneration
      read FOnKeyGeneration
      write FOnKeyGeneration;
    property OnShowState: TOnShowState
      read FOnShowState
      write FOnShowState;
  end;

implementation

function EventHandler(Context: pPGPContext; Event: pPGPEvent; UserValue: PGPUserValue): PGPError; cdecl;
var
  Cancel	: Longbool;
begin
  Result:=0;
  Cancel:=false;
  with TPGPKeysGenerateCustom(UserValue) do begin
    case Event^.EType of
      kPGPEvent_KeyGenEvent: begin
	if TMethod(FOnShowState).Code<>nil then with Event^.EData.KeyGenData do begin
	  FOnShowState(chr(State), Cancel);
	  if Cancel then begin
	    Result:=kPGPError_UserAbort;
	    FAborted:=true;
	  end;
	end;
	ProcessMessages;
      end;
    end;
  end;
end;

constructor TPGPKeysGenerateCustom.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FFastGenerate:=true;
  FMinPassLen:=10;
  FMinPassQual:=100;
  FKeySize:=1024;
  FSubKeySize:=1024;
end;

destructor TPGPKeysGenerateCustom.Destroy;
begin
  inherited Destroy;
end;

function TPGPKeysGenerateCustom.InitKeyGen: PGPError;
begin
  FUserID:='';
  FNewHexID:='';
  FAborted:=false;
  FLegacyKey:=true;
  FPassphrase:=nil;
  FNewKeyHexID:='';
  FNewSubKeyHexID:='';
  Result:=KeyRings.InitKeyRings(FContext, FKeySetMain);
end;

procedure TPGPKeysGenerateCustom.FinitKeyGen(var Result: PGPError);
var
  KeyPropsList	: TKeyPropsList;
begin
  PGPFreeData(FPassphrase);
  KeyRings.FreeKeyRings;
  try
    if not FAborted then begin
      KeyPropsList:=nil;
      FindKeyProps(FMasterKeyHexID, KeyPropsList,
		   spgpKeyPropFlag_IDComplete,
		   KeyFilterFlag_CanSign,
		   Any_Ordering);
    end
    else Result:=kPGPError_UserAbort;
    if Assigned(FOnKeyGeneration) then FOnKeyGeneration(FNewHexID, pKeyPropsRec(KeyPropsList.Objects[0])^, Result<>0);
  finally
    KeyPropsList.Free;
  end;
end;

procedure TPGPKeysGenerateCustom.SetUserName(const Name: String);
var
  ErrorStr	: String;
begin
  if Name<>FUserName then begin
    if pos('@', Name)=0 then
      FUserName:=Trim(Name)
    else begin
      SetLength(ErrorStr, 255);
      PGPGetErrorString(kPGPError_BadParams, 255, PChar(ErrorStr));
      SetLength(ErrorStr, StrLen(PChar(ErrorStr)));
      MessageBox(FParentHandle, PChar(ErrorStr + ': ' + Name), 'PGP', MB_ICONERROR);
    end;
  end;
end;

procedure TPGPKeysGenerateCustom.SetEmailAddress(const Address: String);
var
  ErrorStr	: String;
begin
  if Address<>FEmailAddress then begin
    if (Address='') or (pos('@', Address)<>0) then
      FEmailAddress:=Trim(Address)
    else begin
      SetLength(ErrorStr, 255);
      PGPGetErrorString(kPGPError_BadParams, 255, PChar(ErrorStr));
      SetLength(ErrorStr, StrLen(PChar(ErrorStr)));
      MessageBox(FParentHandle, PChar(ErrorStr + ': ' + Address), 'PGP', MB_ICONERROR);
    end;
  end;
end;

function TPGPKeysGenerateCustom.GetUserID: String;
begin
  if FEmailAddress<>'' then begin
    Result:='<' + FEmailAddress + '>';
    if FUserName<>'' then Result:=FUserName + ' ' + Result;
  end
  else Result:=FUserName;
end;

function TPGPKeysGenerateCustom.GetMasterKeyHexID: Longint;
var
  KeyPropsList	: TKeyPropsList;
begin
  KeyPropsList:=nil;
  try
    Result:=FindKeyProps(FMasterKeyHexID,
			 KeyPropsList,
			 spgpKeyPropFlag_KeyID,
			 KeyFilterFlag_CanSign or KeyFilterFlag_V4,
			 Any_Ordering);
    if Result=0 then begin
      Result:=kPGPError_SecretKeyNotFound;
      Exit;
    end
    else if Result<>1 then begin
      if KeyPropsList<>nil then KeyPropsList.Clear;
      Result:=SelectKeysDialog(FKeyDlgPrompt, KeyPropsList, true, spgpKeyPropFlag_KeyID,
			       KeyFilterFlag_CanSign or KeyFilterFlag_V4, FParentHandle);
      if Result=0 then FMasterKeyHexID:=KeyPropsList.Strings[0];
    end
    else begin
      FMasterKeyHexID:=KeyPropsList.Strings[0];
      Result:=0;
    end;
  finally
    KeyPropsList.Free;
  end;
end;

function TPGPKeysGenerateCustom.GetUserNameAddress: Longint;
begin
  Result:=0;
  if (FUserName='') or (FEmailAddress='') then begin
    if Assigned(FOnGetUserNameAddress) then FOnGetUserNameAddress(FUserName, FEmailAddress);
  end;
  if FUserName='' then Result:=kPGPError_BadParams;
end;

function TPGPKeysGenerateCustom.GetPassphrase(OfMasterKey: Longbool): PGPError;
var
  Cancel	: Longbool;
  KeySetFound	: pPGPKeySet;
  PassBufSize	: Cardinal;
  KeyPropsRec	: TKeyPropsRec;
  KeyPropsList	: TKeyPropsList;
begin
  Result:=0;
  Cancel:=false;
  KeySetFound:=nil;
  PassBufSize:=MaxUTF8Length;
  if Assigned(FOnEnterPassphrase) then begin
    FillChar(KeyPropsRec, SizeOf(TKeyPropsRec), 0);
    KeyPropsList:=nil;
    try
      if OfMasterKey then begin
	if FindKeyProps(FMasterKeyHexID, KeyPropsList, spgpKeyPropFlag_IDComplete,
			KeyFilterFlag_CanSign or KeyFilterFlag_V4, Any_Ordering)=1 then begin
	  KeyPropsRec:=pKeyPropsRec(KeyPropsList.Strings[0])^;
	end
	else begin
	  Result:=kPGPError_SecretKeyNotFound;
	  Exit;
	end;
      end;
      FPassphrase:=PGPNewSecureData(PGPGetDefaultMemoryMgr, 1024, kPGPMemoryMgrFlags_Clear);
      if FPassphrase<>nil then begin
	repeat
	  FOnEnterPassphrase(FPassphrase, KeyPropsRec, FMinPassLen, FMinPassQual, Cancel);
	until Cancel or (StrLen(FPassphrase)>=FMinPassLen) and (PGPEstimatePassphraseQuality(FPassphrase)>=FMinPassQual);
	if not Cancel then begin
	  if PGP8X then AnsiToUtf8PChar(FPassphrase, FPassphrase, PassBufSize);
	end
	else Result:=kPGPError_UserAbort;
      end
      else Result:=kPGPError_OutOfMemory;
    finally
      KeyPropsList.Free;
    end;
  end
  else begin
    if OfMasterKey then begin
      Result:=GetKeySetByAnyIDs(FContext, FKeySetMain, FMasterKeyHexID, KeySetFound);
      if Result<>0 then Exit;
      try
	Result:=KeyPassphraseDialog(FContext, KeySetFound, FPassphrase, FPassDlgPrompt, FParentHandle);
      finally
	PGPFreeKeySet(KeySetFound);
      end;
    end
    else begin
      Result:=ConfirmationPassphraseDialog(FContext, FPassphrase, FMinPassLen, FMinPassQual, true,
					   FPassDlgPrompt, FParentHandle);
    end;
  end;
end;

function TPGPKeysGenerateCustom.GetEntropy(IncludeSubKey: Longbool): PGPError;
var
  OptionList	: pPGPOptionList;
  EntropyNeeded	: PGPUInt32;
  ParamOption	: pPGPoptionList;
begin
  OptionList:=nil;
  Result:=PGPBuildOptionList(FContext, OptionList,
    [
      PGPOKeyGenParams(FContext, PGPPublicKeyAlgorithm(FPubKeyAlgorithm), FKeySize),
      PGPOKeyGenFast(FContext, PGPBoolean(FFastGenerate) and 1)
    ]);
  if Result<>0 then Exit;
  try
    EntropyNeeded:=PGPGetKeyEntropyNeeded(FContext, OptionList, PGPOLastOption(FContext));
  finally
    PGPFreeOptionList(OptionList);
  end;
  if IncludeSubKey then begin
    OptionList:=nil;
    if FPubKeyAlgorithm<>KeyAlgorithm_RSA then
      ParamOption:=PGPOKeyGenParams(FContext, PGPPublicKeyAlgorithm(KeyAlgorithm_DH), FSubKeySize)
    else ParamOption:=PGPOKeyGenParams(FContext, PGPPublicKeyAlgorithm(KeyAlgorithm_RSA), FSubKeySize);
    try
      Result:=PGPBuildOptionList(FContext, OptionList,
	[ParamOption, PGPOKeyGenFast(FContext, PGPBoolean(FFastGenerate) and 1)]);
    finally
      PGPFreeOptionList(ParamOption);
    end;
    if Result<>0 then Exit;
    try
      EntropyNeeded:=EntropyNeeded + PGPGetKeyEntropyNeeded(FContext, OptionList, PGPOLastOption(FContext));
    finally
      PGPFreeOptionList(OptionList);
    end;
  end;
  if FFailNoEntropy and (EntropyNeeded>PGPGlobalRandomPoolGetEntropy) then begin
    Result:=kPGPError_OutOfEntropy;
    Exit;
  end;
  // PGP 8 returns kPGPError_OutOfEntropy if we don't call CollectRandomDataDialog() in any case
  Result:=CollectRandomDataDialog(FContext, EntropyNeeded, FParentHandle);
end;

function TPGPKeysGenerateCustom.KeyGenerate(IncludeSubKey: Longbool): PGPError;
var
  OptionList	: pPGPOptionList;
  CipherAlgs	: TPGPCipherAlgorithms;
  CipherIndex	: Longint;
  UsageOption	: pPGPOptionList;
  UserIDUtf8	: UTF8String;
  NewKey	: pPGPKey;
  function GetPGPCipherAlgorithm(CipherAlgorithm: TCipherAlgorithm): PGPCipherAlgorithm;
  begin
    Result:=PGPCipherAlgorithm(
      ord(CipherAlgorithm) +
      ord(CipherAlgorithm>CipherAlgorithm_CAST5) * ord(pred(kPGPCipherAlgorithm_AES128-kPGPCipherAlgorithm_CAST5))
    );
  end;
begin
  OptionList:=nil;
  if FCipherAlgorithmList[0]=CipherAlgorithm_None then begin
    case FCipherAlgorithm of
      CipherAlgorithm_None: begin
	case FPubKeyAlgorithm of
	  KeyAlgorithm_RSA: FCipherAlgorithm:=CipherAlgorithm_IDEA;
	  KeyAlgorithm_DSS: FCipherAlgorithm:=CipherAlgorithm_CAST5;
	end;
      end;
    end;
    CipherAlgs[0]:=GetPGPCipherAlgorithm(FCipherAlgorithm);
    CipherIndex:=1;
  end
  else begin
    CipherIndex:=0;
    while FCipherAlgorithmList[CipherIndex]<>CipherAlgorithm_None do begin
      CipherAlgs[CipherIndex]:=GetPGPCipherAlgorithm(FCipherAlgorithmList[CipherIndex]);
      inc(CipherIndex);
    end;
  end;
  if PGP7X and not FLegacyKey then
    UsageOption:=PGPOKeyFlags(FContext, kPGPKeyPropertyFlags_UsageSign)
  else UsageOption:=PGPONullOption(FContext);
  if PGP8X then
    UserIDUtf8:=AnsiToUtf8(FUserID)
  else UserIDUtf8:=FUserID;
  try
    Result:=PGPBuildOptionList(FContext, OptionList,
      [
	PGPOKeySetRef(FContext, FKeySetMain),
	PGPOPassphrase(FContext, FPassphrase),
	PGPOKeyGenName(FContext, PChar(UserIDUtf8), Length(UserIDUtf8)),
	PGPOKeyGenParams(FContext, PGPPublicKeyAlgorithm(FPubKeyAlgorithm), FKeySize),
	PGPOPreferredAlgorithms(FContext, CipherAlgs, CipherIndex),
	PGPOKeyGenFast(FContext, PGPBoolean(FFastGenerate) and 1),
	PGPOExpiration(FContext, FExpires),
	UsageOption,
	PGPOEventHandler(FContext, EventHandler, Self)
      ]);
  finally
    PGPFreeOptionList(UsageOption);
  end;
  if Result<>0 then Exit;
  try
    Result:=PGPGenerateKey(FContext, NewKey, OptionList, PGPOLastOption(FContext));
    if Result<>0 then Exit;
    if not IncludeSubKey then begin
      Result:=KeyRings.UpdateKeyRings;
      if Result<>0 then Exit;
    end;
    FNewKeyHexID:=GetKeyPropKeyID(NewKey);
  finally
    PGPFreeOptionList(OptionList);
  end;
end;

function TPGPKeysGenerateCustom.SubKeyGenerate(ForOldMasterKey: Longbool): PGPError;
var
  OptionList	: pPGPOptionList;
  MasterKey	: pPGPKey;
  UsageOption	: pPGPOptionList;
  NewSubKey	: pPGPKey;
begin
  OptionList:=nil;
  Result:=GetKeyByHexID(FKeySetMain, FMasterKeyHexID, MasterKey);
  if Result<>0 then Exit;
  if PGP7X and not FLegacyKey then
    UsageOption:=PGPOKeyFlags(FContext, kPGPKeyPropertyFlags_UsageEncrypt)
  else UsageOption:=PGPONullOption(FContext);
  try
    Result:=PGPBuildOptionList(FContext, OptionList,
      [
	PGPOKeyGenMasterKey(FContext, MasterKey),
	PGPOPassphrase(FContext, FPassphrase),
	PGPOKeyGenParams(FContext, PGPPublicKeyAlgorithm(FPubKeyAlgorithm), FSubKeySize),
	PGPOKeyGenFast(FContext, PGPBoolean(FFastGenerate) and 1),
	PGPOExpiration(FContext, FExpires),
	UsageOption,
	PGPOEventHandler(FContext, EventHandler, Self)
      ]);
  finally
    PGPFreeOptionList(UsageOption);
  end;
  if Result<>0 then Exit;
  try
    Result:=PGPGenerateSubKey(FContext, NewSubKey, OptionList, PGPOLastOption(FContext));
    if Result<>0 then Exit;
    Result:=KeyRings.UpdateKeyRings;
    if Result<>0 then Exit;
    if ForOldMasterKey then FUserID:=GetKeyPropUserID(MasterKey);
    FNewSubKeyHexID:=GetSubKeyPropKeyID(NewSubKey);
  finally
    PGPFreeOptionList(OptionList);
  end;
end;

procedure TPGPKeysGenerateCustom.SetCipherAlgPrefs(const CipherAlgList: Array of TCipherAlgorithm);
var
  CipherCount	: Longint;
  CipherIndex	: Longint;
begin
  CipherCount:=succ(high(CipherAlgList));
  if CipherCount>high(TCipherCount) then CipherCount:=high(TCipherCount);
  for CipherIndex:=0 to pred(CipherCount) do FCipherAlgorithmList[CipherIndex]:=CipherAlgList[CipherIndex];
  FCipherAlgorithmList[CipherCount]:=CipherAlgorithm_None;
end;

function TPGPKeysGenerateCustom.DHDSSKeyGenerate: Longint;
begin
  Result:=0;
  Result:=InitKeyGen;
  if Result<>0 then Exit;
  try
    Result:=GetUserNameAddress;
    if Result<>0 then Exit;
    if FKeySize>1024 then FKeySize:=1024;
    FPubKeyAlgorithm:=KeyAlgorithm_DSS;
    FUserID:=GetUserID;
    Result:=GetPassphrase(false);
    if Result<>0 then Exit;
    Result:=GetEntropy(true);
    if Result<>0 then Exit;
    Result:=KeyGenerate(true);
    if Result<>0 then Exit;
    FPubKeyAlgorithm:=KeyAlgorithm_DH;
    FMasterKeyHexID:=FNewKeyHexID;
    Result:=SubKeyGenerate(false);
    if Result=0 then FNewHexID:=FNewKeyHexID;
  finally
    FinitKeyGen(Result);
  end;
end;

function TPGPKeysGenerateCustom.DHSubKeyGenerate: Longint;
begin
  Result:=0;
  Result:=InitKeyGen;
  if Result<>0 then Exit;
  try
    Result:=GetMasterKeyHexID;
    if Result<>0 then Exit;
    FPubKeyAlgorithm:=KeyAlgorithm_DH;
    Result:=GetPassphrase(true);
    if Result<>0 then Exit;
    Result:=GetEntropy(false);
    if Result<>0 then Exit;
    Result:=SubKeyGenerate(true);
    if Result=0 then FNewHexID:=FNewSubKeyHexID;
  finally
    FinitKeyGen(Result);
  end;
end;

function TPGPKeysGenerateCustom.DSAKeyGenerate: Longint;
begin
  Result:=0;
  Result:=InitKeyGen;
  if Result<>0 then Exit;
  try
    Result:=GetUserNameAddress;
    if Result<>0 then Exit;
    if FKeySize>1024 then FKeySize:=1024;
    FPubKeyAlgorithm:=KeyAlgorithm_DSS;
    FUserID:=GetUserID;
    Result:=GetPassphrase(false);
    if Result<>0 then Exit;
    Result:=GetEntropy(false);
    if Result<>0 then Exit;
    Result:=KeyGenerate(false);
    if Result=0 then FNewHexID:=FNewKeyHexID;
  finally
    FinitKeyGen(Result);
  end;
end;

function TPGPKeysGenerateCustom.RSASubKeyGenerate: Longint;
begin
  Result:=0;
  Result:=InitKeyGen;
  if Result<>0 then Exit;
  try
    Result:=GetMasterKeyHexID;
    if Result<>0 then Exit;
    FPubKeyAlgorithm:=KeyAlgorithm_RSA;
    Result:=GetPassphrase(true);
    if Result<>0 then Exit;
    Result:=GetEntropy(false);
    if Result<>0 then Exit;
    Result:=SubKeyGenerate(true);
    if Result=0 then FNewHexID:=FNewSubKeyHexID;
  finally
    FinitKeyGen(Result);
  end;
end;

function TPGPKeysGenerateCustom.RSAKeyGenerate(Legacy: Longbool): Longint;
begin
  if Legacy or PGP7X then begin
    Result:=0;
    Result:=InitKeyGen;
    if Result<>0 then Exit;
    try
      Result:=GetUserNameAddress;
      if Result<>0 then Exit;
      FPubKeyAlgorithm:=KeyAlgorithm_RSA;
      FUserID:=GetUserID;
      FLegacyKey:=Legacy;
      Result:=GetPassphrase(false);
      if Result<>0 then Exit;
      Result:=GetEntropy(not Legacy);
      if Result<>0 then Exit;
      Result:=KeyGenerate(not Legacy);
      if Result<>0 then Exit;
      if not Legacy then begin
	FPubKeyAlgorithm:=KeyAlgorithm_RSA;
	FMasterKeyHexID:=FNewKeyHexID;
	Result:=SubKeyGenerate(false);
      end;
      if Result=0 then FNewHexID:=FNewKeyHexID;
    finally
      FinitKeyGen(Result);
    end;
  end
  else Result:=kPGPError_UnknownKeyVersion;
end;

end.

