{$J+,Z4}
unit TimeUtils;

{:      Primoz Gabrijelcic's Time Zone Routines v1.2:

	Date/Time Routines to enhance your 32-bit Delphi Programming.

	(c) 1999, 2000 Primoz Gabrijelcic

	===================================================

	original file name gpTimezone.pas

	modified by Michael in der Wiesche <idw.doc@t-online.de>:
	- added two UnixTime conversion functions
	- added a SystemTimeToDateTime routine
	- included required ESBDates routines
	- removed Registry dependent routines
	- added SECONDS_PER_DAY constant
	- renamed MINUTESPERDAY constant
	- changed some function parameters to const
	- some minor optical and comment changes ...

	Portions Copyright (c) Michael in der Wiesche, 2002

	===================================================

	The original routines are used by ESB Consultancy and Primoz Gabrijelcic
	within the development of their Customised Application.
	Primoz Gabrijelcic retains full copyright.
	mailto:gabr@17slon.com
	http://17slon.com/gp/gp/
	http://www.eccentrica.org/gabr/gp/
	http://members.xoom.com/primozg/gp/

	Primoz Gabrijelcic grants users of this code royalty free rights
	to do with this code as they wish.

	Primoz Gabrijelcic makes no guarantees nor excepts any liabilities
	due to the use of these routines.

	We do ask that if this code helps you in your development
	that you send as an email mailto:info@esbconsult.com.au or even
	a local postcard. It would also be nice if you gave us a
	mention in your About Box, Help File or Documentation.

	ESB Consultancy Home Page: http://www.esbconsult.com.au

	Mail Address: PO Box 2259, Boulder, WA 6432 AUSTRALIA }

interface

uses
  Windows,
  SysUtils;

const
  MINUTES_PER_DAY = 24*60;
  SECONDS_PER_DAY = 24*60*60;

  {: Returns true if date1 and date2 are almost the same (difference between
     them is less than 1/10 of a millisecond. }
  function DateEQ(const date1, date2: TDateTime): boolean;

  {: Corrects date part so it will represent exact (as possible) millisecond,
     not maybe small part before or after that. Useful when you want to use
     Trunc/Int and Frac functions to get date or time part from TDateTime
     variable. }
  function FixDT(const date: TDateTime): TDateTime;

  {: Returns the Day of the Month number from a given date/time. }
  function Date2Day(const DT: TDateTime): Word;

  {: Returns the Month number from a given date/time, 1 = Jan, etc. }
  function Date2Month(const DT: TDateTime): Word;

  {: Returns the Year from a given date/time. }
  function Date2Year(const DT: TDateTime): Word;

  {: Returns the Hour from a given date/time. }
  function Time2Hr (const DT: TDateTime): Word;

  {: Returns the Minute from a given date/time. }
  function Time2Min (const DT: TDateTime): Word;

  {: Returns the Second from a given date/time. }
  function Time2Sec (const DT: TDateTime): Word;

  {: Returns the Millisecond from a given date/time. }
  function Time2MSec (const DT: TDateTime): Word;

  {: Is given Year a Leap Year. Thanks to Dr John Stockton
     for suggesting a faster methodology. }
  function IsLeapYear(Year: Word): Boolean;

  {: Is given Date/Time in a Leap Year. Thanks to Dr John Stockton
     for suggesting a faster methodology. }
  function DateIsLeapYear(const DT: TDateTime): Boolean;

  {: Returns the number of days in the Month represented by the given Date. }
  function DaysInMonth(const DT: TDateTime): Byte;

  {: Returns the current Year - from Today's Date. }
  function ThisYear: Word;

  {: Returns the current Month - from Today's Date. }
  function ThisMonth: Word;

  {: Returns the current Day - from Today's Date. }
  function ThisDay: Word;

  {: Returns the current Hour - from the current Time. }
  function ThisHr: Word;

  {: Returns the current Minute - from the current Time. }
  function ThisMin: Word;

  {: Returns the current Second - from the current Time. }
  function ThisSec: Word;

  {: Returns the current Millisecond - from the current Time. }
  function ThisMSec: Word;

  {: Converts 'day of month' syntax to normal date. Set year and month to
     required values, set weekInMonth to required week (1-4, or 5 for last),
     set dayInWeek to required day of week (1 (Sunday) to 7 (Saturday) - Delphi
     style). }
  function DayOfMonth2Date(year, month, weekInMonth, dayInWeek: word): TDateTime;

  {: Converts TIME_ZONE_INFORMATION date to normal date. Time zone information
     can be returned in two formats by Windows API call GetTimeZoneInformation.
     Absolute format specifies an exact date and time when standard/DS time
     begins. In this form, the wYear, wMonth, wDay, wHour, wMinute , wSecond,
     and wMilliseconds members of the TSystemTime structure are used to specify
     an exact date. Year is left intact, if you want to change it, call
     ESBDates.AdjustDateYear (warning: this will clear the time part).
     Day-in-month format is specified by setting the wYear member to zero,
     setting the wDayOfWeek member to an appropriate weekday (0 to 6,
     0 = Sunday), and using a wDay value in the range 1 through 5 to select the
     correct day in the month. Year parameter is used to specify year for this
     date.
     Returns 0 if 'dstDate' is invalid or if it specifies "absolute date" for a
     year not equal to 'year' parameter. }
  function DSTDate2Date(const dstDate: TSystemTime; year: word): TDateTime;

  {: Returns daylight saving information for a specified time zone and year.
     Sets DaylightDate and StandardDate year to specified year if date is
     specified in day-in-month format (see above).
     DaylightDate and StandardDate are returned in local time. To convert them
     to UTC use DaylightDate+StandardBias/MINUTESPERDAY and
     StandardDate+DaylightBias/MINUTESPERDAY.
     Returns false if 'TZ' is invalid or if it specifies "absolute date" for a
     year not equal to 'year' parameter. }
  function GetTZDaylightSavingInfoForYear(
    const TZ: TTimeZoneInformation; year: word;
    var DaylightDate, StandardDate: TDateTime;
    var DaylightBias, StandardBias: longint): boolean;

  {: Returns daylight saving information for a specified time zone and current
     year. See GetTZDaylightSavingInfoForYear for more information. }
  function GetTZDaylightSavingInfo(
    const TZ: TTimeZoneInformation;
    var DaylightDate, StandardDate: TDateTime;
    var DaylightBias, StandardBias: longint): boolean;

  {: Returns daylight saving information for current time zone and specified
     year. See GetTZDaylightSavingInfoForYear for more information. }
  function GetDaylightSavingInfoForYear(
    year: word;
    var DaylightDate, StandardDate: TDateTime;
    var DaylightBias, StandardBias: longint): boolean;

  {: Returns daylight saving information for current time zone and year. See
     GetTZDaylightSavingInfoForYear for more information. }
  function GetDaylightSavingInfo(
    var DaylightDate, StandardDate: TDateTime;
    var DaylightBias, StandardBias: longint): boolean;

  {: Converts local time to UTC according to a given timezone rules. Takes into
     account daylight saving time as it was active at that time. This is not
     very safe as DST rules are always changing.
     Special processing is done for the times during the standard/daylight time
     switch.
     If the specified local time lies in the non-existing area (when clock is
     moved forward), function returns 0.
     If the specified local time lies in the ambigious area (when clock is moved
     backward), function takes into account value of preferDST parameter. If it
     is set to true, time is converted as if it belongs to the daylight time. If
     it is set to false, time is converted as if it belong to the standard time.
  }
  function TZLocalTimeToUTC(
    const TZ: TTimeZoneInformation;
    const loctime: TDateTime;
    preferDST: boolean): TDateTime;

  {: Converts local time to UTC according to a given timezone rules. Takes into
     account daylight saving time as it was active at that time. This is not
     very safe as DST rules are always changing.
     In Windows NT/2000 (but not in 95/98) you can use API function
     SystemTimeToTzSpecificLocalTime instead. }
  function UTCToTZLocalTime(
    const TZ: TTimeZoneInformation;
    const utctime: TDateTime): TDateTime;

  {: Converts local time to UTC according to a current time zone. See
     TzLocalTimeToUTC for more information. }
  function LocalTimeToUTC(const loctime: TDateTime; preferDST: boolean): TDateTime;

  {: Converts UTC time to local time according to a current time zone. See
     UTCToTZLocalTime for more information. }
  function UTCToLocalTime(const utctime: TDateTime): TDateTime;

  {: Returns current UTC date and time. }
  function NowUTC: TDateTime;

  {: Returns current UTC time. }
  function TimeUTC: TDateTime;

  {: Returns current UTC date. }
  function DateUTC: TDateTime;

  {: Compares two TSystemTime records. Returns -1 if st1 < st2, 1 is st1 > st2,
     and 0 if st1 = st2. }
  function CompareSysTime(st1, st2: TSystemTime): integer;

  {: Compares two TTimeZoneInformation records. }
  function IsEqualTZ(tz1, tz2: TTimeZoneInformation): boolean;

  // added by idw =========================================================== //

  {: Converts Windows API TSystemTime structure to Delphi's TDateTime. }
  function SystemTimeToDateTime(const ST: TSystemTime): TDateTime;

  {: Converts Windows API TSystemTime to UNIX formatted UTC. }
  function SystemTimeToUnixTime(const ST: TSystemTime): longint;

  {: Converts UNIX formatted UTC to local time. }
  function UnixTimeToLocalTime(unixtime: longint): TDateTime;

implementation

  function DateEQ(const date1, date2: TDateTime): boolean;
  begin
    Result := (Abs(date1-date2) < 1/(10*MSecsPerDay));
  end; { DateEQ }

  function FixDT(const date: TDateTime): TDateTime;
  var
    ye,mo,da,ho,mi,se,ms: word;
  begin
    try
      DecodeDate(date,ye,mo,da);
      DecodeTime(date,ho,mi,se,ms);
      Result := EncodeDate(ye,mo,da)+EncodeTime(ho,mi,se,ms);
    except
      on E: EConvertError do Result := date;
      else raise;
    end;
  end; { FixDT }

  function Date2Day(const DT: TDateTime): Word;
  var
    m,y: word;
  begin
    DecodeDate(dt,y,m,Result);
  end; { Date2Day }

  function Date2Month(const DT: TDateTime): Word;
  var
    d,y: word;
  begin
    DecodeDate(dt,y,Result,d);
  end; { Date2Month }

  function Date2Year(const DT: TDateTime): Word;
  var
    d,m: word;
  begin
    DecodeDate(dt,Result,m,d);
  end; { Date2Year }

  function Time2Hr(const DT: TDateTime): Word;
  var
    min,sec,msec: word;
  begin
    DecodeTime(dt,Result,min,sec,msec);
  end;

  function Time2Min(const DT: TDateTime): Word;
  var
    hr,sec,msec: word;
  begin
    DecodeTime(dt,hr,Result,sec,msec);
  end;

  function Time2Sec(const DT: TDateTime): Word;
  var
    hr,min,msec: word;
  begin
    DecodeTime(dt,hr,min,Result,msec);
  end;

  function Time2MSec(const DT: TDateTime): Word;
  var
    hr,min,sec: word;
  begin
    DecodeTime(dt,hr,min,sec,Result);
  end;

  function IsLeapYear(Year: Word): Boolean;
  begin
    Result := ((Year and 3) = 0) and ((Year mod 100 > 0) or (Year mod 400 = 0))
  end; { IsLeapYear }

  function DateIsLeapYear(const DT: TDateTime): Boolean;
  begin
    Result := IsLeapYear(Date2Year(DT));
  end; { DateIsLeapYear }

  function DaysInMonth(const DT: TDateTime): Byte;
  begin
    case Date2Month(DT) of
      2: if DateIsLeapYear(DT) then
	   Result := 29
	 else Result := 28;
      4, 6, 9, 11: Result := 30;
      else Result := 31;
    end;
  end; { DaysInMonth }

  function ThisYear: Word;
  begin
    Result := Date2Year(Date);
  end; { ThisYear }

  function ThisMonth: Word;
  begin
    Result := Date2Month(Date);
  end; { ThisMonth }

  function ThisDay: Word;
  begin
    Result := Date2Day(Date);
  end; { ThisDay }

  function ThisHr: Word;
  begin
    Result := Time2Hr(Time);
  end; { ThisHr }

  function ThisMin: Word;
  begin
    Result := Time2Min(Time);
  end; { ThisMin }

  function ThisSec: Word;
  begin
    Result := Time2Sec(Time);
  end; { ThisSec }

  function ThisMSec: Word;
  begin
    Result := Time2MSec(Time);
  end; { ThisYear }

  function DayOfMonth2Date(year,month,weekInMonth,dayInWeek: word): TDateTime;
  var
    days: integer;
    day : integer;
  begin
    if (weekInMonth >= 1) and (weekInMonth <= 4) then begin
      day := DayOfWeek(EncodeDate(year,month,1));		// get first day in month
      day := 1 + dayInWeek-day;					// get first dayInWeek in month
      if day <= 0 then
	Inc(day,7);
      day := day + 7*(weekInMonth-1);				// get weekInMonth-th dayInWeek in month
      Result := EncodeDate(year,month,day);
    end
    else if weekInMonth = 5 then begin				// last week, calculate from end of month
      days := DaysInMonth(EncodeDate(year,month,1));
      day  := DayOfWeek(EncodeDate(year,month,days));		// get last day in month
      day  := days + (dayInWeek-day);
      if day > days then
	Dec(day,7);						// get last dayInWeek in month
      Result := EncodeDate(year,month,day);
    end
    else
      Result := 0;
  end; { DayOfMonth2Date }

  function DSTDate2Date(const dstDate: TSystemTime; year: word): TDateTime;
  begin
    if dstDate.wMonth = 0 then
      Result := 0						// invalid month => no DST info
    else if dstDate.wYear = 0 then begin			// day-of-month notation
      Result :=
	DayOfMonth2Date(year,dstDate.wMonth,dstDate.wDay,dstDate.wDayOfWeek+1{convert to Delphi Style}) +
	EncodeTime(dstDate.wHour,dstDate.wMinute,dstDate.wSecond,dstDate.wMilliseconds);
    end
    else if dstDate.wYear = year then				// absolute format - valid only for specified year
      Result := SystemTimeToDateTime(dstDate)
    else
      Result := 0;
  end; { DSTDate2Date }

  function GetTZDaylightSavingInfoForYear(
    const TZ: TTimeZoneInformation; year: word;
    var DaylightDate, StandardDate: TDateTime;
    var DaylightBias, StandardBias: longint): boolean;
  begin
    Result := false;
    if (TZ.DaylightDate.wMonth <> 0) and
       (TZ.StandardDate.wMonth <> 0) then
    begin
      DaylightDate := DSTDate2Date(TZ.DaylightDate,year);
      StandardDate := DSTDate2Date(TZ.StandardDate,year);
      DaylightBias := TZ.Bias+TZ.DaylightBias;
      StandardBias := TZ.Bias+TZ.StandardBias;
      Result := (DaylightDate <> 0) and (StandardDate <> 0);
    end;
  end; { GetTZDaylightSavingInfoForYear }

  function GetTZDaylightSavingInfo(
    const TZ: TTimeZoneInformation;
    var DaylightDate, StandardDate: TDateTime;
    var DaylightBias, StandardBias: longint): boolean;
  begin
    Result := GetTZDaylightSavingInfoForYear(TZ,ThisYear,DaylightDate,StandardDate,DaylightBias,StandardBias);
  end; { GetTZDaylightSavingInfo }

  function GetDaylightSavingInfoForYear(
    year: word;
    var DaylightDate, StandardDate: TDateTime;
    var DaylightBias, StandardBias: longint): boolean;
  var
    TZ: TTimeZoneInformation;
  begin
    GetTimeZoneInformation(TZ);
    Result := GetTZDaylightSavingInfoForYear(TZ,year,DaylightDate,StandardDate,StandardBias,DaylightBias);
  end; { GetDaylightSavingInfoForYear }

  function GetDaylightSavingInfo(
    var DaylightDate, StandardDate: TDateTime;
    var DaylightBias, StandardBias: longint): boolean;
  var
    TZ: TTimeZoneInformation;
  begin
    GetTimeZoneInformation(TZ);
    Result := GetTZDaylightSavingInfo(TZ,DaylightDate,StandardDate,StandardBias,DaylightBias);
  end; { GetDaylightSavingInfo }

  function TZLocalTimeToUTC(
    const TZ: TTimeZoneInformation;
    const loctime: TDateTime;
    preferDST: boolean): TDateTime;

    function Convert(const startDate, endDate, startOverl, endOverl: TDateTime;
      const startInval, endInval: TDateTime; inBias, outBias, overlBias: longint): TDateTime;
    begin
      if ((locTime > startOverl) or DateEQ(locTime,startOverl)) and (locTime < endOverl) then
	Result := loctime + overlBias/MINUTES_PER_DAY
      else if ((locTime > startInval) or DateEQ(locTime,startInval)) and (locTime < endInval) then
	Result := 0
      else if ((locTime > startDate) or DateEQ(locTime,startDate)) and (locTime < endDate) then
	Result := loctime + inBias/MINUTES_PER_DAY
      else
	Result := loctime + outBias/MINUTES_PER_DAY;
    end; { Convert }

  var
    dltBias : real;
    overBias: longint;
    stdBias : longint;
    dayBias : longint;
    stdDate : TDateTime;
    dayDate : TDateTime;
  begin { TZLocalTimeToUTC }
    if GetTZDaylightSavingInfoForYear(TZ, Date2Year(loctime), dayDate, stdDate, dayBias, stdBias) then begin
      if preferDST then
        overBias := dayBias
      else
        overBias := stdBias;
      dltBias := (stdBias-dayBias)/MINUTES_PER_DAY;
      if dayDate < stdDate then begin				// northern hemisphere
	if dayBias < stdBias then				// overlap at stdDate
	  Result := Convert(dayDate, stdDate, stdDate-dltBias, stdDate,
	    dayDate, dayDate+dltBias, dayBias, stdBias, overBias)
	else 							// overlap at dayDate - that actually never happens
	  Result := Convert(dayDate, stdDate, dayDate+dltBias, dayDate,
	    stdDate, stdDate-dltBias, dayBias, stdBias, overBias);
      end
      else begin						// southern hemisphere
	if dayBias < stdBias then				// overlap at stdDate
	  Result := Convert(stdDate, dayDate, stdDate-dltBias, stdDate,
	    dayDate, dayDate+dltBias, stdBias, dayBias, overBias)
	else							// overlap at dayDate - that actually never happens
	  Result := Convert(stdDate, dayDate, dayDate+dltBias, dayDate,
	    stdDate, stdDate-dltBias, stdBias, dayBias, overBias);
      end;
    end
    else
      Result := loctime + TZ.bias/MINUTES_PER_DAY;		// TZ does not use DST
  end; { TZLocalTimeToUTC }

  function UTCToTZLocalTime(
    const TZ: TTimeZoneInformation;
    const utctime: TDateTime): TDateTime;

    function Convert(const startDate, endDate: TDateTime; inBias, outBias: longint): TDateTime;
    begin
      if ((utctime > startDate) or DateEQ(utctime,startDate)) and (utctime < endDate) then
	Result := utctime - inBias/MINUTES_PER_DAY
      else
	Result := utctime - outBias/MINUTES_PER_DAY;
    end; { Convert }

  var
    stdUTC : TDateTime;
    dayUTC : TDateTime;
    stdBias: longint;
    dayBias: longint;
    stdDate: TDateTime;
    dayDate: TDateTime;

  begin { UTCToTZLocalTime }
    if GetTZDaylightSavingInfoForYear(TZ, Date2Year(utctime), dayDate, stdDate, dayBias, stdBias) then begin
      dayUTC := dayDate + stdBias/MINUTES_PER_DAY;
      stdUTC := stdDate + dayBias/MINUTES_PER_DAY;
      if dayUTC < stdUTC then
	Result := Convert(dayUTC,stdUTC,dayBias,stdBias)	// northern hem.
      else
	Result := Convert(stdUTC,dayUTC,stdBias,dayBias);	// southern hem.
    end
    else
      Result := utctime - TZ.bias/MINUTES_PER_DAY;		// TZ does not use DST
  end; { UTCToTZLocalTime }

  function LocalTimeToUTC(const loctime: TDateTime; preferDST: boolean): TDateTime;
  var
    TZ: TTimeZoneInformation;
  begin
    GetTimeZoneInformation(TZ);
    Result := TZLocalTimeToUTC(TZ,loctime,preferDST);
  end; { LocalTimeToUTC }

  function UTCToLocalTime(const utctime: TDateTime): TDateTime;
  var
    TZ: TTimeZoneInformation;
  begin
    GetTimeZoneInformation(TZ);
    Result := UTCToTZLocalTime(TZ,utctime);
  end; { UTCToLocalTime }

  function CompareSysTime(st1, st2: TSystemTime): integer;
  begin
    if st1.wYear < st2.wYear then
      Result := -1
    else if st1.wYear > st2.wYear then
      Result := 1
    else if st1.wMonth < st2.wMonth then
      Result := -1
    else if st1.wMonth > st2.wMonth then
      Result := 1
    else if st1.wDayOfWeek < st2.wDayOfWeek then
      Result := -1
    else if st1.wDayOfWeek > st2.wDayOfWeek then
      Result := 1
    else if st1.wDay < st2.wDay then
      Result := -1
    else if st1.wDay > st2.wDay then
      Result := 1
    else if st1.wHour < st2.wHour then
      Result := -1
    else if st1.wHour > st2.wHour then
      Result := 1
    else if st1.wMinute < st2.wMinute then
      Result := -1
    else if st1.wMinute > st2.wMinute then
      Result := 1
    else if st1.wSecond < st2.wSecond then
      Result := -1
    else if st1.wSecond > st2.wSecond then
      Result := 1
    else if st1.wMilliseconds < st2.wMilliseconds then
      Result := -1
    else if st1.wMilliseconds > st2.wMilliseconds then
      Result := 1
    else
      Result := 0;
  end; { CompareSysTime }

  function IsEqualTZ(tz1, tz2: TTimeZoneInformation): boolean;
  begin
    Result :=
      (tz1.Bias         = tz2.Bias)         and
      (tz1.StandardBias = tz2.StandardBias) and
      (tz1.DaylightBias = tz2.DaylightBias) and
      (CompareSysTime(tz1.StandardDate,tz2.StandardDate) = 0) and
      (CompareSysTime(tz1.DaylightDate,tz2.DaylightDate) = 0) and
      (WideCharToString(tz1.StandardName) = WideCharToString(tz2.StandardName)) and
      (WideCharToString(tz1.DaylightName) = WideCharToString(tz2.DaylightName));
  end; { IsEqualTZ }

  function NowUTC: TDateTime;
  var
    sysnow: TSystemTime;
  begin
    GetSystemTime(sysnow);
    Result := SystemTimeToDateTime(sysnow);
  end; { NowUTC }

  function TimeUTC: TDateTime;
  begin
    Result := Frac(NowUTC);
  end; { TimeUTC }

  function DateUTC: TDateTime;
  begin
    Result := Int(NowUTC);
  end; { DateUTC }

  // added by idw =========================================================== //

  function SystemTimeToDateTime(const ST: TSystemTime): TDateTime;
  begin
    with ST do Result:=EncodeDate(wYear, wMonth, wDay) + EncodeTime(wHour, wMinute, wSecond, wMilliseconds);
  end;

  function SystemTimeToUnixTime(const ST: TSystemTime): longint;
  var
    UTC: TDateTime;
  begin
    UTC:=SystemTimeToDateTime(ST);
    Result:=Round((UTC - EncodeDate(1970, 1, 1)) * SECONDS_PER_DAY);
  end;

  function UnixTimeToLocalTime(unixtime: longint): TDateTime;
  var
    TZ: TTimeZoneInformation;
  begin
    Result:=EncodeDate(1970, 1, 1) + unixtime/SECONDS_PER_DAY;
    GetTimeZoneInformation(TZ);
    Result:=UTCToTZLocalTime(TZ, Result);
  end;

end.

