{$J+,Z4}
unit PGPDecode;

{------------------------------------------------------------------------------}
{                                                                              }
{                 This unit is partly based on Steve Heller's                  }
{         SPGP sources available from http://www.oz.net/~srheller/spgp/        }
{                                                                              }
{                Portions created by Michael in der Wiesche are                }
{                 Copyright (C) 2001 by Michael in der Wiesche                 }
{                                                                              }
{------------------------------------------------------------------------------}

interface

uses
  Windows,
  Classes,
  SysUtils,
  KeyFuncs,
  KeyPropTypes,
  PGPDialogs,
  pgpSC,
  pgpTLS,
  pgpKeys,
  pgpEvents,
  pgpMemoryMgr,
  pgpUtilities,
  pgpOptionList,
  pgpErrors, pgpPubTypes, pgpBase;

type
  TSigStatus = (
    SIGNED_NOT,
    SIGNED_BAD,
    SIGNED_GOOD,
    SIGNED_NO_KEY
  );
  TSigPropsRec = Record
    sStatus: TSigStatus;
    sHexID: String;
    sUserID: String;
    sCreaTimeStr: String;
    sCreaTimeNum: Longint;
    sChecked: Longbool;
    sVerified: Longbool;
    sValidity: TValidityLevel;
    sRevoked: Longbool;
    sDisabled: Longbool;
    sExpired: Longbool;
    sDetached: Longbool;
    sAxiomatic: Longbool;
  end;
  TAnalysis = (
    ENCRYPTED,
    SIGNED,
    DETACHED_SIGNATURE,
    KEY,
    UNKNOWN,
    X509_CERTIFICATE
  );
  TPassThroughOption = (
    PassThrough_Keys,
    PassThrough_ClearSigned,
    PassThrough_Unrecognized
  );
  TPassOption = (
    AnalyseOnly,
    AnalysePass,
    DecodePass
  );
  TPassThroughOptions = Set of TPassThroughOption;
  TOnAddKeys = procedure(var CancelAdding: Longbool) of Object;
  TOnGetAnalysis = procedure(SectionNumber: Longint; SectionType: TAnalysis; var SkipSection: Longbool) of Object;
  TOnGetInputFileName = procedure(var SuggestedName: String) of Object;
  TOnGetOutputFileName = procedure(var SuggestedName: String) of Object;
  TOnGetSignedFileName = procedure(var SuggestedName: String) of Object;
  TOnGetPassphrase = procedure(const Passphrase: PChar;
			       const DecryptionKeyList: TKeyPropsList;
			       Conventional, BadPassphrase: Longbool;
			       var Cancel: Longbool) of Object;
  TOnGetSignature = procedure(const SigProps: TSigPropsRec) of Object;
  TOnShowProgress = procedure(BytesProcessed, BytesTotal: Longint) of Object;
  TOnWipePassphrase = procedure(const Passphrase: PChar) of Object;
  TPGPDecodeCustom = class(TComponent)
  private
    // internal
    FContext: pPGPContext;
    FKeySetMain: pPGPKeySet;
    FtlsContext: pPGPTLSContext;
    FAllocatedOutputBuffer: PChar;
    FActualOutputSize: PGPSize;
    FInputSize: PGPSize;
    FInputFileName: String;
    FOutputBuffer: String;
    FDetachedSigVerify: Longbool;
    FEyesOnly: Longbool;
    FOutputExists: Longbool;
    FOutputError: PGPError;
    FPassOption: TPassOption;
    FPassphrase: PChar;
    FPassSection: PGPUInt32;
    FRecipientsData: TPGPEventRecipientsData;
    FSectionNumber: PGPUInt32;
    FSecureBuffer: PChar;
    FSecureBufferSize: PGPSize;
    FClearSigned: Longbool;
    // properties
    FFileOutput: Longbool;
    FGetKeyFromServer: Longbool;
    FLogBadSignature: String;
    FLogKeyDisabled: String;
    FLogKeyExpired: String;
    FLogKeyRevoked: String;
    FLogKeyUnknown: String;
    FParentHandle: THandle;
    FPassThroughOptions: TPassThroughOptions;
    FProgressInterval: Cardinal;
    FQueryAddKeys: Longbool;
    FRecursivelyDecode: Longbool;
    FShowSigLog: Longbool;
    FKeyDlgPrompt: String;
    FPassDlgPrompt: String;
    FOutputFileName: String;
    FSignedFileName: String;
    FKeyPropsList: TKeyPropsList;
    FSigPropsRec: TSigPropsRec;
    // events
    FOnAddKeys: TOnAddKeys;
    FOnGetAnalysis: TOnGetAnalysis;
    FOnGetInputFileName: TOnGetInputFileName;
    FOnGetOutputFileName: TOnGetOutputFileName;
    FOnGetSignedFileName: TOnGetSignedFileName;
    FOnGetPassphrase: TOnGetPassphrase;
    FOnGetSignature: TOnGetSignature;
    FOnShowProgress: TOnShowProgress;
    FOnWipePassphrase: TOnWipePassphrase;
    procedure SetRecursive(Value: Longbool);
    function  InitDecode: PGPError;
    procedure FinitDecode;
    function  GetOutputFile: String;
    function  SecureMove: PGPError;
    function  ShowSigInfo: PGPError;
    function  WriteOutputFile(Output: PChar; OutputSize: DWord): PGPError;
    function  Analyze(const Input: String; FileSpec: pPGPFileSpec): PGPError;
    function  GetOptionList(var OptionList: pPGPOptionList): PGPError;
    function  SetOutputOption(var OptionList: pPGPOptionList): PGPError;
    function  Decode(Input: String; IsFile: Longbool): Longint;
    function  AnalyzeHandler(Event: pPGPEvent): PGPError;
    function  RecipientsHandler(Event: pPGPEvent): PGPError;
    function  PassphraseHandler(Event: pPGPEvent): PGPError;
    function  KeyFoundHandler(Event: pPGPEvent): PGPError;
    function  OutputHandler(Event: pPGPEvent): PGPError;
    function  SignatureHandler(Event: pPGPEvent): PGPError;
    function  EndLexHandler(Event: pPGPEvent): PGPError;
  protected
    property OutputBuffer: String
      read FOutputBuffer;
    property SigPropsRec: TSigPropsRec
      read FSigPropsRec;
    property KeyPropsList: TKeyPropsList
      read FKeyPropsList;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    function AnalyseBuffer(const DataBuffer: String): Longint; virtual;
    function AnalyseFile(const FileName: String): Longint; virtual;
    function DecodeBuffer(const DataBuffer: String): Longint; virtual;
    function DecodeFile(const FileName: String): Longint; virtual;
    property ParentHandle: THandle
      read FParentHandle
      write FParentHandle;
  published
    property FileOutput: Longbool
      read FFileOutput
      write FFileOutput;
    property GetKeyFromServer: Longbool
      read FGetKeyFromServer
      write FGetKeyFromServer;
    property LogBadSignature: String
      read FLogBadSignature
      write FLogBadSignature;
    property LogKeyDisabled: String
      read FLogKeyDisabled
      write FLogKeyDisabled;
    property LogKeyExpired: String
      read FLogKeyExpired
      write FLogKeyExpired;
    property LogKeyRevoked: String
      read FLogKeyRevoked
      write FLogKeyRevoked;
    property LogKeyUnknown: String
      read FLogKeyUnknown
      write FLogKeyUnknown;
    property PassThroughOptions: TPassThroughOptions
      read FPassThroughOptions
      write FPassThroughOptions;
    property ProgressInterval: Cardinal
      read FProgressInterval
      write FProgressInterval;
    property RecursivelyDecode: Longbool
      read FRecursivelyDecode
      write SetRecursive;
    property QueryAddKeys: Longbool
      read FQueryAddKeys
      write FQueryAddKeys;
    property ShowSigLog: Longbool
      read FShowSigLog
      write FShowSigLog;
    property KeyDlgPrompt: String
      read FKeyDlgPrompt
      write FKeyDlgPrompt;
    property PassDlgPrompt: String
      read FPassDlgPrompt
      write FPassDlgPrompt;
    property OutputFileName: String
      read FOutputFileName
      write FOutputFileName;
    property SignedFileName: String
      read FSignedFileName
      write FSignedFileName;
    property OnAddKeys: TOnAddKeys
      read FOnAddKeys
      write FOnAddKeys;
    property OnGetAnalysis: TOnGetAnalysis
      read FOnGetAnalysis
      write FOnGetAnalysis;
    property OnGetInputFileName: TOnGetInputFileName
      read FOnGetInputFileName
      write FOnGetInputFileName;
    property OnGetSignedFileName: TOnGetSignedFileName
      read FOnGetSignedFileName
      write FOnGetSignedFileName;
    property OnGetOutputFileName: TOnGetOutputFileName
      read FOnGetOutputFileName
      write FOnGetOutputFileName;
    property OnGetPassphrase: TOnGetPassphrase
      read FOnGetPassphrase
      write FOnGetPassphrase;
    property OnGetSignature: TOnGetSignature
      read FOnGetSignature
      write FOnGetSignature;
    property OnShowProgress: TOnShowProgress
      read FOnShowProgress
      write FOnShowProgress;
    property OnWipePassphrase: TOnWipePassphrase
      read FOnWipePassphrase
      write FOnWipePassphrase;
  end;

implementation

function EventHandler(Context: pPGPContext; Event: pPGPEvent; UserValue: PGPUserValue): PGPError; cdecl;
begin
  Result:=0;
  with TPGPDecodeCustom(UserValue) do begin
    case Event^.EType of
      kPGPEvent_NullEvent:		if (FPassOption=DecodePass)
      					and not (FDetachedSigVerify or FClearSigned) then begin
					  if TMethod(FOnShowProgress).Code<>nil then begin
					    // BytesTotal always stays 0 => use FInputSize
					    with Event^.Data.NullData do FOnShowProgress(BytesWritten, FInputSize);
					  end;
					  ProcessMessages;
					end;
      kPGPEvent_InitialEvent:		;
      kPGPEvent_FinalEvent:		;
      kPGPEvent_ErrorEvent:	  	Result:=Event^.Data.ErrorData.Error;
      kPGPEvent_WarningEvent:		;
      kPGPEvent_PassphraseEvent:        Result:=PassphraseHandler(Event);
      kPGPEvent_AnalyzeEvent:		Result:=AnalyzeHandler(Event);
      kPGPEvent_RecipientsEvent:  	if FPassOption=DecodePass then Result:=RecipientsHandler(Event);
      kPGPEvent_KeyFoundEvent:          if FPassOption=DecodePass then Result:=KeyFoundHandler(Event);
      kPGPEvent_OutputEvent:		if FPassOption=DecodePass then OutputHandler(Event);
      kPGPEvent_SignatureEvent:         if FPassOption=DecodePass then Result:=SignatureHandler(Event);
      kPGPEvent_BeginLexEvent:	  	FSectionNumber:=succ(Event^.Data.BeginLexData.SectionNumber);
      kPGPEvent_EndLexEvent:      	Result:=EndLexHandler(Event);
      kPGPEvent_DetachedSignatureEvent:	;
      kPGPEvent_DecryptionEvent:	;
    end;
  end;
end;

constructor TPGPDecodeCustom.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FKeyPropsList:=TKeyPropsList.Create(0, spgpKeyPropFlag_IDFlags);
  FLogBadSignature:='bad signature';
  FLogKeyDisabled:='disabled key';
  FLogKeyExpired:='expired key';
  FLogKeyRevoked:='revoked key';
  FLogKeyUnknown:='unknown key';
  FProgressInterval:=1000;
end;

destructor TPGPDecodeCustom.Destroy;
begin
  FKeyPropsList.Free;
  inherited Destroy;
end;

procedure TPGPDecodeCustom.SetRecursive(Value: Longbool);
begin
  if Value<>FRecursivelyDecode then FRecursivelyDecode:=Value;
  if FRecursivelyDecode then Include(FPassThroughOptions, PassThrough_ClearSigned);
end;

function TPGPDecodeCustom.InitDecode: PGPError;
begin
  FSectionNumber:=0;
  FOutputError:=0;
  FOutputBuffer:='';
  FSecureBufferSize:=0;
  FSecureBuffer:=nil;
  FActualOutputSize:=0;
  FAllocatedOutputBuffer:=nil;
  FDetachedSigVerify:=false;
  FClearSigned:=false;
  FEyesOnly:=false;
  FKeyPropsList.Clear;
  FPassSection:=0;
  FPassphrase:=nil;
  FillChar(FSigPropsRec, SizeOf(TSigPropsRec), 0);
  FillChar(FRecipientsData, SizeOf(TPGPEventRecipientsData), 0);
  if FPassOption=AnalyseOnly then begin
    FContext:=nil;
    Result:=PGPNewContext(kPGPsdkAPIVersion, FContext);
  end
  else Result:=KeyDlgInit(FContext, FtlsContext, FKeySetMain);
end;

procedure TPGPDecodeCustom.FinitDecode;
begin
  with FRecipientsData do begin
    PGPFreeKeySet(RecipientSet);
    if KeyIDArray<>nil then GlobalFree(GlobalHandle(KeyIDArray));
  end;
  if FPassOption=AnalyseOnly then
    PGPFreeContext(FContext)
  else KeyDlgFree(FContext, FtlsContext, FKeySetMain);
end;

function TPGPDecodeCustom.GetOutputFile: String;
var
  FileExtension	: String;
begin
  if FDetachedSigVerify then begin
    if FSignedFileName='' then begin
      FSignedFileName:=Trim(FInputFileName);
      if LowerCase(ExtractFileExt(FSignedFileName))='.sig' then begin
	Delete(FSignedFileName, Length(FSignedFileName)-3, 4);
      end;
    end;
    if ((FSignedFileName=FInputFileName) or not FileExists(FSignedFileName))
    and Assigned(FOnGetSignedFileName) then begin
      FOnGetSignedFileName(FSignedFileName);
    end;
    Result:=FSignedFileName;
  end
  else begin
    if FOutputFileName='' then begin
      FOutputFileName:=Trim(FInputFileName);
      FileExtension:=LowerCase(ExtractFileExt(FOutputFileName));
      if (FileExtension='.pgp') or (FileExtension='.asc') then begin
	Delete(FOutputFileName, Length(FOutputFileName)-3, 4);
      end;
    end;
    if (FOutputFileName='') or (ExtractFileExt(FOutputFileName)='') or FileExists(FOutputFileName) then begin
      if Assigned(FOnGetOutputFileName) then
	FOnGetOutputFileName(FOutputFileName)
      else FOutputFileName:='';
    end;
    Result:=FOutputFileName;
  end;
end;

function TPGPDecodeCustom.SecureMove: PGPError;
begin
  try
    Result:=PGPReallocData(PGPGetDefaultMemoryMgr,
			   pointer(FSecureBuffer),
			   succ(FSecureBufferSize + FActualOutputSize),
			   kPGPMemoryMgrFlags_Clear);
    if Result=0 then begin
      Move(FAllocatedOutputBuffer[0], FSecureBuffer[FSecureBufferSize], FActualOutputSize);
      inc(FSecureBufferSize, FActualOutputSize);
    end;
  finally
    PGPFreeData(FAllocatedOutputBuffer);
  end;
end;

function TPGPDecodeCustom.ShowSigInfo: PGPError;
var
  Status	: String;
  Source	: String;
  ID		: String;
  Validity	: String;
  Info		: String;
begin
  Result:=0;
  with FSigPropsRec do begin
    if sStatus<>SIGNED_NOT then begin
      // don't change these (constant) PGP strings, see SigEvent.c
      if sStatus=SIGNED_GOOD then
	Status:='Good Signature' + #10
      else Status:='Bad Signature' + #10;
      // these strings may be customized
      if FInputFileName<>'' then begin
	if FDetachedSigVerify then
	  Source:=ExtractFileName(FSignedFileName) + #10
	else Source:=ExtractFileName(FInputFileName) + #10;
      end
      else Source:='(...)' + #10;
      if sStatus=SIGNED_NO_KEY then
	ID:='(' + sHexID + ')' + #10
      else ID:=sUserID + #10;
      // don't change these (constant) PGP strings, see SigEvent.c
      if sAxiomatic then
	Validity:='Implicit Trust' + #10 + sCreaTimeStr
      else begin
	case sValidity of
	  Validity_Unknown, Validity_Invalid: Validity:='Invalid Key' + #10 + sCreaTimeStr;
	  Validity_Marginal: Validity:='Marginal Key' + #10 + sCreaTimeStr;
	  Validity_Complete: Validity:='Valid Key' + #10 + sCreaTimeStr;
	end;
      end;
      // these strings may be customized
      if sUserID='' then
	Info:=FLogKeyUnknown
      else begin
	if not sVerified then
	  Info:=FLogBadSignature
	else begin
	  if sRevoked then Info:=FLogKeyRevoked;
	  if sDisabled then Info:=FLogKeyDisabled;
	  if sExpired then Info:=FLogKeyExpired;
	end;
      end;
      if Info<>'' then  Info:=' (' + Info + ')';
      if not SendPGPlogMsg(FParentHandle, PChar(Status + Source + ID + Validity + Info)) then begin
	Result:=kPGPError_FeatureNotAvailable;
      end;
    end;
  end;
end;

function TPGPDecodeCustom.WriteOutputFile(Output: PChar; OutputSize: DWord): PGPError;
var
  OutputFile	: THandle;
  BytesWritten	: DWord;
begin
  OutputFile:=CreateFile(PChar(FOutputFileName), GENERIC_WRITE, 0, nil, CREATE_ALWAYS,
			 FILE_ATTRIBUTE_ARCHIVE or FILE_FLAG_WRITE_THROUGH, 0);
  if OutputFile<>INVALID_HANDLE_VALUE then begin
    if not WriteFile(OutputFile, Output[0], OutputSize, BytesWritten, nil)
    or (BytesWritten<>OutputSize) then
      Result:=kPGPError_WriteFailed
    else Result:=kPGPError_NoErr;
    CloseHandle(OutputFile);
  end
  else Result:=kPGPError_CantOpenFile;
end;

function TPGPDecodeCustom.Analyze(const Input: String; FileSpec: pPGPFileSpec): PGPError;
var
  OptionList	: pPGPOptionList;
begin
  OptionList:=nil;
  Result:=PGPBuildOptionList(FContext, OptionList,
    [
      PGPODiscardOutput(FContext, PGPBoolean(true)),
      PGPOEventHandler(FContext, EventHandler, Self)
    ]);
  if Result<>0 then Exit;
  if FileSpec<>nil then begin
    Result:=PGPAppendOptionList(OptionList, [PGPOInputFile(FContext, FileSpec)]);
  end
  else if Input<>'' then begin
    Result:=PGPAppendOptionList(OptionList, [PGPOInputBuffer(FContext, @Input[1], Length(Input))]);
  end;
  if Result<>0 then Exit;
  try
    Result:=pgpEvents.PGPDecode(FContext, OptionList, PGPOLastOption(FContext));
    if Result=kPGPError_UserAbort then Result:=0;
  finally
    PGPFreeOptionList(OptionList);
  end;
end;

function TPGPDecodeCustom.GetOptionList(var OptionList: pPGPOptionList): PGPError;
begin
  Result:=PGPBuildOptionList(FContext, OptionList,
    [
      PGPOKeySetRef(FContext, FKeySetMain),
      PGPOSendNullEvents(FContext, FProgressInterval),
      PGPOPassThroughKeys(FContext, PGPBoolean(PassThrough_Keys in PassThroughOptions)),
      PGPOPassThroughClearSigned(FContext, PGPBoolean(PassThrough_ClearSigned in PassThroughOptions)),
      PGPOPassThroughIfUnrecognized(FContext, PGPBoolean(PassThrough_Unrecognized in PassThroughOptions)),
      PGPOSendEventIfKeyFound(FContext, PGPBoolean(true)),
      PGPORecursivelyDecode(FContext, PGPBoolean(FRecursivelyDecode) and 1),
      PGPOEventHandler(FContext, EventHandler, Self)
    ]);
end;

function TPGPDecodeCustom.SetOutputOption(var OptionList: pPGPOptionList): PGPError;
var
  FileSpec	: pPGPFileSpec;
begin
  if FDetachedSigVerify then begin
    if GetOutputFile<>'' then begin
      FileSpec:=nil;
      Result:=PGPNewFileSpecFromFullPath(FContext, PChar(FSignedFileName), FileSpec);
      if Result<>0 then Exit;
      try
	Result:=PGPAppendOptionList(OptionList,
	  [PGPODetachedSig(FContext, PGPOInputFile(FContext, FileSpec), PGPOLastOption(FContext))]);
      finally
	PGPFreeFileSpec(FileSpec);
      end;
    end
    else Result:=kPGPError_DetachedSignatureFound;
  end
  else if FileOutput then begin
    if GetOutputFile<>'' then begin
      Result:=PGPAppendOptionList(OptionList,
	[PGPOAllocatedOutputBuffer(FContext, FAllocatedOutputBuffer, -1, FActualOutputSize)]);
    end
    else Result:=kPGPError_CantOpenFile;
  end
  else begin
    Result:=PGPAppendOptionList(OptionList,
      [PGPOAllocatedOutputBuffer(FContext, FAllocatedOutputBuffer, -1, FActualOutputSize)]);
  end;
end;

function TPGPDecodeCustom.Decode(Input: String; IsFile: Longbool): Longint;
var
  BugFix	: Longbool;
  OptionList	: pPGPOptionList;
  FileSpec	: pPGPFileSpec;
  hInFile	: THandle;
  Output	: PChar;
  OutputSize	: DWord;
begin
  Result:=0;
  BugFix:=false;
  if IsFile then begin
    FInputFileName:=Input;
    if not FileExists(FInputFileName) then begin
      if Assigned(FOnGetInputFileName) then begin
	FOnGetInputFileName(FInputFileName);
	if FInputFileName='' then Result:=kPGPError_FileNotFound;
      end
      else Result:=kPGPError_FileNotFound;
    end;
  end
  else begin
    if Input='' then
      Result:=kPGPError_ItemNotFound
    else if Input[1]>#127 then begin  // to avoid strange decoding error
      Input:='@' + Input;
      BugFix:=true;
    end;
  end;
  if Result=0 then begin
    FileSpec:=nil;
    OptionList:=nil;
    Result:=InitDecode;
    if Result<>0 then Exit;
    try
      if IsFile then begin
	Result:=PGPNewFileSpecFromFullPath(FContext, PChar(FInputFileName), FileSpec);
	if Result<>0 then Exit;
      end;
      try
	Result:=Analyze(Input, FileSpec);
	if Result=kPGPError_BadIntegrity then Result:=0;  // avoid strange error after kPGPError_SkipSection
	if FPassOption<>AnalyseOnly then begin
	  if Result<>0 then Exit;
	  Result:=GetOptionList(OptionList);
	  if Result<>0 then Exit;
	  if IsFile then begin
	    hInFile:=FileOpen(FInputFileName, fmOpenRead or fmShareDenyNone);
	    if hInFile<>INVALID_HANDLE_VALUE then begin
	      FInputSize:=GetFileSize(hInFile, nil);
	      FileClose(hInFile);
	    end
	    else FInputSize:=-1;
	    if (FInputSize=0) or (FInputSize=-1) then begin
	      Result:=kPGPError_ReadFailed;
	      Exit;
	    end;
	    Result:=PGPAppendOptionList(OptionList, [PGPOInputFile(FContext, FileSpec)]);
	  end
	  else begin
	    FInputSize:=Length(Input);
	    Result:=PGPAppendOptionList(OptionList, [PGPOInputBuffer(FContext, @Input[1], FInputSize)]);
	  end;
	  try
	    if Result<>0 then Exit;
	    if FDetachedSigVerify or FClearSigned then begin
	      Result:=SetOutputOption(OptionList);
	      if Result<>0 then Exit;
	    end
	    else FOutputExists:=false;
	    FPassOption:=DecodePass;
	    Result:=pgpEvents.PGPDecode(FContext, OptionList, PGPOLastOption(FContext));
	    try
	      if (FPassOption<>AnalyseOnly) and (Result=0) then begin
		if not (FDetachedSigVerify or FClearSigned) then begin
		  Result:=SecureMove;
		  if Result<>0 then Exit;
		  FAllocatedOutputBuffer:=FSecureBuffer;
		  FActualOutputSize:=FSecureBufferSize;
		  if BugFix and (PassThrough_Unrecognized in PassThroughOptions) then begin // remove leading AT
		    Output:=FAllocatedOutputBuffer + 1;
		    OutputSize:=pred(FActualOutputSize);
		  end
		  else begin // AT already removed
		    Output:=FAllocatedOutputBuffer;
		    OutputSize:=FActualOutputSize;
		  end;
		  if FEyesOnly then begin
		    Result:=TempestViewer(FContext, FParentHandle, Output, OutputSize, true);
		    ProcessMessages;
		    Exit;
		  end;
		end
		else begin
		  if BugFix and (PassThrough_Unrecognized in PassThroughOptions) then begin // remove leading AT
		    Output:=FAllocatedOutputBuffer + 1;
		    OutputSize:=pred(FActualOutputSize);
		  end
		  else begin // AT already removed
		    Output:=FAllocatedOutputBuffer;
		    OutputSize:=FActualOutputSize;
		  end;
		end;
		if FileOutput then
		  WriteOutputFile(Output, OutputSize)
		else begin
		  SetLength(FOutputBuffer, OutputSize);
		  Move(Output[0], FOutputBuffer[1], OutputSize);
		end;
	      end;
	    except
	      on EOutOfMemory do Result:=kPGPError_OutOfMemory;
	    end;
	  finally
	    try
	      PGPFreeOptionList(OptionList);
	    finally
	      try
		if Assigned(FOnWipePassphrase) then FOnWipePassphrase(FPassphrase);
	      finally
		PGPFreeData(FAllocatedOutputBuffer);
		PGPFreeData(FPassphrase);
		FPassphrase:=nil;
	      end;
	    end;
	  end;
	end;
      finally
	PGPFreeFileSpec(FileSpec);
      end;
    finally
      FinitDecode;
    end;
  end;
  // OutputHandler errors don't get returned correctly by PGP
  if (Result=kPGPError_BadParams) and (FOutputError<>0) then Result:=FOutputError;
end;

function TPGPDecodeCustom.AnalyseBuffer(const DataBuffer: String): Longint;
begin
  FPassOption:=AnalyseOnly;
  Result:=Decode(DataBuffer, false);
end;

function TPGPDecodeCustom.AnalyseFile(const FileName: String): Longint;
begin
  FPassOption:=AnalyseOnly;
  Result:=Decode(FileName, true);
end;

function TPGPDecodeCustom.DecodeBuffer(const DataBuffer: String): Longint;
begin
  FPassOption:=AnalysePass;
  Result:=Decode(DataBuffer, false);
end;

function TPGPDecodeCustom.DecodeFile(const FileName: String): Longint;
begin
  FPassOption:=AnalysePass;
  Result:=Decode(FileName, true);
end;

function TPGPDecodeCustom.AnalyzeHandler(Event: pPGPEvent): PGPError;
var
  SkipSection	: Longbool;
begin
  Result:=0;
  SkipSection:=false;
  with Event^.Data.AnalyzeData do begin
    case FPassOption of
      AnalyseOnly: begin
	if Assigned(FOnGetAnalysis) then FOnGetAnalysis(FSectionNumber, TAnalysis(SectionType), SkipSection);
	if SectionType<>kPGPAnalyze_Key then Result:=kPGPError_SkipSection; // avoid key data bug
      end;
      AnalysePass: begin
	case SectionType of
	  kPGPAnalyze_Signed: FClearSigned:=true;
	  kPGPAnalyze_DetachedSignature: FDetachedSigVerify:=true;
	end;
	if FDetachedSigVerify or FClearSigned then
	  Result:=kPGPError_UserAbort
	else if SectionType<>kPGPAnalyze_Key then Result:=kPGPError_SkipSection; // avoid key data bug
      end;
      DecodePass: begin
	if Assigned(FOnGetAnalysis) then FOnGetAnalysis(FSectionNumber, TAnalysis(SectionType), SkipSection);
	Result:=(ord(SkipSection) and 1)*kPGPError_SkipSection;
      end;
    end;
  end;
end;

function TPGPDecodeCustom.RecipientsHandler(Event: pPGPEvent): PGPError;
begin
  Result:=0;
  FRecipientsData:=Event^.Data.RecipientsData;
  PGPIncKeySetRefCount(FRecipientsData.RecipientSet);
end;

function TPGPDecodeCustom.PassphraseHandler(Event: pPGPEvent): PGPError;
var
  KeyPropsList	: TKeyPropsList;
  BadPassphrase	: Longbool;
  Cancel	: Longbool;
  DecryptionKey	: pPGPKey;
begin
  Result:=0;
  with Event^.Data.PassphraseData do begin
    KeyPropsList:=TKeyPropsList.Create(0, spgpKeyPropFlag_IDComplete);
    try
      if (FPassphrase=nil) or (FSectionNumber=FPassSection) then begin
	FPassSection:=FSectionNumber;
	BadPassphrase:=false;
	if FPassphrase<>nil then begin
	  PGPFreeData(FPassphrase);
	  FPassphrase:=nil;
	  BadPassphrase:=true;
	end;
	if Assigned(FOnGetPassphrase) then begin
	  if not boolean(IsConventional) then begin
	    Result:=GetKeySetProps(FContext, KeySet, KeyPropsList, spgpKeyPropFlag_IDComplete,
				   KeyFilterFlag_CanSign, UserID_Ordering);
	    if Result=0 then Result:=kPGPError_SecretKeyNotFound;
	  end;
	  if Result>=0 then begin
	    Cancel:=false;
	    Result:=kPGPError_UserAbort;
	    FPassphrase:=PGPNewSecureData(PGPGetDefaultMemoryMgr, 256, kPGPMemoryMgrFlags_Clear);
	    if FPassphrase<>nil then begin
	      FOnGetPassphrase(FPassphrase, KeyPropsList, boolean(IsConventional), BadPassphrase, Cancel);
	      if Cancel then Exit;
	    end
	    else begin
	      Result:=kPGPError_OutOfMemory;
	      Exit;
	    end;
	  end
	  else Exit;
	end
	else begin
	  if boolean(IsConventional) then
	    Result:=ConvDecPassphraseDialog(FContext, FPassphrase, FPassDlgPrompt, FParentHandle)
	  else with FRecipientsData do begin
	    Result:=DecryptionPassphraseDialog(FContext, RecipientSet, KeyCount, KeyIDArray,
					       DecryptionKey, FPassphrase, true, FPassDlgPrompt, FParentHandle);
	  end;
	  if Result<>0 then Exit;
	end;
      end
      else FPassSection:=FSectionNumber;
    finally
      KeyPropsList.Free;
    end;
  end;
  Result:=PGPAddJobOptions(Event^.Job, PGPOPassphrase(FContext, FPassphrase), PGPOLastOption(FContext));
end;

function TPGPDecodeCustom.KeyFoundHandler(Event: pPGPEvent): PGPError;
var
  CancelAdding	: Longbool;
begin
  Result:=0; // don't cancel decoding whatever happens here
  CancelAdding:=false;
  with Event^.Data.KeyFoundData do begin
    if PGP7X then begin
      try
	if FQueryAddKeys then begin
	  if Assigned(FOnAddKeys) then FOnAddKeys(CancelAdding);
	  if not CancelAdding then begin
	    SelectAddKeysToKeyRing(FContext, FtlsContext, PGPPeekKeyDBRootKeySet(KeysFound), FKeySetMain,
				   FKeyPropsList, PChar(FKeyDlgPrompt),
				   false, spgpKeyPropFlag_IDFlags,
				   FParentHandle);
	  end;
	end;
      finally
	PGPFreeKeyDB(PGPPeekKeySetKeyDB(KeysFound));
      end;
    end
    else begin
      try
	if FQueryAddKeys then begin
	  if Assigned(FOnAddKeys) then FOnAddKeys(CancelAdding);
	  if not CancelAdding then begin
	    SelectAddKeysToKeyRing(FContext, FtlsContext, KeysFound, FKeySetMain,
				   FKeyPropsList, PChar(FKeyDlgPrompt),
				   false, spgpKeyPropFlag_IDFlags,
				   FParentHandle);
	  end;
	end;
      finally
	PGPFreeKeySet(KeysFound);
      end;
    end;
  end;
end;

// causes trouble when used with FPassThroughClearSigned on clear signed input
function TPGPDecodeCustom.OutputHandler(Event: pPGPEvent): PGPError;
begin
  Result:=0;
  with Event^.Data.OutputData do begin
    if boolean(ForYourEyesOnly) then FEyesOnly:=true;
    if FOutputExists then
      Result:=SecureMove
    else if FFileOutput and not FEyesOnly then begin
      if FOutputFileName='' then FOutputFileName:=ExtractFilePath(FInputFileName) + SuggestedName;
      if GetOutputFile='' then Result:=kPGPError_CantOpenFile;
    end;
    if Result=0 then begin
      Result:=PGPAddJobOptions(Event^.Job,
			       PGPOAllocatedOutputBuffer(FContext, FAllocatedOutputBuffer, -1, FActualOutputSize),
			       PGPOLastOption(FContext));
      FOutputExists:=true;
    end
    else FOutputError:=Result;
  end;
end;

function TPGPDecodeCustom.SignatureHandler(Event: pPGPEvent): PGPError;
var
  PGPSignKeyID	: TPGPKeyID7;
  SignKeyID	: TKeyID;
  KeysFound	: Pointer;
  CancelAdding	: Longbool;
begin
  Result:=0; // don't cancel decoding whatever happens here
  CancelAdding:=false;
  FillChar(FSigPropsRec, SizeOf(TSigPropsRec), 0);
  if PGP7X then with Event^.Data.SignatureData.PGPEventSignatureData7, FSigPropsRec do begin
    if boolean(Checked) then begin
      sHexID:=GetKeyPropKeyID(SigningKey);
      sUserID:=GetKeyPropUserID(SigningKey);
      sCreaTimeStr:=UnixTimeToLocalTimeStr(PGPGetStdTimeFromPGPTime(CreationTime));
      sCreaTimeNum:=CreationTime;
      sChecked:=boolean(Checked);
      sVerified:=boolean(Verified);
      sValidity:=TValidityLevel(KeyValidity);
      sRevoked:=boolean(KeyRevoked);
      sDisabled:=boolean(KeyDisabled);
      sExpired:=boolean(KeyExpired);
      sAxiomatic:=GetKeyPropIsAxiomatic(SigningKey);
      sStatus:=TSigStatus(Checked + Verified);
    end
    else begin
      Move(SigningKeyID, PGPSignKeyID, SizeOf(TPGPKeyID7));
      Result:=PGPGetKeyIDString(PGPSignKeyID, kPGPKeyIDString_Full, SignKeyID);
      if Result=0 then begin
	sCreaTimeStr:=UnixTimeToLocalTimeStr(PGPGetStdTimeFromPGPTime(CreationTime));
	sCreaTimeNum:=CreationTime;
	sHexID:=SignKeyID;
      end;
      sStatus:=SIGNED_NO_KEY;
      if (Result=0) and FGetKeyFromServer then begin
	KeysFound:=nil;
	Result:=KeyServerDialog(FContext, FtlsContext, FKeySetMain, SignKeyID, pPGPKeyDB(KeysFound), FParentHandle);
	if (Result=0) and (KeysFound<>nil) then begin
	  try
	    if Assigned(FOnAddKeys) then FOnAddKeys(CancelAdding);
	    if not CancelAdding then begin
	      SelectAddKeysToKeyRing(FContext, FtlsContext, PGPPeekKeyDBRootKeySet(pPGPKeyDB(KeysFound)), FKeySetMain,
				     FKeyPropsList, PChar(FKeyDlgPrompt),
				     false, spgpKeyPropFlag_IDFlags,
				     FParentHandle);
	    end;
	    PGPAddJobOptions(Event^.Job,
			     PGPOKeySetRef(FContext, PGPPeekKeyDBRootKeySet(pPGPKeyDB(KeysFound))),
			     PGPOLastOption(FContext));
	  finally
	    PGPFreeKeyDB(pPGPKeyDB(KeysFound));
	  end;
	end;
      end;
      Result:=0;
    end;
    sDetached:=FDetachedSigVerify;
  end
  else with Event^.Data.SignatureData.PGPEventSignatureData6, FSigPropsRec do begin
    if boolean(Checked) then begin
      sHexID:=GetKeyPropKeyID(SigningKey);
      sUserID:=GetKeyPropUserID(SigningKey);
      sCreaTimeStr:=UnixTimeToLocalTimeStr(PGPGetStdTimeFromPGPTime(CreationTime));
      sCreaTimeNum:=CreationTime;
      sChecked:=boolean(Checked);
      sVerified:=boolean(Verified);
      sValidity:=TValidityLevel(KeyValidity);
      sRevoked:=boolean(KeyRevoked);
      sDisabled:=boolean(KeyDisabled);
      sExpired:=boolean(KeyExpired);
      sAxiomatic:=GetKeyPropIsAxiomatic(SigningKey);
      sStatus:=TSigStatus(Checked + Verified);
    end
    else begin
      Move(SigningKeyID, PGPSignKeyID, SizeOf(TPGPKeyID6));
      Result:=PGPGetKeyIDString(PGPSignKeyID, kPGPKeyIDString_Full, SignKeyID);
      if Result=0 then begin
	sCreaTimeStr:=UnixTimeToLocalTimeStr(PGPGetStdTimeFromPGPTime(CreationTime));
	sCreaTimeNum:=CreationTime;
	sHexID:=SignKeyID;
      end;
      sStatus:=SIGNED_NO_KEY;
      if (Result=0) and FGetKeyFromServer then begin
	KeysFound:=nil;
	Result:=KeyServerDialog(FContext, FtlsContext, FKeySetMain, SignKeyID, pPGPKeySet(KeysFound), FParentHandle);
	if (Result=0) and (KeysFound<>nil) then begin
	  try
	    if Assigned(FOnAddKeys) then FOnAddKeys(CancelAdding);
	    if not CancelAdding then begin
	      SelectAddKeysToKeyRing(FContext, FtlsContext, pPGPKeySet(KeysFound), FKeySetMain,
				     FKeyPropsList, PChar(FKeyDlgPrompt),
				     false, spgpKeyPropFlag_IDFlags,
				     FParentHandle);
	    end;
	    PGPAddJobOptions(Event^.Job,
			     PGPOKeySetRef(FContext, pPGPKeySet(KeysFound)),
			     PGPOLastOption(FContext));
	  finally
	    PGPFreeKeySet(pPGPKeySet(KeysFound));
	  end;
	end;
      end;
      Result:=0;
    end;
    sDetached:=FDetachedSigVerify;
  end;
  if FShowSigLog then ShowSigInfo;
  if Assigned(FOnGetSignature) then FOnGetSignature(SigPropsRec);
end;

function TPGPDecodeCustom.EndLexHandler(Event: pPGPEvent): PGPError;
begin
  Result:=0;
  with FRecipientsData do begin
    PGPFreeKeySet(RecipientSet);
    if KeyIDArray<>nil then GlobalFree(GlobalHandle(KeyIDArray));
  end;
  FillChar(FRecipientsData, SizeOf(TPGPEventRecipientsData), 0);
end;

end.

