{$J+,Z4}
unit KeyFuncs;

{------------------------------------------------------------------------------}
{                                                                              }
{     This unit is partly based on Steve Heller's spgpKeyUtil.pas from his     }
{         SPGP sources available from http://www.oz.net/~srheller/spgp/        }
{                                                                              }
{                Portions created by Michael in der Wiesche are                }
{                 Copyright (C) 2001 by Michael in der Wiesche                 }
{                                                                              }
{------------------------------------------------------------------------------}

interface

uses
  Windows,
  Classes,
  SysUtils,
  TimeUtils,
  KeyPropTypes,
  pgpCl,
  pgpKeys,
  pgpGroups,
  pgpUtilities,
  pgpOptionList,
  pgpErrors, pgpPubTypes, pgpBase;

type
  TKeyRings = Class
  private
    InitCount: Longint;
    RingKeyDB: pPGPKeyDB;
    RingKeySet: pPGPKeySet;
    RingContext: pPGPContext;
  public
    PubringFile: String;
    SecringFile: String;
    GroupsFile: String;
    function UpdateKeyRings: PGPError;
    function SetKeyRings(const Pubring, Secring: String): Longbool;
    function InitKeyRings(var Context: pPGPContext; var KeySetMain: pPGPKeySet): PGPError;
    procedure FreeKeyRings;
  end;

{ "public" functions }
function IsHexID(const ToCheck: String): Boolean;
function KeyRemove(KeyData: PChar): Integer;
function KeyEnable(HexKeyID: PChar): Integer;
function KeyDisable(HexKeyID: PChar): Integer;
function KeyRevoke(HexKeyID, Passphrase: PChar): Integer;
function ChangePassphrase(HexKeyID, OldPassphrase, NewPassphrase: PChar): Integer;
// properties, key & user IDs for all keys matching the specified KeyData
// if KeyData='' then all keys will be checked
// - returns number of keys found or error
function FindKeyProps(const KeyData: String; var KeyPropsList: TKeyPropsList;
		      PropertyFlags, FilterFlags: Integer;
		      KeyOrder: TPGPKeyOrdering): Integer;
{ "private" functions }
function UnixTimeToLocalTimeStr(UnixSeconds: Integer): String;
function GetSubKeyPropKeyID(SubKey: pPGPKey): String;
function GetKeyPropKeyID(Key: pPGPKey): String;
function GetKeyPropUserID(Key: pPGPKey): String;
function GetKeyPropAlg(Key: pPGPKey): TKeyAlgorithm;
function GetKeyPropVersion(Key: pPGPKey): TKeyVersion;
function GetKeyPropIsAxiomatic(Key: pPGPKey): Longbool;
function GetKeyFromKeySet(KeySet: pPGPKeySet; var Key: pPGPKey): PGPError;
function GetKeyByHexID(KeySet: pPGPKeySet; const HexID: String; var Key: pPGPKey): PGPError;
function GetKeyFilter(Context: pPGPContext; FilterFlags: Integer; var KeyFilter: pPGPFilter): PGPError;
function GetKeyFilterByAnyID(Context: pPGPContext; const KeyData: String; var KeyFilter: pPGPFilter): PGPError;
function GetKeySetByAnyIDs(Context: pPGPContext; KeySetMain: pPGPKeySet;
			   const KeyData: String; var KeySetFound: pPGPKeySet): PGPError;
function GetExclusiveKeySet(var KeySetToCheck: pPGPKeySet; KeySetMain: pPGPKeySet;
			    Context: pPGPContext; IgnoreKnownFlag: Integer): PGPError;
function GetKeySetProps(Context: pPGPContext; KeySet: pPGPKeySet;
			var KeyPropsList: TKeyPropsList;
			PropertyFlags, FilterFlags: Integer;
			KeyOrder: TPGPKeyOrdering): Integer; // - returns number of keys found or error
function GetHexIDByAnyID(Context: pPGPContext; KeySetMain: pPGPKeySet;
			 const KeyData: String; var HexID: String): Integer; // - returns number of keys found or error
function PassphraseIsValid(Context: pPGPContext; KeySetMain: pPGPKeySet; KeyData, Passphrase: PChar): Longbool;

const KeyRings: TKeyRings = nil;

implementation

uses PrefFuncs;

function IsHexID(const ToCheck: String): Boolean; assembler;
asm	// EAX=@ToCheck
  OR	EAX,EAX
  JE	@FALSE
  MOV	ECX,[EAX-04h]
  CMP	ECX,18
  JE	@START
  CMP	ECX,10
  JNE	@FALSE
  @START:
  MOV	DX,[EAX]
  AND	DX,0DFFFh
  CMP	DX,'X0'
  JNE	@FALSE
  SUB	ECX,2
  ADD	EAX,2
  @LOOP:
  DEC	ECX
  JS	@TRUE
  MOV	DL,[EAX+ECX]
  CMP	DL,'0'
  JB	@FALSE
  CMP	DL,'9'
  JBE	@LOOP
  AND	DL,0DFh
  CMP	DL,'A'
  JB	@FALSE
  CMP	DL,'F'
  JBE	@LOOP
  @FALSE:
  XOR	EAX,EAX
  RET
  @TRUE:
  MOV	EAX,1
end;

function KeyRemove(KeyData: PChar): Integer;
var
  Context	: pPGPContext;
  KeySetMain	: pPGPKeySet;
  KeySetFound	: pPGPKeySet;
  KeyCount	: PGPUInt32;
begin
  KeySetFound:=nil;
  Result:=KeyRings.InitKeyRings(Context, KeySetMain);
  if Result<>0 then Exit;
  try
    Result:=GetKeySetByAnyIDs(Context, KeySetMain, KeyData, KeySetFound);
    if Result<>0 then Exit;
    try
      Result:=PGPCountKeys(KeySetFound, KeyCount);
      if Result<>0 then Exit;
      if KeyCount<>0 then begin
	Result:=PGPRemoveKeys(KeySetFound, KeySetMain);
	if Result<>0 then Exit;
      end;
    finally
      PGPFreeKeySet(KeySetFound);
    end;
    Result:=KeyRings.UpdateKeyRings;
  finally
    KeyRings.FreeKeyRings;
  end;
end;

function KeyEnable(HexKeyID: PChar): Integer;
var
  Context	: pPGPContext;
  KeySetMain	: pPGPKeySet;
  Key		: pPGPKey;
begin
  Result:=KeyRings.InitKeyRings(Context, KeySetMain);
  if Result<>0 then Exit;
  try
    Result:=GetKeyByHexID(KeySetMain, HexKeyID, Key);
    if Result<>0 then Exit;
    Result:=PGPEnableKey(Key);
    if Result<>0 then Exit;
    Result:=KeyRings.UpdateKeyRings;
  finally
    KeyRings.FreeKeyRings;
  end;
end;

function KeyDisable(HexKeyID: PChar): Integer;
var
  Context	: pPGPContext;
  KeySetMain	: pPGPKeySet;
  Key		: pPGPKey;
begin
  Result:=KeyRings.InitKeyRings(Context, KeySetMain);
  if Result<>0 then Exit;
  try
    Result:=GetKeyByHexID(KeySetMain, HexKeyID, Key);
    if Result<>0 then Exit;
    Result:=PGPDisableKey(Key);
    if Result<>0 then Exit;
    Result:=KeyRings.UpdateKeyRings;
  finally
    KeyRings.FreeKeyRings;
  end;
end;

function KeyRevoke(HexKeyID, Passphrase: PChar): Integer;
var
  Context	: pPGPContext;
  KeySetMain	: pPGPKeySet;
  Key		: pPGPKey;
begin
  Result:=KeyRings.InitKeyRings(Context, KeySetMain);
  if Result<>0 then Exit;
  try
    Result:=GetKeyByHexID(KeySetMain, HexKeyID, Key);
    if Result<>0 then Exit;
    Result:=PGPRevokeKey(Key, PGPOPassphrase(Context, Passphrase), PGPOLastOption(Context));
    if Result<>0 then Exit;
    Result:=KeyRings.UpdateKeyRings;
  finally
    KeyRings.FreeKeyRings;
  end;
end;

function ChangePassphrase(HexKeyID, OldPassphrase, NewPassphrase: PChar): Integer;
var
  Context	: pPGPContext;
  KeySetMain	: pPGPKeySet;
  OptionList	: pPGPOptionList;
  Key		: pPGPKey;
  KeyList	: pPGPKeyList;
  KeyIter	: pPGPKeyIter;
  SubKey	: pPGPSubKey;
begin
  Result:=KeyRings.InitKeyRings(Context, KeySetMain);
  if Result<>0 then Exit;
  OptionList:=nil;
  try
    Result:=PGPBuildOptionList(Context, OptionList,
      [
	PGPOPassphrase(Context, OldPassphrase),
	PGPOPassphrase(Context, NewPassphrase)
      ]);
    if Result<>0 then Exit;
    try
      Result:=GetKeyByHexID(KeySetMain, HexKeyID, Key);
      if Result<>0 then Exit;
      Result:=PGPChangePassphrase(Key, OptionList, PGPOLastOption(Context));
      if Result<>0 then Exit;
      if (GetKeyPropAlg(Key)=KeyAlgorithm_DHDSS) or (GetKeyPropVersion(Key)=KeyVersion_V4) then begin
	KeyList:=nil;
	KeyIter:=nil;
	Result:=PGPOrderKeySet(KeySetMain, kPGPKeyOrdering_Any, PGPBoolean(false), KeyList);
	if Result<>0 then Exit;
	try
	  Result:=PGPNewKeyIter(KeyList, KeyIter);
	  if Result<>0 then Exit;
	  try
	    PGPKeyIterSeek(KeyIter, Key);
	    while PGPKeyIterNextKeyDBObj(KeyIter, kPGPKeyDBObjType_SubKey, SubKey)=0 do begin
	      Result:=PGPChangeSubKeyPassphrase(SubKey, OptionList, PGPOLastOption(Context));
	      if Result<>0 then Exit;
	    end;
	  finally
	    PGPFreeKeyIter(KeyIter);
	  end;
	finally
	  PGPFreeKeyList(KeyList);
	end;
      end;
      Result:=KeyRings.UpdateKeyRings;
    finally
      PGPFreeOptionList(OptionList);
    end;
  finally
    KeyRings.FreeKeyRings;
  end;
end;

function FindKeyProps(const KeyData: String; var KeyPropsList: TKeyPropsList;
		      PropertyFlags, FilterFlags: Integer;
		      KeyOrder: TPGPKeyOrdering): Integer;
var
  Context	: pPGPContext;
  KeySetMain	: pPGPKeySet;
  KeySetFound	: pPGPKeySet;
  KeyFilter	: pPGPFilter;
  KeySetFiltered: pPGPKeySet;
  KeyCount	: PGPUInt32;
begin
  KeySetFound:=nil;
  KeyFilter:=nil;
  KeySetFiltered:=nil;
  Result:=KeyRings.InitKeyRings(Context, KeySetMain);
  if Result<>0 then Exit;
  try
    if KeyData='' then
      KeySetFound:=KeySetMain // all keys on key rings
    else Result:=GetKeySetByAnyIDs(Context, KeySetMain, KeyData, KeySetFound); // key(s) matching KeyData
    if Result<>0 then Exit;
    try
      if PropertyFlags<>0 then begin
	Result:=GetKeySetProps(Context, KeySetFound,
			       KeyPropsList, PropertyFlags,
			       FilterFlags, KeyOrder);
      end
      else begin
	if FilterFlags<>KeyFilterFlag_AllKeys then begin
	  Result:=GetKeyFilter(Context, FilterFlags, KeyFilter);
	  try
	    if Result<>0 then Exit;
	    Result:=PGPFilterKeySet(KeySetFound, KeyFilter, KeySetFiltered);
	  finally
	    PGPFreeFilter(KeyFilter);
	  end;
	end
	else KeySetFiltered:=KeySetFound;
	if Result<>0 then Exit;
	try
	  Result:=PGPCountKeys(KeySetFiltered, KeyCount);
	  if Result=0 then Result:=KeyCount;
	finally
	  if KeySetFiltered<>KeySetFound then PGPFreeKeySet(KeySetFiltered);
	end;
      end;
    finally
      if KeySetFound<>KeySetMain then PGPFreeKeySet(KeySetFound);
    end;
  finally
    KeyRings.FreeKeyRings;
  end;
end;

function GetKeyFromKeySet(KeySet: pPGPKeySet; var Key: pPGPKey): PGPError;
var
  KeyList	: pPGPKeyList;
  KeyIter	: pPGPKeyIter;
begin
  Key:=nil;
  KeyList:=nil;
  KeyIter:=nil;
  Result:=PGPOrderKeySet(KeySet, kPGPKeyOrdering_UserID, PGPBoolean(false), KeyList);
  if Result<>0 then Exit;
  try
    Result:=PGPNewKeyIter(KeyList, KeyIter);
    if Result<>0 then Exit;
    try
      Result:=PGPKeyIterNextKeyDBObj(KeyIter, kPGPKeyDBObjType_Key, Key);
    finally
      PGPFreeKeyIter(KeyIter);
    end;
  finally
    PGPFreeKeyList(KeyList);
  end;
end;

function GetKeyByHexID(KeySet: pPGPKeySet; const HexID: String; var Key: pPGPKey): PGPError;
var
  PGPKeyID	: TPGPKeyID7;
begin
  Key:=nil;
  Result:=PGPGetKeyIDFromString(PChar(HexID), kPGPPublicKeyAlgorithm_Invalid, PGPKeyID);
  if Result=0 then Result:=PGPGetKeyByKeyID(KeySet, PGPKeyID, kPGPPublicKeyAlgorithm_Invalid, Key);
end;

function GetKeyFilter(Context: pPGPContext; FilterFlags: Integer; var KeyFilter: pPGPFilter): PGPError;
var
  KeyBoolFilter	: pPGPFilter;
  KeyAlgFilter	: pPGPFilter;
begin
  KeyFilter:=nil;
  KeyBoolFilter:=nil;
  KeyAlgFilter:=nil;
  Result:=0;
  if FilterFlags and KeyFilterMask_Boolean<>0 then begin
    if FilterFlags and KeyFilterFlag_CanEncrypt<>0 then begin
      Result:=PGPNewKeyDBObjBooleanFilter(Context, kPGPKeyProperty_CanEncrypt, PGPBoolean(true), KeyBoolFilter);
    end
    else if FilterFlags and KeyFilterFlag_CanDecrypt<>0 then begin
      Result:=PGPNewKeyDBObjBooleanFilter(Context, kPGPKeyProperty_CanDecrypt, PGPBoolean(true), KeyBoolFilter);
    end
    else if FilterFlags and KeyFilterFlag_CanSign<>0 then begin
      Result:=PGPNewKeyDBObjBooleanFilter(Context, kPGPKeyProperty_CanSign, PGPBoolean(true), KeyBoolFilter);
    end
    else if FilterFlags and KeyFilterFlag_CanVerify<>0 then begin
      Result:=PGPNewKeyDBObjBooleanFilter(Context, kPGPKeyProperty_CanVerify, PGPBoolean(true), KeyBoolFilter);
    end
    else if FilterFlags and KeyFilterFlag_Enabled<>0 then begin
      Result:=PGPNewKeyDBObjBooleanFilter(Context, kPGPKeyProperty_IsDisabled, PGPBoolean(false), KeyBoolFilter);
    end
    else if FilterFlags and KeyFilterFlag_Disabled<>0 then begin
      Result:=PGPNewKeyDBObjBooleanFilter(Context, kPGPKeyProperty_IsDisabled, PGPBoolean(true), KeyBoolFilter);
    end;
  end;
  if Result<>0 then Exit;
  if FilterFlags and KeyFilterMask_Algorithm<>0 then begin
    if FilterFlags and KeyFilterFlag_DHDSS<>0 then begin
      Result:=PGPNewKeyDBObjNumericFilter(Context, kPGPKeyProperty_AlgorithmID,
					  kPGPPublicKeyAlgorithm_DSA, kPGPMatchEqual, KeyAlgFilter);
    end else if FilterFlags and KeyFilterFlag_RSA<>0 then begin
      Result:=PGPNewKeyDBObjNumericFilter(Context, kPGPKeyProperty_AlgorithmID,
					  kPGPPublicKeyAlgorithm_RSA, kPGPMatchEqual, KeyAlgFilter);
    end else if FilterFlags and KeyFilterFlag_V4<>0 then begin
      if PGP7X then begin
	Result:=PGPNewKeyDBObjNumericFilter(Context, kPGPKeyProperty_Version,
					    kPGPKeyVersion_V4, kPGPMatchEqual, KeyAlgFilter);
      end
      else begin
	Result:=PGPNewKeyDBObjNumericFilter(Context, kPGPKeyProperty_AlgorithmID,
					    kPGPPublicKeyAlgorithm_DSA, kPGPMatchEqual, KeyAlgFilter);
      end;
    end else if FilterFlags and KeyFilterFlag_V3<>0 then begin
      if PGP7X then begin
	Result:=PGPNewKeyDBObjNumericFilter(Context, kPGPKeyProperty_Version,
					    kPGPKeyVersion_V3, kPGPMatchLessOrEqual, KeyAlgFilter);
      end
      else begin
	Result:=PGPNewKeyDBObjNumericFilter(Context, kPGPKeyProperty_AlgorithmID,
					    kPGPPublicKeyAlgorithm_RSA, kPGPMatchEqual, KeyAlgFilter);
      end;
    end;
  end;
  if Result<>0 then Exit;
  if (KeyBoolFilter<>nil) and (KeyAlgFilter<>nil) then
    Result:=PGPIntersectFilters(KeyBoolFilter, KeyAlgFilter, KeyFilter)
  else if KeyAlgFilter<>nil then
    KeyFilter:=KeyAlgFilter
  else if KeyBoolFilter<>nil then
    KeyFilter:=KeyBoolFilter;
end;

function GetKeyFilterByAnyID(Context: pPGPContext; const KeyData: String; var KeyFilter: pPGPFilter): PGPError;
var
  PGPKeyID	: TPGPKeyID7;
begin
  KeyFilter:=nil;
  if IsHexID(KeyData) then begin
    Result:=PGPGetKeyIDFromString(PChar(KeyData), kPGPPublicKeyAlgorithm_Invalid, PGPKeyID);
    if Result=0 then Result:=PGPNewKeyDBObjDataFilter(Context, kPGPKeyProperty_KeyID, @PGPKeyID,
						      SizeOf(TPGPKeyID7), kPGPMatchEqual, KeyFilter);
  end
  else Result:=PGPNewKeyDBObjDataFilter(Context, kPGPUserIDProperty_Name, PChar(KeyData),
					Length(KeyData), kPGPMatchSubString, KeyFilter);
end;

function GetKeySetByAnyIDs(Context: pPGPContext; KeySetMain: pPGPKeySet;
			   const KeyData: String; var KeySetFound: pPGPKeySet): PGPError;
var
  KeyDataList	: TStringList;
  KeyIndex	: PGPUInt32;
  KeyString	: String;
  KeyFilter	: pPGPFilter;
  KeySet	: pPGPKeySet;
  KeyCount	: PGPUInt32;
begin
  KeySetFound:=nil;
  if KeyData<>'' then begin
    KeyDataList:=TStringList.Create;
    KeyFilter:=nil;
    KeySet:=nil;
    try
      KeyDataList.CommaText:=KeyData;
      if PGP7X then
	Result:=PGPNewEmptyKeySet(PGPPeekKeySetKeyDB(KeySetMain), KeySetFound)
      else Result:=PGPNewKeySet(Context, KeySetFound);
      if Result<>0 then Exit;
      for KeyIndex:=0 to pred(KeyDataList.Count) do begin
	KeyString:=Trim(KeyDataList[KeyIndex]);
	if KeyString<>'' then begin
	  Result:=GetKeyFilterByAnyID(Context, KeyString, KeyFilter);
	  if Result<>0 then Exit;
	  try
	    Result:=PGPFilterKeySet(KeySetMain, KeyFilter, KeySet);
	    if Result<>0 then Exit;
	    try
	      Result:=PGPCountKeys(KeySet, KeyCount);
	      if Result<>0 then Exit;
	      if KeyCount<>0 then begin
		Result:=PGPAddKeys(KeySet, KeySetFound);
		if Result<>0 then Exit;
	      end
	      else begin
		Result:=kPGPError_ItemNotFound;
		PGPFreeKeySet(KeySetFound);
		KeySetFound:=nil;
		Exit;
	      end;
	    finally
	      PGPFreeKeySet(KeySet);
	    end;
	  finally
	    PGPFreeFilter(KeyFilter);
	  end;
	end;
      end;
    finally
      KeyDataList.Free;
    end;
  end
  else Result:=kPGPError_ItemNotFound;
end;

function GetExclusiveKeySet(var KeySetToCheck: pPGPKeySet; KeySetMain: pPGPKeySet;
			    Context: pPGPContext; IgnoreKnownFlag: Integer): PGPError;
var
  KeyListToCheck: pPGPKeyList;
  KeyIterToCheck: pPGPKeyIter;
  KeyToCheck	: pPGPKey;
  KeyIDToCheck	: TKeyID;
  UserIDToCheck	: TUserID;
  KeySetToRemove: pPGPKeySet;
  KeyCount	: PGPUInt32;
  IDSize	: PGPSize;
  UserIDFound	: pPGPUserID;
  UserIDProp	: PGPBoolean;
  KeyFilter	: pPGPFilter;
  KeySetFound	: pPGPKeySet;
begin
  KeyToCheck:=nil;
  KeyListToCheck:=nil;
  KeyIterToCheck:=nil;
  try
    if PGP7X then
      Result:=PGPNewKeyIterFromKeySet(KeySetToCheck, KeyIterToCheck)
    else begin
      Result:=PGPOrderKeySet(KeySetToCheck, kPGPKeyOrdering_UserID, PGPBoolean(false), KeyListToCheck);
      if Result<>0 then Exit;
      Result:=PGPNewKeyIter(KeyListToCheck, KeyIterToCheck);
    end;
    if Result<>0 then Exit;
    try
      if (IgnoreKnownFlag and IgnoreFlag_ByHexID)<>0 then begin
	while PGPKeyIterNextKeyDBObj(KeyIterToCheck, kPGPKeyDBObjType_Key, KeyToCheck)=0 do begin
	  KeyFilter:=nil;
	  KeySetFound:=nil;
	  KeySetToRemove:=nil;
	  try
	    if (PGPGetKeyDBObjDataProperty(KeyToCheck, kPGPKeyProperty_KeyID,
					   @KeyIDToCheck, kPGPMaxKeyIDStringSize, IDSize)=0)
	    and (PGPNewKeyDBObjDataFilter(Context, kPGPKeyProperty_KeyID, @KeyIDToCheck,
					  IDSize, kPGPMatchEqual, KeyFilter)=0)
	    and (PGPFilterKeySet(KeySetMain, KeyFilter, KeySetFound)=0)
	    and (PGPCountKeys(KeySetFound, KeyCount)=0) and (KeyCount<>0)
	    and (PGPNewSingletonKeySet(KeyToCheck, KeySetToRemove)=0) then begin
	      PGPRemoveKeys(KeySetToRemove, KeySetToCheck);
	    end;
	  finally
	    PGPFreeFilter(KeyFilter);
	    PGPFreeKeySet(KeySetFound);
	    PGPFreeKeySet(KeySetToRemove);
	  end;
	end;
	PGPKeyIterRewindKeyDBObj(KeyIterToCheck, kPGPKeyDBObjType_Key);
      end;
      if (PGPCountKeys(KeySetToCheck, KeyCount)=0) and (KeyCount=0) then Exit;
      if (IgnoreKnownFlag and IgnoreFlag_ByUserID)<>0 then begin
	while (PGPKeyIterNextKeyDBObj(KeyIterToCheck, kPGPKeyDBObjType_Key, KeyToCheck)=0)
	and (PGPKeyIterRewindKeyDBObj(KeyIterToCheck, kPGPKeyDBObjType_UserID)=0) do begin
	  while (PGPKeyIterNextKeyDBObj(KeyIterToCheck, kPGPKeyDBObjType_UserID, UserIDFound)=0)
	  and (PGPGetKeyDBObjBooleanProperty(UserIDFound, kPGPUserIDProperty_IsAttribute, UserIDProp)=0)
	  and not boolean(UserIDProp) do begin
	    KeyFilter:=nil;
	    KeySetFound:=nil;
	    KeySetToRemove:=nil;
	    try
	      if (PGPGetKeyDBObjDataProperty(UserIDFound, kPGPUserIDProperty_Name,
					     @UserIDToCheck, kPGPMaxUserIDSize, IDSize)=0)
	      and (PGPNewKeyDBObjDataFilter(Context, kPGPUserIDProperty_Name, @UserIDToCheck,
					    IDSize, kPGPMatchEqual, KeyFilter)=0)
	      and (PGPFilterKeySet(KeySetMain, KeyFilter, KeySetFound)=0)
	      and (PGPCountKeys(KeySetFound, KeyCount)=0) and (KeyCount<>0)
	      and (PGPNewSingletonKeySet(KeyToCheck, KeySetToRemove)=0) then begin
		PGPRemoveKeys(KeySetToRemove, KeySetToCheck);
		Break;
	      end;
	    finally
	      PGPFreeFilter(KeyFilter);
	      PGPFreeKeySet(KeySetFound);
	      PGPFreeKeySet(KeySetToRemove);
	    end;
	  end;
	  if (PGPCountKeys(KeySetToCheck, KeyCount)=0) and (KeyCount=0) then Break;
	end;
      end;
    finally
      PGPFreeKeyIter(KeyIterToCheck);
    end;
  finally
    PGPFreeKeyList(KeyListToCheck);
  end;
end;

function GetFingerprintString(const Fingerprint: TFingerprint; FPSize: PGPSize): String; assembler;
asm	// EAX=@Fingerprint, EDX=FPSize, ECX=@Result
  JMP	@BEGIN
  @HEXCHARS:
  DB	'0123456789ABCDEF';
  @BEGIN:
  PUSH	EBX
  PUSH	EDI
  PUSH	ESI
  MOV	ESI,EAX
  MOV	EDI,ECX
  MOV	EAX,EDX
  MOV	EBX,EDX
  SHL	EAX,1
  SHR	EBX,1
  DEC	EBX
  ADD	EAX,EBX
  PUSH	EDX
  PUSH	EAX
  CALL	SYSTEM.@NEWANSISTRING
  MOV	[EDI],EAX
  MOV	EDI,EAX
  POP	EBX
  POP	ECX
  SUB	EBX,2
  DEC	ECX
  @LOOP:
  MOVZX	EAX,BYTE PTR[ESI + ECX]
  MOV	EDX,EAX
  AND	EAX,0Fh
  SHR	EDX,4
  MOV	AH,BYTE PTR[@HEXCHARS + EAX]
  MOV	AL,BYTE PTR[@HEXCHARS + EDX]
  MOV	[EDI + EBX],AX
  MOVZX	EAX,BYTE PTR[ESI + ECX - 1]
  MOV	EDX,EAX
  AND	EAX,0Fh
  SHR	EDX,4
  MOV	AH,BYTE PTR[@HEXCHARS + EAX]
  MOV	AL,BYTE PTR[@HEXCHARS + EDX]
  MOV	[EDI + EBX - 2],AX
  SUB	EBX,5
  SUB	ECX,2
  JS	@END
  MOV	BYTE PTR[EDI + EBX + 2],' '
  JMP	@LOOP
  @END:
  POP	ESI
  POP	EDI
  POP	EBX
end;

function UnixTimeToLocalTimeStr(UnixSeconds: Integer): String;
const
  TIME_ZONE_ID_INVALID  =  $FFFFFFFF;
  TIME_ZONE_ID_UNKNOWN  =  0;
  TIME_ZONE_ID_STANDARD =  1;
  TIME_ZONE_ID_DAYLIGHT =  2;
var
  UnixTime, LocalTime: TDateTime;
  tzInfo: TTimeZoneInformation;
  NoTZInfo: Longbool;
begin
  Result:='';
  try
    // convert a UNIX time to Delphi's TDateTime format
    UnixTime:=EncodeDate(1970, 1, 1) + UnixSeconds / SECONDS_PER_DAY;

    // if Windows local timezone info is not available
    // then take UTC/GMT as the only available valid time format
    // else get local time using Primoz Gabrijelcic's Time Zone Routines
    NoTZInfo:=(GetTimeZoneInformation(tzInfo)=TIME_ZONE_ID_INVALID);
    if NoTZInfo then
      LocalTime:=UnixTime
    else LocalTime:=UTCToTZLocalTime(tzInfo, UnixTime);

    // Format TDateTime to the local date/time format
    if NoTZInfo then
      Result:=FormatDateTime('ddd, dd mmm yyyy, tt "<UTC>"', LocalTime)
    else Result:=FormatDateTime('ddd, dd mmm yyyy, tt', LocalTime);
  except
  end;
end;

function GetSubKeyPropKeyID(SubKey: pPGPKey): String;
var
  PGPKeyID	: TPGPKeyID7;
  KeyID		: TKeyID;
begin
  Result:='';
  if (SubKey<>nil)
  and (PGPGetKeyIDFromSubKey(SubKey, PGPKeyID)=0)
  and (PGPGetKeyIDString(PGPKeyID, kPGPKeyIDString_Full, KeyID)=0) then begin
    Result:=KeyID;
  end;
end;

function GetKeyPropKeyID(Key: pPGPKey): String;
var
  PGPKeyID	: TPGPKeyID7;
  KeyID		: TKeyID;
begin
  Result:='';
  if (Key<>nil)
  and (PGPGetKeyIDFromKey(Key, PGPKeyID)=0)
  and (PGPGetKeyIDString(PGPKeyID, kPGPKeyIDString_Full, KeyID)=0) then begin
    Result:=KeyID;
  end;
end;

function GetKeyPropUserID(Key: pPGPKey): String;
var
  UserIDBuf	: TUserID;
  IDSize	: PGPSize;
begin
  Result:='';
  if (Key<>nil)
  and (PGPGetPrimaryUserIDNameBuffer(Key, @UserIDBuf, kPGPMaxUserIDSize, IDSize)=0) then begin
    Result:=UserIDBuf;
  end;
end;

function GetKeyPropFingerprint(Key: pPGPKey): String;
var
  Fingerprint	: TFingerprint;
  FPSize	: PGPSize;
begin
  Result:='';
  if (Key<>nil)
  and (PGPGetKeyDBObjDataProperty(Key, kPGPKeyProperty_Fingerprint, @Fingerprint, kPGPmaxFingerprintSize, FPSize)=0)
  and (FPSize<>0) then begin
    Result:=GetFingerprintString(Fingerprint, FPSize);
  end;
end;

// local time
function GetKeyPropKeyCreationTimeStr(Key: pPGPKey): String;
var
  CreaTime	: PGPTime;
begin
  Result:='';
  if (Key<>nil) and (PGPGetKeyDBObjTimeProperty(Key, kPGPKeyProperty_Creation, CreaTime)=0) then begin
    Result:=UnixTimeToLocalTimeStr(PGPGetStdTimeFromPGPTime(CreaTime));
  end;
end;

// local time
function GetKeyPropKeyExpirationTimeStr(Key: pPGPKey): String;
var
  ExpTime	: PGPTime;
  CreaTime	: PGPTime;
begin
  Result:='';
  if (Key<>nil)
  and (PGPGetKeyDBObjTimeProperty(Key, kPGPKeyProperty_Expiration, ExpTime)=0)
  and (PGPGetKeyDBObjTimeProperty(Key, kPGPKeyProperty_Creation, CreaTime)=0) then begin
    if ExpTime>CreaTime then Result:=UnixTimeToLocalTimeStr(PGPGetStdTimeFromPGPTime(ExpTime));
  end;
end;

function GetKeyPropKeyBits(Key: pPGPKey): String;
var
  KeyAlg	: PGPInt32;
  KeyBits	: PGPInt32;
  Error		: PGPError;
  KeySet	: pPGPKeySet;
  KeyList	: pPGPKeyList;
  KeyIter	: pPGPKeyIter;
  SubKey	: pPGPSubKey;
  SubKeyBits	: PGPInt32;
begin
  Result:='';
  KeySet:=nil;
  KeyList:=nil;
  KeyIter:=nil;
  SubKey:=nil;
  if (Key<>nil) and (PGPGetKeyDBObjNumericProperty(Key, kPGPKeyProperty_AlgorithmID, KeyAlg)=0) then begin
    case KeyAlg of
      // no subkey(s)
      kPGPPublicKeyAlgorithm_RSA: begin
	if PGPGetKeyDBObjNumericProperty(Key, kPGPKeyProperty_Bits, KeyBits)=0 then Result:=IntToStr(KeyBits) + ' bits';
      end;
      // w or w/o subkey(s)
      kPGPPublicKeyAlgorithm_DSA : begin
	Error:=PGPGetKeyDBObjNumericProperty(Key, kPGPKeyProperty_Bits, KeyBits);
	if Error<>0 then Exit;
	Error:=PGPNewSingletonKeySet(Key, KeySet);
	if Error<>0 then Exit;
	try
	  Error:=PGPOrderKeySet(KeySet, kPGPKeyOrdering_Any, PGPBoolean(false), KeyList);
	  if Error<>0 then Exit;
	  try
	    Error:=PGPNewKeyIter(KeyList, KeyIter);
	    if Error<>0 then Exit;
	    try
	      Error:=PGPKeyIterNextKeyDBObj(KeyIter, kPGPKeyDBObjType_Key, Key);
	      if Error<>0 then Exit;
	      if PGPKeyIterNextKeyDBObj(KeyIter, kPGPKeyDBObjType_SubKey, SubKey)=0 then
		Error:=PGPGetKeyDBObjNumericProperty(SubKey, kPGPSubKeyProperty_Bits, SubKeyBits)
	      else SubKeyBits:=0;
	      if Error=0 then begin
		if SubKeyBits>0 then
		  Result:=IntToStr(SubKeyBits) + '/' + IntToStr(KeyBits) + ' bits'
		else Result:=IntToStr(KeyBits) + ' bits';
	      end;
	    finally
	      PGPFreeKeyIter(KeyIter);
	    end;
	  finally
	    PGPFreeKeyList(KeyList);
	  end;
	finally
	  PGPFreeKeySet(KeySet);
	end;
      end;
    end;
  end;
end;

function GetKeyPropAlg(Key: pPGPKey): TKeyAlgorithm;
var
  KeyAlg	: PGPInt32;
begin
  Result:=KeyAlgorithm_Invalid;
  if (Key<>nil) and (PGPGetKeyDBObjNumericProperty(Key, kPGPKeyProperty_AlgorithmID, KeyAlg)=0) then begin
    Result:=TKeyAlgorithm(KeyAlg);
    if (Result=KeyAlgorithm_DSS) and (pos('/', GetKeyPropKeyBits(Key))<>0) then begin
      inc(Result);
    end;
  end;
end;

function GetKeyPropVersion(Key: pPGPKey): TKeyVersion;
var
  KeyVersion	: PGPInt32;
begin
  Result:=KeyVersion_Unknown;
  if PGP7X and (Key<>nil) and (PGPGetKeyDBObjNumericProperty(Key, kPGPKeyProperty_Version, KeyVersion)=0) then begin
    Result:=TKeyVersion(KeyVersion);
  end;
end;

function GetKeyPropTrust(Key: pPGPKey): TTrustLevel;
var
  KeyTrust	: PGPInt32;
begin
  Result:=KeyTrust_Undefined;
  if (Key<>nil) and (PGPGetKeyDBObjNumericProperty(Key, kPGPKeyProperty_Trust, KeyTrust)=0) then begin
    Result:=TTrustLevel(KeyTrust);
  end;
end;

function GetKeyPropValidity(Key: pPGPKey): TValidityLevel;
var
  KeyValidity	: PGPInt32;
begin
  Result:=Validity_Unknown;
  if (Key<>nil) and (PGPGetKeyDBObjNumericProperty(Key, kPGPKeyProperty_Validity, KeyValidity)=0) then begin
    Result:=TValidityLevel(KeyValidity);
  end;
end;

// UTC/GMT Unix format seconds
function GetKeyPropKeyCreationTime(Key: pPGPKey): Integer;
var
  CreaTime	: PGPTime;
begin
  Result:=0;
  if (Key<>nil) and (PGPGetKeyDBObjTimeProperty(Key, kPGPKeyProperty_Creation, CreaTime)=0) then begin
    Result:=PGPGetStdTimeFromPGPTime(CreaTime);
  end;
end;

// UTC/GMT Unix format seconds
function GetKeyPropKeyExpirationTime(Key: pPGPKey): Integer;
var
  ExpTime	: PGPTime;
  CreaTime	: PGPTime;
begin
  Result:=0;
  if (Key<>nil)
  and (PGPGetKeyDBObjTimeProperty(Key, kPGPKeyProperty_Expiration, ExpTime)=0)
  and (PGPGetKeyDBObjTimeProperty(Key, kPGPKeyProperty_Creation, CreaTime)=0) then begin
    if ExpTime>CreaTime then Result:=PGPGetStdTimeFromPGPTime(ExpTime);
  end;
end;

function GetKeyPropIsSecret(Key: pPGPKey): Longbool;
var
  Prop		: PGPBoolean;
begin
  Result:=false;
  if (Key<>nil) and (PGPGetKeyDBObjBooleanProperty(Key, kPGPKeyProperty_IsSecret, Prop)=0) then begin
    Result:=boolean(Prop);
  end;
end;

function GetKeyPropIsAxiomatic(Key: pPGPKey): Longbool;
var
  Prop		: PGPBoolean;
begin
  Result:=false;
  if (Key<>nil) and (PGPGetKeyDBObjBooleanProperty(Key, kPGPKeyProperty_IsAxiomatic, Prop)=0) then begin
    Result:=boolean(Prop);
  end;
end;

function GetKeyPropIsRevoked(Key: pPGPKey): Longbool;
var
  Prop		: PGPBoolean;
begin
  Result:=false;
  if (Key<>nil) and (PGPGetKeyDBObjBooleanProperty(Key, kPGPKeyProperty_IsRevoked, Prop)=0) then begin
    Result:=boolean(Prop);
  end;
end;

function GetKeyPropIsDisabled(Key: pPGPKey): Longbool;
var
  Prop		: PGPBoolean;
begin
  Result:=false;
  if (Key<>nil) and (PGPGetKeyDBObjBooleanProperty(Key, kPGPKeyProperty_IsDisabled, Prop)=0) then begin
    Result:=boolean(Prop);
  end;
end;

function GetKeyPropIsExpired(Key: pPGPKey): Longbool;
var
  Prop		: PGPBoolean;
begin
  Result:=false;
  if (Key<>nil) and (PGPGetKeyDBObjBooleanProperty(Key, kPGPKeyProperty_IsExpired, Prop)=0) then begin
    Result:=boolean(Prop);
  end;
end;

function GetKeyPropIsSecretShared(Key: pPGPKey): Longbool;
var
  Prop		: PGPBoolean;
begin
  Result:=false;
  if (Key<>nil) and (PGPGetKeyDBObjBooleanProperty(Key, kPGPKeyProperty_IsSecretShared, Prop)=0) then begin
    Result:=boolean(Prop);
  end;
end;

function GetKeyPropCanEncrypt(Key: pPGPKey): Longbool;
var
  Prop		: PGPBoolean;
begin
  Result:=false;
  if (Key<>nil) and (PGPGetKeyDBObjBooleanProperty(Key, kPGPKeyProperty_CanEncrypt, Prop)=0) then begin
    Result:=boolean(Prop);
  end;
end;

function GetKeyPropCanDecrypt(Key: pPGPKey): Longbool;
var
  Prop		: PGPBoolean;
begin
  Result:=false;
  if (Key<>nil) and (PGPGetKeyDBObjBooleanProperty(Key, kPGPKeyProperty_CanDecrypt, Prop)=0) then begin
    Result:=boolean(Prop);
  end;
end;

function GetKeyPropCanSign(Key: pPGPKey): Longbool;
var
  Prop		: PGPBoolean;
begin
  Result:=false;
  if (Key<>nil) and (PGPGetKeyDBObjBooleanProperty(Key, kPGPKeyProperty_CanSign, Prop)=0) then begin
    Result:=boolean(Prop);
  end;
end;

function GetKeyPropCanVerify(Key: pPGPKey): Longbool;
var
  Prop		: PGPBoolean;
begin
  Result:=false;
  if (Key<>nil) and (PGPGetKeyDBObjBooleanProperty(Key, kPGPKeyProperty_CanVerify, Prop)=0) then begin
    Result:=boolean(Prop);
  end;
end;

function GetKeyPropHasRevoker(KeySet: pPGPKeySet; Key: pPGPKey; KeyPropsRec: pKeyPropsRec): Longbool;
var
  RevKeyCount	: PGPUInt32;
  RevKeyIndex	: PGPUInt32;
  RevKey	: pPGPKey;
  RevKeyID	: TPGPKeyID7;
  KeyID		: TKeyID;
begin
  Result:=false;
  if (Key<>nil) and (PGPCountRevocationKeys(Key, RevKeyCount)=0) then begin
    if RevKeyCount>0 then begin
      Result:=true;
      for RevKeyIndex:=0 to pred(RevKeyCount) do begin
	if PGPGetIndexedRevocationKey(Key, KeySet, RevKeyIndex, RevKey, RevKeyID)=0 then begin
	  if PGPGetKeyIDString(RevKeyID, kPGPKeyIDString_Full, KeyID)=0 then begin
	    KeyPropsRec^.kRevKeyIDList.Add(KeyID);
	  end;
	end;
      end;
    end;
  end;
end;

function GetKeyPropHasARR(KeySet: pPGPKeySet; Key: pPGPKey; KeyPropsRec: pKeyPropsRec): TADKType;
var
  ARKeyCount	: PGPUInt32;
  ARKeyIndex	: PGPUInt32;
  ARKey		: pPGPKey;
  ARKeyID	: TPGPKeyID7;
  ARClass	: PGPByte;
  KeyID		: TKeyID;
begin
  Result:=NoADK;
  ARKey:=nil;
  if (Key<>nil) and (PGPCountAdditionalRecipientRequests(Key, ARKeyCount)=0) then begin
    if ARKeyCount>0 then begin
      Result:=SimpleADK;
      for ARKeyIndex:=0 to pred(ARKeyCount) do begin
	if PGPGetIndexedAdditionalRecipientRequestKey(Key, KeySet, ARKeyIndex, ARKey, ARKeyID, ARClass)=0 then begin
	  if ARClass>0 then Result:=EnforcedADK;
	  if PGPGetKeyIDString(ARKeyID, kPGPKeyIDString_Full, KeyID)=0 then begin
	    KeyPropsRec^.kADKeyIDList.Add(KeyID);
	  end;
	end;
      end;
    end;
  end;
end;

function GetKeyPropHasSubKey(Key: pPGPKey): Longbool;
begin
  Result:=(pos('/', GetKeyPropKeyBits(Key))<>0);
end;

function GetKeyPropUserIDs(Key: pPGPKey; KeyPropsRec: pKeyPropsRec; IncludeSignerIDs: Longbool): Integer;
var
  KeySet	: pPGPKeySet;
  KeyList	: pPGPKeyList;
  KeyIter	: pPGPKeyIter;
  UserID	: pPGPUserID;
  UserIDProp	: PGPBoolean;
  UserIDBuf	: TUserID;
  UserIDNum	: PGPUInt32;
  IDSize	: PGPSize;
  IDBuffer	: String;
  KeySig	: pPGPSig;
  SignKey	: TPGPKeyID7;
  KeyID		: TKeyID;
begin
  KeySet:=nil;
  KeyList:=nil;
  KeyIter:=nil;
  Result:=PGPNewSingletonKeySet(Key, KeySet);
  if Result<>0 then Exit;
  try
    Result:=PGPOrderKeySet(KeySet, kPGPKeyOrdering_UserID, PGPBoolean(false), KeyList);
    if Result<>0 then Exit;
    try
      Result:=PGPNewKeyIter(KeyList, KeyIter);
      if Result<>0 then Exit;
      try
	Result:=PGPKeyIterNextKeyDBObj(KeyIter, kPGPKeyDBObjType_Key, Key);
	if Result<>0 then Exit;
	Result:=PGPKeyIterRewindKeyDBObj(KeyIter, kPGPKeyDBObjType_UserID);
	if Result<>0 then Exit;
	// add UserIDs one by one to UserIDList
	while PGPKeyIterNextKeyDBObj(KeyIter, kPGPKeyDBObjType_UserID, UserID)=0 do begin
	  Result:=PGPGetKeyDBObjBooleanProperty(UserID, kPGPUserIDProperty_IsAttribute, UserIDProp);
	  if Result<>0 then Exit;
	  // check for attributes (picture, f.e.)
	  if not boolean(UserIDProp) then begin
	    // UserID validity
	    Result:=PGPGetKeyDBObjNumericProperty(UserID, kPGPUserIDProperty_Validity, UserIDNum);
	    if Result<>0 then Exit;
	    // UserID string
	    Result:=PGPGetKeyDBObjDataProperty(UserID, kPGPUserIDProperty_Name, @UserIDBuf, kPGPMaxUserIDSize, IDSize);
	    if Result<>0 then Exit;
	    try
	      with KeyPropsRec^ do begin
		kUserIDList.AddObject(UserIDBuf, ptr(UserIDNum));
		if IncludeSignerIDs then kSignerIDList.Add('');
	      end;
	    except
	      Result:=kPGPError_OutOfMemory;
	      Exit;
	    end;
	  end;
	  IDBuffer:='';
	  if IncludeSignerIDs then begin
	    // add comma separated HexIDs of signing keys to KeyPropsRec.SignerIDList
	    while PGPKeyIterNextKeyDBObj(KeyIter, kPGPKeyDBObjType_Signature, KeySig)=0 do begin
	      Result:=PGPGetKeyIDOfCertifier(KeySig, SignKey);
	      if Result<>0 then Exit;
	      Result:=PGPGetKeyIDString(SignKey, kPGPKeyIDString_Full, KeyID);
	      if Result<>0 then Exit;
	      IDBuffer:=IDBuffer + ',' + KeyID;
	    end;
	    Delete(IDBuffer, 1, 1);
	    try
	      with KeyPropsRec^.kSignerIDList do Strings[pred(Count)]:=IDBuffer;
	    except
	      Result:=kPGPError_OutOfMemory;
	      Exit;
	    end;
	  end;
	end;
      finally
	PGPFreeKeyIter(KeyIter);
      end;
    finally
      PGPFreeKeyList(KeyList);
    end;
  finally
    PGPFreeKeySet(KeySet);
  end;
end;

function GetGroupsList(Context: pPGPContext; KeySetMain: pPGPKeySet; var GroupsList: TGroupsList): Integer;
var
  GroupFileSpec	: pPGPFileSpec;
  GroupSet	: pPGPGroupSet;
  GroupCount	: PGPUInt32;
  GroupIndex	: PGPUInt32;
  GroupID	: PGPGroupID;
  GroupInfo	: TPGPGroupInfo;
  KeyCount	: PGPUInt32;
  ItemCount	: PGPUInt32;
  GroupItemIter	: pPGPGroupItemIter;
  GroupItem	: TPGPGroupItem;
  HexIDList	: String;
  Key		: pPGPKey;
  HexID		: String;
begin
  Result:=kPGPError_OutOfMemory;
  if GroupsList<>nil then begin
    try
      GroupsList.Clear;
      GroupFileSpec:=nil;
      GroupSet:=nil;
      Result:=PGPNewFileSpecFromFullPath(Context, PChar(KeyRings.GroupsFile), GroupFileSpec);
      if Result<>0 then Exit;
      try
	Result:=PGPNewGroupSetFromFile(Context, GroupFileSpec, GroupSet);
	if Result<>0 then Exit;
	try
	  Result:=PGPCountGroupsInSet(GroupSet, GroupCount);
	  if (Result<>0) or (GroupCount=0) then Exit;
	  for GroupIndex:=0 to pred(GroupCount) do begin
	    Result:=PGPGetIndGroupID(GroupSet, GroupIndex, GroupID);
	    if Result<>0 then Continue;
	    Result:=PGPGetGroupInfo(GroupSet, GroupID, GroupInfo);
	    if Result<>0 then Continue;
	    Result:=PGPCountGroupItems(GroupSet, GroupID, PGPBoolean(true), KeyCount, ItemCount);
	    if (Result<>0) or (KeyCount=0) then Continue;
	    with GroupInfo do begin
	      if Description<>'' then
		GroupsList.Add(Description + ' <' + Name + '>' + #9)
	      else GroupsList.Add('<' + Name + '>' + #9);
	    end;
	    GroupItemIter:=nil;
	    Result:=PGPNewGroupItemIter(GroupSet, GroupID, kPGPGroupIterFlags_AllKeysRecursive, GroupItemIter);
	    if Result<>0 then Continue;
	    try
	      HexIDList:='';
	      while PGPGroupItemIterNext(GroupItemIter, GroupItem)=0 do begin
		with GroupItem.Item.Key do begin
		  if PGP7X then
		    Result:=PGPGetKeyByKeyID(KeySetMain, KeyStruct7.KeyID, kPGPPublicKeyAlgorithm_Invalid, Key)
		  else Result:=PGPGetKeyByKeyID(KeySetMain, KeyStruct6.KeyID, kPGPPublicKeyAlgorithm_Invalid, Key);
		end;
		if Result<>0 then Continue;
		HexID:=GetKeyPropKeyID(Key);
		if HexID<>'' then HexIDList:=HexIDList + ',' + HexID;
	      end;
	      Delete(HexIDList, 1, 1);
	      if HexIDList<>'' then with GroupsList do Strings[GroupIndex]:=Strings[GroupIndex] + HexIDList;
	    finally
	      PGPFreeGroupItemIter(GroupItemIter);
	    end;
	  end;
	  Result:=GroupsList.Count;
	finally
	  if GroupSet<>nil then PGPFreeGroupSet(GroupSet);
	end;
      finally
	PGPFreeFileSpec(GroupFileSpec);
      end;
    except
      Result:=0;
    end;
    if (Result<=0) and (GroupsList<>nil) then GroupsList.Clear;
  end;
end;

function GetKeyPropsData(KeySet: pPGPKeySet; KeyFound: pPGPKey; Flags: Integer;
			 KeyPropsList: TKeyPropsList; KeyIndex: Integer): Integer;
var
  KeyPropsRec	: pKeyPropsRec;
begin
  Result:=-1;
  if KeyPropsList.Objects[KeyIndex]<>nil then begin
    KeyPropsRec:=pKeyPropsRec(KeyPropsList.Objects[KeyIndex]);
    with KeyPropsRec^ do begin
      try
	// "string" properties
	KeyPropsList.Strings[KeyIndex]:=GetKeyPropKeyID(KeyFound);
	if Flags and spgpKeyPropFlag_KeyID<>0 then begin
	  kHexID:='0x' + Copy(KeyPropsList[KeyIndex], Length(KeyPropsList.Strings[KeyIndex])-7, 8);
	end;
	if Flags and spgpKeyPropFlag_UserID<>0 then kUserID:=GetKeyPropUserID(KeyFound);
	if Flags and spgpKeyPropFlag_Fingerprint<>0 then kFingerprint:=GetKeyPropFingerprint(KeyFound);
	if Flags and spgpKeyPropFlag_CreationTimeStr<>0 then kCreaTimeStr:=GetKeyPropKeyCreationTimeStr(KeyFound);
	if Flags and spgpKeyPropFlag_ExpirationTimeStr<>0 then kExpTimeStr:=GetKeyPropKeyExpirationTimeStr(KeyFound);
	// "number" properties
	if Flags and spgpKeyPropFlag_Keybits<>0 then kSize:=GetKeyPropKeyBits(KeyFound);
	if Flags and spgpKeyPropFlag_KeyAlg<>0 then kAlgorithm:=GetKeyPropAlg(KeyFound);
	if Flags and spgpKeyPropFlag_Trust<>0 then kTrust:=GetKeyPropTrust(KeyFound);
	if Flags and spgpKeyPropFlag_Validity<>0 then kValidity:=GetKeyPropValidity(KeyFound);
	if Flags and spgpKeyPropFlag_CreationTime<>0 then kCreaTimeNum:=GetKeyPropKeyCreationTime(KeyFound);
	if Flags and spgpKeyPropFlag_ExpirationTime<>0 then kExpTimeNum:=GetKeyPropKeyExpirationTime(KeyFound);
	// "boolean" properties
	if Flags and spgpKeyPropFlag_IsSecret<>0 then kPrivate:=GetKeyPropIsSecret(KeyFound);
	if Flags and spgpKeyPropFlag_IsAxiomatic<>0 then kImplicitTrust:=GetKeyPropIsAxiomatic(KeyFound);
	if Flags and spgpKeyPropFlag_IsRevoked<>0 then kRevoked:=GetKeyPropIsRevoked(KeyFound);
	if Flags and spgpKeyPropFlag_IsDisabled<>0 then kDisabled:=GetKeyPropIsDisabled(KeyFound);
	if Flags and spgpKeyPropFlag_IsExpired<>0 then kExpired:=GetKeyPropIsExpired(KeyFound);
	if Flags and spgpKeyPropFlag_IsSecretShared<>0 then kSecShared:=GetKeyPropIsSecretShared(KeyFound);
	if Flags and spgpKeyPropFlag_CanEncrypt<>0 then kCanEncrypt:=GetKeyPropCanEncrypt(KeyFound);
	if Flags and spgpKeyPropFlag_CanDecrypt<>0 then kCanDecrypt:=GetKeyPropCanDecrypt(KeyFound);
	if Flags and spgpKeyPropFlag_CanSign<>0 then kCanSign:=GetKeyPropCanSign(KeyFound);
	if Flags and spgpKeyPropFlag_CanVerify<>0 then kCanVerify:=GetKeyPropCanVerify(KeyFound);
	if Flags and spgpKeyPropFlag_HasRevoker<>0 then kHasRevoker:=GetKeyPropHasRevoker(KeySet, KeyFound, KeyPropsRec);
	if Flags and spgpKeyPropFlag_HasADK<>0 then kHasADK:=GetKeyPropHasARR(KeySet, KeyFound, KeyPropsRec);
	if Flags and spgpKeyPropFlag_HasSubKey<>0 then kHasSubKey:=GetKeyPropHasSubKey(KeyFound);
	// "list" properties
	if Flags and spgpKeyPropFlag_IncludeUserIDs<>0 then begin
	  GetKeyPropUserIDs(KeyFound, KeyPropsRec, (Flags and spgpKeyPropFlag_IncludeSignerIDs<>0));
	end;
	Result:=0;
      except
      end;
    end;
  end;
end;

function GetKeySetProps(Context: pPGPContext; KeySet: pPGPKeySet;
			var KeyPropsList: TKeyPropsList;
			PropertyFlags, FilterFlags: Integer;
			KeyOrder: TPGPKeyOrdering): Integer;
var
  KeyFilter	: pPGPFilter;
  KeySetFound	: pPGPKeySet;
  KeyCount	: PGPUInt32;
  KeyList	: pPGPKeyList;
  KeyIter	: pPGPKeyIter;
  KeyFound	: pPGPKey;
  KeyIndex	: PGPUInt32;
begin
  KeyFilter:=nil;
  KeySetFound:=nil;
  KeyList:=nil;
  KeyIter:=nil;
  if FilterFlags<>KeyFilterFlag_AllKeys then begin
    Result:=GetKeyFilter(Context, FilterFlags, KeyFilter);
    try
      if Result<>0 then Exit;
      Result:=PGPFilterKeySet(KeySet, KeyFilter, KeySetFound);
      if Result<>0 then Exit;
      KeySet:=KeySetFound;
    finally
      PGPFreeFilter(KeyFilter);
    end;
  end
  else Result:=0;
  try
    if Result<>0 then Exit;
    Result:=PGPCountKeys(KeySet, KeyCount);
    if Result<>0 then Exit;
    if PropertyFlags<>0 then begin
      if KeyPropsList=nil then begin
	KeyPropsList:=TKeyPropsList.Create(KeyCount, PropertyFlags);
	KeyIndex:=0;
      end
      else KeyIndex:=KeyPropsList.Append(KeyCount);
      if (KeyPropsList=nil) or (KeyIndex<0) then begin
	Result:=kPGPError_OutOfMemory;
	Exit;
      end;
      if (KeyIndex=0) and ((PropertyFlags and spgpKeyPropFlag_IncludeGroupsList)<>0) then begin
	GetGroupsList(Context, KeySet, KeyPropsList.GroupsList);
      end;
      Result:=PGPOrderKeySet(KeySet, PGPKeyOrdering(succ(ord(KeyOrder) shr 1)), PGPBoolean(odd(ord(KeyOrder))), KeyList);
      if Result<>0 then Exit;
      try
	Result:=PGPNewKeyIter(KeyList, KeyIter);
	try
	  if Result<>0 then Exit;
	  while PGPKeyIterNextKeyDBObj(KeyIter, kPGPKeyDBObjType_Key, KeyFound)=0 do begin
	    Result:=GetKeyPropsData(KeySet, KeyFound, PropertyFlags, KeyPropsList, KeyIndex);
	    if Result=0 then
	      inc(KeyIndex)
	    else Exit;
	  end;
	  Result:=KeyPropsList.Count;
	finally
	  PGPFreeKeyIter(KeyIter);
	end;
      finally
	PGPFreeKeyList(KeyList);
      end;
    end
    else Result:=KeyCount;
  finally
    PGPFreeKeySet(KeySetFound);
  end;
end;

function GetHexIDByAnyID(Context: pPGPContext; KeySetMain: pPGPKeySet;
			 const KeyData: String; var HexID: String): Integer;
var
  KeySetFound	: pPGPKeySet;
  KeyCount	: PGPUInt32;
  PGPKey	: pPGPKey;
begin
  HexID:='';
  Result:=0;
  KeySetFound:=nil;
  if KeyData<>'' then begin
    try
      Result:=GetKeySetByAnyIDs(Context, KeySetMain, KeyData, KeySetFound);
      if Result<>0 then Exit;
      Result:=PGPCountKeys(KeySetFound, KeyCount);
      if Result<>0 then Exit;
      if KeyCount>0 then begin
	Result:=GetKeyFromKeySet(KeySetFound, PGPKey);
	if Result<>0 then Exit;
	HexID:=GetKeyPropKeyID(PGPKey);
      end;
      Result:=KeyCount;
    finally
      PGPFreeKeySet(KeySetFound);
    end;
  end;
end;

function PassphraseIsValid(Context: pPGPContext; KeySetMain: pPGPKeySet; KeyData, Passphrase: PChar): Longbool;
var
  KeySetFound	: pPGPKeySet;
  KeyCount	: PGPUInt32;
  Key		: pPGPKey;
begin
  Result:=false;
  KeySetFound:=nil;
  if GetKeySetByAnyIDs(Context, KeySetMain, KeyData, KeySetFound)=0 then begin
    try
      Result:=(PGPCountKeys(KeySetFound, KeyCount)=0) and (KeyCount=1) and (GetKeyFromKeySet(KeySetFound, Key)=0) and
	      (PGPPassphraseIsValid(Key, PGPOPassphrase(Context, Passphrase), PGPOLastOption(Context))<>PGPBoolean(false));
    finally
      PGPFreeKeySet(KeySetFound);
    end;
  end;
end;

function TKeyRings.UpdateKeyRings: PGPError;
begin
  Result:=kPGPError_NoErr;
  if PGP7X then
    Result:=PGPFlushKeyDB(RingKeyDB)
  else if boolean(PGPKeySetNeedsCommit(RingKeySet)) then Result:=PGPCommitKeyringChanges(RingKeySet);
  if Result=0 then PGPclNotifyKeyringChanges(GetCurrentProcessID);
end;

function TKeyRings.SetKeyRings(const Pubring, Secring: String): Longbool;
begin
  PubringFile:=Pubring;
  SecringFile:=Secring;
  Result:=((PubringFile<>'') and (SecringFile<>''));
end;

function TKeyRings.InitKeyRings(var Context: pPGPContext; var KeySetMain: pPGPKeySet): PGPError;
var
  Prefs: TPreferenceRec;
  PubFileSpec: pPGPFileSpec;
  SecFileSpec: pPGPFileSpec;
begin
  Result:=kPGPError_ImproperInitialization;
  if PGPInitErrorCode=ieNone then begin
    if InitCount<=0 then begin
      InitCount:=0;
      RingKeyDB:=nil;
      RingKeySet:=nil;
      RingContext:=nil;
      PubFileSpec:=nil;
      SecFileSpec:=nil;
      Result:=PGPNewContext(kPGPsdkAPIVersion, RingContext);
      if Result<>0 then Exit;
      if (PubRingFile='') or (SecringFile='') then begin
	GetPreferences(Prefs, PrefsFlag_PublicKeyring or PrefsFlag_PrivateKeyring or PrefsFlag_GroupsFile);
	KeyRings.SetKeyRings(Prefs.PublicKeyring, Prefs.PrivateKeyring);
	if GroupsFile='' then GroupsFile:=Prefs.GroupsFile;
      end
      else if (GroupsFile='') and (GetPreferences(Prefs, PrefsFlag_GroupsFile)=0) then GroupsFile:=Prefs.GroupsFile;
      try
	Result:=PGPNewFileSpecFromFullPath(RingContext, PChar(PubringFile), PubFileSpec);
	if Result<>0 then Exit;
	Result:=PGPNewFileSpecFromFullPath(RingContext, PChar(SecringFile), SecFileSpec);
	if Result<>0 then Exit;
	if PGP7X then begin
	  Result:=PGPOpenKeyDBFile(RingContext, kPGPOpenKeyDBFileOptions_Mutable, PubFileSpec, SecFileSpec, RingKeyDB);
	  if Result=0 then RingKeySet:=PGPPeekKeyDBRootKeySet(RingKeyDB);
	end
	else Result:=PGPOpenKeyRingPair(RingContext, kPGPKeyRingOpenFlags_Mutable, PubFileSpec, SecFileSpec, RingKeySet);
      finally
	PGPFreeFileSpec(PubFileSpec);
	PGPFreeFileSpec(SecFileSpec);
      end;
      if Result<>0 then begin
	PGPFreeContext(RingContext);
	RingContext:=nil;
	if PGP7X then begin
	  PGPFreeKeyDB(RingKeyDB);
	  RingKeyDB:=nil;
	end
	else begin
	  PGPFreeKeySet(RingKeySet);
	  RingKeySet:=nil;
	end;
      end;
    end
    else Result:=0;
    if Result=0 then inc(InitCount);
    KeySetMain:=RingKeySet;
    Context:=RingContext;
  end;
end;

procedure TKeyRings.FreeKeyRings;
begin
  try
    if InitCount=1 then begin
      try
	if PGP7X then
	  PGPFreeKeyDB(RingKeyDB)
	else PGPFreeKeySet(RingKeySet);
      finally
	RingKeySet:=nil;
	RingKeyDB:=nil;
	try
	  PGPFreeContext(RingContext);
	finally
	  RingContext:=nil;
	end;
      end;
    end;
  finally
    if InitCount>0 then dec(InitCount);
  end;
end;

initialization
  KeyRings:=TKeyRings.Create;

finalization
  KeyRings.Free;
  KeyRings:=nil;

end.

