{$J+,Z4}
unit PrefFuncs;

{------------------------------------------------------------------------------}
{                                                                              }
{       This unit is based on Steve Heller's spgpPreferences.pas from his      }
{         SPGP sources available from http://www.oz.net/~srheller/spgp/        }
{                                                                              }
{                Portions created by Michael in der Wiesche are                }
{                 Copyright (C) 2001 by Michael in der Wiesche                 }
{                                                                              }
{------------------------------------------------------------------------------}

interface

uses
  Windows,
  Classes,
  SysUtils,
  KeyPropTypes,
  KeyFuncs,
  pgpKeys,
  pgpCL,
  pgpSDKPrefs,
  pgpKeyServer,
  pgpMemoryMgr,
  pgpUtilities,
  pgpErrors, pgpPubTypes, pgpBase;

const
  E = '';
  BS = '\';
  CM = ',';
  QU = '"';
  SP = ' ';
  CRLF = #13#10;
  AppDataRegPath = 'Software\Microsoft\Windows\CurrentVersion\Explorer\Shell Folders';
  NAIAppDataPath = '\Network Associates\PGP\';
  PrefsFileName = 'PGPprefs.txt';
  PubFile = 'PublicKeyringFile';
  SecFile = 'PrivateKeyringFile';
  RNGFile = 'RNGSeedFile';
  DefKeyID ='DefaultKeyID';
  KeyServers = 'KeyServerList';
  WipeWarning = 'WarnOnWipe';

  PrefsFlag_Default		= $00;
  PrefsFlag_PublicKeyring	= $01;
  PrefsFlag_PrivateKeyring	= $02;
  PrefsFlag_RandomSeedFile	= $04;
  PrefsFlag_DefaultKeyID	= $08;
  PrefsFlag_GroupsFile		= $10;

type
  pPreferenceRec = ^TPreferenceRec;
  TPreferenceRec = Record
    PublicKeyring: String;
    PrivateKeyring: String;
    RandomSeedFile: String;
    GroupsFile: String;
    DefaultKeyHexID: String;
  end;

type
  TPrefsFile = Class
    PrefsData: TStringList;
    PrefsFilePath: String;
    constructor Create;
    destructor Destroy; override;
    function LoadPrefs: Boolean;
    function SavePrefs: Boolean;
    function GetShortHexID(const LongHexID: String): String;
    function GetAppPathRegEntry(const RegPath, RegKey: String): String;
  end;

var
  PrefsFile: TPrefsFile;

function GetPreferences(var Prefs: TPreferenceRec; Flags: Longint): Longint;
function SetPreferences(const Prefs: TPreferenceRec; Flags: Longint): Longint;
function GetServerList(var ServerList: TStringList): Longint;
function GetPrefWarnOnWipe(var WarnOnWipe: Longbool): Longint;
function SetPrefWarnOnWipe(WarnOnWipe: Longbool): Longint;

implementation

function ShortPos(Find: Char; const Dest: String; Pos: Longint): Longint; assembler;
asm	// EAX=Find, EDX=@Dest, ECX=Pos
  PUSH	EDI
  OR	Dest,Dest
  JZ	@NOT
  OR	Pos,Pos
  JLE	@NOT
  MOV	EDI,Dest
  MOV	EDX,[EDI-04h]
  ADD	EDI,Pos
  SUB	ECX,EDX
  JG	@NOT
  NEG	ECX
  DEC	EDI
  INC	ECX
  @LOOP:
  CMP	AL,[EDI]
  JZ	@END
  INC	EDI
  DEC	ECX
  JNZ	@LOOP
  JMP	@NOT
  @END:
  SUB	EDX,ECX
  MOV	EAX,EDX
  POP	EDI
  INC	EAX
  RET
  @NOT:
  POP	EDI
  XOR	EAX,EAX
end;

function ShortStrPos(const Find, Dest: String; Pos: Longint): Longint; assembler;
asm	// EAX=@Find, EDX=@Dest, ECX=Pos
  OR	Find,Find
  JZ	@NOT
  OR	Dest,Dest
  JZ	@NOT
  OR	Pos,Pos
  JLE	@NOT
  PUSH	EDI
  PUSH	ESI
  PUSH	EBP
  PUSH	EBX
  MOV	ESI,Find
  MOV	EDI,Dest
  MOV	EBP,[ESI-04h]
  CMP	EBP,1
  JZ	@CHAR
  MOV	EAX,[EDI-04h]
  ADD	EBP,Pos
  PUSH	EAX
  INC	EAX
  CMP	EAX,EBP
  JS	@BREAK
  DEC	EDI
  SUB	EBP,Pos
  ADD	EDI,Pos
  MOV	DL,[ESI]
  SUB	EAX,Pos
  INC	ESI
  DEC	EBP
  @SCASB:
  CMP	DL,[EDI]
  JZ	@MATCH
  INC	EDI
  DEC	EAX
  JG	@SCASB
  JMP	@BREAK
  @MATCH:
  INC	EDI
  DEC	EAX
  MOV	ECX,EBP
  @CMPSB:
  DEC	ECX
  JS	@FOUND
  MOV	BL,[ESI+ECX]
  CMP	BL,[EDI+ECX]
  JZ	@CMPSB
  JMP	@SCASB
  @CHAR:
  XOR	EAX,EAX
  MOV	AL,[ESI]
  CALL	ShortPos
  JMP	@EXIT
  @BREAK:
  POP	ECX
  XOR	EAX,EAX
  JMP	@EXIT
  @FOUND:
  POP	ECX
  NEG	EAX
  ADD	EAX,ECX
  @EXIT:
  POP	EBX
  POP	EBP
  POP	ESI
  POP	EDI
  RET
  @NOT:
  XOR	EAX,EAX
end;

function ExtractStr(const Str: String): String;
var iPos: Longint;
begin
  Result:=Str;
  iPos:=Length(Result);
  if iPos>1 then begin
    if (Result[iPos]=BS) and (Result[pred(iPos)]=SP) then begin
      Delete(Result, pred(iPos), 2);
      dec(iPos, 2);
    end;
    if (iPos<>0) and (Result[1]=QU) and (Result[iPos]=QU) then begin
      Delete(Result, iPos, 1);
      Delete(Result, 1, 1);
    end;
  end;
end;

constructor TPrefsFile.Create;
begin
  inherited;
  PrefsData:=TStringList.Create;
  PrefsFilePath:=GetAppPathRegEntry(AppDataRegPath, 'AppData') + NAIAppDataPath + PrefsFileName;
end;

destructor TPrefsFile.Destroy;
begin
  if PrefsData<>nil then PrefsData.Free;
  inherited;
end;

function TPrefsFile.LoadPrefs: Boolean;
var Buffer: String; iIndex: Longint;
begin
  Result:=false;
  if (PrefsData<>nil) and (PrefsFilePath<>E) then with PrefsData do begin
    try
      LoadFromFile(PrefsFilePath);
      Buffer:=Text;
      iIndex:=1;
      repeat
	iIndex:=ShortStrPos(SP + BS + CRLF, Buffer, iIndex);
	if iIndex>1 then begin
	  if (Buffer[pred(iIndex)]=QU) then
	    System.Delete(Buffer, pred(iIndex), Length(QU + SP + BS + CRLF + QU))
	  else System.Delete(Buffer, iIndex, Length(SP + BS + CRLF));
	end;
      until iIndex=0;
      Text:=Buffer;
      Result:=(Text<>E);
    except
    end;
  end;
end;

function TPrefsFile.SavePrefs: Boolean;
begin
  Result:=false;
  if (PrefsData<>nil) and (PrefsFilePath<>E) then with PrefsData do begin
    try
      SaveToFile(PrefsFilePath);
      Result:=true;
    except
    end;
  end;
end;

function TPrefsFile.GetShortHexID(const LongHexID: String): String;
var iPos: Longint;
begin
  Result:=LongHexID;
  iPos:=Length(Result);
  if iPos<>0 then begin
    while Result[iPos]='0' do dec(iPos);
    if (iPos>10) and (Result[1]='0') and ((ord(Result[2]) and $DF)=ord('X')) then begin
      Delete(Result, succ(iPos), Length(Result));
      Delete(Result, 3, iPos-10);
    end;
  end;
end;

function TPrefsFile.GetAppPathRegEntry(const RegPath, RegKey: String): String;
var hFilePath: hKey; dwBufSize, dwKeyType: DWord; aFilePath: Array[0..MAX_PATH] of Char;
begin
  Result:=E;
  dwBufSize:=MAX_PATH;
  if (RegOpenKeyEx(HKEY_CURRENT_USER, PChar(RegPath), 0, KEY_QUERY_VALUE, hFilePath)=ERROR_SUCCESS) then begin
    if (RegQueryValueEx(hFilePath, PChar(RegKey), nil, @dwKeyType, @aFilePath, @dwBufSize)=ERROR_SUCCESS) then begin
      Result:=aFilePath;
    end;
    RegCloseKey(hFilePath);
  end;
end;

function GetPreferences(var Prefs: TPreferenceRec; Flags: Longint): Longint;
var
  Context	: pPGPContext;
  FileSpec	: pPGPFileSpec;
  FileOut	: PChar;
  KeyOut	: PChar;
  PGPKeyID	: TPGPKeyID7;
  IDSize	: PGPSize;
  KeyID		: TKeyID;
begin
  FillChar(Prefs, SizeOf(Prefs), 0);
  if PGP7X then begin
    Result:=kPGPError_PrefNotFound;
    if PrefsFile<>nil then with PrefsFile do if LoadPrefs then begin
      try
	if Flags and PrefsFlag_PublicKeyring<>0 then Prefs.PublicKeyring:=ExtractStr(PrefsData.Values[PubFile]);
	if Flags and PrefsFlag_PrivateKeyring<>0 then Prefs.PrivateKeyring:=ExtractStr(PrefsData.Values[SecFile]);
	if Flags and PrefsFlag_RandomSeedFile<>0 then Prefs.RandomSeedFile:=ExtractStr(PrefsData.Values[RNGFile]);
	// doesn't have a groups file entry
	if Flags and PrefsFlag_DefaultKeyID<>0 then Prefs.DefaultKeyHexID:=GetShortHexID(PrefsData.Values[DefKeyID]);
	Result:=0;
      except
      end;
      PrefsData.Clear;
    end;
  end
  else begin
    Context := nil;
    FileSpec:=nil;
    Result:=PGPNewContext(kPGPsdkAPIVersion, Context);
    if Result<>0 then Exit;
    try
      Result:=PGPsdkLoadDefaultPrefs(Context);
      if Result<>0 then Exit;

      // Pubring
      if Flags and PrefsFlag_PublicKeyring<>0 then
      begin
	Result:=PGPsdkPrefGetFileSpec(Context, kPGPsdkPref_PublicKeyring, FileSpec);
	try
	  if Result=0 then begin
	    FileOut:=nil;
	    Result:=PGPGetFullPathFromFileSpec(FileSpec, FileOut);
	    try
	      Prefs.PublicKeyring:=FileOut;
	    finally
	      PGPFreeData(FileOut);
	    end;
	  end;
	finally
	  PGPFreeFileSpec(FileSpec);
	end;
      end;
      if Result<>0 then Exit;

      // Secring
      if Flags and PrefsFlag_PrivateKeyring<>0 then
      begin
	Result:=PGPsdkPrefGetFileSpec(Context, kPGPsdkPref_PrivateKeyring, FileSpec);
	try
	  if Result=0 then begin
	    FileOut:=nil;
	    Result:=PGPGetFullPathFromFileSpec(FileSpec, FileOut);
	    try
	      Prefs.PrivateKeyring:=FileOut;
	    finally
	      PGPFreeData(FileOut);
	    end;
	  end;
	finally
	  PGPFreeFileSpec(FileSpec);
	end;
      end;
      if Result<>0 then Exit;

      // Randseed file
      if Flags and PrefsFlag_RandomSeedFile<>0 then
      begin
	Result:=PGPsdkPrefGetFileSpec(Context, kPGPsdkPref_RandomSeedFile, FileSpec);
	try
	  if Result=0 then begin
	    FileOut:=nil;
	    Result:=PGPGetFullPathFromFileSpec(FileSpec, FileOut);
	    try
	      Prefs.RandomSeedFile:=FileOut;
	    finally
	      PGPFreeData(FileOut);
	    end;
	  end;
	finally
	  PGPFreeFileSpec(FileSpec);
	end;
      end;
      if Result<>0 then Exit;

      // Groups file
      if Flags and PrefsFlag_GroupsFile<>0 then
      begin
	Result:=PGPsdkPrefGetFileSpec(Context, kPGPsdkPref_GroupsFile, FileSpec);
	try
	  if Result=0 then begin
	    FileOut:=nil;
	    Result:=PGPGetFullPathFromFileSpec(FileSpec, FileOut);
	    try
	      Prefs.GroupsFile:=FileOut;
	    finally
	      PGPFreeData(FileOut);
	    end;
	  end;
	finally
	  PGPFreeFileSpec(FileSpec);
	end;
      end;
      if Result<>0 then Exit;

      // Private key
      if Flags and PrefsFlag_DefaultKeyID<>0 then
      begin
	KeyOut:=nil;
	Result:=PGPsdkPrefGetData(Context, kPGPsdkPref_DefaultKeyID, @KeyOut, IDSize);
	if Result<>0 then Exit;
	try
	  Result:=PGPImportKeyID(KeyOut, PGPKeyID);
	finally
	  PGPFreeData(KeyOut);
	end;
	if Result<>0 then Exit;
	Result:=PGPGetKeyIDString(PGPKeyID, kPGPKeyIDString_Abbreviated, KeyID);
	if Result=0 then Prefs.DefaultKeyHexID:=KeyID;
      end;
    finally
      PGPFreeContext(Context);
    end;
  end;
end;

function SetPreferences(const Prefs: TPreferenceRec; Flags: Longint): Longint;
var
  Context	: pPGPContext;
  FileSpec	: pPGPFileSpec;
  KeyPropsList	: TKeyPropsList;
  MainKeySet	: pPGPKeySet;
  Key		: pPGPKey;
  KeyIn		: TPGPKeyID7;
  PGPKeyID	: TPGPKeyID7;
  IDSize	: PGPSize;
begin
  if PGP7X then begin
    Result:=kPGPError_PrefNotFound;
    if PrefsFile<>nil then with PrefsFile do if LoadPrefs then begin
      try
	if Flags and PrefsFlag_PublicKeyring<>0 then PrefsData.Values[PubFile]:=QU + Prefs.PublicKeyring + QU;
	if Flags and PrefsFlag_PrivateKeyring<>0 then PrefsData.Values[SecFile]:=QU + Prefs.PrivateKeyring + QU;
	if Flags and PrefsFlag_RandomSeedFile<>0 then PrefsData.Values[RNGFile]:=QU + Prefs.RandomSeedFile + QU;
	// doesn't have a groups file entry
	if Flags and PrefsFlag_DefaultKeyID<>0 then PrefsData.Values[DefKeyID]:=Prefs.DefaultKeyHexID;
	if SavePrefs then Result:=0;
      except
      end;
      PrefsData.Clear;
    end;
  end
  else begin
    Context:=nil;
    FileSpec:=nil;
    Result:=PGPNewContext(kPGPsdkAPIVersion, Context);
    if Result<>0 then Exit;
    try
      Result:=PGPsdkLoadDefaultPrefs(Context);
      if Result<>0 then Exit;

      // Pubring
      if Flags and PrefsFlag_PublicKeyring<>0 then begin
	if not FileExists(Prefs.PublicKeyring) then begin
	  Result:=kPGPError_FileNotFound;
	  Exit;
	end;
	Result:=PGPNewFileSpecFromFullPath(Context, PChar(Prefs.PublicKeyring), FileSpec);
	try
	  if Result=0 then Result:=PGPsdkPrefSetFileSpec(Context, kPGPsdkPref_PublicKeyring, FileSpec);
	  if Result<>0 then Exit;
	finally
	  PGPFreeFileSpec(FileSpec);
	end;
      end;

      // Secring
      if Flags and PrefsFlag_PrivateKeyring<>0 then begin
	if not FileExists(Prefs.PrivateKeyring) then begin
	  Result:=kPGPError_FileNotFound;
	  Exit;
	end;
	Result:=PGPNewFileSpecFromFullPath(Context, PChar(Prefs.PrivateKeyring), FileSpec);
	try
	  if Result=0 then Result:=PGPsdkPrefSetFileSpec(Context, kPGPsdkPref_PrivateKeyring, FileSpec);
	  if Result<>0 then Exit;
	finally
	  PGPFreeFileSpec(FileSpec);
	end;
      end;

      // Randseed file
      if Flags and PrefsFlag_RandomSeedFile<>0 then begin
	if not FileExists(Prefs.RandomSeedFile) then begin
	  Result:=kPGPError_FileNotFound;
	  Exit;
	end;
	Result:=PGPNewFileSpecFromFullPath(Context, PChar(Prefs.RandomSeedFile), FileSpec);
	try
	  if Result=0 then Result:=PGPsdkPrefSetFileSpec(Context, kPGPsdkPref_RandomSeedFile, FileSpec);
	  if Result<>0 then Exit;
	finally
	  PGPFreeFileSpec(FileSpec);
	end;
      end;

      // Groups file
      if Flags and PrefsFlag_GroupsFile<>0 then begin
	if not FileExists(Prefs.GroupsFile) then begin
	  Result:=kPGPError_FileNotFound;
	  Exit;
	end;
	Result:=PGPNewFileSpecFromFullPath(Context, PChar(Prefs.GroupsFile), FileSpec);
	try
	  if Result=0 then Result:=PGPsdkPrefSetFileSpec(Context, kPGPsdkPref_GroupsFile, FileSpec);
	  if Result<>0 then Exit;
	finally
	  PGPFreeFileSpec(FileSpec);
	end;
      end;

      // Private key
      if Flags and PrefsFlag_DefaultKeyID<>0 then begin
	KeyPropsList:=nil;
	if FindKeyProps(Prefs.DefaultKeyHexID, KeyPropsList, 0, KeyFilterFlag_CanSign, Any_Ordering)=1 then begin
	  Result:=PGPOpenDefaultKeyRings(Context, 0, MainKeySet);
	  if Result<>0 then Exit;
	  Result:=GetKeyByHexID(MainKeySet, Prefs.DefaultKeyHexID, Key);
	  if Result<>0 then Exit;
	  Result:=PGPGetKeyIDFromString(PChar(Prefs.DefaultKeyHexID), kPGPPublicKeyAlgorithm_Invalid, KeyIn);
	  if Result<>0 then Exit;
	  Result:=PGPExportKeyID(@KeyIn, PGPKeyID, IDSize);
	  if Result<>0 then Exit;
	  Result:=PGPsdkPrefSetData(Context, kPGPsdkPref_DefaultKeyID, @PGPKeyID, IDSize);
	  if Result<>0 then Exit;
	end
	else Result:=kPGPError_SecretKeyNotFound;
	if Result<>0 then Exit;
      end;
      Result:=PGPsdkSavePrefs(Context);
    finally
      PGPFreeContext(Context);
    end;
  end;
  if Result=0 then begin
    PGPclNotifyKeyringChanges(GetCurrentProcessID);
    PGPclNotifyPrefsChanges(GetCurrentProcessID);
  end;
end;

function GetServerList(var ServerList: TStringList): Longint;
const
  Protocols: Array[kPGPKeyServerProtocol_Invalid..kPGPKeyServerProtocol_HTTPS] of String = (
    E, 'ldap', 'http', 'ldaps', 'https'
  );
var
  Entry		 : TStringList;
  PGPPref	 : pPGPPref;
  KeyServerList	 : pPGPKeyServerEntry;
  KeyServerEntry : pPGPKeyServerEntry;
  ServerCount	 : PGPUInt32;
  ServerURL	 : String;
begin
  if PGP7X then begin
    Result:=kPGPError_PrefNotFound;
    if PrefsFile<>nil then with PrefsFile do if LoadPrefs then begin
      try
	Entry:=TStringList.Create;
	ServerList:=TStringList.Create;
	if (Entry<>nil) and (ServerList<>nil) then with Entry do begin
	  try
	    ServerCount:=0;
	    repeat
	      inc(ServerCount);
	      CommaText:=PrefsData.Values[KeyServers + IntToStr(ServerCount)];
	      if (Text<>E) and (Count=8) then begin
		if Strings[0]<>E then ServerURL:=Protocols[StrToInt(Strings[0])];
		if (ServerURL<>E) and (Strings[3]<>E) then begin
		  ServerURL:=ServerURL + '://' + Strings[3];
		  if (Strings[4]<>E) and (Strings[4]<>'0') then ServerURL:=ServerURL + ':' + Strings[4];
		  ServerList.Add(ServerURL);
		end;
	      end;
	    until CommaText=E;
	  finally
	    Entry.Free;
	  end;
	  Result:=0;
	end;
      except
      end;
      PrefsData.Clear;
    end;
  end
  else begin
    PGPPref:=nil;
    KeyServerList:=nil;
    ServerList:=TStringList.Create;
    if ServerList<>nil then begin
      Result:=PGPclOpenClientPrefs(PGPGetDefaultMemoryMgr, PGPPref);
      if Result=0 then begin
	try
	  Result:=PGPGetKeyServerPrefs(PGPPref, KeyServerList, ServerCount);
	  if Result=0 then begin
	    try
	      if ServerCount<>0 then begin
		KeyServerEntry:=KeyServerList;
		for ServerCount:=ServerCount downto 1 do with KeyServerEntry^ do begin
		  if Protocol<>kPGPKeyServerProtocol_Invalid then ServerURL:=Protocols[Protocol] + '://';
		  if ServerDNS<>E then begin
		    ServerURL:=ServerURL + ServerDNS;
		    if ServerPort<>0 then ServerURL:=ServerURL + ':' + IntToStr(ServerPort);
		    ServerList.Add(ServerURL);
		  end;
		  inc(KeyServerEntry);
		end;
	      end
	      else Result:=kPGPError_PrefNotFound;
	    finally
	      PGPFreeData(KeyServerList);
	    end;
	  end;
	finally
	  PGPclCloseClientPrefs(PGPPref, false);
	end;
      end;
    end
    else Result:=kPGPError_OutOfMemory;
  end;
end;

function GetPrefWarnOnWipe(var WarnOnWipe: Longbool): Longint;
var
  PGPPref	 : pPGPPref;
  Pref           : PGPBoolean;
begin
  WarnOnWipe:=false;
  if PGP7X then begin
    Result:=kPGPError_PrefNotFound;
    if PrefsFile<>nil then with PrefsFile do if LoadPrefs then begin
      try
	WarnOnWipe:=(CompareText(PrefsData.Values[WipeWarning], 'TRUE')=0);
	Result:=0;
      except
      end;
      PrefsData.Clear;
    end;
  end
  else begin
    PGPPref:=nil;
    Result:=PGPclOpenClientPrefs(PGPGetDefaultMemoryMgr, PGPPref);
    if Result=0 then begin
      try
	Result:=PGPclGetPrefBoolean(PGPPref, kPGPPrefWarnOnWipe, Pref);
	if Result=0 then WarnOnWipe:=Boolean(Pref);
      finally
	PGPclCloseClientPrefs(PGPPref, false);
      end;
    end;
  end;
end;

function SetPrefWarnOnWipe(WarnOnWipe: Longbool): Longint;
var
  PGPPref	 : pPGPPref;
begin
  if PGP7X then begin
    Result:=kPGPError_PrefNotFound;
    if PrefsFile<>nil then with PrefsFile do if LoadPrefs then begin
      try
	if WarnOnWipe then
	  PrefsData.Values[WipeWarning]:='TRUE'
	else PrefsData.Values[WipeWarning]:='FALSE';
	if SavePrefs then Result:=0;
      except
      end;
      PrefsData.Clear;
    end;
  end
  else begin
    PGPPref:=nil;
    Result:=PGPclOpenClientPrefs(PGPGetDefaultMemoryMgr, PGPPref);
    if Result=0 then begin
      try
	Result:=PGPclSetPrefBoolean(PGPPref, kPGPPrefWarnOnWipe, PGPBoolean(WarnOnWipe) and 1);
      finally
	PGPclCloseClientPrefs(PGPPref, true);
      end;
    end;
  end;
  if Result=0 then PGPclNotifyPrefsChanges(GetCurrentProcessID);
end;

initialization
  PrefsFile:=TPrefsFile.Create;

finalization
  if PrefsFile<>nil then PrefsFile.Free;

end.

