{$J+,Z4}
unit PGPDialogs;

{------------------------------------------------------------------------------}
{                                                                              }
{                 This unit is partly based on Steve Heller's                  }
{         SPGP sources available from http://www.oz.net/~srheller/spgp/        }
{                                                                              }
{                Portions created by Michael in der Wiesche are                }
{                 Copyright (C) 2001 by Michael in der Wiesche                 }
{                                                                              }
{------------------------------------------------------------------------------}

interface

uses
  Windows,
  Classes,
  SysUtils,
  KeyPropTypes,
  KeyFuncs,
  PrefFuncs,
  pgpUI,
  pgpCL,
  pgpTLS,
  pgpKeys,
  pgpKeyServer,
  pgpMemoryMgr,
  pgpUtilities,
  pgpOptionList,
  pgpErrors, pgpPubTypes, pgpBase;

type
  TSignOption = (soEnc, soClear, soDetached);

{ "public" functions }
function PreferencesDialog(PrefsPage, WinHandle: Integer): Integer;
function SelectKeysDialog(const Prompt: String; var KeyPropsList: TKeyPropsList;
			  SingleKey: Longbool; PropertyFlags, FilterFlags, WinHandle: Integer): Integer;
function SendKeyToServerDialog(const Prompt: String; var KeyPropsList: TKeyPropsList;
			       KeyPropFlags, WinHandle: Integer): Integer;
function GetKeyFromServerDialog(const Prompt, KeyData: String; var KeyPropsList: TKeyPropsList;
				KeyPropFlags, WinHandle: Integer): Integer;
function KeyImportDialog(const Prompt, KeyData: String; var KeyPropsList: TKeyPropsList;
			 FromFile: Longbool; IgnoreKnownFlag: Integer; KeyPropFlags, WinHandle: Integer): Integer;
function KeyExportDialog(Prompt: String; const KeyIDs: TStrings; var KeyData: String; var KeyPropsList: TKeyPropsList;
			 ToFile, ExportCompatible, ExportPrivate: Longbool; KeyPropFlags, WinHandle: Integer): Integer;
function KeyRevokeDialog(const KeyHexID, PassPrompt: String; WinHandle: Integer): Integer;
function KeyPassChangeDialog(const KeyHexID, OldPrompt, NewPrompt: String;
			     MinPassLength, MinPassQuality, WinHandle: Integer): Integer;

{ "private" functions }
function SelectAddKeysToKeyRing(Context: pPGPContext; tlsContext: pPGPtlsContext;
				KeysToAdd: pPGPKeySet; KeySetMain: pPGPKeySet;
				var KeyPropsList: TKeyPropsList;
				const Prompt: String; SingleKey: Longbool;
				KeyPropFlags, WinHandle: Integer): PGPError;
function RecipientsDialog(Context: pPGPContext; AllKeys: pPGPKeySet;
			  AlwaysDisplayDialog: PGPBoolean; var Armor, EyesOnly, MIME, Textmode: PGPUInt32;
			  const DefaultKey: String; var RecipientKeys: Pointer;
			  const Prompt: String; WinHandle: Integer): PGPError;
function KeyServerDialog(Context: pPGPContext; tlsContext: pPGPtlsContext;
			 KeySetMain: pPGPKeySet; const KeyData: String;
			 var KeysFound: Pointer; WinHandle: Integer): PGPError;
function CollectRandomDataDialog(Context: pPGPContext; NeededEntropyBits: PGPUInt32; WinHandle: Integer): PGPError;
function ConvEncPassphraseDialog(Context: pPGPContext; var Passphrase: PChar;
				 const Prompt: String; WinHandle: Integer): PGPError;
function ConvDecPassphraseDialog(Context: pPGPContext; var Passphrase: PChar;
				 const Prompt: String; WinHandle: Integer): PGPError;
function KeyPassphraseDialog(Context: pPGPContext; KeySet: pPGPKeySet; var Passphrase: PChar;
			     const Prompt: String; WinHandle: Integer): PGPError;
function SigningPassphraseDialog(Context: pPGPContext; KeySetMain: pPGPKeySet;
				 var SigningKey: pPGPKey; var SignKeyPass: PChar;
				 const DefaultKeyID: String; FindMatchingKey: Longbool; SignOption: TSignOption;
				 var Armor, MIME: PGPUInt32; const Prompt: String; WinHandle: Integer): PGPError;
function DecryptionPassphraseDialog(Context: pPGPContext; RecipientKeys: pPGPKeySet;
				    KeyIDCount: PGPUInt32; const KeyIDList: pPGPKeyID7;
				    var DecryptionKey: pPGPKey; var DecryptKeyPass: PChar;
				    FindMatchingKey: Longbool; const Prompt: String; WinHandle: Integer): PGPError;
function ConfirmationPassphraseDialog(Context: pPGPContext; var Passphrase: PChar;
				      MinPassLength, MinPassQuality: Integer;
				      ShowPassphraseQuality: Longbool;
				      const Prompt: String; WinHandle: Integer): PGPError;

function KeyDlgInit(var Context: pPGPContext; var tlsContext: pPGPtlsContext; var KeySetMain: pPGPKeySet): Integer;
procedure KeyDlgFree(Context: pPGPContext; tlsContext: pPGPtlsContext; KeySetMain: pPGPKeySet);

implementation

function KeyDlgInit(var Context: pPGPContext; var tlsContext: pPGPtlsContext; var KeySetMain: pPGPKeySet): Integer;
begin
  tlsContext:=nil;
  Result:=KeyRings.InitKeyRings(Context, KeySetMain);
  if Result<>0 then Exit;
  Result:=PGPNewTLSContext(Context, tlsContext);
  if Result<>0 then Exit;
  Result:=PGPclInitLibrary(Context, kPGPFlags_ForceLocalExecution);
end;

procedure KeyDlgFree(Context: pPGPContext; tlsContext: pPGPtlsContext; KeySetMain: pPGPKeySet);
begin
  try PGPclCloseLibrary except end;
  PGPFreeTLSContext(tlsContext);
  KeyRings.FreeKeyRings;
end;

function PreferencesDialog(PrefsPage, WinHandle: Integer): Integer;
var
  Context	 : pPGPContext;
  tlsContext	 : pPGPtlsContext;
  KeySetMain	 : pPGPKeySet;
begin
  Result:=KeyDlgInit(Context, tlsContext, KeySetMain);
  if Result<>0 then Exit;
  try
    Result:=PGPclPreferences(Context, WinHandle, PrefsPage, KeySetMain);
    ProcessMessages;
  finally
    KeyDlgFree(Context, tlsContext, KeySetMain);
  end;
end;

function SelectKeysDialog(const Prompt: String; var KeyPropsList: TKeyPropsList;
			  SingleKey: Longbool; PropertyFlags, FilterFlags, WinHandle: Integer): Integer;
var
  Context	 : pPGPContext;
  tlsContext	 : pPGPtlsContext;
  KeySetMain	 : pPGPKeySet;
  KeyFilter	 : pPGPFilter;
  KeySetFiltered : pPGPKeySet;
  KeySetSelected : pPGPKeySet;
  KeyCount	 : PGPUInt32;
begin
  KeyFilter:=nil;
  KeySetFiltered:=nil;
  KeySetSelected:=nil;
  try
    Result:=KeyDlgInit(Context, tlsContext, KeySetMain);
    if Result<>0 then Exit;
    if FilterFlags<>KeyFilterFlag_AllKeys then begin
      Result:=GetKeyFilter(Context, FilterFlags, KeyFilter);
      if Result<>0 then Exit;
      try
	Result:=PGPFilterKeySet(KeySetMain, KeyFilter, KeySetFiltered);
	if Result<>0 then Exit;
      finally
	PGPFreeFilter(KeyFilter);
      end;
    end
    else KeySetFiltered:=KeySetMain;
    try
      Result:=PGPclSelectKeys(Context, tlsContext, WinHandle, PChar(Prompt), KeySetFiltered,
			      KeySetMain, (ord(SingleKey) and 1)*PGPCL_SINGLESELECTION, KeySetSelected);
      ProcessMessages;
      if Result<>0 then Exit;
    finally
      if KeySetFiltered<>KeySetMain then PGPFreeKeySet(KeySetFiltered);
    end;
    try
      Result:=PGPCountKeys(KeySetSelected, KeyCount);
      if Result<>0 then Exit;
      if KeyCount<>0 then begin
	Result:=GetKeySetProps(Context, KeySetSelected, KeyPropsList, PropertyFlags,
			       KeyFilterFlag_AllKeys, UserID_Ordering);
	if Result>0 then Result:=0;
      end;
    finally
      PGPFreeKeySet(KeySetSelected);
    end;
  finally
    KeyDlgFree(Context, tlsContext, KeySetMain);
  end;
end;

function InitKeyServer(Context: pPGPContext; var ServerThreadStorage: pPGPKeyServerThreadStorage;
		       var KeyServerSpecList: pPGPKeyServerSpec; var KeyServerCount: PGPUInt32): PGPError;
var
  KeyServerEntry : pPGPKeyServerSpec;
  KeyServerList	 : TStringList;
  KeyServerIndex : Integer;
begin
  ServerThreadStorage:=nil;
  KeyServerSpecList:=nil;
  KeyServerCount:=0;
  KeyServerList:=nil;
  Result:=PGPKeyServerInit;
  if Result<>0 then Exit;
  Result:=PGPKeyServerCreateThreadStorage(ServerThreadStorage);
  if Result<>0 then Exit;
  try
    Result:=GetServerList(KeyServerList);
    if (Result=0) and (KeyServerList<>nil) then with KeyServerList do begin
      KeyServerSpecList:=AllocMem(SizeOf(PGPKeyServerSpec)*Count);
      KeyServerEntry:=KeyServerSpecList;
      for KeyServerIndex:=0 to pred(Count) do begin
	Result:=PGPNewKeyServer(Context, kPGPKeyServerClass_PGP, KeyServerEntry.Server,
				PGPONetURL(Context, PChar(Strings[KeyServerIndex])), PGPOLastOption(Context));
	if Result<>0 then Exit;
	inc(KeyServerCount);
	inc(KeyServerEntry);
      end;
    end;
  finally
    KeyServerList.Free;
  end;
end;

procedure FreeKeyServer(ServerThreadStorage: pPGPKeyServerThreadStorage;
			KeyServerSpecList: pPGPKeyServerSpec; KeyServerCount: PGPUInt32);
var
  KeyServerEntry : pPGPKeyServerSpec;
begin
  KeyServerEntry:=KeyServerSpecList;
  try
    for KeyServerCount:=KeyServerCount downto 1 do begin
      PGPFreeKeyServer(KeyServerEntry.Server);
      inc(KeyServerEntry);
    end;
  finally
    FreeMem(KeyServerSpecList);
    PGPKeyServerDisposeThreadStorage(ServerThreadStorage);
    PGPKeyServerCleanup;
  end;
end;

function SendKeyToServerDialog(const Prompt: String; var KeyPropsList: TKeyPropsList;
			       KeyPropFlags, WinHandle: Integer): Integer;
var
  Context	 : pPGPContext;
  tlsContext	 : pPGPtlsContext;
  KeySetMain	 : pPGPKeySet;
  KeySetToSend	 : pPGPKeySet;
  ThreadStorage	 : pPGPKeyServerThreadStorage;
  KeyServerSpecs : pPGPKeyServerSpec;
  KeyServerCount : PGPUInt32;
  KeySetFailed	 : pPGPKeySet;
begin
  KeySetToSend:=nil;
  KeySetFailed:=nil;
  try
    Result:=KeyDlgInit(Context, tlsContext, KeySetMain);
    if Result<>0 then Exit;
    Result:=PGPclSelectKeys(Context, tlsContext, WinHandle, PChar(Prompt), KeySetMain, KeySetMain, 0, KeySetToSend);
    ProcessMessages;
    if Result<>0 then Exit;
    try
      Result:=InitKeyServer(Context, ThreadStorage, KeyServerSpecs, KeyServerCount);
      if Result<>0 then Exit;
      Result:=PGPSendToKeyServerDialog(Context, KeyServerSpecs^, tlsContext, KeySetToSend, KeySetFailed,
				       PGPOUIParentWindowHandle(Context, WinHandle), PGPOLastOption(Context));
      Sleep(100); // trying to prevent a strange cancel crash
      ProcessMessages;
      if (KeySetFailed<>nil) and (KeySetToSend<>nil) then PGPRemoveKeys(KeySetFailed, KeySetToSend);
      GetKeySetProps(Context, KeySetToSend, KeyPropsList, KeyPropFlags, KeyFilterFlag_AllKeys, UserID_Ordering);
    finally
      PGPFreeKeySet(KeySetFailed);
      PGPFreeKeySet(KeySetToSend);
      FreeKeyServer(ThreadStorage, KeyServerSpecs, KeyServerCount);
    end;
  finally
    KeyDlgFree(Context, tlsContext, KeySetMain);
  end;
end;

function GetKeyFromServerDialog(const Prompt, KeyData: String; var KeyPropsList: TKeyPropsList;
				KeyPropFlags, WinHandle: Integer): Integer;
var
  Context	 : pPGPContext;
  tlsContext	 : pPGPtlsContext;
  KeySetMain	 : pPGPKeySet;
  KeyFilter	 : pPGPFilter;
  ThreadStorage	 : pPGPKeyServerThreadStorage;
  KeyServerSpecs : pPGPKeyServerSpec;
  KeyServerCount : PGPUInt32;
  OptionList	 : pPGPOptionList;
  KeysFound	 : Pointer;
begin
  KeyFilter:=nil;
  OptionList:=nil;
  KeysFound:=nil;
  try
    Result:=KeyDlgInit(Context, tlsContext, KeySetMain);
    if Result<>0 then Exit;
    Result:=GetKeyFilterByAnyID(Context, KeyData, KeyFilter);
    if Result<>0 then Exit;
    try
      Result:=InitKeyServer(Context, ThreadStorage, KeyServerSpecs, KeyServerCount);
      if Result<>0 then Exit;
      Result:=PGPBuildOptionList(Context, OptionList,
	[
	  PGPOKeyServerSearchFilter(Context, KeyFilter),
	  PGPOUIParentWindowHandle(Context, WinHandle)
	]);
      try
	Result:=PGPSearchKeyServerDialog(Context, KeyServerCount, KeyServerSpecs, tlsContext, PGPBoolean(false),
					 KeysFound, OptionList, PGPOLastOption(Context));
	Sleep(100); // trying to prevent a strange cancel crash
	ProcessMessages;
      finally
	PGPFreeOptionList(OptionList);
      end;
      if Result<>0 then Exit;
      if PGP7X then begin
	try
	  Result:=SelectAddKeysToKeyRing(Context, tlsContext, PGPPeekKeyDBRootKeySet(pPGPKeyDB(KeysFound)),
					 KeySetMain, KeyPropsList, Prompt, false, KeyPropFlags, WinHandle);
	finally
	  PGPFreeKeyDB(pPGPKeyDB(KeysFound));
	end;
      end
      else begin
	try
	  Result:=SelectAddKeysToKeyRing(Context, tlsContext, pPGPKeySet(KeysFound),
					 KeySetMain, KeyPropsList, Prompt, false, KeyPropFlags, WinHandle);
	finally
	  PGPFreeKeySet(pPGPKeySet(KeysFound));
	end;
      end;
    finally
      FreeKeyServer(ThreadStorage, KeyServerSpecs, KeyServerCount);
      PGPFreeFilter(KeyFilter);
    end;
  finally
    KeyDlgFree(Context, tlsContext, KeySetMain);
  end;
end;

function KeyImportDialog(const Prompt, KeyData: String; var KeyPropsList: TKeyPropsList;
			 FromFile: Longbool; IgnoreKnownFlag: Integer; KeyPropFlags, WinHandle: Integer): Integer;
var
  Context	 : pPGPContext;
  tlsContext	 : pPGPtlsContext;
  KeySetMain	 : pPGPKeySet;
  FileSpec	 : pPGPFileSpec;
  KeysToImport	 : Pointer;
  KeySetSelToAdd : pPGPKeySet;
begin
  FileSpec:=nil;
  KeysToImport:=nil;
  KeySetSelToAdd:=nil;
  try
    Result:=KeyDlgInit(Context, tlsContext, KeySetMain);
    if Result<>0 then Exit;
    if FromFile then begin
      Result:=PGPNewFileSpecFromFullPath(Context, PChar(KeyData), FileSpec);
      if Result<>0 then Exit;
      try
	if PGP7X then begin
	  Result:=PGPImport(Context, KeysToImport,
			    PGPOInputFile(Context, FileSpec),
			    PGPOLastOption(Context));
	end
	else begin
	  Result:=PGPImportKeySet(Context, KeysToImport,
				  PGPOInputFile(Context, FileSpec),
				  PGPOLastOption(Context));
	end;
      finally
	PGPFreeFileSpec(FileSpec);
      end;
    end
    else begin
      if PGP7X then begin
	Result:=PGPImport(Context, KeysToImport,
			  PGPOInputBuffer(Context, PChar(KeyData), Length(KeyData)),
			  PGPOLastOption(Context));
      end
      else begin
	Result:=PGPImportKeySet(Context, KeysToImport,
				PGPOInputBuffer(Context, PChar(KeyData), Length(KeyData)),
				PGPOLastOption(Context));
      end;
    end;
    if Result<>0 then Exit;
    if PGP7X then begin
      try
	KeySetSelToAdd:=PGPPeekKeyDBRootKeySet(pPGPKeyDB(KeysToImport));
	Result:=GetExclusiveKeySet(KeySetSelToAdd, KeySetMain, Context, IgnoreKnownFlag);
	KeySetSelToAdd:=nil;
	if Result<>0 then Exit;
	if Prompt='' then begin
	  Result:=PGPCopyKeys(PGPPeekKeyDBRootKeySet(pPGPKeyDB(KeysToImport)),
	  		      PGPPeekKeySetKeyDB(KeySetMain), KeySetSelToAdd);
	  if Result<>0 then Exit;
	  Result:=KeyRings.UpdateKeyRings;
	  if Result<>0 then Exit;
	  Result:=GetKeySetProps(Context, KeySetSelToAdd, KeyPropsList,
				 KeyPropFlags, KeyFilterFlag_AllKeys, UserID_Ordering);
	  if Result>0 then Result:=0;
	end
	else begin
	  Result:=SelectAddKeysToKeyRing(Context, tlsContext, PGPPeekKeyDBRootKeySet(pPGPKeyDB(KeysToImport)),
					 KeySetMain, KeyPropsList, Prompt, false, KeyPropFlags, WinHandle);
	end;
      finally
	PGPFreeKeySet(KeySetSelToAdd);
	PGPFreeKeyDB(pPGPKeyDB(KeysToImport));
      end;
    end
    else begin
      try
	Result:=GetExclusiveKeySet(pPGPKeySet(KeysToImport), KeySetMain, Context, IgnoreKnownFlag);
	if Result<>0 then Exit;
	if Prompt='' then begin
	  Result:=PGPAddKeys(pPGPKeySet(KeysToImport), KeySetMain);
	  if Result<>0 then Exit;
	  Result:=KeyRings.UpdateKeyRings;
	  if Result<>0 then Exit;
	  Result:=GetKeySetProps(Context, pPGPKeySet(KeysToImport), KeyPropsList,
				 KeyPropFlags, KeyFilterFlag_AllKeys, UserID_Ordering);
	  if Result>0 then Result:=0;
	end
	else begin
	  Result:=SelectAddKeysToKeyRing(Context, tlsContext, pPGPKeySet(KeysToImport),
					 KeySetMain, KeyPropsList, Prompt, false, KeyPropFlags, WinHandle);
	end;
      finally
	PGPFreeKeySet(pPGPKeySet(KeysToImport));
      end;
    end;
  finally
    KeyDlgFree(Context, tlsContext, KeySetMain);
  end;
end;

function KeyExportDialog(Prompt: String; const KeyIDs: TStrings; var KeyData: String; var KeyPropsList: TKeyPropsList;
			 ToFile, ExportCompatible, ExportPrivate: Longbool; KeyPropFlags, WinHandle: Integer): Integer;
var
  Context	 : pPGPContext;
  tlsContext	 : pPGPtlsContext;
  KeySetMain	 : pPGPKeySet;
  KeySetSelected : pPGPKeySet;
  KeyCount	 : PGPUInt32;
  ExportFormat	 : PGPExportFormat;
  FileSpec	 : pPGPFileSpec;
  OptionList	 : pPGPOptionList;
  OptionListBuf	 : pPGPOptionList;
  KeyOutput	 : PChar;
  KeyLength	 : PGPUInt32;
begin
  KeySetSelected:=nil;
  FileSpec:=nil;
  OptionList:=nil;
  OptionListBuf:=nil;
  try
    Result:=KeyDlgInit(Context, tlsContext, KeySetMain);
    if Result<>0 then Exit;
    if KeyIDs.Count<>0 then begin
      GetKeySetByAnyIDs(Context, KeySetMain, KeyIDs.CommaText, KeySetSelected);
      if KeySetSelected<>nil then begin
	Result:=PGPCountKeys(KeySetSelected, KeyCount);
	if (Result<>0) or (KeyCount<>KeyIDs.Count) then begin
	  PGPFreeKeySet(KeySetSelected);
	  KeySetSelected:=nil;
	end;
      end;
    end;
    if KeySetSelected=nil then begin
      Result:=PGPclSelectKeys(Context, tlsContext, WinHandle, PChar(Prompt), KeySetMain, KeySetMain, 0, KeySetSelected);
      ProcessMessages;
    end;
    if Result<>0 then Exit;
    try
      Result:=PGPCountKeys(KeySetSelected, KeyCount);
      if Result<>0 then Exit;
      if KeyCount<>0 then begin
	if ExportCompatible then
	  ExportFormat:=kPGPExportFormat_Basic
	else ExportFormat:=kPGPExportFormat_Complete;
	Result:=PGPBuildOptionList(Context, OptionList,
	  [
	    PGPOExportFormat(Context, ExportFormat),
	    PGPOExportPrivateKeys(Context, PGPBoolean(ExportPrivate) and 1),
	    PGPOVersionString(Context, PChar(MyVersion))
	  ]);
	if Result<>0 then Exit;
	try
	  if ToFile then begin
	    Result:=PGPNewFileSpecFromFullPath(Context, PChar(KeyData), FileSpec);
	    if Result<>0 then Exit;
	    try
	      if Result<>0 then Exit;
	      Result:=PGPAppendOptionList(OptionList, [PGPOOutputFile(Context, FileSpec)]);
	      if Result<>0 then Exit;
	      if PGP7X then begin
		Result:=PGPAppendOptionList(OptionList, [PGPOExportKeySet(Context, KeySetSelected)]);
		if Result=0 then Result:=PGPExport(Context, OptionList, PGPOLastOption(Context));
	      end
	      else Result:=PGPExportKeySet(KeySetSelected, OptionList, PGPOLastOption(Context));
	      if Result=0 then Result:=GetKeySetProps(Context, KeySetSelected, KeyPropsList, KeyPropFlags,
						      KeyFilterFlag_AllKeys, UserID_Ordering);
	      if Result>0 then Result:=0;
	    finally
	      PGPFreeFileSpec(FileSpec);
	    end;
	  end
	  else begin
	    Result:=PGPCopyOptionList(OptionList, OptionListBuf);
	    if Result<>0 then Exit;
	    Result:=PGPAppendOptionList(OptionList, [PGPOAllocatedOutputBuffer(Context, KeyOutput, 1024, KeyLength)]);
	    if Result<>0 then Exit;
	    if PGP7X then begin
	      Result:=PGPAppendOptionList(OptionList, [PGPOExportKeySet(Context, KeySetSelected)]);
	      if Result=0 then Result:=PGPExport(Context, OptionList, PGPOLastOption(Context));
	    end
	    else Result:=PGPExportKeySet(KeySetSelected, OptionList, PGPOLastOption(Context));
	    if Result=kPGPError_OutputBufferTooSmall then begin
	      PGPFreeData(KeyOutput);
	      Result:=PGPAppendOptionList(OptionListBuf, [PGPOAllocatedOutputBuffer(Context, KeyOutput, KeyLength, KeyLength)]);
	      if Result<>0 then Exit;
	      if PGP7X then begin
		Result:=PGPAppendOptionList(OptionListBuf, [PGPOExportKeySet(Context, KeySetSelected)]);
		if Result=0 then Result:=PGPExport(Context, OptionListBuf, PGPOLastOption(Context));
	      end
	      else Result:=PGPExportKeySet(KeySetSelected, OptionListBuf, PGPOLastOption(Context));
	    end;
	    if (Result=0) and (KeyLength<>0) then begin
	      try
		SetLength(KeyData, KeyLength);
		StrPLCopy(PChar(KeyData), KeyOutput, KeyLength);
	      finally
		PGPFreeData(KeyOutput);
	      end;
	    end
	    else KeyData:='';
	    if Result=0 then Result:=GetKeySetProps(Context, KeySetSelected, KeyPropsList, KeyPropFlags,
						    KeyFilterFlag_AllKeys, UserID_Ordering);
	    if Result>0 then Result:=0;
	  end;
	finally
	  PGPFreeOptionList(OptionListBuf);
	  PGPFreeOptionList(OptionList);
	end;
      end;
    finally
      PGPFreeKeySet(KeySetSelected);
    end;
  finally
    KeyDlgFree(Context, tlsContext, KeySetMain);
  end;
end;

function KeyRevokeDialog(const KeyHexID, PassPrompt: String; WinHandle: Integer): Integer;
var
  Context	 : pPGPContext;
  KeySetMain	 : pPGPKeySet;
  KeySetFound	 : pPGPKeySet;
  Passphrase	 : PChar;
  Key		 : pPGPKey;
begin
  KeySetFound:=nil;
  Passphrase:=nil;
  try
    Result:=KeyRings.InitKeyRings(Context, KeySetMain);
    if Result<>0 then Exit;
    Result:=GetKeySetByAnyIDs(Context, KeySetMain, KeyHexID, KeySetFound);
    if Result<>0 then Exit;
    try
      Result:=KeyPassphraseDialog(Context, KeySetFound, Passphrase, PassPrompt, WinHandle);
      ProcessMessages;
      if Result<>0 then Exit;
    finally
      PGPFreeKeySet(KeySetFound);
    end;
    try
      Result:=GetKeyByHexID(KeySetMain, KeyHexID, Key);
      if Result<>0 then Exit;
      Result:=PGPRevokeKey(Key, PGPOPassphrase(Context, Passphrase), PGPOLastOption(Context));
      if Result<>0 then Exit;
      Result:=KeyRings.UpdateKeyRings;
    finally
      PGPFreeData(Passphrase);
    end;
  finally
    KeyRings.FreeKeyRings;
  end;
end;

function KeyPassChangeDialog(const KeyHexID, OldPrompt, NewPrompt: String;
			     MinPassLength, MinPassQuality, WinHandle: Integer): Integer;
var
  Context	 : pPGPContext;
  KeySetMain	 : pPGPKeySet;
  KeySetFound	 : pPGPKeySet;
  OldPassphrase	 : PChar;
  NewPassphrase  : PChar;
  OptionList	 : pPGPOptionList;
  Key		 : pPGPKey;
  KeyList	 : pPGPKeyList;
  KeyIter	 : pPGPkeyIter;
  SubKey	 : pPGPKey;
begin
  KeySetFound:=nil;
  OldPassphrase:=nil;
  NewPassphrase:=nil;
  OptionList:=nil;
  try
    Result:=KeyRings.InitKeyRings(Context, KeySetMain);
    if Result<>0 then Exit;
    Result:=GetKeySetByAnyIDs(Context, KeySetMain, KeyHexID, KeySetFound);
    if Result<>0 then Exit;
    try
      Result:=KeyPassphraseDialog(Context, KeySetFound, OldPassphrase, OldPrompt, WinHandle);
      ProcessMessages;
      if Result<>0 then Exit;
    finally
      PGPFreeKeySet(KeySetFound);
    end;
    try
      Result:=ConfirmationPassphraseDialog(Context, NewPassphrase,
					   MinPassLength, MinPassQuality, true,
					   NewPrompt, WinHandle);
      ProcessMessages;
      if Result<>0 then Exit;
      try
	Result:=PGPBuildOptionList(Context, OptionList,
	  [
	    PGPOPassphrase(Context, OldPassphrase),
	    PGPOPassphrase(Context, NewPassphrase)
	  ]);
	if Result<>0 then Exit;
	try
	  Result:=GetKeyByHexID(KeySetMain, KeyHexID, Key);
	  if Result<>0 then Exit;
	  Result:=PGPChangePassphrase(Key, OptionList, PGPOLastOption(Context));
	  if Result<>0 then Exit;
	  if GetKeyPropAlg(Key)=KeyAlgorithm_DHDSS then begin
	    KeyList:=nil;
	    KeyIter:=nil;
	    Result:=PGPOrderKeySet(KeySetMain, kPGPKeyOrdering_Any, PGPBoolean(false), KeyList);
	    if Result<>0 then Exit;
	    try
	      Result:=PGPNewKeyIter(KeyList, KeyIter);
	      if Result<>0 then Exit;
	      try
		PGPKeyIterSeek(KeyIter, Key);
		while PGPKeyIterNextKeyDBObj(KeyIter, kPGPKeyDBObjType_SubKey, SubKey)=0 do begin
		  Result:=PGPChangeSubKeyPassphrase(SubKey, OptionList, PGPOLastOption(Context));
		  if Result<>0 then Exit;
		end;
	      finally
		PGPFreeKeyIter(KeyIter);
	      end;
	    finally
	      PGPFreeKeyList(KeyList);
	    end;
	  end;
	  Result:=KeyRings.UpdateKeyRings;
	finally
	  PGPFreeOptionList(OptionList);
	end;
      finally
	PGPFreeData(NewPassphrase);
      end;
    finally
      PGPFreeData(OldPassphrase);
    end;
  finally
    KeyRings.FreeKeyRings;
  end;
end;

function SelectAddKeysToKeyRing(Context: pPGPContext; tlsContext: pPGPtlsContext;
				KeysToAdd: pPGPKeySet; KeySetMain: pPGPKeySet;
				var KeyPropsList: TKeyPropsList;
				const Prompt: String; SingleKey: Longbool;
				KeyPropFlags, WinHandle: Integer): PGPError;
var
  KeyCount	 : PGPUInt32;
  KeySetSelected : pPGPKeySet;
  KeySetSelToAdd : pPGPKeySet;
begin
  KeySetSelected:=nil;
  KeySetSelToAdd:=nil;
  Result:=PGPCountKeys(KeysToAdd, KeyCount);
  if Result<>0 then Exit;
  if KeyCount<>0 then begin
    Result:=PGPclSelectKeys(Context, tlsContext, WinHandle, PChar(Prompt), KeysToAdd,
			    KeySetMain, (ord(SingleKey) and 1)*PGPCL_SINGLESELECTION, KeySetSelected);
    ProcessMessages;
    if Result<>0 then Exit;
    try
      if PGP7X then
	Result:=PGPCopyKeys(KeySetSelected, PGPPeekKeySetKeyDB(KeySetMain), KeySetSelToAdd)
      else begin
	KeySetSelToAdd:=KeySetSelected;
	Result:=PGPAddKeys(KeySetSelToAdd, KeySetMain);
      end;
      if Result<>0 then Exit;
      Result:=KeyRings.UpdateKeyRings;
      if Result<>0 then Exit;
      Result:=GetKeySetProps(Context, KeySetSelToAdd, KeyPropsList, KeyPropFlags,
			     KeyFilterFlag_AllKeys, UserID_Ordering);
      if Result>0 then Result:=0;
    finally
      if PGP7X then PGPFreeKeySet(KeySetSelected);
      PGPFreeKeySet(KeySetSelToAdd);
    end;
  end
  else Result:=kPGPError_PublicKeyNotFound;
end;

function RecipientsDialog(Context: pPGPContext; AllKeys: pPGPKeySet;
			  AlwaysDisplayDialog: PGPBoolean; var Armor, EyesOnly, MIME, Textmode: PGPUInt32;
			  const DefaultKey: String; var RecipientKeys: Pointer;
			  const Prompt: String; WinHandle: Integer): PGPError;
var
  OptionList	 : pPGPOptionList;
  FormOptions	 : pPGPOptionList;
  RecipientSpec	 : PGPRecipientSpec;
begin
  RecipientKeys:=nil;
  OptionList:=nil;
  FormOptions:=nil;
  Result:=PGPBuildOptionList(Context, FormOptions,
    [
      PGPOUICheckbox(Context, 801, '&Armor', nil, Armor, Armor, PGPOLastOption(Context)),
      PGPOUICheckbox(Context, 802, '&MIME', nil, MIME, MIME, PGPOLastOption(Context)),
      PGPOUICheckbox(Context, 803, '&Textmode', nil, Textmode, Textmode, PGPOLastOption(Context)),
      PGPOUICheckbox(Context, 804, 'For your &eyes only', nil, EyesOnly, EyesOnly, PGPOLastOption(Context))
    ]);
  if Result<>0 then Exit;
  try
    Result:=PGPBuildOptionList(Context, OptionList,
      [
	PGPOUIDialogOptions(Context, FormOptions, PGPOLastOption(Context)),
	PGPOUIDisplayMarginalValidity(Context, PGPBoolean(true)),
	PGPOUIWindowTitle(Context, 'PGP'),
	PGPOUIDialogPrompt(Context, PChar(Prompt)),
	PGPOUIParentWindowHandle(Context, WinHandle)
      ]);
    if Result<>0 then Exit;
    if DefaultKey<>'' then begin
      FillChar(RecipientSpec, SizeOf(PGPRecipientSpec), 0);
      RecipientSpec.AType:=kPGPRecipientSpecType_Key;
      Result:=GetKeyByHexID(AllKeys, DefaultKey, RecipientSpec.ID.Key);
      if Result=0 then PGPAppendOptionList(OptionList, [PGPOUIDefaultRecipients(Context, 1, @RecipientSpec)]);
    end;
    try
      if PGP7X then begin
	Result:=PGPRecipientDialog(Context, PGPPeekKeySetKeyDB(AllKeys), AlwaysDisplayDialog,
				   pPGPKeyDB(RecipientKeys) ,OptionList, PGPOLastOption(Context));
      end
      else begin
	Result:=PGPRecipientDialog(Context, AllKeys, AlwaysDisplayDialog,
				   pPGPKeySet(RecipientKeys), OptionList, PGPOLastOption(Context));
      end;
      ProcessMessages;
    finally
      PGPFreeOptionList(OptionList);
    end;
  finally
    PGPFreeOptionList(FormOptions);
  end;
end;

function KeyServerDialog(Context: pPGPContext; tlsContext: pPGPtlsContext;
			 KeySetMain: pPGPKeySet; const KeyData: String;
			 var KeysFound: Pointer; WinHandle: Integer): PGPError;
var
  KeyFilter	 : pPGPFilter;
  ThreadStorage	 : pPGPKeyServerThreadStorage;
  KeyServerSpecs : pPGPKeyServerSpec;
  KeyServerCount : PGPUInt32;
  OptionList	 : pPGPOptionList;
begin
  KeysFound:=nil;
  KeyFilter:=nil;
  OptionList:=nil;
  Result:=GetKeyFilterByAnyID(Context, KeyData, KeyFilter);
  if Result<>0 then Exit;
  try
    Result:=InitKeyServer(Context, ThreadStorage, KeyServerSpecs, KeyServerCount);
    if Result<>0 then Exit;
    Result:=PGPBuildOptionList(Context, OptionList,
      [
	PGPOKeyServerSearchFilter(Context, KeyFilter),
	PGPOUIParentWindowHandle(Context, WinHandle)
      ]);
    try
      Result:=PGPSearchKeyServerDialog(Context, KeyServerCount, KeyServerSpecs, tlsContext, PGPBoolean(false),
				       KeysFound, OptionList, PGPOLastOption(Context));
      Sleep(100); // trying to prevent a strange cancel crash
      ProcessMessages;
    finally
      PGPFreeOptionList(OptionList);
    end;
  finally
    FreeKeyServer(ThreadStorage, KeyServerSpecs, KeyServerCount);
    PGPFreeFilter(KeyFilter);
  end;
end;

function CollectRandomDataDialog(Context: pPGPContext; NeededEntropyBits: PGPUInt32; WinHandle: Integer): PGPError;
begin
  Result:=PGPCollectRandomDataDialog(Context,
				     NeededEntropyBits,
				     PGPOUIParentWindowHandle(Context, WinHandle),
				     PGPOLastOption(Context));
  ProcessMessages;
end;

function ConvEncPassphraseDialog(Context: pPGPContext; var Passphrase: PChar;
				 const Prompt: String; WinHandle: Integer): PGPError;
var
  OptionList	 : pPGPOptionList;
begin
  Passphrase:=nil;
  OptionList:=nil;
  Result:=PGPBuildOptionList(Context, OptionList,
    [
      PGPOUIWindowTitle(Context, 'PGP'),
      PGPOUIDialogPrompt(Context, PChar(Prompt)),
      PGPOUIOutputPassphrase(Context, Passphrase),
      PGPOUIParentWindowHandle(Context, WinHandle)
    ]);
  if Result<>0 then Exit;
  try
    Result:=PGPConventionalEncryptionPassphraseDialog(Context, OptionList, PGPOLastOption(Context));
    ProcessMessages;
  finally
    PGPFreeOptionList(OptionList);
  end;
end;

function ConvDecPassphraseDialog(Context: pPGPContext; var Passphrase: PChar;
				 const Prompt: String; WinHandle: Integer): PGPError;
var
  OptionList	 : pPGPOptionList;
begin
  Passphrase:=nil;
  OptionList:=nil;
  Result:=PGPBuildOptionList(Context, OptionList,
    [
      PGPOUIWindowTitle(Context, 'PGP'),
      PGPOUIDialogPrompt(Context, PChar(Prompt)),
      PGPOUIOutputPassphrase(Context, Passphrase),
      PGPOUIParentWindowHandle(Context, WinHandle)
    ]);
  if Result<>0 then Exit;
  try
    Result:=PGPConventionalDecryptionPassphraseDialog(Context, OptionList, PGPOLastOption(Context));
    ProcessMessages;
  finally
    PGPFreeOptionList(OptionList);
  end;
end;

function KeyPassphraseDialog(Context: pPGPContext; KeySet: pPGPKeySet; var Passphrase: PChar;
			     const Prompt: String; WinHandle: Integer): PGPError;
var
  Key		 : pPGPKey;
  OptionList	 : pPGPOptionList;
begin
  Passphrase:=nil;
  OptionList:=nil;
  Result:=GetKeyFromKeySet(KeySet, Key);
  if Result<>0 then Exit;
  Result:=PGPBuildOptionList(Context, OptionList,
    [
      PGPOUIWindowTitle(Context, 'PGP'),
      PGPOUIDialogPrompt(Context, PChar(Prompt)),
      PGPOUIOutputPassphrase(Context, Passphrase),
      PGPOUIParentWindowHandle(Context, WinHandle)
    ]);
  if Result<>0 then Exit;
  try
    Result:=PGPKeyPassphraseDialog(Context, Key, OptionList, PGPOLastOption(Context));
    ProcessMessages;
  finally
    PGPFreeOptionList(OptionList);
  end;
end;

function SigningPassphraseDialog(Context: pPGPContext; KeySetMain: pPGPKeySet;
				 var SigningKey: pPGPKey; var SignKeyPass: PChar;
				 const DefaultKeyID: String; FindMatchingKey: Longbool; SignOption: TSignOption;
				 var Armor, MIME: PGPUInt32; const Prompt: String; WinHandle: Integer): PGPError;
var
  DefaultKey	 : pPGPKey;
  OptionList	 : pPGPOptionList;
begin
  SigningKey:=nil;
  SignKeyPass:=nil;
  OptionList:=nil;
  if (DefaultKeyID='') or (GetKeyByHexID(KeySetMain, PChar(DefaultKeyID), DefaultKey)<>0) then DefaultKey:=nil;
  Result:=PGPBuildOptionList(Context, OptionList,
    [
      PGPOUIWindowTitle(Context, 'PGP'),
      PGPOUIDialogPrompt(Context, PChar(Prompt)),
      PGPOUIDefaultKey(Context, DefaultKey),
      PGPOUIFindMatchingKey(Context, PGPBoolean(FindMatchingKey) and 1),
      PGPOUIVerifyPassphrase(Context, PGPBoolean(true)),
      PGPOUIOutputPassphrase(Context, SignKeyPass),
      PGPOUIParentWindowHandle(Context, WinHandle)
    ]);
  if Result<>0 then Exit;
  case SignOption of
    soClear: begin
      Result:=PGPAppendOptionList(OptionList,
	[
	  PGPOUIDialogOptions(Context,
			      PGPOUICheckbox(Context, 802, '&MIME', nil, MIME, MIME, PGPOLastOption(Context)),
			      PGPOLastOption(Context))
	]);
    end;
    soDetached: begin
      Result:=PGPAppendOptionList(OptionList,
	[
	  PGPOUIDialogOptions(Context,
			      PGPOUICheckbox(Context, 801, '&Armor', nil, Armor, Armor, PGPOLastOption(Context)),
			      PGPOLastOption(Context))
	]);
    end;
  end;
  if Result<>0 then Exit;
  try
    if PGP7X then begin
      Result:=PGPSigningPassphraseDialog(Context, PGPPeekKeySetKeyDB(KeySetMain), SigningKey,
					 OptionList, PGPOLastOption(Context));
    end
    else Result:=PGPSigningPassphraseDialog(Context, KeySetMain, SigningKey, OptionList, PGPOLastOption(Context));
    ProcessMessages;
  finally
    PGPFreeOptionList(OptionList);
  end;
end;

function DecryptionPassphraseDialog(Context: pPGPContext; RecipientKeys: pPGPKeySet;
				    KeyIDCount: PGPUInt32; const KeyIDList: pPGPKeyID7;
				    var DecryptionKey: pPGPKey; var DecryptKeyPass: PChar;
				    FindMatchingKey: Longbool; const Prompt: String; WinHandle: Integer): PGPError;
var
  OptionList	 : pPGPOptionList;
begin
  DecryptionKey:=nil;
  DecryptKeyPass:=nil;
  OptionList:=nil;
  Result:=PGPBuildOptionList(Context, OptionList,
    [
      PGPOUIWindowTitle(Context, 'PGP'),
      PGPOUIDialogPrompt(Context, PChar(Prompt)),
      PGPOUIFindMatchingKey(Context, PGPBoolean(FindMatchingKey) and 1),
      PGPOUIVerifyPassphrase(Context, PGPBoolean(true)),
      PGPOUIOutputPassphrase(Context, DecryptKeyPass),
      PGPOUIParentWindowHandle(Context, WinHandle)
    ]);
  if Result<>0 then Exit;
  try
    Result:=PGPDecryptionPassphraseDialog(Context, RecipientKeys, KeyIDCount, KeyIDList,
					  DecryptionKey, OptionList, PGPOLastOption(Context));
    ProcessMessages;
  finally
    PGPFreeOptionList(OptionList);
  end;
end;

function ConfirmationPassphraseDialog(Context: pPGPContext; var Passphrase: PChar;
				      MinPassLength, MinPassQuality: Integer;
				      ShowPassphraseQuality: Longbool;
				      const Prompt: String; WinHandle: Integer): PGPError;
var
  OptionList	 : pPGPOptionList;
begin
  Passphrase:=nil;
  OptionList:=nil;
  Result:=PGPBuildOptionList(Context, OptionList,
    [
      PGPOUIWindowTitle(Context, 'PGP'),
      PGPOUIDialogPrompt(Context, PChar(Prompt)),
      PGPOUIMinimumPassphraseLength(Context, MinPassLength),
      PGPOUIMinimumPassphraseQuality(Context, MinPassQuality),
      PGPOUIShowPassphraseQuality(Context, PGPBoolean(ShowPassphraseQuality) and 1),
      PGPOUIOutputPassphrase(Context, Passphrase),
      PGPOUIParentWindowHandle(Context, WinHandle)
    ]);
  if Result<>0 then Exit;
  try
    Result:=PGPConfirmationPassphraseDialog(Context, OptionList, PGPOLastOption(Context));
    ProcessMessages;
  finally
    PGPFreeOptionList(OptionList);
  end;
end;

end.

