{$J+,Z4}
unit KeyPropTypes;

{------------------------------------------------------------------------------}
{                                                                              }
{                                 This code is                                 }
{                 Copyright (C) 2001 by Michael in der Wiesche                 }
{                                                                              }
{------------------------------------------------------------------------------}

interface

uses
  Classes;

{ My tribute to Steve Heller's great work with creating SPGP & DPGP }
const
  spgpKeyPropFlag_None			=  0;
  spgpKeyPropFlag_All			= -1;

  // "string" properties
  spgpKeyPropFlag_KeyID			= $00000001;
  spgpKeyPropFlag_UserID		= $00000002;
  spgpKeyPropFlag_Fingerprint		= $00000004;
  spgpKeyPropFlag_CreationTimeStr	= $00000008;
  spgpKeyPropFlag_ExpirationTimeStr	= $00000010;

  // "numeric" properties
  spgpKeyPropFlag_Keybits		= $00000080;
  spgpKeyPropFlag_KeyAlg		= $00000100;
  spgpKeyPropFlag_Trust			= $00000200;
  spgpKeyPropFlag_Validity		= $00000400;
  spgpKeyPropFlag_CreationTime		= $00000800;
  spgpKeyPropFlag_ExpirationTime	= $00001000;

  // "boolean" properties
  spgpKeyPropFlag_IsSecret		= $00008000;
  spgpKeyPropFlag_IsAxiomatic		= $00010000;
  spgpKeyPropFlag_IsRevoked		= $00020000;
  spgpKeyPropFlag_IsDisabled		= $00040000;
  spgpKeyPropFlag_IsExpired		= $00080000;
  spgpKeyPropFlag_IsSecretShared	= $00100000;

  spgpKeyPropFlag_CanEncrypt		= $00200000;
  spgpKeyPropFlag_CanDecrypt		= $00400000;
  spgpKeyPropFlag_CanSign		= $00800000;
  spgpKeyPropFlag_CanVerify		= $01000000;

  spgpKeyPropFlag_HasRevoker		= $02000000;
  spgpKeyPropFlag_HasADK		= $04000000;
  spgpKeyPropFlag_HasSubKey		= $08000000;

  // "list" flags
  spgpKeyPropFlag_IncludeUserIDs	= $10000000;
  spgpKeyPropFlag_IncludeSignerIDs	= $20000000;

  // "convenience" flags
  spgpKeyPropFlag_IDFlags		= spgpKeyPropFlag_KeyID or spgpKeyPropFlag_UserID;
  spgpKeyPropFlag_IDComplete		= spgpKeyPropFlag_IDFlags or spgpKeyPropFlag_Keybits or spgpKeyPropFlag_KeyAlg;

const
  KeyFilterFlag_AllKeys			= $0000;
  // boolean property filters
  KeyFilterFlag_CanEncrypt		= $0001;
  KeyFilterFlag_CanDecrypt		= $0002;
  KeyFilterFlag_CanSign			= $0004;
  KeyFilterFlag_CanVerify		= $0008;
  // ability filters
  KeyFilterFlag_Enabled			= $0010;
  KeyFilterFlag_Disabled		= $0020;
  // algorithm filters
  KeyFilterFlag_DHDSS			= $0100;
  KeyFilterFlag_RSA			= $0200;
  KeyFilterFlag_V4			= $0400;

  // the groups can be combined
  KeyFilterMask_Boolean			= $00FF;
  KeyFilterMask_Algorithm		= $FF00;

const
  // pre-selection flag for key import
  IgnoreFlag_ByHexID			= 1;
  IgnoreFlag_ByUserID			= 2;

type
  TPGPKeyOrdering = (
    Invalid_Ordering,
    Any_Ordering,
    UserID_Ordering,
    ReverseUserID_Ordering,
    KeyID_Ordering,
    ReverseKeyID_Ordering,
    Validity_Ordering,
    ReverseValidity_Ordering,
    Trust_Ordering,
    ReverseTrust_Ordering,
    EncryptKeySize_Ordering,
    ReverseEncryptKeySize_Ordering,
    SigKeySize_Ordering,
    ReverseSigKeySize_Ordering,
    Creation_Ordering,
    ReverseCreation_Ordering,
    Expiration_Ordering,
    ReverseExpiration_Ordering
  );
  TTrustLevel = (
    KeyTrust_Undefined,
    KeyTrust_Unknown,
    KeyTrust_Never,
    KeyTrust_Reserved1,
    KeyTrust_Reserved2,
    KeyTrust_Marginal,
    KeyTrust_Complete,
    KeyTrust_Ultimate
  );
  TValidityLevel = (
    Validity_Unknown,
    Validity_Invalid,
    Validity_Marginal,
    Validity_Complete
  );
  TKeyAlgorithm = (
    KeyAlgorithm_Invalid,
    KeyAlgorithm_RSA,
    KeyAlgorithm_RSAEncryptOnly,
    KeyAlgorithm_RSASignOnly,
    KeyAlgorithm_Reserved01,
    KeyAlgorithm_Reserved02,
    KeyAlgorithm_Reserved03,
    KeyAlgorithm_Reserved04,
    KeyAlgorithm_Reserved05,
    KeyAlgorithm_Reserved06,
    KeyAlgorithm_Reserved07,
    KeyAlgorithm_Reserved08,
    KeyAlgorithm_Reserved09,
    KeyAlgorithm_Reserved10,
    KeyAlgorithm_Reserved11,
    KeyAlgorithm_Reserved12,
    KeyAlgorithm_DH,
    KeyAlgorithm_DSS,
    KeyAlgorithm_DHDSS
  );
  TCipherAlgorithm = (
    CipherAlgorithm_None,
    CipherAlgorithm_IDEA,
    CipherAlgorithm_3DES,
    CipherAlgorithm_CAST5,
    CipherAlgorithm_AES128,
    CipherAlgorithm_AES192,
    CipherAlgorithm_AES256,
    CipherAlgorithm_Twofish256
  );
  TADKType = (
    NoADK,
    SimpleADK,
    EnforcedADK
  );

  TUserIDs = TStringList;
  TSignerIDs = TStringList;
  TADKeyIDs = TStringList;
  TRevKeyIDs = TStringList;

  pKeyPropsRec = ^TKeyPropsRec;
  TKeyPropsRec = Record
    kHexID: String;
    kUserID: String;
    kFingerprint: String;
    kCreaTimeStr: String;
    kExpTimeStr: String;
    kSize: String;
    kAlgorithm: TKeyAlgorithm;
    kTrust: TTrustLevel;
    kValidity: TValidityLevel;
    kCreaTimeNum: Longint;
    kExpTimeNum: Longint;
    kPrivate: Longbool;
    kImplicitTrust: Longbool;
    kRevoked: Longbool;
    kDisabled: Longbool;
    kExpired: Longbool;
    kSecShared: Longbool;
    kCanEncrypt: Longbool;
    kCanDecrypt: Longbool;
    kCanSign: Longbool;
    kCanVerify: Longbool;
    kHasRevoker: Longbool;
    kHasADK: TADKType;
    kHasSubKey: Longbool;
    kUserIDList: TUserIDs;
    kSignerIDList: TSignerIDs;
    kADKeyIDList: TADKeyIDs;
    kRevKeyIDList: TRevKeyIDs;
  end;

  pKeyPropsList = ^TKeyPropsList;
  TKeyPropsList = class(TStringList)
  private
    IncludeUserIDs: Longbool;
    IncludeSignerIDs: Longbool;
    IncludeADKeyIDs: Longbool;
    IncludeRevKeyIDs: Longbool;
    function	AllocKeyPropsRec: pKeyPropsRec;
    function    FreeKeyPropsRec(const KeyPropsRec: pKeyPropsRec): pKeyPropsRec;
  public
    constructor	Create(ItemCount: Cardinal; PropertyFlags: Longint);
    destructor	Destroy; override;
    procedure	Clear; override;
    procedure	Delete(Index: Integer); override;
    procedure   Move(CurIndex, NewIndex: Integer); override;
    function    Add(const S: String): Integer; override;
    {$IFNDEF VER140}
    function    AddObject(const S: String; AObject: TObject): Integer; override;
    {$ENDIF}
    procedure   Insert(Index: Integer; const S: String); override;
    // returns first free item index
    function	Append(ItemCount: Cardinal): Integer;
    // returns true if requested record item exists
    function	GetKeyPropsRec(var KeyPropsRec: TKeyPropsRec; Index: Integer): Longbool;
  end;

implementation

uses
  KeyFuncs;

function TKeyPropsList.AllocKeyPropsRec: pKeyPropsRec;
begin
  New(Result);
  if Result<>nil then begin
    FillChar(Result^, SizeOf(TKeyPropsRec), 0);
    if IncludeUserIDs then Result.kUserIDList:=TStringList.Create;
    if IncludeSignerIDs then Result.kSignerIDList:=TStringList.Create;
    if IncludeADKeyIDs then Result.kADKeyIDList:=TStringList.Create;
    if IncludeRevKeyIDs then Result.kRevKeyIDList:=TStringList.Create;
  end;
end;

function TKeyPropsList.FreeKeyPropsRec(const KeyPropsRec: pKeyPropsRec): pKeyPropsRec;
begin
  Result:=KeyPropsRec;
  if Result<>nil then begin
    Result^.kUserIDList.Free;
    Result^.kSignerIDList.Free;
    Result^.kADKeyIDList.Free;
    Result^.kRevKeyIDList.Free;
    Dispose(Result);
    Result:=nil;
  end;
end;

constructor TKeyPropsList.Create(ItemCount: Cardinal; PropertyFlags: Longint);
var Index: Cardinal;
begin
  inherited Create;
  IncludeUserIDs:=((PropertyFlags and spgpKeyPropFlag_IncludeUserIDs)<>0);
  IncludeSignerIDs:=((PropertyFlags and spgpKeyPropFlag_IncludeSignerIDs)<>0);
  IncludeADKeyIDs:=((PropertyFlags and spgpKeyPropFlag_HasADK)<>0);
  IncludeRevKeyIDs:=((PropertyFlags and spgpKeyPropFlag_HasRevoker)<>0);
  {$IFDEF VER140}
  if ItemCount>0 then for Index:=1 to ItemCount do AddObject('', TObject(AllocKeyPropsRec));
  {$ELSE}
  if ItemCount>0 then for Index:=1 to ItemCount do Objects[inherited Add('')]:=TObject(AllocKeyPropsRec);
  {$ENDIF}
end;

destructor TKeyPropsList.Destroy;
var Index: Integer;
begin
  if Count<>0 then for Index:=0 to pred(Count) do FreeKeyPropsRec(pKeyPropsRec(Objects[Index]));
  inherited Destroy;
end;

procedure TKeyPropsList.Clear;
var Index: Integer;
begin
  if Count<>0 then for Index:=0 to pred(Count) do FreeKeyPropsRec(pKeyPropsRec(Objects[Index]));
  inherited Clear;
end;

procedure TKeyPropsList.Delete(Index: Integer);
begin
  FreeKeyPropsRec(pKeyPropsRec(Objects[Index]));
  inherited Delete(Index);
end;

procedure TKeyPropsList.Move(CurIndex, NewIndex: Integer);
var TempStr: String; TempObj: TObject;
begin
  if CurIndex<>NewIndex then begin
    TempStr:=Strings[CurIndex];
    TempObj:=Objects[CurIndex];
    inherited Delete(CurIndex);
    InsertObject(NewIndex, TempStr, TempObj);
  end;
end;

function TKeyPropsList.Add(const S: String): Integer;
begin
  {$IFDEF VER140}
  Result:=AddObject(S, TObject(AllocKeyPropsRec));
  {$ELSE}
  Result:=inherited Add(S);
  Objects[Result]:=TObject(AllocKeyPropsRec);
  {$ENDIF}
end;

{$IFNDEF VER140}
function TKeyPropsList.AddObject(const S: String; AObject: TObject): Integer;
begin
  Result:=inherited Add(S);
  Objects[Result]:=AObject;
end;
{$ENDIF}

procedure TKeyPropsList.Insert(Index: Integer; const S: String);
begin
  inherited Insert(Index, S);
  Objects[Index]:=TObject(AllocKeyPropsRec);
end;

function TKeyPropsList.Append(ItemCount: Cardinal): Integer;
var Index: Cardinal;
begin
  Result:=Count;
  try
    {$IFDEF VER140}
    if ItemCount>0 then for Index:=1 to ItemCount do AddObject('', TObject(AllocKeyPropsRec));
    {$ELSE}
    if ItemCount>0 then for Index:=1 to ItemCount do Objects[inherited Add('')]:=TObject(AllocKeyPropsRec);
    {$ENDIF}
  except
    Result:=-1;
  end;
end;

function TKeyPropsList.GetKeyPropsRec(var KeyPropsRec: TKeyPropsRec; Index: Integer): Longbool;
begin
  Result:=false;
  FillChar(KeyPropsRec, SizeOf(TKeyPropsRec), 0);
  if (Index<Count) and (Objects[Index]<>nil) then begin
    KeyPropsRec:=pKeyPropsRec(Objects[Index])^;
    Result:=true;
  end;
end;

end.

