{$J+,Z4}
unit PGPKeyGenerate;

{------------------------------------------------------------------------------}
{                                                                              }
{                 This unit is partly based on Steve Heller's                  }
{         SPGP sources available from http://www.oz.net/~srheller/spgp/        }
{                                                                              }
{                Portions created by Michael in der Wiesche are                }
{                 Copyright (C) 2001 by Michael in der Wiesche                 }
{                                                                              }
{------------------------------------------------------------------------------}

interface

uses
  Windows,
  Classes,
  SysUtils,
  KeyPropTypes,
  PGPDialogs,
  KeyFuncs,
  pgpUI,
  pgpCL,
  pgpKeys,
  pgpEvents,
  pgpMemoryMgr,
  pgpUtilities,
  pgpOptionList,
  pgpRandomPool,
  pgpErrors, pgpPubTypes, pgpBase;

type
  TKeySize = (
    KeySize_512,
    KeySize_1024,
    KeySize_2048
  );
  TSubKeySize = (
    SubKeySize_512,
    SubKeySize_1024,
    SubKeySize_2048,
    SubKeySize_4096
  );
  TMinPassLen = 1..255;
  TMinPassQual = 1..100;

  TOnGetPassphrase = procedure(const Passphrase: PChar; const MasterKeyProps: TKeyPropsRec;
			       MinPassLen: TMinPassLen; MinPassQual: TMinPassQual;
			       var Cancel: Longbool) of Object;
  TOnGetUserNameAddress = procedure(var UserName, EmailAddress: String) of Object;
  TOnKeyGeneration = procedure(const NewUserID, NewHexID: String; Aborted: Longbool) of Object;
  TOnShowState = procedure(State: Char; var Cancel: Longbool) of Object;
  TPGPKeysGenerateCustom = Class(TComponent)
  private
    FAborted: Longbool;
    FContext: pPGPContext;
    FKeySetMain: pPGPKeySet;
    FPubKeyAlgorithm: TKeyAlgorithm;
    FCipherAlgorithm: TCipherAlgorithm;
    FExpires: Longint;
    FFastGenerate: Longbool;
    FFailNoEntropy: Longbool;
    FMinPassLen: TMinPassLen;
    FMinPassQual: TMinPassQual;
    FPassphrase: PChar;
    FSize: Longint;
    FUserID: String;
    FNewHexID: String;
    FUserName: String;
    FEmailAddress: String;
    FMasterKeyHexID: String;
    FNewKeyHexID: String;
    FNewSubKeyHexID: String;
    FKeySize: TKeySize;
    FSubKeySize: TSubKeySize;
    FParentHandle: THandle;
    FKeyDlgPrompt: String;
    FPassDlgPrompt: String;
    FOnGetPassphrase: TOnGetPassphrase;
    FOnGetUserNameAddress: TOnGetUserNameAddress;
    FOnKeyGeneration: TOnKeyGeneration;
    FOnShowState: TOnShowState;
    function InitKeyGen: PGPError;
    procedure FinitKeyGen(var Result: PGPError);
    function GetUserID: String;
    function GetKeySize: Longint;
    function GetSubKeySize: Longint;
    function GetMasterKeyHexID: Longint;
    function GetUserNameAddress: Longint;
    function GetPassphrase(OfMasterKey: Longbool): PGPError;
    function GetEntropy(IncludeSubKey: Longbool): PGPError;
    function KeyGenerate(IncludeSubKey: Longbool): PGPError;
    function SubKeyGenerate(ForOldMasterKey: Longbool): PGPError;
  protected
    procedure SetUserName(const Name: String);
    procedure SetEmailAddress(const Address: String);
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    function DHDSSKeyGenerate: Longint; virtual;
    function DHSubKeyGenerate: Longint; virtual;
    function DSAKeyGenerate: Longint; virtual;
    function RSAKeyGenerate: Longint; virtual;
    property ParentHandle: THandle
      read FParentHandle
      write FParentHandle;
  published
    property CipherAlgorithm: TCipherAlgorithm
      read FCipherAlgorithm
      write FCipherAlgorithm;
    property Expires: Longint
      read FExpires
      write FExpires;
    property FastGenerate: Longbool
      read FFastGenerate
      write FFastGenerate;
    property FailNoEntropy: Longbool
      read FFailNoEntropy
      write FFailNoEntropy;
    property MinPassLen: TMinPassLen
      read FMinPassLen
      write FMinPassLen;
    property MinPassQual: TMinPassQual
      read FMinPassQual
      write FMinPassQual;
    property UserName: String
      read FUserName
      write SetUserName;
    property EmailAddress: String
      read FEmailAddress
      write SetEmailAddress;
    property MasterKeyHexID: String
      read FMasterKeyHexID
      write FMasterKeyHexID;
    property KeySize: TKeySize
      read FKeySize
      write FKeySize;
    property SubKeySize: TSubKeySize
      read FSubKeySize
      write FSubKeySize;
    property KeyDlgPrompt: String
      read FKeyDlgPrompt
      write FKeyDlgPrompt;
    property PassDlgPrompt: String
      read FPassDlgPrompt
      write FPassDlgPrompt;
    property OnGetPassphrase: TOnGetPassphrase
      read FOnGetPassphrase
      write FOnGetPassphrase;
    property OnGetUserNameAddress: TOnGetUserNameAddress
      read FOnGetUserNameAddress
      write FOnGetUserNameAddress;
    property OnKeyGeneration: TOnKeyGeneration
      read FOnKeyGeneration
      write FOnKeyGeneration;
    property OnShowState: TOnShowState
      read FOnShowState
      write FOnShowState;
  end;

implementation

function EventHandler(Context: pPGPContext; Event: pPGPEvent; UserValue: PGPUserValue): PGPError; cdecl;
var
  Cancel	: Longbool;
begin
  Result:=0;
  Cancel:=false;
  with TPGPKeysGenerateCustom(UserValue) do begin
    case Event^.EType of
      kPGPEvent_KeyGenEvent: begin
	if TMethod(FOnShowState).Code<>nil then with Event^.Data.KeyGenData do begin
	  FOnShowState(chr(State), Cancel);
	  if Cancel then begin
	    Result:=kPGPError_UserAbort;
	    FAborted:=true;
	  end;
	end;
	ProcessMessages;
      end;
    end;
  end;
end;

constructor TPGPKeysGenerateCustom.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FFastGenerate:=true;
  FMinPassLen:=10;
  FMinPassQual:=100;
end;

destructor TPGPKeysGenerateCustom.Destroy;
begin
  inherited Destroy;
end;

function TPGPKeysGenerateCustom.InitKeyGen: PGPError;
begin
  FSize:=0;
  FUserID:='';
  FNewHexID:='';
  FAborted:=false;
  FPassphrase:=nil;
  FNewKeyHexID:='';
  FNewSubKeyHexID:='';
  Result:=InitKeyRings(FContext, FKeySetMain);
  if Result=0 then PGPclInitLibrary(FContext);
end;

procedure TPGPKeysGenerateCustom.FinitKeyGen(var Result: PGPError);
begin
  try PGPclCloseLibrary except end;
  PGPFreeData(FPassphrase);
  FreeKeyRings(FContext, FKeySetMain);
  if FAborted then Result:=kPGPError_UserAbort;
  if Assigned(FOnKeyGeneration) then FOnKeyGeneration(FUserID, FNewHexID, Result<>0);
end;

procedure TPGPKeysGenerateCustom.SetUserName(const Name: String);
var
  ErrorStr	: String;
begin
  if Name<>FUserName then begin
    if pos('@', Name)=0 then
      FUserName:=Trim(Name)
    else begin
      SetLength(ErrorStr, 255);
      PGPGetErrorString(kPGPError_BadParams, 255, PChar(ErrorStr));
      SetLength(ErrorStr, StrLen(PChar(ErrorStr)));
      MessageBox(FParentHandle, PChar(ErrorStr + ': ' + Name), 'PGP', MB_ICONERROR);
    end;
  end;
end;

procedure TPGPKeysGenerateCustom.SetEmailAddress(const Address: String);
var
  ErrorStr	: String;
begin
  if Address<>FEmailAddress then begin
    if (Address='') or (pos('@', Address)<>0) then
      FEmailAddress:=Trim(Address)
    else begin
      SetLength(ErrorStr, 255);
      PGPGetErrorString(kPGPError_BadParams, 255, PChar(ErrorStr));
      SetLength(ErrorStr, StrLen(PChar(ErrorStr)));
      MessageBox(FParentHandle, PChar(ErrorStr + ': ' + Address), 'PGP', MB_ICONERROR);
    end;
  end;
end;

function TPGPKeysGenerateCustom.GetUserID: String;
begin
  if FEmailAddress<>'' then begin
    Result:='<' + FEmailAddress + '>';
    if FUserName<>'' then Result:=FUserName + ' ' + Result;
  end
  else Result:=FUserName;
end;

function TPGPKeysGenerateCustom.GetKeySize: Longint;
begin
  Result:=0;
  case FKeySize of
    KeySize_512: Result:=512;
    KeySize_1024: Result:=1024;
    KeySize_2048: Result:=2048;
  end;
end;

function TPGPKeysGenerateCustom.GetSubKeySize: Longint;
begin
  Result:=0;
  case FSubKeySize of
    SubKeySize_512: Result:=512;
    SubKeySize_1024: Result:=1024;
    SubKeySize_2048: Result:=2048;
    SubKeySize_4096: Result:=4096;
  end;
end;

function TPGPKeysGenerateCustom.GetMasterKeyHexID: Longint;
var
  KeyPropsList	: TKeyPropsList;
begin
  KeyPropsList:=nil;
  Result:=kPGPError_BadParams;
  try
    if (FMasterKeyHexID='') or (FindKeyProps(FMasterKeyHexID,
					     KeyPropsList,
					     spgpKeyPropFlag_KeyID,
					     KeyFilterFlag_CanSign or KeyFilterFlag_DHDSS,
					     Any_Ordering)<>1) then begin
      if KeyPropsList<>nil then KeyPropsList.Clear;
      Result:=SelectKeysDialog(FKeyDlgPrompt, KeyPropsList, true, spgpKeyPropFlag_KeyID,
			       KeyFilterFlag_CanSign or KeyFilterFlag_DHDSS, FParentHandle);
      if Result=0 then FMasterKeyHexID:=KeyPropsList.Strings[0];
    end
    else FMasterKeyHexID:=KeyPropsList.Strings[0];
  finally
    KeyPropsList.Free;
  end;
end;

function TPGPKeysGenerateCustom.GetUserNameAddress: Longint;
begin
  Result:=0;
  if (FUserName='') or (FEmailAddress='') then begin
    if Assigned(FOnGetUserNameAddress) then FOnGetUserNameAddress(FUserName, FEmailAddress);
  end;
  if FUserName='' then Result:=kPGPError_BadParams;
end;

function TPGPKeysGenerateCustom.GetPassphrase(OfMasterKey: Longbool): PGPError;
var
  KeySetFound	: pPGPKeySet;
  KeyPropsRec	: TKeyPropsRec;
  KeyPropsList	: TKeyPropsList;
  Cancel	: Longbool;
begin
  Result:=0;
  Cancel:=false;
  KeySetFound:=nil;
  if Assigned(FOnGetPassphrase) then begin
    FillChar(KeyPropsRec, SizeOf(TKeyPropsRec), 0);
    KeyPropsList:=nil;
    try
      if OfMasterKey then begin
	if FindKeyProps(FMasterKeyHexID, KeyPropsList, spgpKeyPropFlag_IDComplete,
			KeyFilterFlag_CanSign or KeyFilterFlag_DHDSS, Any_Ordering)=1 then begin
	  KeyPropsRec:=pKeyPropsRec(KeyPropsList[0])^;
	end;
      end;
      FPassphrase:=PGPNewSecureData(PGPGetDefaultMemoryMgr, 256, kPGPMemoryMgrFlags_Clear);
      repeat
	FOnGetPassphrase(FPassphrase, KeyPropsRec, FMinPassLen, FMinPassQual, Cancel);
	if Cancel then begin
	  Result:=kPGPError_UserAbort;
	  Break;
	end;
      until (StrLen(FPassphrase)>=FMinPassLen) and (PGPEstimatePassphraseQuality(FPassphrase)>=FMinPassQual);
    finally
      KeyPropsList.Free;
    end;
  end
  else begin
    if OfMasterKey then begin
      Result:=GetKeySetByAnyIDs(FContext, FKeySetMain, FMasterKeyHexID, KeySetFound);
      if Result<>0 then Exit;
      try
	Result:=KeyPassphraseDialog(FContext, KeySetFound, FPassphrase, FPassDlgPrompt, FParentHandle);
      finally
	PGPFreeKeySet(KeySetFound);
      end;
    end
    else begin
      Result:=ConfirmationPassphraseDialog(FContext, FPassphrase, FMinPassLen, FMinPassQual, true,
					   FPassDlgPrompt, FParentHandle);
    end;
  end;
end;

function TPGPKeysGenerateCustom.GetEntropy(IncludeSubKey: Longbool): PGPError;
var
  OptionList	: pPGPOptionList;
  EntropyNeeded	: PGPUInt32;
begin
  OptionList:=nil;
  Result:=PGPBuildOptionList(FContext, OptionList,
    [
      PGPOKeyGenParams(FContext, PGPPublicKeyAlgorithm(FPubKeyAlgorithm), FSize),
      PGPOKeyGenFast(FContext, PGPBoolean(FFastGenerate))
    ]);
  if Result<>0 then Exit;
  try
    EntropyNeeded:=PGPGetKeyEntropyNeeded(FContext, OptionList, PGPOLastOption(FContext));
  finally
    PGPFreeOptionList(OptionList);
  end;
  if IncludeSubKey then begin
    OptionList:=nil;
    Result:=PGPBuildOptionList(FContext, OptionList,
      [
	PGPOKeyGenParams(FContext, PGPPublicKeyAlgorithm(KeyAlgorithm_DH), FSize),
	PGPOKeyGenFast(FContext, PGPBoolean(FFastGenerate))
      ]);
    if Result<>0 then Exit;
    try
      EntropyNeeded:=EntropyNeeded + PGPGetKeyEntropyNeeded(FContext, OptionList, PGPOLastOption(FContext));
    finally
      PGPFreeOptionList(OptionList);
    end;
  end;
  if EntropyNeeded>=PGPGlobalRandomPoolGetEntropy then begin
    if FFailNoEntropy then
      Result:=kPGPError_OutOfEntropy
    else Result:=CollectRandomDataDialog(FContext, EntropyNeeded, FParentHandle);
  end;
end;

function TPGPKeysGenerateCustom.KeyGenerate(IncludeSubKey: Longbool): PGPError;
var
  OptionList	: pPGPOptionList;
  NewKey	: pPGPKey;
begin
  OptionList:=nil;
  case FCipherAlgorithm of
    CipherAlgorithm_None: begin
      case FPubKeyAlgorithm of
	KeyAlgorithm_RSA: FCipherAlgorithm:=CipherAlgorithm_IDEA;
	KeyAlgorithm_DSS: FCipherAlgorithm:=CipherAlgorithm_CAST5;
      end;
    end;
  end;
  Result:=PGPBuildOptionList(FContext, OptionList,
    [
      PGPOKeySetRef(FContext, FKeySetMain),
      PGPOPassphrase(FContext, FPassphrase),
      PGPOKeyGenName(FContext, PChar(FUserID), Length(FUserID)),
      PGPOKeyGenParams(FContext, PGPPublicKeyAlgorithm(FPubKeyAlgorithm), FSize),
      PGPOPreferredAlgorithms(FContext, PGPCipherAlgorithm(FCipherAlgorithm), 1),
      PGPOKeyGenFast(FContext, PGPBoolean(FFastGenerate)),
      PGPOExpiration(FContext, FExpires),
      PGPOEventHandler(FContext, EventHandler, Self)
    ]);
  if Result<>0 then Exit;
  try
    Result:=PGPGenerateKey(FContext, NewKey, OptionList, PGPOLastOption(FContext));
    if Result<>0 then Exit;
    if not IncludeSubKey then begin
      if longbool(PGPKeySetNeedsCommit(FKeySetMain)) then begin
	Result:=PGPCommitKeyRingChanges(FKeySetMain);
	if Result<>0 then Exit;
	PGPclNotifyKeyringChanges(GetCurrentProcessID);
      end;
    end;
    FNewKeyHexID:=GetKeyPropKeyID(NewKey);
  finally
    PGPFreeOptionList(OptionList);
  end;
end;

function TPGPKeysGenerateCustom.SubKeyGenerate(ForOldMasterKey: Longbool): PGPError;
var
  OptionList	: pPGPOptionList;
  MasterKey	: pPGPKey;
  NewSubKey	: pPGPKey;
begin
  OptionList:=nil;
  Result:=GetKeyByHexID(FKeySetMain, FMasterKeyHexID, MasterKey);
  if Result<>0 then Exit;
  Result:=PGPBuildOptionList(FContext, OptionList,
    [
      PGPOKeyGenMasterKey(FContext, MasterKey),
      PGPOPassphrase(FContext, FPassphrase),
      PGPOKeyGenParams(FContext, PGPPublicKeyAlgorithm(FPubKeyAlgorithm), FSize),
      PGPOKeyGenFast(FContext, PGPBoolean(FFastGenerate)),
      PGPOExpiration(FContext, FExpires),
      PGPOEventHandler(FContext, EventHandler, Self)
    ]);
  if Result<>0 then Exit;
  try
    Result:=PGPGenerateSubKey(FContext, NewSubKey, OptionList, PGPOLastOption(FContext));
    if Result<>0 then Exit;
    if longbool(PGPKeySetNeedsCommit(FKeySetMain)) then begin
      Result:=PGPCommitKeyRingChanges(FKeySetMain);
      if Result<>0 then Exit;
      PGPclNotifyKeyringChanges(GetCurrentProcessID);
    end;
    if ForOldMasterKey then FUserID:=GetKeyPropUserID(MasterKey);
    FNewSubKeyHexID:=GetSubKeyPropKeyID(NewSubKey);
  finally
    PGPFreeOptionList(OptionList);
  end;
end;

function TPGPKeysGenerateCustom.DHDSSKeyGenerate: Longint;
begin
  Result:=InitKeyGen;
  try
    if Result<>0 then Exit;
    Result:=GetUserNameAddress;
    if Result<>0 then Exit;
    if FKeySize>KeySize_1024 then FKeySize:=KeySize_1024;
    FPubKeyAlgorithm:=KeyAlgorithm_DSS;
    FUserID:=GetUserID;
    FSize:=GetKeySize;
    Result:=GetPassphrase(false);
    if Result<>0 then Exit;
    Result:=GetEntropy(true);
    if Result<>0 then Exit;
    Result:=KeyGenerate(true);
    FPubKeyAlgorithm:=KeyAlgorithm_DH;
    FMasterKeyHexID:=FNewKeyHexID;
    FSize:=GetSubKeySize;
    Result:=SubKeyGenerate(false);
    if Result=0 then FNewHexID:=FNewKeyHexID;
  finally
    FinitKeyGen(Result);
  end;
end;

function TPGPKeysGenerateCustom.DHSubKeyGenerate: Longint;
begin
  Result:=InitKeyGen;
  try
    if Result<>0 then Exit;
    Result:=GetMasterKeyHexID;
    if Result<>0 then Exit;
    FPubKeyAlgorithm:=KeyAlgorithm_DH;
    FSize:=GetSubKeySize;
    Result:=GetPassphrase(true);
    if Result<>0 then Exit;
    Result:=GetEntropy(false);
    if Result<>0 then Exit;
    Result:=SubKeyGenerate(true);
    if Result=0 then FNewHexID:=FNewSubKeyHexID;
  finally
    FinitKeyGen(Result);
  end;
end;

function TPGPKeysGenerateCustom.DSAKeyGenerate: Longint;
begin
  Result:=InitKeyGen;
  try
    if Result<>0 then Exit;
    Result:=GetUserNameAddress;
    if Result<>0 then Exit;
    if FKeySize>KeySize_1024 then FKeySize:=KeySize_1024;
    FPubKeyAlgorithm:=KeyAlgorithm_DSS;
    FUserID:=GetUserID;
    FSize:=GetKeySize;
    Result:=GetPassphrase(false);
    if Result<>0 then Exit;
    Result:=GetEntropy(false);
    if Result<>0 then Exit;
    Result:=KeyGenerate(false);
    if Result=0 then FNewHexID:=FNewKeyHexID;
  finally
    FinitKeyGen(Result);
  end;
end;

function TPGPKeysGenerateCustom.RSAKeyGenerate: Longint;
begin
  Result:=InitKeyGen;
  try
    if Result<>0 then Exit;
    Result:=GetUserNameAddress;
    if Result<>0 then Exit;
    FPubKeyAlgorithm:=KeyAlgorithm_RSA;
    FUserID:=GetUserID;
    FSize:=GetKeySize;
    Result:=GetPassphrase(false);
    if Result<>0 then Exit;
    Result:=GetEntropy(false);
    if Result<>0 then Exit;
    Result:=KeyGenerate(false);
    if Result=0 then FNewHexID:=FNewKeyHexID;
  finally
    FinitKeyGen(Result);
  end;
end;

end.

