{$J+,Z4}
unit PGPEncode;

{------------------------------------------------------------------------------}
{                                                                              }
{                 This unit is partly based on Steve Heller's                  }
{         SPGP sources available from http://www.oz.net/~srheller/spgp/        }
{                                                                              }
{                Portions created by Michael in der Wiesche are                }
{                 Copyright (C) 2001 by Michael in der Wiesche                 }
{                                                                              }
{------------------------------------------------------------------------------}

interface

uses
  Windows,
  Classes,
  SysUtils,
  KeyPropTypes,
  PGPDialogs,
  PrefFuncs,
  KeyFuncs,
  pgpCL,
  pgpTLS,
  pgpKeys,
  pgpEvents,
  pgpMemoryMgr,
  pgpUtilities,
  pgpOptionList,
  pgpErrors, pgpPubTypes, pgpBase;

type
  TSignAlgorithm = (
    HashAlgorithm_Default,
    HashAlgorithm_MD5,
    HashAlgorithm_SHA,
    HashAlgorithm_RIPEMD160
  );
  TConventionalAlgorithm = (
    CipherAlgorithm_IDEA,
    CipherAlgorithm_3DES,
    CipherAlgorithm_CAST5
  );
  TFormatOption = (
    Format_Armor,
    Format_MIME,
    Format_Textmode
  );
  TFormatOptions = Set of TFormatOption;
  TOnGetInputFileName = procedure(var SuggestedName: String) of Object;
  TOnGetOutputFileName = procedure(var SuggestedName: String) of Object;
  TOnGetPassphrase = procedure(const Passphrase: PChar;
			       const SigningKeyList: TKeyPropsList;
			       var SelectedKey: Longint) of Object;
  TOnShowProgress = procedure(BytesProcessed, BytesTotal: Longint) of Object;
  TOnWipePassphrase = procedure(const Passphrase: PChar) of Object;
  TPGPEncodeCustom = class(TComponent)
  private
    // internal
    FContext: pPGPContext;
    FKeySetMain: pPGPKeySet;
    FtlsContext: pPGPTLSContext;
    FAllocatedOutputBuffer: PChar;
    FActualOutputSize: PGPSize;
    FInputSize: PGPSize;
    FInputFileName: String;
    FOutputBuffer: String;
    FPassphrase: PChar;
    FEncryptKeySet: pPGPKeySet;
    FClear: Longbool;
    FCompress: Longbool;
    FConventional: Longbool;
    FDetachedSign: Longbool;
    FEncrypt: Longbool;
    FSign: Longbool;
    // properties
    FComment: String;
    FConventionalAlgorithm: TConventionalAlgorithm;
    FEyesOnly: Longbool;
    FFileOutput: Longbool;
    FFormatOptions: TFormatOptions;
    FMimeBodyOffset: Longint;
    FMimeSeparator: String;
    FOmitMimeVersion: Longbool;
    FParentHandle: THandle;
    FProgressInterval: Cardinal;
    FSignAlgorithm: TSignAlgorithm;
    FEncryptKeyIDs: TStrings;
    FSignKeyID: String;
    FKeyDlgPrompt: String;
    FPassDlgPrompt: String;
    FOutputFileName: String;
    // events
    FOnGetInputFileName: TOnGetInputFileName;
    FOnGetOutputFileName: TOnGetOutputFileName;
    FOnGetPassphrase: TOnGetPassphrase;
    FOnShowProgress: TOnShowProgress;
    FOnWipePassphrase: TOnWipePassphrase;
    procedure SetEncryptKeyIDs(const Value: TStrings);
    procedure SetClear(Value: Longbool);
    procedure SetConventional(Value: Longbool);
    procedure SetEncrypt(Value: Longbool);
    procedure SetSign(Value: Longbool);
    function  InitEncode: PGPError;
    procedure FinitEncode;
    function  WriteOutputFile: PGPError;
    function  GetEncryptKeyIDs(const EncryptKeyIDs: TStrings; SignKeyID: String): PGPError;
    function  GetOptionList(var OptionList: pPGPOptionList): PGPError;
    function  SetOutputOption(var OptionList: pPGPOptionList): PGPError;
    function  Encode(const Input: String; IsFile: Longbool): Longint;
  protected
    property OutputBuffer: String
      read FOutputBuffer;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    function KeyEncryptBuffer(const DataBuffer: String; Sign: Longbool): Longint; virtual;
    function KeyEncryptFile(const FileName: String; Sign: Longbool): Longint; virtual;
    function ConventionalEncryptBuffer(const DataBuffer: String): Longint; virtual;
    function ConventionalEncryptFile(const FileName: String): Longint; virtual;
    function ClearSignBuffer(const DataBuffer: String): Longint; virtual;
    function ClearSignFile(const FileName: String): Longint; virtual;
    function ArmorBuffer(const DataBuffer: String): Longint; virtual;
    function ArmorFile(const FileName: String): Longint; virtual;
    function DetachedSignBuffer(const DataBuffer: String): Longint; virtual;
    function DetachedSignFile(const FileName: String): Longint; virtual;
    property MimeBodyOffset: Longint
      read FMimeBodyOffset;
    property MimeSeparator: String
      read FMimeSeparator;
    property ParentHandle: THandle
      read FParentHandle
      write FParentHandle;
  published
    property Comment: String
      read FComment
      write FComment;
    property ConventionalAlgorithm: TConventionalAlgorithm
      read FConventionalAlgorithm
      write FConventionalAlgorithm;
    property EyesOnly: Longbool
      read FEyesOnly
      write FEyesOnly;
    property FileOutput: Longbool
      read FFileOutput
      write FFileOutput;
    property FormatOptions: TFormatOptions
      read FFormatOptions
      write FFormatOptions;
    property OmitMimeVersion: Longbool
      read FOmitMimeVersion
      write FOmitMimeVersion;
    property ProgressInterval: Cardinal
      read FProgressInterval
      write FProgressInterval;
    property SignAlgorithm: TSignAlgorithm
      read FSignAlgorithm
      write FSignAlgorithm;
    property EncryptKeyIDs: TStrings
      read FEncryptKeyIDs
      write SetEncryptKeyIDs;
    property SignKeyID: String
      read FSignKeyID
      write FSignKeyID;
    property KeyDlgPrompt: String
      read FKeyDlgPrompt
      write FKeyDlgPrompt;
    property PassDlgPrompt: String
      read FPassDlgPrompt
      write FPassDlgPrompt;
    property OutputFileName: String
      read FOutputFileName
      write FOutputFileName;
    property OnGetInputFileName: TOnGetInputFileName
      read FOnGetInputFileName
      write FOnGetInputFileName;
    property OnGetOutputFileName: TOnGetOutputFileName
      read FOnGetOutputFileName
      write FOnGetOutputFileName;
    property OnGetPassphrase: TOnGetPassphrase
      read FOnGetPassphrase
      write FOnGetPassphrase;
    property OnShowProgress: TOnShowProgress
      read FOnShowProgress
      write FOnShowProgress;
    property OnWipePassphrase: TOnWipePassphrase
      read FOnWipePassphrase
      write FOnWipePassphrase;
  end;

implementation

function EventHandler(Context: pPGPContext; Event: pPGPEvent; UserValue: PGPUserValue): PGPError; cdecl;
begin
  Result:=0;
  with TPGPEncodeCustom(UserValue) do begin
    case Event^.EType of
      kPGPEvent_NullEvent:	if not (FDetachedSign or FClear) then begin
				  if TMethod(FOnShowProgress).Code<>nil then begin
				    // BytesTotal always stays 0 => use FInputSize
				    with Event^.Data.NullData do FOnShowProgress(BytesWritten, FInputSize);
				  end;
				  ProcessMessages;
				end;
      kPGPEvent_InitialEvent:	;
      kPGPEvent_FinalEvent:	;
      kPGPEvent_ErrorEvent:	Result:=Event^.Data.ErrorData.Error;
      kPGPEvent_WarningEvent:	;
    end;
  end;
end;

constructor TPGPEncodeCustom.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FEncryptKeyIDs:=TStringList.Create;
  FProgressInterval:=1000;
end;

destructor TPGPEncodeCustom.Destroy;
begin
  EncryptKeyIDs.Free;
  inherited Destroy;
end;

procedure TPGPEncodeCustom.SetEncryptKeyIDs(const Value: TStrings);
begin
  FEncryptKeyIDs.Assign(Value);
end;

procedure TPGPEncodeCustom.SetClear(Value: Longbool);
begin
  if Value<>FClear then FClear:=Value;
  if FClear then begin
    FSign:=true;
    FEncrypt:=false;
    FCompress:=false;
    FConventional:=false;
    FDetachedSign:=false;
    Include(FFormatOptions, Format_Armor);
    Include(FFormatOptions, Format_Textmode);
  end;
end;

procedure TPGPEncodeCustom.SetConventional(Value: Longbool);
begin
  if Value<>FConventional then FConventional:=Value;
  if FConventional then begin
    FDetachedSign:=false;
    FEncrypt:=false;
    FClear:=false;
    FSign:=false;
  end;
end;

procedure TPGPEncodeCustom.SetEncrypt(Value: Longbool);
begin
  if Value<>FEncrypt then FEncrypt:=Value;
  if FEncrypt then begin
    FDetachedSign:=false;
    FConventional:=false;
    FClear:=false;
  end;
end;

procedure TPGPEncodeCustom.SetSign(Value: Longbool);
begin
  if Value<>FSign then FSign:=Value;
  if FSign then FConventional:=false;
end;

function TPGPEncodeCustom.InitEncode: PGPError;
begin
  FOutputBuffer:='';
  FActualOutputSize:=0;
  FAllocatedOutputBuffer:=nil;
  Result:=KeyDlgInit(FContext, FtlsContext, FKeySetMain);
end;

procedure TPGPEncodeCustom.FinitEncode;
begin
  KeyDlgFree(FContext, FtlsContext, FKeySetMain);
end;

function TPGPEncodeCustom.WriteOutputFile: PGPError;
var
  OutputFile	: THandle;
  BytesWritten	: DWord;
begin
  OutputFile:=CreateFile(PChar(FOutputFileName), GENERIC_WRITE, 0, nil, CREATE_ALWAYS,
			 FILE_ATTRIBUTE_ARCHIVE or FILE_FLAG_WRITE_THROUGH, 0);
  if OutputFile<>INVALID_HANDLE_VALUE then begin
    if not WriteFile(OutputFile, FAllocatedOutputBuffer[0], FActualOutputSize, BytesWritten, nil)
    or (BytesWritten<>DWord(FActualOutputSize)) then
      Result:=kPGPError_WriteFailed
    else Result:=kPGPError_NoErr;
    CloseHandle(OutputFile);
  end
  else Result:=kPGPError_CantOpenFile;
end;

function TPGPEncodeCustom.GetEncryptKeyIDs(const EncryptKeyIDs: TStrings; SignKeyID: String): PGPError;
var
  PrefsRec	: TPreferenceRec;
  RecipientKeys	: pPGPKeySet;
  Armor		: PGPUInt32;
  MIME		: PGPUInt32;
  Textmode	: PGPUInt32;
  KeyPropsList	: TKeyPropsList;
  KeyCount	: PGPUInt32;
begin
  RecipientKeys:=nil;
  if (SignKeyID='') and (GetPreferences(PrefsRec, PrefsFlag_DefaultKeyID)=0) then SignKeyID:=PrefsRec.DefaultKeyHexID;
  try
    Result:=GetKeySetByAnyIDs(FContext, FKeySetMain, EncryptKeyIDs.CommaText, RecipientKeys);
    if Result<>0 then begin
      EncryptKeyIDs.Clear;
      Armor:=ord(Format_Armor in FFormatOptions);
      MIME:=ord(Format_MIME in FFormatOptions);
      Textmode:=ord(Format_Textmode in FFormatOptions);
      Result:=RecipientsDialog(FContext, FKeySetMain, PGPBoolean(true), Armor, PGPUInt32(FEyesOnly), MIME, Textmode,
			       SignKeyID, RecipientKeys, FKeyDlgPrompt, FParentHandle);
      if Result=0 then begin
	FFormatOptions:=[];
	if Armor<>0 then Include(FFormatOptions, Format_Armor);
	if MIME<>0 then Include(FFormatOptions, Format_MIME);
	if Textmode<>0 then Include(FFormatOptions, Format_Textmode);
	KeyPropsList:=nil;
	try
	  KeyCount:=GetKeySetProps(FContext,
				   RecipientKeys,
				   KeyPropsList,
				   spgpKeyPropFlag_KeyID,
				   KeyFilterFlag_CanEncrypt,
				   UserID_Ordering);
	  if KeyCount>0 then EncryptKeyIDs.Text:=TrimRight(KeyPropsList.Text);
	finally
	  KeyPropsList.Free;
	end;
      end;
    end
    else if Trim(EncryptKeyIDs.Text)<>'' then EncryptKeyIDs.Add(SignKeyID);
  finally
    PGPFreeKeySet(RecipientKeys);
  end;
end;

function TPGPEncodeCustom.GetOptionList(var OptionList: pPGPOptionList): PGPError;
var
  KeyPropsList	: TKeyPropsList;
  SelectedKey	: PGPUInt32;
  PrefsRec	: TPreferenceRec;
  SigningKey	: pPGPKey;
begin
  KeyPropsList:=TKeyPropsList.Create(0, spgpKeyPropFlag_IDComplete);
  try
    if FDetachedSign then begin
      Result:=PGPBuildOptionList(FContext, OptionList,
	[
	  PGPOSendNullEvents(FContext, FProgressInterval),
	  PGPOEventHandler(FContext, EventHandler, Self),
	  PGPOArmorOutput(FContext, PGPBoolean(Format_Armor in FFormatOptions)),
	  PGPODetachedSig(FContext, PGPOLastOption(FContext), nil),
	  PGPOCommentString(FContext, PChar(FComment)),
	  PGPOVersionString(FContext, PChar(MyVersion))
	]);
    end
    else begin
      Result:=PGPBuildOptionList(FContext, OptionList,
	[
	  PGPOSendNullEvents(FContext, FProgressInterval),
	  PGPOEventHandler(FContext, EventHandler, Self),
	  PGPOClearSign(FContext, PGPBoolean(FClear)),
	  PGPOCompression(FContext, PGPBoolean(FCompress)),
	  PGPOCommentString(FContext, PChar(FComment)),
	  PGPOVersionString(FContext, PChar(MyVersion))
	]);
    end;
    if Result<>0 then Exit;
    if FConventional then begin
      Result:=PGPAppendOptionList(OptionList,
	[PGPOCipherAlgorithm(FContext, PGPCipherAlgorithm(succ(FConventionalAlgorithm)))]);
      if Result<>0 then Exit;
      if Assigned(FOnGetPassphrase) then begin
	SelectedKey:=-1;
	FPassphrase:=PGPNewSecureData(PGPGetDefaultMemoryMgr, 256, kPGPMemoryMgrFlags_Clear);
	FOnGetPassphrase(FPassphrase, KeyPropsList, SelectedKey);
      end
      else begin
	Result:=ConvEncPassphraseDialog(FContext, FPassphrase, FPassDlgPrompt, FParentHandle);
	if Result<>0 then Exit;
      end;
      Result:=PGPAppendOptionList(OptionList,
	[PGPOConventionalEncrypt(FContext, PGPOPassphrase(FContext, FPassphrase), PGPOLastOption(FContext))]);
      if Result<>0 then Exit;
    end
    else begin
      if FSign or FDetachedSign then begin
	SigningKey:=nil;
	if Assigned(FOnGetPassphrase) then begin
	  if FindKeyProps('', KeyPropsList, spgpKeyPropFlag_IDComplete,
			  KeyFilterFlag_CanSign, UserID_Ordering)>0 then begin
	    Result:=GetHexIDByAnyID(FContext, FKeySetMain, FSignKeyID, FSignKeyID);
	    if (Result<>1) and (GetPreferences(PrefsRec, PrefsFlag_DefaultKeyID)=0) then begin
	      FSignKeyID:=PrefsRec.DefaultKeyHexID;
	      Result:=1;
	    end;
	  end;
	  if Result=1 then begin
	    with KeyPropsList do if Count>0 then begin
	      SelectedKey:=IndexOf(FSignKeyID);
	      if SelectedKey>0 then Move(SelectedKey, 0);
	    end;
	  end
	  else begin
	    Result:=kPGPError_KeyUnusableForSignature;
	    Exit;
	  end;
	  SelectedKey:=0;
	  FPassphrase:=PGPNewSecureData(PGPGetDefaultMemoryMgr, 256, kPGPMemoryMgrFlags_Clear);
	  FOnGetPassphrase(FPassphrase, KeyPropsList, SelectedKey);
	  if (KeyPropsList<>nil) and (SelectedKey>=0) and (SelectedKey<KeyPropsList.Count) then begin
	    FSignKeyID:=KeyPropsList.Strings[SelectedKey];
	    Result:=GetKeyByHexID(FKeySetMain, PChar(FSignKeyID), SigningKey);
	  end
	  else Result:=kPGPError_KeyUnusableForSignature;
	end
	else begin
	  Result:=SigningPassphraseDialog(FContext, FKeySetMain, SigningKey, FPassphrase, FSignKeyID,
					  false, FPassDlgPrompt, FParentHandle);
	end;
	if Result<>0 then Exit;
	if FSignAlgorithm=HashAlgorithm_Default then begin
	  case GetKeyPropAlg(SigningKey) of
	    KeyAlgorithm_RSA..KeyAlgorithm_RSASignOnly: FSignAlgorithm:=HashAlgorithm_MD5;
	    KeyAlgorithm_DH..KeyAlgorithm_DHDSS: FSignAlgorithm:=HashAlgorithm_SHA;
	  else
	    Result:=kPGPError_UnknownPublicKeyAlgorithm;
	    Exit;
	  end;
	end;
	Result:=PGPAppendOptionList(OptionList,
	  [
	    PGPOSignWithKey(FContext, SigningKey, PGPOPassphrase(FContext, FPassphrase), PGPOLastOption(FContext)),
	    PGPOHashAlgorithm(FContext, PGPHashAlgorithm(FSignAlgorithm))
	  ]);
	if Result<>0 then Exit;
      end;
      if SigningKey<>nil then FSignKeyID:=GetKeyPropKeyID(SigningKey);
      if FEncrypt then begin
	Result:=GetEncryptKeyIDs(FEncryptKeyIDs, FSignKeyID);
	if Result<>0 then Exit;
	Result:=GetKeySetByAnyIDs(FContext, FKeySetMain, FEncryptKeyIDs.CommaText, FEncryptKeySet);
	if Result<>0 then Exit;
	Result:=PGPAppendOptionList(OptionList, [PGPOEncryptToKeySet(FContext, FEncryptKeySet)]);
	if Result<>0 then Exit;
      end;
      if not FDetachedSign then begin
	Result:=PGPAppendOptionList(OptionList,
	  [
	    PGPOForYourEyesOnly(FContext, PGPBoolean(FEyesOnly)),
	    PGPOArmorOutput(FContext, PGPBoolean(Format_Armor in FFormatOptions)),
	    PGPODataIsASCII(FContext, PGPBoolean(Format_Textmode in FFormatOptions))
	  ]);
	if Result<>0 then Exit;
	if Format_Mime in FFormatOptions then begin
	  SetLength(FMimeSeparator, SizeOf(TMimeSeparator));
	  FMimeBodyOffset:=0;
	  FMimeSeparator[1]:=#0;
	  Result:=PGPAppendOptionList(OptionList,
	    [
	      PGPOPGPMIMEEncoding(FContext, PGPBoolean(true), PGPSize(FMimeBodyOffset), @FMimeSeparator[1]),
	      PGPOOmitMIMEVersion(fContext, PGPBoolean(FOmitMimeVersion))
	    ]);
	  if Result<>0 then Exit;
	end;
      end;
    end;
  finally
    KeyPropsList.Free;
  end;
end;

function TPGPEncodeCustom.SetOutputOption(var OptionList: pPGPOptionList): PGPError;
begin
  if FFileOutput then begin
    if (FOutputFileName='') or FileExists(FOutputFileName) then begin
      if FOutputFileName='' then begin
	if FDetachedSign then
	  FOutputFileName:=FInputFileName + '.sig'
	else FOutputFileName:=FInputFileName + '.pgp';
	if Assigned(FOnGetOutputFileName) then
	  FOnGetOutputFileName(FOutputFileName)
	else FOutputFileName:='';
      end;
    end;
    if FOutputFileName='' then begin
      Result:=kPGPError_FileNotFound;
      Exit;
    end;
  end;
  Result:=PGPAppendOptionList(OptionList,
    [PGPOAllocatedOutputBuffer(FContext, FAllocatedOutputBuffer, -1, FActualOutputSize)]);
end;

function TPGPEncodeCustom.Encode(const Input: String; IsFile: Longbool): Longint;
var
  OptionList	: pPGPOptionList;
  FileSpec	: pPGPFileSpec;
  hInFile	: THandle;
begin
  Result:=0;
  if IsFile then begin
    FInputFileName:=Input;
    if not FileExists(FInputFileName) then begin
      if Assigned(FOnGetInputFileName) then begin
	FOnGetInputFileName(FInputFileName);
	if FInputFileName='' then Result:=kPGPError_FileNotFound;
      end
      else Result:=kPGPError_FileNotFound;
    end;
  end
  else if Input='' then Result:=kPGPError_ItemNotFound;
  if Result=0 then begin
    FileSpec:=nil;
    OptionList:=nil;
    FEncryptKeySet:=nil;
    Result:=InitEncode;
    try
      if Result<>0 then Exit;
      if IsFile then begin
	Result:=PGPNewFileSpecFromFullPath(FContext, PChar(FInputFileName), FileSpec);
	if Result<>0 then Exit;
      end;
      try
	FPassphrase:=nil;
	Result:=GetOptionList(OptionList);
	try
	  if Result<>0 then Exit;
	  if IsFile then begin
	    hInFile:=FileOpen(FInputFileName, fmOpenRead or fmShareDenyNone);
	    if hInFile<>INVALID_HANDLE_VALUE then begin
	      FInputSize:=GetFileSize(hInFile, nil);
	      FileClose(hInFile);
	    end
	    else FInputSize:=-1;
	    if (FInputSize=0) or (FInputSize=-1) then begin
	      Result:=kPGPError_ReadFailed;
	      Exit;
	    end;
	    Result:=PGPAppendOptionList(OptionList, [PGPOInputFile(FContext, FileSpec)]);
	  end
	  else begin
	    FInputSize:=Length(Input);
	    Result:=PGPAppendOptionList(OptionList, [PGPOInputBuffer(FContext, @Input[1], Length(Input))]);
	  end;
	  if Result<>0 then Exit;
	  Result:=SetOutputOption(OptionList);
	  if Result<>0 then Exit;
	  Result:=pgpEvents.PGPEncode(FContext, OptionList, PGPOLastOption(FContext));
	  try
	    if (Result=0) and (FActualOutputSize<>0) then begin
	      if FFileOutput then
		Result:=WriteOutputFile
	      else begin
		SetLength(FOutputBuffer, FActualOutputSize);
		Move(FAllocatedOutputBuffer[0], FOutputBuffer[1], FActualOutputSize);
	      end;
	    end;
	  except
	    on EOutOfMemory do Result:=kPGPError_OutOfMemory;
	  end;
	finally
	  try
	    SetLength(FMimeSeparator, StrLen(PChar(FMimeSeparator)));
	    PGPFreeOptionList(OptionList);
	    PGPFreeKeySet(FEncryptKeySet);
	    if Assigned(FOnWipePassphrase) then FOnWipePassphrase(FPassphrase);
	  finally
	    PGPFreeData(FAllocatedOutputBuffer);
	    PGPFreeData(FPassphrase);
	    FPassphrase:=nil;
	  end;
	end;
      finally
	PGPFreeFileSpec(FileSpec);
      end;
    finally
      FinitEncode;
    end;
  end;
end;

function TPGPEncodeCustom.KeyEncryptBuffer(const DataBuffer: String; Sign: Longbool): Longint;
begin
  SetSign(Sign);
  SetEncrypt(true);
  FCompress:=true;
  Result:=Encode(DataBuffer, false);
end;

function TPGPEncodeCustom.KeyEncryptFile(const FileName: String; Sign: Longbool): Longint;
begin
  SetSign(Sign);
  SetEncrypt(true);
  FCompress:=true;
  Result:=Encode(FileName, true);
end;

function TPGPEncodeCustom.ConventionalEncryptBuffer(const DataBuffer: String): Longint;
begin
  FCompress:=true;
  SetConventional(true);
  Result:=Encode(DataBuffer, false);
end;

function TPGPEncodeCustom.ConventionalEncryptFile(const FileName: String): Longint;
begin
  FCompress:=true;
  SetConventional(true);
  Result:=Encode(FileName, true);
end;

function TPGPEncodeCustom.ClearSignBuffer(const DataBuffer: String): Longint;
begin
  SetClear(true);
  Result:=Encode(DataBuffer, false);
end;

function TPGPEncodeCustom.ClearSignFile(const FileName: String): Longint;
begin
  SetClear(true);
  Result:=Encode(FileName, true);
end;

function TPGPEncodeCustom.ArmorBuffer(const DataBuffer: String): Longint;
begin
  FClear:=false;
  FCompress:=true;
  FDetachedSign:=false;
  FEncrypt:=false;
  FSign:=false;
  Include(FFormatOptions, Format_Armor);
  Result:=Encode(DataBuffer, false);
end;

function TPGPEncodeCustom.ArmorFile(const FileName: String): Longint;
begin
  FClear:=false;
  FCompress:=true;
  FDetachedSign:=false;
  FEncrypt:=false;
  FSign:=false;
  Include(FFormatOptions, Format_Armor);
  Result:=Encode(FileName, true);
end;

function TPGPEncodeCustom.DetachedSignBuffer(const DataBuffer: String): Longint;
begin
  FConventional:=false;
  FEncrypt:=false;
  FSign:=false;
  FDetachedSign:=true;
  Result:=Encode(DataBuffer, false);
end;

function TPGPEncodeCustom.DetachedSignFile(const FileName: String): Longint;
begin
  FConventional:=false;
  FEncrypt:=false;
  FSign:=false;
  FDetachedSign:=true;
  Result:=Encode(FileName, true);
end;

end.

