{$J+,Z4}
unit pgpMemoryMgr;

{**********************************************************************************}
{                                                                                  }
{ The contents of this file are subject to the Mozilla Public License Version 1.1  }
{ (the "License"); you may not use this file except in compliance with the         }
{ License. You may obtain a copy of the License at http://www.mozilla.org/MPL/.    }
{                                                                                  }
{ Software distributed under the License is distributed on an "AS IS" basis,       }
{ WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License for the }
{ specific language governing rights and limitations under the License.            }
{                                                                                  }
{ The Original Code is pgpMemoryMgr.pas based on pgpMemoryMgr.h and pgpUtilities.h }
{ from the PGP 6.5.1i sources which are Copyright (C) Network Associates Inc.      }
{ and affiliated companies.                                                        }
{                                                                                  }
{ The Initial Developer of the Original Code is Michael in der Wiesche.            }
{                                                                                  }
{ Portions created by Michael in der Wiesche are Copyright (C) 2001                }
{ Michael in der Wiesche. All Rights Reserved.                                     }
{                                                                                  }
{**********************************************************************************}

interface

uses
  pgpPubTypes, pgpBase;

type
  pPGPMemoryMgr = ^TPGPMemoryMgr;
  TPGPMemoryMgr = Pointer;
const
  kInvalidPGPMemoryMgr = nil;

type
  PGPMemoryMgrFlags = PGPFlags;
const
  kPGPMemoryMgrFlags_None	= 0;
  kPGPMemoryMgrFlags_Clear	= 1;

type
  { type declarations for PGPNewMemoryMgrStruct below }
  PGPMemoryMgrAllocationProc = function(Mgr: pPGPMemoryMgr; UserValue: PGPUserValue;
					RequestSize: PGPSize; Flags: PGPMemoryMgrFlags): Pointer;
  // realloc not be implemented using PGPNewData()
  PGPMemoryMgrReallocationProc = function(Mgr: pPGPMemoryMgr; UserValue: PGPUserValue;
					  var Allocation: Pointer; NewAllocationSize: PGPSize;
					  Flags: PGPMemoryMgrFlags; ExistingSize: PGPSize): PGPError;
  PGPMemoryMgrDeallocationProc = function(Mgr: pPGPMemoryMgr; UserValue: PGPUserValue;
					  Allocation: Pointer; AllocationSize: PGPSize): PGPError;
  PGPMemoryMgrSecureAllocationProc = function(Mgr: pPGPMemoryMgr; UserValue: PGPUserValue;
					      RequestSize: PGPSize; Flags: PGPMemoryMgrFlags;
					      var IsNonPageable: PGPBoolean): Pointer;
  // deallocation proc need not clear the memory upon deallocation since PGPFreeData() does it automatically
  PGPMemoryMgrSecureDeallocationProc = function(Mgr: pPGPMemoryMgr; UserValue: PGPUserValue;
						Allocation: Pointer; AllocationSize: PGPSize;
						WasLocked: PGPBoolean): PGPError;

type
  { uses the above declared types }
  PGPNewMemoryMgrStruct = Record
    SizeOfStruct: PGPUInt32; // SizeOfStruct must be inited to SizeOf(PGPNewMemoryMgrStruct)
    ReservedFlags: PGPFlags;

    AllocProc: PGPMemoryMgrAllocationProc;
    ReallocProc: PGPMemoryMgrReallocationProc;
    DeallocProc: PGPMemoryMgrDeallocationProc;

    SecureAllocProc: PGPMemoryMgrSecureAllocationProc;
    Reserved: Pointer; // MUST be zeroed
    SecureDeallocProc: PGPMemoryMgrSecureDeallocationProc;

    CustomValue: PGPUserValue;
    Pad8: Array[0..1] of Pointer; // MUST be zeroed
  end;

{____________________________________________________________________________
	Block Info:
		kPGPMemoryMgrBlockInfo_Valid		it's a valid block
		kPGPMemoryMgrBlockInfo_Secure		block is a secure allocation
		kPGPMemoryMgrBlockInfo_NonPageable	block cannot be paged by VM

	Secure blocks are always wiped before being disposed,
	but may or may not be pageable, depending on the OS facilities.  Some
	OSs may not provide the ability to make blocks non-pageable.

	You should check these flags if the information matters to you.
_____________________________________________________________________________}
const
  kPGPMemoryMgrBlockInfo_Valid		= 1;
  kPGPMemoryMgrBlockInfo_Secure		= 2;
  kPGPMemoryMgrBlockInfo_NonPageable	= 4;

var
{ memory manager functions }
  PGPNewMemoryMgr: function(Reserved: PGPFlags; var NewMemoryMgr: pPGPMemoryMgr): PGPError; cdecl;
  PGPNewMemoryMgrCustom: function(const Custom: PGPNewMemoryMgrStruct; var NewMemoryMgr: pPGPMemoryMgr): PGPError; cdecl;
  PGPFreeMemoryMgr: function(Mgr: pPGPMemoryMgr): PGPError; cdecl;
  PGPGetMemoryMgrCustomValue: function(Mgr: pPGPMemoryMgr; var CustomValue: PGPUserValue): PGPError; cdecl;
  PGPSetMemoryMgrCustomValue: function(Mgr: pPGPMemoryMgr; CustomValue: PGPUserValue): PGPError; cdecl;

{ memory allocation functions }
// allocate a block of the specified size
  PGPNewData: function(Mgr: pPGPMemoryMgr; RequestSize: PGPSize; Flags: PGPMemoryMgrFlags): Pointer; cdecl;
// allocate a block of the specified size in non-pageable memory
// *isSecure is TRUE if the block definitely can't be paged
  PGPNewSecureData: function(Mgr: pPGPMemoryMgr; RequestSize: PGPSize; Flags: PGPMemoryMgrFlags): Pointer; cdecl;
// properly reallocs secure or non-secure blocks
// WARNING: the block may move, even if its size is being reduced
  PGPReallocData: function(Mgr: pPGPMemoryMgr; var Allocation: Pointer; NewAllocationSize: PGPSize; Flags: PGPMemoryMgrFlags): PGPError; cdecl;
// properly frees secure or non-secure blocks
  PGPFreeData: function(Allocation: Pointer): PGPError; cdecl;

{ default memory manager }
  PGPGetDefaultMemoryMgr: function: pPGPMemoryMgr; cdecl;
  PGPSetDefaultMemoryMgr: function(MemoryMgr: pPGPMemoryMgr): PGPError; cdecl;

{ context memory manager }
  PGPGetContextMemoryMgr: function(Context: pPGPContext): pPGPMemoryMgr; cdecl;

{ block info }
  PGPGetMemoryMgrDataInfo: function(Allocation: Pointer): PGPFlags; cdecl;

{____________________________________________________________________________
	Mini-tutorial:

	A PGPMemoryMgr is an object which implements memory management, including
	allocation, reallocation, deallocation, and secure versions of the same.

	*** Using it ***
	A typical sequence of calls is as follows:
		PGPNewMemoryMgr
		...
		PGPNewData or PGPNewSecureData
		PGPFreeData
		...
		PGPFreeMemoryMgr

	Typically, a program will create one PGPMemoryMgr per thread at
	thread creation time and use that memory mgr until the thread dies.
	Generally, an individual PGPMemoryMgr instance is not thread-safe;
	you must either synchronize or use one PGPMemoryMgr per thread.


	*** Custom Allocators ***

	Default allocators are supplied, but the client can create a custom
	PGPMemoryMgr using PGPNewMemoryMgrCustom() which uses client-supplied
	routines.
	Custom routines need only concern themselves with the actual
	allocation and deallocation.
	The following should be kept in mind for user supplied routines:
		- they can ignore the allocation flags passed
		- leaks, memory clearing, etc is done by the PGPMemoryMgr
		- secure allocator must set 'isNonPageable' to TRUE only if the
		memory really can't be paged.
		- the user value is not interpreted by the PGPMemoryMgr. Typically,
		it would be a pointer to some data the allocation routines use
		to store state.


	*** Secure memory allocation ***

	Blocks can be allocated as "Secure" blocks.  Secure blocks are guaranteed
	to be wiped when they are deallocated.  Additionally, if the operating
	system and the current conditions allow, the block will be allocated
	in non-pageable memory.  You can determine the attributes of a block using
	PGPGetMemoryMgrDataInfo().


	*** Leaks tracking ***

	Leaks tracking is implemented when debugging is on,
	but currently reporting is limited to reporting the number of leaks
	outstanding when the PGPMemoryMgr is disposed.


	*** Debugging ***

	For debugging purposes, blocks may be larger in debug mode to accomodate
	various schemes to detect stray pointers, etc.
_____________________________________________________________________________}

{ evaluation 'macro' }
function PGPMemoryMgrIsValid(Mgr: pPGPMemoryMgr): PGPBoolean;

implementation

uses
  Windows;

function PGPMemoryMgrIsValid(Mgr: pPGPMemoryMgr): PGPBoolean;
begin
  Result:=ord(Mgr<>kInvalidPGPMemoryMgr);
end;

initialization

  PGPNewMemoryMgr:=GetProcAddress(hPGPsdkLib, 'PGPNewMemoryMgr');
  PGPNewMemoryMgrCustom:=GetProcAddress(hPGPsdkLib, 'PGPNewMemoryMgrCustom');
  PGPFreeMemoryMgr:=GetProcAddress(hPGPsdkLib, 'PGPFreeMemoryMgr');
  PGPGetMemoryMgrCustomValue:=GetProcAddress(hPGPsdkLib, 'PGPGetMemoryMgrCustomValue');
  PGPSetMemoryMgrCustomValue:=GetProcAddress(hPGPsdkLib, 'PGPSetMemoryMgrCustomValue');
  PGPNewData:=GetProcAddress(hPGPsdkLib, 'PGPNewData');
  PGPNewSecureData:=GetProcAddress(hPGPsdkLib, 'PGPNewSecureData');
  PGPReallocData:=GetProcAddress(hPGPsdkLib, 'PGPReallocData');
  PGPFreeData:=GetProcAddress(hPGPsdkLib, 'PGPFreeData');
  PGPGetDefaultMemoryMgr:=GetProcAddress(hPGPsdkLib, 'PGPGetDefaultMemoryMgr');
  PGPSetDefaultMemoryMgr:=GetProcAddress(hPGPsdkLib, 'PGPSetDefaultMemoryMgr');
  PGPGetContextMemoryMgr:=GetProcAddress(hPGPsdkLib, 'PGPGetContextMemoryMgr');
  PGPGetMemoryMgrDataInfo:=GetProcAddress(hPGPsdkLib, 'PGPGetMemoryMgrDataInfo');

end.

