{$J+,Z4}
unit KeyFuncs;

{------------------------------------------------------------------------------}
{                                                                              }
{     This unit is partly based on Steve Heller's spgpKeyUtil.pas from his     }
{         SPGP sources available from http://www.oz.net/~srheller/spgp/        }
{                                                                              }
{                Portions created by Michael in der Wiesche are                }
{                 Copyright (C) 2001 by Michael in der Wiesche                 }
{                                                                              }
{------------------------------------------------------------------------------}

interface

uses
  Windows,
  Classes,
  SysUtils,
  TimeUtils,
  KeyPropTypes,
  pgpCl,
  pgpKeys,
  pgpUtilities,
  pgpOptionList,
  pgpErrors, pgpPubTypes, pgpBase;

{ "public" functions }
function GetHexIDFromIDNum(const iKey: Integer): String;
function GetIDNumFromHexID(const KeyData: String): Integer;
function KeyRemove(KeyData: PChar): Integer;
function KeyEnable(HexKeyID: PChar): Integer;
function KeyDisable(HexKeyID: PChar): Integer;
// properties, key & user IDs for all keys matching the specified KeyData
// if KeyData='' then all keys will be checked
// - returns number of keys found or error
function FindKeyProps(const KeyData: String; var KeyPropsList: TKeyPropsList;
		      PropertyFlags, FilterFlags: Integer;
		      KeyOrder: TPGPKeyOrdering): Integer;
{ "private" functions }
function UnixTimeToLocalTimeStr(UnixSeconds: Integer): String;
function GetSubKeyPropKeyID(SubKey: pPGPKey): String;
function GetKeyPropKeyID(Key: pPGPKey): String;
function GetKeyPropUserID(Key: pPGPKey): String;
function GetKeyPropAlg(Key: pPGPKey): TKeyAlgorithm;
function GetKeyPropIsAxiomatic(Key: pPGPKey): Longbool;
function GetKeyFromKeySet(KeySet: pPGPKeySet; var Key: pPGPKey): PGPError;
function GetKeyByHexID(KeySet: pPGPKeySet; const HexID: String; var Key: pPGPKey): PGPError;
function GetKeyFilter(Context: pPGPContext; FilterFlags: Integer; var KeyFilter: pPGPFilter): PGPError;
function GetKeyFilterByAnyID(Context: pPGPContext; const KeyData: String; var KeyFilter: pPGPFilter): PGPError;
function GetKeySetByAnyIDs(Context: pPGPContext; KeySetMain: pPGPKeySet;
			   const KeyData: String; var KeySetFound: pPGPKeySet): PGPError;
function GetExclusiveKeySet(var KeySetToCheck: pPGPKeySet; KeySetMain: pPGPKeySet;
			    Context: pPGPContext; IgnoreKnownFlag: Integer): PGPError;
function AddNonPrivateKeysFromList(Context: pPGPContext; KeySetMain: pPGPKeySet;
				   const KeyIDList: pPGPKeyID; KeyCount: PGPUInt32;
				   var KeySetToAddTo: pPGPKeySet): PGPError;
function GetKeySetProps(Context: pPGPContext; KeySet: pPGPKeySet;
			var KeyPropsList: TKeyPropsList;
			PropertyFlags, FilterFlags: Integer;
			KeyOrder: TPGPKeyOrdering): Integer; // - returns number of keys found or error
function GetHexIDByAnyID(Context: pPGPContext; KeySetMain: pPGPKeySet;
			   const KeyData: String; var HexID: String): Integer; // - returns number of keys found or error
function PassphraseIsValid(Context: pPGPContext; KeySetMain: pPGPKeySet; KeyData, Passphrase: PChar): Longbool;
function InitKeyRings(var Context: pPGPContext; var MainKeySet: pPGPKeySet): PGPError;
procedure FreeKeyRings(Context: pPGPContext; MainKeySet: pPGPKeySet);

implementation

function GetHexIDFromIDNum(const iKey: Integer): String;
begin
  Result:='0x' + IntToHex(integer(iKey), 8);
end;

function GetIDNumFromHexID(const KeyData: String): Integer;
var iPos: Integer;
begin
  Result:=-1;
  try
    iPos:=pos('x', LowerCase(KeyData));
    if (iPos<>0) and (Length(KeyData)>=iPos+8) then begin
      Result:=StrToInt('$' + Copy(KeyData, succ(iPos), 8));
    end;
  except
  end;
end;

function KeyRemove(KeyData: PChar): Integer;
var
  Context	: pPGPContext;
  KeySetMain	: pPGPKeySet;
  KeySetFound	: pPGPKeySet;
  KeyCount	: PGPUInt32;
begin
  KeySetFound:=nil;
  Result:=InitKeyRings(Context, KeySetMain);
  try
    if Result<>0 then Exit;
    Result:=GetKeySetByAnyIDs(Context, KeySetMain, KeyData, KeySetFound);
    if Result<>0 then Exit;
    try
      Result:=PGPCountKeys(KeySetFound, KeyCount);
      if Result<>0 then Exit;
      if KeyCount<>0 then begin
	Result:=PGPRemoveKeys(KeySetFound, KeySetMain);
	if Result<>0 then Exit;
      end;
    finally
      PGPFreeKeySet(KeySetFound);
    end;
    if longbool(PGPKeySetNeedsCommit(KeySetMain)) then begin
      Result:=PGPCommitKeyringChanges(KeySetMain);
      if Result<>0 then Exit;
      if PGPclInitLibrary(Context)=0 then begin
	try
	  PGPclNotifyKeyringChanges(GetCurrentProcessID);
	finally
	  PGPclCloseLibrary;
	end;
      end;
    end;
  finally
    FreeKeyRings(Context, KeySetMain);
  end;
end;

function KeyEnable(HexKeyID: PChar): Integer;
var
  Context	: pPGPContext;
  KeySetMain	: pPGPKeySet;
  Key		: pPGPKey;
begin
  Result:=InitKeyRings(Context, KeySetMain);
  try
    if Result<>0 then Exit;
    Result:=GetKeyByHexID(KeySetMain, HexKeyID, Key);
    if Result<>0 then Exit;
    Result:=PGPEnableKey(Key);
    if Result<>0 then Exit;
    if longbool(PGPKeySetNeedsCommit(KeySetMain)) then begin
      Result:=PGPCommitKeyringChanges(KeySetMain);
      if Result<>0 then Exit;
      if PGPclInitLibrary(Context)=0 then begin
	try
	  PGPclNotifyKeyringChanges(GetCurrentProcessID);
	finally
	  PGPclCloseLibrary;
	end;
      end;
    end;
  finally
    FreeKeyRings(Context, KeySetMain);
  end;
end;

function KeyDisable(HexKeyID: PChar): Integer;
var
  Context	: pPGPContext;
  KeySetMain	: pPGPKeySet;
  Key		: pPGPKey;
begin
  Result:=InitKeyRings(Context, KeySetMain);
  try
    if Result<>0 then Exit;
    Result:=GetKeyByHexID(KeySetMain, HexKeyID, Key);
    if Result<>0 then Exit;
    Result:=PGPDisableKey(Key);
    if Result<>0 then Exit;
    if longbool(PGPKeySetNeedsCommit(KeySetMain)) then begin
      Result:=PGPCommitKeyringChanges(KeySetMain);
      if Result<>0 then Exit;
      if PGPclInitLibrary(Context)=0 then begin
	try
	  PGPclNotifyKeyringChanges(GetCurrentProcessID);
	finally
	  PGPclCloseLibrary;
	end;
      end;
    end;
  finally
    FreeKeyRings(Context, KeySetMain);
  end;
end;

function FindKeyProps(const KeyData: String; var KeyPropsList: TKeyPropsList;
		      PropertyFlags, FilterFlags: Integer;
		      KeyOrder: TPGPKeyOrdering): Integer;
var
  Context	: pPGPContext;
  KeySetMain	: pPGPKeySet;
  KeySetFound	: pPGPKeySet;
  KeyFilter	: pPGPFilter;
  KeySetFiltered: pPGPKeySet;
  KeyCount	: PGPUInt32;
begin
  KeySetFound:=nil;
  KeyFilter:=nil;
  KeySetFiltered:=nil;
  Result:=InitKeyRings(Context, KeySetMain);
  try
    if Result<>0 then Exit;
    if KeyData='' then
      KeySetFound:=KeySetMain // all keys on key rings
    else Result:=GetKeySetByAnyIDs(Context, KeySetMain, KeyData, KeySetFound); // key(s) matching KeyData
    if Result<>0 then Exit;
    try
      if PropertyFlags<>0 then begin
	Result:=GetKeySetProps(Context, KeySetFound,
			       KeyPropsList, PropertyFlags,
			       FilterFlags, KeyOrder);
      end
      else begin
	if FilterFlags<>KeyFilterFlag_AllKeys then begin
	  Result:=GetKeyFilter(Context, FilterFlags, KeyFilter);
	  try
	    if Result<>0 then Exit;
	    Result:=PGPFilterKeySet(KeySetFound, KeyFilter, KeySetFiltered);
	  finally
	    PGPFreeFilter(KeyFilter);
	  end;
	end
	else KeySetFiltered:=KeySetFound;
	if Result<>0 then Exit;
	try
	  Result:=PGPCountKeys(KeySetFiltered, KeyCount);
	  if Result=0 then Result:=KeyCount;
	finally
	  if KeySetFiltered<>KeySetFound then PGPFreeKeySet(KeySetFiltered);
	end;
      end;
    finally
      if KeySetFound<>KeySetMain then PGPFreeKeySet(KeySetFound);
    end;
  finally
    FreeKeyRings(Context, KeySetMain);
  end;
end;

function GetKeyFromKeySet(KeySet: pPGPKeySet; var Key: pPGPKey): PGPError;
var
  KeyList	: pPGPKeyList;
  KeyIter	: pPGPKeyIter;
begin
  Key:=nil;
  KeyList:=nil;
  KeyIter:=nil;
  Result:=PGPOrderKeySet(KeySet, kPGPUserIDOrdering, KeyList);
  if Result<>0 then Exit;
  try
    Result:=PGPNewKeyIter(KeyList, KeyIter);
    if Result<>0 then Exit;
    try
      Result:=PGPKeyIterNext(KeyIter, Key);
    finally
      PGPFreeKeyIter(KeyIter);
    end;
  finally
    PGPFreeKeyList(KeyList);
  end;
end;

function GetKeyByHexID(KeySet: pPGPKeySet; const HexID: String; var Key: pPGPKey): PGPError;
var
  PGPKeyID	: TPGPKeyID;
begin
  Key:=nil;
  Result:=PGPGetKeyIDFromString(PChar(HexID), PGPKeyID);
  if Result=0 then Result:=PGPGetKeyByKeyID(KeySet, PGPKeyID, kPGPPublicKeyAlgorithm_Invalid, Key);
end;

function GetKeyFilter(Context: pPGPContext; FilterFlags: Integer; var KeyFilter: pPGPFilter): PGPError;
var
  KeyBoolFilter	: pPGPFilter;
  KeyAlgFilter	: pPGPFilter;
begin
  KeyFilter:=nil;
  KeyBoolFilter:=nil;
  KeyAlgFilter:=nil;
  Result:=0;
  if FilterFlags and KeyFilterMask_Boolean<>0 then begin
    if FilterFlags and KeyFilterFlag_CanEncrypt<>0 then begin
      Result:=PGPNewKeyBooleanFilter(Context, kPGPKeyPropCanEncrypt, PGPBoolean(true), KeyBoolFilter);
    end
    else if FilterFlags and KeyFilterFlag_CanDecrypt<>0 then begin
      Result:=PGPNewKeyBooleanFilter(Context, kPGPKeyPropCanDecrypt, PGPBoolean(true), KeyBoolFilter);
    end
    else if FilterFlags and KeyFilterFlag_CanSign<>0 then begin
      Result:=PGPNewKeyBooleanFilter(Context, kPGPKeyPropCanSign, PGPBoolean(true), KeyBoolFilter);
    end
    else if FilterFlags and KeyFilterFlag_CanVerify<>0 then begin
      Result:=PGPNewKeyBooleanFilter(Context, kPGPKeyPropCanVerify, PGPBoolean(true), KeyBoolFilter);
    end
    else if FilterFlags and KeyFilterFlag_Enabled<>0 then begin
      Result:=PGPNewKeyDisabledFilter(Context, PGPBoolean(false), KeyBoolFilter);
    end
    else if FilterFlags and KeyFilterFlag_Disabled<>0 then begin
      Result:=PGPNewKeyDisabledFilter(Context, PGPBoolean(true), KeyBoolFilter);
    end;
  end;
  if Result<>0 then Exit;
  if FilterFlags and KeyFilterMask_Algorithm<>0 then begin
    if FilterFlags and KeyFilterFlag_DHDSS<>0 then begin
      Result:=PGPNewKeySigAlgorithmFilter(Context, kPGPPublicKeyAlgorithm_DSA, KeyAlgFilter);
    end else if FilterFlags and KeyFilterFlag_RSA<>0 then begin
      Result:=PGPNewKeySigAlgorithmFilter(Context, kPGPPublicKeyAlgorithm_RSA, KeyAlgFilter);
    end;
  end;
  if Result<>0 then Exit;
  if (KeyBoolFilter<>nil) and (KeyAlgFilter<>nil) then
    Result:=PGPIntersectFilters(KeyBoolFilter, KeyAlgFilter, KeyFilter)
  else if KeyAlgFilter<>nil then
    KeyFilter:=KeyAlgFilter
  else if KeyBoolFilter<>nil then
    KeyFilter:=KeyBoolFilter;
end;

function GetKeyFilterByAnyID(Context: pPGPContext; const KeyData: String; var KeyFilter: pPGPFilter): PGPError;
var
  PGPKeyID	: TPGPKeyID;
begin
  KeyFilter:=nil;
  if (KeyData<>'') and (KeyData[1]='0') and ((ord(KeyData[2]) and $DF)=ord('X')) then begin
    Result:=PGPGetKeyIDFromString(PChar(KeyData), PGPKeyID);
    if Result=0 then Result:=PGPNewKeyIDFilter(Context, PGPKeyID, KeyFilter);
  end
  else Result:=PGPNewUserIDStringFilter(Context, PChar(KeyData), kPGPMatchSubString, KeyFilter);
end;

function GetKeySetByAnyIDs(Context: pPGPContext; KeySetMain: pPGPKeySet;
			   const KeyData: String; var KeySetFound: pPGPKeySet): PGPError;
var
  KeyDataList	: TStringList;
  KeyIndex	: PGPUInt32;
  KeyString	: String;
  KeyFilter	: pPGPFilter;
  KeySet	: pPGPKeySet;
  KeyCount	: PGPUInt32;
begin
  KeySetFound:=nil;
  if KeyData<>'' then begin
    KeyDataList:=TStringList.Create;
    KeyFilter:=nil;
    KeySet:=nil;
    try
      KeyDataList.CommaText:=KeyData;
      Result:=PGPNewKeySet(Context, KeySetFound);
      if Result<>0 then Exit;
      for KeyIndex:=0 to pred(KeyDataList.Count) do begin
	KeyString:=Trim(KeyDataList[KeyIndex]);
	if KeyString<>'' then begin
	  Result:=GetKeyFilterByAnyID(Context, KeyString, KeyFilter);
	  if Result<>0 then Exit;
	  try
	    Result:=PGPFilterKeySet(KeySetMain, KeyFilter, KeySet);
	    if Result<>0 then Exit;
	    try
	      Result:=PGPCountKeys(KeySet, KeyCount);
	      if Result<>0 then Exit;
	      if KeyCount<>0 then begin
		Result:=PGPAddKeys(KeySet, KeySetFound);
		if Result<>0 then Exit;
	      end
	      else begin
		Result:=kPGPError_ItemNotFound;
		PGPFreeKeySet(KeySetFound);
		KeySetFound:=nil;
		Exit;
	      end;
	    finally
	      PGPFreeKeySet(KeySet);
	    end;
	  finally
	    PGPFreeFilter(KeyFilter);
	  end;
	end;
      end;
    finally
      KeyDataList.Free;
    end;
  end
  else Result:=kPGPError_ItemNotFound;
end;

function GetExclusiveKeySet(var KeySetToCheck: pPGPKeySet; KeySetMain: pPGPKeySet;
			    Context: pPGPContext; IgnoreKnownFlag: Integer): PGPError;
var
  KeyListToCheck: pPGPKeyList;
  KeyIterToCheck: pPGPKeyIter;
  KeyToCheck	: pPGPKey;
  UserIDToCheck	: TUserID;
  KeySetToRemove: pPGPKeySet;
  KeyCount	: PGPUInt32;
  IDSize	: PGPSize;
  UserIDFound	: pPGPUserID;
  UserIDProp	: PGPBoolean;
  KeyFilter	: pPGPFilter;
  KeySetFound	: pPGPKeySet;
begin
  Result:=0;
  KeyToCheck:=nil;
  KeyListToCheck:=nil;
  KeyIterToCheck:=nil;
  try
    if (PGPOrderKeySet(KeySetToCheck, kPGPUserIDOrdering, KeyListToCheck)=0)
    and (PGPNewKeyIter(KeyListToCheck, KeyIterToCheck)=0) then begin
      if (IgnoreKnownFlag and IgnoreFlag_ByHexID)<>0 then begin
	while PGPKeyIterNext(KeyIterToCheck, KeyToCheck)=0 do begin
	  KeySetToRemove:=nil;
	  if boolean(PGPKeySetIsMember(KeyToCheck, KeySetMain))
	  and (PGPNewSingletonKeySet(KeyToCheck, KeySetToRemove)=0) then begin
	    try
	      PGPRemoveKeys(KeySetToRemove, KeySetToCheck);
	    finally
	      PGPFreeKeySet(KeySetToRemove);
	    end;
	  end;
	end;
	PGPKeyIterRewind(KeyIterToCheck);
      end;
      if (PGPCountKeys(KeySetToCheck, KeyCount)=0) and (KeyCount=0) then Exit;
      if (IgnoreKnownFlag and IgnoreFlag_ByUserID)<>0 then begin
	while (PGPKeyIterNext(KeyIterToCheck, KeyToCheck)=0)
	and (PGPKeyIterRewindUserID(KeyIterToCheck)=0) do begin
	  while (PGPKeyIterNextUserID(KeyIterToCheck, UserIDFound)=0)
	  and (PGPGetUserIDBoolean(UserIDFound, kPGPUserIDPropIsAttribute, UserIDProp)=0)
	  and not boolean(UserIDProp) do begin
	    KeyFilter:=nil;
	    KeySetFound:=nil;
	    KeySetToRemove:=nil;
	    try
	      if (PGPGetUserIDStringBuffer(UserIDFound, kPGPUserIDPropName, kPGPMaxUserIDSize, @UserIDToCheck, IDSize)=0)
	      and (PGPNewUserIDStringFilter(Context, UserIDToCheck, kPGPMatchEqual, KeyFilter)=0)
	      and (PGPFilterKeySet(KeySetMain, KeyFilter, KeySetFound)=0)
	      and (PGPCountKeys(KeySetFound, KeyCount)=0) and (KeyCount<>0)
	      and (PGPNewSingletonKeySet(KeyToCheck, KeySetToRemove)=0) then begin
		PGPRemoveKeys(KeySetToRemove, KeySetToCheck);
		Break;
	      end;
	    finally
	      PGPFreeFilter(KeyFilter);
	      PGPFreeKeySet(KeySetFound);
	      PGPFreeKeySet(KeySetToRemove);
	    end;
	  end;
	  if (PGPCountKeys(KeySetToCheck, KeyCount)=0) and (KeyCount=0) then Break;
	end;
      end;
    end;
  finally
    PGPFreeKeyIter(KeyIterToCheck);
    PGPFreeKeyList(KeyListToCheck);
  end;
end;

function AddNonPrivateKeysFromList(Context: pPGPContext; KeySetMain: pPGPKeySet;
				   const KeyIDList: pPGPKeyID; KeyCount: PGPUInt32;
				   var KeySetToAddTo: pPGPKeySet): PGPError;
var
  KeyIndex	: PGPUInt32;
  PGPKeyID	: pPGPKeyID;
  Key		: pPGPKey;
  KeyNotFound	: Longbool;
  KeySecFilter	: pPGPFilter;
  KeyPubFilter	: pPGPFilter;
  KeyIDFilter	: pPGPFilter;
  KeyFilter	: pPGPFilter;
  KeySet	: pPGPKeySet;
  KeyFoundCount	: PGPUInt32;
begin
  Result:=kPGPError_SecretKeyNotFound;
  if (KeySetToAddTo<>nil) and (KeyIDList<>nil) and (KeyCount<>0) then begin
    KeySecFilter:=nil;
    KeyPubFilter:=nil;
    KeyIDFilter:=nil;
    KeyFilter:=nil;
    KeySet:=nil;
    PGPKeyID:=KeyIDList;
    for KeyIndex:=0 to pred(KeyCount) do begin
      try
	Key:=nil;
	KeyNotFound:=(PGPGetKeyByKeyID(KeySetMain, PGPKeyID^, kPGPPublicKeyAlgorithm_Invalid, Key)=kPGPError_ItemNotFound);
	if KeyNotFound then
	  Result:=PGPNewSubKeyBooleanFilter(Context, kPGPKeyPropIsSecret, PGPBoolean(true), KeySecFilter)
	else Result:=PGPNewKeyBooleanFilter(Context, kPGPKeyPropIsSecret, PGPBoolean(true), KeySecFilter);
	if Result<>0 then Exit;
	Result:=PGPNegateFilter(KeySecFilter, KeyPubFilter);
	if Result<>0 then Exit; // KeySecFilter is automatically freed
	if KeyNotFound then
	  Result:=PGPNewSubKeyIDFilter(Context, PGPKeyID^, KeyIDFilter)
	else Result:=PGPNewKeyIDFilter(Context, PGPKeyID^, KeyIDFilter);
	if Result<>0 then begin
	  PGPFreeFilter(KeyPubFilter); // KeySecFilter is automatically freed
	  Exit;
	end;
	Result:=PGPIntersectFilters(KeyPubFilter, KeyIDFilter, KeyFilter);
	if Result<>0 then Exit; // KeyPubFilter & KeyIDFilter are automatically freed
	try
	  Result:=PGPFilterKeySet(KeySetMain, KeyFilter, KeySet);
	  if Result<>0 then begin
	    PGPFreeFilter(KeyFilter); // KeyPubFilter & KeyIDFilter are automatically freed
	    Exit;
	  end;
	  try
	    Result:=PGPCountKeys(KeySet, KeyFoundCount);
	    if (Result=0) and (KeyFoundCount<>0) then Result:=PGPAddKeys(KeySet, KeySetToAddTo);
	  finally
	    PGPFreeKeySet(KeySet);
	  end;
	finally
	  PGPFreeFilter(KeyFilter);
	end;
      except
	Exit;
      end;
      inc(PGPKeyID);
    end;
  end;
end;

function UnixTimeToLocalTimeStr(UnixSeconds: Integer): String;
const
  TIME_ZONE_ID_INVALID  =  $FFFFFFFF;
  TIME_ZONE_ID_UNKNOWN  =  0;
  TIME_ZONE_ID_STANDARD =  1;
  TIME_ZONE_ID_DAYLIGHT =  2;
var
  UnixTime, LocalTime: TDateTime;
  tzInfo: TTimeZoneInformation;
  NoTZInfo: Longbool;
begin
  Result:='';
  try
    // convert a UNIX time to Delphi's TDateTime format
    UnixTime:=EncodeDate(1970, 1, 1) + UnixSeconds / SECONDS_PER_DAY;

    // if Windows local timezone info is not available
    // then take UTC/GMT as the only available valid time format
    // else get local time using Primoz Gabrijelcic's Time Zone Routines
    NoTZInfo:=(GetTimeZoneInformation(tzInfo)=TIME_ZONE_ID_INVALID);
    if NoTZInfo then
      LocalTime:=UnixTime
    else LocalTime:=UTCToTZLocalTime(tzInfo, UnixTime);

    // Format TDateTime to the local date/time format
    if NoTZInfo then
      Result:=FormatDateTime('ddd, dd mmm yyyy, tt "<UTC>"', LocalTime)
    else Result:=FormatDateTime('ddd, dd mmm yyyy, tt', LocalTime);
  except
  end;
end;

function GetSubKeyPropKeyID(SubKey: pPGPKey): String;
var
  PGPKeyID	: TPGPKeyID;
  KeyID		: TKeyID;
begin
  Result:='';
  if (SubKey<>nil)
  and (PGPGetKeyIDFromSubKey(SubKey, PGPKeyID)=0)
  and (PGPGetKeyIDString(PGPKeyID, kPGPKeyIDString_Abbreviated, KeyID)=0) then begin
    Result:=KeyID;
  end;
end;

function GetKeyPropKeyID(Key: pPGPKey): String;
var
  PGPKeyID	: TPGPKeyID;
  KeyID		: TKeyID;
begin
  Result:='';
  if (Key<>nil)
  and (PGPGetKeyIDFromKey(Key, PGPKeyID)=0)
  and (PGPGetKeyIDString(PGPKeyID, kPGPKeyIDString_Abbreviated, KeyID)=0) then begin
    Result:=KeyID;
  end;
end;

function GetKeyPropUserID(Key: pPGPKey): String;
var
  UserIDBuf	: TUserID;
  IDSize	: PGPSize;
begin
  Result:='';
  if (Key<>nil)
  and (PGPGetPrimaryUserIDNameBuffer(Key, kPGPMaxUserIDSize, @UserIDBuf, IDSize)=0) then begin
    Result:=UserIDBuf;
  end;
end;

function GetKeyPropFingerprint(Key: pPGPKey): String;
var
  Fingerprint	: TFingerprint;
  FPSize	: PGPSize;
  FPIndex	: PGPSize;
begin
  Result:='';
  if (Key<>nil)
  and (PGPGetKeyPropertyBuffer(Key, kPGPKeyPropFingerprint, kPGPmaxFingerprintSize, @Fingerprint, FPSize)=0) then begin
    for FPIndex:=0 to pred(FPSize) do begin
      if (FPIndex mod 2)=0
	then Result:=Result + ' ' + IntToHex(Fingerprint[FPIndex], 2)
      else Result:=Result + IntToHex(Fingerprint[FPIndex], 2);
    end;
    Delete(Result, 1, 1);
  end;
end;

// local time
function GetKeyPropKeyCreationTimeStr(Key: pPGPKey): String;
var
  CreaTime	: PGPTime;
begin
  Result:='';
  if (Key<>nil) and (PGPGetKeyTime(Key, kPGPKeyPropCreation, CreaTime)=0) then begin
    Result:=UnixTimeToLocalTimeStr(PGPGetStdTimeFromPGPTime(CreaTime));
  end;
end;

// local time
function GetKeyPropKeyExpirationTimeStr(Key: pPGPKey): String;
var
  ExpTime	: PGPTime;
  CreaTime	: PGPTime;
begin
  Result:='';
  if (Key<>nil)
  and (PGPGetKeyTime(Key, kPGPKeyPropExpiration, ExpTime)=0)
  and (PGPGetKeyTime(Key, kPGPKeyPropCreation, CreaTime)=0) then begin
    if ExpTime>CreaTime then Result:=UnixTimeToLocalTimeStr(PGPGetStdTimeFromPGPTime(ExpTime));
  end;
end;

function GetKeyPropKeyBits(Key: pPGPKey): String;
var
  KeyAlg	: PGPInt32;
  KeyBits	: PGPInt32;
  Error		: PGPError;
  KeySet	: pPGPKeySet;
  KeyList	: pPGPKeyList;
  KeyIter	: pPGPKeyIter;
  SubKey	: pPGPSubKey;
  SubKeyBits	: PGPInt32;
begin
  Result:='';
  KeySet:=nil;
  KeyList:=nil;
  KeyIter:=nil;
  SubKey:=nil;
  if (Key<>nil) and (PGPGetKeyNumber(Key, kPGPKeyPropAlgID, KeyAlg)=0) then begin
    case KeyAlg of
      // no subkey(s)
      kPGPPublicKeyAlgorithm_RSA: begin
	if PGPGetKeyNumber(Key, kPGPKeyPropBits, KeyBits)=0 then Result:=IntToStr(KeyBits) + ' bits';
      end;
      // w or w/o subkey(s)
      kPGPPublicKeyAlgorithm_DSA : begin
	Error:=PGPGetKeyNumber(Key, kPGPKeyPropBits, KeyBits);
	if Error<>0 then Exit;
	Error:=PGPNewSingletonKeySet(Key, KeySet);
	if Error<>0 then Exit;
	try
	  Error:=PGPOrderKeySet(KeySet, kPGPAnyOrdering, KeyList);
	  if Error<>0 then Exit;
	  try
	    Error:=PGPNewKeyIter(KeyList, KeyIter);
	    if Error<>0 then Exit;
	    try
	      Error:=PGPKeyIterNext(KeyIter, Key);
	      if Error<>0 then Exit;
	      if PGPKeyIterNextSubKey(KeyIter, SubKey)=0 then
		Error:=PGPGetSubKeyNumber(SubKey, kPGPKeyPropBits, SubKeyBits)
	      else SubKeyBits:=0;
	      if Error=0 then begin
		if SubKeyBits>0 then
		  Result:=IntToStr(SubKeyBits) + '/' + IntToStr(KeyBits) + ' bits'
		else Result:=IntToStr(KeyBits) + ' bits';
	      end;
	    finally
	      PGPFreeKeyIter(KeyIter);
	    end;
	  finally
	    PGPFreeKeyList(KeyList);
	  end;
	finally
	  PGPFreeKeySet(KeySet);
	end;
      end;
    end;
  end;
end;

function GetKeyPropAlg(Key: pPGPKey): TKeyAlgorithm;
var
  KeyAlg	: PGPInt32;
begin
  Result:=KeyAlgorithm_Invalid;
  if (Key<>nil) and (PGPGetKeyNumber(Key, kPGPKeyPropAlgID, KeyAlg)=0) then begin
    Result:=TKeyAlgorithm(KeyAlg);
    if (Result=KeyAlgorithm_DSS) and (pos('/', GetKeyPropKeyBits(Key))<>0) then begin
      inc(Result);
    end;
  end;
end;

function GetKeyPropTrust(Key: pPGPKey): TTrustLevel;
var
  KeyTrust	: PGPInt32;
begin
  Result:=KeyTrust_Undefined;
  if (Key<>nil) and (PGPGetKeyNumber(Key, kPGPKeyPropTrust, KeyTrust)=0) then begin
    Result:=TTrustLevel(KeyTrust);
  end;
end;

function GetKeyPropValidity(Key: pPGPKey): TValidityLevel;
var
  KeyValidity	: PGPInt32;
begin
  Result:=Validity_Unknown;
  if (Key<>nil) and (PGPGetKeyNumber(Key, kPGPKeyPropValidity, KeyValidity)=0) then begin
    Result:=TValidityLevel(KeyValidity);
  end;
end;

// UTC/GMT Unix format seconds
function GetKeyPropKeyCreationTime(Key: pPGPKey): Integer;
var
  CreaTime	: PGPTime;
begin
  Result:=0;
  if (Key<>nil) and (PGPGetKeyTime(Key, kPGPKeyPropCreation, CreaTime)=0) then begin
    Result:=PGPGetStdTimeFromPGPTime(CreaTime);
  end;
end;

// UTC/GMT Unix format seconds
function GetKeyPropKeyExpirationTime(Key: pPGPKey): Integer;
var
  ExpTime	: PGPTime;
  CreaTime	: PGPTime;
begin
  Result:=0;
  if (Key<>nil)
  and (PGPGetKeyTime(Key, kPGPKeyPropExpiration, ExpTime)=0)
  and (PGPGetKeyTime(Key, kPGPKeyPropCreation, CreaTime)=0) then begin
    if ExpTime>CreaTime then Result:=PGPGetStdTimeFromPGPTime(ExpTime);
  end;
end;

function GetKeyPropIsSecret(Key: pPGPKey): Longbool;
var
  Prop		: PGPBoolean;
begin
  Result:=false;
  if (Key<>nil) and (PGPGetKeyBoolean(Key, kPGPKeyPropIsSecret, Prop)=0) then begin
    Result:=boolean(Prop);
  end;
end;

function GetKeyPropIsAxiomatic(Key: pPGPKey): Longbool;
var
  Prop		: PGPBoolean;
begin
  Result:=false;
  if (Key<>nil) and (PGPGetKeyBoolean(Key, kPGPKeyPropIsAxiomatic, Prop)=0) then begin
    Result:=boolean(Prop);
  end;
end;

function GetKeyPropIsRevoked(Key: pPGPKey): Longbool;
var
  Prop		: PGPBoolean;
begin
  Result:=false;
  if (Key<>nil) and (PGPGetKeyBoolean(Key, kPGPKeyPropIsRevoked, Prop)=0) then begin
    Result:=boolean(Prop);
  end;
end;

function GetKeyPropIsDisabled(Key: pPGPKey): Longbool;
var
  Prop		: PGPBoolean;
begin
  Result:=false;
  if (Key<>nil) and (PGPGetKeyBoolean(Key, kPGPKeyPropIsDisabled, Prop)=0) then begin
    Result:=boolean(Prop);
  end;
end;

function GetKeyPropIsExpired(Key: pPGPKey): Longbool;
var
  Prop		: PGPBoolean;
begin
  Result:=false;
  if (Key<>nil) and (PGPGetKeyBoolean(Key, kPGPKeyPropIsExpired, Prop)=0) then begin
    Result:=boolean(Prop);
  end;
end;

function GetKeyPropIsSecretShared(Key: pPGPKey): Longbool;
var
  Prop		: PGPBoolean;
begin
  Result:=false;
  if (Key<>nil) and (PGPGetKeyBoolean(Key, kPGPKeyPropIsSecretShared, Prop)=0) then begin
    Result:=boolean(Prop);
  end;
end;

function GetKeyPropCanEncrypt(Key: pPGPKey): Longbool;
var
  Prop		: PGPBoolean;
begin
  Result:=false;
  if (Key<>nil) and (PGPGetKeyBoolean(Key, kPGPKeyPropCanEncrypt, Prop)=0) then begin
    Result:=boolean(Prop);
  end;
end;

function GetKeyPropCanDecrypt(Key: pPGPKey): Longbool;
var
  Prop		: PGPBoolean;
begin
  Result:=false;
  if (Key<>nil) and (PGPGetKeyBoolean(Key, kPGPKeyPropCanDecrypt, Prop)=0) then begin
    Result:=boolean(Prop);
  end;
end;

function GetKeyPropCanSign(Key: pPGPKey): Longbool;
var
  Prop		: PGPBoolean;
begin
  Result:=false;
  if (Key<>nil) and (PGPGetKeyBoolean(Key, kPGPKeyPropCanSign, Prop)=0) then begin
    Result:=boolean(Prop);
  end;
end;

function GetKeyPropCanVerify(Key: pPGPKey): Longbool;
var
  Prop		: PGPBoolean;
begin
  Result:=false;
  if (Key<>nil) and (PGPGetKeyBoolean(Key, kPGPKeyPropCanVerify, Prop)=0) then begin
    Result:=boolean(Prop);
  end;
end;

function GetKeyPropHasRevoker(KeySet: pPGPKeySet; Key: pPGPKey; KeyPropsRec: pKeyPropsRec): Longbool;
var
  RevKeyCount	: PGPUInt32;
  RevKeyIndex	: PGPUInt32;
  RevKey	: pPGPKey;
  RevKeyID	: TPGPKeyID;
  KeyID		: TKeyID;
begin
  Result:=false;
  if (Key<>nil) and (PGPCountRevocationKeys(Key, RevKeyCount)=0) then begin
    if RevKeyCount>0 then begin
      Result:=true;
      for RevKeyIndex:=0 to pred(RevKeyCount) do begin
	if PGPGetIndexedRevocationKey(Key, KeySet, RevKeyIndex, RevKey, RevKeyID)=0 then begin
	  if PGPGetKeyIDString(RevKeyID, kPGPKeyIDString_Abbreviated, KeyID)=0 then begin
	    KeyPropsRec^.kRevKeyIDList.Add(KeyID);
	  end;
	end;
      end;
    end;
  end;
end;

function GetKeyPropHasARR(KeySet: pPGPKeySet; Key: pPGPKey; KeyPropsRec: pKeyPropsRec): TADKType;
var
  ARKeyCount	: PGPUInt32;
  ARKeyIndex	: PGPUInt32;
  ARKey		: pPGPKey;
  ARKeyID	: TPGPKeyID;
  ARClass	: PGPByte;
  KeyID		: TKeyID;
begin
  Result:=NoADK;
  ARKey:=nil;
  if (Key<>nil) and (PGPCountAdditionalRecipientRequests(Key, ARKeyCount)=0) then begin
    if ARKeyCount>0 then begin
      Result:=SimpleADK;
      for ARKeyIndex:=0 to pred(ARKeyCount) do begin
	if PGPGetIndexedAdditionalRecipientRequestKey(Key, KeySet, ARKeyIndex, ARKey, ARKeyID, ARClass)=0 then begin
	  if ARClass>0 then Result:=EnforcedADK;
	  if PGPGetKeyIDString(ARKeyID, kPGPKeyIDString_Abbreviated, KeyID)=0 then begin
	    KeyPropsRec^.kADKeyIDList.Add(KeyID);
	  end;
	end;
      end;
    end;
  end;
end;

function GetKeyPropHasSubKey(Key: pPGPKey): Longbool;
begin
  Result:=(pos('/', GetKeyPropKeyBits(Key))<>0);
end;

function GetKeyPropUserIDs(Key: pPGPKey; KeyPropsRec: pKeyPropsRec; IncludeSignerIDs: Longbool): Integer;
var
  KeySet	: pPGPKeySet;
  KeyList	: pPGPKeyList;
  KeyIter	: pPGPKeyIter;
  UserID	: pPGPUserID;
  UserIDProp	: PGPBoolean;
  UserIDBuf	: TUserID;
  IDSize	: PGPSize;
  IDBuffer	: String;
  KeySig	: pPGPSig;
  SignKey	: TPGPKeyID;
  KeyID		: TKeyID;
begin
  KeySet:=nil;
  KeyList:=nil;
  KeyIter:=nil;
  Result:=PGPNewSingletonKeySet(Key, KeySet);
  if Result<>0 then Exit;
  try
    Result:=PGPOrderKeySet(KeySet, kPGPUserIDOrdering, KeyList);
    if Result<>0 then Exit;
    try
      Result:=PGPNewKeyIter(KeyList, KeyIter);
      if Result<>0 then Exit;
      try
	Result:=PGPKeyIterNext(KeyIter, Key);
	if Result<>0 then Exit;
	Result:=PGPKeyIterRewindUserID(KeyIter);
	if Result<>0 then Exit;
	// add UserIDs one by one to UserIDList
	while PGPKeyIterNextUserID(KeyIter, UserID)=0 do begin
	  Result:=PGPGetUserIDBoolean(UserID, kPGPUserIDPropIsAttribute, UserIDProp);
	  if Result<>0 then Exit;
	  // check for attributes (picture, f.e.)
	  if not boolean(UserIDProp) then begin
	    Result:=PGPGetUserIDStringBuffer(UserID, kPGPUserIDPropName, kPGPMaxUserIDSize, @UserIDBuf, IDSize);
	    if Result<>0 then Exit;
	    try
	      with KeyPropsRec^ do begin
		kUserIDList.Add(UserIDBuf);
		if IncludeSignerIDs then kSignerIDList.Add('');
	      end;
	    except
	      Result:=kPGPError_OutOfMemory;
	      Exit;
	    end;
	  end;
	  IDBuffer:='';
	  if IncludeSignerIDs then begin
	    // add comma separated HexIDs of signing keys to KeyPropsRec.SignerIDList
	    while PGPKeyIterNextUIDSig(KeyIter, KeySig)=0 do begin
	      Result:=PGPGetKeyIDOfCertifier(KeySig, SignKey);
	      if Result<>0 then Exit;
	      Result:=PGPGetKeyIDString(SignKey, kPGPKeyIDString_Abbreviated, KeyID);
	      if Result<>0 then Exit;
	      IDBuffer:=IDBuffer + ',' + KeyID;
	    end;
	    Delete(IDBuffer, 1, 1);
	    try
	      with KeyPropsRec^.kSignerIDList do Strings[pred(Count)]:=IDBuffer;
	    except
	      Result:=kPGPError_OutOfMemory;
	      Exit;
	    end;
	  end;
	end;
      finally
	PGPFreeKeyIter(KeyIter);
      end;
    finally
      PGPFreeKeyList(KeyList);
    end;
  finally
    PGPFreeKeySet(KeySet);
  end;
end;

function GetKeyPropsData(KeySet: pPGPKeySet; KeyFound: pPGPKey; Flags: Integer;
			 KeyPropsList: TKeyPropsList; KeyIndex: Integer): Integer;
var KeyPropsRec	: pKeyPropsRec;
begin
  Result:=-1;
  if KeyPropsList.Objects[KeyIndex]<>nil then begin
    KeyPropsRec:=pKeyPropsRec(KeyPropsList.Objects[KeyIndex]);
    with KeyPropsRec^ do begin
      try
	// "string" properties
	KeyPropsList.Strings[KeyIndex]:=GetKeyPropKeyID(KeyFound);
	if Flags and spgpKeyPropFlag_KeyID<>0 then kHexID:=KeyPropsList[KeyIndex];
	if Flags and spgpKeyPropFlag_UserID<>0 then kUserID:=GetKeyPropUserID(KeyFound);
	if Flags and spgpKeyPropFlag_Fingerprint<>0 then kFingerprint:=GetKeyPropFingerprint(KeyFound);
	if Flags and spgpKeyPropFlag_CreationTimeStr<>0 then kCreaTimeStr:=GetKeyPropKeyCreationTimeStr(KeyFound);
	if Flags and spgpKeyPropFlag_ExpirationTimeStr<>0 then kExpTimeStr:=GetKeyPropKeyExpirationTimeStr(KeyFound);
	// "number" properties
	if Flags and spgpKeyPropFlag_Keybits<>0 then kSize:=GetKeyPropKeyBits(KeyFound);
	if Flags and spgpKeyPropFlag_KeyAlg<>0 then kAlgorithm:=GetKeyPropAlg(KeyFound);
	if Flags and spgpKeyPropFlag_Trust<>0 then kTrust:=GetKeyPropTrust(KeyFound);
	if Flags and spgpKeyPropFlag_Validity<>0 then kValidity:=GetKeyPropValidity(KeyFound);
	if Flags and spgpKeyPropFlag_CreationTime<>0 then kCreaTimeNum:=GetKeyPropKeyCreationTime(KeyFound);
	if Flags and spgpKeyPropFlag_ExpirationTime<>0 then kExpTimeNum:=GetKeyPropKeyExpirationTime(KeyFound);
	// "boolean" properties
	if Flags and spgpKeyPropFlag_IsSecret<>0 then kPrivate:=GetKeyPropIsSecret(KeyFound);
	if Flags and spgpKeyPropFlag_IsAxiomatic<>0 then kImplicitTrust:=GetKeyPropIsAxiomatic(KeyFound);
	if Flags and spgpKeyPropFlag_IsRevoked<>0 then kRevoked:=GetKeyPropIsRevoked(KeyFound);
	if Flags and spgpKeyPropFlag_IsDisabled<>0 then kDisabled:=GetKeyPropIsDisabled(KeyFound);
	if Flags and spgpKeyPropFlag_IsExpired<>0 then kExpired:=GetKeyPropIsExpired(KeyFound);
	if Flags and spgpKeyPropFlag_IsSecretShared<>0 then kSecShared:=GetKeyPropIsSecretShared(KeyFound);
	if Flags and spgpKeyPropFlag_CanEncrypt<>0 then kCanEncrypt:=GetKeyPropCanEncrypt(KeyFound);
	if Flags and spgpKeyPropFlag_CanDecrypt<>0 then kCanDecrypt:=GetKeyPropCanDecrypt(KeyFound);
	if Flags and spgpKeyPropFlag_CanSign<>0 then kCanSign:=GetKeyPropCanSign(KeyFound);
	if Flags and spgpKeyPropFlag_CanVerify<>0 then kCanVerify:=GetKeyPropCanVerify(KeyFound);
	if Flags and spgpKeyPropFlag_HasRevoker<>0 then kHasRevoker:=GetKeyPropHasRevoker(KeySet, KeyFound, KeyPropsRec);
	if Flags and spgpKeyPropFlag_HasADK<>0 then kHasADK:=GetKeyPropHasARR(KeySet, KeyFound, KeyPropsRec);
	if Flags and spgpKeyPropFlag_HasSubKey<>0 then kHasSubKey:=GetKeyPropHasSubKey(KeyFound);
	// "list" properties
	if Flags and spgpKeyPropFlag_IncludeUserIDs<>0 then begin
	  GetKeyPropUserIDs(KeyFound, KeyPropsRec, (Flags and spgpKeyPropFlag_IncludeSignerIDs<>0));
	end;
	Result:=0;
      except
      end;
    end;
  end;
end;

function GetKeySetProps(Context: pPGPContext; KeySet: pPGPKeySet;
			var KeyPropsList: TKeyPropsList;
			PropertyFlags, FilterFlags: Integer;
			KeyOrder: TPGPKeyOrdering): Integer;
var
  KeyFilter	: pPGPFilter;
  KeySetFound	: pPGPKeySet;
  KeyCount	: PGPUInt32;
  KeyList	: pPGPKeyList;
  KeyIter	: pPGPKeyIter;
  KeyFound	: pPGPKey;
  KeyIndex	: PGPUInt32;
begin
  KeyFilter:=nil;
  KeySetFound:=nil;
  KeyList:=nil;
  KeyIter:=nil;
  if FilterFlags<>KeyFilterFlag_AllKeys then begin
    Result:=GetKeyFilter(Context, FilterFlags, KeyFilter);
    try
      if Result<>0 then Exit;
      Result:=PGPFilterKeySet(KeySet, KeyFilter, KeySetFound);
      if Result<>0 then Exit;
      KeySet:=KeySetFound;
    finally
      PGPFreeFilter(KeyFilter);
    end;
  end
  else Result:=0;
  try
    if Result<>0 then Exit;
    Result:=PGPCountKeys(KeySet, KeyCount);
    if Result<>0 then Exit;
    if PropertyFlags<>0 then begin
      if KeyPropsList=nil then begin
	KeyPropsList:=TKeyPropsList.Create(KeyCount, PropertyFlags);
	KeyIndex:=0;
      end
      else KeyIndex:=KeyPropsList.Append(KeyCount);
      if (KeyPropsList=nil) or (KeyIndex<0) then begin
	Result:=kPGPError_OutOfMemory;
	Exit;
      end;
      Result:=PGPOrderKeySet(KeySet, PGPKeyOrdering(KeyOrder), KeyList);
      if Result<>0 then Exit;
      try
	Result:=PGPNewKeyIter(KeyList, KeyIter);
	try
	  if Result<>0 then Exit;
	  while PGPKeyIterNext(KeyIter, KeyFound)=0 do begin
	    Result:=GetKeyPropsData(KeySet, KeyFound, PropertyFlags, KeyPropsList, KeyIndex);
	    if Result=0 then
	      inc(KeyIndex)
	    else Exit;
	  end;
	  Result:=KeyPropsList.Count;
	finally
	  PGPFreeKeyIter(KeyIter);
	end;
      finally
	PGPFreeKeyList(KeyList);
      end;
    end
    else Result:=KeyCount;
  finally
    PGPFreeKeySet(KeySetFound);
  end;
end;

function GetHexIDByAnyID(Context: pPGPContext; KeySetMain: pPGPKeySet;
			 const KeyData: String; var HexID: String): Integer;
var
  KeySetFound	: pPGPKeySet;
  KeyCount	: PGPUInt32;
  PGPKey	: pPGPKey;
begin
  HexID:='';
  Result:=0;
  KeySetFound:=nil;
  if KeyData<>'' then begin
    try
      Result:=GetKeySetByAnyIDs(Context, KeySetMain, KeyData, KeySetFound);
      if Result<>0 then Exit;
      Result:=PGPCountKeys(KeySetFound, KeyCount);
      if Result<>0 then Exit;
      if KeyCount>0 then begin
	Result:=GetKeyFromKeySet(KeySetFound, PGPKey);
	if Result<>0 then Exit;
	HexID:=GetKeyPropKeyID(PGPKey);
      end;
      Result:=KeyCount;
    finally
      PGPFreeKeySet(KeySetFound);
    end;
  end;
end;

function PassphraseIsValid(Context: pPGPContext; KeySetMain: pPGPKeySet; KeyData, Passphrase: PChar): Longbool;
var
  KeySetFound	: pPGPKeySet;
  KeyCount	: PGPUInt32;
  Key		: pPGPKey;
begin
  Result:=false;
  KeySetFound:=nil;
  if GetKeySetByAnyIDs(Context, KeySetMain, KeyData, KeySetFound)=0 then begin
    try
      Result:=(PGPCountKeys(KeySetFound, KeyCount)=0) and (KeyCount=1) and (GetKeyFromKeySet(KeySetFound, Key)=0) and
	      (PGPPassphraseIsValid(Key, PGPOPassphrase(Context, Passphrase), PGPOLastOption(Context))=ord(true));
    finally
      PGPFreeKeySet(KeySetFound);
    end;
  end;
end;

function InitKeyRings(var Context: pPGPContext; var MainKeySet: pPGPKeySet): PGPError;
begin
  Context:=nil;
  MainKeySet:=nil;
  Result:=PGPNewContext(kPGPsdkAPIVersion, Context);
  if Result=0 then Result:=PGPOpenDefaultKeyRings(Context, kPGPKeyRingOpenFlags_Mutable, MainKeySet);
end;

procedure FreeKeyRings(Context: pPGPContext; MainKeySet: pPGPKeySet);
begin
  PGPFreeKeySet(MainKeySet);
  PGPFreeContext(Context);
end;

end.

