unit PathCombo;

// TPATHCOMBOBOX Component
//  Angus Johnson
// ajohnson@rpi.net.au
// Version 2.0
// September 1997.

// DESCRIPTION: File Path combobox component which appears and behaves
// almost identically to the combobox in the Win95 Open & Save Dialogs.
// Can be used with the excellent TFileList component
// written by Amedeo Lanza <amldc@tin.it>.
// (Source code - Flist010.zip - available from DSP ->
// http://SunSITE.icm.edu.pl/delphi/)

// USAGE:
// 1. To get the full path name (including drive name)...
// FullPathName := PathComboBox1.path
// 2. To programatically set the path ...
// PathComboBox1.path := 'c:\Program Files\';
// 3. Assign PathComboBox1's OnChange method ...
// FileList1.directory := PathComboBox1.path;
// etc.
// NOTE: Selecting the Virtual folder "Drives" will return an empty Path!

// KNOWN "FEATURES":
// Changing ItemIndex will not change Path.
// eg: itemindex := itemindex-1 {DOES NOT CHANGE PATH}
// do: Path := items[itemindex-1] {HOWEVER THIS DOES.}

// CHANGES FROM VERSION 1.0 :
// 1. DrawItem method deleted (no longer called) as
// all drawing now done in CNDrawItem method.
// Now only the display text is highlighted on selection.

// DISCLAIMER:
// This software may be freely used but no guarantees are given
// as to reliability. Please keep this header to acknowledge source.
// USE AT YOUR OWN RISK.

// PROBLEMS/COMMENTS/THANKS ...
// ajohnson@rpi.net.au

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  StdCtrls, ShellApi, ShlObj;

type
  TDriveSet = set of 0..25;

  TPathComboBox = class(TCustomComboBox)
  private
    FImageList: TImageList;
    DriveSet: TDriveSet;
    FDesktop, FDrives, FDesktopPath: string;
    FiDesktop, FiDrives: integer;
    FPath: string;
    {Workaround to stop OnChange event being called twice under some circumstances}
    FNotifyChange: boolean;
    procedure SetPath(NewPath: string);
    procedure CMFontChanged(var Message: TMessage); message CM_FONTCHANGED;
    procedure WMCHAR(var Message: TMessage); message WM_CHAR;
    procedure CNDrawItem(var Message: TWMDrawItem); message CN_DrawItem;
  protected
    procedure CreateWnd; override;
    procedure Click; override;
    procedure Change; override;
    procedure BuildList; virtual;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    property Path: string read FPath write SetPath;
  published
    property Color;
    property Ctl3D;
    property DragMode;
    property DragCursor;
    property Enabled;
    property Font;
    property ParentColor;
    property ParentCtl3D;
    property ParentFont;
    property ParentShowHint;
    property PopupMenu;
    property ShowHint;
    property TabOrder;
    property TabStop;
    property Visible;
    property OnClick;
    property OnChange;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnDropDown;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnKeyDown;
    property OnKeyPress;
    property OnKeyUp;
    property OnStartDrag;
  end;

procedure Register;

implementation

constructor TPathComboBox.Create(AOwner: TComponent);
var
  sfi : tshfileinfo;
  pidl: PITEMIDLIST;
  zTempPath: array [0..MAX_PATH] of char;
begin
  inherited Create(AOwner);
  width := 230;
  Style := csOwnerDrawFixed;
  FImageList := TImageList.create(self);
  FImageList.handle :=
    shgetfileinfo('',0,sfi,sizeof(tshfileinfo), shgfi_sysiconindex or shgfi_smallicon);
  FImageList.shareimages := true;
  FImageList.BlendColor := clHighlight;

  // get valid drives
  integer(DriveSet) := GetLogicalDrives;
  // set current path
  GetDir(0, FPath);
  if FPath[length(FPath)] <> '\' then appendstr(FPath,'\');

  if (SHGetSpecialFolderLocation(0,CSIDL_DESKTOPDIRECTORY,pidl) = NOERROR) and
            SHGetPathFromIDList(pidl,zTempPath) then begin
    FDesktopPath := zTempPath;
    appendstr(FDesktopPath,'\');
  end;
  if SHGetSpecialFolderLocation(0,CSIDL_DESKTOP,pidl) = NOERROR then begin
    SHGetFileInfo(pchar(pidl),0,sfi,sizeof(sfi), shgfi_sysiconindex or
      shgfi_smallicon or SHGFI_DISPLAYNAME or SHGFI_PIDL);
    FDesktop := sfi.szDisplayName;
    FiDesktop := sfi.iIcon;
  end;
  if SHGetSpecialFolderLocation(0,CSIDL_DRIVES,pidl) = NOERROR then begin
    SHGetFileInfo(pchar(pidl),0,sfi,sizeof(sfi), shgfi_sysiconindex or
      shgfi_smallicon or SHGFI_DISPLAYNAME or SHGFI_PIDL);
    FDrives := sfi.szDisplayName;
    FiDrives := sfi.iIcon;
  end;

end;

destructor TPathComboBox.Destroy;
begin
  FImageList.handle := 0;
  FImageList.free;
  inherited Destroy;
end;

procedure TPathComboBox.SetPath(NewPath: string);
begin
  if (NewPath <> '') and (NewPath[length(NewPath)] <> '\') then appendstr(NewPath,'\');
  if Uppercase(FPath) = Uppercase(NewPath) then exit;
  FPath := NewPath;
  BuildList;
  FNotifyChange := true;
  Change; {The only way to notify OnChange event}
  FNotifyChange := false;
end;

procedure TPathComboBox.CreateWnd;
begin
  inherited CreateWnd;
  BuildList;
end;

procedure TPathComboBox.BuildList;
var
  Drive: char;
  i, level, l, r, Drivenum, ItmNdx: integer;
  TmpStr: string;
begin
  ItmNdx := -1;
  Items.beginupdate;
  items.clear;

  // NOTE:
  // the data pointer is used to store 2 numbers !!!
  // the least signif. bit is a flag - directory open/closed
  // the next bit is not used
  // the other bits store the nested level of the directory.
  // eg: pointer(1*4) -> Level = 1; {100 binary}
  // eg: pointer(2*4) -> Level = 2; {1000 binary}
  // eg: pointer(3*4+1) -> Level = 3 & use open icon. {1101 binary}


  //Add Desktop & NullDrive folders
  items.addobject(FDesktopPath,pointer(0));
  items.addobject('',pointer(4));

  // If a Virtual Folder then just add drives and exit;
  if (fPath = '') or (fPath = FDesktopPath) then begin
    for i := 0 to 25 do
      if i in DriveSet then begin
        Drive := chr(ord('A')+i);
        items.addobject(Drive+':\',pointer(8));
      end;
    if (fPath = '') then ItemIndex := 1
    else ItemIndex := 0;
    Items.endupdate;
    exit;
  end;

  // A Path has been specified...

  Drive := UpCase(FPath[1]); {UNC not implimented}
  if not ((Drive >= 'A') and (Drive <= 'Z')) then exit; {ERROR: shouldn't happen!}
  DriveNum := ord(Drive)-ord('A'); {A=0, B=1, ...}

  // add drives up to & incl. current drive
  for i := 0 to Drivenum do
    if i in DriveSet then begin
      Drive := chr(ord('A')+i);
      items.addobject(Drive+':\',pointer(8));
    end;

  // add folders if any
  level := 2;
  l := 4;
  while l < length(FPath) do begin
    inc(level);
    r := l; {right = left}
    while FPath[r] <> '\' do inc(r);
    TmpStr := copy(FPath,1,r);
    if r = length(FPath) then begin{open folder}
      items.addobject(TmpStr, pointer((level shl 2)+1));
      ItmNdx := items.count-1;
      end
    else  {get closed folder icon}
      items.addobject(TmpStr, pointer((level shl 2)+0));
    {end}
    l := r+1;
  end;
  if ItmNdx = -1 then ItmNdx := items.count-1;

  // add drives after current drive
  for i := Drivenum+1 to 25 do
    if i in DriveSet then begin
      Drive := chr(ord('A')+i);
      items.addobject(Drive+':\',pointer(8));
    end;

  ItemIndex := ItmNdx;
  Items.endupdate;
end;

procedure TPathComboBox.CNDrawItem(var Message: TWMDrawItem);
var
  sfi : tshfileinfo;
  Level: integer;
  IsOpenFolder: boolean;
begin
  with Message.DrawItemStruct^ do
  begin
    Canvas.Handle := hDC;
    Canvas.Font := Font;
    Canvas.Brush := Brush;
    if Integer(itemID) < 0 then Canvas.FillRect(rcItem)
    else with canvas do begin
      // extract 2 values from object pointer
      IsOpenFolder := odd(integer(ItemData)); {ie: what's at bit 1 of ItemData pointer}
      Level := integer(ItemData) shr 2; {ie: ItemData Div 4}

      // This next line prevents 'edit selection' text from being indented as well.
      if (rcItem.top <> 3) then rcItem.left := rcItem.left + 10*Level;

      if integer(itemID) = 0 then begin {Virtual "Desktop" Folder}
        StrPCopy(sfi.szdisplayname,FDesktop);
        sfi.iIcon := FiDesktop;
        end
      else if integer(itemID) = 1 then begin {Virtual "Drives" Folder}
        StrPCopy(sfi.szdisplayname,FDrives);
        sfi.iIcon := FiDrives;
        end
      else if IsOpenFolder then
        shgetfileinfo(pchar(items[ItemID]),0,sfi,sizeof(sfi),shgfi_sysiconindex or
            shgfi_smallicon or shgfi_openicon or shgfi_displayname)
      else
        shgetfileinfo(pchar(items[ItemID]),0,sfi,sizeof(sfi),shgfi_sysiconindex or
            shgfi_smallicon or shgfi_displayname);

      if (Integer(itemID) >= 0) and (ods_Selected and ItemState <> 0) then
        FimageList.DrawingStyle := dsFocus
      else
        FimageList.DrawingStyle := dsNormal;

      FillRect(rcItem);
      FImageList.draw(canvas,rcItem.left+2,rcItem.top,sfi.iicon);
      rcItem.left := rcItem.left+FImageList.width+4;

      if (Integer(itemID) >= 0) and (ods_Selected and ItemState <> 0) then begin
        rcItem.right := rcItem.left+textwidth(sfi.szdisplayname)+4;
        Canvas.Brush.Color := clHighlight;
        Canvas.Font.Color := clHighlightText;
        FillRect(rcItem);
      end;

      textout(rcItem.left+2, rcItem.top, sfi.szdisplayname);
    end; {with canvas}

    if (ods_Focus and ItemState <> 0) then DrawFocusRect(hDC, rcItem);
    Canvas.Handle := 0;
  end;
end;

procedure TPathComboBox.Click;
begin
  inherited Click;
  // Note: Path not changed while dropdown list visible
  if (ItemIndex >= 0) and (FPath <> items[ItemIndex]) and
    (sendmessage(handle,CB_GETDROPPEDSTATE,0,0)=0) then
     Path := items[ItemIndex];
end;

procedure TPathComboBox.Change;
begin
  if FNotifyChange then inherited Change;
end;

procedure TPathComboBox.WMCHAR(var Message: TMessage);
begin
  inherited;
  if (TWMKey(Message).CharCode = VK_RETURN) or
   (TWMKey(Message).CharCode = VK_ESCAPE) then Click;
end;

procedure TPathComboBox.CMFontChanged(var Message: TMessage);

  function GetItemHeight(Font: TFont): Integer;
  var
    DC: HDC;
    SaveFont: HFont;
    Metrics: TTextMetric;
  begin
    DC := GetDC(0);
    SaveFont := SelectObject(DC, Font.Handle);
    GetTextMetrics(DC, Metrics);
    SelectObject(DC, SaveFont);
    ReleaseDC(0, DC);
    Result := Metrics.tmHeight;
  end;

var
  IHeight: integer;
begin
  inherited;
  IHeight := GetItemHeight(Font)+2;
  if IHeight < FImageList.height then IHeight := FImageList.height;
  ItemHeight := IHeight;
  RecreateWnd;
end;



procedure Register;
begin
  RegisterComponents('Samples', [TPathComboBox]);
end;

end.
