unit Grafxes;
(***********************************************************************
 *                                                                     *
 *  Copyright (C) 1996-1998                                            *
 *  Erol S. Uzuner                                                     *
 *                                                                     *
 *  Modul       :       Grafxes                                        *
 *  Version     :       1.0 (26.05.98)                                 *
 *                                                                     *
 *  Beschreibung:       Grafxes Unit                                   *
 *                                                                     *
 *  Autor       :       Erol S. Uzuner                                 *
 *                                                                     *
 *  Datum       :       26.05.98  10:44:00 1998                        *
 *                                                                     *
 *                                                                     *
 *  Aenderungen :                                                      *
 *  Datum        Autor          Beschreibung                           *
 *
 ***********************************************************************)


interface
uses  WinTypes, Graphics, ExtCtrls, Classes;



procedure DrawTiled(DC : HDC; CR: TRect; Bitmap : TBitmap);
procedure BitmapRotateHorizontal(Bitmap : TBitmap);
procedure BitmapRotateVertical(Bitmap : TBitmap);
procedure BitmapRotate90(Bitmap : TBitmap);
procedure DrawFrameBorder(Canvas: TCanvas; X, Y, W, H : Integer);
procedure DrawTransparentBitmap( canvas: TCanvas; bmp: TBitmap;
                                 fromRect, toRect : TRect;
                                 cTransparentColor: LongInt);
function Rechteck(left, top, right, bottom : integer) : TRect;
procedure BitmapGray(Bitmap : TBitmap);
function DarkerColor(c1 : TColor; darkness: Byte): TColor;
function LighterColor(c1 : TColor; lightness: Byte): TColor;

type

 TGradientStyle = (gsHorizontal, gsVertical, gsElliptic, gsRectangle,
                                     gsVertCenter, gsHorizCenter);
{ Gradient filling routine }
 {this Gradient filling routine is from harmans@uswest.net}
procedure GradientFillRect(Canvas: TCanvas; Rect: TRect;
                           BeginClr, EndClr: TColor; style: TGradientStyle);

type
 TMyColor = record
     case Boolean of
        TRUE :  (C : TColor);
        FALSE :  ( R, G, B, A : Byte);
       // FALSE :  ( G, R, B, A : Byte);
  end;

implementation
uses Math;
const
{ TBitmap.GetTransparentColor from GRAPHICS.PAS use this value }
  PaletteMask = $02000000;

function WidthOf(R: TRect): Integer;
begin Result := R.Right - R.Left;
end;

function HeightOf(R: TRect): Integer;
begin Result := R.Bottom - R.Top;
end;

function LighterColor(c1 : TColor; lightness: Byte): TColor;
var C:TmyColor;
begin
   C.R := (TmyColor(c1).R + lightness) MOD 255;
   C.B := (TmyColor(c1).B + lightness) MOD 255;
   C.G := (TmyColor(c1).G + lightness) MOD 255;
   C.A := TmyColor(c1).A;//(TmyColor(c1).A - darkness) MOD 255;
   Result := C.C;
end;

function DarkerColor(c1 : TColor; darkness: Byte): TColor;
var C:TmyColor;
begin
   C.R := (TmyColor(c1).R - darkness) MOD 255;
   C.B := (TmyColor(c1).B - darkness) MOD 255;
   C.G := (TmyColor(c1).G - darkness) MOD 255;
   C.A := TmyColor(c1).A;//(TmyColor(c1).A - darkness) MOD 255;
   Result := C.C;
end;

{----------------[Gradient Fill Methods]---------}
function Muldv(a,b,c : integer) : longint;
ASM
  MOV EAX, a
  IMUL b
  IDIV c
end;


procedure GradientFillRect(Canvas: TCanvas; Rect: TRect;
               BeginClr, EndClr: TColor; style : TGradientStyle);

var Height, Width : Integer;

    {I'll explain a little about the Horizontal gradient, the other styles are all
     consistent with their logic.  The six R, G, and B values are passed to us.
     We define some local variables we'll need: a rectangle, a FOR loop counter,
     and our own RGB numbers.  For a horizontal gradient, we'll draw a series of
     rectangles, each one a little closer in color to the EndClr value.  A horizontal
     gradient rectangle will always be from the top to the bottom of the canvas,
     so we set top to 0 and bottom to however tall our control is.  Then, we draw
     a series of 255 rectangles.  The starting point and width of each will depend
     on the actual width of our control.  It starts out on the left, draws the
     first rectangle in a color that's a percentage of the difference plus the
     starting color.  As I increments through the loop, the rectangles move to the
     right and the color gets closer and closer to the EndClr.}
    procedure DoHorizontal(fr, fg, fb, dr, dg, db : Integer);
    var
      ColorRect     : TRect;
      I             : Integer;
      R, G, B : Byte;
    begin
      ColorRect.Top:= Rect.Top;          //Set rectangle top
      ColorRect.Bottom := Rect.Bottom;
      for I := 0 to 255 do begin         //Make lines (rectangles) of color
        ColorRect.Left := Rect.Left + Muldv (I, Width, 256);        //Find left for this color
        ColorRect.Right:= Rect.Left + Muldv (I + 1, Width, 256);   //Find Right
        R := fr + Muldv(I, dr, 255);    //Find the RGB values
        G := fg + Muldv(I, dg, 255);
        B := fb + Muldv(I, db, 255);
        Canvas.Brush.Color := RGB(R, G, B);   //Plug colors into brush
        Canvas.FillRect(ColorRect);           //Draw on Bitmap
      end;
    end;

    procedure DoVertical(fr, fg, fb, dr, dg, db : Integer);
    var
      ColorRect: TRect;
      I: Integer;
      R, G, B : Byte;
    begin
      ColorRect.Left:= Rect.Left;        //Set rectangle left&right
      ColorRect.Right:= Rect.Right;
      for I := 0 to 255 do begin         //Make lines (rectangles) of color
        ColorRect.Top:= Rect.Top + Muldv (I, Height, 256);    //Find top for this color
        ColorRect.Bottom:= Rect.Top + Muldv (I + 1, Height, 256);   //Find Bottom
        R := fr + Muldv(I, dr, 255);    //Find the RGB values
        G := fg + Muldv(I, dg, 255);
        B := fb + Muldv(I, db, 255);
        Canvas.Brush.Color := RGB(R, G, B);   //Plug colors into brush
        Canvas.FillRect(ColorRect);           //Draw on Bitmap
      end;
    end;

    procedure DoElliptic(fr, fg, fb, dr, dg, db : Integer);

      procedure ClippingRect(r : TRect);
      var Rgn : THandle;
      begin
           LPToDP(Canvas.Handle, r, 2);
           Rgn := CreateRectRgnIndirect(r);
           SelectClipRgn(Canvas.Handle, Rgn);
           DeleteObject(Rgn);
      end;

    var
      I: Integer;
      R, G, B : Byte;
      Pw, Ph, Dw, Dh : integer;
      x1,y1,x2,y2 : integer;
      oldClipRect : TRect;
    {The elliptic is a bit different, since I had to use real numbers. I cut down
     on the number (to 155 instead of 255) of iterations in an attempt to speed
     things up, to no avail.  I think it just takes longer for windows to draw an
     ellipse as opposed to a rectangle.}
    begin
      oldClipRect := Canvas.ClipRect;
      ClippingRect(Rect);
      Canvas.Pen.Style := psClear;
      Canvas.Pen.Mode := pmCopy;
      x1 := Width div-3;
      x2 := Width + (Width div 3);
      y1 := Height div-3;
      y2 := Height + (Height div 3);
      Pw := x2 - x1;
      Ph := y2 - y1;
      for I := 0 to 50 do begin         //Make ellipses of color
        R := fr + Muldv(I, dr, 50);    //Find the RGB values
        G := fg + Muldv(I, dg, 50);
        B := fb + Muldv(I, db, 50);
        Canvas.Brush.Color := R or (G shl 8) or (b shl 16);   //Plug colors into brush
        Dw := Pw * i div 100;
        Dh := Ph * i div 100;
        Canvas.Ellipse(x1 + Dw,y1 + Dh,x2 - Dw,y2 - Dh);
      end;
      Canvas.Pen.Style := psSolid;
      ClippingRect(oldClipRect);
    end;

    procedure DoRectangle(fr, fg, fb, dr, dg, db : Integer);
    var
      I: Integer;
      R, G, B : Byte;
      Pw, Ph : Real;
      x1,y1,x2,y2 : Real;
      r1, r2, r3, r4 : Integer;
    begin
      Canvas.Pen.Style := psClear;
      Canvas.Pen.Mode := pmCopy;
      x1 := 0;
      x2 := Width+2;
      y1 := 0;
      y2 := Height+2;
      Pw := (Width / 2) / 255;
      Ph := (Height / 2) / 255;
      for I := 0 to 255 do begin         //Make rectangles of color
        x1 := x1 + Pw;
        x2 := X2 - Pw;
        y1 := y1 + Ph;
        y2 := y2 - Ph;
        R := fr + Muldv(I, dr, 255);    //Find the RGB values
        G := fg + Muldv(I, dg, 255);
        B := fb + Muldv(I, db, 255);
        Canvas.Brush.Color := RGB(R, G, B);   //Plug colors into brush
        r1 := Rect.Left + Trunc(x1);
        r2 := Rect.Top+Trunc(y1);
        r3 := Rect.Left+Trunc(x2);
        r4 := Rect.Top+Trunc(y2);
        if r3 > Rect.Right then r3 := Rect.Right;
        if r4 > Rect.Bottom then r4 := Rect.Bottom;
        Canvas.FillRect(Classes.Rect(r1, r2,r3,r4));
      end;
      Canvas.Pen.Style := psSolid;
    end;

    procedure DoVertCenter(fr, fg, fb, dr, dg, db : Integer);
    var
      ColorRect: TRect;
      I: Integer;
      R, G, B : Byte;
      Haf : Integer;
    begin
      Haf := Height Div 2;
      ColorRect.Left := Rect.Left;
      ColorRect.Right := Rect.Right;
      for I := 0 to Haf do begin
        ColorRect.Top := Rect.Top + Muldv (I, Haf, Haf);
        ColorRect.Bottom := Rect.Top + Muldv (I + 1, Haf, Haf);
        R := fr + Muldv(I, dr, Haf);
        G := fg + Muldv(I, dg, Haf);
        B := fb + Muldv(I, db, Haf);
        Canvas.Brush.Color := RGB(R, G, B);
        Canvas.FillRect(ColorRect);
        ColorRect.Top := Rect.Top + Height - (Muldv (I, Haf, Haf));
        ColorRect.Bottom := Rect.Top +  Height - (Muldv (I + 1, Haf, Haf));
        Canvas.FillRect(ColorRect);
      end;
    end;

    procedure DoHorizCenter(fr, fg, fb, dr, dg, db : Integer);
    var
      ColorRect: TRect;
      I: Integer;
      R, G, B : Byte;
      Haf : Integer;
    begin
      Haf := Width Div 2;
      ColorRect.Top := Rect.Top;
      ColorRect.Bottom := Rect.Bottom;
      for I := 0 to Haf do begin
        ColorRect.Left := Rect.Left + Muldv (I, Haf, Haf);
        ColorRect.Right := Rect.Left + Muldv (I + 1, Haf, Haf);
        R := fr + Muldv(I, dr, Haf);
        G := fg + Muldv(I, dg, Haf);
        B := fb + Muldv(I, db, Haf);
        Canvas.Brush.Color := RGB(R, G, B);
        Canvas.FillRect(ColorRect);
        ColorRect.Left := Rect.Left + Width - (Muldv (I, Haf, Haf));
        ColorRect.Right := Rect.Left + Width - (Muldv (I + 1, Haf, Haf));
        Canvas.FillRect(ColorRect);
      end;
    end;


var
  FromR, FromG, FromB : Integer; //These are the separate color values for RGB
  DiffR, DiffG, DiffB : Integer; // of color values.
begin
  FromR := BeginClr and $000000ff;  //Strip out separate RGB values
  FromG := (BeginClr shr 8) and $000000ff;
  FromB := (BeginClr shr 16) and $000000ff;
  DiffR := (EndClr  and $000000ff) - FromR;   //Find the difference
  DiffG := ((EndClr shr 8) and $000000ff) - FromG;
  DiffB := ((EndClr shr 16) and $000000ff) - FromB;

  Height := Rect.Bottom-Rect.Top;
  Width  := Rect.Right-Rect.Left;

    //Depending on gradient style selected, go draw it on the Bitmap canvas.
  case style of
    gsHorizontal  : DoHorizontal(FromR, FromG, FromB, DiffR, DiffG, DiffB);
    gsVertical    : DoVertical(FromR, FromG, FromB, DiffR, DiffG, DiffB);
    gsElliptic    : DoElliptic(FromR, FromG, FromB, DiffR, DiffG, DiffB);
    gsRectangle   : DoRectangle(FromR, FromG, FromB, DiffR, DiffG, DiffB);
    gsVertCenter  : DoVertCenter(FromR, FromG, FromB, DiffR, DiffG, DiffB);
    gsHorizCenter : DoHorizCenter(FromR, FromG, FromB, DiffR, DiffG, DiffB);
  end;

end;

(*
{ Gradient fill procedure - displays a gradient beginning with a chosen    }
{ color and ending with another chosen color. Based on TGradientFill       }
{ component source code written by Curtis White, cwhite@teleport.com.      }
procedure GradientFillRect(Canvas: TCanvas; Rect: TRect; BeginColor,
  EndColor: TColor; Direction: TFillDirection; Colors: Byte);
var
  BeginRGBValue: array[0..2] of Byte;    { Begin RGB values }
  RGBDifference: array[0..2] of Integer; { Difference between begin and end RGB values }
  ColorBand: TRect; { Color band rectangular coordinates }
  I, Delta: Integer;
  R, G, B: Byte;    { Color band Red, Green, Blue values }
begin
  if IsRectEmpty(Rect) then Exit;
  BeginColor := ColorToRGB(BeginColor);
  EndColor := ColorToRGB(EndColor);
  case Direction of
    fdTopToBottom, fdLeftToRight: begin
      { Set the Red, Green and Blue colors }
      BeginRGBValue[0] := GetRValue(BeginColor);
      BeginRGBValue[1] := GetGValue(BeginColor);
      BeginRGBValue[2] := GetBValue(BeginColor);
      { Calculate the difference between begin and end RGB values }
      RGBDifference[0] := GetRValue(EndColor) - BeginRGBValue[0];
      RGBDifference[1] := GetGValue(EndColor) - BeginRGBValue[1];
      RGBDifference[2] := GetBValue(EndColor) - BeginRGBValue[2];
    end;
    fdBottomToTop, fdRightToLeft: begin
      { Set the Red, Green and Blue colors }
      { Reverse of TopToBottom and LeftToRight directions }
      BeginRGBValue[0] := GetRValue(EndColor);
      BeginRGBValue[1] := GetGValue(EndColor);
      BeginRGBValue[2] := GetBValue(EndColor);
      { Calculate the difference between begin and end RGB values }
      { Reverse of TopToBottom and LeftToRight directions }
      RGBDifference[0] := GetRValue(BeginColor) - BeginRGBValue[0];
      RGBDifference[1] := GetGValue(BeginColor) - BeginRGBValue[1];
      RGBDifference[2] := GetBValue(BeginColor) - BeginRGBValue[2];
    end;
  end; {case}
  { Calculate the color band's coordinates }
  ColorBand := Rect;
  if Direction in [fdTopToBottom, fdBottomToTop] then begin
    Colors := Math.Max(1, Min(Colors, HeightOf(Rect)));
    Delta := HeightOf(Rect) div Colors;
  end
  else begin
    Colors := Max(1, Min(Colors, WidthOf(Rect)));
    Delta := WidthOf(Rect) div Colors;
  end;
  with Canvas.Pen do begin { Set the pen style and mode }
    Style := psSolid;
    Mode := pmCopy;
  end;
  { Perform the fill }
  if Delta > 0 then
    for I := 0 to Colors do begin
      case Direction of
        { Calculate the color band's top and bottom coordinates }
        fdTopToBottom, fdBottomToTop: begin
          ColorBand.Top := Rect.Top + I * Delta;
          ColorBand.Bottom := ColorBand.Top + Delta;
        end;
        { Calculate the color band's left and right coordinates }
        fdLeftToRight, fdRightToLeft: begin
          ColorBand.Left := Rect.Left + I * Delta;
          ColorBand.Right := ColorBand.Left + Delta;
        end;
      end; {case}
      { Calculate the color band's color }
      if Colors > 1 then begin
        R := BeginRGBValue[0] + MulDiv(I, RGBDifference[0], Colors - 1);
        G := BeginRGBValue[1] + MulDiv(I, RGBDifference[1], Colors - 1);
        B := BeginRGBValue[2] + MulDiv(I, RGBDifference[2], Colors - 1);
      end
      else begin
        { Set to the Begin Color if set to only one color }
        R := BeginRGBValue[0];
        G := BeginRGBValue[1];
        B := BeginRGBValue[2];
      end;
      with Canvas do begin
        Brush.Color := RGB(R, G, B) or PaletteMask;
        FillRect(ColorBand);
      end;
    end;
  if Direction in [fdTopToBottom, fdBottomToTop] then
    Delta := HeightOf(Rect) mod Colors
  else Delta := WidthOf(Rect) mod Colors;
  if Delta > 0 then begin
    case Direction of
      { Calculate the color band's top and bottom coordinates }
      fdTopToBottom, fdBottomToTop: begin
        ColorBand.Top := Rect.Bottom - Delta;
        ColorBand.Bottom := ColorBand.Top + Delta;
      end;
      { Calculate the color band's left and right coordinates }
      fdLeftToRight, fdRightToLeft: begin
        ColorBand.Left := Rect.Right - Delta;
        ColorBand.Right := ColorBand.Left + Delta;
      end;
    end; {case}
    with Canvas do begin
      case Direction of
        fdTopToBottom, fdLeftToRight:
          Brush.Color := EndColor or PaletteMask;
        fdBottomToTop, fdRightToLeft:
          Brush.Color := BeginColor or PaletteMask;
      end; {case}
      FillRect(ColorBand);
    end;
  end;
end;

*)

function Rechteck(left, top, right, bottom : integer) : TRect;
begin Result.left := left; Result.top :=  top;
      Result.Right :=  right; Result.Bottom :=  bottom;
end;


procedure DrawTiled(DC : HDC; CR: TRect; Bitmap : TBitmap);
var
  Row, Col : Integer;
  NumRows, NumCols: Integer;
  DrawX, DrawY, DrawWidth, DrawHeight: Integer;
  OldRect : TRect;
begin

  NumRows := CR.Bottom DIV Bitmap.Height;
  NumCols := CR.Right DIV Bitmap.Width;
  for Row := 0 to NumRows+1 do
    for Col := 0 to NumCols+1  do
    begin
      DrawX :=  (Col * Bitmap.Width) + CR.Left;
      DrawY :=  (Row * Bitmap.Height) + CR.Top;
      if (DrawX + Bitmap.Width) > CR.Right then
         DrawWidth :=  CR.Right - DrawX
      else
         DrawWidth :=  Bitmap.Width;

      if (DrawY + Bitmap.Height) > CR.Bottom then
         DrawHeight :=  CR.Bottom - DrawY
      else
         DrawHeight := Bitmap.Height;
      GetClipBox(DC, OldRect);
      IntersectCliprect(DC, CR.Left, CR.Top, CR.Right, CR.Bottom);
      BitBlt( DC,
              DrawX, DrawY, DrawWidth, DrawHeight,
              Bitmap.Canvas.Handle,
              0, 0, SRCCOPY);
      IntersectCliprect(DC, OldRect.Left, OldRect.Top, OldRect.Right, OldRect.Bottom);
    end;
end;

procedure DrawFrameBorder(Canvas: TCanvas; X, Y, W, H : Integer);
var
   nx, ny, wi, hi : Integer;
begin

   Canvas.Pen.Width := 1;
   Canvas.Pen.Style := psSolid;
   Canvas.Brush.Style := bsClear;
   Canvas.Pen.Color := clBlack;
   Canvas.Rectangle(0, 0, W, H);

   nx := 1;
   ny := 1;
   wi := W-1;
   hi := H-1;

   Canvas.Pen.Color := clBtnHighlight;
   Canvas.MoveTo(nx, hi-1);
   Canvas.LineTo(nx, ny);
   Canvas.LineTo(wi-1, ny);

   Canvas.Pen.Color := clBtnShadow;
   Canvas.MoveTo(wi-1, ny+1);
   Canvas.LineTo(wi-1, hi-1);
   Canvas.LineTo(nx-1, hi-1);

   Canvas.Pen.Color := clBtnFace;
   Canvas.Pen.Width := 2;
   Canvas.Rectangle(3, 3, wi-1, hi-1);
   Canvas.Pen.Width := 1;

end;

procedure DrawTransparentBitmap( canvas: TCanvas; bmp: TBitmap;
                                 fromRect, toRect : TRect;
                                 cTransparentColor: LongInt);
begin
     Canvas.BrushCopy(fromRect, bmp, toRect, cTransparentColor);
end;


procedure BitmapRotateHorizontal(Bitmap : TBitmap);
VAR aStream : TMemorystream;
    dc:hDC;
    Aptr: Pchar;
    linewidth:INTEGER;

VAR j,k : INTEGER;
 ptr1 : Pchar;
 ptr2 : Pchar;
 aint : LongInt;

BEGIN
   aStream := Tmemorystream.Create;
   aStream.SetSize(  SizeOf (TBITMAPINFOHEADER)
                    +   Bitmap.Height
                     * (Bitmap.Width+4) * 3);
   WITH TBITMAPINFOHEADER(aStream.Memory^) DO
    BEGIN
      biSize := SizeOf(TBITMAPINFOHEADER);
      biWidth := Bitmap.Width;
      biHeight := Bitmap.Height;
      biPlanes := 1;
      biBitCount := 24;
      biCompression := bi_RGB;
      biSizeImage := 0;
      biXPelsPerMeter :=1;
      biYPelsPerMeter :=1;
      biClrUsed :=0;
      biClrImportant :=0;
    END;

    dc := GetDC(0);
    Aptr := Pchar(aStream.Memory) +SizeOf (TBITMAPINFOHEADER);
    GetDIBits(dc,
             Bitmap.Handle,
             0,
             Bitmap.Height,
             Aptr,
             TBITMAPINFO(aStream.Memory^),
             dib_RGB_Colors);

    linewidth := (Bitmap.Width*3);
    linewidth := ((linewidth+3)DIV 4)*4;



    FOR j:=0 TO Bitmap.Height DO
     FOR k:=0 TO Bitmap.Width DIV 2 DO
      BEGIN
       ptr1 := Aptr +
          + j*linewidth
          + k * 3;
        ptr2 := Aptr +
          + j*linewidth
          + (Bitmap.Width-k)*3;
        Move(ptr1^,aint ,3);
        Move(ptr2^,ptr1^,3);
        Move(aint ,ptr2^,3);
      END;



    Bitmap.Handle :=
          CreateDIBitmap(dc,
              TBITMAPINFOHEADER(aStream.Memory^),
              cbm_Init,
              Aptr,
              TBITMAPINFO(aStream.Memory^),
              dib_RGB_Colors);
    aStream.Free;
    ReleaseDC(0,dc);
END;


PROCEDURE BitmapRotateVertical(Bitmap : TBitmap);
VAR aStream : Tmemorystream;
    dc:hDC;
    Aptr: Pchar;
    linewidth:INTEGER;

VAR j : INTEGER;
 ptr1 : Pchar;
 ptr2 : Pchar;
 ptr3 : Pchar;

BEGIN
    aStream := Tmemorystream.Create;
    aStream.SetSize(  SizeOf (TBITMAPINFOHEADER)
                     +   Bitmap.Height
                      * (Bitmap.Width+4) * 3);
    WITH TBITMAPINFOHEADER(aStream.Memory^) DO
     BEGIN
       biSize := SizeOf(TBITMAPINFOHEADER);
       biWidth := Bitmap.Width;
       biHeight := Bitmap.Height;
       biPlanes := 1;
       biBitCount := 24;
       biCompression := bi_RGB;
       biSizeImage := 0;
       biXPelsPerMeter :=1;
       biYPelsPerMeter :=1;
       biClrUsed :=0;
       biClrImportant :=0;
     END;

     dc := GetDC(0);
     Aptr := Pchar(aStream.Memory) +SizeOf (TBITMAPINFOHEADER);
     GetDIBits(dc,
              Bitmap.Handle,
              0,
              Bitmap.Height,
              Aptr,
              TBITMAPINFO(aStream.Memory^),
              dib_RGB_Colors);

     linewidth := (Bitmap.Width*3);
     linewidth := ((linewidth+3)DIV 4)*4;


     GetMem(ptr3,linewidth);
     IF NOT (ptr3 = NIL) THEN
      FOR j:=0 TO Bitmap.Height DIV 2 DO
       BEGIN
        ptr1 := Aptr + j*linewidth;
        ptr2 := Aptr + (Bitmap.Height-j)*linewidth;
        Move(ptr1^,ptr3^,linewidth);
        Move(ptr2^,ptr1^,linewidth);
        Move(ptr3^,ptr2^,linewidth);
       END;

     FreeMem(ptr3);



     Bitmap.Handle :=
           CreateDIBitmap(dc,
               TBITMAPINFOHEADER(aStream.Memory^),
               cbm_Init,
               Aptr,
               TBITMAPINFO(aStream.Memory^),
               dib_RGB_Colors);
     aStream.Free;
     ReleaseDC(0,dc);
END;

PROCEDURE BitmapRotate90(Bitmap : TBitmap);
VAR aStream,bstream : Tmemorystream;
    dc:hDC;
    Aptr,Bptr: Pchar;
    linewidth,blinewidth:INTEGER;

VAR j,k : INTEGER;
 ptr1 : Pchar;
 ptr2 : Pchar;

BEGIN
   aStream := Tmemorystream.Create;
   aStream.SetSize(  SizeOf (TBITMAPINFOHEADER)
                    +   Bitmap.Height
                     * (Bitmap.Width+4) * 3);
   bstream := Tmemorystream.Create;
   bstream.SetSize(  SizeOf (TBITMAPINFOHEADER)
                    +   Bitmap.Width
                     * (Bitmap.Height+4) * 3);
   WITH TBITMAPINFOHEADER(aStream.Memory^) DO
    BEGIN
      biSize := SizeOf(TBITMAPINFOHEADER);
      biWidth := Bitmap.Width;
      biHeight := Bitmap.Height;
      biPlanes := 1;
      biBitCount := 24;
      biCompression := bi_RGB;
      biSizeImage := 0;
      biXPelsPerMeter :=1;
      biYPelsPerMeter :=1;
      biClrUsed :=0;
      biClrImportant :=0;
    END;

   WITH TBITMAPINFOHEADER(bstream.Memory^) DO
    BEGIN
      biSize := SizeOf(TBITMAPINFOHEADER);
      biWidth := Bitmap.Height;
      biHeight := Bitmap.Width;
      biPlanes := 1;
      biBitCount := 24;
      biCompression := bi_RGB;
      biSizeImage := 0;
      biXPelsPerMeter :=1;
      biYPelsPerMeter :=1;
      biClrUsed :=0;
      biClrImportant :=0;
    END;

    dc := GetDC(0);
    Aptr := Pchar(aStream.Memory) +SizeOf (TBITMAPINFOHEADER);
    Bptr := Pchar(bstream.Memory) +SizeOf (TBITMAPINFOHEADER);
    GetDIBits(dc,
             Bitmap.Handle,
             0,
             Bitmap.Height,
             Aptr,
             TBITMAPINFO(aStream.Memory^),
             dib_RGB_Colors);

    linewidth := (Bitmap.Width*3);
    linewidth := ((linewidth+3)DIV 4)*4;
    blinewidth := (Bitmap.Height*3);
    blinewidth := ((blinewidth+3)DIV 4)*4;


    FOR j:=0 TO Bitmap.Height DO
    FOR k:=0 TO Bitmap.Width DO
     BEGIN
      ptr1 := Aptr + j*linewidth + k * 3;
      ptr2 := Bptr + k* blinewidth + (Bitmap.Height-j)*3;
      Move(ptr1^,ptr2^,3);
     END;


    Bitmap.Handle :=
          CreateDIBitmap(dc,
              TBITMAPINFOHEADER(bstream.Memory^),
              cbm_Init,
              Bptr,
              TBITMAPINFO(bstream.Memory^),
              dib_RGB_Colors);
    aStream.Free;
    bStream.Free;
    ReleaseDC(0,dc);
END;


type PByte  = ^Byte;

procedure BitmapGray(Bitmap : TBitmap);
VAR aStream : Tmemorystream;
    dc:hDC;
    Aptr: PChar;
    linewidth:INTEGER;

VAR j, k, e : INTEGER;
 ptr : PChar;
 tc, ac : TMyColor;
 y: Integer;


 procedure getColor(p : PChar; var c : TMyColor);
 begin
   c.B := Byte(p^);
   c.G := Byte((p+1)^);
   c.R := Byte((p+2)^);
 end;

 procedure setColor(p : PChar; var c : TMyColor);
 begin
   p^ := Char(c.B);
   (p+1)^ := Char(c.G);
   (p+2)^ := Char(c.R);
 end;


BEGIN
   aStream := Tmemorystream.Create;
   aStream.SetSize(  SizeOf (TBITMAPINFOHEADER)
                    +   Bitmap.Height
                     * (Bitmap.Width+4) * 3);
   WITH TBITMAPINFOHEADER(aStream.Memory^) DO
    BEGIN
      biSize := SizeOf(TBITMAPINFOHEADER);
      biWidth := Bitmap.Width;
      biHeight := Bitmap.Height;
      biPlanes := 1;
      biBitCount := 24;
      biCompression := bi_RGB;
      biSizeImage := 0;
      biXPelsPerMeter :=1;
      biYPelsPerMeter :=1;
      biClrUsed :=0;
      biClrImportant :=0;
    END;

    dc := GetDC(0);
    Aptr := PChar(aStream.Memory) +SizeOf (TBITMAPINFOHEADER);
    GetDIBits(dc,
             Bitmap.Handle,
             0,
             Bitmap.Height,
             Aptr,
             TBITMAPINFO(aStream.Memory^),
             dib_RGB_Colors);

    linewidth := (Bitmap.Width*3);
    linewidth := ((linewidth+3)DIV 4)*4;
    tc := TMyColor(Bitmap.TransparentColor);

    For j:= 0 TO Bitmap.Height do begin
      k := (j*lineWidth);
      e := (j+1)* lineWidth;
      while k <  e do begin
          ptr := APtr + k;
          Inc(k, 3);
          getColor(ptr, ac);
          if (tc.R <> ac.R) OR (tc.G <> ac.G) OR (tc.B <> ac.B) then begin
           y := (ac.R + ac.G + ac.B) div 3;
           if y > 200 then y := 200;
           ac.R := y;
           ac.G := y;
           ac.B := y;
           setColor(ptr, ac);
           {
            else Inc(ac.R,180);
           if (ac.G + 180) > 255 then ac.G := 255
            else Inc(ac.G,180);
            setColor(ptr, ac);
            }
           { if (val + 100) > 255 then val := 255
            else Inc(val,100);}
           {if (ac.R + 100) > 255 then ac.R := 255
            else Inc(ac.R,100);
           if (ac.G + 100) > 255 then ac.G := 255
            else Inc(ac.G,100);
           if (ac.B + 100) > 255 then ac.B := 255
            else Inc(ac.B,100);
            setColor(ptr, ac);
            }

{
            y :=  (ac.R+ac.G+ac.B)/ 1.02;
            u := (y-ac.R) *1.8;
            v := (y-ac.G) *1.8;
            x := (y-ac.B) *1.8;
            ac.R := Round(y-u);
            ac.G := Round(y-v);
            ac.B := Round(y-x);
            SetColor(ptr, ac);
 }
            {if (k > 0) then begin
                ptr := APtr + (k-3);
                getColor(ptr, ac);
                if (tc.R = ac.R) AND (tc.G = ac.G) AND (tc.B = ac.B) then begin
                  ac.R := 255; ac.G := 255; ac.B := 0;
                  setColor(ptr, ac)
                end;
            end;
             }
          end;
      end;
    end;

        {
    FOR j:=0 TO Bitmap.Height * linewidth DO begin
        ptr1 := Aptr + j;
        val := Byte(ptr1^);
        if (val + 100) > 255 then val := 255
        else Inc(val,100);
        ptr1^ := Chr(val);
     END;
     }

    Bitmap.Handle :=
          CreateDIBitmap(dc,
              TBITMAPINFOHEADER(astream.Memory^),
              cbm_Init,
              Aptr,
              TBITMAPINFO(astream.Memory^),
              dib_RGB_Colors);
    aStream.Free;
    ReleaseDC(0,dc);
END;

end.
