//
//  Copyright 1996, Artemis Alliance, Inc.
//  St. Paul, Mn 55101
//  (612) 227-7172
//
unit frmcube3;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  GL, GLU;

type
  TfrmCube = class(TForm)
    procedure FormCreate(Sender: TObject);
    procedure FormResize(Sender: TObject);
    procedure FormPaint(Sender: TObject);
    procedure FormDestroy(Sender: TObject);

  private
    hrc: HGLRC;
    Palette: HPALETTE;

    procedure DrawScene;
    procedure SetDCPixelFormat;

  protected
    procedure WMQueryNewPalette(var Msg : TWMQueryNewPalette); message WM_QUERYNEWPALETTE;
    procedure WMPaletteChanged(var Msg : TWMPaletteChanged); message WM_PALETTECHANGED;

  public
  end;

var
  frmCube: TfrmCube;

implementation

{$R *.DFM}

procedure TfrmCube.SetDCPixelFormat;
var
  hHeap: THandle;
  nColors, i: Integer;
  lpPalette: PLogPalette;
  byRedMask, byGreenMask, byBlueMask: Byte;
  nPixelFormat: Integer;
  DC: HDC;
  pfd: TPixelFormatDescriptor;

begin
  DC := Canvas.Handle;
  FillChar(pfd, SizeOf(pfd), 0);

  with pfd do begin
    nSize     := sizeof(pfd);                               // Size of this structure
    nVersion  := 1;                                         // Version number
    dwFlags   := PFD_DRAW_TO_WINDOW or PFD_SUPPORT_OPENGL;  // Flags
    iPixelType:= PFD_TYPE_RGBA;                             // RGBA pixel values
    cColorBits:= 24;                                        // 24-bit color
    cDepthBits:= 32;                                        // 32-bit depth buffer
    iLayerType:= PFD_MAIN_PLANE;                            // Layer type
  end;

  nPixelFormat := ChoosePixelFormat(DC, @pfd);
  SetPixelFormat(DC, nPixelFormat, @pfd);

  DescribePixelFormat(DC, nPixelFormat, sizeof(TPixelFormatDescriptor), pfd);

  if ((pfd.dwFlags and PFD_NEED_PALETTE) <> 0) then begin
    {nColors = 1 << pfd.cColorBits;}
    nColors   := 1 shl pfd.cColorBits;
    hHeap     := GetProcessHeap;
    lpPalette := HeapAlloc(hHeap, 0, sizeof(TLogPalette) + (nColors * sizeof(TPaletteEntry)));

    lpPalette^.palVersion := $300;
    lpPalette^.palNumEntries := nColors;

    byRedMask   := (1 shl pfd.cRedBits) - 1;
    byGreenMask := (1 shl pfd.cGreenBits) - 1;
    byBlueMask  := (1 shl pfd.cBlueBits) - 1;

    for i := 0 to nColors - 1 do begin
      lpPalette^.palPalEntry[i].peRed   := (((i shr pfd.cRedShift)   and byRedMask)   * 255) DIV byRedMask;
      lpPalette^.palPalEntry[i].peGreen := (((i shr pfd.cGreenShift) and byGreenMask) * 255) DIV byGreenMask;
      lpPalette^.palPalEntry[i].peBlue  := (((i shr pfd.cBlueShift)  and byBlueMask)  * 255) DIV byBlueMask;
      lpPalette^.palPalEntry[i].peFlags := 0;
    end;

    Palette := CreatePalette(lpPalette^);
    HeapFree(hHeap, 0, lpPalette);

    if (Palette <> 0) then begin
      SelectPalette(DC, Palette, False);
      RealizePalette(DC);
    end;
  end;
end;

procedure TfrmCube.DrawScene;
const
  glfLightAmbient : Array[0..3] of GLfloat = (0.1, 0.1, 0.1, 1.0);
  glfLightDiffuse : Array[0..3] of GLfloat = (0.7, 0.7, 0.7, 1.0);
  glfLightSpecular: Array[0..3] of GLfloat = (0.0, 0.0, 0.0, 1.0);
  glfMaterialColor: Array[0..3] of GLfloat = (0.0, 0.0, 1.0, 1.0);

begin
  //
  // Enable depth testing and clear the color and depth buffers.
  //
  glEnable(GL_DEPTH_TEST);
  glClear(GL_COLOR_BUFFER_BIT or GL_DEPTH_BUFFER_BIT);


  //
  // Add a light to the scene.
  //
  glMatrixMode(GL_MODELVIEW);
  glLoadIdentity;
  glLightfv(GL_LIGHT0, GL_AMBIENT, @glfLightAmbient);
  glLightfv(GL_LIGHT0, GL_DIFFUSE, @glfLightDiffuse);
  glLightfv(GL_LIGHT0, GL_SPECULAR,@glfLightSpecular);
  glEnable(GL_LIGHTING);
  glEnable(GL_LIGHT0);

  //
  // Define the modelview transformation.
  //
  glTranslatef(0.0, 0.0, -8.0);
  glRotatef(30.0, 1.0, 0.0, 0.0);
  glRotatef(70.0, 0.0, 1.0, 0.0);

  //
  // Define the reflective properties of the cube's faces.
  //
  glMaterialfv(GL_FRONT, GL_AMBIENT_AND_DIFFUSE, @glfMaterialColor);

  //
  // Draw the six faces of the cube.
  //
  glBegin(GL_POLYGON);
    glNormal3f(0.0, 0.0, 1.0);
    glVertex3f(1.0, 1.0, 1.0);
    glVertex3f(-1.0, 1.0, 1.0);
    glVertex3f(-1.0, -1.0, 1.0);
    glVertex3f(1.0, -1.0, 1.0);
  glEnd;

  glBegin(GL_POLYGON);
    glNormal3f(0.0, 0.0, -1.0);
    glVertex3f(1.0, 1.0, -1.0);
    glVertex3f(1.0, -1.0, -1.0);
    glVertex3f(-1.0, -1.0, -1.0);
    glVertex3f(-1.0, 1.0, -1.0);
  glEnd;

  glBegin(GL_POLYGON);
    glNormal3f(-1.0, 0.0, 0.0);
    glVertex3f(-1.0, 1.0, 1.0);
    glVertex3f(-1.0, 1.0, -1.0);
    glVertex3f(-1.0, -1.0, -1.0);
    glVertex3f(-1.0, -1.0, 1.0);
  glEnd;

  glBegin(GL_POLYGON);
    glNormal3f(1.0, 0.0, 0.0);
    glVertex3f(1.0, 1.0, 1.0);
    glVertex3f(1.0, -1.0, 1.0);
    glVertex3f(1.0, -1.0, -1.0);
    glVertex3f(1.0, 1.0, -1.0);
  glEnd;

  glBegin(GL_POLYGON);
    glNormal3f(0.0, 1.0, 0.0);
    glVertex3f(-1.0, 1.0, -1.0);
    glVertex3f(-1.0, 1.0, 1.0);
    glVertex3f(1.0, 1.0, 1.0);
    glVertex3f(1.0, 1.0, -1.0);
  glEnd;

  glBegin(GL_POLYGON);
    glNormal3f(0.0, -1.0, 0.0);
    glVertex3f(-1.0, -1.0, -1.0);
    glVertex3f(1.0, -1.0, -1.0);
    glVertex3f(1.0, -1.0, 1.0);
    glVertex3f(-1.0, -1.0, 1.0);
  glEnd;

  //
  // Flush the drawing pipeline.
  //
  glFlush;
end;


// EVENT HANDLERS

procedure TfrmCube.FormCreate(Sender: TObject);
begin
  // Create a rendering context.
  SetDCPixelFormat;
  hrc := wglCreateContext(Canvas.Handle);
end;

procedure TfrmCube.FormResize(Sender: TObject);
var
  gldAspect : GLdouble;

begin
  // Redefine the viewing volume and viewport when the window size changes.
  wglMakeCurrent(Canvas.Handle, hrc);
  gldAspect := Width / Height;

  glMatrixMode(GL_PROJECTION);
  glLoadIdentity;
  gluPerspective(30.0,           // Field-of-view angle
                 gldAspect,      // Aspect ratio of viewing volume
                 1.0,            // Distance to near clipping plane
                 10.0);          // Distance to far clipping plane
  glViewport(0, 0, Width, Height);
  wglMakeCurrent(0, 0);
  Invalidate;
end;

procedure TfrmCube.FormPaint(Sender: TObject);
begin
  // Draw the scene.
  wglMakeCurrent(Canvas.Handle, hrc);
  DrawScene;
  wglMakeCurrent(0, 0);
end;

procedure TfrmCube.FormDestroy(Sender: TObject);
begin
  // Clean up and terminate.
  wglDeleteContext(hrc);

  if (Palette <> 0) then
      DeleteObject(Palette);
end;

procedure TfrmCube.WMQueryNewPalette(var Msg : TWMQueryNewPalette);
begin
  //
  // If the program is using a color palette, realize the palette
  // and update the client area when the window receives the input
  // focus.
  //
  if (Palette <> 0) then begin
    SelectPalette(Canvas.Handle, Palette, False);
    Msg.Result := RealizePalette(Canvas.Handle);

    if (Msg.Result <> GDI_ERROR) then
      Invalidate;
  end;
end;

procedure TfrmCube.WMPaletteChanged(var Msg : TWMPaletteChanged);
begin
  //
  // If the program is using a color palette, realize the palette
  // and update the colors in the client area when another program
  // realizes its palette.
  //
  if ((Palette <> 0) and (THandle(TMessage(Msg).wParam) <> Handle)) then begin
    SelectPalette(Canvas.Handle, Palette, False);

    if (RealizePalette(Canvas.Handle) <> GDI_ERROR) then
        UpdateColors(Canvas.Handle);

    Msg.Result := 0;
  end;
end;

end.

