{*******************************************************}
{                                                       }
{         Vladimir Gaitanoff Delphi VCL Library         }
{         TDispInvokeCache, English locale hook         }
{                                                       }
{         Copyright (c) 1997, 1998                      }
{                                                       }
{*******************************************************}

{$I VG.INC }
{$D-,L-}

unit OleFix;

interface
uses Windows, Classes, OleConst, SysUtils, {$IFDEF _D3_} ActiveX, ComObj {$ELSE} Ole2, OleAuto {$ENDIF};

{$IFDEF _D3_}
const
  MaxDispArgs = 32;
{$ENDIF _D3_}

type
  TDispInvokeCache = class
  private
    FDispatch: {$IFDEF _D3_} Pointer {$ELSE} IDispatch {$ENDIF};
    FEntries: TList;
    function FindCallDesc(CallDesc: TCallDesc): Integer;
    function GetDispIDs(ADispatch:{$IFDEF _D3_} Pointer {$ELSE} IDispatch {$ENDIF};
      CallDesc: PCallDesc; Params: Pointer): Boolean;
    procedure UpdateCache(CallDesc: PCallDesc; Params: Pointer);
  public
    constructor Create;
    destructor Destroy; override;
    procedure ClearCache;
    procedure Reset(Instance: Variant);
    procedure Copy(Cache: TDispInvokeCache);
    procedure Read(Stream: TStream);
    procedure Write(Stream: TStream);
  end;

implementation

type
  TNames = array [0..1023] of Char;
  TDispIDs = array[0..MaxDispArgs - 1] of Integer;

  PDispInvokeEntry = ^TDispInvokeEntry;
  TDispInvokeEntry = record
    Size: Word;
    Names: TNames;
    Count: Byte;
    DispIDs: TDispIDs;
  end;

function CompareMem(P1, P2: Pointer; Length: Integer): Boolean; assembler;
asm
        PUSH    ESI
        PUSH    EDI
        MOV     ESI,P1
        MOV     EDI,P2
        MOV     EDX,ECX
        XOR     EAX,EAX
        AND     EDX,3
        SHR     ECX,1
        SHR     ECX,1
        REPE    CMPSD
        JNE     @@2
        MOV     ECX,EDX
        REPE    CMPSB
        JNE     @@2
@@1:    INC     EAX
@@2:    POP     EDI
        POP     ESI
end;

procedure WideCharToNames(Names: PChar; NameCount: Byte; var WideNames: TNames; var Size: Word);
var
  I, N: Integer;
  Ch: WideChar;
begin
  I := 0;
  N := 0;
  Size := 0;
  repeat
    repeat
      Ch := WideChar(Names[I]);
      WideNames[I] := Char(Ch);
      Inc(Size);
      Inc(I);
    until Char(Ch) = #0;
    Inc(N);
  until N = NameCount;
end;

const
  EnglishLocale = (LANG_ENGLISH + SUBLANG_DEFAULT * 1024) + (SORT_DEFAULT shl 16);

var
  FCacheList: TList = nil;

function FindCache(Dispatch:{$IFDEF _D3_} Pointer {$ELSE} IDispatch {$ENDIF}): TDispInvokeCache;
var
  I: Integer;
begin
  if Assigned(FCacheList) then
  for I := 0 to FCacheList.Count - 1 do
  begin
    Result := FCacheList[I];
    if Result.FDispatch = Dispatch then Exit;
  end;
  Result := nil;
end;

constructor TDispInvokeCache.Create;
begin
  if not Assigned(FCacheList) then FCacheList := TList.Create;
  FCacheList.Add(Self);
end;

destructor TDispInvokeCache.Destroy;
begin
  FCacheList.Remove(Self);
  if FCacheList.Count = 0 then
  begin
    FCacheList.Free;
    FCacheList := nil;
  end;
  ClearCache;
  inherited;
end;

procedure TDispInvokeCache.ClearCache;
var
  P: PDispInvokeEntry;
begin
  if Assigned(FEntries) then
  begin
    while FEntries.Count > 0 do
    begin
      P := FEntries.Last;
      ReallocMem(P, 0);
      FEntries.Delete(FEntries.Count - 1);
    end;
    FEntries.Free;
    FEntries := nil;
  end;
end;

function TDispInvokeCache.FindCallDesc(CallDesc: TCallDesc): Integer;
var
  P: PDispInvokeEntry;
  I: Integer;
  Size: Word;
  Names: TNames;
begin
  Result := -1;
  if Assigned(FEntries) then
  begin
    WideCharToNames(@CallDesc.ArgTypes[CallDesc.ArgCount], CallDesc.NamedArgCount + 1, Names, Size);
    for I := 0 to FEntries.Count - 1 do
    begin
      P := FEntries[I];
      if (Size = P^.Size) and (CompareMem(@P^.Names, @Names, Size)) then
      begin
        Result := I;
        Break;
      end;
    end;
  end;
end;

function TDispInvokeCache.GetDispIDs(ADispatch:{$IFDEF _D3_} Pointer {$ELSE} IDispatch {$ENDIF};
  CallDesc: PCallDesc; Params: Pointer): Boolean;
var
  P: PDispInvokeEntry;
  I: Integer;
begin
  I := FindCallDesc(CallDesc^);
  if (I >= 0) then
  begin
    P := FEntries[I];
    Move(P^.DispIDs, Params^, SizeOf(TDispIDs));
    Result := True;
  end else
    Result := False;
end;

procedure TDispInvokeCache.UpdateCache(CallDesc: PCallDesc; Params: Pointer);
var
  P: PDispInvokeEntry;
  I: Integer;
begin
  I := FindCallDesc(CallDesc^);
  if (I < 0) then
  begin
    GetMem(P, SizeOf(TDispInvokeEntry));
    if not Assigned(FEntries) then FEntries := TList.Create;
    FEntries.Add(P);
    P^.Count := CallDesc^.NamedArgCount + 1;
    WideCharToNames(@CallDesc.ArgTypes[CallDesc.ArgCount], P^.Count, P^.Names, P^.Size);
    Move(Params^, P^.DispIDs, SizeOf(TDispIDs));
  end;
end;

procedure TDispInvokeCache.Reset(Instance: Variant);
begin
  if not VarIsEmpty(Instance) then
  begin
    {$IFDEF _D3_}
    if TVarData(Instance).VType = varDispatch then
      FDispatch := TVarData(Instance).VDispatch
    else if TVarData(Instance).VType = (varDispatch or varByRef) then
      FDispatch := Pointer(TVarData(Instance).VPointer^)
    else
      raise EOleError.Create(SVarNotObject);
    {$ELSE}
    FDispatch := VarToInterface(Instance);
    {$ENDIF}
  end else
    FDispatch := nil;
end;

procedure TDispInvokeCache.Copy(Cache: TDispInvokeCache);
var
  F: TStream;
begin
  F := TMemoryStream.Create;
  try
    Cache.Write(F);
    F.Position := 0;
    Read(F);
  finally
    F.Free;
  end;
end;

procedure TDispInvokeCache.Read(Stream: TStream);
var
  P: PDispInvokeEntry;
  I, Count: Integer;
begin
  ClearCache;
  Stream.ReadBuffer(Count ,SizeOf(I));
  for I := 0 to Count - 1 do
  begin
    GetMem(P, SizeOf(P^));
    if not Assigned(FEntries) then FEntries := TList.Create;
    FEntries.Add(P);
    with P^ do
    begin
      Stream.ReadBuffer(Size, SizeOf(Size));
      Stream.ReadBuffer(Names, Size);
      Stream.ReadBuffer(Count, SizeOf(Count));
      Stream.ReadBuffer(DispIDs, Count * SizeOf(Integer));
    end;
  end;
end;

procedure TDispInvokeCache.Write(Stream: TStream);
var
  P: PDispInvokeEntry;
  I: Integer;
begin
  if Assigned(FEntries) then
  begin
    I := FEntries.Count;
    Stream.WriteBuffer(I ,SizeOf(I));
    for I := 0 to FEntries.Count - 1 do
    begin
      P := FEntries[I];
      with P^ do
      begin
        Stream.WriteBuffer(Size, SizeOf(Size));
        Stream.WriteBuffer(Names, Size);
        Stream.WriteBuffer(Count, SizeOf(Count));
        Stream.WriteBuffer(DispIDs, Count * SizeOf(Integer));
      end;
    end;
  end else begin
    I := 0;
    Stream.WriteBuffer(I ,SizeOf(I));
  end;
end;

{$IFDEF _D3_ }
procedure GetIDsOfNamesFix(const Dispatch: IDispatch; Names: PChar;
  NameCount: Integer; DispIDs: PDispIDList);

  procedure Error;
  begin
    raise EOleError.CreateFmt(SNoMethod, [Names]);
  end;

type
  PNamesArray = ^TNamesArray;
  TNamesArray = array[0..MaxDispArgs - 1] of PWideChar;
var
  N, SrcLen, DestLen: Integer;
  Src: PChar;
  Dest: PWideChar;
  NameRefs: PNamesArray;
  StackTop: Pointer;
  Temp: Integer;
begin
  Src := Names;
  N := 0;
  asm
    MOV  StackTop, ESP
    MOV  EAX, NameCount
    INC  EAX
    SHL  EAX, 2  // sizeof pointer = 4
    SUB  ESP, EAX
    LEA  EAX, NameRefs
    MOV  [EAX], ESP
  end;
  repeat
    SrcLen := StrLen(Src);
    DestLen := MultiByteToWideChar(0, 0, Src, SrcLen, nil, 0) + 1;
    asm
      MOV  EAX, DestLen
      ADD  EAX, EAX
      ADD  EAX, 3      // round up to 4 byte boundary
      AND  EAX, not 3
      SUB  ESP, EAX
      LEA  EAX, Dest
      MOV  [EAX], ESP
    end;
    if N = 0 then NameRefs[0] := Dest else NameRefs[NameCount - N] := Dest;
    MultiByteToWideChar(0, 0, Src, SrcLen, Dest, DestLen);
    Dest[DestLen-1] := #0;
    Inc(Src, SrcLen+1);
    Inc(N);
  until N = NameCount;

  Temp := Dispatch.GetIDsOfNames(GUID_NULL, NameRefs, NameCount,
    EnglishLocale, DispIDs);

  if Temp = DISP_E_UNKNOWNNAME then Error else OleCheck(Temp);
  asm
    MOV  ESP, StackTop
  end;
end;
{$ELSE}
procedure GetIDsOfNamesFix(Dispatch: IDispatch; Names: PChar;
  NameCount: Integer; DispIDs: PDispIDList);
var
  I, N: Integer;
  Ch: WideChar;
  P: PWideChar;
  NameRefs: array[0..MaxDispArgs - 1] of PWideChar;
  WideNames: array[0..1023] of WideChar;
begin
  I := 0;
  N := 0;
  repeat
    P := @WideNames[I];
    if N = 0 then NameRefs[0] := P else NameRefs[NameCount - N] := P;
    repeat
      Ch := WideChar(Names[I]);
      WideNames[I] := Ch;
      Inc(I);
    until Char(Ch) = #0;
    Inc(N);
  until N = NameCount;

  if Dispatch.GetIDsOfNames(GUID_NULL, @NameRefs, NameCount,
       EnglishLocale, DispIDs) <> 0 then
    raise EOleError.CreateResFmt(SNoMethod, [Names]);
end;
{$ENDIF}

procedure VarDispInvokeFix(Result: PVariant; const Instance: Variant;
  CallDesc: PCallDesc; Params: Pointer); cdecl;
var
  Dispatch: {$IFDEF _D3_} Pointer {$ELSE} IDispatch {$ENDIF};
  DispIDs: array[0..MaxDispArgs - 1] of Integer;
  Cache: TDispInvokeCache;
begin
{$IFDEF _D3_}
  if TVarData(Instance).VType = varDispatch then
    Dispatch := TVarData(Instance).VDispatch
  else if TVarData(Instance).VType = (varDispatch or varByRef) then
    Dispatch := Pointer(TVarData(Instance).VPointer^)
  else
    raise EOleError.Create(SVarNotObject);
  Cache := FindCache(Dispatch);
  if not (Assigned(Cache) and Cache.GetDispIDs(Dispatch, CallDesc, @DispIDs)) then
  begin
    GetIDsOfNamesFix(IDispatch(Dispatch), @CallDesc^.ArgTypes[CallDesc^.ArgCount],
      CallDesc^.NamedArgCount + 1, @DispIDs);
    if Assigned(Cache) then
      Cache.UpdateCache(CallDesc, @DispIDs);
  end;

  if Result <> nil then VarClear(Result^);
  DispatchInvoke(IDispatch(Dispatch), CallDesc, @DispIDs, @Params, Result);
{$ELSE}
  Dispatch := VarToInterface(Instance);

  Cache := FindCache(Dispatch);
  if not (Assigned(Cache) and Cache.GetDispIDs(Dispatch, CallDesc, @DispIDs)) then
  begin
    GetIDsOfNamesFix(Dispatch, @CallDesc^.ArgTypes[CallDesc^.ArgCount],
      CallDesc^.NamedArgCount + 1, @DispIDs);
    if Assigned(Cache) then
      Cache.UpdateCache(CallDesc, @DispIDs);
  end;

  if Result <> nil then VarClear(Result^);
  DispInvoke(Dispatch, CallDesc, @DispIDs, @Params, Result);
{$ENDIF}
end;

initialization
  VarDispProc := @VarDispInvokeFix;

finalization

end.
