unit Balloon;

interface

uses
  {$IFDEF WIN32} Windows, {$ELSE} WinTypes, WinProcs, {$ENDIF}
  SysUtils, Messages, Classes, Graphics, Controls, Forms,
  Dialogs, StdCtrls;

type
  TOffBalnPosition = (bsAuto, bsTop, bsLeft, bsRight);
  TOffice97Balloon = class(TForm)
    Button1: TButton;
    Button2: TButton;
    Button3: TButton;
    procedure Button1Click(Sender: TObject);
    procedure Button2Click(Sender: TObject);
    procedure Button3Click(Sender: TObject);
  private
    { Private declarations }
    fFit,fOldH,fOldL,fOldT,fOldW,fOldPos,fZoom: Integer;
    fShowCaption: Boolean;
    fHideOnClick,fMoving,fVisible: Boolean;
    fFrameColor: TColor;
    fCustomGlyph: TBitmap;
    fBalloonPosition: TOffBalnPosition;
    fStickTo: TForm;
    fClientInstance: TFarProc;
    fDefClientProc: TFarProc;
    procedure ClientWndProc(var Message: TMessage);
    procedure DrawTransparentBitmap(Dest:TCanvas;const X,Y:Smallint;srcBmp:TBitmap;const transpColor:TColor);
    procedure SetBalloonPosition(Val: TOffBalnPosition);
    procedure SetCustomGlyph(Val: TBitmap);
    procedure SetFrameColor(Val: TColor);
    procedure SetHideOnClick(Val: Boolean);
    procedure SetShowCaption(Val: Boolean);
    procedure SetStickTo(Val: TForm);
    function ControlFits(cPosition: TOffBalnPosition):Integer;
    {$IFDEF WIN32}
    procedure MakePoly(const Points: array of TPoint);
    {$ENDIF}
    procedure DrawFrame;
    procedure PositionChilds(OldPos,NewPos: integer);
    function PositionControl:Integer;
    procedure Zoom;
    procedure WMLButtonDown(var Message: TWMLButtonDown); message WM_LBUTTONDOWN;
  protected
    { Protected declarations }
  public
    { Public declarations }
    procedure Show;
    procedure Paint; override;
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
  published
    { Published declarations }
    property Font;
    property BalloonPosition: TOffBalnPosition read fBalloonPosition write SetBalloonPosition default bsAuto;
    property CustomGlyph: TBitmap read fCustomGlyph write SetCustomGlyph;
    property FrameColor: TColor read fFrameColor write SetFrameColor default clBlack;
    property HideOnClick: Boolean read fHideOnClick write SetHideOnClick default True;
    property ShowCaption: Boolean read fShowCaption write SetShowCaption default True;
    property StickTo: TForm read fStickTo write SetStickTo;
  end;

var
  Office97Balloon: TOffice97Balloon;

implementation

{$R *.DFM}

uses ExtCtrls, Main;

{ Utility Routines }

procedure Delay(msecs:integer);
var
  FirstTickCount:longint;
begin
  FirstTickCount:=GetTickCount;
  while ((GetTickCount-FirstTickCount) < Longint(msecs)) do
    Application.ProcessMessages;
end;

{ TOffice97Balloon }

const
  CopyRightStr: PChar = '<- TOFFICE97BALLOON Component v1.2 by Jonathan Hosking (15/04/1999) ->';
var
  CopyRightPtr: Pointer;

{ This procedure draws a transparent bitmap }

procedure TOffice97Balloon.DrawTransparentBitmap(Dest:TCanvas;const X,Y:Smallint;srcBmp:TBitmap;const transpColor:TColor);
var
  ANDBitmap,ORBitmap: TBitmap;
  oldCopyMode: TCopyMode;
  src: TRect;
begin
  try
    ANDBitmap := TBitmap.Create;
    ORBitmap := TBitmap.Create;
    Src := Bounds(0,0,srcBmp.Width,srcBmp.Height);
    with ORBitmap do
    begin
      Width := srcBmp.Width;
      Height := srcBmp.Height;
      Canvas.Brush.Color := clBlack;
      Canvas.CopyMode := cmSrcCopy;
      Canvas.BrushCopy(Src,srcBmp,Src,transpColor);
    end;
    with ANDBitmap do
    begin
      Width := srcBmp.Width;
      Height := srcBmp.Height;
      Canvas.Brush.Color := clWhite;
      Canvas.CopyMode := cmSrcInvert;
      Canvas.BrushCopy(Src,srcBmp,Src,transpColor);
    end;
    with Dest do
    begin
      oldCopyMode := CopyMode;
      CopyMode := cmSrcAnd;
      Draw(x,y,ANDBitmap);
      CopyMode := cmSrcPaint;
      Draw(x,y,ORBitmap);
      CopyMode := oldCopyMode;
    end;
  finally
    ORBitmap.Free;
    ANDBitmap.Free;
  end;
end;

constructor TOffice97Balloon.Create(AOwner: TComponent);
begin
  { Setup the control }
  Inherited Create(AOwner);
  CopyRightPtr := @CopyRightStr;
  BorderStyle := bsNone;
  BorderIcons := [];
  Color := $00CEFFFF;
  fCustomGlyph := TBitmap.Create;
  fBalloonPosition := bsAuto;
  fFrameColor := clBlack;
  fHideOnClick := True;
  fShowCaption := True;
  fVisible := True;
  fStickTo := nil;
  fOldH := 0;
  fOldL := 0;
  fOldT := 0;
  fOldW := 0;
  fOldPos := 0;
  fZoom := 0;
  Width := 200;
  Height := 200;
  if Assigned(fStickTo) then
  begin
    fMoving := False;
    fClientInstance := MakeObjectInstance(ClientWndProc);
    fDefClientProc := Pointer(GetWindowLong(fStickTo.Handle, GWL_WNDPROC));
    SetWindowLong(fStickTo.Handle, GWL_WNDPROC, LongInt(fClientInstance));
  end;
end;

destructor TOffice97Balloon.Destroy;
begin
  { Kill the control }
  if Assigned(fStickTo) then
  begin
     SetWindowLong(fStickTo.Handle, GWL_WNDPROC, LongInt(fDefClientProc));
     FreeObjectInstance(fClientInstance);
  end;
  fClientInstance := nil;
  fCustomGlyph.Free;
  Inherited Destroy;
end;

procedure TOffice97Balloon.ClientWndProc(var Message: TMessage);

procedure Default;
begin
  with Message
    do Result := CallWindowProc(fDefClientProc, fStickTo.Handle, Msg, wParam, lParam);
end;

begin
  with Message do
  begin
    case Msg of
      WM_MOVING,WM_SIZING: begin
                             if not fMoving then
                             begin
                               fOldW := fStickTo.Width;
                               fOldH := fStickTo.Height;
                               fOldL := fStickTo.Left;
                               fOldT := fStickTo.Top;
                               Default;
                               Invalidate;
                             end
                             else fMoving := True;
                           end;
      WM_MOVE,WM_SIZE: begin
                         Default;
                         PositionControl;
                         if (fOldW = fStickTo.Width) and (fOldH = fStickTo.Height)
                           and (fOldL = fStickTo.Left) and (fOldT = fStickTo.Top) then
                         begin
                           fMoving := False;
                           Show;
                           Invalidate;
                         end;
                       end;
      else Default;
    end;
  end;
end;

{ Start of component configuration routines }

procedure TOffice97Balloon.SetBalloonPosition(Val: TOffBalnPosition);
begin
  if fBalloonPosition <> Val then
  begin
    fBalloonPosition := Val;
    Invalidate;
  end;
end;

procedure TOffice97Balloon.SetCustomGlyph(Val: TBitmap);
begin
  fCustomGlyph.Assign(Val);
  Invalidate;
end;

procedure TOffice97Balloon.SetFrameColor(Val: TColor);
begin
  if fFrameColor <> Val then
  begin
    fFrameColor := Val;
    Invalidate;
  end;
end;

procedure TOffice97Balloon.SetHideOnClick(Val: Boolean);
begin
  if fHideOnClick <> Val then
  begin
    fHideOnClick := Val;
    Invalidate;
  end;
end;

procedure TOffice97Balloon.SetShowCaption(Val: Boolean);
begin
  if fShowCaption <> Val then
  begin
    fShowCaption := Val;
    Invalidate;
  end;
end;

procedure TOffice97Balloon.SetStickTo(Val: TForm);
begin
  if (fStickTo <> Val) and (Val <> Self) then
  begin
    if Assigned(fStickTo) then
    begin
      SetWindowLong(fStickTo.Handle, GWL_WNDPROC, LongInt(fDefClientProc));
      FreeObjectInstance(fClientInstance);
    end;
    fStickTo := Val;
    if Assigned(fStickTo) then
    begin
      fOldH := 0;
      fOldL := 0;
      fOldT := 0;
      fOldW := 0;
      fOldPos := 0;
      fMoving := False;
      fClientInstance := MakeObjectInstance(ClientWndProc);
      fDefClientProc := Pointer(GetWindowLong(fStickTo.Handle, GWL_WNDPROC));
      SetWindowLong(fStickTo.Handle, GWL_WNDPROC, LongInt(fClientInstance));
    end;
    Invalidate;
  end;
end;

{ End of component configuration routines }

procedure TOffice97Balloon.Show;
begin
  Inherited Show;
  if not fMoving then
    fZoom := 0;
  Invalidate;
end;

function TOffice97Balloon.ControlFits(cPosition: TOffBalnPosition):Integer;
var
  pL,pR,pT,sH,sW: Integer;
begin
  Result := 0;
  pL := fStickTo.Left;
  pT := fStickTo.Top;
  pR := pL + fStickTo.Width;
  sW := Screen.Width;
  sH := Screen.Height;
  case cPosition of
    bsTop:
      if (pT + 12 - Height) >= 0 then
      begin
        if ((pL + ((pR - pL) div 2)) <= (sW div 2)) then
        begin
          if ((pL + Width) <= sW) then
          begin
            if pL < 0 then Left := 0 else Left := pL;
            Top := pT + 12 - Height;
            Result := 1;
          end;
        end
        else
          if ((pR - Width) >= 0) then
          begin
            if pR > sW then Left := sW - Width else
              Left := pR - Width;
            Top := pT + 12 - Height;
            Result := 2;
          end;
      end;
    bsLeft:
      if ((pL + 12 - Width) >= 0) and ((pT + Height) <= sH) then
      begin
        Left := pL + 12 - Width;
        Top := pT;
        Result := 3;
      end;
    bsRight:
      if ((pR - 12 + Width) <= sW) and ((pT + Height) <= sH) then
      begin
        Left := pR - 12;
        Top := pT;
        Result := 4;
      end;
  end;
end;

{$IFDEF WIN32}
procedure TOffice97Balloon.MakePoly(const Points: array of TPoint);
type
  PPoints = ^TPoints;
  TPoints = array[0..0] of TPoint;
begin
  SetWindowRgn(Handle, CreatePolygonRgn(PPoints(@Points)^, High(Points) + 1, Winding), True);
end;
{$ENDIF}

procedure TOffice97Balloon.DrawFrame;
var
  rClient: TRect;
  fB,fL,fSl,fSw,fR: Integer;
begin
  if Assigned(fStickTo) then
  begin
    fSl := fStickTo.Left;
    fSw := fStickTo.Width;
  end;
  if fFit = 1 then
  begin
    if fSl < 0 then
      fL := fSl + fSw
    else
      fL := fSw;
    if fL > Width - 10 then fL := Width - 10;
    if fL < 15 then fL := 15;
  end
  else
  begin
    if fSl < 0 then
      fL := (fSl + fSw) - Left
    else
      fL := fSl - Left;
    if fL < -10 then fL := -10;
    if fL > Screen.Width - Left - 35 then
      fL := Screen.Width - Left - 35;
  end;
  rClient := ClientRect;
  with Canvas do
    with rClient do
    begin
      Pen.Color := fFrameColor;
      Pen.Style := psSolid;
      fB := Bottom;
      fR := Right;
      Case fFit of
        1: begin
             Polyline([Point(5,0),Point(5,1),Point(3,1),Point(3,2),
               Point(2,2),Point(2,3),Point(1,3),Point(1,5),
               Point(Left,5),Point(Left,fB-21),Point(1,fB-21),
               Point(1,fB-19),Point(2,fB-19),Point(2,fB-18),
               Point(4,fB-18),Point(4,fB-17),Point(fL-9,fB-17),
               Point(fL-9,fB-1),Point(fL-9,fB-2),Point(fL-8,fB-3),
               Point(fL-8,fB-4),Point(fL-7,fB-4),Point(fL-7,fB-6),
               Point(fL-6,fB-6),Point(fL-6,fB-8),Point(fL-5,fB-8),
               Point(fL-5,fB-9),Point(fL-4,fB-9),Point(fL-4,fB-11),
               Point(fL-3,fB-11),Point(fL-3,fB-12),Point(fL-2,fB-12),
               Point(fL-2,fB-14),Point(fL-1,fB-14),Point(fL-1,fB-16),
               Point(fL,fB-16),Point(fL,fB-17),Point(fR-5,fB-17),
               Point(fR-5,fB-18),Point(fR-3,fB-18),Point(fR-3,fB-19),
               Point(fR-2,fB-19),Point(fR-2,fB-21),Point(fR-1,fB-21),
               Point(fR-1,5),Point(fR-2,5),Point(fR-2,3),
               Point(fR-3,3),Point(fR-3,2),Point(fR-4,2),
               Point(fR-4,1),Point(fR-6,1),Point(fR-6,0),
               Point(5,0)]);
             {$IFDEF WIN32}
             inc(fB);
             inc(fL);
             inc(fR);
             MakePoly([Point(5,0),Point(5,1),Point(3,1),Point(3,2),
               Point(2,2),Point(2,3),Point(1,3),Point(1,5),
               Point(Left,5),Point(Left,fB-21),Point(1,fB-21),
               Point(1,fB-19),Point(2,fB-19),Point(2,fB-18),
               Point(4,fB-18),Point(4,fB-17),Point(fL-9,fB-17),
               Point(fL-9,fB-1),Point(fL-9,fB-2),Point(fL-8,fB-3),
               Point(fL-8,fB-4),Point(fL-7,fB-4),Point(fL-7,fB-6),
               Point(fL-6,fB-6),Point(fL-6,fB-8),Point(fL-5,fB-8),
               Point(fL-5,fB-9),Point(fL-4,fB-9),Point(fL-4,fB-11),
               Point(fL-3,fB-11),Point(fL-3,fB-12),Point(fL-2,fB-12),
               Point(fL-2,fB-14),Point(fL-1,fB-14),Point(fL-1,fB-16),
               Point(fL,fB-16),Point(fL,fB-17),Point(fR-5,fB-17),
               Point(fR-5,fB-18),Point(fR-3,fB-18),Point(fR-3,fB-19),
               Point(fR-2,fB-19),Point(fR-2,fB-21),Point(fR-1,fB-21),
               Point(fR-1,5),Point(fR-2,5),Point(fR-2,3),
               Point(fR-3,3),Point(fR-3,2),Point(fR-4,2),
               Point(fR-4,1),Point(fR-6,1),Point(fR-6,0),
               Point(5,0)]);
             {$ENDIF}
           end;
        2: begin
             Polyline([Point(5,0),Point(5,1),Point(3,1),Point(3,2),
               Point(2,2),Point(2,3),Point(1,3),Point(1,5),
               Point(Left,5),Point(Left,fB-21),Point(1,fB-21),
               Point(1,fB-19),Point(2,fB-19),Point(2,fB-18),
               Point(4,fB-18),Point(4,fB-17),Point(fL+19,fB-17),
               Point(fL+19,fB-16),Point(fL+20,fB-16),Point(fL+20,fB-14),
               Point(fL+21,fB-14),Point(fL+21,fB-13),Point(fL+22,fB-13),
               Point(fL+22,fB-11),Point(fL+23,fB-11),Point(fL+23,fB-9),
               Point(fL+24,fB-9),Point(fL+24,fB-8),Point(fL+25,fB-8),
               Point(fL+25,fB-6),Point(fL+26,fB-6),Point(fL+26,fB-5),
               Point(fL+27,fB-5),Point(fL+27,fB-3),Point(fL+28,fB-2),
               Point(fL+28,fB-1),Point(fL+28,fB-17),Point(fR-5,fB-17),
               Point(fR-5,fB-18),Point(fR-3,fB-18),Point(fR-3,fB-19),
               Point(fR-2,fB-19),Point(fR-2,fB-21),Point(fR-1,fB-21),
               Point(fR-1,5),Point(fR-2,5),Point(fR-2,3),Point(fR-3,3),
               Point(fR-3,2),Point(fR-4,2),Point(fR-4,1),Point(fR-6,1),
               Point(fR-6,0),Point(5,0)]);
             {$IFDEF WIN32}
             inc(fB);
             inc(fL);
             inc(fR);
             MakePoly([Point(5,0),Point(5,1),Point(3,1),Point(3,2),
               Point(2,2),Point(2,3),Point(1,3),Point(1,5),
               Point(Left,5),Point(Left,fB-21),Point(1,fB-21),
               Point(1,fB-19),Point(2,fB-19),Point(2,fB-18),
               Point(4,fB-18),Point(4,fB-17),Point(fL+19,fB-17),
               Point(fL+19,fB-16),Point(fL+20,fB-16),Point(fL+20,fB-14),
               Point(fL+21,fB-14),Point(fL+21,fB-13),Point(fL+22,fB-13),
               Point(fL+22,fB-11),Point(fL+23,fB-11),Point(fL+23,fB-9),
               Point(fL+24,fB-9),Point(fL+24,fB-8),Point(fL+25,fB-8),
               Point(fL+25,fB-6),Point(fL+26,fB-6),Point(fL+26,fB-5),
               Point(fL+27,fB-5),Point(fL+27,fB-3),Point(fL+28,fB-2),
               Point(fL+28,fB-1),Point(fL+28,fB-17),Point(fR-5,fB-17),
               Point(fR-5,fB-18),Point(fR-3,fB-18),Point(fR-3,fB-19),
               Point(fR-2,fB-19),Point(fR-2,fB-21),Point(fR-1,fB-21),
               Point(fR-1,5),Point(fR-2,5),Point(fR-2,3),Point(fR-3,3),
               Point(fR-3,2),Point(fR-4,2),Point(fR-4,1),Point(fR-6,1),
               Point(fR-6,0),Point(5,0)]);
             {$ENDIF}
           end;
        3: begin
             Polyline([Point(5,0),Point(5,1),Point(3,1),Point(3,2),
               Point(2,2),Point(2,3),Point(1,3),Point(1,5),
               Point(Left,5),Point(Left,fB-5),Point(1,fB-5),
               Point(1,fB-3),Point(2,fB-3),Point(2,fB-2),
               Point(4,fB-2),Point(4,fB-1),Point(fR-21,fB-1),
               Point(fR-21,fB-2),Point(fR-19,fB-2),Point(fR-19,fB-3),
               Point(fR-18,fB-3),Point(fR-18,fB-5),Point(fR-17,fB-5),
               Point(fR-17,35),Point(fR-16,35),Point(fR-16,34),
               Point(fR-14,34),Point(fR-14,33),Point(fR-12,33),
               Point(fR-12,32),Point(fR-11,32),Point(fR-11,31),
               Point(fR-9,31),Point(fR-9,30),Point(fR-8,30),
               Point(fR-8,29),Point(fR-6,29),Point(fR-6,28),
               Point(fR-4,28),Point(fR-4,27),Point(fR-3,27),
               Point(fR-2,26),Point(fR-1,26),Point(fR-17,26),
               Point(fR-17,5),Point(fR-18,5),Point(fR-18,3),
               Point(fR-19,3),Point(fR-19,2),Point(fR-20,2),
               Point(fR-20,1),Point(fR-22,1),Point(fR-22,0),
               Point(5,0)]);
             {$IFDEF WIN32}
             inc(fB);
             inc(fR);
             MakePoly([Point(5,0),Point(5,1),Point(3,1),Point(3,2),
               Point(2,2),Point(2,3),Point(1,3),Point(1,5),
               Point(Left,5),Point(Left,fB-5),Point(1,fB-5),
               Point(1,fB-3),Point(2,fB-3),Point(2,fB-2),
               Point(4,fB-2),Point(4,fB-1),Point(fR-21,fB-1),
               Point(fR-21,fB-2),Point(fR-19,fB-2),Point(fR-19,fB-3),
               Point(fR-18,fB-3),Point(fR-18,fB-5),Point(fR-17,fB-5),
               Point(fR-17,35),Point(fR-16,35),Point(fR-16,34),
               Point(fR-14,34),Point(fR-14,33),Point(fR-12,33),
               Point(fR-12,32),Point(fR-11,32),Point(fR-11,31),
               Point(fR-9,31),Point(fR-9,30),Point(fR-8,30),
               Point(fR-8,29),Point(fR-6,29),Point(fR-6,28),
               Point(fR-4,28),Point(fR-4,27),Point(fR-3,27),
               Point(fR-2,26),Point(fR-1,26),Point(fR-17,26),
               Point(fR-17,5),Point(fR-18,5),Point(fR-18,3),
               Point(fR-19,3),Point(fR-19,2),Point(fR-20,2),
               Point(fR-20,1),Point(fR-22,1),Point(fR-22,0),
               Point(5,0)]);
             {$ENDIF}
           end;
        4: begin
             Polyline([Point(21,0),Point(21,1),Point(19,1),Point(19,2),
               Point(18,2),Point(18,3),Point(17,3),Point(17,5),
               Point(16,5),Point(16,26),Point(0,26),Point(1,26),
               Point(2,27),Point(3,27),Point(3,28),Point(5,28),
               Point(5,29),Point(7,29),Point(7,30),Point(8,30),
               Point(8,31),Point(10,31),Point(10,32),Point(11,32),
               Point(11,33),Point(13,33),Point(13,34),Point(15,34),
               Point(15,35),Point(16,35),Point(16,fB-5),Point(17,fB-5),
               Point(17,fB-3),Point(18,fB-3),Point(18,fB-2),
               Point(20,fB-2),Point(20,fB-1),Point(fR-5,fB-1),
               Point(fR-5,fB-2),Point(fR-3,fB-2),Point(fR-3,fB-3),
               Point(fR-2,fB-3),Point(fR-2,fB-5),Point(fR-1,fB-5),
               Point(fR-1,5),Point(fR-2,5),Point(fR-2,3),Point(fR-3,3),
               Point(fR-3,2),Point(fR-4,2),Point(fR-4,1),Point(fR-6,1),
               Point(fR-6,0),Point(21,0)]);
             {$IFDEF WIN32}
             inc(fB);
             inc(fR);
             MakePoly([Point(21,0),Point(21,1),Point(19,1),Point(19,2),
               Point(18,2),Point(18,3),Point(17,3),Point(17,5),
               Point(16,5),Point(16,26),Point(0,26),Point(1,26),
               Point(2,27),Point(3,27),Point(3,28),Point(5,28),
               Point(5,29),Point(7,29),Point(7,30),Point(8,30),
               Point(8,31),Point(10,31),Point(10,32),Point(11,32),
               Point(11,33),Point(13,33),Point(13,34),Point(15,34),
               Point(15,35),Point(16,35),Point(16,fB-5),Point(17,fB-5),
               Point(17,fB-3),Point(18,fB-3),Point(18,fB-2),
               Point(20,fB-2),Point(20,fB-1),Point(fR-5,fB-1),
               Point(fR-5,fB-2),Point(fR-3,fB-2),Point(fR-3,fB-3),
               Point(fR-2,fB-3),Point(fR-2,fB-5),Point(fR-1,fB-5),
               Point(fR-1,5),Point(fR-2,5),Point(fR-2,3),Point(fR-3,3),
               Point(fR-3,2),Point(fR-4,2),Point(fR-4,1),Point(fR-6,1),
               Point(fR-6,0),Point(21,0)]);
             {$ENDIF}
           end;
      end;
    end;
end;

procedure TOffice97Balloon.PositionChilds(OldPos,NewPos: integer);
var
  ChildMove,Count: integer;
begin
  ChildMove := 0;
  if OldPos = 4 then ChildMove := 1; { Left }
  if NewPos = 4 then ChildMove := 2; { Right }
  if (ChildMove = 0) or (OldPos = NewPos) or (OldPos = 0) or
    (NewPos = 0) then Exit;
  for Count := 0 to ComponentCount - 1 do
    if (Components[Count] is TControl) then
  begin
    if ChildMove = 1 then
      TControl(Components[Count]).left := TControl(Components[Count]).left - 16
    else
      TControl(Components[Count]).left := TControl(Components[Count]).left + 16;
  end;
end;

function TOffice97Balloon.PositionControl:Integer;
var
  tmp: Integer;
begin
  Result := 0;
  tmp := 0;
  if fBalloonPosition <> bsAuto then
    tmp := ControlFits(fBalloonPosition);
  if tmp > 0 then Result := tmp else
  begin
    tmp := ControlFits(bsTop);
    if tmp > 0 then Result := tmp else
    begin
      tmp := ControlFits(bsLeft);
      if tmp > 0 then Result := tmp else
      begin
        tmp := ControlFits(bsRight);
        if tmp > 0 then Result := tmp
          else Result := 1;
      end;
    end;
  end;
  PositionChilds(fOldPos,Result);
  fOldPos := Result;
end;

procedure TOffice97Balloon.Zoom;
var
  tZh,tZi,tZw,tZx,tZy: Integer;
begin
  with Canvas do
  begin
    { Paint "Zoom" }
    Brush.Color := clBtnShadow;
    Brush.Style := bsClear;
    Pen.Color := clBtnShadow;
    Pen.Style := psSolid;
    for tZi := 1 to 2 do
    begin
      tZx := (Width div 4) - (tZi * 15);
      tZy := (Height div 4) - (tZi * 15);
      tZw := Width - tZx;
      tZh := Height - tZy;
      Rectangle(tZx,tZy,tZw,tZh);
      Delay(40);
    end;
    fZoom := 1;
  end;
end;

procedure TOffice97Balloon.Paint;
var
  Glyph: TBitmap;
  bitRect,txtRect: TRect;
  gH,gW,tH: Integer;
  TempCap: array[0..255] of char;
begin
  with Canvas do
  begin
    { Position control }
    if fStickTo <> nil then fFit := PositionControl
      else fFit := 1;
    { "Zoom" control if nessacary }
    if fZoom = 0 then Zoom;
    { Fill control background }
    Brush.Color := Color;
    Brush.Style := bsSolid;
    FillRect(ClientRect);
    { Draw control frame }
    DrawFrame;
    if fShowCaption then
    begin
      { Draw Text }
      if fCustomGlyph.Empty then
      begin
        gH := Icon.Height;
        gW := Icon.Width;
        bitRect := Rect(8, 8, gW + 8, gH + 8);
        if fOldPos = 4 then Draw(bitRect.Left + 16, bitRect.Top, Icon)
          else Draw(bitRect.Left, bitRect.Top, Icon);
      end
      else
      begin
        Glyph := fCustomGlyph;
        gH := Glyph.Height;
        gW := Glyph.Width;
        bitRect := Rect(8, 8, gW + 8, gH + 8);
        if fOldPos = 4 then
          DrawTransparentBitmap(Canvas,bitRect.Left + 16,bitRect.Top,Glyph,clOlive)
        else
          DrawTransparentBitmap(Canvas,bitRect.Left,bitRect.Top,Glyph,clOlive);
      end;
      tH := TextHeight(Caption);
      if fOldPos = 4 then
        txtRect := Rect(gW + 32, 8 + (gH div 2) - (tH div 2), Width - 8, 8 + (gH div 2) + tH)
      else
        txtRect := Rect(gW + 16, 8 + (gH div 2) - (tH div 2), Width - 24, 8 + (gH div 2) + tH);
      StrPCopy(TempCap, Caption);
      DrawText(Canvas.handle, TempCap, StrLen(TempCap), txtRect, 0);
    end;
  end;
end;

procedure TOffice97Balloon.WMLButtonDown(var Message: TWMLButtonDown);
begin
  { Hide the control on a left mouse click }
  Inherited;
  if fHideOnClick then Hide;
end;

procedure TOffice97Balloon.Button1Click(Sender: TObject);
begin
  office97balloon.stickto := form1;
end;

procedure TOffice97Balloon.Button2Click(Sender: TObject);
begin
  office97balloon.stickto := nil;
end;

procedure TOffice97Balloon.Button3Click(Sender: TObject);
begin
  application.terminate;
end;

end.
