#
# File
# 	genhlp.tcl
# Description
# 	Object Domain script for generating Windows help file
# Created
#	03/17/1997	Johan Nilsson, Autoliv Sverige AB
# Revisions

###############################################################################
# String procedures
###############################################################################

#
# strsubst
#	Substitutes expression <e2> for <e1> in a string <str>.
#	Returns string with substitutions made.

proc strsubst { str e1 e2 } {
	set left $str
	set res ""
	while -1 {
		set pos [string first $e1 $left]
		if {$pos == -1} {
        	return [strcat $res $left]
		} else {
			set res [strcat $res [string range $left 0 [expr $pos - 1]]]
			set res [strcat $res $e2]
			set left [string range $left [expr $pos + [string length $e1]] end]
        }
	}
}

#
# strsep
# 	Separates a string into rows of <width> characters.
#	Returns a list of strings.

proc strsep { str width } {
	set strlist [list]
	set remainder [strsubst $str "\r\n" " "]
	while {$remainder != ""} {

		if {[string length $remainder] < $width} {
			set cut [string length $remainder]
		} else {
			# search for next whitespace
			set cut [string last " " [string range $remainder 0 $width]]

			if {$cut == -1} {
				set cut [string first " " $remainder]
			}
			if {$cut == -1} {
				set cut [string length $remainder]
			}
		}

		# add row to list
		concat strlist [string range $remainder 0 $cut]

		# fortstt med resten av strngen
		set remainder [string range $remainder [expr $cut + 1] end]
	}
	return $strlist
}

#
# strcat
#	Concatenates two strings.

proc strcat { s1 s2 } {
	return [format "%s%s" $s1 $s2]
}

#
# streq
#	Compares two strings and returns true if they are equal

proc streq { s1 s2 } {
	return [expr {[string compare $s1 $s2] == 0}]
}

#
# strfill
#	Returns a string filled with characters.

proc strfill { c n } {
	set s ""
	for { set i 0 } {$i < $n} {set i [expr $i + 1]} {
    	set s [strcat $s $c]
	}
    return $s
}

###############################################################################
# Property procedures
###############################################################################

#
# prop_is_defined
#	Returns -1 if a property is defined for an object

proc prop_is_defined { obj name } {
	return [expr [string compare [$obj get property $name] "OD_no_value"] != 0]
}

#
# obj_list_find_property
#	Searches an object list for a property name and a value pattern.
#	Returns a list of values.

proc obj_list_find_property { objlist name pat } {
	set rlist [list]
	foreach obj $objlist {
		set rlist [concat $rlist [obj_find_property $obj $name $pat]]
	}
	return $rlist
}

#
# obj_find_property
#	Searches for a property name and a value pattern in an object's properties.
#	Returns a list of values.

proc obj_find_property { obj name pat } {

	# read property string
	set propstr [$obj get properties]
	if {$propstr == "OD_no_value"} {
		return ""
	}

	# find name-value-pair
	set sgn "\{ $name \{"
	set pos 0
	set rlist [list]
	set tmp [string first $sgn $propstr]
	while {$tmp != -1} {

		# go to beginning of property
		set pos [expr $pos + $tmp]

		# read value
		set pos [expr $pos + [string length $sgn]]
		set len [string first "\}" [string range $propstr $pos end]]

    	# check that value matches pattern
		set value [string range $propstr $pos [expr $pos + $len - 1]]
		if {[string match $pat $value] == 1} {
           	# add value to result list
			set rlist [linsert $rlist end $value]
		}

		# find next type property
		set tmp [string first $sgn [string range $propstr $pos end]]
	}

	# return value
	return $rlist
}

###############################################################################
# Class object procedures
###############################################################################

proc cl_name { cl isref } {
	if $isref {
		return [cl_pname $cl]
	} else {
    	return [cl_tname $cl]
	}
}

#
# cl_tname
#	Creates a class name with a leading T for a class

proc cl_tname { cl } {

	# add a T in front of class name if not present
	if {[string index [$cl get name] 0] == "T"} {
		return [$cl get name]
	}
    return "T[$cl get name]"
}

#
# cl_pname
#	Creates a pointer type name for a class

proc cl_pname { cl } {
	if {[string index [$cl get name] 0] == "P"} {
		return [$cl get name]
	}
	if {[string index [$cl get name] 0] == "T"} {
		return "P[string range [$cl get name] 1 end]"
	}
	return "P[$cl get name]"
}

#
# cl_topic_id
#	Creates a topic id for a class

proc cl_topic_id { cl } {
	return "[cl_tname $cl]_topic"
}

#
# cl_sccnt
#	Returns the number of super classes for a class

proc cl_sccnt { cl } {
	set sclist [$cl get superclasses]
	return [llength $sclist]
}

#
# cl_superclass
# 	Returns the parent of a class

proc cl_superclass { cl } {
	set sclist [$cl get superclasses]
	set scl [lindex $sclist 0]
	set sci [expr [llength $scl] - 1]
#	switch $sci {
#		1 { puts -nonewline $fp "[lindex $scl 0]" }
#		2 { puts -nonewline $fp "[lrange $scl 0 1]" }
#	}
	return [lindex $scl $sci]
}

#
# cl_is_superclass
# 	Returns -1 if cl1 is a super class to cl2, 0 if not

proc cl_is_superclass { cl1 cl2 } {
	if {$cl1 == $cl2} {
		return -1
	}
	if {[cl_sccnt $cl2] == 0} {
		return 0
	}
    return [cl_is_superclass $cl1 [cl_superclass $cl2]]
}

#
# cl_is_component
#	Determines if a class is a component.

proc cl_is_component { cl } {
	if {[prop_is_defined $cl component]} {
		return -1
	}
	if {[cl_sccnt $cl] == 0} {
		return 0
	}
	return [cl_is_component [cl_superclass $cl]]
}

#
# cl_from_name
#	Finds a class object with a class name

proc cl_from_name { name } {
	set cllist [OD_getClasses]
	foreach cl $cllist {
    	if {[$cl get name] == $name} {
            return $cl
		}
	}
	return ""
}

#
# cl_hierarchical_order
#	Returns 0 if cl1 is a superclass of cl2 or if they are not related
#	through inheritance, 1 otherwise.
#	Function is used for sorting lists of classes.

proc cl_hierarchical_order { cl1 cl2 } {
	if {![cl_is_superclass $cl1 $cl2]} {
		return 0
	} else {
		return 1
	}
}

proc cl_alpha_order { cl1 cl2 } {
	return [string compare [$cl1 get name] [$cl2 get name]]
}


###############################################################################
# Attribute object procedures
###############################################################################

#
# att_is_property
#	Determines if an attribute is declared as property

proc att_is_property { att } {
	return [prop_is_defined $att property]
}

#
# prop_is_event
#	Determines if a property is an event, by searching for "Event" in the
#	property type name.

proc prop_is_event { att } {
	return [expr [string first "Event" [$att get className]] != -1]
}

#
# att_type
#	Returns the attribute type: attribute, property or event

proc att_type { att } {

	if [att_is_property $att] {
		if [prop_is_event $att] {
			return event
		} else {
			return property
		}
	} else {
		return attribute
	}
}

#
# att_topic_id
#	Creates a unique topic id for an attribute, property or event

proc att_topic_id { cl att } {
	return "[cl_tname $cl]_[$att get name]_topic"
}

proc att_alpha_order { att1 att2 } {
	return [string compare [$att1 get name] [$att2 get name]]
}

###############################################################################
# 						Operation object procedures
###############################################################################


#
# op_is_override
# purpose
#	determines if an operation should be declared override
# returns
#	-1 (operation overrides base class) or 0 (else)

proc op_is_override { cl op } {

	# check if there is a superclass
	if {[cl_sccnt $cl] == 0} {
		return 0
	}

	# search for base class operation with same name
	set sc [cl_superclass $cl]
	foreach scop [$sc get operations] {
    	if {[$scop get name] == [$op get name]} {
			return 1
		}
	}

	# go one level up in hierarchy
	return [op_is_override $sc $op]
}

#
# method_topic_id
#	Creates a unique topic id for a method

proc method_topic_id { cl op } {
	return "[cl_tname $cl]_[$op get name]_topic"
}

###############################################################################
# 						Module object procedures
###############################################################################

#
# module_from_class
# 	finds body module where a class is defined

proc module_from_class { cl } {

	# loop through all diagrams
	set dgmlist [OD_getDiagrams]
	foreach dgm $dgmlist {
		if {[$dgm get type] != "module"} {
			continue
		}

		# loop through all objects in the diagram
		set objlist [$dgm get objects]
		foreach obj $objlist {
			if {[$obj get objectType] != "module"} {
				continue
			}

			# loop thorugh all classes in the module
			set dcllist [$obj get classes]
	        foreach dcl $dcllist {
				if {$dcl == $cl} {

					# return diagram
					return $obj
            	}
			}
		}
	}
	return ""
}

###############################################################################
# 						Topic generation procedures
###############################################################################

proc topic_id { t } {
	return "$t\_topic"
}

#
# dump_header
# 	generates a RTF file header for a module

proc dump_header { fp } {

	puts $fp "\{"
	puts $fp "\\rtf1\\ansi\\deff0\\deflang1024"

	puts $fp "\{\\fonttbl"
	puts $fp "\t\{\\f0\\froman Times New Roman;\}"
	puts $fp "\t\{\\f1\\froman Symbol;\}"
	puts $fp "\t\{\\f2\\fswiss Arial;\}"
	puts $fp "\t\{\\f3\\froman MS Serif;\}"
	puts $fp "\t\{\\f4\\fswiss MS Sans Serif;\}"
	puts $fp "\}"
}

#
# dump_att_link_list
#	generates attribute links for a class
# parameters
#	fp = file
#	cl = class
#	attlist = list of attributes
#	type = attribute type, can be attribute, property or event
#	hdr = header to write

proc dump_att_link_list { fp mod cl attlist type hdr } {
	set cnt 0
	foreach att $attlist {

		# check attribute type
		if {![streq [att_type $att] $type]} {
			continue
		}

		# check that export control is at least public
		if {![streq [$att get exportControl] public] ||
		    ![prop_is_defined $att published]} {
			continue
		}

		# static attributes are implemented as global variables
		if {[$att get isStatic] == 1} continue

		# determine if header should be written
		if {$cnt == 0} {
			puts $fp "\\par \\plain \\b $hdr"
		}
		incr cnt

		# write attribute link
		puts $fp "\\par \\plain \\tab \{\\uldb [$att get name]\}"
		puts $fp "\{\\v [att_topic_id $cl $att]\}"
	}
}

proc dump_att_topic_list { fp mod cl attlist type } {
	foreach att $attlist {

		# check attribute type
		if {![streq [att_type $att] $type]} {
			continue
		}

		# check that export control is at least public
		if {![streq [$att get exportControl] public] ||
		    ![prop_is_defined $att published]} {
			continue
		}

        # write attribute topic
		dump_att_topic $fp $mod $cl $att $type
	}
}

#
# dump_att_topic
#	generates an attribute topic
# parameters
#	fp = file
#	cl = class
#	att = attribute
#	type = can be "attribute", "property" or "event"

proc dump_att_topic { fp mod cl att type } {

	# write topic id
    puts $fp "#\{\\footnote [att_topic_id $cl $att]\}"

	# write topic title
	puts $fp "\$\{\\footnote [$cl get name].[$att get name] $type\}"

	# write keyword
    puts $fp "K\{\\footnote [$att get name]\}"

	# write heading
	puts $fp "\\par \\plain [$att get name] $type"

	# write class link
	puts $fp "\\par"
	puts $fp "\\par Applies to "
	puts $fp "\{\\uldb [cl_tname $cl]\}"
	puts $fp "\{\\v [cl_topic_id $cl]\}"

	# write declaration
	puts $fp "\\par"
	puts $fp "\\par \\plain \\b Declaration"
	puts $fp "\\par \\plain "

	# possibly write keyword property
    puts $fp "\\b property \\b0 "

	# write attribute name
  	puts $fp "[$att get name]"

	# write attribute type, possibly as link to a class
	puts $fp " : "
	set attclname [$att get className]
	set attcl [cl_from_name $attclname]
	set isref [expr {[expr [$att get isPointer] == 1] ||
	                 [expr [$att get isReference] == 1]}]
	set tdlist [obj_list_find_property [OD_getClasses] "type" "$attclname*=*"]
	if {$attcl != ""} {
		# write attribute type as link to a class topic
		puts -nonewline $fp "\{\\uldb "
		if {$isref} {
			puts -nonewline $fp "[cl_pname $attcl]"
		} else {
			puts -nonewline $fp "[cl_tname $attcl]"
        }
		puts -nonewline $fp "\} "
		puts $fp "\{\\v [cl_topic_id $attcl]\}"
	} else {
    if {[llength $tdlist] > 0} {

	    	# write attribute type as link to a type definition topic
			puts -nonewline $fp "\{\\uldb $attclname\}"
			puts $fp "\{\\v [topic_id $attclname]\}"
	} else {

		# write attribute type as plain text
		if $isref {
			puts $fp $attclname
		} else {
			puts $fp $attclname
		}
    }}

	# write declaration terminator
	puts $fp ";"

	# write description
	puts $fp "\\par"
	puts $fp "\\par \\plain \\b Description"
    puts $fp "\\par \\plain [$att get description]"

	# end attribute topic
    puts $fp "\\page"
}

#
# dump_method_syntax
#	Writes a method header with a link to the method's class

proc dump_method_syntax { fp mod cl op } {

	# begin declaration
	puts -nonewline $fp "\\par \\plain "

	# detect static operations
       	if {[$op get isStatic] == 1} {
		puts -nonewline $fp "\\b class \\b0 "
	}

	# determine if operation has return value
	set returnType [$op get returnType]
	if {[prop_is_defined $op constructor]} {
		puts $fp "\\b constructor \\b0 "
	} else {
	if {[prop_is_defined $op destructor]} {
		puts $fp "\\b destructor \\b0 "
	} else {
	if {$returnType != ""} {
		puts $fp "\\b function \\b0 "
	} else {
		puts $fp "\\b procedure \\b0 "
	}}}

	# write class prefix as a link
	puts $fp "\{\\uldb [cl_tname $cl]\}"
	puts $fp "\{\\v [cl_topic_id $cl]\}."

	# write operation name
	puts -nonewline $fp "[$op get name]"

	# create parameter string
	set plist [$op get parameters]
	set noParameters -1
	foreach p $plist {
		if {$noParameters} {
			# begin parameter list
			puts -nonewline $fp "( "
			set noParameters 0
		} else {
			# add comma
			puts -nonewline $fp "; "
		}
		puts -nonewline $fp "[$p get name]"
	}
	if {!$noParameters} {
  		puts -nonewline $fp " )"
	}

	# write return type, if any
	if {$returnType != ""} {
		puts $fp " : "
		set rtclass [cl_from_name $returnType]
		set isref [expr {[$op get isReturnTypePointer] ||
		                 [$op get isReturnTypeReference]}]
		if {$rtclass != ""} {
			dump_class_link $fp $mod $rtclass $isref
		} else {
			puts $fp "$returnType"
        }
	}

	# write terminating semicolon
	puts -nonewline $fp "; "

	# write method directives
	if {[op_is_override $cl $op]} {
		puts $fp "\\b override; \\b0"
	} else { if {[$op get isPure] == 1} {
		puts $fp "\\b virtual; abstract; \\b0"
	} else { if {[$op get isVirtual] == 1} {
		puts $fp "\\b virtual; \\b0"
	}}}
}

#
# dump_method_link_list
#	Writes links to methods in a class topic

proc dump_method_link_list { fp mod cl oplist } {
	set cnt 0
	foreach op $oplist {

		# check public export control
		if {[$op get exportControl] != "public"} {
			continue
		}

		# determine if header should be written
		if {$cnt == 0} {
			puts $fp "\\par \\plain \\b Methods"
		}
		incr cnt

		# write method link
		puts $fp "\\par \\plain \\tab \{\\uldb [$op get name]\}"
		puts $fp "\{\\v [method_topic_id $cl $op]\}"
	}
}

proc dump_method_topic_list { fp mod cl oplist } {
	foreach op $oplist {

		# check public export control
		if {[$op get exportControl] != "public"} {
        	continue
		}

        # write attribute topic
		dump_method_topic $fp $mod $cl $op
	}
}

#
# dump_method_topic
#	generates a template for a class method
# parameters
#		fp = file
#		cl = class
#		op = operation

proc dump_method_topic { fp mod cl op } {

	# write topic title
	puts $fp "\$\{\\footnote [$cl get name].[$op get name] method\}"

	# write topic id
	puts $fp "#\{\\footnote [method_topic_id $cl $op]\}"

	# write keyword
	puts $fp "K\{\\footnote [$op get name]\}"

	# write heading
	puts $fp "\\par \\plain [$op get name] method"

	# write Delphi declaration
	puts $fp "\\par"
	puts $fp "\\par \\plain \\b Declaration"
	dump_method_syntax $fp $mod $cl $op

	# write description
	puts $fp "\\par"
	puts $fp "\\par \\plain \\b Description"
	puts $fp "\\par \\plain [strsubst [$op get description] "\n" " \\par "]"

	# write topic end
	puts $fp "\\page"
}

proc dump_type_link_list { fp mod } {

	# find module type definitions
	set tdlist [obj_find_property $mod "type" "*"]

	# find class type definitions
	foreach cl [$mod get classes] {
		set tdlist [concat $tdlist [obj_find_property $cl "type" "*"]]
	}
	if {[llength $tdlist] == 0} {
		return
	}

	# sort the list alphabetically
	set tdlist [lsort $tdlist]

	# write type links
    puts $fp "\\par \\plain \\b Types"
	for {set i 0} {$i < [llength $tdlist]} {set i [expr $i + 1]} {
		# write link
		set td [string trim [lindex $tdlist $i] "\{\} "]
		set tname [string range $td 0 [expr [string first " " $td] - 1]]
		puts $fp "\\par \\plain \\tab"
		puts $fp "\{\\uldb $tname\}"
		puts $fp "\{\\v [topic_id $tname]\}"
	}
}

proc dump_type_topic_list { fp mod } {

	# find module type definitions
	set tdlist [obj_find_property $mod "type" "*"]

	# find class type definitions
	foreach cl [$mod get classes] {
		set tdlist [concat $tdlist [obj_find_property $cl "type" "*"]]
	}
	if {[llength $tdlist] == 0} {
		return
	}

	# write topics for all definitions
	for {set i 0} {$i < [llength $tdlist]} {set i [expr $i + 1]} {
		set td [string trim [lindex $tdlist $i] "\{\} "]
		set tname [string range $td 0 [expr [string first " " $td] - 1]]

		# write topic id
		puts $fp "#\{\\footnote [topic_id $tname]\}"

		# write topic title
		puts $fp "\$\{\\footnote $tname type\}"

		# write keyword
		puts $fp "K\{\\footnote $tname type\}"

		# write header
	    puts $fp "\\par \\plain \\b $tname type"

		# write declaration
		puts $fp "\\par"
		puts $fp "\\par \\plain \\b Declaration"
		puts $fp "\\par \\plain $td;"

		# write topic end
		puts $fp "\\page"
	}
}


#
# dump_relation_list
#	Writes relations from the class as links.

proc dump_relation_list { fp mod cl relType text cnt } {
	set cvlist [$cl get views]
  	foreach cv $cvlist {
		# check all the relations from the class
	 	set rellist [$cv get connections]
	 	foreach rel $rellist {
			if {[$rel get objectType] != "relation"} {
				continue
			}
			if {[$rel get relationType] != $relType} {
				continue
			}
			set frcl [$rel get from]
			if {[$frcl get class] != $cl} {
				continue
			}

			# write heading
			if {$cnt == 0} {
			    puts $fp "\\par \\plain \\b Relations"
			}
			incr cnt

			# write relation with a link to the "to" class
			set tocl [$rel get to]
		    puts -nonewline $fp "\\par \\plain \\tab $text "
			dump_class_link $fp $mod [$tocl get class] 0
		}
	}
	return $cnt
}

#
# dump_class_link
#	Writes a link to a class which can be outside the current module

proc dump_class_link { fp mod cl isref } {

	# determine if class is defined in this module
	if {[expr [lsearch -exact [$mod get classes] $cl] != -1]} {
		# same module: generate local jump
		puts $fp "\{\\uldb [cl_name $cl $isref]\}"
		puts $fp "\{\\v [cl_topic_id $cl]\}"
	} else {
		# other module: generate far jump
		set clmod [module_from_class $cl]
		if [streq $clmod ""] {
        	puts $fp [cl_name $cl $isref]
		} else {
			set hlpname "[$clmod get name].hlp"
			puts $fp "\{\\uldb [cl_name $cl $isref]\}"
			puts $fp "\{\\v [cl_topic_id $cl] @$hlpname\}"
		}
	}
}

#
# dump_class_topic_list
#	writes a list of class topics

proc dump_class_topic_list { fp mod classlist } {

	# loop through all classes in the module
	foreach cl $classlist {

		# write topic for class
		dump_class_topic $fp $mod $cl
		set oplist [$cl get operations]
		set attlist [$cl get attributes]

		# write topics for attributes, properties, methods and events
		dump_att_topic_list $fp $mod $cl $attlist attribute
		dump_att_topic_list $fp $mod $cl $attlist property
		dump_method_topic_list $fp $mod $cl $oplist
		dump_att_topic_list $fp $mod $cl $attlist event
	}
}

#
# dump_class_topic
#	writes a class topic

proc dump_class_topic { fp mod cl } {

	# set type to "class" or "component"
	if [cl_is_component $cl] {
		set type "component"
	} else {
		set type "class"
	}

	# write topic title
	puts $fp "\$\{\\footnote [cl_tname $cl] $type\}"

	# write topic id
	puts $fp "#\{\\footnote [cl_topic_id $cl]\}"

	# write keyword
	puts $fp "K\{\\footnote [cl_tname $cl]\}"

	# write heading
	puts $fp "\\par \\plain \\b [cl_tname $cl] $type"

	# write unit as link
	puts $fp "\\par \\plain \\b Unit"
	puts $fp "\\par \\plain \\tab"
	puts $fp "\{\\uldb [$mod get name]\}"
	puts $fp "\{\\v [topic_id [$mod get name]]\}"

	# write responsibilities
    puts $fp "\\par \\plain \\b Responsibilities"
	set rowlist [strsep [$cl get responsibilities] 72 ]
	foreach row $rowlist {
		puts $fp "\\par \\plain \\tab $row"
	}

	# write relations
	set cnt 0
	set cnt [dump_relation_list $fp $mod $cl "inherits" "inherits" $cnt]
	set cnt [dump_relation_list $fp $mod $cl "has" "has" $cnt]
	set cnt [dump_relation_list $fp $mod $cl "use" "uses" $cnt]

	# find all attributes and operations
	set oplist [lsort -command cl_alpha_order [$cl get operations]]
	set attlist [lsort -command att_alpha_order [$cl get attributes]]

	# dump attributes
	dump_att_link_list $fp $mod $cl $attlist attribute Attributes

	# dump static attributes (implemented as variables)

	# dump methods
	dump_method_link_list $fp $mod $cl $oplist

	# dump properties
	dump_att_link_list $fp $mod $cl $attlist property Properties

	# dump events
	dump_att_link_list $fp $mod $cl $attlist event Events

	# end topic
	puts $fp "\\page"
}


#
# dump_module_topic
#	Writes a module topic with links to defined classes.

proc dump_module_topic { fp mod } {

	# write topic id
	puts $fp "#\{\\footnote [topic_id [$mod get name]]\}"

	# write topic title
	puts $fp "\$\{\\footnote [$mod get name] unit\}"

	# write heading
	puts $fp "\\par \\plain \\b [$mod get name] unit"

	# write constant definitions

	# write type definitions
	puts $fp "\\par"
	dump_type_link_list $fp $mod

	# write class links
	puts $fp "\\par"
    puts $fp "\\par \\b Classes"
	set cllist [lsort -command cl_hierarchical_order [$mod get classes]]
	foreach cl $cllist {
		puts $fp "\\par \\plain \\tab \{\\uldb [cl_tname $cl]\}"
		puts $fp "\{\\v [cl_topic_id $cl]\}"
	}

	# write topic end
	puts $fp "\\page"
}

#
# dump_end
#	generates end of RTF file

proc dump_end { fp } {

	# write closing bracket
	puts $fp "\}"
}

#
# generate_rtf
# 	Generates a RTF file for a module.

proc generate_rtf { workdir mod } {

	# generate file name
	set modulename [string trim [$mod get name]]
	if {[string match [string toupper $modulename] "*.*"] == 1} {
		error "Module name must not have extension: [$modulename]"
	}
	set filename [format "%s\\%s.rtf" $workdir $modulename]

	# create file
	set fp [open $filename w]

	# write header
	dump_header $fp

	# write module topic
	dump_module_topic $fp $mod

	# write type definition topics
	dump_type_topic_list $fp $mod

	# write class topics
	dump_class_topic_list $fp $mod [$mod get classes]

    # write end of RTF file
	dump_end $fp

	# close output file
  	close $fp
}

#
# generate_hjp
# 	Generates a help project file for a module.

proc generate_hjp { workdir mod } {

	# generate file name
	set modulename [string trim [$mod get name]]
	if {[string match [string toupper $modulename] "*.*"] == 1} {
		error "Module name must not have extension: [$modulename]"
	}
	set filename [format "%s\\%s.hpj" $workdir $modulename]

	# create file
	set fp [open $filename w]

    # write options section
	puts $fp "\[OPTIONS\]"
	puts $fp "ROOT=$workdir"
	puts $fp "HLP=$workdir\\$modulename.HLP"
	puts $fp "TITLE=$modulename"
    puts $fp "CONTENTS=$modulename\_Contents"
	puts $fp "BUILD=WINDOWS"
	puts $fp "WARNING=3"
	puts $fp "COMPRESS=FALSE"
	puts $fp "OLDKEYPHRASE=FALSE"
	puts $fp "OPTCDROM=FALSE"
	puts $fp "REPORT=FALSE"
	puts $fp "ERRORLOG=$workdir\\$modulename.ERR"
	puts $fp ""

    # write buildtags section
	puts $fp "\[BUILDTAGS\]"
	puts $fp "WINDOWS"
	puts $fp ""

	# write config section
	puts $fp "\[CONFIG\]"
	puts $fp "BrowseButtons()"
	puts $fp ""

	# write files section
	puts $fp "\[FILES\]"
	puts $fp "$workdir\\$modulename.RTF"
	puts $fp ""

	# close output file
  	close $fp
}

#
# compile_hlp
# 	Runs Microsoft Help Compiler to create a Windows help file from
#	a project file and a RTF file.

proc compile_hlp { workdir mod } {

	# determine help project file name
	set modulename [string trim [$mod get name]]
	if {[string match [string toupper $modulename] "*.*"] == 1} {
		error "Module name must not have extension: [$modulename]"
	}
	set projname [format "%s\\%s.hpj" $workdir $modulename]

	# run help compiler
	exec "hcw " "/c" "/m" "/n" "/e" $projname
}



#######################################################################################
# main
#######################################################################################

	# check diagram type
	set workdir [OD_getInput "Output directory" "" [strsubst [pwd] "/" "\\"]]
    if {$workdir == ""} {
    	return
	}
	set diagram [OD_getActiveDiagram]
	set dt [$diagram get type]

	# loop over all selected items in the diagram
	set itemlist [$diagram get objects]
	foreach item $itemlist {

		# ignore unselected items
		if {[$item get isSelected] != 1} {
			continue
        }

		# determine which module to generate
		if {[$item get objectType] == "module"} {
			set mod $item
        } else {
		if {[$item get objectType] == "class"} {
			set mod [module_from_class [$item get class]]
            if {$mod == ""} {
                continue
			}
        } else {
			continue
		}}

		# check module type
		set mtype [$mod get moduleType]
		if {$mtype == "specification"} {
			continue
		}

		# check contents of module
		if {[llength [$mod get classes]] == 0} {
			continue
		}

        # generate RTF file
		generate_rtf $workdir $mod
		generate_hjp $workdir $mod
		compile_hlp $workdir $mod
    }

