#
# File
#	gendelph.tcl
# Description
# 	Object Domain script for generating Delphi code skeletons.
# Created
#	11/20/1996	Johan Nilsson, Autoliv Sverige AB
# Revisions
#	01/17/1997	JN	Support for generating visual components

# global variables
set VERSION 	"GenDelph 1.00"

###############################################################################
# 							String procedures
###############################################################################

#
# strsubst
#	Substitutes expression <e2> for <e1> in a string <str>.
#	Returns string with substitutions made.

proc strsubst { str e1 e2 } {
	set left $str
	set res ""
	while -1 {
		set pos [string first $e1 $left]
		if {$pos == -1} {
        	return [strcat $res $left]
		} else {
			set res [strcat $res [string range $left 0 [expr $pos - 1]]]
			set res [strcat $res $e2]
			set left [string range $left [expr $pos + [string length $e1]] end]
        }
	}
}

#
# strsep
# 	Separates a string into rows of <width> characters.
#	Returns a list of strings.

proc strsep { str width } {
	set strlist [list]
	set remainder [strsubst $str "\r\n" " "]
	while {$remainder != ""} {

		if {[string length $remainder] < $width} {
			set cut [string length $remainder]
		} else {
			# search for next whitespace
			set cut [string last " " [string range $remainder 0 $width]]

			if {$cut == -1} {
				set cut [string first " " $remainder]
			}
			if {$cut == -1} {
				set cut [string length $remainder]
			}
		}

		# add row to list
		concat strlist [string range $remainder 0 $cut]

		# fortstt med resten av strngen
		set remainder [string range $remainder [expr $cut + 1] end]
	}
	return $strlist
}

#
# strcat
#	Concatenates two strings.

proc strcat { s1 s2 } {
	return [format "%s%s" $s1 $s2]
}

#
# streq
#	Compares two strings and returns true if they are equal

proc streq { s1 s2 } {
	return [expr {[string compare $s1 $s2] == 0}]
}

#
# strfill
#	Returns a string filled with characters.

proc strfill { c n } {
	set s ""
	for {set i 0} {$i < $n} {set i [expr $i + 1]} {
    	set s [strcat $s $c]
	}
    return $s
}

###############################################################################
# 							Property procedures
###############################################################################

#
# prop_is_defined
#	Returns -1 if a property is defined for an object

proc prop_is_defined { obj name } {
	return [expr [string compare [$obj get property $name] "OD_no_value"] != 0]
}

#
# obj_list_find_property
#	Searches an object list for a property name and a value pattern.
#	Returns a list of values.

proc obj_list_find_property { objlist name pat } {
	set rlist [list]
	foreach obj $objlist {
		set rlist [concat $rlist [obj_find_property $obj $name $pat]]
	}
	return $rlist
}

#
# obj_find_property
#	Searches for a property name and a value pattern in an object's properties.
#	Returns a list of values.

proc obj_find_property { obj name pat } {

	# read property string
	set propstr [$obj get properties]
	if {$propstr == "OD_no_value"} {
		return ""
	}

	# find name-value-pair
	set sgn "\{ $name \{"
	set pos 0
	set rlist [list]
	set tmp [string first $sgn $propstr]
	while {$tmp != -1} {

		# go to beginning of property
		set pos [expr $pos + $tmp]

		# read value
		set pos [expr $pos + [string length $sgn]]
		set len [string first "\}" [string range $propstr $pos end]]

    	# check that value matches pattern
		set value [string range $propstr $pos [expr $pos + $len - 1]]
		if {[string match $pat $value] == 1} {
           	# add value to result list
			set rlist [linsert $rlist end $value]
		}

		# find next type property
		set tmp [string first $sgn [string range $propstr $pos end]]
	}

	# return value
	return $rlist
}


################################################################################
# 							Class object procedures
################################################################################

#
# tname
#	Inserts a T in front of a type name.

proc tname { n } {
	# does name begin with a T and a capital?
	if {[expr {[string index $n 0] == "T"}] &&
	    [expr {[string toupper [string index $n 1]] == [string index $n 1]}]} {
		# yes, return name as it is
		return $n
	} else {
		# no, add a T
		return [strcat "T" $n]
	}
}

#
# tname
#	Inserts a P in front of a type name.

proc pname { n } {
	# replace T with a P
	return [strcat "P" [string range [tname $n] 1 end]]
}

#
# cl_tname
#	Creates a class name with a leading T for a class

proc cl_tname { cl } {
    return [tname [$cl get name]]
}

#
# cl_pname
#	Creates a pointer type name for a class

proc cl_pname { cl } {
	return [pname [$cl get name]]
}

#
# cl_sccnt
#	Returns the number of super classes for a class

proc cl_sccnt { cl } {
	set sclist [$cl get superclasses]
	return [llength $sclist]
}

#
# cl_superclass
# 	Returns the parent of a class

proc cl_superclass { cl } {
	set sclist [$cl get superclasses]
	set scl [lindex $sclist 0]
	set sci [expr [llength $scl] - 1]
#	switch $sci {
#		1 { puts -nonewline $fp "[lindex $scl 0]" }
#		2 { puts -nonewline $fp "[lrange $scl 0 1]" }
#	}
	return [lindex $scl $sci]
}

#
# cl_is_superclass
# 	Returns -1 if cl1 is a super class to cl2, 0 if not

proc cl_is_superclass { cl1 cl2 } {
	if {$cl1 == $cl2} {
		return -1
	}
	if {[cl_sccnt $cl2] == 0} {
		return 0
	}
    return [cl_is_superclass $cl1 [cl_superclass $cl2]]
}

#
# cl_is_component
#	Determines if a class is a component.

proc cl_is_component { cl } {
	if {[prop_is_defined $cl component]} {
		return -1
	}
	if {[cl_sccnt $cl] == 0} {
		return 0
	}
	return [cl_is_component [cl_superclass $cl]]
}

#
# cl_from_name
#	Finds a class object with a class name

proc cl_from_name { name } {
	set cllist [OD_getClasses]
	foreach cl $cllist {
    	if {[$cl get name] == $name} {
            return $cl
		}
	}
	return ""
}

#
# cl_hierarchical_order
#	Returns 0 if cl1 is a superclass of cl2 or if they are not related
#	through inheritance, 1 otherwise.
#	Function is used for sorting lists of classes.

proc cl_hierarchical_order { cl1 cl2 } {
	if {![cl_is_superclass $cl1 $cl2]} {
		return 0
	} else {
		return 1
	}
}

proc cl_alpha_order { cl1 cl2 } {
	return [string compare [$cl1 get name] [$cl2 get name]]
}

################################################################################
# 						Attribute object procedures
################################################################################

#
# att_type_tname
#	Returns the attribute type. If the type is a class in the model,
#	a preceding T is added.

proc att_type_tname { att } {
	set type [$att get className]
	set cl [cl_from_name $type]
	if {$cl != ""} {
		return [cl_tname $cl]
	}
	return $type
}

#
# att_type_tname
#	Returns the attribute type. If the type is a class in the model,
#	a preceding P is added.

proc att_type_pname { att } {
	set type [$att get className]
	set cl [cl_from_name $type]
	if {$cl != ""} {
		return [cl_pname $cl]
	}
	return $type
}

#
# att_is_property
#	Determines if an attribute is declared as property

proc att_is_property { att } {
	return [prop_is_defined $att property]
}

#
# prop_is_event
#	Determines if a property is an event, by searching for "Event" in the
#	property type name.

proc prop_is_event { att } {
	return [expr [string first "Event" [$att get className]] != -1]
}

#
# prop_is_override

proc prop_is_override { cl prop } {

	# check if there is a superclass
	if {[cl_sccnt $cl] == 0} {
		return 0
	}

	# search for base class attribute with same name
	set sc [cl_superclass $cl]
	foreach att [$sc get attributes] {
    	if {[$att get name] == [$prop get name]} {
			return 1
		}
	}

	# go one level up in hierarchy
	return [prop_is_override $sc $prop]
}

#
# att_type
#	Returns the type of attribute: attribute, property or event

proc att_type { att } {

	if [att_is_property $att] {
		if [prop_is_event $att] {
			return event
		} else {
			return property
		}
	} else {
		return attribute
	}
}

#
# att_alpha_order
#	Returns the alphabetic order between two attributes.
#	Used to sort attribute lists.

proc att_alpha_order { att1 att2 } {
	return [string compare [$att1 get name] [$att2 get name]]
}

################################################################################
# 						Operation object procedures
################################################################################

#
# op_is_override
# purpose
#	determines if an operation should be declared override
# returns
#	-1 (operation overrides base class) or 0 (else)

proc op_is_override { cl op } {

	# check if there is a superclass
	if {[cl_sccnt $cl] == 0} {
		return 0
	}

	# search for base class operation with same name
	set sc [cl_superclass $cl]
	foreach scop [$sc get operations] {
    	if {[$scop get name] == [$op get name]} {
			return 1
		}
	}

	# go one level up in hierarchy
	return [op_is_override $sc $op]
}

################################################################################
# 						Module object procedures
################################################################################

proc module_has_component { mod } {

	set cllist [$mod get classes]
	foreach cl $cllist {
		if {[cl_is_component $cl]} {
			return -1
		}
	}
	return 0
}

#
# module_from_class
# 	finds body module where a class is defined

proc module_from_class { cl } {

	# loop through all diagrams
	set dgmlist [OD_getDiagrams]
	foreach dgm $dgmlist {
		if {[$dgm get type] != "module"} {
			continue
		}

		# loop through all objects in the diagram
		set objlist [$dgm get objects]
		foreach obj $objlist {
			if {[$obj get objectType] != "module"} {
				continue
			}

			# loop thorugh all classes in the module
			set dcllist [$obj get classes]
	        foreach dcl $dcllist {
				if {$dcl == $cl} {

					# return diagram
					return $obj
            	}
			}
		}
	}
	return ""
}

################################################################################
# 					Interface definition procedures
################################################################################

#
# dump_header
# 	generates a file header for a module

proc dump_header { fp mod } {
	global VERSION

	# write file header
	puts $fp "//"
	puts $fp "// File"
	puts $fp "//\t[$mod get name]\.pas"
	puts $fp "// Created"
	puts $fp "//\t"
	puts $fp "// Revisions"
	puts $fp "//\t"
    puts $fp ""
	puts $fp "unit [$mod get name]\;\n"
	flush $fp
}

#
# dump_dependencies
#	generates the USES directives for the interface part of a unit

proc dump_dependencies { fp mod } {

	# determine if module contains components
	set has_component [module_has_component $mod]

	# import "Classes" unit if there is a component
	set independent -1
	if $has_component {
		set independent 0
		puts -nonewline $fp "uses\n\tClasses"
	}

	# get all connections
	set deplist [$mod get connections]
	foreach dep $deplist {

		# check object type
		if {[$dep get objectType] != "dependency"} continue

		# check if this is the client
		set supplier [$dep get to]
		if {$supplier == $mod} continue

		# write
		if {$independent} {
			puts $fp "uses"
			set independent 0
		} else {
			puts $fp "\,"
		}
		puts -nonewline $fp "\t[$supplier get name]"
	}
	if {!$independent} {
		puts $fp "\;\n"
	}
	flush $fp
}

#
# dump_property_specifier
#	writes directives for a component property

proc dump_property_specifier { fp obj } {

	# write "read" directive
	set read_prop [$obj get property read]
	if {[$obj get property read] != "OD_no_value"} {
		puts -nonewline $fp "\n\t\tread "
		puts -nonewline $fp [string trim $read_prop "\{\} "]
	}

	# write "write" directive
	set write_prop [$obj get property write]
	if {[$obj get property write] != "OD_no_value"} {
		puts -nonewline $fp "\n\t\twrite "
		puts -nonewline $fp [string trim $write_prop "\{\} "]
	}

	# write "default" directive
	if {[$obj get property default] != "OD_no_value"} {
		puts -nonewline $fp "\n\t\tdefault "
		puts -nonewline $fp [$obj get defaultValue]
	}

	# write "stored" directive
	set prop_stored [$obj get property stored]
	if {[$obj get property stored] != "OD_no_value"} {
		puts -nonewline $fp "\n\t\tstored "
		puts -nonewline $fp [string trim $prop_stored "\{\} "]
	}
}

#
# dump_attribute_list
#	generates attribute definitions for a class and export control
# parameters
#	fp = file
#	cl = class
#	attlist = list of attributes
#	ectrl = public, protected, public or published
#   cnt = number of prior attributes, operations or has relations
#	under the export control
#	dir = export control directive to use
# returns
#	updated count

proc dump_attribute_list { fp cl attlist ectrl cnt dir } {
	foreach att $attlist {

		# check export control
		if {[$att get property published] == "OD_no_value"} {
			if {[$att get exportControl] != $ectrl} {
				continue
			}
		} else {
	       		if {$ectrl != "published"} {
				continue
			}
		}

		# static attributes are implemented as global variables
		if {[$att get isStatic] == 1} continue

		# determine if export control directive should be written
		if {$cnt == 0} {
			puts $fp $dir
		}
		incr cnt

		# write attribute
        dump_attribute $fp $cl $att
	}
    return $cnt
}

#
# dump_attribute
#	generates code for attributes in a class declaration
# parameters
#	fp = file
#	cl = class
#	att = attribute

proc dump_attribute { fp cl att } {

	# detect constant attributes
  	if {[$att get isConst] == 1} {
		error "Constant attributes not supported: [$att get name]"
	}

	# indent
	puts -nonewline $fp "\t"

	# write property directive
    	set is_prop 0
	set is_override 0
	if {[$att get property property] != "OD_no_value"} {
    	set is_prop -1
    	puts -nonewline $fp "property "
		set is_override [prop_is_override $cl $att]
	}

	# write attribute name
  	puts -nonewline $fp "[$att get name]"

	# write attribute type
	if {!$is_override} {
		puts -nonewline $fp " : "
        if {[expr [$att get isPointer] == 1] || [expr [$att get isReference] == 1]} {
			puts -nonewline $fp [att_type_pname $att]
	    } else {
			puts -nonewline $fp [att_type_tname $att]
		}
	}

	# write property definition
	if {$is_prop} {
		dump_property_specifier $fp $att
	}

	# end attribute definition
    puts $fp ";"
}

#
# op_is_override
# purpose
#	determines if an operation should be declared override
# returns
#	-1 (operation overrides base class) or 0 (else)

proc op_is_override { cl op } {

	# check if there is a superclass
	if {[cl_sccnt $cl] == 0} {
		return 0
	}

	# search for base class operation with same name
	set sc [cl_superclass $cl]
	foreach scop [$sc get operations] {
    	if {[$scop get name] == [$op get name]} {
			return 1
		}
	}

	# go one level up in hierarchy
	return [op_is_override $sc $op]
}

#
# create_op_header
#	creates an operation header
# argument
#	op = operation
#	cl = class
#   pFlag = -1 if header should be prefixed, 0 if not
# returns
#	header

proc create_op_header { op cl pFlag } {

	# create operation name
	set opname [$op get name]

	# determine if operation has return value
	set returnType [$op get returnType]

	# create parameter string
	set args ""
	set plist [$op get parameters]
	set noParameters -1
	foreach p $plist {
		if {$noParameters} {
			set args "( "
			set noParameters 0
		} else {
			# add comma
			set args [strcat $args "; "]
		}
		set args [strcat $args "[$p get name]"]
	}
	if {!$noParameters} {
  		set args [strcat $args " )"]
	}

	# create class prefix
	if {$pFlag} {
		set prefix "[cl_tname $cl]."
	} else {
		set prefix ""
	}

	# return header
	if {[$op get property constructor] != "OD_no_value"} {
		return [format "constructor %s%s%s" $prefix $opname $args]
	}
	if {[$op get property destructor] != "OD_no_value"} {
		return [format "destructor %s%s%s" $prefix $opname $args]
	}
	if {$returnType != ""} {
		# function
		return [format "function %s%s%s : %s" $prefix $opname $args $returnType]
	}
	return [format "procedure %s%s%s" $prefix $opname $args]
}

#
# dump_op_hdr_list
#	writes operation headers for a class and for an export control mode
# parameters
#	fp = file
#	cl = class
#	oplist = operation list
#	ectrl = public, protected or private
#	cnt = number of generated operation headers
#	dir = export directive
# returns
#	updated number of generated headers

proc dump_op_hdr_list {fp cl oplist ectrl cnt dir} {

	foreach op $oplist {

		# check export control
		if {[$op get property published] != ""} {
			if {[$op get exportControl] != $ectrl} {
				continue
			}
		} else {
	       		if {$ectrl != "published"} {
				continue
			}
		}

        # write export directive the first time
		if {$cnt == 0} {
			puts $fp $dir
		}
		incr cnt

		# write operation header
		dump_op_hdr $fp $cl $op
	}
	return $cnt
}

#
# dump_op_hdr
#	writes an operation header in a class declaration

proc dump_op_hdr { fp cl op } {

	# detect unsupported features
	if {[$op get isConst] == 1}  {
		error "Constant operations not supported: [$op get name]"
	}
	if {[$op get isReturnTypeConst] == 1} {
		error "Constant return values not supported: [$op get name]"
	}
	if { [$op get isInline] == 1 } {
		error "Inline functions not supported: [$op get name]"
	}

	# indent
	puts -nonewline $fp "\t"

	# detect static operations
       	if {[$op get isStatic] == 1} {
		puts -nonewline $fp "class "
	}

	# write header without class prefix
	set header [create_op_header $op $cl 0]
	puts -nonewline $fp "$header;"

	# write method directives
	if {[op_is_override $cl $op]} {
		puts -nonewline $fp " override;"
	} else { if {[$op get isPure] == 1} {
		puts -nonewline $fp " virtual; abstract;"
	} else { if {[$op get isVirtual] == 1} {
		puts -nonewline $fp " virtual;"
	}}}

	# write end of line
	puts $fp ""
}

#
# dump_relations
#	writes all relations from the class

proc dump_relations { fp cl relType text cnt } {
	set cvlist [$cl get views];
  	foreach cv $cvlist {

		# check all the relations from the class
	 	set rellist [$cv get connections]
	 	foreach rel $rellist {
			if {[$rel get objectType] != "relation"} {
				continue
			}
			if {[$rel get relationType] != $relType} {
				continue
			}
			set frcl [$rel get from]
			if {[$frcl get class] != $cl} {
				continue
			}
			if {$cnt == 0} {
				puts $fp "relations"
			}
			incr cnt
			set tocl [$rel get to]
			puts $fp "\t$text [cl_tname $tocl]"
		}
	}
	return $cnt
}

#
# dump_class_definition
#	generates a class declaration

proc dump_class_definition { fp cl } {

	# begin class header
	puts $fp "\{[strfill "*" 79]"
	puts $fp "Class: [cl_tname $cl]"

	# write responsibilities
	puts $fp "responsibilities"
	set rowlist [strsep [$cl get responsibilities] 72 ]
	foreach row $rowlist {
		puts $fp "\t$row"
	}

	# write relations
	set cnt 0
	set cnt [dump_relations $fp $cl "inherits" "inherits" $cnt]
	set cnt [dump_relations $fp $cl "has" "has" $cnt]
	set cnt [dump_relations $fp $cl "use" "uses" $cnt]

	# end class header
	puts $fp "[strfill "*" 79]\}"
	puts $fp ""

	# write type declarations associated to class
	set tdlist [obj_find_property $cl "type" "*"]
	foreach td $tdlist {
		puts $fp $td
	}
	if [llength $tdlist] {
		puts $fp ""
	}

	# write class statement
	puts -nonewline $fp "[cl_tname $cl] = class"

	# find superclasses
	set sclist [$cl get superclasses]
	set sccnt [llength $sclist]

	# check multiple inheritance
	if {$sccnt > 1} {
		error "Multiple inheritance not supported: [cl_tname $cl]"
	}

	# write superclass
	if {$sccnt == 0} {
		# determine if the class should be declared as a component
		if {[cl_is_component $cl]} {
			puts $fp "( TComponent )"
		} else {
			# no superclass
			puts $fp ""
		}
	} else {
		# one superclass
		set sc [cl_superclass $cl]
		puts $fp "( [cl_tname $sc] )"
	}

	# find all attributes and operations
	set oplist [$cl get operations]
	set attlist [$cl get attributes]

	# dump private operations
	set cnt 0
	set dir "private"
	set cnt [dump_op_hdr_list $fp $cl $oplist private $cnt $dir]

	# dump private attributes
	set cnt 0
	set cnt [dump_attribute_list $fp $cl $attlist private $cnt $dir]

    # dump protected operations
	set cnt 0
	set dir "protected"
	set cnt [dump_op_hdr_list $fp $cl $oplist protected $cnt $dir]

    # dump protected attributes
	set cnt 0
    set cnt [dump_attribute_list $fp $cl $attlist protected $cnt $dir]

    # dump public operations
	set cnt 0
	set dir "public"
	set cnt [dump_op_hdr_list $fp $cl $oplist public $cnt $dir]

	# dump public attributes
	set cnt 0
	set cnt [dump_attribute_list $fp $cl $attlist public $cnt $dir]

	# dump published properties
	set cnt 0
	set cnt [dump_attribute_list $fp $cl $attlist published $cnt $dir]

	# finish header
	puts $fp "end;"
	puts $fp ""
}


#
# dump_interface
#	Generates the interface specification for a module.

proc dump_interface { fp mod } {

	# write interface statement
	puts $fp "interface"

	# write uses statement and dependencies
	dump_dependencies $fp $mod

	# write type statement
	puts $fp "type"
	puts $fp ""

	# generate forward declarations and pointer declarations for each class
	set cllist [lsort -command cl_hierarchical_order [$mod get classes]]
	foreach cl $cllist {

		# detect template classes
		if {[$cl get type] == "parameterized_class"} {
			error "Parameterized classes not supported: [cl_tname $cl]"
        }

		# write forward declaration
		puts $fp "[cl_tname $cl] = class;"

		# write pointer declaration
		puts $fp "[cl_pname $cl] = ^[cl_tname $cl];"
	}
	puts $fp ""

	# write type definition associated to module
	set tdlist [obj_find_property $mod "type" "*"]
	foreach td $tdlist {
		puts $fp $td
	}
	if [llength $tdlist] {
		puts $fp ""
	}

	# write the class declarations
	foreach cl $cllist {
		dump_class_definition $fp $cl
	}

}

################################################################################
# 						Implementation procedures
################################################################################

#
# dump_static_att_list
#	writes static attributes of a class as global variables

proc dump_static_att_list { fp cl attlist } {

	set noStaticAtt -1
	foreach att $attlist {
		if {[$att get isStatic] != 1} continue
		if {$noStaticAtt} {
			puts $fp "\nvar"
			puts $fp "\t// static attributes for [cl_tname $cl]"
			set noStaticAtt 0
		}
		dump_static_att $fp $cl $att
        }
	if {!$noStaticAtt} {
		puts $fp ""
	}
}

#
# dump_static_att
#	writes static attribute as a global variable

proc dump_static_att { fp cl att } {

	# write attribute name
  	puts -nonewline $fp "\t[cl_tname $cl] : "

	# write attribute type
    if {[expr [$att get isPointer] == 1] || [expr [$att get isReference] == 1]} {
       	puts -nonewline $fp "^"
    }
	puts $fp "[att_type_tname $att];"
}


#
# dump_op_body_list
#	generate operation template for a class
# parameters
#	fp = file
#	cl = class
#	oplist = list of operations for class
#	exportControl = public, protected or private

proc dump_op_body_list { fp cl oplist exportControl } {

	# dump all operations templates
	foreach op $oplist {
		if {[$op get exportControl] != $exportControl} continue
		if {[$op get isPure] == 1} continue
        dump_op_body $fp $cl $op
	}
}

#
# dump_op_body
#	generates a template for a class method
# parameters
#		fp = file
#		cl = class
#		op = operation

proc dump_op_body { fp cl op } {

	# create header with class prefix

	# write operation header
	puts $fp "//"
	puts $fp "// [cl_tname $cl].[$op get name]"

	# write description
	puts $fp "// description"
	set rowlist [strsep [$op get description] 72 ]
	foreach row $rowlist {
		puts $fp "//\t$row"
	}
	puts $fp ""

	# write header
	set header [create_op_header $op $cl -1]
	puts $fp "$header;"

	# write BEGIN
	puts $fp "begin"

	# write semantics
	puts $fp "\t[strsubst [$op get semantics] "\r\n" "\n\t"]"

	# write END
	puts $fp "end;"
	puts $fp ""
}

#
# dump_class_implementation
#	generates operation templates and static variables for a class
# parameters
# 	fp = file
#	cl = class

proc dump_class_implementation { fp cl } {

	# get all attributes and operations
	set oplist [$cl get operations]
	set attlist [$cl get attributes]

	# write static attributes as global variables
	dump_static_att_list $fp $cl $attlist

	# write private operation templates
   	dump_op_body_list $fp $cl $oplist private

	# write protected operation templates
   	dump_op_body_list $fp $cl $oplist protected

	# write public operation templates
   	dump_op_body_list $fp $cl $oplist public
}

#
# dump_implementation
#	generates implementation code for a module

proc dump_implementation { fp mod } {

	puts $fp "implementation\n"

	# write implementation for all classes in the module
	set cllist [lsort -command cl_hierarchical_order [$mod get classes]]
	foreach cl $cllist {
		dump_class_implementation $fp $cl
	}

}

#
# dump_end
#	generates module end

proc dump_end { fp mod } {

	# write module end
	puts $fp "end."
}

#
# dump_module
# 	generates Delphi code for a module

proc dump_module { workdir mod } {

	# check module type
	set mtype [$mod get moduleType]
	if {$mtype == "specification"} {
		error "Specification modules are not allowed: [$mod get name]"
	}

	# check contents of module
	if {[llength [$mod get classes]] == 0} {
		return
	}

	# generate file name
	set modulename [string trim [$mod get name]]
	if {[string match [string toupper $modulename] "*.PAS"] == 1} {
		error "Module name cannot have extension: [$modulename]"
	}
	set filename [format "%s\\%s.pas" $workdir $modulename]

	# create file
	set fp [open $filename w]

	# generate Delphi unit source file
	dump_header $fp $mod
	dump_interface $fp $mod
	dump_implementation $fp $mod
	dump_end $fp $mod

	# close output file
  	close $fp
}

################################################################################
# main
################################################################################

	# check diagram type
	set workdir [OD_getInput "Output directory" "" [pwd]]
	set diagram [OD_getActiveDiagram]
	set dt [$diagram get type]
	if {[expr {$dt != "module"}] && [expr {$dt != "class"}]} {
		error "Select a module or class diagram"
	}

	# loop over all selected items in the diagram
	set itemlist [$diagram get objects]
	foreach item $itemlist {

		# ignore unselected items
		if {[$item get isSelected] != 1} {
			continue
        }

		# determine which module to generate
		if {[$item get objectType] == "module"} {
			set mod $item
        } else {
		if {[$item get objectType] == "class"} {
			set mod [module_from_class [$item get class]]
            if {$mod == ""} {
                continue
			}
        } else {
			continue
		}}

        # generate delphi module
		dump_module $workdir $mod
    }

