unit NTNetWk;

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Controls, Dialogs;

const
  NI_CANCEL_ERROR   = 0;
  NI_UNKNOWN_ERROR  = 1;
  NI_SHUTDOWN_ERROR = 2;

type
  ENTNetError = class(Exception);
  TNTNetErrorReporting = (niNone, niFatalErrorsOnly, niAllErrors);

  TNTNetInfo = class(TComponent)
  private
    FReportFatals: boolean;
    FNukeOnError: boolean;
    FVerbose: boolean;
    FComputerName: string;//used to check if the name is being changed
    procedure niRaiseError(errorcode: integer);
    function niGetUserName: string;
    function niGetComputerName: string;
    function niGetErrorReporting: TNTNetErrorReporting;
    procedure niBitBucket(Value: string);
    procedure niSetComputerName(Value: string);
    procedure niSetErrorReporting(Value: TNTNetErrorReporting);
  public
    constructor Create(AOwner: TComponent); override;
  published
    //property to control use of exception raising
    property ErrorReporting: TNTNetErrorReporting
                             read niGetErrorReporting
                             write niSetErrorReporting
                             default niFatalErrorsOnly;

    property UserName: string
                       read niGetUserName
                       write niBitBucket;
    property ComputerName: string
                           read niGetComputerName
                           write niSetComputerName
                           stored False; //otherwise, causes problems when loading

    //property to control display of reboot dialog when changing computer name
    property VerboseMode: boolean
                          read FVerbose
                          write FVerbose
                          default True;
  end;

procedure Register;

implementation

procedure TNTNetInfo.niSetErrorReporting(Value: TNTNetErrorReporting);
begin
  if Value <> niGetErrorReporting then
    case Value of
      niNone            : begin
                          FReportFatals := False;
                          FNukeOnError  := False;
                          end;
      niFatalErrorsOnly : begin
                          FReportFatals := True;
                          FNukeOnError  := False;
                          end;
      niAllErrors       : begin
                          FReportFatals := True;
                          FNukeOnError  := True;
                          end;
      end; //case
end;

function TNTNetInfo.niGetErrorReporting: TNTNetErrorReporting;
begin
  if FReportFatals then
    begin
    if FNukeOnError then
      Result := niAllErrors
      else
        Result := niFatalErrorsOnly;
    end
    else
      Result := niNone;
end;

procedure TNTNetInfo.niBitBucket(Value: string);
begin
  {This procedure discards the passed string.
   It is necessary for the associated property to
   show up in the OI.}
end;

function TNTNetInfo.niGetUserName:String;
Var
  lpBuffer: lpstr;
  success: Boolean;
  nSize: DWord;
Begin
  nSize := 255;
  Result := '';
  GetMem(lpBuffer, 255);
  try
    success := GetUserName(lpBuffer, nSize);
    if (not success) and (FReportFatals)
      then niRaiseError(NI_UNKNOWN_ERROR);
    Result := StrPas(lpBuffer);
  finally
    FreeMem(lpBuffer);
  end;
End;

function TNTNetInfo.niGetComputerName:String;
Var
  lpBuffer: lpstr;
  success: Boolean;
  nSize: DWord;
Begin
  nSize := 255;
  Result := '';
  GetMem(lpBuffer, 255);
  try
    success := GetComputerName(lpBuffer, nSize);
    if (not success) and (FReportFatals)
      then niRaiseError(NI_UNKNOWN_ERROR);
    FComputerName := StrPas(lpBuffer);
    Result := FComputerName;
  finally
    FreeMem(lpBuffer);
  end;
End;

procedure TNTNetInfo.niSetComputerName(Value : string);
Var
  lpBuffer: lpstr;
  success, canceled: Boolean;
begin
  //Perform some quick validation...
  Value := Trim(Value);
  if (length(Value) > MAX_COMPUTERNAME_LENGTH)
    then Value := copy(Value, 0, MAX_COMPUTERNAME_LENGTH);

  if Value <> FComputerName then
    begin
    GetMem(lpBuffer, 255);
    try
      success := True;
      canceled := false;
      if FVerbose then
        begin
        if(MessageDlg('Are you sure you want to change the name' +
                      chr(13) +
                      'of this computer? (This action will require' +
                      chr(13) +
                      'a system restart!)',
                      mtWarning,
                      mbOKCancel,
                      0) = mrOK) then
          success := SetComputerName(StrPCopy(lpBuffer, Value))
          else
            begin
            canceled := True;
            if FNukeOnError then
              niRaiseError(NI_CANCEL_ERROR);
            end;
        end
      else
        success := SetComputerName(StrPCopy(lpBuffer, Value));
      if (not success) and (FReportFatals)
        then niRaiseError(NI_UNKNOWN_ERROR);
      if (success and FVerbose and (not canceled))then
        if (MessageDlg('You have made changes which require you to restart the' +
                       chr(13) +
                       'system.  Click "OK" to restart now, or "No" to restart later.',
                       mtWarning,
                       [mbOK, mbNo],
                       0) = mrOK) then
          if (ExitWindowsEx(EWX_REBOOT, 0) = False) then
            begin
            ShowMessage('Unable to restart system at this time.  The changes' +
                        chr(13) +
                        'will not take effect until the system is restarted.');
            if FNukeOnError then
              niRaiseError(NI_SHUTDOWN_ERROR);
            end;
    finally
      FreeMem(lpBuffer);
    end;
  end;
end;

procedure TNTNetInfo.niRaiseError(errorcode: integer);
Var
  StError:String;
Begin
  case errorcode of
    NI_UNKNOWN_ERROR  : StError:='NTNetInfo error: An unknown error occurred ' +
                                 'while accessing the system information.';
    NI_CANCEL_ERROR   : StError:='NTNetInfo error: The computer name change ' +
                                 'request was cancelled.';
    NI_SHUTDOWN_ERROR : StError:='NTNetInfo error: The process was unable to' +
                                 chr(13) +
                                 'shut the system down at this time.  The' +
                                 chr(13) +
                                 'changes made will not take effect until you' +
                                 chr(13) +
                                 'restart.';
    end;
  Raise ENTNetError.Create(StError);
End;

constructor TNTNetInfo.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FReportFatals := True;
  FNukeOnError := False;
  FVerbose := True;
  niGetComputerName;  //force component to 'learn' the current computer name
end;

procedure Register;
begin
  RegisterComponents('The Good Package', [TNTNetInfo]);
end;

end.
