{
Unit        : NTELog.Pas
Description : Delphi component wrapper for NT event logging API
Version     : 1.01, 12 March 2000
Status      : Freeware.
Copyright   : 2000, First Internet Software House
Contact     : First Internet Software House - support@fishouse.com

History:
  v1.01     : Rebranded for FISH
              12 March 2000

Notes:
  Required files:
    fisEVT.MC     - message source
    MC.EXE        - compiles source to .RC - Available from MSDN online
    fisEVT.RES    - Compiled resource file
    fisEVT.PAS    - Converted from fisevt.H, which is created by MC.EXE and
                    contians required constants.
}

unit fisNTElog;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  registry,
  fisEVT; // this file converted from C header file created by MC.EXE

const

  cREG_KEY    = 'SYSTEM\CurrentControlSet\Services\EventLog\Application\';

  {Copied from WINNT.H via MSDN}
  EVENTLOG_SUCCESS          = $0000;
  EVENTLOG_ERROR_TYPE       = $0001;
  EVENTLOG_WARNING_TYPE     = $0002;
  EVENTLOG_INFORMATION_TYPE = $0004;
  EVENTLOG_AUDIT_SUCCESS    = $0008;
  EVENTLOG_AUDIT_FAILURE    = $0010;

type

  ENTLogError = class(Exception);

  TfisNTEventLog = class(TComponent)
  private
    { Private declarations }
    FEnabled: boolean;        // Did the initialisation proceed correctly?
    FAppName: string;         // name of application in registry

    evtHandle: THandle;       // Used for RegisterEventSource.

  protected
    { Protected declarations }
    procedure SetAppName(Value: string);
  public
    { Public declarations }
    procedure LogError(msg: string);
    procedure LogWarning(msg: string);
    procedure LogInfo(msg: string);
    procedure InitLogging;
    procedure CreateRegEntry;
    procedure DeleteRegEntry;
    constructor Create(aOwner: TComponent); override;
    destructor Destroy; override;
  published
    { Published declarations }
    property AppName: string read FAppName write SetAppName;
    property Enabled: boolean read FEnabled;
  end;

procedure Register;

implementation

{$r fisEVT.res}

procedure Register;
begin
  RegisterComponents('FISH', [TfisNTEventLog]);
end;

constructor TfisNTEventLog.Create(aOwner: TComponent);
begin
  inherited Create(aOwner);
  FEnabled := false;
  FAppName := '<DelphiApp>';
  evtHandle := 0;
end;

{Automatically deregisters the app if it was registered}
destructor TfisNTEventLog.Destroy;
begin
  if not(csDesigning in ComponentState) then
    if evtHandle <> 0 then
      DeregisterEventSource(evtHandle);

  inherited Destroy;
end;

procedure TfisNTEventLog.SetAppName(Value: string);
begin
  if FAppName = value then exit;
  if Value <> '' then FAppName := Value else FAppName := '<DelphiApp>';
end;

{Sets up the registry to handle logging, and registers the app}
procedure TfisNTEventLog.InitLogging;
begin
  CreateRegEntry;
  evtHandle := RegisterEventSource(nil, Pchar(FAppName));
  if evtHandle = 0 then
  begin
    FEnabled := false;
    raise(ENTLogError.Create('Unable to register event source.'));
  end;
  FEnabled := true; // initialisation successful
end;

{Creates the correct entry in the registry}
procedure TfisNTEventLog.CreateRegEntry;
var LogType: integer;
    Subkey: string;
    r: TRegistry;
begin
  LogType:= EVENTLOG_INFORMATION_TYPE or EVENTLOG_WARNING_TYPE or EVENTLOG_ERROR_TYPE;
  Subkey := cREG_KEY+FAppName;
  r := TRegistry.Create;
  r.RootKey := HKEY_LOCAL_MACHINE;
  {don't create entry if it already exists}
  if not r.OpenKey(Subkey, false) then
  begin
    if not r.CreateKey(subkey) then
    begin
      r.free;
      raise(ENTLogError.Create('Unable to register application.'));
    end;
    r.OpenKey(SubKey, false);
    r.WriteString('EventMessageFile', Application.Exename);
    r.WriteInteger('TypesSupported', LogType);
  end;
  r.CloseKey;
  r.free;
end;

{Remove the application entry from the registry. Rarely called, but
included for completness}
procedure TfisNTEventLog.DeleteRegEntry;
var r: TRegistry;
    Subkey: string;
begin
  Subkey := cREG_KEY+FAppName;
  r := TRegistry.Create;
  r.RootKey := HKEY_LOCAL_MACHINE;
  if not r.DeleteKey(subkey) then
  begin
    raise(ENTLogError.Create('Unable to remove registry entries.'));
  end;
  r.closekey;
  r.free;
end;

{Writes a message to the log as 'information' type}
procedure TfisNTEventLog.LogInfo(msg: string);
var Strings: Array[0..0] Of PChar;
begin
  if not FEnabled then exit;
  Strings[0]:= PChar(msg);
  ReportEvent(evtHandle, EVENTLOG_INFORMATION_TYPE, 0, LOGMSG_INFO,
                       nil, 1, 0, @Strings, nil);
end;

{Writes a message to the log as 'Warning' type}
procedure TfisNTEventLog.LogWarning(msg: string);
var Strings: Array[0..0] Of PChar;
begin
  if not FEnabled then exit;
  Strings[0]:= PChar(msg);
  ReportEvent(evtHandle, EVENTLOG_WARNING_TYPE, 0, LOGMSG_WARNING,
                       nil, 1, 0, @Strings, nil);
end;

{Writes a message to the log as 'Error' type}
procedure TfisNTEventLog.LogError(msg: string);
var Strings: Array[0..0] Of PChar;
begin
  if not FEnabled then exit;
  Strings[0]:= PChar(msg);
  ReportEvent(evtHandle, EVENTLOG_ERROR_TYPE, 0, LOGMSG_ERROR,
                       nil, 1, 0, @Strings, nil);
end;

end.
