{+--------------------------------------------------------------------------+
 | Component:   TTokenList
 | Created:     06.97 - 04.98
 | Author:      Martin Waldenburg
 | Copyright    1997-1998, all rights reserved.
 | Description: TTokenList is a general tokenizer, very fast.
 | Version:     1.5
 | State:       FreeWare
 +--------------------------------------------------------------------------+}
 { Disclaimer:
   This is provided as is, expressly without a warranty of any kind.
   You use it at your own risc.}

unit mwTokenList;

{$R-}


interface

uses SysUtils, Windows, Dialogs;

type

 { TTokenList class }
  TDelimiter = array[#0..#255] of ByteBool;
  PTokenArray = ^TTokenArray;
  TTokenArray = array[0..0] of PChar;
  TTokenList = class(TObject)
  private
    Run, Start: PChar;
    FTokenPos: PChar;
    FTokenList: PTokenArray;
    FCount: Integer;
    fCapacity: Integer;
    fOrigin: PChar;
    fDelimiterTable: TDelimiter;
    procedure Expand;
    procedure SetCapacity(NewCapacity: Integer);
    function GetToken(Index: Integer): String;
    function GetPosition(Index: Integer): Integer;
    function GetLength(Index: Integer): Integer;
        function GetFinished: BOOLEAN; 
  protected
    function GetItems(Index: Integer): PChar;
    procedure SetItems(Index: Integer; Item: PChar);
  public
    constructor create(OriginPtr: PChar);
    destructor Destroy; override;
    function Add(Item: PChar): Integer;
    procedure Clear;
    procedure Pack;
    function Next: String;
    procedure Tokenize(StartCapacity: Integer);
    property Capacity: Integer read fCapacity write SetCapacity;
    property Count: Integer read FCount;
    property Items[Index: Integer]: PChar read GetItems write SetItems; default;
    property Origin: PChar read fOrigin write fOrigin;
    property Token[Index: Integer]: String read GetToken;
    property Position[Index: Integer]: Integer read GetPosition;
    property Length[Index: Integer]: Integer read GetLength;
    property Delimiters: TDelimiter read fDelimiterTable write fDelimiterTable;
    property Finished: Boolean read GetFinished;
  published
  end;

implementation

{ TTokenList }

constructor TTokenList.create(OriginPtr: PChar);
begin
  inherited Create;
  fOrigin := OriginPtr;
  Start := FOrigin;
  Run := Start;
end;

procedure TTokenList.Clear;
begin
  FCount := 0;
  FCapacity := 0;
  ReallocMem(FTokenList, 0 * SizeOf(PChar));
end;

destructor TTokenList.Destroy;
begin
  Clear;
  Inherited Destroy;
end;

procedure TTokenList.Expand;
begin
  SetCapacity(FCapacity + 1024);
end;

procedure TTokenList.SetCapacity(NewCapacity: Integer);
begin
  FCapacity := NewCapacity;
  ReallocMem(FTokenList, FCapacity * SizeOf(PChar));
end;

function TTokenList.Add(Item: PChar): Integer;
begin
  Result := FCount;
  if Result = FCapacity then Expand;
  FTokenList[Result] := Item;
  Inc(FCount);
end;

function TTokenList.GetItems(Index: Integer): PChar;
begin
  Result := FTokenList[Index];
end;

procedure TTokenList.SetItems(Index: Integer; Item: PChar);
begin
  FTokenList[Index] := Item;
end;

procedure TTokenList.Pack;
begin
  ReallocMem(FTokenList, FCount * SizeOf(PChar));
end;

function TTokenList.GetLength(Index: Integer): Integer;
var
  RunPtr, StartPtr: PChar;
begin
  StartPtr := FTokenList[Index];
  RunPtr := StartPtr;
    Case FDelimiterTable[RunPtr^] of
      True: if RunPtr^ <> #0 then Inc(RunPtr);
      False:
        repeat
          Inc(RunPtr);
        until FDelimiterTable[RunPtr^];
    end;
  Result := RunPtr - StartPtr;
end;

function TTokenList.GetPosition(Index: Integer): Integer;
begin
  FTokenPos := FTokenList[Index];
  Result := FTokenPos - FOrigin;
end;

function TTokenList.GetToken(Index: Integer): String;
var
  StartPtr: PChar;
  TokenLen: Integer;
begin
  StartPtr := FTokenList[Index];
  TokenLen := GetLength(Index);
  SetString(Result, StartPtr, TokenLen);
end;

procedure TTokenList.Tokenize(StartCapacity: Integer);
begin
  SetCapacity(StartCapacity);
  Run := FOrigin;
  while (Run^ <> #0) do
  begin
    Add(Run);
    Case FDelimiterTable[Run^] of
      True: Inc(Run);
      False:
        repeat
          Inc(Run);
        until FDelimiterTable[Run^];
    end;
  end;
end;

function TTokenList.Next: String;
begin
  Start := Run;
  Case FDelimiterTable[Run^] of
    True: Inc(Run);
    False:
      repeat
        Inc(Run);
      until FDelimiterTable[Run^];
  end;
  SetString(Result, Start, Run - Start);
end;

function TTokenList.GetFinished: Boolean;
begin
  Case Run^ = #0 of
    True: Result:= True;
    False: Result:= False;
  end;
end;

end.

{ Martin Waldenburg
  Landaeckerstrasse 27
  71642 Ludwigsburg
  Germany }

