{+--------------------------------------------------------------------------+
 | Unit:        mwEPParser
 | Created:     12.97
 | Author:      Martin Waldenburg
 | Copyright    1997, all rights reserved.
 | Description: Enhanced Pascal parser
 | Version:     1.1
 | Status:      FreeWare
 | DISCLAIMER:  This is provided as is, expressly without a warranty of any kind.
 |              You use it at your own risc.
 +--------------------------------------------------------------------------+}

unit mwEPParser;

interface

uses
  Windows, SysUtils, ComCtrls, mwPasParser;

type
  TmInfoKind = (ikClass, ikClEmpty, ikClEnd, ikClFunction, ikClForward,
    ikClProcedure, ikClReference, ikConst, ikConstructor,
    ikDestructor, ikField, ikFunction, ikNull, ikObject, ikPrivate,
    ikProcedure, ikProperty, ikProtected, ikPublic, ikPublished,
    ikType, ikUnit, ikUses, ikInterface, ikImplementation, ikUnknown,
    ikVar);


  TmAdditionalInfo = class(TObject)
    aiUnit: String;
    aiPath: String;
    Ready: Boolean;
    constructor Create;
  end; { TmInfoPas }

  TmPasInfo = class(TObject)
    ID: TmInfoKind;
    Data: String;
    LineNumber: Integer;
    LinePos: Integer;
    AI: TmAdditionalInfo;
    constructor Create;
    destructor Destroy; override;
  end; { TmInfoPas }

  TmEParser = class(TmPasParser)
  private
    Helper: TmPasParser;
    FInfo: TmPasInfo;
    procedure ResetHelper;
  protected
  public
    constructor Create;
    destructor Destroy; override;
    procedure Init(NewOrigin: PChar);
    function NextClassKind: TmInfoKind;
    procedure LoadField;
    procedure LoadNonFieldDeclaration;
    function NextClassElement: TmInfoKind;
    property Info: TmPasInfo read FInfo;
  published
  end; { TmEParser }

implementation

constructor TmAdditionalInfo.Create;
begin
  inherited Create;
  aiUnit := '';
  aiPath := '';
  Ready := False;
end;

constructor TmPasInfo.Create;
begin
  inherited Create;
  ID := ikUnknown;
  Data := '';
  LineNumber := 0;
  LinePos := 0;
  AI := nil;
end;

destructor TmPasInfo.Destroy;
begin
  if AI <> nil then AI.Free;
  inherited Destroy;
end;

destructor TmEParser.Destroy;
begin
  Helper.Free;
  FInfo.Free;
  inherited Destroy;
end; { Destroy }

constructor TmEParser.Create;
begin
  inherited Create;
  Helper := TmPasParser.Create;
  FInfo := TmPasInfo.Create;
end; { Create }

procedure TmEParser.Init(NewOrigin: PChar);
begin
  Origin := NewOrigin;
  Helper.Origin := NewOrigin;
end; { Init }

procedure TmEParser.ResetHelper;
begin
  Helper.Origin:= Origin;
  Helper.Token.LineNumber := Token.LineNumber;
  Helper.Token.LinePos := Token.LinePos;
  Helper.Comments := Comments;
  Helper.Visibility := Visibility;
  Helper.RunPos := RunPos;
  Case Helper.Token.ID of
    tkAnsiComment: Helper.NextNonJunk;
    tkBorComment: Helper.NextNonJunk;
    tkCRLF: Helper.NextNonJunk;
    tkCRLFCo: Helper.NextNonJunk;
    tkSlashesComment: Helper.NextNonJunk;
    tkSpace: Helper.NextNonJunk;
  end;
end; { ResetHelper }

function TmEParser.NextClassKind: TmInfoKind;
var
  TempRunPos: LongInt;
begin
  NextClassLine;
  FInfo.LineNumber := Token.LineNumber;
  FInfo.LinePos := Token.LinePos;
  TempRunPos := RunPos;
  FInfo.Data := GetSubString(LastIdentPos, RunPos);
  NextNonJunk;
  Case Token.ID of
    tkEnd:
      begin
        FInfo.ID := ikClEmpty;
        NextToken;
      end;
    tkNull: FInfo.ID := ikNull;
    tkClass, tkPrivate, tkProtected, tkPublic, tkPublished:
      begin
        FInfo.ID := ikClass;
        if Token.ID = tkIdentifier then Visibility := tkPublic;
      end;
    tkSemiColon:
      begin
        FInfo.ID := ikClForward;
        FInfo.Data := FInfo.Data + ';';
        Visibility := tkUnknown;
        EndCount := 0;
        NextToken;
      end;
    tkOf:
      begin
        FInfo.ID := ikClReference;
        Visibility := tkUnknown;
        repeat
          NextToken;
        until Token.ID = tkSemiColon;
        FInfo.Data := FInfo.Data + GetSubString(TempRunPos, RunPos);
        EndCount := 0;
        NextToken;
      end;
    tkRoundOpen:
      begin
        FInfo.ID := ikClass;
        FInfo.Data := FInfo.Data + '(';
        NextNonComment;
        while Token.ID <> tkRoundClose do
        begin
          Case Token.ID of
            tkCRLF, tkSpace: FInfo.Data := FInfo.Data + ' ';
          else FInfo.Data := FInfo.Data + Token.Data;
          end;
          NextNonComment;
        end;
        FInfo.Data := FInfo.Data + ')';
        NextNonJunk;
        if Token.ID = tkSemiColon then
        begin
          FInfo.ID := ikClEmpty;
          FInfo.Data := FInfo.Data + ';';
          NextToken;
        end;
      end;
  else
    begin
      if Token.ID in IdentDirect then
      begin
        FInfo.ID := ikClass;
        if Token.ID = tkIdentifier then Visibility := tkPublic;
      end;
    end;
  end;
  Result := FInfo.ID;
end; { NextClassKind }

procedure TmEParser.LoadField;
begin
  RoundCount := 0;
  SquareCount := 0;
  FInfo.ID := ikField;
  FInfo.LineNumber := Token.LineNumber;
  FInfo.LinePos := Token.LinePos;
  FInfo.Data := Token.Data;
  NextNonJunk;
  if Token.ID = tkColon then
  begin
    while (not ((Token.Id = tkSemiColon) and (RoundCount = 0) and
      (SquareCount = 0))) and (Token.ID <> tkNull) do
    begin
      Case Token.ID of
        tkCRLF, tkSpace: FInfo.Data := FInfo.Data + ' ';
      else FInfo.Data := FInfo.Data + Token.Data;
      end;
      NextNonComment;
    end;
    FInfo.Data := FInfo.Data + ';';
  end
  else
    if Token.ID = tkComma then
    begin
      ResetHelper;
      Helper.RoundCount := 0;
      Helper.SquareCount := 0;
      while Helper.Token.ID <> tkColon do Helper.NextNonJunk;
      while (not ((Helper.Token.Id = tkSemiColon) and (Helper.RoundCount = 0) and
        (Helper.SquareCount = 0))) and (Helper.Token.ID <> tkNull) do
      begin
        Case Helper.Token.ID of
          tkCRLF, tkSpace: FInfo.Data := FInfo.Data + ' ';
        else FInfo.Data := FInfo.Data + Helper.Token.Data;
        end;
        Helper.NextNonComment;
      end;
      FInfo.Data := FInfo.Data + ';';
    end;
  NextToken;
end; { LoadField }

procedure TmEParser.LoadNonFieldDeclaration;
begin
  while (not ((Token.Id = tkSemiColon) and (RoundCount = 0) and
    (SquareCount = 0))) and (Token.ID <> tkNull) do
  begin
    Case Token.ID of
      tkCRLF, tkSpace: FInfo.Data := FInfo.Data + ' ';
    else FInfo.Data := FInfo.Data + Token.Data;
    end;
    NextNonComment;
  end;
  FInfo.Data := FInfo.Data + ';';
  NextNonComment;
  while (not (Token.Id in [tkClass, tkConstructor, tkDestructor, tkEnd,
    tkFunction, tkPrivate, tkProcedure, tkProperty,
      tkProtected, tkPublic, tkPublished])) and
    (Token.ID <> tkNull) do
  begin
    Case Token.ID of
      tkCRLF, tkSpace: FInfo.Data := FInfo.Data + ' ';
    else FInfo.Data := FInfo.Data + Token.Data;
    end;
    NextNonComment;
  end;
end; { LoadNonFieldDeclaration }

function TmEParser.NextClassElement: TmInfoKind;
begin
  if Token.ID <> tkNull then
  begin
    if IsJunk then NextNonJunk;
    Case Token.ID of
      tkClass:
        begin
          FInfo.LineNumber := Token.LineNumber;
          FInfo.LinePos := Token.LinePos;
          FInfo.Data := Token.Data + ' ';
          NextNonJunk;
          Case Token.ID of
            tkFunction:
              begin
                FInfo.ID := ikClFunction;
                LoadNonFieldDeclaration;
              end;
            tkProcedure:
              begin
                FInfo.ID := ikClProcedure;
                LoadNonFieldDeclaration;
              end;
          end;
        end;

      tkConstructor:
        begin
          FInfo.ID := ikConstructor;
          FInfo.LineNumber := Token.LineNumber;
          FInfo.LinePos := Token.LinePos;
          FInfo.Data := '';
          LoadNonFieldDeclaration;
        end;

      tkDestructor:
        begin
          FInfo.ID := ikDestructor;
          FInfo.LineNumber := Token.LineNumber;
          FInfo.LinePos := Token.LinePos;
          FInfo.Data := '';
          LoadNonFieldDeclaration;
        end;

      tkEnd:
        begin
          FInfo.ID := ikClEnd;
          FInfo.LineNumber := Token.LineNumber;
          FInfo.LinePos := Token.LinePos;
          FInfo.Data := Token.Data;
          NextNonJunk;
          FInfo.Data := FInfo.Data + Token.Data;
          NextToken;
        end;

      tkFunction:
        begin
          FInfo.ID := ikFunction;
          FInfo.LineNumber := Token.LineNumber;
          FInfo.LinePos := Token.LinePos;
          FInfo.Data := '';
          LoadNonFieldDeclaration;
        end;

      tkPrivate:
        begin
          FInfo.ID := ikPrivate;
          FInfo.LineNumber := Token.LineNumber;
          FInfo.LinePos := Token.LinePos;
          FInfo.Data := Token.Data;
          NextNonJunk;
        end;

      tkProcedure:
        begin
          FInfo.ID := ikProcedure;
          FInfo.LineNumber := Token.LineNumber;
          FInfo.LinePos := Token.LinePos;
          FInfo.Data := '';
          LoadNonFieldDeclaration;
        end;

      tkProperty:
        begin
          FInfo.ID := ikProperty;
          FInfo.LineNumber := Token.LineNumber;
          FInfo.LinePos := Token.LinePos;
          FInfo.Data := '';
          LoadNonFieldDeclaration;
        end;

      tkProtected:
        begin
          FInfo.ID := ikProtected;
          FInfo.LineNumber := Token.LineNumber;
          FInfo.LinePos := Token.LinePos;
          FInfo.Data := Token.Data;
          NextNonJunk;
        end;

      tkPublic:
        begin
          FInfo.ID := ikPublic;
          FInfo.LineNumber := Token.LineNumber;
          FInfo.LinePos := Token.LinePos;
          FInfo.Data := Token.Data;
          NextNonJunk;
        end;

      tkPublished:
        begin
          FInfo.ID := ikPublished;
          FInfo.LineNumber := Token.LineNumber;
          FInfo.LinePos := Token.LinePos;
          FInfo.Data := Token.Data;
          NextNonJunk;
        end;
    else
      begin
        if Token.ID in IdentDirect then
        begin
          FInfo.Data := '';
          LoadField;
        end;
      end;
    end;
  end;
  Result := FInfo.ID;
end; { NextClassElement }

end.

