{-------------------------------------------------------------------}
{ MULTISHAPE - VCL component to create graphic effects v.1.00       }
{-------------------------------------------------------------------}
{ v. 1.00 April, 4 1996                                             }
{-------------------------------------------------------------------}
{ Copyright Enrico Lodolo                                           }
{ via F.Bolognese 27/3 - 40129 Bologna - Italy                      }
{ CIS 100275,1255 - Internet e.lodolo@bo.nettuno.it                 }
{-------------------------------------------------------------------}

unit MShape;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs;

const DefSize=10;
      DefSpacing=20;
      DefShadowOfs=2;

type TMultiShapeType=
       (msRectangle,msRoundRect,msDiamond,msEllipse,
        msTriangle,msLine,msText);
     TRepeatMode=(rpNone,rpVert,rpHoriz,rpBoth);

type TShapeStr=string[255];

type
  TMultiShape = class(TGraphicControl)
  private
    FAngle:Integer;
    FAutoSize:Boolean;
    FNewFont,FOldFont:HFont;
    FDX,FDY:Integer;
    FFilled:Boolean;
    FRepeatMode:TRepeatMode;
    FShapeType:TMultiShapeType;
    FShapeH:Integer;
    FShapeW:Integer;
    FXSpacing:Integer;
    FYSpacing:Integer;
    FXMargin:Integer;
    FYMargin:Integer;
    FBorder:Boolean;
    FBorderColor:TColor;
    FBorderWidth:Integer;
    FShadow:Boolean;
    FShadowColor:TColor;
    FShadowX:Integer;
    FShadowY:Integer;
    procedure CMTextChanged(var Message: TMessage);
      message CM_TEXTCHANGED;
    procedure CMFontChanged(var Message: TMessage);
      message CM_FONTCHANGED;
    procedure CMColorChanged(var Message: TMessage);
      message CM_COLORCHANGED;
  protected
    {-- Property Handlers --------------------------------}
    procedure SetAngle(Value:Integer);
    procedure SetAutoSize(Value:Boolean);
    procedure SetFilled(Value:Boolean);
    procedure SetRepeatMode(Value:TRepeatMode);
    procedure SetShapeType(Value:TMultiShapeType);
    procedure SetShapeH(Value:Integer);
    procedure SetShapeW(Value:Integer);
    procedure SetXSpacing(Value:Integer);
    procedure SetYSpacing(Value:Integer);
    procedure SetXMargin(Value:Integer);
    procedure SetYMargin(Value:Integer);
    procedure SetBorder(Value:Boolean);
    procedure SetBorderColor(Value:TColor);
    procedure SetBorderWidth(Value:Integer);
    procedure SetShadow(Value:Boolean);
    procedure SetShadowColor(Value:TColor);
    procedure SetShadowX(Value:Integer);
    procedure SetShadowY(Value:Integer);
    {-- Protected methods --------------------------------}
    procedure PrepareText;
    procedure UnprepareText;
    procedure AdjustShapeSize;
    procedure AdjustControlSize;
    procedure DrawRectangle(X,Y:Integer);
    procedure DrawRoundRect(X,Y:Integer);
    procedure DrawDiamond(X,Y:Integer);
    procedure DrawEllipse(X,Y:Integer);
    procedure DrawTriangle(X,Y:Integer);
    procedure DrawLine(X,Y:Integer);
    procedure DrawText(X,Y:Integer);
    procedure Paint; override;
    property AutoSize:Boolean read FAutoSize write SetAutoSize;
  public
    procedure SetBounds(ALeft,ATop,AWidth,AHeight:Integer); override;
  published
    constructor Create(AOwner:TComponent); override;
    property Align;
    property Color;
    property Font;
    property ParentColor;
    property ParentFont;
    property Text;
    property Angle:Integer read FAngle write SetAngle;
    property Filled:Boolean
      read FFilled write SetFilled default True;
    property RepeatMode:TRepeatMode
      read FRepeatMode write SetRepeatMode default rpBoth;
    property ShapeType:TMultiShapeType
      read FShapeType write SetShapeType;
    property ShapeH:Integer
      read FShapeH write SetShapeH default DefSize;
    property ShapeW:Integer
      read FShapeW write SetShapeW default DefSize;
    property XSpacing:Integer
      read FXSpacing write SetXSpacing default DefSpacing;
    property YSpacing:Integer
      read FYSpacing write SetYSpacing default DefSpacing;
    property XMargin:Integer
      read FXMargin write SetXMargin;
    property YMargin:Integer
      read FYMargin write SetYMargin;
    property Border:Boolean
      read FBorder write SetBorder;
    property BorderColor:TColor
      read FBorderColor write SetBorderColor default clBlack;
    property BorderWidth:Integer
      read FBorderWidth write SetBorderWidth default 1;
    property Shadow:Boolean
      read FShadow write SetShadow;
    property ShadowColor:TColor
      read FShadowColor write SetShadowColor default clGray;
    property ShadowX:Integer
      read FShadowX write SetShadowX default DefShadowOfs;
    property ShadowY:Integer
      read FShadowY write SetShadowY default DefShadowOfs;
  end;

procedure Register;

implementation

{--------------------------------------------------------------}
{                       Message Handlers                       }
{--------------------------------------------------------------}

procedure TMultiShape.CMTextChanged(var Message:TMessage);

begin
     Invalidate;
end;

procedure TMultiShape.CMFontChanged(var Message:TMessage);

begin
     inherited;
     Color:=Font.Color;
end;

procedure TMultiShape.CMColorChanged(var Message:TMessage);

begin
     inherited;
     Font.Color:=Color;
end;

{--------------------------------------------------------------}
{                         Basic methods                        }
{--------------------------------------------------------------}

procedure TMultiShape.AdjustShapeSize;

begin
     if FAutoSize then
       begin
         FShapeW:=Width-FXMargin*2;
         FShapeH:=Height-FYMargin*2;
         if Shadow then
           begin
             Dec(FShapeW,ShadowX);
             Dec(FShapeH,ShadowY);
           end;
       end;
end;

procedure TMultiShape.AdjustControlSize;

var H,W:Integer;

begin
     if FAutoSize then
       begin
         W:=FShapeW+FXMargin*2;
         H:=FShapeH+FYMargin*2;
         if FShadow then
           begin
             Inc(W,ShadowX);
             Inc(H,ShadowY);
           end;
         Width:=W;
         Height:=H;
       end;
end;

procedure TMultiShape.SetBounds(ALeft,ATop,AWidth,AHeight:Integer);

begin
     inherited SetBounds(ALeft,ATop,AWidth,AHeight);
     AdjustShapeSize;
end;

{--------------------------------------------------------------}
{                       Property Handlers                      }
{--------------------------------------------------------------}

procedure TMultiShape.SetAngle(Value:Integer);

begin
     if Value<>FAngle then
       begin
         FAngle:=Value;
         {-- Normalization: -179 .. +180 ---------------}
         FAngle:=FAngle mod 360;
         if FAngle=-180 then FAngle:=180;
         if FAngle>180 then FAngle:=-(360-FAngle);
         if FAngle<-180 then FAngle:=(360+FAngle);
         {-- Only 45 steps allowed for triangles --------}
         if ShapeType in [msTriangle,msLine] then
           FAngle:=Round(FAngle/45)*45;
         {-- Refresh -------------------------------------}
         Invalidate;
       end;
end;

procedure TMultiShape.SetAutoSize(Value:Boolean);

begin
     if Value<>FAutoSize then
       begin
         FAutoSize:=Value;
         if FAutoSize then
           begin
             FShapeW:=Width-FXMargin*2;
             FShapeH:=Height-FYMargin*2;
           end;
         Invalidate;
       end;
end;

procedure TMultiShape.SetFilled(Value:Boolean);

begin
     if Value<>FFilled then
       begin
         FFilled:=Value;
         Invalidate;
       end;
end;

procedure TMultiShape.SetRepeatMode(Value:TRepeatMode);

begin
     if Value<>FRepeatMode then
       begin
         FRepeatMode:=Value;
         AutoSize:=FRepeatMode=rpNone;
         Invalidate;
       end;
end;

procedure TMultiShape.SetShapeType(Value:TMultiShapeType);

begin
     if Value<>FShapeType then
       begin
         FShapeType:=Value;
         Invalidate;
       end;
end;

procedure TMultiShape.SetShapeH(Value:Integer);

begin
     if Value<>FShapeH then
       begin
         FShapeH:=Value;
         AdjustControlSize;
         Invalidate;
       end;
end;

procedure TMultiShape.SetShapeW(Value:Integer);

begin
     if Value<>FShapeW then
       begin
         FShapeW:=Value;
         AdjustControlSize;
         Invalidate;
       end;
end;

procedure TMultiShape.SetXSpacing(Value:Integer);

begin
     if Value<>FXSpacing then
       begin
         FXSpacing:=Value;
         Invalidate;
       end;
end;

procedure TMultiShape.SetYSpacing(Value:Integer);

begin
     if Value<>FYSpacing then
       begin
         FYSpacing:=Value;
         Invalidate;
       end;
end;

procedure TMultiShape.SetXMargin(Value:Integer);

begin
     if Value<>FXMargin then
       begin
         FXMargin:=Value;
         AdjustControlSize;
         Invalidate;
       end;
end;

procedure TMultiShape.SetYMargin(Value:Integer);

begin
     if Value<>FYMargin then
       begin
         FYMargin:=Value;
         AdjustControlSize;
         Invalidate;
       end;
end;

procedure TMultiShape.SetBorder(Value:Boolean);

begin
     if Value<>FBorder then
       begin
         FBorder:=Value;
         Invalidate;
       end;
end;

procedure TMultiShape.SetBorderColor(Value:TColor);

begin
     if Value<>FBorderColor then
       begin
         FBorderColor:=Value;
         Invalidate;
       end;
end;

procedure TMultiShape.SetBorderWidth(Value:Integer);

begin
     if Value<>FBorderWidth then
       begin
         FBorderWidth:=Value;
         Invalidate;
       end;
end;

procedure TMultiShape.SetShadow(Value:Boolean);

begin
     if Value<>FShadow then
       begin
         FShadow:=Value;
         AdjustControlSize;
         Invalidate;
       end;
end;

procedure TMultiShape.SetShadowColor(Value:TColor);

begin
     if Value<>FShadowColor then
       begin
         FShadowColor:=Value;
         Invalidate;
       end;
end;

procedure TMultiShape.SetShadowX(Value:Integer);

begin
     if Value<>FShadowX then
       begin
         FShadowX:=Value;
         AdjustControlSize;
         Invalidate;
       end;
end;

procedure TMultiShape.SetShadowY(Value:Integer);

begin
     if Value<>FShadowY then
       begin
         FShadowY:=Value;
         AdjustControlSize;
         Invalidate;
       end;
end;

{--------------------------------------------------------------}
{                          Draw methods                        }
{--------------------------------------------------------------}

procedure TMultiShape.PrepareText;

var Rad:Extended;
    TL,TH:Integer;
    Sz:TSize;
    FontName:string[255];

begin
     {-- Sets the font ----------------------------------------}
     FontName:=Font.Name+#0;
     FNewFont:= CreateFont(
          { Height      } Font.Height,
          { Width       } 0,
          { Escapement  } FAngle*10,
          { Orientation } 0,
          { Weight      } Ord(fsBold in Font.Style)*FW_BOLD,
          { Italic      } Ord(fsItalic in Font.Style),
          { Underline   } Ord(fsUnderLine in Font.Style),
          { StrikeOut   } Ord(fsStrikeOut in Font.Style),
          { CharSet     } DEFAULT_CHARSET,
          { OutputPrec. } OUT_DEFAULT_PRECIS,
          { ClipPrec.   } CLIP_DEFAULT_PRECIS,
          { Quality     } DEFAULT_QUALITY,
          { Picth       } DEFAULT_PITCH + FF_DONTCARE,
          { FaceName    } @FontName[1]);
     FOldFont:=SelectObject(Canvas.Handle,FNewFont);
     Canvas.Brush.Style:= bsClear;
     {-- Calculates text offset from shape center -------------}
     GetTextExtentPoint32(Canvas.Handle,@Text[1],Length(Text),Sz);
     TL:=Abs(Sz.CX);TH:=Abs(Sz.CY);
     Rad:=FAngle*Pi/180;
     FDX:=Round(TL/2*cos(Rad)+TH/2*sin(Rad));
     FDY:=Round(TL/2*sin(Rad)-TH/2*cos(Rad));
end;

procedure TMultiShape.UnprepareText;

begin
     SelectObject(Canvas.Handle,FOldFont);
     DeleteObject(FNewFont);
end;

procedure TMultiShape.DrawRectangle(X,Y:Integer);

var SX,SY,i:Integer;
    Pt,ShPt:array[1..5] of TPoint;

begin
     Pt[1].X:=X;Pt[1].Y:=Y;
     Pt[2].X:=X+ShapeW;Pt[2].Y:=Y;
     Pt[3].X:=X+ShapeW;Pt[3].Y:=Y+ShapeH;
     Pt[4].X:=X;Pt[4].Y:=Y+ShapeH;
     Pt[5]:=Pt[1];
     if Shadow then
       begin
         for i:=1 to 5 do
           begin
             ShPt[i]:=Pt[i];
             Inc(ShPt[i].X,ShadowX);
             Inc(ShPt[i].Y,ShadowY);
           end;
         SX:=X+ShadowX;SY:=Y+ShadowY;
         Canvas.Pen.Color:=ShadowColor;
         Canvas.Pen.Width:=BorderWidth;
         Canvas.Brush.Color:=ShadowColor;
         if Filled then Canvas.Rectangle(SX,SY,SX+ShapeW,SY+ShapeH)
         else Canvas.PolyLine(ShPt);
       end;
     if Border then Canvas.Pen.Color:=BorderColor
     else Canvas.Pen.Color:=Color;
     Canvas.Pen.Width:=BorderWidth;
     Canvas.Brush.Color:=Color;
     if Filled then Canvas.Rectangle(X,Y,X+ShapeW,Y+ShapeH)
     else Canvas.PolyLine(Pt);
end;

procedure TMultiShape.DrawRoundRect(X,Y:Integer);

var SX,SY:Integer;

begin
     if Shadow then
       begin
         SX:=X+ShadowX;SY:=Y+ShadowY;
         Canvas.Pen.Color:=ShadowColor;
         Canvas.Pen.Width:=1;
         Canvas.Brush.Color:=ShadowColor;
         Canvas.RoundRect(SX,SY,SX+ShapeW,SY+ShapeH,ShapeW div 2,ShapeH div 2);
       end;
     if Border then Canvas.Pen.Color:=BorderColor
     else Canvas.Pen.Color:=Color;
     Canvas.Pen.Width:=BorderWidth;
     Canvas.Brush.Color:=Color;
     Canvas.RoundRect(X,Y,X+ShapeW,Y+ShapeH,ShapeW div 2,ShapeH div 2);
end;

procedure TMultiShape.DrawDiamond(X,Y:Integer);

var i:Integer;
    Pt,ShPt:array[1..5] of TPoint;

begin
     Pt[1].X:=X+ShapeW div 2;Pt[1].Y:=Y;
     Pt[2].X:=X+(ShapeW div 2)*2;Pt[2].Y:=Y+ShapeH div 2;
     Pt[3].X:=Pt[1].X;Pt[3].Y:=Y+(ShapeH div 2)*2;
     Pt[4].X:=X;Pt[4].Y:=Pt[2].Y;
     Pt[5]:=Pt[1];
     if Shadow then
       begin
         for i:=1 to 5 do
           begin
             ShPt[i]:=Pt[i];
             Inc(ShPt[i].X,ShadowX);
             Inc(ShPt[i].Y,ShadowY);
           end;
         Canvas.Pen.Color:=ShadowColor;
         Canvas.Pen.Width:=BorderWidth;
         Canvas.Brush.Color:=ShadowColor;
         if Filled then Canvas.Polygon(ShPt)
         else Canvas.PolyLine(ShPt);
       end;
     if Border then Canvas.Pen.Color:=BorderColor
     else Canvas.Pen.Color:=Color;
     Canvas.Pen.Width:=BorderWidth;
     Canvas.Brush.Color:=Color;
     if Filled then Canvas.Polygon(Pt)
     else Canvas.PolyLine(Pt);
end;

procedure TMultiShape.DrawEllipse(X,Y:Integer);

var SX,SY:Integer;

begin
     if Shadow then
       begin
         SX:=X+ShadowX;SY:=Y+ShadowY;
         Canvas.Pen.Color:=ShadowColor;
         Canvas.Pen.Width:=BorderWidth;
         Canvas.Brush.Color:=ShadowColor;
         if Filled then Canvas.Ellipse(SX,SY,SX+ShapeW,SY+ShapeH)
         else Canvas.Arc(SX,SY,SX+ShapeW,SY+ShapeH,SX,SY,SX,SY);
       end;
     if Border then Canvas.Pen.Color:=BorderColor
     else Canvas.Pen.Color:=Color;
     Canvas.Pen.Width:=BorderWidth;
     Canvas.Brush.Color:=Color;
     if Filled then Canvas.Ellipse(X,Y,X+ShapeW,Y+ShapeH)
     else Canvas.Arc(X,Y,X+ShapeW,Y+ShapeH,X,Y,X,Y);
end;

procedure TMultiShape.DrawTriangle(X,Y:Integer);

var i,SW,SH:Integer;
    Pt,ShPt:array[1..4] of TPoint;

begin
     {-- Odd sizes not allowed for triangles }
     SW:=(ShapeW div 2) * 2;
     SH:=(ShapeH div 2) * 2;
     case Angle of
       -135:begin
              Pt[1].X:=X;Pt[1].Y:=Y;
              Pt[2].X:=X;Pt[2].Y:=Y+SH;
              Pt[3].X:=X+SW;Pt[3].Y:=Y+SH;
            end;
        -90:begin
              Pt[1].X:=X;Pt[1].Y:=Y;
              Pt[2].X:=X+SW;Pt[2].Y:=Y;
              Pt[3].X:=X+SW div 2;Pt[3].Y:=Y+SH;
            end;
        -45:begin
              Pt[1].X:=X+SW;Pt[1].Y:=Y;
              Pt[2].X:=X+SW;Pt[2].Y:=Y+SH;
              Pt[3].X:=X;Pt[3].Y:=Y+SH;
            end;
          0:begin
              Pt[1].X:=X;Pt[1].Y:=Y;
              Pt[2].X:=X+SW;Pt[2].Y:=Y+SH div 2;
              Pt[3].X:=X;Pt[3].Y:=Y+SH;
            end;
         45:begin
              Pt[1].X:=X;Pt[1].Y:=Y;
              Pt[2].X:=X+SW;Pt[2].Y:=Y;
              Pt[3].X:=X+SW;Pt[3].Y:=Y+SH;
            end;
         90:begin
              Pt[1].X:=X+SW div 2;Pt[1].Y:=Y;
              Pt[2].X:=X+SW;Pt[2].Y:=Y+SH;
              Pt[3].X:=X;Pt[3].Y:=Y+SH;
            end;
        135:begin
              Pt[1].X:=X;Pt[1].Y:=Y;
              Pt[2].X:=X+SW;Pt[2].Y:=Y;
              Pt[3].X:=X;Pt[3].Y:=Y+SH;
            end;
        180:begin
              Pt[1].X:=X;Pt[1].Y:=Y+SH div 2;
              Pt[2].X:=X+SW;Pt[2].Y:=Y;
              Pt[3].X:=X+SW;Pt[3].Y:=Y+SH;
            end;
     end;
     Pt[4]:=Pt[1];
     if Shadow then
       begin
         for i:=1 to 4 do
           begin
             ShPt[i]:=Pt[i];
             Inc(ShPt[i].X,ShadowX);
             Inc(ShPt[i].Y,ShadowY);
           end;  
         Canvas.Pen.Color:=ShadowColor;
         Canvas.Pen.Width:=BorderWidth;
         Canvas.Brush.Color:=ShadowColor;
         if Filled then Canvas.Polygon(ShPt)
         else Canvas.PolyLine(ShPt);
       end;
     if Border then Canvas.Pen.Color:=BorderColor
     else Canvas.Pen.Color:=Color;
     Canvas.Pen.Width:=BorderWidth;
     Canvas.Brush.Color:=Color;
     if Filled then Canvas.Polygon(Pt)
     else Canvas.PolyLine(Pt);
end;

procedure TMultiShape.DrawLine(X,Y:Integer);

var A,X1,Y1,X2,Y2,CX,CY:Integer;

begin
     if Angle<0 then A:=180-Angle else A:=Angle; 
     A:=A mod 180;
     CX:=ShapeW div 2;
     CY:=ShapeH div 2;
     case A of
        0:begin
            X1:=X;Y1:=Y+CY;
            X2:=X+ShapeW;Y2:=Y+CY;
          end;
       45:begin
            X1:=X;Y1:=Y+ShapeH;
            X2:=X+ShapeW;Y2:=Y;
          end;
       90:begin
            X1:=X+CX;Y1:=Y;
            X2:=X+CX;Y2:=Y+ShapeH;
          end;
      135:begin
            X1:=X;Y1:=Y;
            X2:=X+ShapeW;Y2:=Y+ShapeH;
          end;
     else begin
            X1:=X+CX;Y1:=Y+CY;
            X2:=CX+ShapeW;Y2:=Y+CY;
          end;
     end;
     Canvas.Pen.Width:=BorderWidth;
     if Shadow then
       begin
         Canvas.Pen.Color:=ShadowColor;
         Inc(X1,ShadowX);Inc(Y1,ShadowY);
         Inc(X2,ShadowX);Inc(Y2,ShadowY);
         Canvas.MoveTo(X1,Y1);
         Canvas.LineTo(X2,Y2);
         Dec(X1,ShadowX);Dec(Y1,ShadowY);
         Dec(X2,ShadowX);Dec(Y2,ShadowY);
       end;
     Canvas.Pen.Color:=Color;
     Canvas.MoveTo(X1,Y1);
     Canvas.LineTo(X2,Y2);
end;

procedure TMultiShape.DrawText(X,Y:Integer);

var TX,TY,SX,SY:Integer;

begin
     TX:=X+ShapeW div 2-FDX;
     TY:=Y+ShapeH div 2+FDY;
     if Shadow then
       begin
         SetTextColor(Canvas.Handle,ColorToRGB(ShadowColor));
         SX:=TX+ShadowX;SY:=TY+ShadowY;
         TextOut(Canvas.Handle,SX,SY,@Text[1],Length(Text));
       end;
     SetTextColor(Canvas.Handle,ColorToRGB(Color));
     TextOut(Canvas.Handle,TX,TY,@Text[1],Length(Text));
end;

constructor TMultiShape.Create(AOwner:TComponent);

begin
     inherited Create(AOwner);
     Width:=30;
     Height:=30;
     Color:=clNavy;
     FFilled:=True;
     FShapeH:=DefSize;
     FShapeW:=DefSize;
     FRepeatMode:=rpBoth;
     FXSpacing:=DefSpacing;
     FYSpacing:=DefSpacing;
     FBorderColor:=clBlack;
     FBorderWidth:=1;
     FShadowColor:=clGray;
     FShadowX:=DefShadowOfs;
     FShadowY:=DefShadowOfs;
end;

procedure TMultiShape.Paint;

var i,j,XN,YN,X,Y:Integer;

begin
     inherited Paint;
     if FShapeType=msText then PrepareText;
     XN:=1;YN:=1;
     if (RepeatMode in [rpHoriz,rpBoth]) and (XSpacing>0) then
       XN:=Width div XSpacing+1;
     if (RepeatMode in [rpVert,rpBoth]) and (YSpacing>0) then
       YN:=Height div YSpacing+1;
     for i:=1 to YN do
       for j:=1 to XN do
         begin
           X:=(j-1)*XSpacing+XMargin;
           Y:=(i-1)*YSpacing+YMargin;
           case FShapeType of
             msRectangle:DrawRectangle(X,Y);
             msRoundRect:DrawRoundRect(X,Y);
               msDiamond:DrawDiamond(X,Y);
               msEllipse:DrawEllipse(X,Y);
              msTriangle:DrawTriangle(X,Y);
                  msLine:DrawLine(X,Y);
                  msText:DrawText(X,Y);
           end;
         end;
     if FShapeType=msText then UnprepareText;
end;

procedure Register;

begin
     RegisterComponents('Samples', [TMultiShape]);
end;

end.
