{----------------------------------------------------------------------

  TmProgressWnd - simple modal-like progress window for Delphi 2+

  Properties: Show, Hide, Min, Max, Position, Width
  Event handlers: OnPositionWindow

  Freeware by Ahto Tanner (ahto@moonsoftware.ee), Moon Software (http://www.moonsoftware.ee)
  You may not sell this component and include it in your commercial collection.

  Version 1.0, May 30, 1997
   - initial version

-----------------------------------------------------------------------}

unit mProgressForm;

interface

uses
  Windows, Classes, Controls, Forms, ComCtrls, StdCtrls;

type
   TfrmProgress = class(TForm)
      ProgressBar: TProgressBar;
      procedure FormClose(Sender: TObject; var Action: TCloseAction);
   private
      procedure CreateParams(var Params: TCreateParams); override;
   end;

   TWindowPositionEvent = procedure(var LeftPos, TopPos: LongInt) of object;

   TmProgressWnd = class(TComponent)
   private
      FShowing: boolean;
      FPosition: integer;
      FMax: integer;
      FMin: integer;
      //FStep: integer;
      FCaption: string;
      FWidth: integer;
      FOnWindowPosition: TWindowPositionEvent;

      procedure SetPosition(const NewPos: integer);
      procedure SetCaption(const NewCaption: string);
      function ReadHeight: integer;
      function CalculateCenterPos: TPoint;

   protected

   public
      constructor Create(AOwner: TComponent); override;
      // creates and shows progress form
      procedure Show;
      // closes and destroys form
      procedure Hide;
      // sets surrent progressbar position
      property Position: integer write SetPosition default 0;
      // read-only property to read form height
      // useful for setting form position in OnPositionWindow event handler
      property Height: integer read ReadHeight;

   published
      // min, max & step values of progressbar
      property Min: integer read FMin write FMin default 0;
      property Max: integer read FMax write FMax default 100;
      //property Step: integer read FStep write FStep default 10;
      // window caption, can be updated on runtime
      property Caption: string read FCaption write SetCaption;
      // window width
      property Width: integer read FWidth write FWidth;
      // event handler to use custom form position
      // default is centered to owner
      // values are in screen coordinates
      property OnPositionWindow: TWindowPositionEvent read FOnWindowPosition write FOnWindowPosition;

   end;

var
   frmProgress: TfrmProgress;
   FParentWnd: integer;

const
   // space on left, right, top, bottom of progress bar
   BORDER_OFFSET = 20;
   // default form width
   DEFAULT_WIDTH = 256;

procedure Register;

implementation

{$R *.DFM}

{---------------------------------------------------------------------------}

procedure TfrmProgress.CreateParams(var Params: TCreateParams);
begin
   inherited;
   with Params do begin
      WndParent := FParentWnd;
      Style := WS_DLGFRAME or WS_CAPTION;
      ExStyle := WS_EX_TOOLWINDOW;
   end;
end;

{---------------------------------------------------------------------------}

procedure TfrmProgress.FormClose(Sender: TObject;
  var Action: TCloseAction);
begin
   // prevent Alt+F4 closing ;-)
   Action := caNone;
end;

{---------------------------------------------------------------------------}
{-- mProgressWnd component implementation ----------------------------------}
{---------------------------------------------------------------------------}

constructor TmProgressWnd.Create(AOwner: TComponent);
begin
   inherited;
   FMin := 0;
   FMax := 100;
   //FStep := 10;
   FShowing := false;
   // potential problem here :)
   FParentWnd := TWinControl(AOwner).Handle;
   FCaption := 'Progress';
   FWidth := DEFAULT_WIDTH;
end;

{---------------------------------------------------------------------------}

procedure TmProgressWnd.Show;
var
   WindowPos: TPoint;
begin
   // if already showing, avoid another instance ;-)
   if FShowing then Exit;
   frmProgress := TfrmProgress.Create(Application);
   with frmProgress do begin
      Width := FWidth;
      ClientHeight := 2 * BORDER_OFFSET + ProgressBar.Height;
      ProgressBar.SetBounds(BORDER_OFFSET, BORDER_OFFSET, ClientWidth - 2 * BORDER_OFFSET, ProgressBar.Height);
      ProgressBar.Min := FMin;
      ProgressBar.Max := FMax;
      //ProgressBar.Step := FStep;
      Caption := FCaption;
      WindowPos := CalculateCenterPos;
      if Assigned(FOnWindowPosition) then
         FOnWindowPosition(WindowPos.X, WindowPos.Y);
      Left := WindowPos.X;
      Top := WindowPos.Y;
      Show;
   end;
   // disable input to owner window
   EnableWindow(FParentWnd, false);
   FShowing := true;
end;

{---------------------------------------------------------------------------}

procedure TmProgressWnd.Hide;
begin
   // if not showing, nothing to hide
   if not FShowing then Exit;
   EnableWindow(FParentWnd, true);
   frmProgress.Free;
   FShowing := false;
end;

{---------------------------------------------------------------------------}

function TmProgressWnd.CalculateCenterPos: TPoint;
var
   OwnerForm: TWinControl;
begin
   // potential problem here again ;)
   OwnerForm := Owner as TWinControl;
   Result.X := OwnerForm.Left + OwnerForm.Width div 2 - frmProgress.Width div 2;
   Result.Y := OwnerForm.Top + OwnerForm.Height div 2 - frmProgress.Height div 2;
end;

{---------------------------------------------------------------------------}

procedure TmProgressWnd.SetPosition(const NewPos: integer);
begin
   if FShowing then begin
      frmProgress.Progressbar.Position := NewPos;
      // remove ProcessMessages, if it seems to slow stuff down
      Application.ProcessMessages;
   end;
end;

{---------------------------------------------------------------------------}

function TmProgressWnd.ReadHeight: integer;
begin
   if frmProgress <> nil then
      Result := frmProgress.Height
   else
      Result := 0;
end;

{---------------------------------------------------------------------------}

procedure TmProgressWnd.SetCaption(const NewCaption: string);
begin
   if FShowing then
      begin
         frmProgress.Caption := NewCaption;
         Application.ProcessMessages;
      end
   else
      FCaption := NewCaption;
end;

{---------------------------------------------------------------------------}

procedure Register;
begin
   RegisterComponents('Moon', [TmProgressWnd]);
end;

{---------------------------------------------------------------------------}

end.
