// ..................................................................
//
//                          mdMailSlot
//
//              Copyright  1997 by Martin Djerns
//
// ..................................................................
// Initial Date : 2 April 1997 - MD
// Version 1.0  : 3 April 1997 - MD
// 8 April 1997 - MD :
// + SendBufToMailSlot added
// 13 April 1997 - MD :
// + WaitThread added for signaling that a message is arrived
// - TmdAutoMailSlot (substituted by TmdMailSlot)
//   ...not installed by default, but just remove the {} arund
//      the Class in the Register procedure in the buttom of the unit
//   ...The class still exists, but with a new source.
// 17 April 1997 - MD :
// + TmdCustomWinPopup class added
//   ...with a easy message event and a send function
// + TmdWinPopup class added
// Version 1.1 : 17 April 1997 - MD
// 19 Maj 1997 - MD
// + Sleep in MailSlotWaitThread (Thanks to Matt Behrens)
// Version 1.2 : 10 Juli 1997 - MD
// 28 July 1997 - MD
// % The paramerters in TwpMessage got new names - there were a naming
//   conflict (Thanks to Channing Corn for reminding me)
// Version 1.3 : 28 Juli 1997 - MD
// 23 September 1997 - MD
// - Two wait threads is one to many !
// 24 October 1997 - AO / MD
// + Added success return value in SendBufToMailSlot (Alexander Orlov)
// + Added safe load of value at startup (AO + MD)
// Version 1.4 : 24 October 1997 - MD
// + TmdCustomSecureMail + TmdSecureMail added (prevent duplicates
//   and guarantee is integrity with a CRC16
// + SendSecureMail function added
// ..................................................................

unit mdMailSlot;

interface

{$LONGSTRINGS ON}  // Equal {$H+}

uses
  ExtCtrls, Windows, SysUtils, Classes;

Const
  // Missing constant!
  MAILSLOT_WAIT_FOREVER = $FFFFFFFF; // DWORD(-1) ! Don't work :(

  // Error strings
  sCantOpen             = 'Cannot open mailslot';
  sNotActive            = 'Not active';
  sGetMailSlotInfoError = 'Error in GetMailSlotInfo';
  sNoMessages           = 'No messages waiting';
  sReadError            = 'Error reading message';
  sExists               = 'Mailslot allready exists';
  sBadName              = 'Bad mailslot name';

  // Default values
  msDefActive = False;
  msDefServer  = '.';
  msDefSlot = 'mdMailSlot1';
  msDefMaxSize = 0;

  wpSlot = 'messngr';

type
// .........................................................
//              Forward declerations
// .........................................................
  TmdCustomMailSlot = Class;

// .........................................................
//               Wait for MessageAvail thread
// .........................................................
  TmdMailSlotWaitThread = class(TThread)
  private
    FMailSlotHandle : THandle;
    FMailSlot : TmdCustomMailSlot;
  protected
    procedure Execute; override;
    Procedure SignalMsgReady;
    Procedure SignalError;
  Public
    Constructor Create(MailSlot : TmdCustomMailSlot);
  end;

// .........................................................
//                Component types
// .........................................................

  TmsMessageAvail = Procedure (Sender : TObject; Msg : String) of Object;
  TwpMessage = Procedure (Sender : TObject; AReciever, ASender, AMsg : String) Of Object;

// .........................................................
//       EXCEPTION :     EmdMailSlot
// .........................................................

  EmdMailSlot = Class(Exception);

// .........................................................
//                       TmdCustomMailSlot
// .........................................................

  TmdCustomMailSlot = class(TComponent)
  private          { Private declarations }

    // Property storage variables
    FLoadedActiveValue : Boolean; // Ver 1.4
    FActive : Boolean;
    FHandle : THandle;
    FMaxSize : DWord;
    FServer : String;
    FSlot : String;

    FWaiting : DWord;
    FNextSize : DWord;

    // Event storage variables
    FOpen : TNotifyEvent;
    FClose : TNotifyEvent;
    FMessageAvail : TmsMessageAvail;

    // Misc values
    FSlotChanged : Boolean;
    FWaitThread : TmdMailSlotWaitThread;

    // Property modification functions
    Procedure SetActive(Value : Boolean);
    Procedure SetSlot(Const Value : String);
    Procedure SetServer(Const Value : String);
    Procedure SetMaxSize(Value : DWord);

    Function GetWaiting : DWord;
    Function GetNextSize : DWord;
  protected        { Protected declarations }

    // Event process functions
    Procedure DoOpen; Virtual;
    Procedure DoClose; Virtual;
    Procedure DoMessageAvail(Const Msg : String); Virtual;

    // Special functions
    Procedure PerformError(Const Err : String); Virtual;
    Procedure UpdateSlotInfo;
  public  { Public declarations }
    // Default functions
    Constructor Create(AOwner : TComponent); Override;
    Procedure SetName(const NewName: TComponentName); Override;
    Procedure Loaded; Override; // Ver 1.4

    // Special functions
    Procedure Open; Virtual;
    Procedure Close; Virtual;
    Function ReadNext : String; Virtual;
    Procedure ReadMessage;

    // properties - Read Only
    Property Handle : THandle Read FHandle;
    Property Waiting : DWord Read GetWaiting;
    Property NextSize : DWord Read GetNextSize;
    // properties - Read/Write
    Property Active : Boolean Read FActive Write SetActive Default msDefActive;
    Property Server : String Read FServer Write SetServer;
    Property Slot : String Read FSlot Write SetSlot Stored FSlotChanged;
    Property MaxSize : DWord Read FMaxSize Write SetMaxSize Default msDefMaxSize;

    // Events
    Property OnOpen : TNotifyEvent Read FOpen Write FOpen;
    Property OnClose : TNotifyEvent Read FClose Write FClose;
    Property OnMessageAvail : TmsMessageAvail Read FMessageAvail Write FMessageAvail;
  published        { Published declarations }
  end;

// .........................................................
//                       TmdMailSlot
// .........................................................

  TmdMailSlot = class(TmdCustomMailSlot)
  Published
    // Properties
    Property Active;
    Property Server;
    Property Slot;
    Property MaxSize;
    // Events
    Property OnOpen;
    Property OnClose;
    Property OnMessageAvail;
  end;

// .........................................................
//                       TmdAutoMailSlot
// .........................................................
// Note : This class are created for automatically getting
//        a Delphi event when something comes into the
//        mailslot. While I not yet know about any way of
//        getting this information from Windows, have I
//        implemented a class which are using a TTimer
//        class for checking every second if anything is
//        arrived.
//        In case of that I find a bether way (registere a
//        mailslot change i Windows o.e.) I'll keep the
//        functions (interface) in this class as thay are,
//        and implement the function directly in
//        TmdCustomMailSlot. This class will then get the
//        functionallity from the base class, but it will
//        be backward compatible.
// .........................................................

  TmdAutoMailSlot = Class(TmdMailSlot);  // For backward compatibility

// Removed now when the TmdCustomMailSlot have a WaitThread
// build in!
(*  TmdAutoMailSlot = Class(TmdMailSlot)
  Private
    FTimer : TTimer;
    FMessageList : TStringList;
    FMessageAvail : TmsMessageAvail;
    Procedure TimerProc(Sender : TObject);
  Protected
    Procedure AddMessage(Const Msg : String);
    Function GetMessage : String;
    Procedure DoMessageAvail(Const Msg : String); Virtual;
  Public
    Constructor Create(AOwner : TComponent); Override;
    Procedure Open; Override;
    Procedure Close; Override;
  Published
    Property OnMessageAvail : TmsMessageAvail Read FMessageAvail Write FMessageAvail;
  end;
*)

// .........................................................
//                       TmdCustomWinPopup
// .........................................................
  TmdCustomWinPopup = Class(TmdCustomMailSlot)
  Private
    FMessage : TwpMessage;
  Protected
    Procedure DoMessageAvail(Const Msg : String); Override;
  Public
    Constructor Create(AOwner : TComponent); Override;
    Function Send(AServer, ASender, AReciever, AMsg : String) : Boolean;
    Property Slot : String Read FSlot; // Should be read only...
    Property OnMessage : TwpMessage Read FMessage Write FMessage;
  end;

// .........................................................
//                       TmdWinPopup
// .........................................................
  TmdWinPopup = Class(TmdCustomWinPopup)
  Published
    // Properties
    Property Active;     // TmdCustomMailSlot
    Property MaxSize;    // TmdCustomMailSlot
    // Events
    Property OnOpen;     // TmdCustomMailSlot
    Property OnClose;    // TmdCustomMailSlot
    Property OnMessage;  // TmdCustomWinPopup
  end;

// .........................................................
//                     TmdCustomSecureMail
// .........................................................
  TmdCustomSecureMail = Class(TmdCustomMailSlot)
  Private
    FError : TmsMessageAvail;
    FDuplicated : TNotifyEvent;
    FLastMessage : PChar;
  Protected
    Procedure DoMessageAvail(Const Msg : String); Override;
    Procedure DoError(Const Msg : String); Virtual;
    Procedure DoDuplicated; Virtual;
  Public
    Property OnError : TmsMessageAvail Read FError Write FError;
    Property OnDuplicated : TNotifyEvent Read FDuplicated Write FDuplicated;
  end;

// .........................................................
//                     TmdSecureMail
// .........................................................
  TmdSecureMail = Class(TmdCustomSecureMail)
  Published
    // Properties
    Property Active;
    Property Server;
    Property Slot;
    Property MaxSize;
    // Events
    Property OnOpen;
    Property OnClose;
    Property OnMessageAvail;
    Property OnError;
    Property OnDuplicated;
  end;

// .........................................................
//                       Functions
// .........................................................

// Send a single message to a mailslot
// If Server is "." then the current machine are used
// If Server is "*" then the current domain are used
Function SendToMailSlot(Const Server, Slot, Mail : String) : Boolean;
Function SendBufToMailSlot(Const Server, Slot : String; Data : PChar; Length : Integer) : Boolean;

// Send with preample for preventing duplicate messages, and
// a CRC16 for preventing errors in the data.
Function SendSecureMail(Const Server, Slot : String; Mail : String) : Boolean;
Function SendBufSecure(Const Server, Slot : String; Data : PChar; Length : Integer) : Boolean;

procedure Register;

implementation


// .........................................................
//                        CRC16
// Actually this is a unit by me, but for only having to
// have one file I copied it in here.
// .........................................................

Const
  IntegerCRCTable     : Array[0..255] of Word =
    ($0000, $C0C1, $C181, $0140, $C301, $03C0, $0280, $C241,
     $C601, $06C0, $0780, $C741, $0500, $C5C1, $C481, $0440,
     $CC01, $0CC0, $0080, $CD41, $0F00, $CFC1, $CE81, $0E40,
     $0A00, $CAC1, $C881, $0B40, $C901, $09C0, $0880, $C841,
     $D801, $18C0, $1980, $D941, $1B00, $DBC1, $DA81, $1A40,
     $1E00, $DEC1, $DF81, $1F40, $DD01, $1DC0, $1C80, $DC41,
     $1400, $D4C1, $D581, $1540, $D701, $17C0, $1680, $0641,
     $D201, $12C0, $1380, $D341, $1100, $D1C1, $D081, $1040,
     $F001, $30C0, $3180, $F141, $3300, $F3C1, $F281, $3240,
     $3600, $F6C1, $F781, $3740, $F501, $35C0, $3480, $F441,
     $3C00, $FCC1, $FD81, $3D40, $FF01, $3FC0, $3E80, $FE41,
     $FA01, $3AC0, $3B80, $FB41, $3900, $F9C1, $F881, $3840,
     $2800, $E8C1, $E981, $2940, $EB01, $2BC0, $2A80, $EA41,
     $EE01, $2EC0, $2F80, $EF41, $2D00, $EDC1, $EC81, $2C40,
     $E401, $24C0, $2580, $E541, $2700, $E7C1, $E681, $2640,
     $2200, $E2C1, $E381, $2340, $E101, $21C0, $2080, $E041,
     $A001, $60C0, $6180, $A141, $6300, $A3C1, $A281, $6240,
     $6600, $A6C1, $A781, $6740, $A501, $65C0, $6480, $A441,
     $6C00, $ACC1, $AD81, $6D40, $AF01, $6FC0, $6E80, $AE41,
     $AA01, $6AC0, $6B80, $AB41, $6900, $A9C1, $A881, $6840,
     $7800, $B8C1, $B981, $7940, $BB01, $7BC0, $7A80, $BA41,
     $BE01, $7EC0, $7F80, $BF41, $7D00, $BDC1, $BC81, $7C40,
     $B401, $74C0, $7580, $B541, $7700, $B7C1, $B681, $7640,
     $7200, $B2C1, $B381, $7340, $B101, $71C0, $7080, $B041,
     $5000, $90C1, $9181, $5140, $9301, $53C0, $5280, $9241,
     $9601, $56C0, $5780, $9741, $5500, $95C1, $9481, $5440,
     $9C01, $5CC0, $5D80, $9041, $5F00, $9FC1, $9E81, $5E40,
     $5A00, $9AC1, $9B81, $5B40, $9901, $59C0, $5880, $9841,
     $8801, $48C0, $4980, $8941, $4B00, $8BC1, $8A81, $4A40,
     $4E00, $8EC1, $8F81, $4F40, $8D01, $4DC0, $4C80, $8C41,
     $4400, $84C1, $8581, $4540, $8701, $47C0, $4680, $8641,
     $8201, $42C0, $4380, $8341, $4100, $81C1, $8081, $4040);

Type
  CRCRecord = Record
    Low         : Byte;
    High        : Byte;
  end;

Var
  CRCTable : Array[0..255] of CRCRecord absolute IntegerCRCTable;

Function CalcCRC16(Const S;Const L : Integer) : Word;
Var
  i, j  : Integer;
  CRC   : CRCRecord;
Begin
  CRC.High := 0;
  CRC.Low := 0;
  For i := 0 to L-1 do
  Begin
    j := Ord(PChar(S)[i]) XOR CRC.Low;
    CRC.Low := CRCTable[j].Low XOR CRC.High;
    CRC.High := CRCTable[j].High;
  end;
  CalcCRC16 := (CRC.High SHL 8) or CRC.Low;
end;

// .........................................................
//                      SendToMailSlot
// .........................................................

Function OpenMailSlot(Const Server, Slot : String): THandle;
Var
  FullSlot : String;
Begin
  FullSlot := '\\'+Server+'\mailslot\'+Slot;  // MailSlot string
  Result := CreateFile(
    PChar(FullSlot),
    GENERIC_WRITE,
    FILE_SHARE_READ,
    NIL,
    OPEN_EXISTING,
    FILE_ATTRIBUTE_NORMAL,
    -1                    );
end;

Function SendToMailSlot(Const Server, Slot, Mail : String) : Boolean;
Var
  hToSlot : THandle;
  BytesWritten : DWord;
Begin
  Result := False; // Default value
  hToSlot := OpenMailSlot(Server,Slot);
  If hToSlot = INVALID_HANDLE_VALUE Then
    Exit; // Error
  Try
    BytesWritten := 0;
    If (NOT WriteFile(hToSlot,
                      Pointer(Mail)^,
                      Length(Mail),
                      BytesWritten,
                      NIL))         OR
        (BytesWritten <> Length(Mail)) Then
      Exit; // Error

    Result := True;  // Everthing went good
  Finally
    CloseHandle(hToSlot);
  end;
end;

Function SendBufToMailSlot(Const Server, Slot : String; Data : PChar; Length : Integer) : Boolean;
Var
  hToSlot : THandle;
  BytesWritten : Integer;
Begin
  Result := False;
  hToSlot := OpenMailSlot(Server,Slot);
  If hToSlot = INVALID_HANDLE_VALUE Then
    Exit; // Error
  Try
    BytesWritten := 0;
    If (NOT WriteFile(hToSlot,
                      Pointer(Data)^,
                      Length,
                      BytesWritten,
                      NIL))         OR
        (BytesWritten <> Length) Then
      Exit; // Error
    Result := True; // Alexander Orlov - ver 1.4
  Finally
    CloseHandle(hToSlot);
  end;
end;

Function SendSecureMail(Const Server, Slot : String; Mail : String) : Boolean;
Begin
  Mail := IntToHex(GetTickCount MOD 10000,4)+Mail;
  Mail := Mail + IntToHex(CalcCRC16(Mail,Length(Mail)),4);
  Result := SendToMailSlot(Server, Slot, Mail);
end;

Function SendBufSecure(Const Server, Slot : String; Data : PChar; Length : Integer) : Boolean;
Var
  TempData : PChar;
Begin
  TempData := StrAlloc(Length+9);
  Try
    StrCopy(TempData,PChar(IntToHex(GetTickCount MOD 10000,4)));
    StrCat(TempData,Data);
    StrCat(TempData,PChar(IntToHex(CalcCRC16(TempData,Length+4),4)));
    Result := SendBufToMailSlot(Server, Slot, TempData, Length+8);
  Finally
    StrDispose(TempData);
  end;
end;

// .........................................................
//               Wait for MessageAvail thread
// .........................................................

Constructor TmdMailSlotWaitThread.Create(MailSlot : TmdCustomMailSlot);
Begin
  Inherited Create(False);
//  Priority := tpLowest;
  Priority := tpIdle;
  FMailSlot := MailSlot;
end;

Procedure TmdMailSlotWaitThread.Execute;
Var
  Waiting,
  NextSize : DWord;
Begin
  While Not Terminated do
  Begin
    GetMailSlotInfo(FMailSlot.Handle,NIL, NextSize, @Waiting, NIL);
    If Waiting > 0 Then
      Synchronize(SignalMsgReady);
    Sleep(1);  // Suggested by Matt Behrens.
  end;
end;

Procedure TmdMailSlotWaitThread.SignalMsgReady;
Begin
  FMailSlot.ReadMessage;
end;

Procedure TmdMailSlotWaitThread.SignalError;
Begin
end;

// .........................................................
//                    TmdWaitThread
// .........................................................
(*
Type
  TmdWaitCall = Procedure of object;
  TmdWaitThread = Class(TThread)
  Protected
    FPath : String;
    FChanged : TmdWaitCall;
  Public
    Constructor Create(Const Path : String; Changed : TmdWaitCall);
    Procedure Execute; Override;
  end;

Constructor TmdWaitThread.Create(Const Path : String; Changed : TmdWaitCall);
Begin
  Inherited Create(False);
  FPath := Path;
  FChanged := Changed;
  FreeOnTerminate := True;
end;

Procedure TmdWaitThread.Execute;
Var
  Handle : THandle;
Begin
  Handle := FindFirstChangeNotification(PChar(FPath),False,
                                    FILE_NOTIFY_CHANGE_SIZE OR
                                    FILE_NOTIFY_CHANGE_LAST_WRITE );
  If Handle <> INVALID_HANDLE_VALUE Then
  Begin
    If (WaitForSingleObject(Handle,Infinite) <> WAIT_FAILED) AND
       Assigned(FChanged) Then
      FChanged;
    FindCloseChangeNotification(Handle);
  end;
end;
  *)
// .........................................................
//                    TmdCustomMailSlot
// .........................................................

// Default functions

Constructor TmdCustomMailSlot.Create(AOwner : TComponent);
Begin
  Inherited Create(AOwner);
  FWaitThread := NIL;
  FSlotChanged := False;
  FHandle := INVALID_HANDLE_VALUE;
  FSlot := msDefSlot;
  FServer := msDefServer;
  FMaxSize := msDefMaxSize;

  If msDefActive Then
    Open
  else
    Close;
end;

Procedure TmdCustomMailSlot.SetName(const NewName: TComponentName);
Begin
  Inherited SetName(NewName);
  If NOT FSlotChanged Then
    FSlot := NewName;
end;

Procedure TmdCustomMailSlot.Loaded; // Ver 1.4
Begin
  Inherited Loaded;
  SetActive(FLoadedActiveValue);
end;

// Special functions

Procedure TmdCustomMailSlot.PerformError(Const Err : String);
Begin
  Raise EmdMailSlot.Create(Err);
end;

Procedure TmdCustomMailSlot.Open;
Var
  MailSlotName : String;
Begin
  If Active Then
    Close;
  MailSlotName := '\\'+FServer+'\mailslot\'+FSlot;
  If NOT (csDesigning IN ComponentState) Then
  Begin
    FHandle := CreateMailSlot(PChar(MailSlotName),
                            MaxSize,
                            MAILSLOT_WAIT_FOREVER,
                            NIL);
    FActive := Handle <> INVALID_HANDLE_VALUE;
    If NOT Active Then
    Begin
      Case GetLastError of
        ERROR_ALREADY_EXISTS : PerformError(sExists);
        ERROR_BAD_PATHNAME   : PerformError(sBadName);
      else
        PerformError(sCantOpen);
      end;
    end
    else
    Begin
      DoOpen;
      If FWaitThread = NIL Then
        FWaitThread := TmdMailSlotWaitThread.Create(Self);
    end;
  end
  else
    FActive := True;
end;

Procedure TmdCustomMailSlot.Close;
Begin
  If Active Then
  Begin
    If FWaitThread <> NIL Then
    Begin
      FWaitThread.Terminate;
      FWaitThread := NIL;
    end;

    CloseHandle(FHandle);
    FHandle := INVALID_HANDLE_VALUE;
    FActive := False;
    DoClose;
  end;
end;

Procedure TmdCustomMailSlot.UpdateSlotInfo;
Begin
  If NOT Active Then
    PerformError(sNotActive);

  If NOT GetMailSlotInfo(Handle,NIL, FNextSize, @FWaiting, NIL) Then
    PerformError(sGetMailSlotInfoError);
end;

Function TmdCustomMailSlot.ReadNext : String;
Var
  ReadSize : DWord;
Begin
  UpdateSlotInfo;
  If FWaiting = 0 Then
    PerformError(sNoMessages);
  SetLength(Result,FNextSize);
  ReadFile(Handle,PChar(Result)^,FNextSize,ReadSize,NIL);
  If FNextSize <> ReadSize Then
    PerformError(sReadError);
end;

Procedure TmdCustomMailSlot.ReadMessage;
Begin
  DoMessageAvail(ReadNext);
end;

// Property modification functions

Procedure TmdCustomMailSlot.SetActive(Value : Boolean);
Begin
  If csReading IN ComponentState Then // Ver 1.4
  Begin
    FLoadedActiveValue := Value;
    Exit;
  end;

  FActive := Value;
  If Value Then
    Open
  else
    Close;
end;

Procedure TmdCustomMailSlot.SetServer(Const Value : String);
Begin
  If Value = FServer Then
    Exit;
  FServer := Value;
  If Active Then
    Open;
end;

Procedure TmdCustomMailSlot.SetSlot(Const Value : String);
Begin
  If Value = FSlot Then
    Exit;
  FSlotChanged := True;
  FSlot := Value;
  If Active Then
    Open;
end;

Procedure TmdCustomMailSlot.SetMaxSize(Value : DWord);
Begin
  If Value = FMaxSize Then
    Exit;
  FMaxSize := Value;
  If Active  Then
    Open;
end;

Function TmdCustomMailSlot.GetWaiting : DWord;
Begin
  Result := 0;
  // I want that it is possible to check on
  // waiting without testing Active first
  If NOT Active Then
    Exit;
  UpdateSlotInfo;
  Result := FWaiting;
end;

Function TmdCustomMailSlot.GetNextSize : DWord;
Begin
  UpdateSlotInfo;
  Result := FNextSize;
end;

// Event process functions

Procedure TmdCustomMailSlot.DoOpen;
Begin
  If Assigned(FOpen) Then
    FOpen(Self);
end;

Procedure TmdCustomMailSlot.DoClose;
Begin
  If Assigned(FClose) Then
    FClose(Self);
end;

Procedure TmdCustomMailSlot.DoMessageAvail(Const Msg : String);
Begin
  If Assigned(FMessageAvail) Then
    FMessageAvail(Self, Msg);
end;

// .........................................................
//                     TmdAutoMailSlot
// .........................................................
(*
Constructor TmdAutoMailSlot.Create(AOwner : TComponent);
Begin
  FTimer := NIL;
  FMessageList := NIL;
  Inherited Create(AOwner);
end;

Procedure TmdAutoMailSlot.AddMessage(Const Msg : String);
Begin
  If FMessageList = NIL Then
    FMessageList := TStringList.Create;
  FMessageList.Add(Msg);
end;

Function TmdAutoMailSlot.GetMessage : String;
Begin
  Result := '';
  If FMessageList <> NIL Then
  Begin
    If FMessageList.Count > 0 Then
    Begin
      Result := FMessageList[0];
      FMessageList.Delete(0);
    end;
    If FMessageList.Count = 0 Then
    Begin
      FMessageList.Free;
      FMessageList := NIL;
    end;
  end;
end;

Procedure TmdAutoMailSlot.Open;
Begin
  Inherited Open;
  If Active AND (NOT (csDesigning IN ComponentState))Then
  Begin
    If FTimer = NIL Then
      FTimer := TTimer.Create(Self);
    FTimer.Interval := 1000;
    FTimer.OnTimer := TimerProc;
  end;
end;

Procedure TmdAutoMailSlot.Close;
Begin
  If FTimer <> NIL Then
    FTimer.Free;
  FTimer := NIL;
  Inherited Close;
end;

Procedure TmdAutoMailSlot.TimerProc(Sender : TObject);
Begin
  While Waiting > 0 do
    AddMessage(ReadNext);

  If (FMessageList <> NIL) AND (FMessageList.Count > 0) Then
    DoMessageAvail(GetMessage);
end;

Procedure TmdAutoMailSlot.DoMessageAvail(Const Msg : String);
Begin
  If Assigned(FMessageAvail) Then
    FMessageAvail(Self, Msg);
end;
*)

// .........................................................
//                        TmdCustomWinPopup
// .........................................................

// Overriden functions

Constructor TmdCustomWinPopup.Create(AOwner : TComponent);
Begin
  Inherited Create(AOwner);
  Inherited Slot := wpSlot;
end;

Procedure TmdCustomWinPopup.DoMessageAvail(Const Msg : String);
var
  P : PChar;
  ASender, AReciever, AMessage : String;
Begin
  If Assigned(FMessage) Then
  Begin
    P := PChar(Msg);
    OemToChar(P,P);
    ASender := StrPas(P);
    While NOT (P[0] = #0) do
      Inc(P);
    Inc(P);
    OemToChar(P,P);
    AReciever := StrPas(P);
    While NOT (P[0] = #0) do
      Inc(P);
    Inc(P);
    OemToChar(P,P);
    AMessage := StrPas(P);
    FMessage(Self, AReciever, ASender, AMessage);
  end;
end;

// New functions

Function TmdCustomWinPopup.Send(AServer, ASender, AReciever, AMsg : String):Boolean;
Begin
  CharToOEM(PChar(AServer), PChar(AServer));
  CharToOEM(PChar(ASender),PChar(ASender));
  CharToOEM(PChar(AReciever),PChar(AReciever));
  CharToOEM(PChar(AMsg),PChar(AMsg));
  Result := SendToMailSlot(AServer, wpSlot, ASender+#0+AReciever+#0+AMsg);
end;

// .........................................................
//                       TmdCustomSecureMail
// .........................................................

Procedure TmdCustomSecureMail.DoMessageAvail(Const Msg : String);
Var
  L : Integer;
  S,C,
  SubStr : String;
Begin
  L := Length(Msg);
  SubStr := Copy(Msg,1,L-4);
  S := Copy(Msg,L-3,4);
  C := IntTohex(CalcCRC16(SubStr,L-4),4);
  If S <> C Then
    DoError(Msg)
  else  // OK !
  Begin
    If (FLastMessage <> NIL) AND
       (StrComp(FLastMessage,PChar(Msg)) = 0) AND
       (Copy(String(FLastMessage),1,4) = Copy(Msg,1,4)) Then
    Begin
      // Inform about this mail is ignores since it is a duplicate
      DoDuplicated;
    end
    else
    Begin
      If FLastMessage <> NIL Then
        StrDispose(FLastMessage);
      FLastMessage := StrNew(PChar(Msg));
      Inherited DoMessageAvail(Copy(Msg,5,L-8));
    end;
  end;
end;

Procedure TmdCustomSecureMail.DoError(Const Msg : String);
Begin
  If Assigned(FError) Then
    FError(Self,Msg);
end;

Procedure TmdCustomSecureMail.DoDuplicated;
Begin
  If Assigned(FDuplicated) Then
    FDuplicated(Self);
end;

// .........................................................
//                        Register
// .........................................................

procedure Register;
begin
  RegisterComponents('mdVCL', [TmdMailSlot{,
                               TmdAutoMailSlot},
                               TmdWinPopup,
                               TmdSecureMail]);
end;

end.
