// ..................................................................
//
//                          mdMailSlot
//
//              Copyright  1997 by Martin Djerns
//
// ..................................................................
// Initial Date : 2 April 1997 - MD
// Version 1.0  : 3 April 1997 - MD
// 8 April 1997 - MD :
// + SendBufToMailSlot added
// 13 April 1997 - MD :
// + WaitThread added for signaling that a message is arrived
// - TmdAutoMailSlot (substituted by TmdMailSlot)
//   ...not installed by default, but just remove the {} arund
//      the Class in the Register procedure in the buttom of the unit
//   ...The class still exists, but with a new source.
// 17 April 1997 - MD :
// + TmdCustomWinPopup class added
//   ...with a easy message event and a send function
// + TmdWinPopup class added
// Version 1.1 : 17 April 1997 - MD
// 19 Maj 1997 - MD
// + Sleep in MailSlotWaitThread (Thanks to Matt Behrens)
// Version 1.2 : 10 Juli 1997 - MD
// 28 July 1997 - MD
// % The paramerters in TwpMessage got new names - there were a naming
//   conflict (Thanks to Channing Corn for reminding me)
// Version 1.3 : 28 Juli 1997 - MD
// 23 September 1997 - MD
// - Two wait threads is one to many !
// 24 October 1997 - AO / MD
// + Added success return value in SendBufToMailSlot (Alexander Orlov)
// + Added safe load of value at startup (AO + MD)
// ..................................................................

unit mdMailSlot;

interface

{$LONGSTRINGS ON}  // Equal {$H+}

uses
  ExtCtrls, Windows, SysUtils, Classes;

Const
  // Missing constant!
  MAILSLOT_WAIT_FOREVER = $FFFFFFFF; // DWORD(-1) ! Don't work :(

  // Error strings
  sCantOpen             = 'Cannot open mailslot';
  sNotActive            = 'Not active';
  sGetMailSlotInfoError = 'Error in GetMailSlotInfo';
  sNoMessages           = 'No messages waiting';
  sReadError            = 'Error reading message';
  sExists               = 'Mailslot allready exists';
  sBadName              = 'Bad mailslot name';

  // Default values
  msDefActive = False;
  msDefServer  = '.';
  msDefSlot = 'mdMailSlot1';
  msDefMaxSize = 0;

  wpSlot = 'messngr';

type
// .........................................................
//              Forward declerations
// .........................................................
  TmdCustomMailSlot = Class;

// .........................................................
//               Wait for MessageAvail thread
// .........................................................
  TmdMailSlotWaitThread = class(TThread)
  private
    FMailSlotHandle : THandle;
    FMailSlot : TmdCustomMailSlot;
  protected
    procedure Execute; override;
    Procedure SignalMsgReady;
    Procedure SignalError;
  Public
    Constructor Create(MailSlot : TmdCustomMailSlot);
  end;

// .........................................................
//                Component types
// .........................................................

  TmsMessageAvail = Procedure (Sender : TObject; Msg : String) of Object;
  TwpMessage = Procedure (Sender : TObject; AReciever, ASender, AMsg : String) Of Object;

// .........................................................
//       EXCEPTION :     EmdMailSlot
// .........................................................

  EmdMailSlot = Class(Exception);

// .........................................................
//                       TmdCustomMailSlot
// .........................................................

  TmdCustomMailSlot = class(TComponent)
  private          { Private declarations }

    // Property storage variables
    FLoadedActiveValue : Boolean; // Ver 1.4
    FActive : Boolean;
    FHandle : THandle;
    FMaxSize : DWord;
    FServer : String;
    FSlot : String;

    FWaiting : DWord;
    FNextSize : DWord;

    // Event storage variables
    FOpen : TNotifyEvent;
    FClose : TNotifyEvent;
    FMessageAvail : TmsMessageAvail;

    // Misc values
    FSlotChanged : Boolean;
    FWaitThread : TmdMailSlotWaitThread;

    // Property modification functions
    Procedure SetActive(Value : Boolean);
    Procedure SetSlot(Const Value : String);
    Procedure SetServer(Const Value : String);
    Procedure SetMaxSize(Value : DWord);

    Function GetWaiting : DWord;
    Function GetNextSize : DWord;
  protected        { Protected declarations }

    // Event process functions
    Procedure DoOpen; Virtual;
    Procedure DoClose; Virtual;
    Procedure DoMessageAvail(Const Msg : String); Virtual;

    // Special functions
    Procedure PerformError(Const Err : String); Virtual;
    Procedure UpdateSlotInfo;
  public  { Public declarations }
    // Default functions
    Constructor Create(AOwner : TComponent); Override;
    Procedure SetName(const NewName: TComponentName); Override;
    Procedure Loaded; Override; // Ver 1.4

    // Special functions
    Procedure Open; Virtual;
    Procedure Close; Virtual;
    Function ReadNext : String; Virtual;
    Procedure ReadMessage;

    // properties - Read Only
    Property Handle : THandle Read FHandle;
    Property Waiting : DWord Read GetWaiting;
    Property NextSize : DWord Read GetNextSize;
    // properties - Read/Write
    Property Active : Boolean Read FActive Write SetActive Default msDefActive;
    Property Server : String Read FServer Write SetServer;
    Property Slot : String Read FSlot Write SetSlot Stored FSlotChanged;
    Property MaxSize : DWord Read FMaxSize Write SetMaxSize Default msDefMaxSize;

    // Events
    Property OnOpen : TNotifyEvent Read FOpen Write FOpen;
    Property OnClose : TNotifyEvent Read FClose Write FClose;
    Property OnMessageAvail : TmsMessageAvail Read FMessageAvail Write FMessageAvail;
  published        { Published declarations }
  end;

// .........................................................
//                       TmdMailSlot
// .........................................................

  TmdMailSlot = class(TmdCustomMailSlot)
  Published
    // Properties
    Property Active;
    Property Server;
    Property Slot;
    Property MaxSize;
    // Events
    Property OnOpen;
    Property OnClose;
    Property OnMessageAvail;
  end;

// .........................................................
//                       TmdAutoMailSlot
// .........................................................
// Note : This class are created for automatically getting
//        a Delphi event when something comes into the
//        mailslot. While I not yet know about any way of
//        getting this information from Windows, have I
//        implemented a class which are using a TTimer
//        class for checking every second if anything is
//        arrived.
//        In case of that I find a bether way (registere a
//        mailslot change i Windows o.e.) I'll keep the
//        functions (interface) in this class as thay are,
//        and implement the function directly in
//        TmdCustomMailSlot. This class will then get the
//        functionallity from the base class, but it will
//        be backward compatible.
// .........................................................

  TmdAutoMailSlot = Class(TmdMailSlot);  // For backward compatibility

// Removed now when the TmdCustomMailSlot have a WaitThread
// build in!
(*  TmdAutoMailSlot = Class(TmdMailSlot)
  Private
    FTimer : TTimer;
    FMessageList : TStringList;
    FMessageAvail : TmsMessageAvail;
    Procedure TimerProc(Sender : TObject);
  Protected
    Procedure AddMessage(Const Msg : String);
    Function GetMessage : String;
    Procedure DoMessageAvail(Const Msg : String); Virtual;
  Public
    Constructor Create(AOwner : TComponent); Override;
    Procedure Open; Override;
    Procedure Close; Override;
  Published
    Property OnMessageAvail : TmsMessageAvail Read FMessageAvail Write FMessageAvail;
  end;
*)

// .........................................................
//                       TmdCustomWinPopup
// .........................................................
  TmdCustomWinPopup = Class(TmdCustomMailSlot)
  Private
    FMessage : TwpMessage;
  Protected
    Procedure DoMessageAvail(Const Msg : String); Override;
  Public
    Constructor Create(AOwner : TComponent); Override;
    Function Send(AServer, ASender, AReciever, AMsg : String) : Boolean;
    Property Slot : String Read FSlot; // Should be read only...
    Property OnMessage : TwpMessage Read FMessage Write FMessage;
  end;

// .........................................................
//                       TmdWinPopup
// .........................................................
  TmdWinPopup = Class(TmdCustomWinPopup)
  Published
    // Properties
    Property Active;     // TmdCustomMailSlot
    Property MaxSize;    // TmdCustomMailSlot
    // Events
    Property OnOpen;     // TmdCustomMailSlot
    Property OnClose;    // TmdCustomMailSlot
    Property OnMessage;  // TmdCustomWinPopup
  end;

// .........................................................
//                       Functions
// .........................................................

// Send a single message to a mailslot
// If Server is "." then the current machine are used
// If Server is "*" then the current domain are used
Function SendToMailSlot(Const Server, Slot, Mail : String) : Boolean;
Function SendBufToMailSlot(Const Server, Slot : String; Data : PChar; Length : Integer) : Boolean;

procedure Register;

implementation

// .........................................................
//                      SendToMailSlot
// .........................................................

Function OpenMailSlot(Const Server, Slot : String): THandle;
Var
  FullSlot : String;
Begin
  FullSlot := '\\'+Server+'\mailslot\'+Slot;  // MailSlot string
  Result := CreateFile(
    PChar(FullSlot),
    GENERIC_WRITE,
    FILE_SHARE_READ,
    NIL,
    OPEN_EXISTING,
    FILE_ATTRIBUTE_NORMAL,
    -1                    );
end;

Function SendToMailSlot(Const Server, Slot, Mail : String) : Boolean;
Var
  hToSlot : THandle;
  BytesWritten : DWord;
Begin
  Result := False; // Default value
  hToSlot := OpenMailSlot(Server,Slot);
  If hToSlot = INVALID_HANDLE_VALUE Then
    Exit; // Error
  Try
    BytesWritten := 0;
    If (NOT WriteFile(hToSlot,
                      Pointer(Mail)^,
                      Length(Mail),
                      BytesWritten,
                      NIL))         OR
        (BytesWritten <> Length(Mail)) Then
      Exit; // Error

    Result := True;  // Everthing went good
  Finally
    CloseHandle(hToSlot);
  end;
end;

Function SendBufToMailSlot(Const Server, Slot : String; Data : PChar; Length : Integer) : Boolean;
Var
  hToSlot : THandle;
  BytesWritten : Integer;
Begin
  Result := False;
  hToSlot := OpenMailSlot(Server,Slot);
  If hToSlot = INVALID_HANDLE_VALUE Then
    Exit; // Error
  Try
    BytesWritten := 0;
    If (NOT WriteFile(hToSlot,
                      Pointer(Data)^,
                      Length,
                      BytesWritten,
                      NIL))         OR
        (BytesWritten <> Length) Then
      Exit; // Error
    Result := True; // Alexander Orlov - ver 1.4
  Finally
    CloseHandle(hToSlot);
  end;
end;

// .........................................................
//               Wait for MessageAvail thread
// .........................................................

Constructor TmdMailSlotWaitThread.Create(MailSlot : TmdCustomMailSlot);
Begin
  Inherited Create(False);
//  Priority := tpLowest;
  Priority := tpIdle;
  FMailSlot := MailSlot;
end;

Procedure TmdMailSlotWaitThread.Execute;
Var
  Waiting,
  NextSize : DWord;
Begin
  While Not Terminated do
  Begin
    GetMailSlotInfo(FMailSlot.Handle,NIL, NextSize, @Waiting, NIL);
    If Waiting > 0 Then
      Synchronize(SignalMsgReady);
    Sleep(1);  // Suggested by Matt Behrens.
  end;
end;

Procedure TmdMailSlotWaitThread.SignalMsgReady;
Begin
  FMailSlot.ReadMessage;
end;

Procedure TmdMailSlotWaitThread.SignalError;
Begin
end;

// .........................................................
//                    TmdWaitThread
// .........................................................
(*
Type
  TmdWaitCall = Procedure of object;
  TmdWaitThread = Class(TThread)
  Protected
    FPath : String;
    FChanged : TmdWaitCall;
  Public
    Constructor Create(Const Path : String; Changed : TmdWaitCall);
    Procedure Execute; Override;
  end;

Constructor TmdWaitThread.Create(Const Path : String; Changed : TmdWaitCall);
Begin
  Inherited Create(False);
  FPath := Path;
  FChanged := Changed;
  FreeOnTerminate := True;
end;

Procedure TmdWaitThread.Execute;
Var
  Handle : THandle;
Begin
  Handle := FindFirstChangeNotification(PChar(FPath),False,
                                    FILE_NOTIFY_CHANGE_SIZE OR
                                    FILE_NOTIFY_CHANGE_LAST_WRITE );
  If Handle <> INVALID_HANDLE_VALUE Then
  Begin
    If (WaitForSingleObject(Handle,Infinite) <> WAIT_FAILED) AND
       Assigned(FChanged) Then
      FChanged;
    FindCloseChangeNotification(Handle);
  end;
end;
  *)
// .........................................................
//                    TmdCustomMailSlot
// .........................................................

// Default functions

Constructor TmdCustomMailSlot.Create(AOwner : TComponent);
Begin
  Inherited Create(AOwner);
  FWaitThread := NIL;
  FSlotChanged := False;
  FHandle := INVALID_HANDLE_VALUE;
  FSlot := msDefSlot;
  FServer := msDefServer;
  FMaxSize := msDefMaxSize;

  If msDefActive Then
    Open
  else
    Close;
end;

Procedure TmdCustomMailSlot.SetName(const NewName: TComponentName);
Begin
  Inherited SetName(NewName);
  If NOT FSlotChanged Then
    FSlot := NewName;
end;

Procedure TmdCustomMailSlot.Loaded; // Ver 1.4
Begin
  Inherited Loaded;
  SetActive(FLoadedActiveValue);
end;

// Special functions

Procedure TmdCustomMailSlot.PerformError(Const Err : String);
Begin
  Raise EmdMailSlot.Create(Err);
end;

Procedure TmdCustomMailSlot.Open;
Var
  MailSlotName : String;
  Buffer : String;
  ReadSize : DWord;
Begin
  If Active Then
    Close;
  MailSlotName := '\\'+FServer+'\mailslot\'+FSlot;
  If NOT (csDesigning IN ComponentState) Then
  Begin
    FHandle := CreateMailSlot(PChar(MailSlotName),
                            MaxSize,
                            MAILSLOT_WAIT_FOREVER,
                            NIL);
    FActive := Handle <> INVALID_HANDLE_VALUE;
    If NOT Active Then
    Begin
      Case GetLastError of
        ERROR_ALREADY_EXISTS : PerformError(sExists);
        ERROR_BAD_PATHNAME   : PerformError(sBadName);
      else
        PerformError(sCantOpen);
      end;
    end
    else
    Begin
      DoOpen;
      If FWaitThread = NIL Then
        FWaitThread := TmdMailSlotWaitThread.Create(Self);
    end;
  end
  else
    FActive := True;
end;

Procedure TmdCustomMailSlot.Close;
Begin
  If Active Then
  Begin
    If FWaitThread <> NIL Then
    Begin
      FWaitThread.Terminate;
      FWaitThread := NIL;
    end;

    CloseHandle(FHandle);
    FHandle := INVALID_HANDLE_VALUE;
    FActive := False;
    DoClose;
  end;
end;

Procedure TmdCustomMailSlot.UpdateSlotInfo;
Begin
  If NOT Active Then
    PerformError(sNotActive);

  If NOT GetMailSlotInfo(Handle,NIL, FNextSize, @FWaiting, NIL) Then
    PerformError(sGetMailSlotInfoError);
end;

Function TmdCustomMailSlot.ReadNext : String;
Var
  ReadSize : DWord;
Begin
  UpdateSlotInfo;
  If FWaiting = 0 Then
    PerformError(sNoMessages);
  SetLength(Result,FNextSize);
  ReadFile(Handle,PChar(Result)^,FNextSize,ReadSize,NIL);
  If FNextSize <> ReadSize Then
    PerformError(sReadError);
end;

Procedure TmdCustomMailSlot.ReadMessage;
Begin
  DoMessageAvail(ReadNext);
end;

// Property modification functions

Procedure TmdCustomMailSlot.SetActive(Value : Boolean);
Begin
  If csReading IN ComponentState Then // Ver 1.4
  Begin
    FLoadedActiveValue := Value;
    Exit;
  end;

  FActive := Value;
  If Value Then
    Open
  else
    Close;
end;

Procedure TmdCustomMailSlot.SetServer(Const Value : String);
Begin
  If Value = FServer Then
    Exit;
  FServer := Value;
  If Active Then
    Open;
end;

Procedure TmdCustomMailSlot.SetSlot(Const Value : String);
Begin
  If Value = FSlot Then
    Exit;
  FSlotChanged := True;
  FSlot := Value;
  If Active Then
    Open;
end;

Procedure TmdCustomMailSlot.SetMaxSize(Value : DWord);
Begin
  If Value = FMaxSize Then
    Exit;
  FMaxSize := Value;
  If Active  Then
    Open;
end;

Function TmdCustomMailSlot.GetWaiting : DWord;
Begin
  Result := 0;
  // I want that it is possible to check on
  // waiting without testing Active first
  If NOT Active Then
    Exit;
  UpdateSlotInfo;
  Result := FWaiting;
end;

Function TmdCustomMailSlot.GetNextSize : DWord;
Begin
  UpdateSlotInfo;
  Result := FNextSize;
end;

// Event process functions

Procedure TmdCustomMailSlot.DoOpen;
Begin
  If Assigned(FOpen) Then
    FOpen(Self);
end;

Procedure TmdCustomMailSlot.DoClose;
Begin
  If Assigned(FClose) Then
    FClose(Self);
end;

Procedure TmdCustomMailSlot.DoMessageAvail(Const Msg : String);
Begin
  If Assigned(FMessageAvail) Then
    FMessageAvail(Self, Msg);
end;

// .........................................................
//                     TmdAutoMailSlot
// .........................................................
(*
Constructor TmdAutoMailSlot.Create(AOwner : TComponent);
Begin
  FTimer := NIL;
  FMessageList := NIL;
  Inherited Create(AOwner);
end;

Procedure TmdAutoMailSlot.AddMessage(Const Msg : String);
Begin
  If FMessageList = NIL Then
    FMessageList := TStringList.Create;
  FMessageList.Add(Msg);
end;

Function TmdAutoMailSlot.GetMessage : String;
Begin
  Result := '';
  If FMessageList <> NIL Then
  Begin
    If FMessageList.Count > 0 Then
    Begin
      Result := FMessageList[0];
      FMessageList.Delete(0);
    end;
    If FMessageList.Count = 0 Then
    Begin
      FMessageList.Free;
      FMessageList := NIL;
    end;
  end;
end;

Procedure TmdAutoMailSlot.Open;
Begin
  Inherited Open;
  If Active AND (NOT (csDesigning IN ComponentState))Then
  Begin
    If FTimer = NIL Then
      FTimer := TTimer.Create(Self);
    FTimer.Interval := 1000;
    FTimer.OnTimer := TimerProc;
  end;
end;

Procedure TmdAutoMailSlot.Close;
Begin
  If FTimer <> NIL Then
    FTimer.Free;
  FTimer := NIL;
  Inherited Close;
end;

Procedure TmdAutoMailSlot.TimerProc(Sender : TObject);
Begin
  While Waiting > 0 do
    AddMessage(ReadNext);

  If (FMessageList <> NIL) AND (FMessageList.Count > 0) Then
    DoMessageAvail(GetMessage);
end;

Procedure TmdAutoMailSlot.DoMessageAvail(Const Msg : String);
Begin
  If Assigned(FMessageAvail) Then
    FMessageAvail(Self, Msg);
end;
*)

// .........................................................
//                        TmdCustomWinPopup
// .........................................................

// Overriden functions

Constructor TmdCustomWinPopup.Create(AOwner : TComponent);
Begin
  Inherited Create(AOwner);
  Inherited Slot := wpSlot;
end;

Procedure TmdCustomWinPopup.DoMessageAvail(Const Msg : String);
var
  P : PChar;
  ASender, AReciever, AMessage : String;
Begin
  If Assigned(FMessage) Then
  Begin
    P := PChar(Msg);
    OemToChar(P,P);
    ASender := StrPas(P);
    While NOT (P[0] = #0) do
      Inc(P);
    Inc(P);
    OemToChar(P,P);
    AReciever := StrPas(P);
    While NOT (P[0] = #0) do
      Inc(P);
    Inc(P);
    OemToChar(P,P);
    AMessage := StrPas(P);
    FMessage(Self, AReciever, ASender, AMessage);
  end;
end;

// New functions

Function TmdCustomWinPopup.Send(AServer, ASender, AReciever, AMsg : String):Boolean;
Begin
  CharToOEM(PChar(AServer), PChar(AServer));
  CharToOEM(PChar(ASender),PChar(ASender));
  CharToOEM(PChar(AReciever),PChar(AReciever));
  CharToOEM(PChar(AMsg),PChar(AMsg));
  Result := SendToMailSlot(AServer, wpSlot, ASender+#0+AReciever+#0+AMsg);
end;

// .........................................................
//                        Register
// .........................................................

procedure Register;
begin
  RegisterComponents('mdVCL', [TmdMailSlot{,
                               TmdAutoMailSlot},
                               TmdWinPopup]);
end;

end.
