unit MKTable;

{
               tTableDBF and TTableFilter VCL Components
                             Version 1.10
             Copyright 1995-96 Absolute Computer Consulting
                         All Rights Reserved
                           24 February 1996

             Please see mktable.txt for more information.

    This VCL component and the associated files can be freely used and
    distributed in commercial and private environments, provided this
    notice is not modified in any way without my expressed written
    consent.  The tTableDBF component is released as is.  No warranty
    is implied and we are not responsible for any problems that might
    arise from the use of this unit.  You use this unit entirely at
    your own risk.
}

(**) interface (**)

uses SysUtils,
     Classes,
     WinTypes,
     WinProcs,
     DB,
     DBTables,
     DbiTypes,
     DbiProcs,
     DbiErrs;

type

  { Exceptions for TablePack }
  EPackError             = class(Exception);
  EPack_InvalidParam     = class(EPackError);
  EPack_InvalidHndl      = class(EPackError);
  EPack_InvalidDBSpec    = class(EPackError);
  EPack_NoConfigFile     = class(EPackError);
  EPack_NoSuchTable      = class(EPackError);
  EPack_NotSupported     = class(EPackError);
  EPack_UnknownTblType   = class(EPackError);
  EPack_UnknownDB        = class(EPackError);
  EPack_NeedExclAccess   = class(EPackError);
  EPack_IncorrectTblType = class(EPackError);
  EPack_DBLimit          = class(EPackError);

  {
    Define a descendant class from Exception to handle errors from search
    routines.  This exception will be called with a meaningful error
    message. :)
  }
  EDBFTableSearch = class(Exception);

{ --------------------------------------------------------------------------- }

  {
    Definition of the filter component.
  }

  tFilterEvent = function (DataSet : tDataset) : boolean of object;
  TTableFilter = class(tComponent)
  private
    fFilter      : boolean;            { are we currently filtering data.     }
    fOnFilter    : tFilterEvent;       { Holder for FILTER event.             }
    FilterHandle : hDBIFilter;         { hold the filter handle.              }
    FTable       : tTable;             { Table the user specified.            }
    FSameTable   : tTable;             { a PRIVATE SECRET table.              }
    function  DoFilter(RecNum : longint) : integer;
    procedure ApplyFilter;
    procedure CancelFilter;
    procedure SetFilter(incoming : boolean);
  published
    property    OnFilter : tFilterEvent read fOnFilter write fOnFilter;
    property    Filter   : Boolean      read fFilter   write SetFilter default false;
    property    Table    : tTable       read FTable    write FTable;
  end; { class TTableFilter }

{ --------------------------------------------------------------------------- }

  {
    Definition of derivate table component.
  }

  TTableDBF = class(tTable)
  private
    { Private declarations }
    procedure PackDBaseTable;
    procedure PackParadoxTable;
    function  FindRecordNumber: longint;
  protected
    { Protected declarations }
    function Chk(rslt: DbiResult): DbiResult; virtual;
    function GetTableType: PChar;
  public
    { Public declarations }
    procedure Pack;
    function  FindKeyDBF(KeyValue : STRING) : boolean;
    procedure FindNearestDBF(KeyValue : STRING);
    procedure ApplyRangeDBF(BeginKey, EndKey : STRING);
    procedure CancelRangeDBF;
    procedure RegenerateIndexes;
    { tBookMark is defined as: TBookmark = Pointer }
    function  CompareBookmarks(First, Second : tBookmark) : boolean;
    property  RecordNumber : longint      read FindRecordNumber;
  end;

{ --------------------------------------------------------------------------- }

procedure Register;                    { procedure to register tTableDBF comp }

(**) implementation (**)


{
  This is a call-back function that the BDE will use to communciate with
  the TTableFilter component.
}

function FilterMK(ulClientData : Longint; pRecBuf : Pointer;
                  iPhyRecNum : Longint) : SmallInt; stdcall; export;
var
  TmpFilter : tTableFilter;
begin
  TmpFilter := Pointer(ulClientData);
  Result    := TmpFilter.DoFilter(iPhyRecNum);
end;

{ --------------------------------------------------------------------------- }
{ -------  tTableFilter component code starts here  ------------------------- }
{ --------------------------------------------------------------------------- }

{
  This function acts like an interface to the BDE call-back function
  (MKFILTER) that was defined at the top of this module.
}

function  tTableFilter.DoFilter(RecNum : longint) : integer;
begin
  Result := 1;

  if not FSameTable.Active then FSameTable.Active := TRUE;

  {
    We now want to move the record pointer in the "secret" table
    to the record that is currently being filtered in the user's
    table.  We will then pass the secret table to the user and
    they will never know the difference :)
  }
  Check( DBISetToRecordNo(FSameTable.Handle, RecNum) );
  FSameTable.ReSync([]);

  {
    IF the user created a "OnFilter event, call it now.
  }
  if Assigned(fOnFilter) then
    if fOnFilter(FSameTable) = false then
      Result := 0;

end;

{ --------------------------------------------------------------------------- }

{
  Procedure to tell tTableFilter to apply the filter condition given in the
  user defined method FilterRecord.
}

procedure tTableFilter.ApplyFilter;
begin

  if FTable <> NIL then
  begin
    if (FTable.State = dsBrowse) then
    begin
      { use DBE function to ADD a filter }
      if Assigned(FilterHandle) then CancelFilter;

      {
        When the user gives us the table to filter, we will extract some
        information and store it in the secret table.  When the
        user wants to filter the data, we will be able to use our
        secret table.
      }

      FSameTable := tTable.Create(nil);
      with FSameTable do
      begin
        {
          We want to ensure that this table is as FAST as possible within the
          constraints of the BDE.
        }
        DisableControls;
        ReadOnly := TRUE;
        DatabaseName := FTable.DatabaseName;
        TableName    := FTable.TableName;
        Active       := TRUE;
      end;

      {

        Here we are being sneaky....

        The DBIAddFilter method has a spot where we can store
        "client supplied data". For us, we will choose the client data to be
        a pointer to this instance of the tTableFilter class.  With this
        information, the FilterMK (BDE call-back) routine will be able to
        call methods in this instance of the TTableFilter class.

      }
      Check( DBIAddFilter(FTable.Handle, LongInt(Self), 1, FALSE,
                          NIL, FilterMK, FilterHandle) );
      Check( DBIActivateFilter(FTable.Handle, FilterHandle) );

      FTable.First;
      FFilter := TRUE;
    end; { if }
  end; { if }
end; { procedure }

{ --------------------------------------------------------------------------- }

{
  Procedure to tell tTableFilter to remove the filter condition given in the
  user defined method FilterRecord.
}

procedure tTableFilter.CancelFilter;
begin
  if FTable <> NIL then
  begin
    if FTable.Active then
    { use DBI function to REMOVE a fitler. }
    if Assigned(FilterHandle) then
    begin
      Check( DBIDeactivateFilter(FTable.Handle, FilterHandle) );
      Check( DBIDropFilter(FTable.Handle, FilterHandle) );
      FilterHandle := NIL;
      FTable.First;
      { Release our "secret table" }
      FSameTable.Active := FALSE;
      FSameTable.Free;
    end;
  end;
  FFilter := FALSE;
end;

{ --------------------------------------------------------------------------- }

procedure tTableFilter.SetFilter(incoming : boolean);
begin

  if not (csDesigning in ComponentState) then
   if incoming = true then
     ApplyFilter
   else
     CancelFilter;

end;

{ --------------------------------------------------------------------------- }
{ -----------------  tTableDBF component  ----------------------------------- }
{ --------------------------------------------------------------------------- }


{find record number}
function tTableDBF.FindRecordNumber: longint;
var
 cP: CurProps;
 rP: RECProps;
begin
 {Return 0 if dataset is not Paradox or dBase}
 Result := 0;

   if state = dsInactive then
    exit;

   {we need to make this call to grab the cursor's iSeqNums}
   Check(DBiGetCursorProps(Handle,cP));

   {synchronize the BDE cursor with the dataset's cursor}
   UpdateCursorPos;

   {fill rP with the current record's properties}
   Check(DBiGetRecord(Handle,DBiNOLOCK,nil,@rP));

   {what kind of dataset are we looking at?}
   case cP.iSeqNums of
    0:
     result := rP.iPhyRecNum; {dBase}
    1:
     result := rP.iSeqNum;    {Paradox}
   end;

end;

{ --------------------------------------------------------------------------- }

{
  Procedure to regenerate all indexes for the table.
}

procedure tTableDBF.RegenerateIndexes;
begin

  Check( DBIRegenIndexes(Handle) );

end;

{ --------------------------------------------------------------------------- }

{
  A function that will compare two bookmarks and return if they point to the
  same data record.
}

function tTableDBF.CompareBookmarks(First, Second : tBookmark) : boolean;
var
  BDEResult : integer;
begin

  Result := FALSE;

  if Assigned(First) and Assigned(Second) then
  begin
    Check( DBICompareBookMarks(Handle, First, Second, BDEResult) );
    if BDEResult = 0 then Result := TRUE;
  end;

end;

{ --------------------------------------------------------------------------- }

{

  Function that will find a matching record in the table by using the
  passed KeyValue to do the search.  This method was designed to be
  used with dBase expression indexes.

}

function tTableDBF.FindKeyDBF(KeyValue : STRING)     : boolean;
var
  TableType : array[0..40] of char;
  DataKey   : array[0..255] of char;
begin

  if Active = FALSE then
    raise eDBFTableSearch.Create('Table must be open to search it.');

  strcopy(TableType, GetTableType);    { make sure we have a dBase file.      }
  if StrComp(TableType, szdBase) <> 0 then
  begin
    { we do not have a dBase table.  Throw an exception. }
    raise eDBFTableSearch.Create('Must be a dBase table.');
  end;

  CheckBrowseMode;
  CursorPosChanged;
  StrPCopy(DataKey, KeyValue);         { convert passed String into a pChar   }
  Result := FALSE;                     { Assume no matching key value         }

  if DBIGetRecordForKey(Handle,        { Table handle                         }
                        TRUE,          { we are passing the key directly.     }
                        1,             { we are passing 1 field??? the key?   }
                        0,             { partial key length ?                 }
                        @DataKey,      { Search key. (pointer to it.)         }
                        nil) = 0 then
  begin
    Resync([rmExact, rmCenter]);       { resync BDE and Delphi                }
    Result := True;
  end;

end; { FindKeyDBF }

{ --------------------------------------------------------------------------- }

{

  Function that will find the nearest matching record (greater than or
  euqal) in the table by using the passed KeyValue to do the search.
  This method was designed to be used with dBase expression indexes.

}
procedure tTableDBF.FindNearestDBF(KeyValue : STRING);
var
  TableType : array[0..40] of char;
  DataKey   : array[0..255] of char;
  SearchCond: DBISearchCond;
begin

  if Active = FALSE then
    raise eDBFTableSearch.Create('Table must be open to search it.');

  strcopy(TableType, GetTableType);    { make sure we have a dBase file.      }
  if StrComp(TableType, szdBase) <> 0 then
  begin
    { we do not have a dBase table.  Throw an exception. }
    raise eDBFTableSearch.Create('Must be a dBase table.');
  end;

  CheckBrowseMode;
  CursorPosChanged;
  StrPCopy(DataKey, KeyValue);         { convert passed String into a pChar   }
  SearchCond := keySEARCHGEQ;          { set the search condition             }
  check(DBISetToKey(Handle,            { table's BDE handle                   }
                    SearchCond,        { define the search condition.         }
                    TRUE,              { we are directly passing the key value}
                    1,
                    0,
                    @DataKey));        { a pointer to the key value.          }
  Resync([rmCenter]);                  { resync BDE and Delphi                }
end;

{ --------------------------------------------------------------------------- }

{
  This method is used to apply a range to the database table.  It uses
  the passed STRING values to set the range.  It was designed to use
  with dBase expression indexes.

  NOTE:  This method can be used to setup master-detail forms.  Just
  use the master table to set the range in the detail table.

  NOTE:  If BeginKey and EndKey are the same, an ASCII 255 is added to
  the EndKey so that all matching records for BeginKey are displayed.
}

procedure tTableDBF.ApplyRangeDBF(BeginKey, EndKey : STRING);
var
  Start,
  Finish : array[0..255] of char;
begin

  { 21 Jan 1996:  Added code abort range if no active dataset. }
  if not Active then exit;

  CheckBrowseMode;

  StrPCopy(Start,  BeginKey);
  StrPCopy(Finish, EndKey);

  {
    We want the range to be inclusive.  The BDE doesn't like that idea
    too much.  So we tack on a "fudge factor" to the Finish key.
  }
  StrCat(Finish, #255);

  Check( DBISetRange( Handle,          { handle for the table, passed to BDE  }
                      TRUE,            { we are directly passing the key.     }
                      1,               { set to 1 for expression searching.   }
                      0,
                      @Start,          { pointer to the starting key value.   }
                      TRUE,            { inclusive start key please!.         }
                      1,               { set to 1 for expression searching.   }
                      0,
                      @Finish,         { pointer to the ending key value.     }
                      TRUE) );         { inclusive end key please!            }

   First;                              { go to the top of the table.          }

end;

{ --------------------------------------------------------------------------- }

{
  This method will remove a range that was applied by the
  ApplyRangeDBF method.  It will more than likely remove any ranges
  that are set!
}

procedure tTableDBF.CancelRangeDBF;
begin

  CheckBrowseMode;
  UpdateCursorPos;
  Check( DBIResetRange(Handle) );
  Resync([]);                          { resync BDE and Delphi                }

end;

{ --------------------------------------------------------------------------- }

{
  The code that will pack a dBase or Paradox table was written by
  Steve Teixeira (Borland Tech Support).  I found the code on a web
  page.  It had no copyright notice or even an author in the source
  code, so I incorporated it into my class.  I thank Steve for
  releasing this code to the masses.

  His class was called tTablePack.  It allowed the packing of dBase
  and Paradox tables.
}

{ --------------------------------------------------------------------------- }

function tTableDBF.GetTableType: PChar;
var
  { FCurProp Holds information about the structure of the table }
  FCurProp: CurProps;

begin
  { Find out what type of table is currently opened.  NOTE: This is
    different than tTableDBF.TableType }
  Chk(DbiGetCursorProps(Handle, FCurProp));
  GetTableType := FCurProp.szTableType;
end;

procedure tTableDBF.Pack;
var
  TType: array[0..40] of char;

begin
  { The table must be opened to get directory information about the table
    before closing }
  if Active <> True then
    raise EPackError.Create('Table must be opened to be packed');
  { Get the type of table }
  strcopy(TType, GetTableType);
  if strcomp(TType, szParadox) = 0 then
    { Call PackParadoxTable procedure if PARADOX table }
    PackParadoxTable
  else
    if strcomp(TType, szDBase) = 0 then
      { Call PackDBaseTable procedure if dBase table }
      PackDBaseTable
    else
      { PARADOX and dBase table are the only types that can be packed }
      raise EPack_IncorrectTblType.Create('Incorrect table type: ' +
                                          StrPas(TType));
end;

procedure tTableDBF.PackParadoxTable;
var
  { Specific information about the table structure, indexes, etc. }
  TblDesc: CRTblDesc;
  { Uses as a handle to the database }
  hDb: hDbiDb;
  { Path to the currently opened table }
  TablePath: array[0..dbiMaxPathLen] of char;

begin
  { Initialize the table descriptor }
  FillChar(TblDesc, SizeOf(CRTblDesc), 0);
  with TblDesc do
  begin
    { Place the table name in descriptor }
    StrPCopy(szTblName, TableName);
    { Place the table type in descriptor }
    StrCopy(szTblType, GetTableType);
    { Set the packing option to true }
    bPack := True;
  end;
  { Initialize the DB handle }
  hDb := nil;
  { Get the current table's directory.  This is why the table MUST be
    opened until now }
  Chk(DbiGetDirectory(DBHandle, True, TablePath));
  { Close the table }
  Close;
  { NOW: since the DbiDoRestructure call needs a valid DB handle BUT the
    table cannot be opened, call DbiOpenDatabase to get a valid handle.
    Just leaving Active = FALSE does not give you a valid handle }
  Chk(DbiOpenDatabase(nil, 'STANDARD', dbiReadWrite, dbiOpenExcl, nil,
                        0, nil, nil, hDb));
  { Set the table's directory to the old directory }
  Chk(DbiSetDirectory(hDb, TablePath));
  { Pack the PARADOX table }
  Chk(DbiDoRestructure(hDb, 1, @TblDesc, nil, nil, nil, FALSE));
  { Re-Open the table }
  Open;
end;

procedure tTableDBF.PackDBaseTable;
begin
  { Pack the dBase Table }
  Chk(DbiPackTable(DBHandle, Handle, nil, nil, True));
end;

function tTableDBF.Chk(rslt: DbiResult): DbiResult;
var
  ErrInfo: DbiErrInfo;
  ErrStr: string;
  pErr: array[0..dbiMaxMsgLen] of char;

begin
  { Only enter the error routine if an error has occured }
  if rslt <> dbiErr_None then
  begin
    { Get information on the error that occured.  ALWAYS DO THIS FIRST!! }
    DbiGetErrorInfo(False, ErrInfo);
    if ErrInfo.iError = rslt then
    begin
      ErrStr := Format('%s  ', [ErrInfo.szErrCode]);
      if StrComp(ErrInfo.szContext[1], '') = 0 then
        ErrStr := Format('%s  %s', [ErrStr, ErrInfo.szContext[1]]);
      if StrComp(ErrInfo.szContext[2], '') = 0 then
        ErrStr := Format('%s  %s', [ErrStr, ErrInfo.szContext[2]]);
      if StrComp(ErrInfo.szContext[3], '') = 0 then
        ErrStr := Format('%s  %s', [ErrStr, ErrInfo.szContext[3]]);
      if StrComp(ErrInfo.szContext[4], '') = 0 then
        ErrStr := Format('%s  %s', [ErrStr, ErrInfo.szContext[4]]);
    end
    else
    begin
      DbiGetErrorString(rslt, pErr);
      ErrStr := StrPas(pErr);
    end;
    ErrStr := Format('Table Pack Error: %d.  %s', [rslt, ErrStr]);

    MessageBeep(mb_IconExclamation);
    { Raise the corrisponding exception }
    case rslt of
      dbiErr_InvalidParam:
        raise EPack_InvalidParam.Create(ErrStr);
      dbiErr_InvalidHndl:
        raise EPack_InvalidHndl.Create(ErrStr);
      dbiErr_InvalidDBSpec:
        raise EPack_InvalidDBSpec.Create(ErrStr);
      dbiErr_NoSuchTable:
        raise EPack_NoSuchTable.Create(ErrStr);
      dbiErr_NoConfigFile:
        raise EPack_NoConfigFile.Create(ErrStr);
      dbiErr_NotSupported:
        raise EPack_NotSupported.Create(ErrStr);
      dbiErr_UnknownTblType:
        raise EPack_UnknownTblType.Create(ErrStr);
      dbiErr_UnknownDB:
        raise EPack_UnknownDB.Create(ErrStr);
      dbiErr_NeedExclAccess:
        raise EPack_NeedExclAccess.Create(ErrStr);
      dbiErr_DBLimit:
        raise EPack_DBLimit.Create(ErrStr);
      else
        raise EPackError.Create(ErrStr);
    end;
  end;
end;

{ --------------------------------------------------------------------------- }

{
  Procedure to register my class in the component library.  I am
  choosing to add this Component to the Data Access tab.
}

procedure Register;
begin
  RegisterComponents('MKTools', [TTableDBF]);
  RegisterComponents('MKTools', [TTableFilter]);
end;

{ --------------------------------------------------------------------------- }
end.

