(***************************************************
TMemoryMappedFile->TComponent

Encapsulates a Win32 memory-mapped file.
***************************************************)

unit MemoryMappedFile;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs, DsgnIntf;

type

  EMemoryMappedFile = class( Exception );
  TAccessMode = ( amRead, amReadWrite );
  TShareMode = ( smNone, smRead, smWrite, smReadWrite );
  TCreationDistribution = ( cdCreateNew, cdCreateAlways, cdOpenExisting, cdOpenAlways, cdTruncateExisting );

  TMemoryMappedFile = class( TComponent )
  private
     FAccessMode: TAccessMode;
     FActive: boolean;
     FCreationDistribution: TCreationDistribution;
     FFileData: pointer;
     FFileName: TFileName;
     FShareMode: TShareMode;
     FSize: DWORD;
     hFile: THandle;
     hFileMapping: THandle;
     dwProtect: DWORD;
     dwMapAccess: DWORD;
     dwHighWord: DWORD;
  protected
     procedure CreateFileHandle;
     procedure SetActive( b: boolean );
     procedure SetFileName( s: TFileName );
  public
     constructor Create( AOwner: TComponent ); override;
     destructor Destroy; override;
     procedure Close;
     procedure Flush;
     procedure Open;
     property FileData: pointer read FFileData;
  published
     property AccessMode: TAccessMode read FAccessMode write FAccessMode default amReadWrite;
     property Active: boolean read FActive write SetActive;
     property CreationDistribution: TCreationDistribution read FCreationDistribution write FCreationDistribution default cdOpenAlways;
     property FileName: TFileName read FFileName write SetFileName;
     property ShareMode: TShareMode read FShareMode write FShareMode default smRead;
     property Size: DWORD read FSize write FSize;
  end;

  TFileNameProperty = class( TPropertyEditor )
  private
  protected
  public
     procedure Edit; override;
     function GetAttributes: TPropertyAttributes; override;
     function GetValue: string; override;
     procedure SetValue( const Value: string ); override;
  end;

procedure Register;

implementation

constructor TMemoryMappedFile.Create( AOwner: TComponent );
begin
  inherited Create( AOwner );
  FAccessMode := amReadWrite;
  FCreationDistribution := cdOpenAlways;
  hFile := INVALID_HANDLE_VALUE;
  FShareMode := smRead;
end;

destructor TMemoryMappedFile.Destroy;
begin
  Close;
  if hFile <> INVALID_HANDLE_VALUE then
     CloseHandle( hFile );
  inherited Destroy;
end;

(***************************************************
Open and Close methods are simply another way of
changing the Active property.
***************************************************)
procedure TMemoryMappedFile.Close;
begin
  Active := FALSE;
end;

procedure TMemoryMappedFile.Open;
begin
  Active := TRUE;
end;

(***************************************************
Only allow change of file name if not open.  Setting
the file name actually creates a kernal File object,
which is used to determine the file size.
***************************************************)
procedure TMemoryMappedFile.SetFileName( s: TFileName );
begin
  if not FActive then
     begin
        FFileName := s;
        if CreationDistribution <> cdCreateNew then
           begin
              CreateFileHandle;

{ Get the size of the file and make this the default value for the mapped file }
              FSize := GetFileSize( hFile, @dwHighWord );
              CloseHandle( hFile );
              hFile := INVALID_HANDLE_VALUE;
           end
        else
           FSize := 0;

{ Don't allow use of files larger than 4gig ... for now }
        if dwHighWord <> 0 then
           begin
              FFileName := '';
              FSize := 0;
              raise EMemoryMappedFile.Create( 'Files larger than 4gig not supported' );
           end
     end
end;
(***************************************************
This routine creates the file kernal object.
***************************************************)
procedure TMemoryMappedFile.CreateFileHandle;
var
  dwAccess, dwShare, dwCreate: DWORD;
begin

{ If a handle is open, make sure to close it }
  if hFile <> INVALID_HANDLE_VALUE then
     begin
        CloseHandle( hFile );
        hFile := INVALID_HANDLE_VALUE;
        FSize := 0;
     end;

{ Determine settings for CreateFile call }
  case FAccessMode of
     amRead:
        begin
           dwAccess := GENERIC_READ;
           dwProtect := PAGE_READONLY;
           dwMapAccess := FILE_MAP_READ;
        end;
     amReadWrite:
        begin
           dwAccess := GENERIC_READ or GENERIC_WRITE;
           dwProtect := PAGE_READWRITE;
           dwMapAccess := FILE_MAP_ALL_ACCESS;
        end;
  end;

  case FShareMode of
     smNone:
        dwShare := 0;
     smRead:
        dwShare := FILE_SHARE_READ;
     smWrite:
        dwShare := FILE_SHARE_WRITE;
     smReadWrite:
        dwShare := FILE_SHARE_READ or FILE_SHARE_WRITE;
  end;

  case FCreationDistribution of
     cdCreateNew:
        dwCreate := CREATE_NEW;
     cdCreateAlways:
        dwCreate := CREATE_ALWAYS;
     cdOpenExisting:
        dwCreate := OPEN_EXISTING;
     cdOpenAlways:
        dwCreate := OPEN_ALWAYS;
     cdTruncateExisting:
        dwCreate := TRUNCATE_EXISTING;
  end;

{ Call CreateFile, and check for success }
  hFile := CreateFile( PChar( FFileName ), dwAccess, dwShare, nil, dwCreate,
     FILE_ATTRIBUTE_NORMAL, 0 );
  if hFile = INVALID_HANDLE_VALUE then
     begin
        FFileName := '';
        FSize := 0;
        raise EMemoryMappedFile.Create( 'CreateFile failed with Error Code: ' +
           IntToStr( GetLastError ) );
     end;

end;
(***************************************************
Setting Active to true establishes the mapping and
commits physical storage to the region.
***************************************************)
procedure TMemoryMappedFile.SetActive( b: boolean );
var
  nRC: DWORD;
begin
  if FActive <> b then
     begin
        if b then
           begin
              CreateFileHandle;
              hFileMapping := CreateFileMapping( hFile, nil, dwProtect, 0, FSize, nil );
              if hFileMapping = 0 then
                 raise EMemoryMappedFile.Create( 'CreateFileMapping failed with Error Code: '
                    + IntToStr( GetLastError ) );
              FFileData := MapViewOfFile( hFileMapping, dwMapAccess, 0, 0, FSize );
              if FFileData = nil then
                 begin
                    nRC := GetLastError;
                    CloseHandle( hFileMapping );
                    raise EMemoryMappedFile.Create( 'MapViewOfFile failed with Error Code: '
                       + IntToStr( nRC ) );
                 end
           end
        else
           begin
              UnmapViewOfFile( FFileData );
              CloseHandle( hFileMapping );
           end;
        FActive := b;
     end;
end;

(***************************************************
Allow the user to flush the data if desired.
***************************************************)
procedure TMemoryMappedFile.Flush;
begin
  if FActive then
     FlushViewOfFile( FFileData, FSize );
end;

(*********************************************
Property editors.
*********************************************)
procedure TFileNameProperty.Edit;
var
  d: TOpenDialog;
begin
  d := TOpenDialog.Create( nil );
  d.FileName := GetStrValue;
  d.Title := 'Select File Name';
  if d.Execute then
     SetStrValue( d.FileName );
  d.Free;
end;

function TFileNameProperty.GetValue: string;
begin
  Result := GetStrValue;
end;

procedure TFileNameProperty.SetValue( const Value: string );
begin
  SetStrValue( Value );
end;

function TFileNameProperty.GetAttributes: TPropertyAttributes;
begin
  Result := [paDialog];
end;

procedure Register;
begin
  RegisterPropertyEditor( TypeInfo( TFileName ), TMemoryMappedFile, 'FileName', TFileNameProperty );
  RegisterComponents( 'NonVis', [TMemoryMappedFile] );
end;

end.
