{  Common unit for Make Update and Update

  Copyright (c) 1998 Gregory L. Bullock (bullock@tsppd.com).
  Freeware: May be freely distributed and modified. Use at your own risk.
  This program draws heavily on

    - the aDiff unit, Copyright (c) 1997 S.Kurinny & S.Kostinsky

    - the Lh5Unit unit, which had various contributers, as noted in its
      source code

  History:
   14-12-99 Added ability to copy over any existing "old" file without
            first verifying that it passes the CRC version checking. This
            is equivalent to deleting the old file then adding the new one.

     4-5-99 Added option to update-all-or-nothing in case of failure.

     3-4-99 Retired support for 16-bit compilation; it was getting too
            troublesome.
            Store file date/time in .Upd file so Update can set the time
            stamp on the updated files to match the original.
            Add ability to add new files or delete obsolete files.
            Use the file sizes in addition to the CRCs to help identify files.

     3-7-98 Added a CRC for the *compressed* differences to better detect
            whether an .Upd is corrupted. Incremented the UpdateFileHeader
            to account for this change to the .Upd file.

    16-5-98 Modified to compile under D3 (32-bits) as well as D1 (16-bits).

            Added a new compiler directive, MAKE16AND32COMPATIBLE, which
            will ensure that the *.Upd file is compatible independent
            of the platform (16- or 32-bit) which creates it or uses it.
            On the one test case I ran, I found that using MAKE16AND32COMPATIBLE
            increased the size of the *.Upd file.  The file sizes I got were
              29304  for the 32-bit .Upd file
              30746  for the 16-bit .Upd file
              30759  for the compatible .Upd file

            Improved one of the log messages and corrected the help file.

    23-4-98 Initial version.

  If you fix any bugs or make significant enhancements, I ask you to send
  me your modifications.

  For example, at present, these programs support an auto-updating system over
  a LAN or WAN, but not over the WEB.  Perhaps a future version of these
  programs will enable Update to support auto-updating system over the WEB.

  If you make any changes to the structure of the Update File,
  change the UpdateFileHeader in the Common Unit so older versions
  of Update won't try to read an Update File that they won't understand.

}

unit Common;

interface

uses
  Windows, SysUtils, StdCtrls, Classes, FileCtrl, aCRC32;

type
  TFileProcessingIndex = (FPNothing, FPUpdate, FPAdd, FPDelete, FPCopyOver);

  TUniqueFileIDInfo = packed record
    CRC,
    HiSize,
    LoSize  : longint;
  end;

  TFileUpdateInfo = packed record
    Name    : string[255];
    OldID,
    NewID   : TUniqueFileIDInfo;
    NewDate : integer;
    FileProcessing  : TFileProcessingIndex;
  end;
  PFileUpdateInfo  = ^TFileUpdateInfo;

  PFileDate = ^integer;

const
  UpdateFileHeader : string[15] = 'Update-32  1.03';
    {Change the UpdateFileHeader if you make any changes to the structure
    of the Update File, so older versions of Update won't try to read
    an Update File that they won't understand.}

  AlsoAllowedUpdateFileHeader : string[15] = 'Update-32  1.02';
    {The only difference between an "Update-32  1.02" file and an
    "Update-32  1.03" file is that the latter can have FileProcessing
    = FPCopyOver, i.e., it allows copying over any existing old file,
    replacing it with the new one.  This means that an Update executable that
    can read a "1.03" file can also read a "1.02" file, so we'll allow that
    with AlsoAllowedUpdateFileHeader.}


function GetFullPathTo(FileName: TFileName; DirectoryListBox: TDirectoryListBox) : TFileName;
function CalculateStreamCRC(Stream: TStream): LongInt;
procedure CalculateUniqueFileIDInfo(const FileName: TFileName; var UniqueFileID: TUniqueFileIDInfo; pDate: PFileDate);
{$IFDEF VER90}  {D2 Doesn't know about CompareMem}
function CompareMem(P1, P2: Pointer; Length: Integer): Boolean;
{$ENDIF}

var
  UpdateStream,
  OldStream,
  NewStream     : TFileStream;
  UpdAccumulateStream,
  DiffStream    : TMemoryStream;
  UpdateAllOrNothing  : Boolean;

implementation


function GetFullPathTo(FileName: TFileName; DirectoryListBox: TDirectoryListBox) : TFileName;
begin
  Result := AnsiLowerCase(DirectoryListBox.Directory);
  if Result[Length(Result)] <> '\' then
    Result := Result + '\';
  Result := Result + FileName;
end;

function CalculateStreamCRC(Stream: TStream): LongInt;
Const
  BufSize=$FFF0;
type
  PLongInt=^LongInt;
  TByteArray=Array[1..BufSize] of byte;
  PByteArray=^TByteArray;
var
  Buf : PByteArray;
  BufRead : LongInt;
  SavePosition  : LongInt;
begin
  Buf := nil;
  SavePosition := Stream.Position;
  try
    Result := 0;
    GetMem(Buf,BufSize);
    Result := CRC32Start;
    repeat
      BufRead := Stream.Read(Buf^,BufSize);
      if BufRead = 0 then Break;
      Result := CRC32Calc(Buf^[1],Result,BufRead);
    until BufRead = 0;
    Result := CRC32Finish(Result); {This ensures compatibility with pkzip's
    CRC32, but since that's just the bitwise complement of the CRC32 we
    already had, there's probably no advantage here.}
  finally
    if Buf <> nil then
      FreeMem(Buf,BufSize);
    Stream.Position := SavePosition;
  end;
end;

procedure CalculateUniqueFileIDInfo(const FileName: TFileName; var UniqueFileID: TUniqueFileIDInfo; pDate: PFileDate);
var
  InStream  : TFileStream;
begin
  {Compute the file CRC and Size }
  InStream := nil;
  try
    InStream := TFileStream.Create(FileName,fmOpenRead);
    UniqueFileID.LoSize := GetFileSize(InStream.Handle,@UniqueFileID.HiSize);
    if pDate <> nil then
      pDate^ := FileGetDate(InStream.Handle);
    InStream.Position := 0;
    UniqueFileID.CRC := CalculateStreamCRC(InStream);
  finally
    InStream.Free; 
  end;
end;

{$IFDEF VER90}  {D2 Doesn't know about CompareMem}
function CompareMem(P1, P2: Pointer; Length: Integer): Boolean;
type
  TByteArray = packed array[0..0] of byte;
  PByteArray = ^TByteArray;
var
  i : integer;
begin
  Result := True;
  i := 0;
  while Result and (i < Length) do
  begin
    Result := PByteArray(P1)^[i] = PByteArray(P2)^[i];
    Inc(i);
  end;
end;
{$ENDIF}

initialization
  UpdateStream  := nil;
  OldStream     := nil;
  NewStream     := nil;
  UpdAccumulateStream := nil;
  DiffStream    := nil;
  UpdateAllOrNothing  := False;
end.
