{  Main form for Make Update

  Copyright (c) 1998 Gregory L. Bullock (bullock@tsppd.com).
  Freeware: May be freely distributed and modified. Use at your own risk.
  This program draws heavily on

    - the aDiff unit, copyright (c) 1997 S.Kurinny & S.Kostinsky

    - the Lh5Unit unit, which had various contributers, as noted in its
      source code

  History:
    28-1-00 Thanks to Dzintars for fixing a bug when the FileListbox is empty.

   14-12-99 Added ability to copy over any existing "old" file without
            first verifying that it passes the CRC version checking. This
            is equivalent to deleting the old file then adding the new one.

     4-5-99 Added option to update-all-or-nothing in case of failure.
            Removed the forcing of file names to lower case.

     3-4-99 Retired support for 16-bit compilation; it was getting too
            troublesome.
            Store file date/time in .Upd file so Update can set the time
            stamp on the added and updated files to match the original.
            Add ability to add new files or delete obsolete files.
            Use the file sizes in addition to the CRCs to help identify files.
            Show progress bar for all files in addition to progress bar for
            the current file.

            Thanks to Jens Liebermann for figuring out how to maintain D2
            compatibility.

   28-10-98 Changed Browse button's action to automatically add all selected
            files to the list. (Previously, it would add all but the last
            selected file to the list). Thanks to Massimiliano Orso for
            reporting this bug.  Made Progress dialog get updated more
            frequently under 32-bits.

     3-7-98 Added a CRC for the *compressed* differences to better detect
            whether an .Upd is corrupted.

    16-5-98 Modified to compile under D3 (32-bits) as well as D1 (16-bits).

            Added a new compiler directive, MAKE16AND32COMPATIBLE, which
            will ensure that the *.Upd file is compatible independent
            of the platform (16- or 32-bit) which creates it or uses it.
            On the one test case I ran, I found that using MAKE16AND32COMPATIBLE
            increased the size of the *.Upd file.  The file sizes I got were
              29304  for the 32-bit .Upd file
              30746  for the 16-bit .Upd file
              30759  for the compatible .Upd file

            Improved one of the log messages and corrected the help file.

            Thanks to Massimiliano Orso, Kurt Senfer, Jeremy Coulter, Mavarik,
            Holger Dors, and Fabrice Avaux for reporting bugs and bug fixes.

    23-4-98 Initial version.

  If you fix any bugs or make significant enhancements, I ask you to send
  me your modifications.

  For example, at present, these programs support an auto-updating system over
  a LAN or WAN, but not over the WEB.  Perhaps a future version of these
  programs will enable Update to support auto-updating system over the WEB.

  If you make any changes to the structure of the Update File,
  change the UpdateFileHeader in the Common Unit so older versions
  of Update won't try to read an Update File that they won't understand.
}

unit Updprops;

interface

uses
  SysUtils, Windows, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, TabNotBk, StdCtrls, Buttons, FileCtrl, ExtCtrls, IniFiles,
  Common, ComCtrls, LH5Unit, Progress, ImgList;

type
  TTabbedNotebookDlg = class(TForm)
    MakeBtn: TBitBtn;
    CloseBtn: TBitBtn;
    HelpBtn: TBitBtn;
    OpenDialog: TOpenDialog;
    LoadBtn: TBitBtn;
    SaveBtn: TBitBtn;
    SaveDialog: TSaveDialog;
    AboutBtn: TBitBtn;
    FileProcessingImageList: TImageList;
    PageControl1: TPageControl;
    TabSheet1: TTabSheet;
    TabSheet2: TTabSheet;
    TabSheet3: TTabSheet;
    GroupBox1: TGroupBox;
    OldDirectoryLabel: TLabel;
    OldDirectoryListBox: TDirectoryListBox;
    OldDriveComboBox: TDriveComboBox;
    OldVersionLabel: TLabel;
    OldVersion: TEdit;
    GroupBox2: TGroupBox;
    NewDirectoryLabel: TLabel;
    NewDirectoryListBox: TDirectoryListBox;
    NewDriveComboBox: TDriveComboBox;
    NewVersionLabel: TLabel;
    NewVersion: TEdit;
    Label1: TLabel;
    FileListBox: TListBox;
    FileNameLabel: TLabel;
    FileName: TEdit;
    BrowseButton: TButton;
    AddBtn: TBitBtn;
    DeleteBtn: TBitBtn;
    MoveUpBtn: TBitBtn;
    MoveDownBtn: TBitBtn;
    RefreshCRCBtn: TBitBtn;
    Image1: TImage;
    Label2: TLabel;
    Image2: TImage;
    Label3: TLabel;
    Image3: TImage;
    Image4: TImage;
    Label4: TLabel;
    Label5: TLabel;
    FailureMode: TRadioGroup;
    Image5: TImage;
    Label6: TLabel;
    NoVersionChecking: TCheckBox;
    procedure OldDirectoryListBoxChange(Sender: TObject);
    procedure FileNameChange(Sender: TObject);
    procedure FileListBoxClick(Sender: TObject);
    procedure BrowseButtonClick(Sender: TObject);
    procedure AddBtnClick(Sender: TObject);
    procedure DeleteBtnClick(Sender: TObject);
    procedure MoveUpBtnClick(Sender: TObject);
    procedure MoveDownBtnClick(Sender: TObject);
    procedure FileListBoxDrawItem(Control: TWinControl; Index: Integer;
      Rect: TRect; State: TOwnerDrawState);
    procedure RefreshCRCBtnClick(Sender: TObject);
    procedure FormCreate(Sender: TObject);
    procedure LoadBtnClick(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure SaveBtnClick(Sender: TObject);
    procedure CloseBtnClick(Sender: TObject);
    procedure FileNameExit(Sender: TObject);
    procedure MakeBtnClick(Sender: TObject);
    procedure AboutBtnClick(Sender: TObject);
    procedure ContolEnter(Sender: TObject);
    procedure NewDirectoryListBoxChange(Sender: TObject);
    procedure PageControl1Change(Sender: TObject);
    procedure NoVersionCheckingClick(Sender: TObject);
  private
    { Private declarations }
    procedure ClearFileList;
  protected
    procedure RefreshFileProcessingIndicators;
  public
    { Public declarations }
    function DetermineRequiredProcessing(var FileInfo: TFileUpdateInfo; const ErrMsg: string): Boolean;
    function CanAddFileNameToList : Boolean;
    function OldAndNewDirectoriesOK : Boolean;
  end;

var
  TabbedNotebookDlg: TTabbedNotebookDlg;

implementation

uses
  AboutM, aDiff, aCRC32;

{$R *.DFM}

{$IFDEF VER90}  {D2 Doesn't know about resourcestring}
const
{$ELSE}
resourcestring
{$ENDIF}
  IDS_FileNotFoundFmt = 'Cannot find file'#13#10'  %s'#13#10'%s';
  IDS_CantOpenFile = 'Error opening file'#13#10'  %s'#13#10'%s %s';
  IDS_FileNotAdded = 'File not added.';
  IDS_FileNotIncluded = 'File not included in Update file.';
  IDS_CRCNotCalculated = '"Old" and "new" CRCs not refreshed.';
  IDS_FileNotInDirectoryTreeFmt = 'The file'#13#10'   %s'#13#10'is not in the directory or a subdirectory of'#13#10'   %s';
  IDS_IdenticalOldNewDirectories = 'The "Old" and "new" directories are the same, so the CRCs are going to be identical (and probably invalid for your update file). Add files anyway?';
  IDS_UpdateSessionFilter = 'Update Session (*.UpS)|*.UpS';
  IDS_OpenSessionTitle = 'Open Update Session';
  IDS_SaveSessionTitle = 'Save Update Session';
  IDS_OldFilesSection = 'Old Files';
  IDS_NewFilesSection = 'New Files';
  IDS_DirectoryLabel = 'Directory';
  IDS_VersionInfoLabel = 'VersionInfo';
  IDS_FilesListLabel = 'Files List';
  IDS_OptionsLabel = 'Update Options';
  IDS_UpdateAllOrNothing = 'Update All Or Nothing';
  IDS_NumberFilesLabel = 'NumberOfFiles';
  IDS_FileNameLabelFmt = 'FileName%d';
  IDS_NoVersionCheckingLabelFmt = 'NoVersionChecking%d';


var
  SessionFileName  : TFileName;

procedure TTabbedNotebookDlg.FormCreate(Sender: TObject);
begin
  SessionFileName  := 'Default.UpS';
end;

procedure TTabbedNotebookDlg.FormDestroy(Sender: TObject);
begin
  ClearFileList;
end;

procedure TTabbedNotebookDlg.ClearFileList;
var
  i : integer;
begin
  with FileListBox, Items do
  begin
    for i := 0 to Pred(Count) do
      Dispose(PFileUpdateInfo(Objects[i]));
    Clear;
  end;
  FileListBoxClick(Self);
end;

function TTabbedNotebookDlg.DetermineRequiredProcessing(var FileInfo: TFileUpdateInfo; const ErrMsg: string): Boolean;
var
  FileName  : TFileName;
  SaveFileProcessing : TFileProcessingIndex;
  OldExists,
  NewExists : Boolean;
begin
  Result := False;
  SaveFileProcessing := FileInfo.FileProcessing;
  FileInfo.FileProcessing := FPNothing;
  {Compute the Old & New file CRCs }
  try
    with FileInfo do
    begin
      FileName := GetFullPathTo(Name,OldDirectoryListBox);
      OldExists := FileExists(FileName);
      if OldExists then
        CalculateUniqueFileIDInfo(FileName, OldID, nil)
      else
        FillChar(OldID, SizeOf(OldID), 0);

      FileName := GetFullPathTo(Name,NewDirectoryListBox);
      NewExists := FileExists(FileName);
      if NewExists then
        CalculateUniqueFileIDInfo(FileName, NewID, @NewDate)
      else
        FillChar(NewID, SizeOf(NewID), 0);

      if (OldExists and NewExists) then
        if CompareMem(@OldID, @NewID, SizeOf(NewID)) then
          FileProcessing := FPNothing
        else
          if SaveFileProcessing = FPCopyOver then
            // Restore CopyOver option if it was selected and is still available
            FileProcessing := FPCopyOver
          else
            FileProcessing := FPUpdate
      else if NewExists then
        FileProcessing := FPAdd
      else
        FileProcessing := FPDelete;
      Result := True;
    end;
  except
    on E: Exception do
      MessageDlg(E.Message+'. '+ErrMsg,mtError,[mbOk],0);
  end;
end;

procedure TTabbedNotebookDlg.OldDirectoryListBoxChange(Sender: TObject);
begin
  OpenDialog.InitialDir := OldDirectoryListBox.Directory;
  RefreshFileProcessingIndicators;
end;

procedure TTabbedNotebookDlg.NewDirectoryListBoxChange(Sender: TObject);
begin
  RefreshFileProcessingIndicators;
end;

procedure TTabbedNotebookDlg.RefreshFileProcessingIndicators;
var
  i : integer;
  Save_Cursor : TCursor;
begin
  Save_Cursor := Screen.Cursor;
  try
    Screen.Cursor := crHourglass;    { Show hourglass cursor }
    with FileListBox.Items do
    begin
      for i := 0 to Pred(Count) do
        DetermineRequiredProcessing(PFileUpdateInfo(Objects[i])^, IDS_FileNotIncluded);
      Invalidate;
    end;
  finally
    Screen.Cursor := Save_Cursor;  { Always restore to normal }
  end;
end;

function TTabbedNotebookDlg.CanAddFileNameToList : Boolean;
var
  i : integer;
begin
  Result := FileName.Text <> '';
  i := 0;
  with FileListBox,Items do
  while Result and (i < Count) do
  begin
    Result := CompareText(PFileUpdateInfo(Objects[i])^.Name,FileName.Text) <> 0;
    Inc(i);
  end;
end;

procedure TTabbedNotebookDlg.FileNameChange(Sender: TObject);
begin
  AddBtn.Enabled := CanAddFileNameToList;
end;

procedure TTabbedNotebookDlg.FileNameExit(Sender: TObject);
begin
  with FileListBox do
    if CanAddFileNameToList and (ItemIndex > -1) then
    with PFileUpdateInfo(Items.Objects[ItemIndex])^ do
    begin
      FillChar(Name, SizeOf(Name), 0); {Fill with zeros to clear "random" bytes from the .Upd file.}
      Name := FileName.Text;
      Invalidate;
    end
end;

procedure TTabbedNotebookDlg.FileListBoxClick(Sender: TObject);
begin
  with FileListBox,Items do
  begin
    DeleteBtn.Enabled := ItemIndex > -1;
    RefreshCRCBtn.Enabled := ItemIndex > -1;
    MoveUpBtn.Enabled := ItemIndex > 0;
    MoveDownBtn.Enabled := (ItemIndex > -1) and (ItemIndex < Pred(Items.Count));
    if (ItemIndex > -1) and (Items.Count > 0) then
    begin
      NoVersionChecking.Enabled := PFileUpdateInfo(Objects[ItemIndex])^.FileProcessing in [FPUpdate,FPCopyOver];
      NoVersionChecking.Checked := NoVersionChecking.Enabled and
        (PFileUpdateInfo(Objects[ItemIndex])^.FileProcessing = FPCopyOver);
      FileName.Text := PFileUpdateInfo(Objects[ItemIndex])^.Name;
    end
    else
      NoVersionChecking.Enabled := False;
  end;
end;

procedure TTabbedNotebookDlg.BrowseButtonClick(Sender: TObject);
var
  OldDirNam,
  NewDirNam,
  FilNam  : TFileName;
  i,
  PathLength  : integer;
  CheckDirectories  : Boolean;
begin
  OldDirNam := OldDirectoryListBox.Directory;
  NewDirNam := NewDirectoryListBox.Directory;
  CheckDirectories := True;
  if OpenDialog.Execute then
  for i := 0 to Pred(OpenDialog.Files.Count) do
  begin
    FilNam := OpenDialog.Files[i];
    PathLength := 0;
    if (OldDirNam <> '') and (Pos(OldDirNam,FilNam) = 1) then
      PathLength := Length(OldDirNam)
    else if (NewDirNam <> '') and (Pos(NewDirNam,FilNam) = 1) then
      PathLength := Length(NewDirNam);
    if PathLength > 0 then
    begin
      Delete(FilNam,1,PathLength);
      if FilNam[1] = '\' then
        Delete(FilNam,1,1);
      if (CheckDirectories and not OldAndNewDirectoriesOK) then
        Exit;
      FileName.Text := FilNam;
      AddBtnClick(Sender); {Automatically add each file to the list}
      CheckDirectories := False;
    end
    else
      MessageDlg(Format(IDS_FileNotInDirectoryTreeFmt,[FilNam,OldDirNam]),mtError,[mbOk],0);
  end;
end;

function TTabbedNotebookDlg.OldAndNewDirectoriesOK : Boolean;
begin
  Result := (CompareText(OldDirectoryListBox.Directory,NewDirectoryListBox.Directory) <> 0)
    or (MessageDlg(IDS_IdenticalOldNewDirectories,mtConfirmation,[mbYes,mbNo],0) = mrYes);
end;

procedure TTabbedNotebookDlg.AddBtnClick(Sender: TObject);
var
  FileInfo  : PFileUpdateInfo;
begin
  if ((Sender = AddBtn) and not OldAndNewDirectoriesOK)
    or not CanAddFileNameToList then
      Exit;

  New(FileInfo);
  FillChar(FileInfo^,SizeOf(TFileUpdateInfo),0);
  FileInfo^.Name := FileName.Text;
  if DetermineRequiredProcessing(FileInfo^, IDS_FileNotAdded) then
  begin
    {Add the FileName and CRCs to the List}
    FileListBox.Items.AddObject('',TObject(FileInfo));
    AddBtn.Enabled := False;
  end
  else
    Dispose(FileInfo);
  FileListBoxClick(Sender);
end;

procedure TTabbedNotebookDlg.DeleteBtnClick(Sender: TObject);
begin
  with FileListBox,Items do
    if ItemIndex > -1 then
    begin
      Dispose(PFileUpdateInfo(Objects[ItemIndex]));
      Delete(ItemIndex);
    end;
  FileNameChange(Sender); {To adjust the Add button enabling}
  FileListBoxClick(Sender);
end;

procedure TTabbedNotebookDlg.MoveUpBtnClick(Sender: TObject);
var
  SelectedItem : integer;
begin
  with FileListBox,Items do
    if ItemIndex > 0 then
    begin
      SelectedItem := ItemIndex;
      Exchange(ItemIndex,Pred(ItemIndex));
      ItemIndex := Pred(SelectedItem);
      FileListBoxClick(Sender);
    end;
end;

procedure TTabbedNotebookDlg.MoveDownBtnClick(Sender: TObject);
var
  SelectedItem : integer;
begin
  with FileListBox,Items do
    if (ItemIndex > -1) and (ItemIndex < Pred(Count)) then
    begin
      SelectedItem := ItemIndex;
      Exchange(ItemIndex,Succ(ItemIndex));
      ItemIndex := Succ(SelectedItem);
      FileListBoxClick(Sender);
    end;
end;

procedure TTabbedNotebookDlg.RefreshCRCBtnClick(Sender: TObject);
begin
  with FileListBox,Items do
    if ItemIndex > -1 then
    begin
      DetermineRequiredProcessing(PFileUpdateInfo(Objects[ItemIndex])^, IDS_CRCNotCalculated);
      Invalidate;
    end;
end;

procedure TTabbedNotebookDlg.FileListBoxDrawItem(Control: TWinControl;
  Index: Integer; Rect: TRect; State: TOwnerDrawState);
begin
  with (Control as TListBox),Canvas,PFileUpdateInfo(Items.Objects[Index])^ do
  begin
    FillRect(Rect);
    FileProcessingImageList.Draw(Canvas, Rect.Left, Rect.Top,
      integer(FileProcessing));
    TextOut(Rect.Left + FileProcessingImageList.Width + 2, Rect.Top, Name);
  end;
end;

procedure TTabbedNotebookDlg.LoadBtnClick(Sender: TObject);
var
  SaveTitle,
  SaveFileName: string[127];
  SaveFilter  : string;
  SessionFile : TIniFile;
  SectionLabel: string[63];
  Directory   : TFileName;
  FileInfo    : PFileUpdateInfo;
  WantCopyOver: Boolean;
  i, NumFiles : integer;
begin
  SaveFileName := OpenDialog.FileName;
  SaveFilter := OpenDialog.Filter;
  SaveTitle := OpenDialog.Title;
  OpenDialog.Filter := IDS_UpdateSessionFilter;
  OpenDialog.Title := IDS_OpenSessionTitle;
  OpenDialog.FileName := SessionFileName;
  if OpenDialog.Execute then
  begin
    SessionFileName := OpenDialog.FileName;
    SessionFile := TIniFile.Create(SessionFileName);

    try
      SectionLabel := IDS_NewFilesSection;
      Directory := SessionFile.ReadString(SectionLabel,IDS_DirectoryLabel,NewDirectoryListBox.Directory);
      if FileGetAttr(Directory) >= 0 then
        NewDirectoryListBox.Directory := Directory;
      NewVersion.Text := SessionFile.ReadString(SectionLabel,IDS_VersionInfoLabel,NewVersion.Text);

      SectionLabel := IDS_OldFilesSection;
      Directory := SessionFile.ReadString(SectionLabel,IDS_DirectoryLabel,OldDirectoryListBox.Directory);
      if FileGetAttr(Directory) >= 0 then
        OldDirectoryListBox.Directory := Directory;
      OldVersion.Text := SessionFile.ReadString(SectionLabel,IDS_VersionInfoLabel,OldVersion.Text);

      SectionLabel := IDS_OptionsLabel;
      FailureMode.ItemIndex := SessionFile.ReadInteger(SectionLabel,IDS_UpdateAllOrNothing,0);

      ClearFileList;
      SectionLabel := IDS_FilesListLabel;
      NumFiles := SessionFile.ReadInteger(SectionLabel,IDS_NumberFilesLabel,0);
      for i := 1 to NumFiles do
      begin
        New(FileInfo);
        FillChar(FileInfo^,SizeOf(TFileUpdateInfo),0);
        FileInfo^.Name := SessionFile.ReadString(SectionLabel,Format(IDS_FileNameLabelFmt,[i]),'');
        if SessionFile.ReadBool(SectionLabel,Format(IDS_NoVersionCheckingLabelFmt,[i]),False) then
          FileInfo^.FileProcessing := FPCopyOver;
        FileListBox.Items.AddObject('',TObject(FileInfo));
      end;

      RefreshFileProcessingIndicators;

    finally
      SessionFile.Free;
    end;

  end;
  OpenDialog.FileName := SaveFileName;
  OpenDialog.Filter := SaveFilter;
  OpenDialog.Title := SaveTitle;
end;

procedure TTabbedNotebookDlg.SaveBtnClick(Sender: TObject);
var
  SessionFile : TIniFile;
  SectionLabel: string[63];
  i           : integer;
begin
  SaveDialog.FileName := SessionFileName;
  if SaveDialog.Execute then
  begin
    SessionFileName := SaveDialog.FileName;
    SessionFile := TIniFile.Create(SessionFileName);

    try
      SectionLabel := IDS_OldFilesSection;
      SessionFile.WriteString(SectionLabel,IDS_DirectoryLabel,OldDirectoryListBox.Directory);
      SessionFile.WriteString(SectionLabel,IDS_VersionInfoLabel,OldVersion.Text);

      SectionLabel := IDS_NewFilesSection;
      SessionFile.WriteString(SectionLabel,IDS_DirectoryLabel,NewDirectoryListBox.Directory);
      SessionFile.WriteString(SectionLabel,IDS_VersionInfoLabel,NewVersion.Text);

      SectionLabel := IDS_FilesListLabel;
      SessionFile.EraseSection(SectionLabel);
      SessionFile.WriteInteger(SectionLabel,IDS_UpdateAllOrNothing,FailureMode.ItemIndex);
      with FileListBox.Items do
      begin
        SessionFile.WriteInteger(SectionLabel,IDS_NumberFilesLabel,Count);
        for i := 1 to Count do
        with PFileUpdateInfo(Objects[Pred(i)])^ do
        begin
          SessionFile.WriteString(SectionLabel,Format(IDS_FileNameLabelFmt,[i]),Name);
          if FileProcessing = FPCopyOver then
            SessionFile.WriteBool(SectionLabel,Format(IDS_NoVersionCheckingLabelFmt,[i]),True);
        end;
      end;

    finally
      SessionFile.Free;
    end;

  end;
end;

procedure TTabbedNotebookDlg.CloseBtnClick(Sender: TObject);
begin
  Close;
end;

function HasExtension(const Name : string; var DotPos : Word) : Boolean;
  {-Return whether and position of extension separator dot in a pathname}
var
  I : Word;
begin
  DotPos := 0;
  for I := Length(Name) downto 1 do
    if (Name[I] = '.') and (DotPos = 0) then
      DotPos := I;
  HasExtension := (DotPos > 0) and (Pos('\', Copy(Name, Succ(DotPos), 64)) = 0);
end;

function ForceExtension(const Name, Ext : string) : string;
  {-Return a pathname with the specified extension attached}
var
  DotPos : Word;
begin
  if HasExtension(Name, DotPos) then
    ForceExtension := Copy(Name, 1, DotPos)+Ext
  else
    ForceExtension := Name+'.'+Ext;
end;

function JustFilename(const PathName : string) : string;
  {-Return just the filename of a pathname}
var
  I : Word;
begin
  I := Succ(Word(Length(PathName)));
  repeat
    Dec(I);
  until (PathName[I] in ['\', ':', #0]) or (I = 0);
  JustFilename := Copy(PathName, Succ(I), 64);
end;

procedure TTabbedNotebookDlg.MakeBtnClick(Sender: TObject);
var
  i : integer;
  VersionInfo   : string[31];
  SavePosition,
  CRC,
  DiffSize      : LongInt;
  FileInfo      : PFileUpdateInfo;
begin
  {Create Update file}
  UpdateStream := nil;
  DiffStream := nil;
  UpdAccumulateStream := nil;
  if SessionFileName = '' then
    SaveDialog.FileName := 'Default.Upd'
  else
    SaveDialog.FileName := ForceExtension(JustFileName(SessionFileName),'Upd');
  SaveDialog.FilterIndex := 2;
  try
    if SaveDialog.Execute then
    begin
      UpdateStream  := TFileStream.Create(SaveDialog.FileName,fmCreate);
      UpdateStream.WriteBuffer(UpdateFileHeader[1],Length(UpdateFileHeader));
      FillChar(VersionInfo,SizeOf(VersionInfo),0); {Fill with zeros to clear "random" bytes from the .Upd.}
      VersionInfo := OldVersion.Text;
      UpdateStream.WriteBuffer(VersionInfo,SizeOf(VersionInfo));
      FillChar(VersionInfo,SizeOf(VersionInfo),0); {Fill with zeros to clear "random" bytes from the .Upd.}
      VersionInfo := NewVersion.Text;
      UpdateStream.WriteBuffer(VersionInfo,SizeOf(VersionInfo));
      
      UpdateAllOrNothing := FailureMode.ItemIndex = 1;
      UpdateStream.WriteBuffer(UpdateAllOrNothing,SizeOf(UpdateAllOrNothing));

      DiffStream := TMemoryStream.Create;
      UpdAccumulateStream := TMemoryStream.Create;

      RefreshFileProcessingIndicators;
      ProgressDlg.Total := 0;
      with FileListBox.Items do
        for i := 0 to Pred(Count) do
        with PFileUpdateInfo(Objects[i])^ do
          if FileProcessing <> FPNothing then
            ProgressDlg.Total := ProgressDlg.Total + NewID.LoSize;
      ProgressDlg.Completed := 0;
      ProgressDlg.Show;

      with FileListBox.Items do
      for i := 0 to Pred(Count) do
      if PFileUpdateInfo(Objects[i])^.FileProcessing <> FPNothing then
      begin
        OldStream := nil;
        NewStream := nil;
        FileInfo  := PFileUpdateInfo(Objects[i]);
        try
          ProgressDlg.FileNameLabel.Caption := GetFullPathTo(FileInfo^.Name,NewDirectoryListBox);
          ProgressDlg.CurrentCompleted := 0;

          UpdAccumulateStream.WriteBuffer(FileInfo^.Name, SizeOf(TFileUpdateInfo));
          Case FileInfo^.FileProcessing of

            FPCopyOver,
            FPAdd : begin
              NewStream := TFileStream.Create(GetFullPathTo(FileInfo^.Name,NewDirectoryListBox),fmOpenRead);
              UpdAccumulateStream.CopyFrom(NewStream,0);
            end;

            FPUpdate : begin
              OldStream := TFileStream.Create(GetFullPathTo(FileInfo^.Name,OldDirectoryListBox),fmOpenRead);
              NewStream := TFileStream.Create(GetFullPathTo(FileInfo^.Name,NewDirectoryListBox),fmOpenRead);
              DiffStream.Clear;
              Screen.Cursor := crHourglass;
              DiffStreamCompress(NewStream,OldStream,DiffStream,UpdateCurrentProgress,1000);
              DiffSize := DiffStream.Size + SizeOf(CRC);
              UpdAccumulateStream.WriteBuffer(DiffSize,SizeOf(DiffSize));
              DiffStream.Position := 0;
              UpdAccumulateStream.CopyFrom(DiffStream,DiffStream.Size);
              DiffStream.Position := 0;
              CRC := CalculateStreamCRC(DiffStream);
              UpdAccumulateStream.WriteBuffer(CRC,SizeOf(CRC));
            end;

          end; { of Case FileInfo^.FileProcessing of }
        finally
          OldStream.Free;
          NewStream.Free;
          Screen.Cursor := crDefault;
          ProgressDlg.Completed := ProgressDlg.Completed + FileInfo^.NewID.LoSize;
        end;
      end;  { of for i := 0 to Pred(FileListBox.ItemsCount) do}
      UpdAccumulateStream.Position := 0;
      SavePosition := UpdateStream.Position;
      UpdateStream.WriteBuffer(CRC,SizeOf(CRC)); {Make a place holder for the compressed CRC.}

      LHACompress(UpdAccumulateStream, UpdateStream); {Compress the stream}

      UpdateStream.Position := SavePosition + SizeOf(CRC); {Calculate the compressed CRC.}
      CRC := CalculateStreamCRC(UpdateStream);
      UpdateStream.Position := SavePosition;
      UpdateStream.WriteBuffer(CRC,SizeOf(CRC)); {Now write the actual compressed CRC.}
    end;
  finally
    ProgressDlg.Hide;
    SaveDialog.FilterIndex := 1;
    UpdateStream.Free;
    DiffStream.Free;
    UpdAccumulateStream.Free;
  end;
end;

procedure TTabbedNotebookDlg.AboutBtnClick(Sender: TObject);
begin
  AboutBox := TAboutBox.Create(Self);
  AboutBox.ShowModal;
  AboutBox.Free;
  AboutBox := nil;
end;

procedure TTabbedNotebookDlg.ContolEnter(Sender: TObject);
var
  Control: TWinControl;
begin
  if Sender is TWinControl then
  begin
    Control := TWinControl(Sender);
    while (Control <> nil) and (Control.HelpContext = 0) do
      Control := Control.Parent;
    if Control <> nil then
      HelpBtn.HelpContext := Control.HelpContext;
  end;
end;

procedure TTabbedNotebookDlg.PageControl1Change(Sender: TObject);
begin
  HelpBtn.HelpContext := PageControl1.ActivePage.HelpContext;
end;

procedure TTabbedNotebookDlg.NoVersionCheckingClick(Sender: TObject);
begin
  with FileListBox,Items do
  if (ItemIndex > -1) and (PFileUpdateInfo(Objects[ItemIndex])^.FileProcessing in [FPUpdate,FPCopyOver]) then
  begin
    if NoVersionChecking.Checked then
      PFileUpdateInfo(Objects[ItemIndex])^.FileProcessing := FPCopyOver
    else
      PFileUpdateInfo(Objects[ItemIndex])^.FileProcessing := FPUpdate;
    Invalidate;
  end;
end;

end.
