unit DoUpdate;

interface
uses
  Windows, SysUtils, Classes, aCRC32, common, LH5unit,ADiff, Progress,
  LogForm, forms, Controls, UpdateGlobals, KSAutoupdate;

procedure ApplyUpdateFile(const ChangesFileName: string);


implementation

type
  EDifferenceError = class(Exception);
  EHarmlessDifferenceError = class(EDifferenceError);
  EUpdateFileError = class(Exception);

  TUndoUpdateInfo = class
    FileName,
    UndoFileName  : TFileName;
  end;

{$IFDEF VER90}  {D2 Doesn't know about resourcestring}
const
{$ELSE}
resourcestring
{$ENDIF}
  IDS_UpdFileNotFoundFmt = 'Can''t find file containing changes'#13#10'   %s';
  IDS_CannotReadUpdateFileFmt = 'Error reading the changes in file'#13#10'   %s';
  IDS_InvalidUpdateFileFmt = 'The file'#13#10'   %s'#13#10'is not an Update File supported by this version of Update.'#13#10'Expecting ID label "%s" but found "%s" instead.';
  IDS_UpdatesNotAppliedFmt = 'Updates not applied.'#13#10'%s';
  IDS_SkippingFileFmt = 'Skipping file: %s'#13#10'   %s'#13#10;
  IDS_FileNotFound = 'File not found.';
  IDS_CannotOpenFile = '%s'#13#10'File may be in use by another process or may be marked read-only.';
  IDS_FileAlreadyUpdatedFmt = 'File already updated to version "%s"';
  IDS_FileWrongVersionFmt = 'File is not expected version "%s"';
  IDS_InvalidChangesForFile = 'Invalid changes. The Update File may be corrupt.';
  IDS_ErrorApplyingChanges = 'Error applying changes.';
  IDS_UpdatedFileFailsCRC = 'Updated file fails CRC check.';
  IDS_CannotDeleteOldFile = 'Cannot delete old file. The file may be read-only. If so, change the attribute and try again.';
  IDS_CannotRenameUpdatedFile = 'Cannot replace file. The file may be read-only. If so, change the attribute and try again.';
  IDS_UpdatedFileFmt = 'Updated file to version "%s":'#13#10'   %s'#13#10;
  IDS_DeletedFileFmt = 'Deleted obsolete file:'#13#10'   %s'#13#10;
  IDS_AddedFileFmt = 'Added new file:'#13#10'   %s'#13#10;
  IDS_AddedFileFailsCRC = 'Added file fails CRC check.';
  IDS_CantMakeTemporaryFile = 'Cannot create a unique name for a temporary file.';
  IDS_UndoingChanges = 'Undoing changes because this update must be "all or nothing."';
  IDS_CannotDeleteChangedFileFmt = 'Cannot delete changed version of file'#13#10'   %s';
  IDS_CannotRestoreOriginalFileFmt = 'Cannot restore original file'#13#10'   %s'#13#10'which had been temporarily renamed'#13#10'   %s';
  IDS_NoMessages = 'There are no messages to report.';


{*************************************************************************}
function GetUniqueFileName : TFileName;
// Return a unique name for a (temporary) file in the TargetDir
begin
  SetLength(Result, MAX_PATH);
  if GetTempFileName(PChar(TargetDir),PChar('Update'),0,PChar(Result)) = 0 then
    raise EDifferenceError.Create(IDS_CantMakeTemporaryFile);

  SetLength(Result, StrLen(PChar(Result)));

  // GetTempFileName insists on creating the file in order to verify that it is unique
  // So delete it now.
  SysUtils.DeleteFile(Result);
end;


{*************************************************************************}
Procedure DeleteFileReversibly(const FileName: TFileName; UndoUpdateActions: TList);
var
  TempFileName  : TFileName;
begin
  if UndoUpdateActions <> nil then
  // Rename the file so the "delete" action is reversible
  begin
    // Add an "undo" information record to the list regardless of whether the file exists
    UndoUpdateActions.Add(TUndoUpdateInfo.Create);

    //If the actual file exists, rename it
    if FileExists(FileName) then
    begin
      // Come up with a temporary file name we can use for doing a reversible renaming
      TempFileName := GetUniqueFileName;

      if not RenameFile(FileName,TempFileName) then
        raise EDifferenceError.Create(IDS_CannotDeleteOldFile);
      // If we get here, rename was successful so we can save the "undo" information
      with TUndoUpdateInfo(UndoUpdateActions.Items[UndoUpdateActions.Count-1]) do
        UndoFileName := TempFileName;
    end;
  end
  else //Delete the file in a nonreversible manner
    if FileExists(FileName) and not SysUtils.DeleteFile(FileName) then
      raise EDifferenceError.Create(IDS_CannotDeleteOldFile);
end;

{*************************************************************************}
procedure AddFileToTargetDir(const NewFileName, OldFileName: TFileName; var NextUpdPosition: longint;
         const FileInfo: TFileUpdateInfo; UndoUpdateActions: TList);
var
  UniqueFileID  : TUniqueFileIDInfo;
begin
  try
    NextUpdPosition := UpdAccumulateStream.Position + FileInfo.NewID.LoSize;
    //create new file with temp name
    NewStream := TFileStream.Create(NewFileName,fmCreate);
    NewStream.CopyFrom(UpdAccumulateStream, FileInfo.NewID.LoSize);
    FileSetDate(NewStream.Handle,FileInfo.NewDate);
    UniqueFileID.LoSize := GetFileSize(NewStream.Handle,@UniqueFileID.HiSize);
    NewStream.Position := 0;
    UniqueFileID.CRC := CalculateStreamCRC(NewStream);
    NewStream.Free;
    NewStream := nil;

  except
    on E: Exception do
      raise EDifferenceError.CreateFmt(IDS_CannotOpenFile,[E.Message]);
  end;

  if not CompareMem(@UniqueFileID, @FileInfo.NewID, SizeOf(UniqueFileID)) then
    raise EDifferenceError.Create(IDS_AddedFileFailsCRC);

  // if we made it this far, then we're about to do something that may need to be undone
  DeleteFileReversibly(OldFileName, UndoUpdateActions);

  // Rename temp file to actual filename
  if not RenameFile(NewFileName,OldFileName) then
    raise EDifferenceError.Create(IDS_CannotRenameUpdatedFile);

  // If we get here, rename was successful so we can save the "undo" information
  if UndoUpdateActions <> nil then
    with TUndoUpdateInfo(UndoUpdateActions.Items[UndoUpdateActions.Count-1]) do
      FileName := OldFileName;

  UpdLogForm.LogMessage.Lines.Add(Format(IDS_AddedFileFmt,[OldFileName]));
end;

{*************************************************************************}
Procedure DeleteFileFromTargetDir(const OldFileName: TFileName;
        var NextUpdPosition: longint; UndoUpdateActions: TList);
begin

  NextUpdPosition := UpdAccumulateStream.Position;

  //If file doesn't exist raise a "harmless" error (so UpdateError won't be
  //set to True and UndoChanges won't be executed. Otherwise delete it
  if not FileExists(OldFileName) then
    raise EHarmlessDifferenceError.Create(IDS_FileNotFound)
  else begin
    // if we made it this far, then we're about to do something that may need to be undone
    DeleteFileReversibly(OldFileName, UndoUpdateActions);
    if UndoUpdateActions <> nil then
      // If we get here, rename was successful so we can save the "undo" information
      with TUndoUpdateInfo(UndoUpdateActions.Items[UndoUpdateActions.Count-1]) do
        FileName := OldFileName;
  end;

  UpdLogForm.LogMessage.Lines.Add(Format(IDS_DeletedFileFmt,[OldFileName]));

end;

{*************************************************************************}
procedure UpdateFileInTargetDir(const NewFileName, OldFileName: TFileName;
        var NextUpdPosition: longint; DiffSize, CRC: longint;
        const FileInfo: TFileUpdateInfo; const NewVersion, OldVersion: string;
        UndoUpdateActions: TList);
var
  UniqueFileID  : TUniqueFileIDInfo;
begin
  //If file doesn't exist raise an error
  if not FileExists(OldFileName) then
    raise EDifferenceError.Create(IDS_FileNotFound);

  try
    CalculateUniqueFileIDInfo(OldFileName, UniqueFileID, nil);
    OldStream := TFileStream.Create(OldFileName,fmOpenRead);
    //create new updated file with temp name
    NewStream := TFileStream.Create(NewFileName,fmCreate);
  except
    on E: Exception do
      raise EDifferenceError.CreateFmt(IDS_CannotOpenFile,[E.Message]);
  end;

  if CompareMem(@UniqueFileID, @FileInfo.NewID, SizeOf(UniqueFileID)) then
    raise EDifferenceError.CreateFmt(IDS_FileAlreadyUpdatedFmt,[NewVersion]);

  if not CompareMem(@UniqueFileID, @FileInfo.OldID, SizeOf(UniqueFileID)) then
    raise EDifferenceError.CreateFmt(IDS_FileWrongVersionFmt,[OldVersion]);

  try
    {$IFDEF DebugLog}
    LogMsg := 'ApplyUpdateFile 5: ' + IntToStr(CRC) + #13#10;
    LogStream.WriteBuffer(LogMsg[1], Length(LogMsg));
    {$ENDIF}

    DiffStream.Clear;
    UpdAccumulateStream.ReadBuffer(DiffSize,SizeOf(DiffSize));

    {$IFDEF DebugLog}
    LogMsg := 'ApplyUpdateFile 6: ' + IntToStr(DiffSize) + #13#10;
    LogStream.WriteBuffer(LogMsg[1], Length(LogMsg));
    {$ENDIF}

    NextUpdPosition := UpdAccumulateStream.Position + DiffSize;
    DiffStream.CopyFrom(UpdAccumulateStream,DiffSize-SizeOf(CRC));
    UpdAccumulateStream.ReadBuffer(CRC,SizeOf(CRC));
  except
    raise EDifferenceError.Create(IDS_InvalidChangesForFile);
  end;


  DiffStream.Position := 0;

  if CRC <> CalculateStreamCRC(DiffStream) then
    raise EDifferenceError.Create(IDS_InvalidChangesForFile);

  try
    DiffStreamExtract(DiffStream,OldStream,NewStream,UpdateCurrentProgress);
  except
    raise EDifferenceError.Create(IDS_ErrorApplyingChanges);
  end;

  FileSetDate(NewStream.Handle,FileInfo.NewDate);
  OldStream.Free;
  NewStream.Free;
  OldStream := nil;
  NewStream := nil;

  try
    CalculateUniqueFileIDInfo(NewFileName, UniqueFileID, nil);
  except
    on E: Exception do
      raise EDifferenceError.CreateFmt(IDS_CannotOpenFile,[E.Message]);
  end;


  if not CompareMem(@UniqueFileID, @FileInfo.NewID, SizeOf(UniqueFileID)) then
    raise EDifferenceError.Create(IDS_UpdatedFileFailsCRC);

  // if we made it this far, then we're about to do something that may need to be undone
  DeleteFileReversibly(OldFileName, UndoUpdateActions);

  {$IFDEF DebugLog}
  LogMsg := 'ApplyUpdateFile 7: ' + OldFileName + #13#10;
  LogStream.WriteBuffer(LogMsg[1], Length(LogMsg));
  {$ENDIF}

  //Rename temp file to actual filename
  if not RenameFile(NewFileName,OldFileName) then
    raise EDifferenceError.Create(IDS_CannotRenameUpdatedFile);

  // If we get here, rename was successful so we can save the "undo" information
  if UndoUpdateActions <> nil then
    with TUndoUpdateInfo(UndoUpdateActions.Items[UndoUpdateActions.Count-1]) do
      FileName := OldFileName;

  UpdLogForm.LogMessage.Lines.Add(Format(IDS_UpdatedFileFmt,[NewVersion,OldFileName]));
end;

{*************************************************************************}


Procedure ProcessOneFile(var OldFileName: TFileName; const NewFileName: TFileName;
                const FileInfo: TFileUpdateInfo; var NextUpdPosition: longint;
                   DiffSize, CRC: longint; const NewVersion, OldVersion: string;
                   UndoUpdateActions: TList);
begin
  try
    OldStream := nil;
    NewStream := nil;

    OldFileName := TargetDir + fileInfo.Name;

    ProgressDlg.FileNameLabel.Caption := OldFileName;
    ProgressDlg.CurrentCompleted := 0;
    ProgressDlg.Completed := UpdAccumulateStream.Position;

    {$IFDEF DebugLog}
    LogMsg := 'ApplyUpdateFile 4: ' + OldFileName + #13#10;
    LogStream.WriteBuffer(LogMsg[1], Length(LogMsg));
    {$ENDIF}

    Case FileInfo.FileProcessing of
      FPDelete : DeleteFileFromTargetDir(OldFileName, NextUpdPosition, UndoUpdateActions);

      FPAdd    : AddFileToTargetDir(NewFileName, OldFileName, NextUpdPosition, FileInfo, UndoUpdateActions);

      FPCopyOver:
        begin
          DeleteFileFromTargetDir(OldFileName, NextUpdPosition, UndoUpdateActions);
          AddFileToTargetDir(NewFileName, OldFileName, NextUpdPosition, FileInfo, UndoUpdateActions);
        end;

      FPUpdate : UpdateFileInTargetDir(NewFileName, OldFileName, NextUpdPosition, DiffSize,
                            CRC, FileInfo, NewVersion, OldVersion, UndoUpdateActions);

    end; //Case FileInfo.FileProcessing

  finally
    OldStream.Free;
    NewStream.Free;
    SysUtils.DeleteFile(NewFileName); //Delete temp file
  end;

end;


{*************************************************************************}


Procedure UndoChanges(UndoUpdateActions: TList);
var
  i : integer;
begin
  UpdLogForm.LogMessage.Lines.Add(IDS_UndoingChanges);

  for i := 0 to UndoUpdateActions.Count-1 do
  with TUndoUpdateInfo(UndoUpdateActions.Items[i]) do
  begin
    if (FileName <> '') and FileExists(FileName)
      and not SysUtils.DeleteFile(FileName) then
        UpdLogForm.LogMessage.Lines.Add(Format(IDS_CannotDeleteChangedFileFmt,[FileName]));

    // try to undo a reversible deletion
    if (UndoFileName <> '') and not RenameFile(UndoFileName,FileName) then
      UpdLogForm.LogMessage.Lines.Add(Format(IDS_CannotRestoreOriginalFileFmt,[FileName,UndoFileName]));
  end;
end;

{*************************************************************************}


Procedure ProcessUpdateFile(const ChangesFileName: string);
var
  FileHeader        : string[15];
  OldVersion,
  NewVersion        : string[31];
  NextUpdPosition,
  CRC, DiffSize     : LongInt;
  Attribute, i      : integer;
  NewFileName,
  OldFileName       : TFileName;
  FileInfo          : TFileUpdateInfo;
  UndoUpdateActions : TList;  // List of TUndoUpdateInfo used when UpdateAllOrNothing = True

begin
  DiffSize := 0;
  NextUpdPosition := 0;
  UpdateStream := nil;
  DiffStream := nil;
  UpdAccumulateStream := nil;
  UndoUpdateActions := nil;

  try //Protect Streams

    if not FileExists(ChangesFileName) then
      raise EUpdateFileError.CreateFmt(IDS_UpdFileNotFoundFmt,[ChangesFileName]);


    try //read UpdateFile header
      UpdateStream  := TFileStream.Create(ChangesFileName,fmOpenRead);
      FileHeader := UpdateFileHeader; {This copies the length byte}
      UpdateStream.ReadBuffer(FileHeader[1],Length(UpdateFileHeader));
    except
      raise EUpdateFileError.CreateFmt(IDS_CannotReadUpdateFileFmt,[ChangesFileName]);
    end;


    //Check UpdateFile version
    if (FileHeader <> UpdateFileHeader) and (FileHeader <> AlsoAllowedUpdateFileHeader) then
      raise EUpdateFileError.CreateFmt(IDS_InvalidUpdateFileFmt,[ChangesFileName,UpdateFileHeader,FileHeader]);


    try //Read Size of old and new file and UpdateAllOrNothing from UpdateFile
      UpdateStream.ReadBuffer(OldVersion,SizeOf(OldVersion));
      UpdateStream.ReadBuffer(NewVersion,SizeOf(NewVersion));
      UpdateStream.ReadBuffer(UpdateAllOrNothing,SizeOf(UpdateAllOrNothing));
      {$IFDEF DebugLog}
      LogMsg := 'ApplyUpdateFile 2: ' + '"' + OldVersion + '", "' + NewVersion + '"'#13#10;
      LogStream.WriteBuffer(LogMsg[1], Length(LogMsg));
      {$ENDIF}
    except
      raise EUpdateFileError.CreateFmt(IDS_CannotReadUpdateFileFmt,[ChangesFileName]);
    end;

    //Make a unique name for a (temporary) updated file
    NewFileName := GetUniqueFileName;

    {$IFDEF DebugLog}
    LogMsg := 'ApplyUpdateFile 3: ' + NewFileName + #13#10;
    LogStream.WriteBuffer(LogMsg[1], Length(LogMsg));
    {$ENDIF}


    //test CRC for old file (read from UpdateFile) against calculated CRC for old file
    UpdateStream.ReadBuffer(CRC,SizeOf(CRC));
    if CRC <> CalculateStreamCRC(UpdateStream) then
      raise EUpdateFileError.Create(IDS_InvalidChangesForFile);


    DiffStream := TMemoryStream.Create;
    UpdAccumulateStream := TMemoryStream.Create;
    LHAExpand(UpdateStream, UpdAccumulateStream);
    UpdAccumulateStream.Position := 0;


    ProgressDlg.Total := UpdAccumulateStream.Size;
    ProgressDlg.Show;

    // Maintain an undo list in case of any errors
    if UpdateAllOrNothing then
      UndoUpdateActions := TList.Create;

    //process every single file inside the main update file
    while UpdAccumulateStream.Read(FileInfo, SizeOf(TFileUpdateInfo)) = SizeOf(TFileUpdateInfo) do
    begin
      try //catch errors in single files inside the main update file
        ProcessOneFile(OldFileName, NewFileName, FileInfo, NextUpdPosition, DiffSize, CRC, NewVersion, OldVersion, UndoUpdateActions);
      except
        on E: EDifferenceError do
        begin
          if not (E is EHarmlessDifferenceError) then
            UpdateError := True; //KS
          {$IFDEF DebugLog}
          LogMsg := 'ApplyUpdateFile except: ' + E.Message + #13#10;
          LogStream.WriteBuffer(LogMsg[1], Length(LogMsg));
          {$ENDIF}
          UpdLogForm.LogMessage.Lines.Add(Format(IDS_SkippingFileFmt,[OldFileName,E.Message]));

          // If this is an all-or-nothing update, undo the changes, and raise an exception to abort processing
          if (UndoUpdateActions <> nil) and not (E is EHarmlessDifferenceError) then
          begin
            UndoChanges(UndoUpdateActions);
            raise EUpdateFileError.Create(''); // Don't proceed to next file
          end;
        end;
      end; // Try - except


      if NextUpdPosition = 0 then
      begin
        if DiffSize = 0 then
          UpdAccumulateStream.ReadBuffer(DiffSize,SizeOf(DiffSize));
        NextUpdPosition := UpdAccumulateStream.Position + DiffSize;
      end;


      UpdAccumulateStream.Position := NextUpdPosition;
      DiffSize := 0;
      NextUpdPosition := 0;
    end; // While

    // Update was successful, so we can delete all temporary files in the undo list
    if UndoUpdateActions <> nil then
      for i := 0 to UndoUpdateActions.Count-1 do
      with TUndoUpdateInfo(UndoUpdateActions.Items[i]) do
        if (UndoFileName <> '') then
        begin
          Attribute := FileGetAttr(UndoFileName);
          if Attribute <> -1 then
            FileSetAttr(UndoFileName, Attribute and not faReadOnly);
          SysUtils.DeleteFile(UndoFileName);
        end;


  finally
    if UndoUpdateActions <> nil then
      for i := 0 to UndoUpdateActions.Count-1 do
        TUndoUpdateInfo(UndoUpdateActions.Items[i]).Free;
    UndoUpdateActions.Free;
    UpdAccumulateStream.Free;
    UpdateStream.Free;
    DiffStream.Free;
  end;

end;



{*************************************************************************}


procedure ApplyUpdateFile(const ChangesFileName: string);
{$IFDEF DebugLog}
var
  i   : integer;
{$ENDIF}
begin

  {$IFDEF DebugLog}
  LogMsg := 'ApplyUpdateFile 1: ' + ChangesFileName + #13#10;
  LogStream.WriteBuffer(LogMsg[1], Length(LogMsg));
  {$ENDIF}

  UpdLogForm.LogMessage.Lines.Clear;

  try //protects log writing and Update finalization
    Screen.Cursor := crHourglass;

    try //Catch errors in main Update file
      ProcessUpdateFile(ChangesFileName);
    except
      on E: EUpdateFileError do
      begin
        UpdateError := True; //KS
        UpdLogForm.LogMessage.Lines.Add(Format(IDS_UpdatesNotAppliedFmt,[E.Message]));
        {$IFDEF DebugLog}
        LogMsg := 'ApplyUpdateFile outer except: ' + E.Message + #13#10;
        LogStream.WriteBuffer(LogMsg[1], Length(LogMsg));
        {$ENDIF}
      end;
    end;

  finally
    {$IFDEF DebugLog}
    LogMsg := 'ApplyUpdateFile 8: '#13#10;
    LogStream.WriteBuffer(LogMsg[1], Length(LogMsg));
    {$ENDIF}

    Screen.Cursor := crDefault;
    ProgressDlg.Hide;
    {Now show the log messages...}
    if UpdLogForm.LogMessage.Lines.Count = 0
      then UpdLogForm.LogMessage.Lines.Add(IDS_NoMessages);

    {$IFDEF DebugLog}
    for i := 0 to Pred(UpdLogForm.LogMessage.Lines.Count) do
    begin
      LogMsg := 'ApplyUpdateFile 9: ' + IntToStr(i) + '"' + UpdLogForm.LogMessage.Lines[i] + '"'#13#10;
      LogStream.WriteBuffer(LogMsg[1], Length(LogMsg));
    end;
    {$ENDIF}

    if itsAutoUpdate
      then AutoUpdateFinish(ChangesFileName)
      else UpdLogForm.ShowModal;

  end;

end;


end.

