unit LZHBlob;
{---------------------------------------------
    
    TLZHCompressor
----------------------------------------------}
interface

uses SysUtils, Windows, Messages, Classes, Controls, Forms, Graphics, Menus,
	StdCtrls, ExtCtrls, DB, DBTables, Mask, Buttons, DBCtrls, ClipBrd, Lzh_op;

type
	{      }
	TProgressEvent = procedure(Sender : TObject; Position, Size : integer) of object;
   
	{    Memo- }
  
  TCDBMemo = class(TCustomMemo)
  private

    { Private declarations }
    FDataLink: TFieldDataLink;
    FAutoDisplay: Boolean;
    FFocused: Boolean;
    FMemoLoaded: Boolean;
    FPaintControl: TPaintControl;
    {---[   ]--------}
    Compressor : TLZHCompressor;
    InStream, OutStream : TMemoryStream;
    FOnProgress : TProgressEvent;
    procedure CompressStream;
    procedure DecompressStream;
    procedure PutData(Sender:TObject; var DTA : BufType; NBytes : WORD;
    	var Bytes_Got : WORD);
    procedure GetData(Sender:TObject; var DTA : BufType; NBytes : WORD;
    	var Bytes_Got : WORD);
    function GetInBufferSize : integer;
    function GetOutBufferSize : integer;
    procedure SetInBufferSize( NewValue : integer );
    procedure SetOutBufferSize( NewValue : integer );
    procedure SetProgressEvent( NewValue : TProgressEvent);
    {------[								]------}
    procedure DataChange(Sender: TObject);
    procedure EditingChange(Sender: TObject);
    function GetDataField: string;
    function GetDataSource: TDataSource;
    function GetField: TField;
    function GetReadOnly: Boolean;
    procedure SetDataField(const Value: string);
    procedure SetDataSource(Value: TDataSource);
    procedure SetReadOnly(Value: Boolean);
    procedure SetAutoDisplay(Value: Boolean);
    procedure SetFocused(Value: Boolean);
    procedure UpdateData(Sender: TObject);
    procedure WMCut(var Message: TMessage); message WM_CUT;
    procedure WMPaste(var Message: TMessage); message WM_PASTE;
    procedure CMEnter(var Message: TCMEnter); message CM_ENTER;
    procedure CMExit(var Message: TCMExit); message CM_EXIT;
    procedure WMLButtonDblClk(var Message: TWMLButtonDblClk); message WM_LBUTTONDBLCLK;
    procedure WMPaint(var Message: TWMPaint); message WM_PAINT;
    procedure CMGetDataLink(var Message: TMessage); message CM_GETDATALINK;
  protected
    { Protected declarations }
    procedure Change; override;
    procedure KeyDown(var Key: Word; Shift: TShiftState); override;
    procedure KeyPress(var Key: Char); override;
    procedure Notification(AComponent: TComponent;
      Operation: TOperation); override;
    procedure WndProc(var Message: TMessage); override;
  public
    { Public declarations }
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure LoadMemo;
    property Field: TField read GetField;
  published
    { Published declarations }
    property Align;
    property Alignment;
    property AutoDisplay: Boolean read FAutoDisplay write SetAutoDisplay default True;
    property BorderStyle;
    property InBufferSize : Integer read GetInBufferSize write SetInBufferSize
    		default $400;
    property OutBufferSize : Integer read GetOutBufferSize write SetOutBufferSize
    		default $400;
    property Color;
    property Ctl3D;
    property DataField: string read GetDataField write SetDataField;
    property DataSource: TDataSource read GetDataSource write SetDataSource;
    property DragCursor;
    property DragMode;
    property Enabled;
    property Font;
    property MaxLength;
    property ParentColor;
    property ParentCtl3D;
    property ParentFont;
    property ParentShowHint;
    property PopupMenu;
    property ReadOnly: Boolean read GetReadOnly write SetReadOnly default False;
    property ScrollBars;
    property ShowHint;
    property TabOrder;
    property TabStop;
    property Visible;
    property WantTabs;
    property WordWrap;
    property OnChange;
    property OnClick;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnKeyDown;
    property OnKeyPress;
    property OnKeyUp;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
    property OnProgress : TProgressEvent read FOnProgress	write SetProgressEvent;
    property OnStartDrag;
  end;

{-----[     TCDBImage ]------}

  TCDBImage = class(TCustomControl)
  private
    FDataLink: TFieldDataLink;
    FPicture: TPicture;
    FBorderStyle: TBorderStyle;
    FAutoDisplay: Boolean;
    FStretch: Boolean;
    FCenter: Boolean;
    FPictureLoaded: Boolean;
    FQuickDraw: Boolean;
{--------[   ]---------}
    Compressor : TLZHCompressor;
    InStream, OutStream : TMemoryStream;
    FOnProgress : TProgressEvent;
    procedure CompressStream;
    procedure DecompressStream;
    procedure PutData(Sender:TObject; var DTA : BufType; NBytes : WORD;
    	var Bytes_Got : WORD);
    procedure GetData(Sender:TObject; var DTA : BufType; NBytes : WORD;
    	var Bytes_Got : WORD);
    function GetInBufferSize : integer;
    function GetOutBufferSize : integer;
    procedure SetInBufferSize( NewValue : integer );
    procedure SetOutBufferSize( NewValue : integer );
    procedure SetProgressEvent( NewValue : TProgressEvent);
{------[							]-------}
    procedure DataChange(Sender: TObject);
    function GetDataField: string;
    function GetDataSource: TDataSource;
    function GetField: TField;
    function GetReadOnly: Boolean;
    procedure PictureChanged(Sender: TObject);
    procedure SetAutoDisplay(Value: Boolean);
    procedure SetBorderStyle(Value: TBorderStyle);
    procedure SetCenter(Value: Boolean);
    procedure SetDataField(const Value: string);
    procedure SetDataSource(Value: TDataSource);
    procedure SetPicture(Value: TPicture);
    procedure SetReadOnly(Value: Boolean);
    procedure SetStretch(Value: Boolean);
    procedure UpdateData(Sender: TObject);
    procedure CMEnter(var Message: TCMEnter); message CM_ENTER;
    procedure CMExit(var Message: TCMExit); message CM_EXIT;
    procedure WMLButtonDown(var Message: TWMLButtonDown); message WM_LBUTTONDOWN;
    procedure WMLButtonDblClk(var Message: TWMLButtonDblClk); message WM_LBUTTONDBLCLK;
    procedure WMCut(var Message: TMessage); message WM_CUT;
    procedure WMCopy(var Message: TMessage); message WM_COPY;
    procedure WMPaste(var Message: TMessage); message WM_PASTE;
    procedure CMTextChanged(var Message: TMessage); message CM_TEXTCHANGED;
  protected
    procedure CreateParams(var Params: TCreateParams); override;
    function GetPalette: HPALETTE; override;
    procedure KeyDown(var Key: Word; Shift: TShiftState); override;
    procedure KeyPress(var Key: Char); override;
    procedure Notification(AComponent: TComponent;
      Operation: TOperation); override;
    procedure Paint; override;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure CopyToClipboard;
    procedure CutToClipboard;
    procedure LoadPicture;
    procedure PasteFromClipboard;
    property Field: TField read GetField;
    property Picture: TPicture read FPicture write SetPicture;
  published
    property Align;
    property AutoDisplay: Boolean read FAutoDisplay write SetAutoDisplay default True;
    property BorderStyle: TBorderStyle read FBorderStyle write SetBorderStyle default bsSingle;
    property InBufferSize : Integer read GetInBufferSize write SetInBufferSize
    		default $1000;
    property OutBufferSize : Integer read GetOutBufferSize write SetOutBufferSize
    		default $1000;
    property Center: Boolean read FCenter write SetCenter default True;
    property Color;
    property Ctl3D;
    property DataField: string read GetDataField write SetDataField;
    property DataSource: TDataSource read GetDataSource write SetDataSource;
    property DragCursor;
    property DragMode;
    property Enabled;
    property Font;
    property ParentColor default False;
    property ParentCtl3D;
    property ParentFont;
    property ParentShowHint;
    property PopupMenu;
    property ReadOnly: Boolean read GetReadOnly write SetReadOnly default False;
    property QuickDraw: Boolean read FQuickDraw write FQuickDraw default True;
    property ShowHint;
    property Stretch: Boolean read FStretch write SetStretch default False;
    property TabOrder;
    property TabStop default True;
    property Visible;
    property OnClick;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnKeyDown;
    property OnKeyPress;
    property OnKeyUp;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
    property OnProgress : TProgressEvent read FOnProgress	write SetProgressEvent;
    property OnStartDrag;
  end;

procedure Register;

implementation

{      }

procedure TCDBMemo.PutData(Sender:TObject; var DTA : BufType; NBytes : WORD;
		var Bytes_Got : WORD);
begin
	OutStream.Write(DTA,NBytes);
  Bytes_Got := NBytes;
end;

procedure TCDBMemo.GetData(Sender:TObject; var DTA : BufType; NBytes : WORD;
		var Bytes_Got : WORD);
begin
  if Assigned(FOnProgress) then
  	FOnProgress(Sender, InStream.Position, InStream.Size);
	if InStream.Position+NBytes <= InStream.Size then
  begin
  	InStream.Read(DTA,NBytes);
    Bytes_Got := NBytes;
  end else
  begin
  	Bytes_Got := InStream.Size-InStream.Position;
    InStream.Read(DTA,Bytes_Got)
  end;
end;

procedure TCDBMemo.CompressStream;
begin
	InStream.Seek(soFromBeginning,0);
  OutStream.Clear;
  Compressor.Compress;
  OutStream.Seek(soFromBeginning,0);
end;

procedure TCDBMemo.DecompressStream;
begin
	InStream.Seek(soFromBeginning,0);
  OutStream.Clear;
  Compressor.DeCompress;
  OutStream.Seek(soFromBeginning,0);
end;

function TCDBMemo.GetInBufferSize : integer;
begin
	Result := Compressor.ReadBufSize;
end;

function TCDBMemo.GetOutBufferSize : integer;
begin
	Result := Compressor.WriteBufSize;
end;

procedure TCDBMemo.SetInBufferSize( NewValue : integer );
begin
	Compressor.WriteBufSize := NewValue;
end;

procedure TCDBMemo.SetOutBufferSize( NewValue : integer );
begin
	Compressor.ReadBufSize := NewValue;
end;

procedure TCDBMemo.SetProgressEvent( NewValue : TProgressEvent );
begin
	FOnProgress := NewValue;
end;

{ TCDBMemo }

constructor TCDBMemo.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  inherited ReadOnly := True;
  FAutoDisplay := True;
  FDataLink := TFieldDataLink.Create;
  FDataLink.Control := Self;
  FDataLink.OnDataChange := DataChange;
  FDataLink.OnEditingChange := EditingChange;
  FDataLink.OnUpdateData := UpdateData;
  FPaintControl := TPaintControl.Create(Self, 'EDIT');
  Compressor := TLZHCompressor.Create(Self);
  Compressor.OnPutBytes := PutData;
  Compressor.OnGetBytes := GetData;
  InStream := TMemoryStream.Create;
  OutStream := TMemoryStream.Create;
end;

destructor TCDBMemo.Destroy;
begin
  Compressor.Free;
  InStream.Free;
  OutStream.Free;
  FPaintControl.Free;
  FDataLink.Free;
  FDataLink := nil;
  inherited Destroy;
end;

procedure TCDBMemo.Notification(AComponent: TComponent;
  Operation: TOperation);
begin
  inherited Notification(AComponent, Operation);
  if (Operation = opRemove) and (FDataLink <> nil) and
    (AComponent = DataSource) then DataSource := nil;
end;

procedure TCDBMemo.KeyDown(var Key: Word; Shift: TShiftState);
begin
  inherited KeyDown(Key, Shift);
  if FMemoLoaded then
  begin
    if (Key = VK_DELETE) or ((Key = VK_INSERT) and (ssShift in Shift)) then
      FDataLink.Edit;
  end else
    Key := 0;
end;

procedure TCDBMemo.KeyPress(var Key: Char);
begin
  inherited KeyPress(Key);
  if FMemoLoaded then
  begin
    if (Key in [#32..#255]) and (FDataLink.Field <> nil) and
      not FDataLink.Field.IsValidChar(Key) then
    begin
      MessageBeep(0);
      Key := #0;
    end;
    case Key of
      ^H, ^I, ^J, ^M, ^V, ^X, #32..#255:
        FDataLink.Edit;
      #27:
        FDataLink.Reset;
    end;
  end else
  begin
    if Key = #13 then LoadMemo;
    Key := #0;
  end;
end;

procedure TCDBMemo.Change;
begin
  if FMemoLoaded then FDataLink.Modified;
  FMemoLoaded := True;
  inherited Change;
end;

function TCDBMemo.GetDataSource: TDataSource;
begin
  Result := FDataLink.DataSource;
end;

procedure TCDBMemo.SetDataSource(Value: TDataSource);
begin
  FDataLink.DataSource := Value;
  if Value <> nil then Value.FreeNotification(Self);
end;

function TCDBMemo.GetDataField: string;
begin
  Result := FDataLink.FieldName;
end;

procedure TCDBMemo.SetDataField(const Value: string);
begin
  FDataLink.FieldName := Value;
end;

function TCDBMemo.GetReadOnly: Boolean;
begin
  Result := FDataLink.ReadOnly;
end;

procedure TCDBMemo.SetReadOnly(Value: Boolean);
begin
  FDataLink.ReadOnly := Value;
end;

function TCDBMemo.GetField: TField;
begin
  Result := FDataLink.Field;
end;

procedure TCDBMemo.LoadMemo;
begin
  if not FMemoLoaded and (FDataLink.Field is TBlobField) then
  begin
    try
      {----[   ]----}
      InStream.Clear;
      TBlobField(FDataLink.Field).SaveToStream(InStream);
			DecompressStream;
      Lines.LoadFromStream(OutStream);
      {---[ 			]----}
      FMemoLoaded := True;
    except
      Lines.Text := 'Memo-  ';
    end;
    EditingChange(Self);
  end;
end;

procedure TCDBMemo.DataChange(Sender: TObject);
begin
  if FDataLink.Field <> nil then
    if FDataLink.Field is TBlobField then
    begin
      if FAutoDisplay or (FDataLink.Editing and FMemoLoaded) then
      begin
        FMemoLoaded := False;
        LoadMemo;
      end else
      begin
        Text := '(' + FDataLink.Field.DisplayLabel + ')';
        FMemoLoaded := False;
      end;
    end else
    begin
      if FFocused and FDataLink.CanModify then
      	Text := FDataLink.Field.Text
      else Text := FDataLink.Field.DisplayText;

      FMemoLoaded := True;
    end
  else
  begin
    if csDesigning in ComponentState then Text := Name else Text := '';
    FMemoLoaded := False;
  end;
end;

procedure TCDBMemo.EditingChange(Sender: TObject);
begin
  inherited ReadOnly := not (FDataLink.Editing and FMemoLoaded);
end;

procedure TCDBMemo.UpdateData(Sender: TObject);
begin
  InStream.Clear;
  Lines.SaveToStream(InStream);
  CompressStream;
  TBlobField(FDataLink.Field).LoadFromStream(OutStream);
end;

procedure TCDBMemo.SetFocused(Value: Boolean);
begin
  if FFocused <> Value then
  begin
    FFocused := Value;
    if not (FDataLink.Field is TBlobField) then FDataLink.Reset;
  end;
end;

procedure TCDBMemo.WndProc(var Message: TMessage);
begin
  with Message do
    if (Msg = WM_CREATE) or (Msg = WM_WINDOWPOSCHANGED) or
      (Msg = CM_FONTCHANGED) then FPaintControl.DestroyHandle;
  inherited;
end;

procedure TCDBMemo.CMEnter(var Message: TCMEnter);
begin
  SetFocused(True);
  inherited;
end;

procedure TCDBMemo.CMExit(var Message: TCMExit);
begin
  try
    FDataLink.UpdateRecord;
  except
    SetFocus;
    raise;
  end;
  SetFocused(False);
  inherited;
end;

procedure TCDBMemo.SetAutoDisplay(Value: Boolean);
begin
  if FAutoDisplay <> Value then
  begin
    FAutoDisplay := Value;
    if Value then LoadMemo;
  end;
end;

procedure TCDBMemo.WMLButtonDblClk(var Message: TWMLButtonDblClk);
begin
  if not FMemoLoaded then LoadMemo else inherited;
end;

procedure TCDBMemo.WMCut(var Message: TMessage);
begin
  FDataLink.Edit;
  inherited;
end;

procedure TCDBMemo.WMPaste(var Message: TMessage);
begin
  FDataLink.Edit;
  inherited;
end;

procedure TCDBMemo.CMGetDataLink(var Message: TMessage);
begin
  Message.Result := Integer(FDataLink);
end;

procedure TCDBMemo.WMPaint(var Message: TWMPaint);
var
  S: string;
begin
  if not (csPaintCopy in ControlState) then inherited else
  begin
    if FDataLink.Field <> nil then
      if FDataLink.Field is TBlobField then
      begin
      	{InStream.Clear;
      	TBlobField(FDataLink.Field).SaveToStream(InStream);
      	DecompressStream;
      	Lines.LoadFromStream(OutStream);}
        S := Lines.Text;
      end  else
        S := FDataLink.Field.DisplayText;

    SendMessage(FPaintControl.Handle, WM_SETTEXT, 0, Integer(PChar(S)));
    SendMessage(FPaintControl.Handle, WM_PAINT, Message.DC, 0);
  end;
end;

{----------------------------[ TCDBImage ]-----------------------------------}

{---[       ]----}
procedure TCDBImage.PutData(Sender:TObject; var DTA : BufType; NBytes : WORD;
		var Bytes_Got : WORD);
begin
	OutStream.Write(DTA,NBytes);
  Bytes_Got := NBytes;
end;

procedure TCDBImage.GetData(Sender:TObject; var DTA : BufType; NBytes : WORD;
		var Bytes_Got : WORD);
begin
  if Assigned( FOnProgress ) then
  	FOnProgress(Sender,InStream.Position,InStream.Size);
	if InStream.Position+NBytes <= InStream.Size then
  begin
  	InStream.Read(DTA,NBytes);
    Bytes_Got := NBytes;
  end else
  begin
  	Bytes_Got := InStream.Size-InStream.Position;
    InStream.Read(DTA,Bytes_Got)
  end;
end;

procedure TCDBImage.CompressStream;
begin
	InStream.Seek(soFromBeginning,0);
  OutStream.Clear;
  Compressor.Compress;
  OutStream.Seek(soFromBeginning,0);
end;

procedure TCDBImage.DecompressStream;
begin
	InStream.Seek(soFromBeginning,0);
  OutStream.Clear;
  Compressor.DeCompress;
  OutStream.Seek(soFromBeginning,0);
end;

function TCDBImage.GetInBufferSize : integer;
begin
	Result := Compressor.ReadBufSize;
end;

function TCDBImage.GetOutBufferSize : integer;
begin
	Result := Compressor.WriteBufSize;
end;

procedure TCDBImage.SetInBufferSize( NewValue : integer );
begin
	Compressor.WriteBufSize := NewValue;
end;

procedure TCDBImage.SetOutBufferSize( NewValue : integer );
begin
	Compressor.ReadBufSize := NewValue;
end;

procedure TCDBImage.SetProgressEvent( NewValue : TProgressEvent );
begin
	FOnProgress := NewValue;
end;

constructor TCDBImage.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  ControlStyle := ControlStyle + [csFramed, csOpaque];
  Width := 105;
  Height := 105;
  TabStop := True;
  ParentColor := False;
  FPicture := TPicture.Create;
  FPicture.OnChange := PictureChanged;
  FBorderStyle := bsSingle;
  FAutoDisplay := True;
  FCenter := True;
  FDataLink := TFieldDataLink.Create;
  FDataLink.Control := Self;
  FDataLink.OnDataChange := DataChange;
  FDataLink.OnUpdateData := UpdateData;
  FQuickDraw := True;
  Compressor := TLZHCompressor.Create(Self);
  Compressor.OnPutBytes := PutData;
  Compressor.OnGetBytes := GetData;
  InStream := TMemoryStream.Create;
  OutStream := TMemoryStream.Create;
end;

destructor TCDBImage.Destroy;
begin
  FPicture.Free;
  FDataLink.Free;
  FDataLink := nil;
  Compressor.Free;
  InStream.Free;
  OutStream.Free;
  inherited Destroy;
end;

function TCDBImage.GetDataSource: TDataSource;
begin
  Result := FDataLink.DataSource;
end;

procedure TCDBImage.SetDataSource(Value: TDataSource);
begin
  FDataLink.DataSource := Value;
  if Value <> nil then Value.FreeNotification(Self);
end;

function TCDBImage.GetDataField: string;
begin
  Result := FDataLink.FieldName;
end;

procedure TCDBImage.SetDataField(const Value: string);
begin
  FDataLink.FieldName := Value;
end;

function TCDBImage.GetReadOnly: Boolean;
begin
  Result := FDataLink.ReadOnly;
end;

procedure TCDBImage.SetReadOnly(Value: Boolean);
begin
  FDataLink.ReadOnly := Value;
end;

function TCDBImage.GetField: TField;
begin
  Result := FDataLink.Field;
end;

function TCDBImage.GetPalette: HPALETTE;
begin
  Result := 0;
  if FPicture.Graphic is TBitmap then
    Result := TBitmap(FPicture.Graphic).Palette;
end;

procedure TCDBImage.SetAutoDisplay(Value: Boolean);
begin
  if FAutoDisplay <> Value then
  begin
    FAutoDisplay := Value;
    if Value then LoadPicture;
  end;
end;

procedure TCDBImage.SetBorderStyle(Value: TBorderStyle);
begin
  if FBorderStyle <> Value then
  begin
    FBorderStyle := Value;
    RecreateWnd;
  end;
end;

procedure TCDBImage.SetCenter(Value: Boolean);
begin
  if FCenter <> Value then
  begin
    FCenter := Value;
    Invalidate;
  end;
end;

procedure TCDBImage.SetPicture(Value: TPicture);
begin
  FPicture.Assign(Value);
end;

procedure TCDBImage.SetStretch(Value: Boolean);
begin
  if FStretch <> Value then
  begin
    FStretch := Value;
    Invalidate;
  end;
end;

procedure TCDBImage.Paint;
var
  W, H: Integer;
  R: TRect;
  S: string;
  DrawPict: TPicture;
begin
  with Canvas do
  begin
    Brush.Style := bsSolid;
    Brush.Color := Color;
    if FPictureLoaded or (csPaintCopy in ControlState) then
    begin
      DrawPict := TPicture.Create;
      H := 0;
      try
        if (csPaintCopy in ControlState) and
          Assigned(FDataLink.Field) and (FDataLink.Field is TBlobField) then
        begin
          InStream.Clear;
          TBlobField(FDataLink.Field).SaveToStream(InStream);
          InStream.Seek(soFromBeginning,0);
          DeCompressStream;
          DrawPict.Bitmap.LoadFromStream( OutStream );
          if DrawPict.Graphic is TBitmap then
            DrawPict.Bitmap.IgnorePalette := QuickDraw; //!!
        end
        else
        begin
          DrawPict.Assign(Picture);
          if Focused and (DrawPict.Graphic is TBitmap) and
            (DrawPict.Bitmap.Palette <> 0) then
          begin { Control has focus, so realize the bitmap palette in foreground }
            H := SelectPalette(Handle, DrawPict.Bitmap.Palette, False);
            RealizePalette(Handle);
          end;
        end;
        if Stretch then
          if (DrawPict.Graphic = nil) or DrawPict.Graphic.Empty then
            FillRect(ClientRect)
          else
            StretchDraw(ClientRect, DrawPict.Graphic)
        else
        begin
          SetRect(R, 0, 0, DrawPict.Width, DrawPict.Height);
          if Center then OffsetRect(R, (ClientWidth - DrawPict.Width) div 2,
            (ClientHeight - DrawPict.Height) div 2);
          StretchDraw(R, DrawPict.Graphic);
          ExcludeClipRect(Handle, R.Left, R.Top, R.Right, R.Bottom);
          FillRect(ClientRect);
          SelectClipRgn(Handle, 0);
        end;
      finally
        if H <> 0 then SelectPalette(Handle, H, True);
        DrawPict.Free;
      end;
    end
    else begin
      Font := Self.Font;
      if FDataLink.Field <> nil then
        S := FDataLink.Field.DisplayLabel
      else S := Name;
      S := '(' + S + ')';
      W := TextWidth(S);
      H := TextHeight(S);
      R := ClientRect;
      TextRect(R, (R.Right - W) div 2, (R.Bottom - H) div 2, S);
    end;
    if (GetParentForm(Self).ActiveControl = Self) and
      not (csDesigning in ComponentState) and
      not (csPaintCopy in ControlState) then
    begin
      Brush.Color := clWindowFrame;
      FrameRect(ClientRect);
    end;
  end;
end;

procedure TCDBImage.PictureChanged(Sender: TObject);
begin
  FDataLink.Modified;
  FPictureLoaded := True;
  Invalidate;
end;

procedure TCDBImage.Notification(AComponent: TComponent;
  Operation: TOperation);
begin
  inherited Notification(AComponent, Operation);
  if (Operation = opRemove) and (FDataLink <> nil) and
    (AComponent = DataSource) then DataSource := nil;
end;

procedure TCDBImage.LoadPicture;
begin
  if not FPictureLoaded and (FDataLink.Field is TBlobField) then
  begin
    InStream.Clear;
    TBlobField(FDataLink.Field).SaveToStream(InStream);
    InStream.Seek(soFromBeginning,0);
    DeCompressStream;
    Picture.Bitmap.LoadFromStream(OutStream);
  end;
end;

procedure TCDBImage.DataChange(Sender: TObject);
begin
  Picture.Graphic := nil;
  FPictureLoaded := False;
  if FAutoDisplay then LoadPicture;
end;

procedure TCDBImage.UpdateData(Sender: TObject);
begin
  if FDataLink.Field is TBlobField then
    with TBlobField(FDataLink.Field) do
      if Picture.Graphic is TBitmap then
      begin
       	InStream.Clear;
        Picture.Graphic.SaveToStream(InStream);
        InStream.Seek(soFromBeginning,0);
        CompressStream;
        LoadFromStream(OutStream);
      end else Clear;
end;

procedure TCDBImage.CopyToClipboard;
begin
  if Picture.Graphic <> nil then Clipboard.Assign(Picture);
end;

procedure TCDBImage.CutToClipboard;
begin
  if Picture.Graphic <> nil then
    if FDataLink.Edit then
    begin
      CopyToClipboard;
      Picture.Graphic := nil;
    end;
end;

procedure TCDBImage.PasteFromClipboard;
begin
  if Clipboard.HasFormat(CF_BITMAP) and FDataLink.Edit then
    Picture.Bitmap.Assign(Clipboard);
end;

procedure TCDBImage.CreateParams(var Params: TCreateParams);
begin
  inherited CreateParams(Params);
  if FBorderStyle = bsSingle then
    Params.Style := Params.Style or WS_BORDER;
end;

procedure TCDBImage.KeyDown(var Key: Word; Shift: TShiftState);
begin
  inherited KeyDown(Key, Shift);
  case Key of
    VK_INSERT:
      if ssShift in Shift then PasteFromClipBoard else
        if ssCtrl in Shift then CopyToClipBoard;
    VK_DELETE:
      if ssShift in Shift then CutToClipBoard;
  end;
end;

procedure TCDBImage.KeyPress(var Key: Char);
begin
  inherited KeyPress(Key);
  case Key of
    ^X: CutToClipBoard;
    ^C: CopyToClipBoard;
    ^V: PasteFromClipBoard;
    #13: LoadPicture;
    #27: FDataLink.Reset;
  end;
end;

procedure TCDBImage.CMEnter(var Message: TCMEnter);
begin
  Invalidate; { Draw the focus marker }
  inherited;
end;

procedure TCDBImage.CMExit(var Message: TCMExit);
begin
  Invalidate; { Erase the focus marker }
  inherited;
end;

procedure TCDBImage.CMTextChanged(var Message: TMessage);
begin
  inherited;
  if not FPictureLoaded then Invalidate;
end;

procedure TCDBImage.WMLButtonDown(var Message: TWMLButtonDown);
begin
  if TabStop and CanFocus then SetFocus;
  inherited;
end;

procedure TCDBImage.WMLButtonDblClk(var Message: TWMLButtonDblClk);
begin
  LoadPicture;
  inherited;
end;

procedure TCDBImage.WMCut(var Message: TMessage);
begin
  CutToClipboard;
end;

procedure TCDBImage.WMCopy(var Message: TMessage);
begin
  CopyToClipboard;
end;

procedure TCDBImage.WMPaste(var Message: TMessage);
begin
  PasteFromClipboard;
end;

{  }

procedure Register;
begin
  RegisterComponents('LZHComponents', [ TCDBMemo ,TCDBImage ]);
end;

end.
