unit Linker;
{
Copyright 1996 by Justin Turberville
Address: P.O.Box 122
      	 Kenton-On-Sea
      	 6191
         South Africa
EMail: justint@cyberjoe.co.za

Designed to maintain links to associated components on a form so that
actions by any one can be handled by a single handler which can then
update the associated components via the link. If the "UseTags" property
is True then the Tags of all linked components are updated to the Index
of the Link, making it possible to implement a handler as below where a
DBGrid's fields' visibility can be toggled via either a MenuItem, a popup
MenuItem, or a SpeedButton - just add this to their OnClick Events:-

procedure Form1.LinkClick(Sender: Object);
begin
  with Linker1.Links[Linker.GetLinkIndex(Sender)] do begin
    MenuItem.Checked  := not MenuItem.Checked;
    PopupItem.Checked := MenuItem.Checked;
    SpeedBtn.Down     := MenuItem.Checked;
    DataField.Visible := MenuItem.Checked;
    TLabel(Component1).Caption := DataField.DisplayLabel + ' toggled';
  end;
end;

- Can save a lot of clumsy, inflexible code, hey!

Have added an "OnLinkClick" event to Linker - is never activated by
Linker, but sets the OnClick events of linked TSpeedButtons & TMenuItems
to the same handler if assigned - may save some effort!
Likewise, added an "OnLinkChange" event for linked TEdits & TFields.

Origionally the tags of linked components were used for manipulating
the associated linked items by setting them to their link index via
the "UseTags" property - ie: with Linker1.Link[TComponent(Sender).Tag]
do ..., but have since added "GetLinkIndex" for this purpose, leaving
the tags free for other use, but have left "UseTags" in for backward
compatability and for use if desired. "GetLinkIndex" is a more robust 
way - tags can be changed! Then again using the tags is more efficient 
than searching for the matching link each time.

Note that no checking for duplicated links are made.
The Links property has the normal TCollection properties & methods,
plus "ByName" which returns the first Link with that name, or nil;
The "Name" property of TLink is for convenience only & not required
(not a component name). Otherwise, apart from the pointer properties,
TLink is a TCollectionItem.

Let me know if you find this usefull!
}

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  DB, Buttons, Menus, StdCtrls, DsgnIntf;

type
  TLinker = class;

  TLink = class(TCollectionItem)
  private
    FName: string;
    FSpeedButton: TSpeedButton;
    FDataField:   TField;
    FEditControl: TCustomEdit;
    FMenuItem, FPopupItem: TMenuItem;
    FComponent1, FComponent2: TComponent;
    procedure SetSpeedButton(Value: TSpeedButton);
    procedure SetMenuItem(Value: TMenuItem);
    procedure SetPopupItem(Value: TMenuItem);
    procedure SetEditControl(Value: TCustomEdit);
    procedure SetDataField(Value: TField);
    procedure SetComponent1(Value: TComponent);
    procedure SetComponent2(Value: TComponent);
  public
    Item: Pointer;                      {runtime holder of anything}
    procedure Assign(Source: TPersistent); override;
    function HasRef(Value: TObject): Boolean;
  published
    property Name: string read FName write FName;
    property SpeedButton: TSpeedButton read FSpeedButton write SetSpeedButton;
    property MenuItem:    TMenuItem    read FMenuItem    write SetMenuItem;
    property PopupItem:   TMenuItem    read FPopupItem   write SetPopupItem;
    property EditControl: TCustomEdit  read FEditControl write SetEditControl;
    property DataField:   TField       read FDataField   write SetDataField;
    property Component1:  TComponent   read FComponent1  write SetComponent1;
    property Component2:  TComponent   read FComponent2  write SetComponent2;
  end;

  TLinks = class(TCollection)
  private
    FLinker: TLinker;
    function GetItem(Index: Integer): TLink;
    procedure SetItem(Index: Integer; Value: TLink);
  protected
    procedure Update(Item: TCollectionItem); override;
  public
    constructor Create(Linker: TLinker);
    function Add: TLink;
    function ByName(AName: string): TLink;
    property Items[Index: Integer]: TLink read GetItem write SetItem; default;
  end;

  TLinker = class(TComponent)
  private
    FAbout: string;
    FLinks: TLinks;
    FUseTags: Boolean;
    FOnLinkClick: TNotifyEvent;
    FOnLinkChange: TNotifyEvent;
    procedure SetLinks(const Value: TLinks);
    procedure SetUseTags(Value: Boolean);
    procedure SetOnLinkClick(Value: TNotifyEvent);
    procedure SetOnLinkChange(Value: TNotifyEvent);
  protected
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure Assign(Source: TPersistent); override;
    procedure Notification(AComponent: TComponent; Operation: TOperation);
      override;
    function GetLinkIndex(Value: TObject): Integer;
  published
    property About: string read FAbout;
    property Links: TLinks read FLinks write SetLinks;
    property UseTags: Boolean read FUseTags write SetUseTags default False;
    property OnLinkClick: TNotifyEvent read FOnLinkClick write SetOnLinkClick;
    property OnLinkChange: TNotifyEvent read FOnLinkChange write SetOnLinkChange;
  end;

procedure Register;

implementation

uses
  LinkEdit;

procedure Register;
begin
  RegisterComponents('Samples', [TLinker]);
  RegisterComponentEditor(TLinker, TLinkEditor);
  RegisterPropertyEditor(TypeInfo(TLinks), TLinker, '', TLinksProperty);
end;

{TLink}

procedure TLink.Assign(Source: TPersistent);
begin
  if Source is TLink then begin
    FName        := TLink(Source).Name;
    FSpeedButton := TLink(Source).SpeedButton;
    FMenuItem    := TLink(Source).MenuItem;
    FPopupItem   := TLink(Source).PopupItem;
    FEditControl := TLink(Source).EditControl;
    FDataField   := TLink(Source).DataField;
    FComponent1  := TLink(Source).Component1;
    FComponent2  := TLink(Source).Component2;
    Item         := TLink(Source).Item;
  end
  else inherited Assign(Source);
end;

function TLink.HasRef(Value: TObject): Boolean;
begin
  Result := (Value = FSpeedButton) or
    (Value = FMenuItem)    or (Value = FPopupItem) or
    (Value = FEditControl) or (Value = FDataField) or
    (Value = FComponent1)  or (Value = FComponent2) or
    (Value = Item);
end;

procedure TLink.SetSpeedButton(Value: TSpeedButton);
begin
  FSpeedButton := Value;
  Changed(False);
end;

procedure TLink.SetMenuItem(Value: TMenuItem);
begin
  FMenuItem := Value;
  Changed(False);
end;

procedure TLink.SetPopupItem(Value: TMenuItem);
begin
  FPopupItem := Value;
  Changed(False);
end;

procedure TLink.SetEditControl(Value: TCustomEdit);
begin
  FEditControl := Value;
  Changed(False);
end;

procedure TLink.SetDataField(Value: TField);
begin
  FDataField := Value;
  Changed(False);
end;

procedure TLink.SetComponent1(Value: TComponent);
begin
  FComponent1 := Value;
  Changed(False);
end;

procedure TLink.SetComponent2(Value: TComponent);
begin
  FComponent2 := Value;
  Changed(False);
end;

{TLinks}

constructor TLinks.Create(Linker: TLinker);
begin
  inherited Create(TLink);
  FLinker := Linker;
end;

function TLinks.Add: TLink;
var
  i, suffix: Integer;
  b: Boolean;
  s: string;
begin
  suffix := 1;
  if Count > 0 then begin
    i := Count-1;
    b := False;
    while (i >= 0) and not b do begin
      s := TLink(Items[i]).Name;
      if Pos('Link', s) = 1 then begin   {if name starts with "Link"...}
        Delete(s, 1, 4);                 {get number suffix}
        suffix := StrToIntDef(s, 0);     {try converting}
        if suffix > 0 then begin         {if a genuine integer...}
          b := True;                     {last number suffix found}
          Inc(suffix);                   {increase for new link}
        end;
      end;
      Dec(i);
    end;
  end;
  Result := TLink(inherited Add);
  Result.Name := 'Link' + IntToStr(suffix);
end;

function TLinks.ByName(AName: string): TLink;
var
  i: Integer;
begin
  Result := nil;
  i := 0;
  while i < Count do
    if TLink(Items[i]).Name = AName then
      Result := TLink(Items[i]) else
      Inc(i); 
end;

function TLinks.GetItem(Index: Integer): TLink;
begin
  Result := TLink(inherited GetItem(Index));
end;

procedure TLinks.SetItem(Index: Integer; Value: TLink);
begin
  inherited SetItem(Index, Value);
end;

procedure TLinks.Update(Item: TCollectionItem);

  procedure SetTags(Link: TLink; ToSet: Boolean);
    procedure SetTag(Comp: TComponent; Index: Integer);
    begin
      if Comp <> nil then Comp.Tag := Index;
    end;
  var
    i: Integer;
  begin
    with Link do begin
      if ToSet then i := Index else i := 0;
      SetTag(SpeedButton, i);
      SetTag(MenuItem,    i);
      SetTag(PopupItem,   i);
      SetTag(EditControl, i);
      SetTag(DataField,   i);
      SetTag(Component1,  i);
      SetTag(Component2,  i);
    end;
  end;

  procedure SetOnClick(Link: TLink; Handler: TNotifyEvent);
  begin
    with Link do begin
      if SpeedButton <> nil then SpeedButton.OnClick := Handler;
      if MenuItem    <> nil then MenuItem.OnClick    := Handler;
      if PopupItem   <> nil then PopupItem.OnClick   := Handler;
    end;
  end;

  procedure SetOnChange(Link: TLink; Handler: TNotifyEvent);
  begin
    with Link do begin
      if EditControl <> nil then
        TEdit(EditControl).OnChange := Handler;
      if DataField <> nil then
        DataField.OnChange := TFieldNotifyEvent(Handler);
    end;
  end;

var
  i: Integer;
begin   {Update}
  if Item <> nil then begin        {update that item only}
    SetTags(TLink(Item), FLinker.UseTags);
    SetOnClick(TLink(Item), FLinker.OnLinkClick);
    SetOnChange(TLink(Item), FLinker.OnLinkChange);
  end
  else for i := 0 to Count-1 do begin      {update all items}
    SetTags(Items[i], FLinker.UseTags);
    SetOnClick(Items[i], FLinker.OnLinkClick);
    SetOnChange(Items[i], FLinker.OnLinkChange);
  end;
end;

{TLinker}

constructor TLinker.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FAbout := 'v1.1 by Justin Turberville - justint@cyberjoe.co.za';
  FLinks := TLinks.Create(Self);
end;

destructor TLinker.Destroy;
begin
  FLinks.Free;
  inherited Destroy;
end;

procedure TLinker.Assign(Source: TPersistent);
begin
  if Source is TLinker then begin
    FUseTags := TLinker(Source).UseTags;
    FLinks.Assign(TLinker(Source).Links);
  end
  else inherited Assign(Source);
end;

procedure TLinker.Notification(AComponent: TComponent; Operation: TOperation);
  procedure CheckLink(Link: TLink);
  begin
    with Link do begin    {remove any reference to component to be deleted...}
      if SpeedButton = AComponent then SpeedButton := nil;
      if MenuItem    = AComponent then MenuItem    := nil;
      if PopupItem   = AComponent then PopupItem   := nil;
      if EditControl = AComponent then EditControl := nil;
      if DataField   = AComponent then DataField   := nil;
      if Component1  = AComponent then Component1  := nil;
      if Component2  = AComponent then Component2  := nil;
    end;
  end;
var
  i: Integer;
begin
  inherited Notification(AComponent, Operation);
  if Operation = opRemove then
    for i := 0 to FLinks.Count-1 do
      CheckLink(FLinks[i]);
end;

function TLinker.GetLinkIndex(Value: TObject): Integer;
var
  i: Integer;
begin
  Result := -1;
  for i := 0 to FLinks.Count-1 do
    if FLinks[i].HasRef(Value) then begin
      Result := i;
      Break;
    end;
end;

procedure TLinker.SetLinks(const Value: TLinks);
begin
  FLinks.Assign(Value);
end;

procedure TLinker.SetUseTags(Value: Boolean);
begin
  FUseTags := Value;
  if FLinks.Count > 0 then
    FLinks[0].Changed(True);      {signal for all links to update}
end;

procedure TLinker.SetOnLinkClick(Value: TNotifyEvent);
begin
  FOnLinkClick := Value;
  if FLinks.Count > 0 then
    FLinks[0].Changed(True);
end;

procedure TLinker.SetOnLinkChange(Value: TNotifyEvent);
begin
  FOnLinkChange := Value;
  if FLinks.Count > 0 then
    FLinks[0].Changed(True);
end;

end.
