unit LinkEdit;

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, StdCtrls, Mask, ExtCtrls, Buttons, DsgnIntf,
  ComCtrls, DB, Menus, Linker;

type
  TLinkEditForm = class(TForm)

    IndexEdit: TEdit;
    LinkGroup: TGroupBox;
    C2Label: TLabel;
    C2Combo: TComboBox;
    SBLabel: TLabel;
    SBCombo: TComboBox;
    MILabel: TLabel;
    MICombo: TComboBox;
    MPLabel: TLabel;
    MPCombo: TComboBox;
    C1Label: TLabel;
    C1Combo: TComboBox;
    AddBtn: TButton;
    DeleteBtn: TButton;
    NameList: TListBox;
    NameEdit: TEdit;
    NameLabel: TLabel;
    DFCombo: TComboBox;
    DFLabel: TLabel;
    Label3: TLabel;
    ECCombo: TComboBox;
    ECLabel: TLabel;
    MoveUpSB: TSpeedButton;
    MoveDownSB: TSpeedButton;
    Timer: TTimer;
    OKBtn: TButton;
    CancelBtn: TButton;
    procedure FormShow(Sender: TObject);
    procedure NameListClick(Sender: TObject);
    procedure NameEditChange(Sender: TObject);
    procedure OKBtnClick(Sender: TObject);
    procedure FormCreate(Sender: TObject);
    procedure SBComboChange(Sender: TObject);
    procedure MIComboChange(Sender: TObject);
    procedure MPComboChange(Sender: TObject);
    procedure C1ComboChange(Sender: TObject);
    procedure ComboKeyDown(Sender: TObject; var Key: Word; Shift: TShiftState);
    procedure AddBtnClick(Sender: TObject);
    procedure DeleteBtnClick(Sender: TObject);
    procedure NameListMouseDown(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    procedure NameListDragOver(Sender, Source: TObject; X, Y: Integer;
      State: TDragState; var Accept: Boolean);
    procedure NameListDragDrop(Sender, Source: TObject; X, Y: Integer);
    procedure C2ComboChange(Sender: TObject);
    procedure DFComboChange(Sender: TObject);
    procedure ECComboChange(Sender: TObject);
    procedure MoveUpSBClick(Sender: TObject);
    procedure MoveDownSBClick(Sender: TObject);
    procedure TimerTimer(Sender: TObject);
    procedure NameListKeyDown(Sender: TObject; var Key: Word; Shift: TShiftState);
  private
    ToChange: Boolean;
    GoingUp:  Boolean;
    EditInfo: TLinker;
    procedure SetItem;
    procedure UpdateBtns;
  public
    Linker: TLinker;
  end;

  TLinkEditor = class(TComponentEditor)
  private
    FLinker: TLinker;
  protected
  public
    procedure Edit; override;
    procedure ExecuteVerb(Index: Integer); override;
    function GetVerb(Index: Integer): string; override;
    function GetVerbCount: Integer; override;
  end;

  TLinksProperty = class(TClassProperty)
  public
    function GetAttributes: TPropertyAttributes; override;
    procedure Edit; override;
  end;

var
  LinkEditForm: TLinkEditForm;

implementation

{$R *.DFM}

{TLinksProperty}

function TLinksProperty.GetAttributes: TPropertyAttributes;
begin
  Result := [paDialog];
end;

procedure TLinksProperty.Edit;
begin
  LinkEditForm := TLinkEditForm.Create(Application);
  with LinkEditForm do
  try
    Linker := TLinker(GetComponent(0));
    ShowModal;
{   if ModalResult = mrOK then Designer.Modified;}
  finally
    Free;
  end;
end;

{TLinkEditor}

procedure TLinkEditor.Edit;
begin
  LinkEditForm := TLinkEditForm.Create(Application);
  with LinkEditForm do
  try
    Linker := Component as TLinker;
    ShowModal;
{   if ModalResult = mrOK then
      Designer.Modified;  -- Designer invalid?!}
  finally
    Free;
  end;
end;

procedure TLinkEditor.ExecuteVerb(Index: Integer);
begin
  if Index = 0 then Edit;
end;

function TLinkEditor.GetVerb(Index: Integer): string;
begin
  if Index = 0 then Result := '&Edit Links...';
end;

function TLinkEditor.GetVerbCount: Integer;
begin
  Result := 1;
end;

{FilterInfoForm}

procedure TLinkEditForm.FormCreate(Sender: TObject);
begin
  EditInfo := TLinker.Create(Self);
end;

procedure TLinkEditForm.FormShow(Sender: TObject);

  procedure ClearBlanks(Combo: TComboBox);
  var
    i: Integer;
  begin
    i := 0;
    with Combo do   {clear blank items that appear for some reason...}
      while i < Items.Count do
        if Items[i] = '' then
          Items.Delete(i) else
          Inc(i);
  end;

  procedure InitCombos;
  var
    i, n: Integer;
  begin
    with Linker.Owner do
      for i := 0 to ComponentCount-1 do begin
        if Components[i] is TDataSource then begin
          with TDataSource(Components[i]) do
            if DataSet <> nil then
              with DataSet do
                for n := 0 to FieldCount-1 do
                  DFCombo.Items.AddObject(Fields[n].Name, Fields[n]);
        end
        else if Components[i] is TSpeedButton then
          SBCombo.Items.AddObject(Components[i].Name, Components[i])
        else if Components[i] is TCustomEdit then
          ECCombo.Items.AddObject(Components[i].Name, Components[i])
        else if Components[i] is TMenuItem then
          if TMenuItem(Components[i]).Caption <> '-' then begin  {skip dividers}
            MICombo.Items.AddObject(Components[i].Name, Components[i]);
            MPCombo.Items.AddObject(Components[i].Name, Components[i]);
          end;
        C1Combo.Items.AddObject(Components[i].Name, Components[i]);
        C2Combo.Items.AddObject(Components[i].Name, Components[i]);
      end;
    ClearBlanks(MICombo);        {caused by menu items, somehow...}
    ClearBlanks(MPCombo);
    ClearBlanks(C1Combo);
    ClearBlanks(C2Combo);
  end;

var
  i: Integer;
begin  {FormShow}
  if not (Linker.Owner is TForm) then begin
    MessageDlg('Cannot find form components from a data module!',
      mtWarning, [mbOK], 0);
    ModalResult := mrAbort;
  end
  else begin
    EditInfo.Assign(Linker);
    if EditInfo.Links.Count > 0 then
      for i := 0 to EditInfo.Links.Count-1 do
        NameList.Items.Add(EditInfo.Links[i].Name);
    InitCombos;
    if NameList.Items.Count > 0 then
      NameList.ItemIndex := 0;
    SetItem;
    DeleteBtn.Enabled := NameList.ItemIndex = 0;
  end;
  UpdateBtns;
end;

procedure TLinkEditForm.NameListClick(Sender: TObject);
begin
  SetItem;
  UpdateBtns;
end;

procedure TLinkEditForm.SetItem;

  procedure SetCombo(Combo: TComboBox; Link: TComponent);
  begin
    if Link <> nil then
      Combo.ItemIndex := Combo.Items.IndexOf(Link.Name) else
      Combo.ItemIndex := -1;
  end;

begin   {SetItem}
  ToChange := False;
  if NameList.ItemIndex > -1 then begin
    with EditInfo.Links[NameList.ItemIndex] do begin
      NameEdit.Text := Name;
      SetCombo(SBCombo, SpeedButton);
      SetCombo(MICombo, MenuItem);
      SetCombo(MPCombo, PopupItem);
      SetCombo(ECCombo, EditControl);
      SetCombo(DFCombo, DataField);
      SetCombo(C1Combo, Component1);
      SetCombo(C2Combo, Component2);
    end;
    IndexEdit.Text := IntToStr(NameList.ItemIndex);
  end
  else begin
    NameEdit.Text  := '';
    IndexEdit.Text := '';
    SBCombo.ItemIndex := -1;
    MICombo.ItemIndex := -1;
    MPCombo.ItemIndex := -1;
    ECCombo.ItemIndex := -1;
    DFCombo.ItemIndex := -1;
    C1Combo.ItemIndex := -1;
    C2Combo.ItemIndex := -1;
  end;
  ToChange := True;
  NameEdit.Enabled  := NameList.ItemIndex > -1;
  LinkGroup.Enabled := NameEdit.Enabled;
  DeleteBtn.Enabled := NameEdit.Enabled;
end;

procedure TLinkEditForm.NameEditChange(Sender: TObject);
var
  i: Integer;
begin
  if ToChange then begin
    i := NameList.ItemIndex;
    EditInfo.Links[i].Name := NameEdit.Text;
    NameList.Items[i] := NameEdit.Text;
    NameList.ItemIndex := i;              {restore eselection}
  end;
end;

procedure TLinkEditForm.SBComboChange(Sender: TObject);
begin
  if ToChange then
    with EditInfo.Links[NameList.ItemIndex] do
      if SBCombo.ItemIndex < 0 then
        SpeedButton := nil else
        SpeedButton := TSpeedButton(SBCombo.Items.Objects[SBCombo.ItemIndex]);
end;

procedure TLinkEditForm.MIComboChange(Sender: TObject);
begin
  if ToChange then
    with EditInfo.Links[NameList.ItemIndex] do
      if MICombo.ItemIndex < 0 then
        MenuItem := nil else
        MenuItem := TMenuItem(MICombo.Items.Objects[MICombo.ItemIndex]);
end;

procedure TLinkEditForm.MPComboChange(Sender: TObject);
begin
  if ToChange then
    with EditInfo.Links[NameList.ItemIndex] do
      if MPCombo.ItemIndex < 0 then
        PopupItem := nil else
        PopupItem := TMenuItem(MPCombo.Items.Objects[MPCombo.ItemIndex]);
end;

procedure TLinkEditForm.ECComboChange(Sender: TObject);
begin
  if ToChange then
    with EditInfo.Links[NameList.ItemIndex] do
      if ECCombo.ItemIndex < 0 then
        EditControl := nil else
        EditControl := TCustomEdit(ECCombo.Items.Objects[ECCombo.ItemIndex]);
end;

procedure TLinkEditForm.C1ComboChange(Sender: TObject);
begin
  if ToChange then
    with EditInfo.Links[NameList.ItemIndex] do
      if C1Combo.ItemIndex < 0 then
        Component1 := nil else
        Component1 := TComponent(C1Combo.Items.Objects[C1Combo.ItemIndex]);
end;

procedure TLinkEditForm.C2ComboChange(Sender: TObject);
begin
  if ToChange then
    with EditInfo.Links[NameList.ItemIndex] do
      if C2Combo.ItemIndex < 0 then
        Component2 := nil else
        Component2 := TComponent(C2Combo.Items.Objects[C2Combo.ItemIndex]);
end;

procedure TLinkEditForm.DFComboChange(Sender: TObject);
begin
  if ToChange then
    with EditInfo.Links[NameList.ItemIndex] do
      if DFCombo.ItemIndex < 0 then
        DataField := nil else
        DataField := TField(DFCombo.Items.Objects[DFCombo.ItemIndex]);
end;

procedure TLinkEditForm.OKBtnClick(Sender: TObject);
begin
  Linker.Assign(EditInfo);
end;

procedure TLinkEditForm.ComboKeyDown(Sender: TObject; var Key: Word;
  Shift: TShiftState);
begin
  if Key = VK_DELETE then begin
    (Sender as TComboBox).ItemIndex := -1;
    (Sender as TComboBox).OnChange(nil);
  end;
end;

procedure TLinkEditForm.AddBtnClick(Sender: TObject);
var
  link: TLink;
begin
  link := EditInfo.Links.Add;
  with NameList do begin
    Items.Add(link.Name);
    ItemIndex := Items.Count-1;
  end;
  SetItem;
  UpdateBtns;
end;

procedure TLinkEditForm.DeleteBtnClick(Sender: TObject);
var
  i: Integer;
begin
  with NameList do begin
    i := ItemIndex;
    EditInfo.Links[i].Free;
    NameList.Items.Delete(i);
    if i < Items.Count then
      ItemIndex := i else
      ItemIndex := Items.Count-1;
  end;
  SetItem;
  UpdateBtns;
end;

procedure TLinkEditForm.NameListMouseDown(Sender: TObject;
  Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
begin
  with Sender as TListBox do
    if (Items.Count > 1) and (Button = mbLeft) then
      if ItemAtPos(Point(X, Y), True) >= 0 then
        BeginDrag(False);
end;

procedure TLinkEditForm.NameListDragOver(Sender, Source: TObject; X,
  Y: Integer; State: TDragState; var Accept: Boolean);
begin
  Accept := (Source = Sender) and
    (TListBox(Sender).ItemAtPos(Point(X,Y), False) >= 0);
  if Accept then
    with TListBox(Sender) do
      if Y > Height - ItemHeight then begin
        GoingUp := False;
        Timer.Enabled := True;
      end
      else if Y < ItemHeight then begin
        GoingUp := True;
        Timer.Enabled := True;
      end
      else Timer.Enabled := False;
end;

procedure TLinkEditForm.NameListDragDrop(Sender, Source: TObject; X,Y: Integer);
var
  i: Integer;
begin
  with NameList do begin
    i := ItemAtPos(Point(X, Y), True);
    if i = -1 then i := Items.Count-1;   {beyond last item}
    if i = ItemIndex then Exit;
    EditInfo.Links[ItemIndex].Index := i;
    Items.Move(ItemIndex, i);
    ItemIndex := i;
  end;
  SetItem;
  UpdateBtns;
end;

procedure TLinkEditForm.MoveUpSBClick(Sender: TObject);
begin
  with NameList do
    Items.Move(ItemIndex -1, ItemIndex);
  UpdateBtns;
end;

procedure TLinkEditForm.MoveDownSBClick(Sender: TObject);
begin
  with NameList do
    Items.Move(ItemIndex +1, ItemIndex);
  UpdateBtns;
end;

procedure TLinkEditForm.UpdateBtns;
begin
  with NameList do begin
    DeleteBtn.Enabled  := ItemIndex >= 0;
    MoveUpSB.Enabled   := ItemIndex > 0;
    MoveDownSB.Enabled := ItemIndex < Items.Count-1;
  end;
end;

procedure TLinkEditForm.TimerTimer(Sender: TObject);
begin
  with NameList do
    if GoingUp then
      if TopIndex > 0 then
        TopIndex := TopIndex -1 else
        Timer.Enabled := False
      else if TopIndex < Items.Count-1 then
        TopIndex := TopIndex +1 else
        Timer.Enabled := False;
end;

procedure TLinkEditForm.NameListKeyDown(Sender: TObject; var Key: Word;
  Shift: TShiftState);
begin
  if ssCtrl in Shift then
    case Key of
      VK_INSERT:
        AddBtnClick(nil);
      VK_DELETE:
        if DeleteBtn.Enabled then DeleteBtnClick(nil);
      VK_UP, VK_LEFT: begin
        if MoveUpSB.Enabled then MoveUpSBClick(nil);
        Key := 0;
      end;
      VK_DOWN, VK_RIGHT: begin
        if MoveDownSB.Enabled then MoveDownSBClick(nil);
        Key := 0;
      end;
      VK_HOME:
        if MoveUpSB.Enabled then begin
          with NameList do Items.Move(ItemIndex, 0);
          UpdateBtns;
        end;
      VK_END:
        if MoveDownSB.Enabled then begin
          with NameList do Items.Move(ItemIndex, Items.Count-1);
          UpdateBtns;
        end;
    end;
end;

end.
