unit linkedit;

{ Copyright 1998 - Horacio Jamilis - Industria Argentina }
{ In this unit you have two similar components.
  One of them is an enhaced TEdit component that could
  have an ellipsis button on it's right side (optional)
  with an event when the user presses this button, and
  Alignment option (left, center or rigth).
  The other one is a data-aware version of the same thing.}

{ This is freeware. So you could use it or change it as much
  as you wish. If you improve this please send me a copy.
  If you have any suggestion or ideas write me too.
  If you like it, YOU MUST WRITE ME!!!
  My email address is: jhoracio@cvtci.com.ar }

{ This unit is based on one that have the same name:
 [copied from original source]
   Copyright 1996
   Vanguard Computer Services Pty Ltd, Jim Wowchuk
   Portions copyrighted by Borland International
   You are permitted to use, copy and adapt this code providing
   doing so does not violate any other existing copyrights and
   you do not attempt to remove, diminish or restrict the copyrights
   of others that have provided this for you.
 [end of copy]

 I expect to not disturb the original autor.
}

{ The last comment:
  This is freeware so you have not any warranty that this
  components do anything else than waste your hard drive
  space. I am not responsible of any damages this code could
  give you. It's clear?

  Thanks for trying this excelent freeware components.
}

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  StdCtrls, Db, DbCtrls, Mask;

type
  TlinkStyle = (lsNormal, lsEllipsis);

  TLinkEdit = class(TCustomEdit)
  private
    { Private declarations }
    fButtonWidth : integer;
    fLinkStyle : TLinkStyle;
    fPressed : boolean;
    fTracking : boolean;
    fOnButtonClick : TNotifyEvent;
    FAlignment: TAlignment;
    FCanvas : TControlCanvas;
    procedure StopTracking;
    procedure SetLinkStyle(Value: TLinkStyle);
    procedure TrackButton(X,Y: Integer);
    procedure WMPaint(var Message: TWMPaint); message wm_Paint;
    procedure WMSetCursor(var Msg: TWMSetCursor); message WM_SETCURSOR;
    function GetTextMargins: TPoint;
  protected
    { Protected declarations }
    procedure KeyPress(var Key: Char); override;
    procedure BoundsChanged;
    procedure EditButtonClick;
    procedure CreateParams(var Params: TCreateParams); override;
    procedure DoEnter; override;
    procedure DoExit; override;
    procedure KeyDown(var Key: Word; Shift: TShiftState); override;
    procedure MouseDown(Button: TMouseButton; Shift: TShiftState;
      X, Y: Integer); override;
    procedure MouseMove(Shift: TShiftState; X, Y: Integer); override;
    procedure MouseUp(Button: TMouseButton; Shift: TShiftState;
      X, Y: Integer); override;
//    procedure PaintWindow(DC: HDC); override;
  public
    { Public declarations }
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
  published
    { Published declarations }
    property AutoSelect;
    property AutoSize;
    property Alignment : TAlignment read FAlignment write FAlignment default TaLeftJustify;
    property BorderStyle;
    property CharCase;
    property Color;
    property Ctl3D;
    property DragCursor;
    property DragMode;
    property Enabled;
    property Font;
    property HideSelection;
    property LinkStyle: TLinkStyle read fLinkStyle write SetLinkStyle default lsNormal;
    property MaxLength;
    property OEMConvert;
    property ParentColor;
    property ParentCtl3D;
    property ParentFont;
    property ParentShowHint;
    property PasswordChar;
    property PopupMenu;
    property ReadOnly;
    property ShowHint;
    property TabOrder;
    property TabStop;
    property Text;
    property Visible;
    property OnButtonClick : TNotifyEvent read fOnButtonClick write fOnButtonClick;
    property OnChange;
    property OnClick;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnKeyDown;
    property OnKeyPress;
    property OnKeyUp;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
    property OnStartDrag;
  end;

  TDbLinkEdit = class(TCustomMaskEdit)
  private
    { Private declarations }
    fButtonWidth : integer;
    fLinkStyle : TLinkStyle;
    fPressed : boolean;
    fTracking : boolean;
    fOnButtonClick : TNotifyEvent;
    FDataLink: TFieldDataLink;
    FAlignment: TAlignment;
    FFocused: Boolean;
    FCanvas : TControlCanvas;
    function GetDataField: string; { returns the name of the data field }
    function GetDataSource: TDataSource; { returns reference to the data source }
    procedure SetDataField(const Value: string); { assigns name of data field }
    procedure SetDataSource(Value: TDataSource); { assigns new data source }
    function GetField: TField;
    function GetReadOnly: Boolean;
    procedure SetFocused(Value: Boolean);
    procedure SetReadOnly(Value: Boolean);
    procedure StopTracking;
    procedure DataChange(Sender: TObject); { must have proper parameters for event }
    procedure EditingChange(Sender: TObject);
    procedure UpdateData(Sender: TObject);
    procedure SetLinkStyle(Value: TLinkStyle);
    procedure TrackButton(X,Y: Integer);
    procedure WMPaint(var Message: TWMPaint); message wm_Paint;
    procedure WMSetCursor(var Msg: TWMSetCursor); message WM_SETCURSOR;
    procedure WMCut(var Message: TMessage); message WM_CUT;
    procedure WMPaste(var Message: TMessage); message WM_PASTE;
    procedure CMEnter(var Message: TCMEnter); message CM_ENTER;
    procedure CMExit(var Message: TWMNoParams); message CM_EXIT;
    procedure CMGetDataLink(var Message: TMessage); message CM_GETDATALINK;
    function GetTextMargins: TPoint;
  protected
    { Protected declarations }
    procedure Change; override;
    function EditCanModify: Boolean; override;
    procedure BoundsChanged;
    procedure EditButtonClick;
    procedure KeyPress(var Key: Char); override;
    procedure Loaded; override;
    procedure CreateParams(var Params: TCreateParams); override;
    procedure DoEnter; override;
    procedure DoExit; override;
    procedure KeyDown(var Key: Word; Shift: TShiftState); override;
    procedure MouseDown(Button: TMouseButton; Shift: TShiftState;
      X, Y: Integer); override;
    procedure MouseMove(Shift: TShiftState; X, Y: Integer); override;
    procedure MouseUp(Button: TMouseButton; Shift: TShiftState;
      X, Y: Integer); override;
    procedure Notification(AComponent: TComponent;Operation: TOperation); override;
    procedure Reset; override;
  public
    { Public declarations }
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    property Field: TField read GetField;
  published
    { Published declarations }
    property AutoSelect;
    property AutoSize;
    property BorderStyle;
    property CharCase;
    property Color;
    property Ctl3D;
    property DataField: string read GetDataField write SetDataField;
    property DataSource: TDataSource read GetDataSource write SetDataSource;
    property DragCursor;
    property DragMode;
    property Enabled;
    property Font;
    property ImeMode;
    property ImeName;
    property LinkStyle: TLinkStyle read fLinkStyle write SetLinkStyle default lsNormal;
    property MaxLength;
    property ParentColor;
    property ParentCtl3D;
    property ParentFont;
    property ParentShowHint;
    property PasswordChar;
    property PopupMenu;
    property ReadOnly : Boolean read GetReadOnly write SetReadOnly default false;
    property ShowHint;
    property TabOrder;
    property TabStop;
    property Visible;
    property OnButtonClick : TNotifyEvent read fOnButtonClick write fOnButtonClick;
    property OnChange;
    property OnClick;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnKeyDown;
    property OnKeyPress;
    property OnKeyUp;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
    property OnStartDrag;
  end;

procedure Register;

implementation

procedure Register;
begin
  RegisterComponents('Free', [TLinkEdit]);
  RegisterComponents('Free', [TDbLinkEdit]);
end;

constructor TLinkEdit.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  fButtonWidth := GetSystemMetrics(SM_CXVSCROLL);
  fLinkStyle := lsEllipsis;
end; // Create

destructor TLinkEdit.Destroy;
begin
  inherited Destroy;
  FCanvas.Free;
end; // Destroy

procedure TLinkEdit.CreateParams(var Params: TCreateParams);
begin
  inherited CreateParams(Params);
    { in order to use the EM_SETRECT later, we must make the edit control
      a type MULTILINE }
  with Params do
  begin
    Style := Style or ES_MULTILINE;
  end;
end;  // CreateParams

procedure TLinkEdit.BoundsChanged;
var
  R: TRect;
begin
    { Determine the size of the text area in the control - it will
      be smaller by the width of the button if one is present }
  SetRect(R, 0, 0, ClientWidth - 2, ClientHeight + 1); // +1 is workaround for windows paint bug
  if (fLinkStyle <> lsNormal) and focused then Dec(R.Right, fButtonWidth);
  SendMessage(Handle, EM_SETRECT, 0, LongInt(@R));
  Repaint;
end; // BoundsChanged

procedure TLinkEdit.SetLinkStyle(Value: TLinkStyle);
begin
    { if the link style is different then change it,
      remember to redraw it if the control is currently
      focused }
  if Value = fLinkStyle then Exit;
  fLinkStyle := Value;
  if not HandleAllocated then exit;
  if focused or (csDesigning in ComponentState) then
    BoundsChanged;
end; // SetLinkStyle

procedure TLinkEdit.EditButtonClick;
begin
  if Assigned(fOnButtonClick) then fOnButtonClick(Self);
end; // EditButtonClick

procedure TLinkEdit.WMPaint(var Message: TWMPaint);
var
  Left: Integer;
  Margins: TPoint;
  R: TRect;
  DC: HDC;
  PS: TPaintStruct;
  S: string;
  Flags: Integer;
  W: Integer;
begin
{ Since edit controls do not handle justification unless multi-line (and
  then only poorly) we will draw right and center justify manually unless
  the edit has the focus. }
  if FCanvas = nil then
  begin
    FCanvas := TControlCanvas.Create;
    FCanvas.Control := Self;
  end;
  DC := Message.DC;
  if DC = 0 then DC := BeginPaint(Handle, PS);
  FCanvas.Handle := DC;
  try
    FCanvas.Font := Font;
    with FCanvas do
    begin
      if (fLinkStyle <> lsNormal) and (focused or (csDesigning in ComponentState)) then
        SetRect(R, ClientWidth - fButtonWidth, 0, ClientWidth, ClientHeight)
      else
        begin
          R := ClientRect;
          if not (NewStyleControls and Ctl3D) and (BorderStyle = bsSingle) then
            begin
              Brush.Color := clWindowFrame;
              FrameRect(R);
              InflateRect(R, -1, -1);
            end;
          Brush.Color := Color;
        end;
      S := Text;
      if PasswordChar <> #0 then
        FillChar(S[1], Length(S), PasswordChar);
      Margins := GetTextMargins;
      case FAlignment of
        taLeftJustify: Left := Margins.X;
        taRightJustify: Left := ClientWidth - TextWidth(S) - Margins.X - 1;
      else
        Left := (ClientWidth - TextWidth(S)) div 2;
      end;
      TextRect(R, Left, Margins.Y, S);
      if (fLinkStyle <> lsNormal) and (focused or (csDesigning in ComponentState)) then
        begin
          Flags := 0;
          if FPressed then
            Flags := BF_FLAT;
          DrawEdge(DC, R, EDGE_RAISED, BF_RECT or BF_MIDDLE or Flags);
          Flags := ((R.Right - R.Left) shr 1) - 1 + Ord(fPressed);
          W := Height shr 3;
          if W = 0 then W := 1;
          PatBlt(DC, R.Left + Flags, R.Top + Flags, W, W, BLACKNESS);
          PatBlt(DC, R.Left + Flags - (W * 2), R.Top + Flags, W, W, BLACKNESS);
          PatBlt(DC, R.Left + Flags + (W * 2), R.Top + Flags, W, W, BLACKNESS);
          ExcludeClipRect(DC, R.Left, R.Top, R.Right, R.Bottom);
          PaintWindow(DC);
        end;
    end;
  finally
    FCanvas.Handle := 0;
    if Message.DC = 0 then EndPaint(Handle, PS);
  end;
end; // WMPaint

(*
procedure TLinkEdit.WMPaint(var Message: TWMPaint);
begin
  PaintHandler(Message)
end; // WMPaint
*)

procedure TLinkEdit.WMSetCursor(var Msg: TWMSetCursor);
var
  P: TPoint;
begin
    { Normally, the Edit control changes the Cursor to an I-bar when over
      the control.  We need to set it back to an arrow when over the button }
  if (fLinkStyle <> lsNormal)
  and PtInRect(Rect(Width - FButtonWidth - 4, 0, ClientWidth, ClientHeight), ScreenToClient(P)) then
    begin
    GetCursorPos(P);
    Windows.SetCursor(LoadCursor(0, idc_Arrow));
    end
  else
    inherited;
end; // WMSetCursor

procedure TLinkEdit.KeyDown(var Key: Word; Shift: TShiftState);
var
  Msg: TMsg;
begin
    { simulate the mouse pressing the ellipsis button from the
      keyboard by the user pressing CTRL+ENTER }
  if  (fLinkStyle = lsEllipsis)
  and (Key = VK_RETURN)
  and (Shift = [ssCtrl]) then
  begin
    EditButtonClick;
    PeekMessage(Msg, Handle, WM_CHAR, WM_CHAR, PM_REMOVE);
  end
  else
    inherited KeyDown(Key, Shift);
end;  // KeyDown

procedure TLinkEdit.KeyPress(var Key: Char);
begin
  if Key = #13 then
    begin
      Key := #0;
      MessageBeep(0);
    end;
  inherited KeyPress(Key);
end;

procedure TLinkEdit.MouseUp(Button: TMouseButton; Shift: TShiftState;
  X, Y: Integer);
var
  WasPressed: Boolean;
begin
     { if the mouse was released (after being pressed) on the button
       then perform its associated action }
  WasPressed := fPressed;
  StopTracking;
  BoundsChanged;
  if (Button = mbLeft) and (fLinkStyle = lsEllipsis) and WasPressed then
    EditButtonClick;
  inherited MouseUp(Button, Shift, X, Y);
end;

procedure TLinkEdit.TrackButton(X,Y: Integer);
var
  NewState: Boolean;
  R: TRect;
begin
    { Check if the position passed is over the area of the button -
      if so then set the state to pressed and redraw the depressed
      button }
  SetRect(R, ClientWidth - fButtonWidth, 0, ClientWidth, ClientHeight);
  NewState := PtInRect(R, Point(X, Y));
  if fPressed <> NewState then
  begin
    fPressed := NewState;
    InvalidateRect(Handle, @R, False);
  end;
end; // TrackButton

(*
procedure TLinkEdit.PaintWindow(DC: HDC);
var
  R: TRect;
  Flags: Integer;
  W: Integer;
begin
    { here's where we draw the little elipsis button when necessary -
      most times it is normal (raised) state, but sometimes it is pressed }
  if (fLinkStyle <> lsNormal) and (focused or (csDesigning in ComponentState)) then
  begin
    SetRect(R, ClientWidth - fButtonWidth, 0, ClientWidth, ClientHeight);
    Flags := 0;
    if FPressed then
      Flags := BF_FLAT;
    DrawEdge(DC, R, EDGE_RAISED, BF_RECT or BF_MIDDLE or Flags);
    Flags := ((R.Right - R.Left) shr 1) - 1 + Ord(fPressed);
    W := Height shr 3;
    if W = 0 then W := 1;
    PatBlt(DC, R.Left + Flags, R.Top + Flags, W, W, BLACKNESS);
    PatBlt(DC, R.Left + Flags - (W * 2), R.Top + Flags, W, W, BLACKNESS);
    PatBlt(DC, R.Left + Flags + (W * 2), R.Top + Flags, W, W, BLACKNESS);
    ExcludeClipRect(DC, R.Left, R.Top, R.Right, R.Bottom);
  end;
  inherited PaintWindow(DC);
end; // PaintWindow
*)

procedure TLinkEdit.MouseDown(Button: TMouseButton; Shift: TShiftState;
  X, Y: Integer);
begin
    { Not only must you press the button with the mouse, but it must be
      released over the same button.  If the button has been pressed, we
      need to redraw it depressed, then track the mouse movements to see
      if the user moves off it before releasing. }
  if (Button = mbLeft) and (fLinkStyle <> lsNormal) and focused
  and PtInRect(Rect(Width - fButtonWidth, 0, Width, Height), Point(X,Y)) then
    begin
    MouseCapture := True;
    FTracking := True;
    TrackButton(X, Y);
    end;
  inherited MouseDown(Button, Shift, X, Y);
end; // MouseDown

procedure TLinkEdit.MouseMove(Shift: TShiftState; X, Y: Integer);
begin
    { if we are tracking the mouse, the mouse must have been pressed over
      the button part of the control.  Check to see we are still over it. }
  if fTracking then
    TrackButton(X, Y);
  inherited MouseMove(Shift, X, Y);
end; // MouseMove

procedure TLinkEdit.StopTracking;
begin
    { we are finished tracking the mouse over the control.  Reset everything }
  if FTracking then
  begin
    TrackButton(-1, -1);
    FTracking := False;
    MouseCapture := False;
  end;
end; // StopTracking;

procedure TLinkEdit.DoEnter;
begin
    { In use the elipsis button is only shown when we the control has focus }
  if (fLinkStyle <> lsNormal)
    then BoundsChanged;
  inherited DoEnter;
  if AutoSelect then
    SelectAll;
end; // DoEnter

procedure TLinkEdit.DoExit;
begin
    { Remove the elipsis button (if present) when we lose focus }
  if (fLinkStyle <> lsNormal)
    then BoundsChanged;
  inherited DoExit;
end;

function TLinkEdit.GetTextMargins: TPoint;
var
  DC: HDC;
  SaveFont: HFont;
  I: Integer;
  SysMetrics, Metrics: TTextMetric;
begin
  if NewStyleControls then
    begin
      if BorderStyle = bsNone then
        I := 0
      else if Ctl3D then
        I := 1
      else
        I := 2;
      Result.X := SendMessage(Handle, EM_GETMARGINS, 0, 0) and $0000FFFF + I;
      Result.Y := I;
    end
  else
    begin
      if BorderStyle = bsNone then
        I := 0
      else
        begin
          DC := GetDC(0);
          GetTextMetrics(DC, SysMetrics);
          SaveFont := SelectObject(DC, Font.Handle);
          GetTextMetrics(DC, Metrics);
          SelectObject(DC, SaveFont);
          ReleaseDC(0, DC);
          I := SysMetrics.tmHeight;
          if I > Metrics.tmHeight then
            I := Metrics.tmHeight;
          I := I div 4;
        end;
      Result.X := I;
      Result.Y := I;
    end;
end;

constructor TDbLinkEdit.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  inherited ReadOnly := True;
  FDataLink := TFieldDataLink.Create; { construct the data-link object }
  FDataLink.Control := Self;
  FDataLink.OnDataChange := DataChange; { attach handler to event }
  FDataLink.OnEditingChange := EditingChange;
  FDataLink.OnUpdateData := UpdateData;
  fButtonWidth := GetSystemMetrics(SM_CXVSCROLL);
  fLinkStyle := lsEllipsis;
end; // Create

destructor TDbLinkEdit.Destroy;
begin
  FDataLink.Free; { always destroy owned objects first... }
  FDataLink := nil; { detach handler before destroying object }
  FCanvas.Free;
  inherited Destroy;
end; // Destroy

procedure TDbLinkEdit.Loaded;
begin
  inherited Loaded;
  if (csDesigning in ComponentState) then DataChange(Self);
end;

procedure TDbLinkEdit.Notification(AComponent: TComponent;Operation: TOperation);
begin
  inherited Notification(AComponent, Operation);
  if (Operation = opRemove) and (FDataLink <> nil) and
    (AComponent = DataSource) then DataSource := nil;
end;

procedure TDbLinkEdit.DataChange(Sender: TObject);
begin
  if FDataLink.Field <> nil then { if there is no field assigned... }
    begin
      if FAlignment <> FDataLink.Field.Alignment then
        begin
          EditText := ''; { ...set to invalid date }
          FAlignment := FDataLink.Field.Alignment;
        end;
      EditMask := FDataLink.Field.EditMask;
      if FFocused and FDataLink.CanModify then
        Text := FDataLink.Field.Text
      else
        begin
          EditText := FDataLink.Field.DisplayText;
          if (FDataLink.Editing) {and (FDataLink.Modified)} then
            Modified := True;
        end;
    end
  else
    begin
      FAlignment := taLeftJustify;
      EditMask := '';
      if csDesigning in ComponentState then
        EditText := Name
      else
        EditText := '';
    end;
end;

procedure TDbLinkEdit.EditingChange(Sender: TObject);
begin
  inherited ReadOnly := not FDataLink.Editing;
end;

procedure TDbLinkEdit.UpdateData(Sender: TObject);
begin
  ValidateEdit;
  FDataLink.Field.Text := Text; { set field link to calendar date }
end;

procedure TDbLinkEdit.Change;
begin
 FDataLink.Modified; { call the Modified method }
 inherited Change; { call the inherited Change method }
end;

procedure TDbLinkEdit.CreateParams(var Params: TCreateParams);
begin
  inherited CreateParams(Params);
    { in order to use the EM_SETRECT later, we must make the edit control
      a type MULTILINE }
  with Params do
  begin
    Style := Style or ES_MULTILINE;
  end;
end;  // CreateParams

procedure TDbLinkEdit.BoundsChanged;
var
  R: TRect;
begin
    { Determine the size of the text area in the control - it will
      be smaller by the width of the button if one is present }
  SetRect(R, 0, 0, ClientWidth - 2, ClientHeight + 1); // +1 is workaround for windows paint bug
  if (fLinkStyle <> lsNormal) and focused then Dec(R.Right, fButtonWidth);
  SendMessage(Handle, EM_SETRECT, 0, LongInt(@R));
  Repaint;
end; // BoundsChanged

procedure TDbLinkEdit.SetLinkStyle(Value: TLinkStyle);
begin
    { if the link style is different then change it,
      remember to redraw it if the control is currently
      focused }
  if Value = fLinkStyle then Exit;
  fLinkStyle := Value;
  if not HandleAllocated then exit;
  if focused or (csDesigning in ComponentState) then
    BoundsChanged;
end; // SetLinkStyle

procedure TDbLinkEdit.EditButtonClick;
begin
  if Assigned(fOnButtonClick) then fOnButtonClick(Self);
end; // EditButtonClick

procedure TDbLinkEdit.WMPaint(var Message: TWMPaint);
var
  Left: Integer;
  Margins: TPoint;
  R: TRect;
  DC: HDC;
  PS: TPaintStruct;
  S: string;
  Flags: Integer;
  W: Integer;
begin
{ Since edit controls do not handle justification unless multi-line (and
  then only poorly) we will draw right and center justify manually unless
  the edit has the focus. }
  if FCanvas = nil then
  begin
    FCanvas := TControlCanvas.Create;
    FCanvas.Control := Self;
  end;
  DC := Message.DC;
  if DC = 0 then DC := BeginPaint(Handle, PS);
  FCanvas.Handle := DC;
  try
    FCanvas.Font := Font;
    with FCanvas do
    begin
      if (fLinkStyle <> lsNormal) and (focused or (csDesigning in ComponentState)) then
        SetRect(R, ClientWidth - fButtonWidth, 0, ClientWidth, ClientHeight)
      else
        begin
          R := ClientRect;
          if not (NewStyleControls and Ctl3D) and (BorderStyle = bsSingle) then
            begin
              Brush.Color := clWindowFrame;
              FrameRect(R);
              InflateRect(R, -1, -1);
            end;
          Brush.Color := Color;
        end;
      if (csPaintCopy in ControlState) and (FDataLink.Field <> nil) then
        begin
          S := FDataLink.Field.DisplayText;
          case CharCase of
            ecUpperCase: S := AnsiUpperCase(S);
            ecLowerCase: S := AnsiLowerCase(S);
          end;
        end
      else
        S := EditText;
      if PasswordChar <> #0 then
        FillChar(S[1], Length(S), PasswordChar);
      Margins := GetTextMargins;
      case FAlignment of
        taLeftJustify: Left := Margins.X;
        taRightJustify: Left := ClientWidth - TextWidth(S) - Margins.X - 1;
      else
        Left := (ClientWidth - TextWidth(S)) div 2;
      end;
      TextRect(R, Left, Margins.Y, S);
      if (fLinkStyle <> lsNormal) and (focused or (csDesigning in ComponentState)) then
        begin
          Flags := 0;
          if FPressed then
            Flags := BF_FLAT;
          DrawEdge(DC, R, EDGE_RAISED, BF_RECT or BF_MIDDLE or Flags);
          Flags := ((R.Right - R.Left) shr 1) - 1 + Ord(fPressed);
          W := Height shr 3;
          if W = 0 then W := 1;
          PatBlt(DC, R.Left + Flags, R.Top + Flags, W, W, BLACKNESS);
          PatBlt(DC, R.Left + Flags - (W * 2), R.Top + Flags, W, W, BLACKNESS);
          PatBlt(DC, R.Left + Flags + (W * 2), R.Top + Flags, W, W, BLACKNESS);
          ExcludeClipRect(DC, R.Left, R.Top, R.Right, R.Bottom);
          PaintWindow(DC);
        end;
    end;
  finally
    FCanvas.Handle := 0;
    if Message.DC = 0 then EndPaint(Handle, PS);
  end;
end; // WMPaint

procedure TDbLinkEdit.WMSetCursor(var Msg: TWMSetCursor);
var
  P: TPoint;
begin
    { Normally, the Edit control changes the Cursor to an I-bar when over
      the control.  We need to set it back to an arrow when over the button }
  if (fLinkStyle <> lsNormal)
  and PtInRect(Rect(Width - FButtonWidth - 4, 0, ClientWidth, ClientHeight), ScreenToClient(P)) then
    begin
    GetCursorPos(P);
    Windows.SetCursor(LoadCursor(0, idc_Arrow));
    end
  else
    inherited;
end; // WMSetCursor

procedure TDbLinkEdit.KeyDown(var Key: Word; Shift: TShiftState);
var
  Msg: TMsg;
begin
    { simulate the mouse pressing the ellipsis button from the
      keyboard by the user pressing CTRL+ENTER }
  if  (fLinkStyle = lsEllipsis)
  and (Key = VK_RETURN)
  and (Shift = [ssCtrl]) then
  begin
    EditButtonClick;
    PeekMessage(Msg, Handle, WM_CHAR, WM_CHAR, PM_REMOVE);
  end
  else
    inherited KeyDown(Key, Shift);
  if (Key = VK_DELETE) or ((Key = VK_INSERT) and (ssShift in Shift)) then
    FDataLink.Edit;
end;  // KeyDown

procedure TDbLinkEdit.KeyPress(var Key: Char);
begin
  if Key = #13 then
    begin
      Key := #0;
      MessageBeep(0);
    end;
  inherited KeyPress(Key);
  if (Key in [#32..#255]) and (FDataLink.Field <> nil) and
    not FDataLink.Field.IsValidChar(Key) then
  begin
    MessageBeep(0);
    Key := #0;
  end;
  case Key of
    ^H, ^V, ^X, #32..#255:
      FDataLink.Edit;
    #27:
      begin
        FDataLink.Reset;
        SelectAll;
        Key := #0;
      end;
  end;
end;

function TDbLinkEdit.EditCanModify: Boolean;
begin
  Result := FDataLink.Edit;
end;

procedure TDbLinkEdit.Reset;
begin
  FDataLink.Reset;
  SelectAll;
end;

procedure TDbLinkEdit.SetFocused(Value: Boolean);
begin
  if FFocused <> Value then
  begin
    FFocused := Value;
    if (FAlignment <> taLeftJustify) and not IsMasked then Invalidate;
    FDataLink.Reset;
  end;
end;

procedure TDbLinkEdit.MouseUp(Button: TMouseButton; Shift: TShiftState;
  X, Y: Integer);
var
  WasPressed: Boolean;
begin
     { if the mouse was released (after being pressed) on the button
       then perform its associated action }
  WasPressed := fPressed;
  StopTracking;
  BoundsChanged;
  if (Button = mbLeft) and (fLinkStyle = lsEllipsis) and WasPressed then
    EditButtonClick;
  inherited MouseUp(Button, Shift, X, Y);
end;

procedure TDbLinkEdit.TrackButton(X,Y: Integer);
var
  NewState: Boolean;
  R: TRect;
begin
    { Check if the position passed is over the area of the button -
      if so then set the state to pressed and redraw the depressed
      button }
  SetRect(R, ClientWidth - fButtonWidth, 0, ClientWidth, ClientHeight);
  NewState := PtInRect(R, Point(X, Y));
  if fPressed <> NewState then
  begin
    fPressed := NewState;
    InvalidateRect(Handle, @R, False);
  end;
end; // TrackButton

procedure TDbLinkEdit.MouseDown(Button: TMouseButton; Shift: TShiftState;
  X, Y: Integer);
begin
    { Not only must you press the button with the mouse, but it must be
      released over the same button.  If the button has been pressed, we
      need to redraw it depressed, then track the mouse movements to see
      if the user moves off it before releasing. }
  if (Button = mbLeft) and (fLinkStyle <> lsNormal) and focused
  and PtInRect(Rect(Width - fButtonWidth, 0, Width, Height), Point(X,Y)) then
    begin
    MouseCapture := True;
    FTracking := True;
    TrackButton(X, Y);
    end;
  inherited MouseDown(Button, Shift, X, Y);
end; // MouseDown

procedure TDbLinkEdit.MouseMove(Shift: TShiftState; X, Y: Integer);
begin
    { if we are tracking the mouse, the mouse must have been pressed over
      the button part of the control.  Check to see we are still over it. }
  if fTracking then
    TrackButton(X, Y);
  inherited MouseMove(Shift, X, Y);
end; // MouseMove

procedure TDbLinkEdit.StopTracking;
begin
    { we are finished tracking the mouse over the control.  Reset everything }
  if FTracking then
  begin
    TrackButton(-1, -1);
    FTracking := False;
    MouseCapture := False;
  end;
end; // StopTracking;

procedure TDbLinkEdit.DoEnter;
begin
    { In use the elipsis button is only shown when we the control has focus }
  if (fLinkStyle <> lsNormal)
    then BoundsChanged;
  inherited DoEnter;
  if AutoSelect then
    SelectAll;
end; // DoEnter

procedure TDbLinkEdit.DoExit;
begin
    { Remove the elipsis button (if present) when we lose focus }
  if (fLinkStyle <> lsNormal)
    then BoundsChanged;
  inherited DoExit;
end;

function TDbLinkEdit.GetDataField: string;
begin
  Result := FDataLink.FieldName;
end;

function TDbLinkEdit.GetDataSource: TDataSource;
begin
  Result := FDataLink.DataSource;
end;

procedure TDbLinkEdit.SetDataField(const Value: string);
begin
  FDataLink.FieldName := Value;
end;

procedure TDbLinkEdit.SetDataSource(Value: TDataSource);
begin
  FDataLink.DataSource := Value;
  if Value <> nil then Value.FreeNotification(Self);
end;

procedure TDbLinkEdit.WMPaste(var Message: TMessage);
begin
  FDataLink.Edit;
  inherited;
end;

procedure TDbLinkEdit.WMCut(var Message: TMessage);
begin
  FDataLink.Edit;
  inherited;
end;

procedure TDbLinkEdit.CMEnter(var Message: TCMEnter);
begin
  SetFocused(True);
  inherited;
  if SysLocale.FarEast and FDataLink.CanModify then
    inherited ReadOnly := False;
end;

procedure TDbLinkEdit.CMExit(var Message: TWMNoParams);
begin
  try
    FDataLink.UpdateRecord; { tell data link to update database }
  except
    SelectAll;
    SetFocus;
    raise;
  end;
  SetFocused(False);
  CheckCursor;
  DoExit;
end;

function TDbLinkEdit.GetReadOnly: Boolean;
begin
  Result := FDataLink.ReadOnly;
end;

procedure TDbLinkEdit.SetReadOnly(Value: Boolean);
begin
  FDataLink.ReadOnly := Value;
end;

function TDbLinkEdit.GetField: TField;
begin
  Result := FDataLink.Field;
end;

procedure TDbLinkEdit.CMGetDataLink(var Message: TMessage);
begin
  Message.Result := Integer(FDataLink);
end;

function TDbLinkEdit.GetTextMargins: TPoint;
var
  DC: HDC;
  SaveFont: HFont;
  I: Integer;
  SysMetrics, Metrics: TTextMetric;
begin
  if NewStyleControls then
    begin
      if BorderStyle = bsNone then
        I := 0
      else if Ctl3D then
        I := 1
      else
        I := 2;
      Result.X := SendMessage(Handle, EM_GETMARGINS, 0, 0) and $0000FFFF + I;
      Result.Y := I;
    end
  else
    begin
      if BorderStyle = bsNone then
        I := 0
      else
        begin
          DC := GetDC(0);
          GetTextMetrics(DC, SysMetrics);
          SaveFont := SelectObject(DC, Font.Handle);
          GetTextMetrics(DC, Metrics);
          SelectObject(DC, SaveFont);
          ReleaseDC(0, DC);
          I := SysMetrics.tmHeight;
          if I > Metrics.tmHeight then
            I := Metrics.tmHeight;
          I := I div 4;
        end;
      Result.X := I;
      Result.Y := I;
    end;
end;

end.
