{ -------------------------------------------------------------------------------------}
{ An "internet shortcuts" component for Delphi32.                                      }
{ Copyright 1996, Patrick Brisacier and Jean-Fabien Connault.  All Rights Reserved.    }
{ This component can be freely used and distributed in commercial and private          }
{ environments, provided this notice is not modified in any way.                       }
{ -------------------------------------------------------------------------------------}
{ Feel free to contact us if you have any questions, comments or suggestions at        }
{   PBrisacier@mail.dotcom.fr (Patrick Brisacier)                                      }
{   JFConnault@mail.dotcom.fr (Jean-Fabien Connault)                                   }
{ You can always find the latest version of this component at:                         }
{   http://www.worldnet.net/~cycocrew/delphi/                                          }
{ -------------------------------------------------------------------------------------}
{ Date last modified:  03/01/97                                                        }
{ -------------------------------------------------------------------------------------}

{ -------------------------------------------------------------------------------------}
{ TIShortCut v1.03                                                                     }
{ -------------------------------------------------------------------------------------}
{ Description:                                                                         }
{   A component that allows you to manipulate Windows 95's internet shortcuts.         }
{ Properties:                                                                          }
{   property FileIcon: TFileIcon;                                                      }
{   property FileName:String;                                                          }
{   property HotKey: TShortCut;                                                        }
{   property ShellFolder: TShellFolder;                                                }
{   property URL: String;                                                              }
{   property WindowState: TWindowState;                                                }
{   property WorkingDir: String;                                                       }
{ Procedures and functions:                                                            }
{   procedure Read;                                                                    }
{   procedure Write;                                                                   }
{ Needs:                                                                               }
{   FileIcon unit from Patrick Brisacier                                               }
{                                                                                      }
{ See example contained in example.zip file for more details.                          }
{ -------------------------------------------------------------------------------------}
{ Revision History:                                                                    }
{ 1.00:  + Initial release                                                             }
{ 1.01:  + Added ShellFolder property and removed Options property                     }
{ 1.02:  + Removed RegFiles package uses                                               }
{ 1.03:  + The ShellOBJ unit has been replaced by the MyShlobj unit build by           }
{          Brad Stowers (bstowers@pobox.com)                                           }
{ -------------------------------------------------------------------------------------}

unit IShrtCut;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  FileIcon, Menus, CommCtrl, MyShlobj, Registry;

type
  TShellFolder = (sfNone, sfDesktop, sfFavorites, sfFonts, sfPersonal, sfPrograms,
                  sfRecent, sfSendTo, sfStartMenu, sfStartup, sfTemplates);
  TIShortCut = class(TComponent)
  private
    { Dclarations prives }
    FFileName: String;
    FURL: String;
    FWorkingDir: String;
    FWindowState: Integer;
    FHotKey: Word;
    FFileIcon: TFileIcon;
    FShellFolder: TShellFolder;
    procedure SetFileName(AFileName: String);
    procedure SetShellFolder(AShellFolder: TShellFolder);
    function GetWindowState: TWindowState;
    procedure SetWindowState(AWindowState: TWindowState);
    function GetHotKey: TShortCut;
    procedure SetHotKey(AHotKey: TShortCut);
    procedure SetFileIcon(AFileIcon: TFileIcon);
  protected
    { Dclarations protges }
  public
    { Dclarations publiques }
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure Read;
    procedure Write;
  published
    { Dclarations publies }
    property FileIcon: TFileIcon read FFileIcon write SetFileIcon;
    property FileName:String read FFileName write SetFileName;
    property HotKey: TShortCut read GetHotKey write SetHotKey;
    property ShellFolder: TShellFolder read FShellFolder write SetShellFolder default sfNone;
    property URL: String read FURL write FURL;
    property WindowState: TWindowState read GetWindowState write SetWindowState;
    property WorkingDir: String read FWorkingDir write FWorkingDir;
  end;

procedure Register;

const
  ShellFolderKeys: array[TShellFolder] of string =
      ('', 'Desktop', 'Favorites', 'Fonts', 'Personal', 'Programs',
      'Recent', 'SendTo', 'Start Menu', 'Startup', 'Templates');

implementation

procedure Register;
begin
  RegisterComponents('Win95', [TIShortCut]);
end;

{***************************************************************************}
{ TIShortCut                                                                }
{***************************************************************************}

{***************************************************************************}
{ TIShortCut.Create                                                         }
{***************************************************************************}
constructor TIShortCut.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FFileIcon := TFileIcon.Create;
  FShellFolder := sfNone;
end; { TIShortCut.Create }

{***************************************************************************}
{ TIShortCut.Destroy                                                        }
{***************************************************************************}
destructor TIShortCut.Destroy;
begin
  FFileIcon.Free;
  inherited Destroy;
end; { TIShortCut.Destroy }

{***************************************************************************}
{ TIShortCut.SetFileName                                                    }
{***************************************************************************}
procedure TIShortCut.SetFileName(AFileName: String);
var
  Registry: TRegistry;
begin
  if FFileName <> AFileName then begin
    if FShellFolder <> sfNone then begin
      Registry := TRegistry.Create;
      try
        Registry.RootKey := HKey_Current_User;
        Registry.OpenKey('Software\Microsoft\Windows\CurrentVersion\Explorer\Shell Folders', False);
        AFileName := Registry.ReadString(ShellFolderKeys[FShellFolder]) + '\' + ExtractFileName(AFileName);
      finally
        Registry.Free;
      end;
    end;
    if UpperCase(ExtractFileExt(AFileName)) <> '.URL' then
      AFileName := AFileName + '.url';
    if UpperCase(ExtractFileName(AFileName)) = '.URL' then
      AFileName := '';
    FFileName := AFileName;
  end;
end; { TIShortCut.SetFileName }

{***************************************************************************}
{ TIShortCut.SetShellFolder                                                 }
{***************************************************************************}
procedure TIShortCut.SetShellFolder(AShellFolder: TShellFolder);
var
  Registry: TRegistry;
begin
  if FShellFolder <> AShellFolder then begin
    FShellFolder := AShellFolder;
    if FShellFolder <> sfNone then begin
      Registry := TRegistry.Create;
      try
        Registry.RootKey := HKey_Current_User;
        Registry.OpenKey('Software\Microsoft\Windows\CurrentVersion\Explorer\Shell Folders', False);
        FFileName := Registry.ReadString(ShellFolderKeys[FShellFolder]) + '\' + ExtractFileName(FFileName);
      finally
        Registry.Free;
      end;
    end;
  end;
end; { TPBShellLink.SetShellFolder }

{***************************************************************************}
{ TIShortCut.GetWindowState                                                 }
{***************************************************************************}
function TIShortCut.GetWindowState: TWindowState;
begin
  case FWindowState of
  SW_SHOWNORMAL:
    Result := wsNormal;
  SW_SHOWMINNOACTIVE:
    Result := wsMinimized;
  SW_SHOWMAXIMIZED:
    Result := wsMaximized;
  else
    Result := wsNormal;
  end;
end; { TIShortCut.GetWindowState }

{***************************************************************************}
{ TIShortCut.SetWindowState                                                 }
{***************************************************************************}
procedure TIShortCut.SetWindowState(AWindowState: TWindowState);
const
  SW: array[TWindowState] of Integer =
          (SW_SHOWNORMAL, SW_SHOWMINNOACTIVE, SW_SHOWMAXIMIZED);
begin
  FWindowState := SW[AWindowState];
end; { TIShortCut.SetWindowState }

{***************************************************************************}
{ TIShortCut.GetHotKey                                                      }
{***************************************************************************}
function TIShortCut.GetHotKey: TShortCut;
const
  Sh: array[Boolean] of TShiftState = ([], [ssShift]);
  Ct: array[Boolean] of TShiftState = ([], [ssCtrl]);
  Al: array[Boolean] of TShiftState = ([], [ssAlt]);
var
  Shift: TShiftState;
begin
  Shift :=  Sh[(Hi(FHotKey) and HOTKEYF_SHIFT = HOTKEYF_SHIFT)]
          + Ct[(Hi(FHotKey) and HOTKEYF_CONTROL = HOTKEYF_CONTROL)]
          + Al[(Hi(FHotKey) and HOTKEYF_ALT = HOTKEYF_ALT)] ;
  Result := ShortCut(Lo(FHotKey), Shift);
end; { TIShortCut.GetHotKey }

{***************************************************************************}
{ TIShortCut.SetHotKey                                                      }
{***************************************************************************}
procedure TIShortCut.SetHotKey(AHotKey: TShortCut);
var
  Key: Word;
  Shift: TShiftState;
begin
  ShortCutToKey(AHotKey, Key, Shift);
  Key := Swap(Key);
  if ssShift in Shift then Key := Key + HOTKEYF_SHIFT;
  if ssCtrl in Shift then Key := Key + HOTKEYF_CONTROL;
  if ssAlt in Shift then Key := Key + HOTKEYF_ALT;
  Key := Swap(Key);
  FHotKey := Key;
end; { TIShortCut.SetHotKey }

{***************************************************************************}
{ TIShortCut.SetFileIcon                                                    }
{***************************************************************************}
procedure TIShortCut.SetFileIcon(AFileIcon: TFileIcon);
begin
  FFileIcon.Assign(AFileIcon);
end; { TIShortCut.SetFileIcon }

{***************************************************************************}
{ TIShortCut.Read                                                           }
{***************************************************************************}
procedure TIShortCut.Read;
var
  SL: TStringList;
begin
  SL := TStringList.Create;
  try
    SL.LoadFromFile(FFileName);
    FURL := SL.Values['URL'];
    FWorkingDir := SL.Values['WorkingDirectory'];
    FFileIcon.FileName := SL.Values['IconFile'];
    try
      FFileIcon.IconIndex := StrToInt(SL.Values['IconIndex']);
    except
      on EConvertError do FFileIcon.IconIndex := 0;
    end;
    try
      FHotKey := StrToInt(SL.Values['HotKey']);
    except
      on EConvertError do FHotKey := 0;
    end;
    try
      FWindowState := StrToInt(SL.Values['ShowCommand']);
    except
      on EConvertError do FWindowState := SW_SHOWNORMAL;
    end;
  finally
    SL.Free;
  end;
end; { TIShortCut.Read }

{***************************************************************************}
{ TIShortCut.Write                                                          }
{***************************************************************************}
procedure TIShortCut.Write;
var
  SL: TStringList;
  PFileName: array [0..255] of Char;
begin
  SL := TStringList.Create;
  try
    SL.Add('[InternetShortcut]');
    SL.Values['URL'] := FURL;
    SL.Values['WorkingDirectory'] := FWorkingDir;
    SL.Values['IconFile'] := FFileIcon.FileName;
    SL.Values['IconIndex'] := IntToStr(FFileIcon.IconIndex);
    SL.Values['HotKey'] := IntToStr(FHotKey);
    if FWindowState <> SW_SHOWNORMAL then
      SL.Values['ShowCommand'] := IntToStr(FWindowState);
    SL.SaveToFile(FFileName);
    { Averti le Shell que l'icone a chang }
    StrPCopy(PFileName, FFileName);
    SHChangeNotify(SHCNE_UPDATEITEM, SHCNF_PATH, @PFileName, nil);
  finally
    SL.Free;
  end;
end; { TIShortCut.Write }

end.
