unit ipl2;
{
 *  INTEL Image Processing Library function prototypes 
 *  Conversion to Delphi 2,3 ver. 0.7 by Peter G.Darakhvelidze
 *  E-mail:     petr@p-media.spb.ru
 *  Works _only_ vith IPL ver. 2.0 and higher
 *  because of INTEL non-compatible changes
 *  Translate from  ipl.h,iplmisc.h,iplerror.h Revision: 31 
 *  Date: 8.02.98 15:49 $
 *  Original comments included
 }
interface
 uses Windows;
 {some useful types}
 type
  float = single;
  pInteger = ^Integer;
  pBoolean = ^Boolean;

  pByteArray = ^TByteArray;
  TByteArray = array[0..65535] of Byte;

  pWordArray = ^TWordArray;
  TWordArray = array[0..65535] of Word;

  pShortintArray = ^TShortintArray;
  TShortintArray = array[0..65535] of Shortint;

  pSmallintArray = ^TSmallintArray;
  TSmallintArray = array[0..65535] of Smallint;

  pIntegerArray = ^TIntegerArray;
  TIntegerArray = array [0..65535] of Integer;

  pFloatArray = ^TFloatArray;
  TFloatArray = Array[0..65535] of Float;

  pSingleArray = ^TSingleArray;
  TSingleArray = Array[0..65535] of Single;
  
 const
 IPL_EPS = 1.0E-12;
 IPL_PI  = 3.14159265358979324;
 IPL_2PI = 6.28318530717958648;
 IPL_PI_2 = 1.57079632679489662;
 IPL_PI_4 = 0.785398163397448310;

 function IPL_DegToRad(deg: double):double;

// ==== Common Definitions ===================================================}
const
 IPL_DEPTH_SIGN = $80000000;               //ci+to select sign tag      }
 IPL_DEPTH_MASK = $7FFFFFFF;               //ci+to select pixel size    }

 IPL_DEPTH_1U   =  1;
 IPL_DEPTH_8U   =  8;
 IPL_DEPTH_16U  = 16;
 IPL_DEPTH_32F  = 32;

 IPL_DEPTH_8S =  IPL_DEPTH_SIGN or 8;       //cs= }
 IPL_DEPTH_16S = IPL_DEPTH_SIGN or 16;       //s= }
 IPL_DEPTH_32S = IPL_DEPTH_SIGN or 32;       //cs= }

 IPL_DATA_ORDER_PIXEL = 0;
 IPL_DATA_ORDER_PLANE = 1;

 IPL_ORIGIN_TL = 0;
 IPL_ORIGIN_BL = 1;

 IPL_ALIGN_DWORD = 4;                       //ci=align in bytes          }
 IPL_ALIGN_QWORD = 8;                       //ci=}

 IPL_GET_TILE_TO_READ  = 1;
 IPL_GET_TILE_TO_WRITE = 2;
 IPL_RELEASE_TILE      = 4;

 IPL_LUT_LOOKUP = 0;                        //ck+ }
 IPL_LUT_INTER  = 1;                        //ck+ }

 IPL_CSEQ_G     = $00000047;      // "G"
 IPL_CSEQ_GRAY  = $59415247;      // "GRAY"
 IPL_CSEQ_BGR   = $00524742;      // "BGR"
 IPL_CSEQ_BGRA  = $41524742;      // "BGRA"
 IPL_CSEQ_RGB   = $00424752;      // "RGB"
 IPL_CSEQ_RGBA  = $41424752;      // "RGBA"

{==== Common Types =========================================================}

type

 pIplROI = ^TIplROI;
 TIplROI = RECORD
    coi  : integer;                {=cg, "unsigned integer"  ->  "integer"    }
    xOffset: integer;
    yOffset: integer;
    width: integer;
    height: integer;
  end;

 pIplImage = ^TIplImage;//   forward
 TIplCallBack = procedure (const img: pIplImage; xIndex,yIndex:integer;
  mode:  integer);
{
   img           - header provided for the parent image
   xIndex,yIndex - indices of the requested tile. They refer to the tile
                   number not pixel number, and count from the origin at (0,0)
   mode          - one of the following:
      IPL_GET_TILE_TO_READ  - get a tile for reading;
                              tile data is returned in "img->tileInfo->tileData",
                              and must not be changed
      IPL_GET_TILE_TO_WRITE - get a tile for writing;
                              tile data is returned in "img->tileInfo->tileData"
                              and may be changed;
                              changes will be reflected in the image
      IPL_RELEASE_TILE      - release tile; commit writes
}
  pIplTileInfo = ^TIplTileInfo;
  TIplTileInfo = record
    callBack: TIplCallBack;         { callback function }
    id : pointer;               { additional identification field !!!}
    tileData: pChar;         { pointer on tile data !!!}
    width: integer;            { width of tile }
    height: integer;           { height of tile }
  end;

 TIplImage = record
    nSize: integer;              { size of iplImage struct          }
    ID: integer;                 { version                          }
    nChannels: integer;
    alphaChannel: integer;
    depth: integer;              { pixel depth in bits              }
    colorModel: array[0..3] of char;
    channelSeq: array[0..3] of char;
    dataOrder: integer;
    origin: integer;
    align: integer;              { 4 or 8 byte align                }
    width: integer;
    height: integer;
    roi: pIplROI;
    maskROI: pIplImage;
    imageId: Pointer;            { use of the application           }
    tileInfo: pIplTileInfo;           { contains information on tiling   }
    imageSize: integer;          { useful size in bytes             }
    imageData: pChar;          {cs=pointer to aligned image       }
    widthStep: integer;          {ci+size of aligned line in bytes  }
    BorderMode: array [0..3] of integer;      { ci+                              }
    BorderConst: array [0..3] of integer;     { ci+                              }
    imageDataOrigin: pChar;    {ci+ptr to full, nonaligned image  }
 end;

 pIplLUT = ^TIplLUT;
 TIplLUT = record
    num: integer;                {ck+no of keys or values   }
    key: pIntegerArray;
    value: pIntegerArray;
    factor: pIntegerArray;
    interpolateType: integer;
 end;

 pIplColorTwist = ^TIplColorTwist;
 TIplColorTwist = record
    data: array [0..15] of integer;
    scalingValue: integer;
 end;

 pIplConvKernel = ^TIplConvKernel;
 TIplConvKernel = record
    nCols: integer;
    nRows: integer;
    anchorX: integer;
    anchorY: integer;
    values:  pIntegerArray;
    nShiftR: integer;
 end;

 pIplConvKernelFP = ^TIplConvKernelFP;
 TIplConvKernelFP = record
    nCols: integer;
    nRows: integer;
    anchorX: integer;
    anchorY: integer;
    values: pSingleArray;
 end;
 //TIplFilter
 //typedef enum
  const
     IPL_PREWITT_3x3_V=1;
     IPL_PREWITT_3x3_H=2;
     IPL_SOBEL_3x3_V  =3;   // vertical */
     IPL_SOBEL_3x3_H  =4; // horizontal */
     IPL_LAPLACIAN_3x3=5;
     IPL_LAPLACIAN_5x5=6;
     IPL_GAUSSIAN_3x3 =7;
     IPL_GAUSSIAN_5x5 =8;
     IPL_HIPASS_3x3   =9;
     IPL_HIPASS_5x5   =10;
     IPL_SHARPEN_3x3  =11;


{==== Functions headers ===================================================}
const
 IPL_BORDER_CONSTANT  = 0;
 IPL_BORDER_REPLICATE = 1;
 IPL_BORDER_REFLECT   = 2;
 IPL_BORDER_WRAP      = 3;

{ Indexes to access IplImage.BorderMode[],IplImage.BorderConst[]          }
 IPL_SIDE_TOP_INDEX   =  0;
 IPL_SIDE_BOTTOM_INDEX = 1;
 IPL_SIDE_LEFT_INDEX   = 2;
 IPL_SIDE_RIGHT_INDEX  = 3;

{ values of argument of iplSetBorderMode(,,border,)                       }
 IPL_SIDE_TOP    =    1 shl IPL_SIDE_TOP_INDEX;
 IPL_SIDE_BOTTOM  =   1 shl IPL_SIDE_BOTTOM_INDEX;
 IPL_SIDE_LEFT    =   1 shl IPL_SIDE_LEFT_INDEX;
 IPL_SIDE_RIGHT   =   1 shl IPL_SIDE_RIGHT_INDEX;

procedure iplSetBorderMode(src: pIplImage; mode: integer; border: integer;
 constVal: integer);stdcall;
{
mode    The following modes are supported.
        IPL_BORDER_CONSTANT     The value constVal will be used for all pixels.
        IPL_BORDER_REPLICATE    The last row or column will be replicated
                                for the border.
        IPL_BORDER_REFLECT      The last n rows or columns will be reflected
                                in reverse order to create the border.
        IPL_BORDER_WRAP         The required border rows or columns are taken
                                from the opposite side of  the image.
border  The side that this function is called for. Can be an OR of one or more
        of the following four sides of an image:
        IPL_SIDE_TOP            Top side.
        IPL_SIDE_BOTTOM         Bottom side.
        IPL_SIDE_LEFT           Left side.
        IPL_SIDE_RIGHT          Right side.
        If  no mode has been set for a side, the default IPL_BORDER_CONSTANT
constVal   The value to use for the border when the mode is set
           to IPL_BORDER_CONSTANT.
}

function iplCreateImageHeader(nChannels: integer; alphaChannel: integer;
         depth:integer;  colorModel, channelSeq:pchar;   dataOrder: integer;
         origin: integer; align: integer;
         width, height:integer; roi: pIplROI; maskROI:pIplImage;
         imageId: Pointer; tileInfo: pIplTileInfo):pIplImage;stdcall;

{
nChannels       Number of channels in the image.
alphaChannel    Alpha channel number (0 if no alpha channel in image).
depth           Bit depth of  pixels. Can be one of
                IPL_DEPTH_1U,
                IPL_DEPTH_8U,
                IPL_DEPTH_8S,
                IPL_DEPTH_16U,
                IPL_DEPTH_16S,
                IPL_DEPTH_32S.
colorModel      A four character array describing the color model,
                e.g. "RGB", "GRAY", "MSI" etc.
channelSeq      The sequence of channels in the image,
                e.g. "BGR" for an RGB image.
dataOrder       IPL_DATA_ORDER_PIXEL or IPL_DATA_ORDER_PLANE.
origin          The origin of the image.
                Can be IPL_ORIGIN_TL or IPL_ORIGIN_BL.
align           Alignment of image data.
                Can be IPL_ALIGN_DWORD or IPL_ALIGN_QWORD.
height          Height of  the image in pixels.
width           Width of  the image in pixels.
roi             Pointer to an ROI (region of interest) structure.
                This can be NULL (implying a region of interest comprising
                all channels and the entire image area).
maskROI         Pointer to an mask ROI for image
imageId         use of the application
tileInfo        contains information on tiling
Return Value    The newly constructed IPL image header.
}

const
 IPL_NOFILL = $FFFFFFFF;                   {ci+nofill flag           }

procedure iplAllocateImage(image: pIplImage; doFill, fillValue: integer);stdcall;

{
image           An IPL image header with a NULL image data pointer.
                The image data pointer will be set to newly allocated
                image data memory after calling this function.
doFill          Use a 0 to leave the pixel data uninitialized.
                Use a not 0 to initialized the pixel data of fillValue
fillValue       The initial value to use for pixel data.
}

procedure iplAllocateImageFP(image: pIplImage; doFill: Integer; fillValue: float);stdcall;
{
image           An IPL image header (IPL_DEPTH_32F) with a NULL image data
                pointer.
                The image data pointer will be set to newly allocated
                image data memory after calling this function.
doFill          Use a 0 to leave the pixel data uninitialized.
                Use a not 0 to initialized the pixel data of fillValue
fillValue       The initial value to use for pixel data.
}

procedure iplDeallocateImage(image: pIplImage);stdcall;

{
image           An IPL image header with a pointer
                to allocated image data memory. The image data pointer
                will be set to NULL after this function executes.
}
const
 IPL_IMAGE_HEADER = 1;
 IPL_IMAGE_DATA   = 2;
 IPL_IMAGE_ROI    = 4;
 IPL_IMAGE_TILE   = 8;
 IPL_IMAGE_MASK  = 16;
 IPL_IMAGE_ALL = IPL_IMAGE_HEADER or IPL_IMAGE_DATA or
                       IPL_IMAGE_TILE or IPL_IMAGE_ROI or IPL_IMAGE_MASK;
 IPL_IMAGE_ALL_WITHOUT_MASK = IPL_IMAGE_HEADER or IPL_IMAGE_DATA or
                       IPL_IMAGE_TILE or IPL_IMAGE_ROI;

procedure iplDeallocate(Image: pIplImage; flag: integer);stdcall;
{
  image         An IPL image header
  flag          what item to free:
     IPL_IMAGE_HEADER - free header structure
     IPL_IMAGE_DATA   - free image data, set pointer to NULL
     IPL_IMAGE_ROI    - free image roi, set pointer to NULL
     IPL_IMAGE_TILE   - free image tile, set pointer to NULL
     IPL_IMAGE_MASK   - free image maskROI, set pointer to NULL
     IPL_IMAGE_ALL    - free image data, roi, header, maskROI, tile
     IPL_IMAGE_ALL_WITHOUT_MASK
                      - as well as IPL_IMAGE_ALL, but maskROI does not release
}


function iplCreateTileInfo( callBack: TIplCallBack;
                              id: Pointer; width, height: integer): pIplTileInfo;stdcall;
{
callBack           callback function
id                 additional identification field
width              width of tile
height             height of tile

Return value      Newly constructed TileInfo structure.
}


procedure iplDeleteTileInfo(tileInfo: pIplTileInfo);stdcall;
{
  tileInfo         An IPL TileInfo header
}

procedure iplSetTileInfo(tileInfo: pIplTileInfo;
          callBack: TIplCallBack; id: Pointer; Width, Height: integer);stdcall;
{
tileInfo           The TileInfo structure to modify.
callBack           callback function
id                 additional identification field
width              width of tile
height             height of tile
}

function iplCreateROI(coi:Integer; xOffset, yOffset:  integer;
   width, height: integer):pIplROI;stdcall;
{
coi            The channel region of interest.
               It can be set to 0 (for all channels) or a specific channel number.
xOffset,
yOffset        The offset from the origin of the rectangular region.
width ,height  The size of the rectangular region.

Return value   Newly constructed ROI structure.
}

procedure iplDeleteROI(roi: pIplROI);stdcall;
{
  roi         An IPL ROI header
}


procedure iplSetROI(roi: pIplROI; coi: integer;
    xOffset,yOffset : integer; width,height: Integer);stdcall;
{
roi            The ROI structure to modify.
coi            The channel region of interest.
               It can be set to 0 (for all channels) or a specific channel number.
xOffset,
yOffset        The offset from the origin of the rectangular region.
width,height   The size of the rectangular region.
}

function iplTranslateDIB(dib: pBITMAPINFOHEADER; var cloneData:BOOLEAN):pIplImage;stdcall;
{
dib            The DIB image.
cloneData      A boolean, result of work of the function. If true, the image
               data pointer in IPL image is made to point to the DIB image data.
               Otherwise the DIB image data was converting to the IPL image data.
Return Value   The constructed IPL image.
}

procedure iplConvertFromDIB(dib: pBITMAPINFOHEADER; Image: pIplImage);stdcall;
{
dib            The input DIB image.
image          The IPL image header with specified attributes.
               If the data pointer is NULL, image data memory
               will be allocated and the pointer set to it.
}
const
 IPL_PALCONV_NONE     = 0;
 IPL_PALCONV_POPULATE = 1;
 IPL_PALCONV_MEDCUT   = 2;

procedure iplConvertToDIB(Image: pIplImage; dib: pBITMAPINFOHEADER; dither: integer;
                     paletteConversion: integer);stdcall;
{
image          The input IPL image.
dib            The output DIB image.
dither         The dithering algorithm to use if applicable.
               Dithering will be done if  the bit depth in the DIB
               is less than that of the IPL image.
               The following algorithms are supported:
               IPL_DITHER_NONE     No dithering is done.
                                   The most significant bits in the IPL
                                   image pixel data are retained.
               IPL_DITHER_STUCKEY  The stuckey dithering algorithm is used.
paletteConversion  Applicable when the DIB is a palette image.
               Specifies the palette algorithm to use when converting
               the IPL absolute color image.
               The following options are supported:
               IPL_PALCONV_NONE    The existing palette in the DIB is used.
               IPL_PALCONV_POPULATE The popularity palette conversion
                                   algorithm is used.
               IPL_PALCONV_MEDCUT  The median cut algorithm palette conversion
                                   algorithm is used.
}
{                       }

procedure iplCopy(srcImage, dstImage: pIplImage);stdcall;
procedure iplSet(dstImage: pIplImage; value: integer);stdcall;
procedure iplSetFP(dstImage: pIplImage; value: Float);stdcall;
procedure iplExchange(Image1, Image2: pIplImage);stdcall;

procedure iplConvert(srcImage, dstImage: pIplImage);stdcall;

procedure iplBitonalToGray(srcImage, dstImage: pIplImage;
           ZeroScale, OneScale: integer);stdcall;


{   Monadic Operations  }

procedure iplAddS(srcImage,dstImage: pIplImage; value:integer);stdcall;
procedure iplAddSFP(srcImage,dstImage: pIplImage; value:float);stdcall;
{
srcImage       The source image.
dstImage       The resultant image.
value          The value to increase the pixel values by.
}

procedure iplSubtractS(srcImage,dstImage:  pIplImage;
                       value: Integer; flip:Boolean);stdcall;
procedure iplSubtractSFP(srcImage,dstImage:  pIplImage;
                       value: float; flip:Boolean);stdcall;
{
srcImage       The source image.
dstImage       The resultant image.
value          The value to decrease the pixel values by.
flip           A boolean that is used to change the order of subtraction.
               If false the result pixel value is computed as
                  result = pixel_value - value, where pixel_value is the input
                                                pixel value.
               If true, the result pixel value is computed as
                  result = value - pixel_value.
}

procedure iplMultiplyS(srcImage, dstImage: pIplImage; value: Integer);stdcall;
procedure iplMultiplySFP(srcImage, dstImage: pIplImage; value: float);stdcall;
{
srcImage       The source image.
dstImage       The resultant image.
value          A positive value to multiply the pixel values with.
}

procedure iplAbs(srcImage, dstImage: pIplImage);stdcall;
{
srcImage        The source image.
dstImage        The resultant image.
}

procedure iplNot(srcImage, dstImage: pIplImage);stdcall;
{
srcImage       The source image.
dstImage       The resultant image.
}

procedure iplLShiftS(srcImage, dstImage: pIplImage; value: Cardinal);stdcall;
{
srcImage       The source image.
dstImage       The resultant image.
nShift         The number of bits to shift each pixel value to the left to.
}

procedure iplRShiftS(srcImage, dstImage: pIplImage; nShift: Cardinal);stdcall;
{
srcImage       The source image.
dstImage       The resultant image.
nShift         The number of bits to shift each pixel value to the right to.
}

procedure iplSquare(srcImage, dstImage: pIplImage);stdcall;
{
srcImage       The source image.
dstImage       The resultant image.
}

procedure iplAndS(srcImage, dstImage: pIplImage; value: Cardinal);stdcall;
{
srcImage       The source image.
dstImage       The resultant image.
value          The bit sequence used to perform the bitwise operation
               on each pixel. Only the number of bits corresponding to the data
               type of the image are used
               (See under "Data Type" in the chapter "Data Architecture").
}

procedure iplOrS(srcImage, dstImage: pIplImage; value: Cardinal);stdcall;
{
srcImage        The source image.
dstImage        The resultant image.
value           The bit sequence used to perform the bitwise operation
                on each pixel. Only the number of bits corresponding
                to the data type of the image are used
                (See under "Data Type" in the chapter "Data Architecture").
}

procedure iplXorS(srcImage, dstImage: pIplImage; value: Cardinal);stdcall;
{
srcImage        The source image.
dstImage        The resultant image.
value           The bit sequence used to perform the bitwise operation
                on each pixel. Only the number of bits corresponding to the data
                type of the image are used
                (See under "Data Type" in the chapter "Data Architecture").
}

{      Dyadic Operations       }
{------------------------------}

procedure iplAnd(srcImageA, srcImageB, dstImage: pIplImage);stdcall;
{
srcImageA       The first source image.
srcImageB       The second source image.
dstImage        The image resulting from the bitwise operation between input
                images srcImageA and srcImageB.
}

procedure iplOr(srcImageA, srcImageB, dstImage: pIplImage);stdcall;
{
srcImageA       The first source image.
srcImageB       The second source image.
dstImage        The image resulting from the bitwise operation
                between input images srcImageA and srcImageB.
}

procedure iplXor(srcImageA, srcImageB, dstImage: pIplImage);stdcall;
{
srcImageA       The first source image.
srcImageB       The second source image.
dstImage        The image resulting from the bitwise operation
                between input images srcImageA and srcImageB.
}

procedure iplAdd(srcImageA, srcImageB, dstImage: pIplImage);stdcall;
{
srcImageA       The first source image.
srcImageB       The second source image.
dstImage        The resultant image obtained as
                dstImage  = srcImageA  + srcImageB.
}

procedure iplSubtract(srcImageA, srcImageB, dstImage: pIplImage; flip:Boolean);stdcall;
{
srcImageA       The first source image.
srcImageB       The second source image.
dstImage        If flip is false, the resultant image obtained as
                dstImage  = srcImageA  - srcImageB, otherwise it is
                dstImage  = srcImageB  - srcImageA
flip            A boolean to modify the order in which the input images
                are subtracted. See description above under dstImage.
}

procedure iplMultiply(srcImageA, srcImageB, dstImage: pIplImage);stdcall;
{
srcImageA       The first source image.
srcImageB       The second source image.
dstImage        The resultant image obtained as
                dstImage  = srcImageA  * srcImageB.
}
procedure iplMultiplySScale(srcImage, dstImage: pIplImage;value: Integer);stdcall;
{
srcImage        The source image.
dstImage        The resultant image.
value           A positive value to multiply the pixel values with
                dstImage  = (srcImage  * value) / VAL_MAX
Notes:          value becomes 0 <= value <= VAL_MAX
}

procedure iplMultiplyScale(srcImageA, srcImageB, dstImage: pIplImage);stdcall;
{
srcImageA       The first source image.
srcImageB       The second source image.
dstImage        The resultant image obtained as
                dstImage  = (srcImageA  * srcImageB) / VAL_MAX
}

const
 IPL_COMPOSITE_OVER  =  0;
 IPL_COMPOSITE_IN    =  1;
 IPL_COMPOSITE_OUT   =  2;
 IPL_COMPOSITE_ATOP  =  3;
 IPL_COMPOSITE_XOR   =  4;
 IPL_COMPOSITE_PLUS  =  5;

procedure iplAlphaComposite(srcImageA, srcImageB, dstImage: pIplImage;
   compositeType: integer;
   alphaImageA, alphaImageB, alphaImageDst: pIplImage;
   premulAlpha, divideMode: boolean);stdcall;

procedure iplAlphaCompositeC(srcImageA, srcImageB, dstImage: pIplImage;
  compositeType, aA, aB: Integer;
  premulALpha, divideMode: Boolean);stdcall;

{
srcImageA       The foreground image.
srcImageB       The background image.
dstImage        The resultant image.
compositeType   The type of composition to perform.
                See the table "Compositing Operations" below for a description.
aA              The constant alpha value to use for the source image srcImageA.
                Should be a positive number.
aB              The constant alpha value to use for the source image srcImageB.
                Should be a positive number.
alphaImageA     The image to use as the  alpha channel for srcImageA.

alphaImageB     The image to use as the  alpha channel for srcImageB.
alphaImageDst   The image to use as the  alpha channel for dstImage.
premulAlpha     Indicates that the input images contain
                premultiplied alpha values.
divideMode      Normally set to false. When true, the result pixel color
                (in table "Image Compositing Operations") is further divided
                by the result alpha value to get the final result pixel color.
}

procedure iplPreMultiplyAlpha(Image: pIplImage; alphaValue: integer);stdcall;
{
image           The image for which the alpha premultiplication is done.
alphaValue      The global alpha value to use in the range 0 to 256.
                If this value is negative (e.g. -1) the internal alpha channel
                of the image is used (it is an error if an alpha channel does
                not exist).
}

{         Linear Filters          }
{---------------------------------}


procedure iplBlur(srcImage, dstImage: pIplImage; nCols, nRows: integer;
          anchorX, anchorY: integer);stdcall;
{
srcImage        The source image.
dstImage        The resultant image.
nRows           Number of rows in the neighbourhood to use.
nCols           Number of columns in the neighbourhood to use.
anchorX,
anchorY         The [x,y] coordinates of the anchor cell in the neighbourhood.
}

function iplCreateConvKernel(nCols, nRows: integer;
   anchorX, anchorY: integer; values: pIntegerArray; nShiftR:integer):pIplConvKernel;stdcall;
function iplCreateConvKernelFP(nCols, nRows: integer;
   anchorX, anchorY: integer; values: pSingleArray):pIplConvKernelFP;stdcall;
function iplCreateConvKernelChar(nCols, nRows: integer;
   anchorX, anchorY: integer; values: pChar; nShiftR:integer):pIplConvKernel;stdcall;
{
nRows           The number of rows in the convolution kernel.
nCols           The number of columns in the convolution kernel.
anchorX,
anchorY         The [x,y] coordinates of the anchor cell in the kernel.
nShiftR         The number of  bits to shift (to the right) the resulting
                output pixel of each convolution.
Return Value    A pointer to the convolution kernel structure IplConvKernel.
}

procedure iplDeleteConvKernel(kernel: pIplConvKernel);stdcall;
procedure iplDeleteConvKernelFP(kernel: pIplConvKernelFP);stdcall;

procedure iplGetConvKernel(kernel: pIplConvKernel; nCols, nRows: pinteger;
          anchorX, anchorY: pinteger; var values: pIntegerArray; var nShiftR:integer);stdcall;//!!!
procedure iplGetConvKernelFP(kernel: pIplConvKernelFP; nCols, nRows: pinteger;
          anchorX, anchorY: pinteger; var values: pSingleArray; var nShiftR:integer);stdcall;//!!!
procedure iplGetConvKernelChar(kernel: pIplConvKernel; nCols, nRows: pinteger;
          anchorX, anchorY: pinteger; var values: pChar; var nShiftR:integer);stdcall;//!!!
{

kernel          The kernel to get the attributes for.
                The attributes are returned in the remaining arguments.
nRows           A pointer to the number of rows in the convolution kernel.
                Set by the function.
nCols           A pointer to the the number of columns in the convolution kernel.
                Set by the function.
anchorX,
anchorY         Pointers to the [x,y] coordinates of the anchor cell
                in the kernel.
nShiftR         A pointer to the number of  bits to shift (to the right)
                the resulting output pixel of each convolution.
                Set by the function.
}
const
 IPL_SUM        = 0;
 IPL_SUMSQ      = 1;
 IPL_SUMSQROOT  = 2;
 IPL_MAX        = 3;
 IPL_MIN        = 4;

procedure iplConvolve2D(srcImage, dstImage: pIplImage;
 kernels: Pointer; nKernels:integer; combineMethod: integer);stdcall;
procedure iplConvolve2DFP(srcImage, dstImage: pIplImage;
 kernels: Pointer; nKernels:integer; combineMethod: integer);stdcall;//!!!
{
srcImage        The source image.
dstImage        The resultant image.
kernel          A pointer to an array of pointers to convolution kernels.
                The length of the array is nKernels. Can be one or more kernels.
nKernels        The number of kernels in the array kernel.
combineMethod   The way in which the results of applying each kernel
                should be combined. This argument is ignored
                when a single kernel is used. The following combinations
                are supported:
                IPL_SUM         Sums the results.
                IPL_SUMSQ       Sums the squares of the  results.
                IPL_SUMSQROOT   Sums the squares of the results
                                and then takes the sqaure root.
                IPL_MAX         Takes the maximum of the results.
                IPL_MIN         Takes the minimum of the results.
}
procedure iplFixedFilter(srcImage, dstImage: pIplImage; filter: Cardinal);stdcall;
procedure iplConvolveSep2D(srcImage, dstImage: pIplImage;
 xKernel, yKernel: pIplConvKernel);stdcall;
{
srcImage        The source image.
dstImage        The resultant image.
xKernel         The x or row kernel. Should contain only one row.
ykernel         The y or column kernel. Should contain only one column.
}


{          Non Linear Filters       }
{-----------------------------------}

procedure iplMedianFilter(srcImage, dstImage: pIplImage;
          nCols, nRows: integer ; anchorX, anchorY: integer);stdcall;
{
srcImage        The source image.
dstImage        The resultant image.
nRows           Number of rows in the neighbourhood to use.
nCols           Number of columns in the neighbourhood to use.
anchorX,
anchorY         The [x,y] coordinates of the anchor cell in the neighbourhood.
}

procedure iplMaxFilter(srcImage, dstImage: pIplImage;
          nCols, nRows: integer; anchorX, anchorY: integer );stdcall;
{
srcImage        The source image.
dstImage        The resultant image.
nRows           Number of rows in the neighbourhood to use.
nCols           Number of columns in the neighbourhood to use.
anchorX,
anchorY         The [x,y] coordinates of the anchor cell in the neighbourhood.
}

procedure iplMinFilter(srcImage, dstImage: pIplImage;
          nCols, nRows: integer; anchorX,anchorY: integer );stdcall;
{
srcImage        The source image.
dstImage        The resultant image.
nRows           Number of rows in the neighbourhood to use.
nCols           Number of columns in the neighbourhood to use.
anchorX,
anchorY         The [x,y] coordinates of the anchor cell in the neighbourhood.
}


{   Fast Fourier Transform }
{--------------------------}

{ch+
  Parameter  flags  in Functions  iplRealFft2D  and  iplCcsFft2D
    IPL_FFT_Forw       - forward transform.
    IPL_FFT_Inv        - inverse transform.
    IPL_FFT_NoScale    - in inverse transform absent to scale.
    IPL_FFT_DoAlpha    - Alpha channel to be transform (if alphaChannel != 0).
    IPL_FFT_UseInt     - use only integer   core.
    IPL_FFT_UseFloat   - use only float core.
    IPL_FFT_OnlyOffsetROIandCalc - take only offset ROI and then calc.
    IPL_FFT_Free       - only free all working arrays and exit.
    IPL_FFT_Save       - save all working arrays on exit.
}
const
  IPL_FFT_Forw                 =   1;
  IPL_FFT_Inv                  =   2;
  IPL_FFT_NoScale              =   4;
  IPL_FFT_DoAlpha              =   8;
  IPL_FFT_UseInt               =  16;
  IPL_FFT_UseFloat             =  32;
  IPL_FFT_OnlyOffsetROIandCalc =  64;
  IPL_FFT_Free                 = 128;
  IPL_FFT_Save                 = 256;

procedure iplRealFft2D(srcImage, dstImage: pIplImage; flags: Integer);stdcall;
{ch+
srcImage        The source image. Any mask ROI specified will be ignored.
dstImage        The resultant image in CcsPerm2D format containing
                the Fourier coefficients. This image cannot be the same
                as the input image srcImage Any mask ROI specified will be
                ignored. This image should be a multi-channel image containing
                as many channels as srcImage. The data type for the image
                should be 8, 16 or 32 bits.
flags           Specifies how the FFT should be performed.
                Integer number in which every bit have next specifications:
                IPL_FFT_Forw       - forward transform,
                IPL_FFT_Inv        - inverse transform,
                IPL_FFT_NoScale    - in inverse transform absent to scale,
                IPL_FFT_DoAlpha    - Alpha channel to be transform
                                     (if alphaChannel != 0),
                IPL_FFT_UseInt     - use only integer   core,
                IPL_FFT_UseFloat   - use only float core,
                IPL_FFT_OnlyOffsetROIandCalc - take only offset ROI
                                               and then calc,
                IPL_FFT_Free       - only free all working arrays and exit,
                IPL_FFT_Save       - save all working arrays on exit.
}

procedure iplCcsFft2D(srcImage, dstImage: pIplImage; flags: Integer);stdcall;
{ch+
srcImage        The source image in CcsPerm2D format.
                Any mask ROI specified will be ignored.
dstImage        The resultant image. This image cannot be the same as
                the input image srcImage.  Any mask ROI specified will be ignored.
                This image should be a multi-channel image containing as many
                channels as srcImage.
flags           Specifies how the FFT should be performed.
                Integer number in which every bit have next specifications:
                IPL_FFT_Forw       - forward transform,
                IPL_FFT_Inv        - inverse transform,
                IPL_FFT_NoScale    - in inverse transform absent to scale,
                IPL_FFT_DoAlpha    - Alpha channel to be transform
                                     (if alphaChannel != 0),
                IPL_FFT_UseInt     - use only integer   core,
                IPL_FFT_UseFloat   - use only float core,
                IPL_FFT_OnlyOffsetROIandCalc - take only offset ROI
                                               and then calc,
                IPL_FFT_Free       - only free all working arrays and exit,
                IPL_FFT_Save       - save all working arrays on exit.
}


{   Discrete Cosine Transform  }
const
  IPL_DCT_Forward   =   1;
  IPL_DCT_Inverse   =   2;
  IPL_DCT_DoAlpha   =   4;
  IPL_DCT_Free      =   8;
  IPL_DCT_UseInpBuf =   16;

procedure iplDCT2D(src,dst: pIplImage; flags: Integer );stdcall;

{
srcImage        The source image. Any mask ROI specified will be ignored.
dstImage        The resultant image containing the DCT coefficients.
                This image cannot be the same as the input image srcImage.
                Any mask ROI specified will be ignored.
                This image should be a multi-channel image containing
                as many channels as srcImage. The data type for the image
                should be 8, 16 or 32 bits.
}

{      Morphological Operations   }
{---------------------------------}

procedure iplErode(srcImage, dstImage: pIplImage; nIterations: integer);stdcall;

{
srcImage        The source image.
dstImage        The resultant image.
nIterations     The number of times to erode the image.
}

procedure iplDilate(srcImage, dstImage: pIplImage; nIterations: integer);stdcall;
{
srcImage        The source image.
dstImage        The resultant image.
nIterations     The number of times to erode the image.
}

procedure iplOpen(srcImage, dstImage: pIplImage; nIterations:integer);stdcall;
{
srcImage        The source image.
dstImage        The resultant image.
nIterations     The number of times to erode and dilate the image.
}

procedure iplClose(srcImage, dstImage: pIplImage; nIterations: integer);stdcall;
{
srcImage        The source image.
dstImage        The resultant image.
nIterations     The number of times to dilate and erode the image.
}


{        Color Space Conversion       }
{-------------------------------------}

const
 IPL_JITTER_NONE    =  0;
 IPL_DITHER_NONE    =  1;
 IPL_DITHER_FS      =  2;
 IPL_DITHER_JJH     =  4;
 IPL_DITHER_STUCKEY =  8;
 IPL_DITHER_BAYER   = 16;

procedure iplReduceBits(srcImage, dstImage: pIplImage;
 jitterType: integer; ditherType: integer;
 levels: integer);stdcall;
{
srcImage        The source image of a higher bit resolution.
dstImage        The resultant image of a lower bit resolution.
jitterType      Number from 0 to 8 of noise added.
ditherType      The type of dithering to be used. The following types are allowed:
    IPL_DITHER_NONE    No dithering is done
    IPL_DITHER_FS      The Floid-Steinberg     dithering algorithm is used.
    IPL_DITHER_JJH     The Jarvice-Judice-Hinke    dithering algorithm is used.
    IPL_DITHER_STUCKEY The Stuckey dithering        algorithm is used
    IPL_DITHER_BAYER   The Bayer dithering  algorithm is used.

levels          Number of levels for dithering. Power of 2 from 1 to N.

}

procedure iplColorToGray(srcImage, dstImage: pIplImage);stdcall;
{
srcImage        The source image.
dstImage        The resultant image.
}

procedure iplGrayToColor(srcImage, dstImage: pIplImage;
 FractR, FractG, FractB: float);stdcall;
{
srcImage        The source image.
dstImage        The resultant image.
}

procedure iplRGB2HSV(rgbImage, hsvImage: pIplImage);stdcall;
{
rgbImage        The source RGB image.
hsvImage        The resultant HSV.
}

procedure iplHSV2RGB(hsvImage, rgbImage: pIplImage);stdcall;
{
hsvImage        The source HSV image.
rgbImage        The resultant RGB.
}

procedure iplRGB2HLS(rgbImage, hlsImage: pIplImage );stdcall;
{
rgbImage        The source RGB image.
hlsImage        The resultant HLS.
}

procedure iplHLS2RGB(hlsImage, rgbImage: pIplImage);stdcall;
{
hlsImage        The source HLS image.
rgbImage        The resultant RGB.
}
procedure iplRGB2XYZ(srcImage, dstImage: pIplImage);stdcall;
{
  Purpose:    Transforms Image from RGB to XYZ color coordinate system
  Arguments:  rgbImage - the source RGB Image consists of integer
              triples which are:
                R - red color
                G - green color
                B - blue  color
              XYZImage - the resultant XYZ imageData consists of integer
              triples which are:
                H - color tone in range 0-360 grades, 0 - red tone.
                S - colour density in range 0-1
                V - whiteness in range 0-1
  32 bit per channel  -> 32  bit per channel
  16 bit per channel  -> 16  bit per channel
   8 bit per channel  ->  8  bit per channel
}

procedure iplXYZ2RGB(srcImage, dstImage:pIplImage);stdcall;
{
  Purpose:    Transforms Image from XYZ  to RGB color coordinate system
  Arguments:  XYZImage - the source XYZ imageData consists of integer
              triples which are:
                 H - color tone in range 0-360 grades, 0 - red tone.
                 S - colour density in range 0-1
                 V - whiteness in range 0-1
              rgbImage - the resultant RGB Image consists of integer
              triples which are:
                R - red color
                G - green color
                B - blue  color
}

procedure iplRGB2LUV(rgbImage, LUVImage: pIplImage);stdcall;
{
rgbImage        The source RGB image.
LUVImage        The resultant LUV.
}

procedure iplLUV2RGB(LUVImage, rgbImage: pIplImage);stdcall;
{
LUVImage        The source LUV image.
rgbImage        The resultant RGB.
}

procedure iplRGB2YUV(rgbImage, YUVImage: pIplImage);stdcall;
{
rgbImage        The source RGB image.
YUVImage        The resultant YUV.
}

procedure iplYUV2RGB(YUVImage, rgbImage: pIplImage);stdcall;
{
YUVImage        The source YUV image.
rgbImage        The resultant RGB.
}

procedure iplRGB2YCrCb(rgbImage, YCrCbImage: pIplImage);stdcall;
{
rgbImage        The source RGB image.
YCrCbImage        The resultant YCrCb.
}

procedure iplYCrCb2RGB(YCrCbImage, rgbImage: pIplImage);stdcall;
{
YCrCbImage        The source YCrCb image.
rgbImage        The resultant RGB.
}

procedure iplYCC2RGB(YCCImage, rgbImage: pIplImage);stdcall;
{
YCCImage        The source YCC image.
rgbImage        The resultant RGB.
}



{         Point Operations     }
{------------------------------}

procedure iplThreshold(srcImage, dstImage: pIplImage;
          threshold: integer);stdcall;
{
srcImage        The source image.
dstImage        The resultant image.
threshold       The threshold value to use for each pixel.
                The pixel value in the output is set to the maximum
                representable value if it is greater than or equal to the
                threshold value (for each channel). Otherwise the pixel value
                in the output is set to minimum representable value.
}


procedure iplContrastStretch(srcImage, dstImage: pIplImage;
          lut: Pointer);stdcall;
{
srcImage        The source image.
dstImage        The resultant image.
lut             An array of pointers to LUTs - one for each channel.
                Each lookup table should have the key and value arrays
                fully initialized. One or more channels may share the same LUT.
                Specifies an intensity transformation.
}

procedure iplComputeHisto(srcImage: pIplImage; lut: Pointer);stdcall;
{
srcImage        The source image for which the Histogram will be computed.
lut             An array of pointers to LUTs - one for each channel.
                Each lookup table should have  the key array fully initialized.
                The value array will be filled by this function.
                The same LUT can be shared by one or more channels.
}

procedure iplHistoEqualize(srcImage, dstImage: pIplImage; lut: Pointer);stdcall;
{
srcImage        The source image for which the Histogram will be computed.
dstImage        The resultant image after equalization.
lut             The Histogram of the image represented as an array of pointers
                to LUTs - one for each channel. Each  lookup table should have
                the key and value arrays fully initialized.
                This LUTs will contain flattened
                Histograms after this function
                is executed.
}


{      Color Twist Matrices    }
{------------------------------}


function iplCreateColorTwist(data: array of integer; scalingValue: integer):pIplColorTwist;stdcall;
{
data            An array containing the sixteen values that constitute
                the color twist matrix. The values are in row wise order.
                Color twist values that lie in the range -1 to 1 should be
                scaled up to lie in the range -231 to 231
                (Simply multiply the floating point number in the -1 to 1
                range by 231)
scalingValue    The scaling value - the exponent of a power of 2 -
                that was used (e.g. 31 if 231  was used to multiply the values)
                to convert to value: Integers. Will be used for normalization.
Return Value    A pointer to the iplColorTwist data structure containing
                the color twist matrix in form suitable for efficient computation.
                This structure can then be used in the function iplApplyColorTwist().
}

procedure iplSetColorTwist(cTwist: pIplColorTwist; data: array of integer;  scalingValue:integer);stdcall;
{
data            An array containing the sixteen values that constitute
                the color twist matrix. The values are in row wise order.
                Color twist values that lie in the range -1 to 1 should be
                scaled up to lie in the range -231 to 231
                (Simply multiply the floating point number in the -1 to 1
                range by 231)
scalingValue    The scaling value - the exponent of a power of 2 -
                that was used (e.g. 31 if 231  was used to multiply the values)
                to convert to value: Integers. Will be used for normalization.
cTwist          The color  twist matrix data structure that was created
                by a call to the function iplCreateColorTwist().
}

procedure iplDeleteColorTwist(cTwist:pIplColorTwist);stdcall;
{
cTwist          The color  twist matrix data structure that was created
                by a call to the function iplCreateColorTwist().
}


procedure iplApplyColorTwist(srcImage, dstImage: pIplImage;
          cTwist:pIplColorTwist; Offset: Integer);stdcall;
{
srcImage        The source image.
dstImage        The resultant image.
cTwist          The color  twist matrix data structure that was prepared
                by a call to the function iplSetColorTwist().
offset          An offset value that will be added to each pixel channel
                after multiplication with the color twist matrix.
}


{         Image Geometric Transforms
 ----------------------------------------}
const
 IPL_INTER_NN        =  0;
 IPL_INTER_LINEAR    =  1;
 IPL_INTER_CUBIC     =  2;

type  //my own types!

 TAffineCoeffs = array [0..1,0..2] of Double;
 TBilinearCoeffs = array [0..1,0..3] of Double;
 TPerspectiveCoeffs = array [0..2,0..2] of Double;
 TAffineRect = array [0..1,0..1] of Double;
 TAffineQuad = array [0..3,0..1] of Double;

//  Name:           iplWarpAffine
//  Purpose:        makes Affine transform of image.
//                  |X'|   |a11 a12| |X| |a13|
//                  |  | = |       |*| |+|   |
//                  |Y'|   |a21 a22| |Y| |a23|
//  Context:
//  Returns:        None.
//  Parameters:
//      srcImage         The source image.
//      dstImage         The resultant image.
//      coeffs           The transform matrix
//      interpolate      The type of interpolation to perform for resampling
//                       the input image. The following are currently supported.
//                       IPL_INTER_NN       Nearest neighbour interpolation.
//                       IPL_INTER_LINEAR   Linear interpolation.
//                       IPL_INTER_CUBIC    Cubic convolution interpolation.
//  Notes:

procedure  iplWarpAffine(srcImage: pIplImage; dstImage: pIplImage; const coeffs: TAffineCoeffs;
                             interpolate: integer);stdcall;



//  Name:           iplShear
//  Purpose:        Makes shear transform of image.
//                  |X'|   |1    xShear| |X|
//                  |  | = |           |*| |
//                  |Y'|   |yShear   1 | |Y|
//  Context:
//  Returns:        None.
//  Parameters:
//      srcImage         The source image.
//      dstImage         The resultant image.
//      interpolate      The type of interpolation to perform for resampling
//                       the input image. The following are currently supported.
//                       IPL_INTER_NN       Nearest neighbour interpolation.
//                       IPL_INTER_LINEAR   Linear interpolation.
//                       IPL_INTER_CUBIC    Cubic convolution interpolation.
//  Notes:



procedure  iplShear(srcImage: pIplImage; dstImage: pIplImage;
                         xShear, yShear:Double; xShift, yShift:Double;
                         interpolate: Integer);stdcall;



//  Name:           iplRotate
//  Purpose:        rotates image about (0, 0) on angle.
//  Context:
//  Returns:        None.
//  Parameters:
//      srcImage         The source image.
//      dstImage         The resultant image.
//      interpolate      The type of interpolation to perform for resampling
//                       the input image. The following are currently supported.
//                       IPL_INTER_NN       Nearest neighbour interpolation.
//                       IPL_INTER_LINEAR   Linear interpolation.
//                       IPL_INTER_CUBIC    Cubic convolution interpolation.
//  Notes:



procedure  iplRotate(srcImage: pIplImage; dstImage: pIplImage; angle:Double;
            xShift, yShift:Double; interpolate: Integer);stdcall;



//  Name:           iplGetAffineQuad
//  Purpose:        calculates coordinates of quadrangle from transformed image ROI.
//  Context:
//  Returns:        None.
//  Parameters:
//      image    The source image.
//      coeffs      The transform matrix
//                  |X'|   |a11 a12| |X| |a13|
//                  |  | = |       |*| |+|   |
//                  |Y'|   |a21 a22| |Y| |a23|
//      quadr       resultant quadrangle
//  Notes:


procedure  iplGetAffineQuad(image: pIplImage; const coeffs: TAffineCoeffs;
                                     var quad: TAffineQuad);stdcall;

//  Name:           iplGetAffineBound
//  Purpose:        calculates bounding rectangle of the transformed image ROI.
//  Context:
//  Returns:        None.
//  Parameters:
//      image    The source image.
//      coeffs      The transform matrix
//                  |X'|   |a11 a12| |X| |a13|
//                  |  | = |       |*| |+|   |
//                  |Y'|   |a21 a22| |Y| |a23|
//      rect        resultant bounding rectangle
//  Notes:


procedure  iplGetAffineBound(image: pIplImage; const coeffs: TAffineCoeffs;
                                 var rect: TAffineRect);stdcall;



//  Name:           iplGetAffineTransform
//  Purpose:        calculates transform matrix from vertexes of quadrangle.
//  Context:
//  Returns:        None.
//  Parameters:
//      image    The source image.
//      coeffs      The resultant transform matrix
//                  |X'|   |a11 a12| |X| |a13|
//                  |  | = |       |*| |+|   |
//                  |Y'|   |a21 a22| |Y| |a23|
//      quadr       quadrangle
//  Notes:


procedure  iplGetAffineTransform(image: pIplImage; var coeffs: TAffineCoeffs;
                         const quad: TAffineQuad);stdcall;



//  Name:           iplGetRotateShift
//  Purpose:        recalculates shifts for rotation around point (x, y).
//  Context:
//  Returns:        None.
//  Parameters:
//                  xCenter, yCenter    new center of rotation
//                  angle               the angle of rotation
//  Notes:


procedure  iplGetRotateShift(xCenter, yCenter:Double; angle: Double;
                              var xShift, yShift: Double);stdcall;

const
IPL_WARP_R_TO_Q = 0;
IPL_WARP_Q_TO_R = 1;


//  Name:           iplWarpBilinear, iplWarpBilinearQ
//  Purpose:        makes bilinear transform of image.
//                  from rectangle to quadrangle use transform
//                  |X|   |a11|      |a12 a13| |J|   |a14|
//                  | | = |   |*JI + |       |*| | + |   |
//                  |Y|   |a21|      |a22 a23| |I|   |a24|
//                  from quadrangle to rectungle use inverse transform
//                  0 = J*J * (-(x1-x0)*(y3-y2)+(x3-x2)*(y1-y0)) +
//                      J * (X*(y3-y2+y1-y0) - Y*(x3-x2+x1-x0) +
//                      ((x0-x3)*(y1-y0)-(x1-x0)*(y0-y3))) +
//                      (X*(y0-y3) - Y*(x0-x3))
//                  0 = I*I * (-(x0-x3)*(y2-y1)+(x2-x1)*(y0-y3)) +
//                      I * (X*(y3-y2+y1-y0) - Y*(x3-x2+x1-x0) +
//                      (-(x0-x3)*(y1-y0)+(y0-y3)*(x1-x0))) +
//                      (-X*(y1-y0) + Y*(x1-x0))
//                 In addition, I & J are bounded by the relation:
//                 0 = I * (-(x0-x3)*(y2-y1)+(x2-x1)*(y0-y3)) +
//                     J * (-(x1-x0)*(y3-y2)+(x3-x2)*(y1-y0)) +
//                    (X * (y3-y2+y1-y0) - Y * (x3-x2+x1-x0))
//                 J=0..1, I=0..1;
//                 J=j/jmax, I=i/imax;
//                 (j,i) - coordinates of a pixel in the square rectangle
//                 X=x-x0, Y=y-y0;
//                 (x,y) - coordinates of the pixel in the qudrangle
//
//  Context:
//  Returns:        None.
//  Parameters:
//      srcImage         The source image.
//      dstImage         The resultant image.
//      quad             The vertexes of quadrangle.
//      warpFlag         If warpFlag is IPL_WARP_R_TO_Q, the transform is from rectangle
//                       srcImage->roi to quadrangle in dstImage.
//                       If warpFlag is IPL_WARP_Q_TO_R, the transform is from quadrangle
//                       in srcImage to rectangle dstImage->roi.
//      interpolate      The type of interpolation to perform for resampling
//                       the input image. The following are currently supported.
//                       IPL_INTER_NN       Nearest neighbour interpolation.
//                       IPL_INTER_LINEAR   Linear interpolation.
//                       IPL_INTER_CUBIC    Cubic convolution interpolation.
//  Notes:


procedure  iplWarpBilinear(srcImage: pIplImage; dstImage: pIplImage;
                                const coeffs: TBilinearCoeffs;warpFlag: integer;
                                interpolate: Integer);stdcall;


procedure  iplWarpBilinearQ(srcImage: pIplImage; dstImage: pIplImage;
                                const quad: TAffineQuad; warpFlag: integer;
                                interpolate: Integer);stdcall;



//  Name:           iplWarpPerspective, iplWarpPerspectiveQ
//  Purpose:        makes perspective transform of image.
//                  from rectangle to quadrangle use transform
//                      a11*j + a12*i + a13
//                  x = -------------------
//                      a31*j + a32*i + a33
//
//                      a21*j + a22*i + a23
//                  y = -------------------
//                      a31*j + a32*i + a33
//
//                        |x0-x1+x2-x3  x3-x2|   | x1-x2  x3-x2|
//                  a31 = |                  | / |             |
//                        |y0-y1+y2-y3  y3-y2|   | y1-y2  y3-y2|
//
//                        |x1-x2  x0-x1+x2-x3|   | x1-x2  x3-x2|
//                  a21 = |                  | / |             |
//                        |y1-y2  y0-y1+y2-y3|   | y1-y2  y3-y2|
//
//                  a11 = (x1-x0+a31*x1)/jmax
//                  a12 = (x3-x0+a32*x3)/imax
//                  a13 = x0
//                  a21 = (y1-y0+a31*y1)/jmax
//                  a22 = (y3-y0+a32*y3)/imax
//                  a23 = y0
//                  from quadrangle to rectangle use inverse transform
//                  |a22*a33-a32*a23  a32*a13-a12*a33  a12*a23-a22*a13|
//                  |a31*a23-a21*a33  a11*a33-a31*a13  a21*a13-a11*a23|
//                  |a21*a32-a31*a22  a31*a12-a11*a32  a11*a22-a21*a12|
//
//  Context:
//  Returns:        None.
//  Parameters:
//      srcImage         The source image.
//      dstImage         The resultant image.
//      quad             The vertexes of quadrangle.
//      warpFlag         If warpFlag is IPL_WARP_R_TO_Q, the transform is from rectangle
//                       srcImage->roi to quadrangle (x0,y0)(x1,y1)(x2,y2)(x3,y3) in dstImage.
//                       If warpFlag is IPL_WARP_Q_TO_R, the transform is from quadrangle
//                       (x0,y0)(x1,y1)(x2,y2)(x3,y3) in srcImage to rectangle dstImage->roi.
//      interpolate      The type of interpolation to perform for resampling
//                       the input image. The following are currently supported.
//                       IPL_INTER_NN       Nearest neighbour interpolation.
//                       IPL_INTER_LINEAR   Linear interpolation.
//                       IPL_INTER_CUBIC    Cubic convolution interpolation.
//  Notes:


procedure  iplWarpPerspective(srcImage: pIplImage; dstImage: pIplImage;
                                  const coeffs: TPerspectiveCoeffs; warpFlag: Integer;
                                  interpolate: Integer);stdcall;


procedure  iplWarpPerspectiveQ(srcImage: pIplImage; dstImage: pIplImage;
                                   const quad: TAffineQuad; warpFlag: Integer;
                                   interpolate: Integer);stdcall;



//  Name:           iplGetBilinearQuad
//  Purpose:        calculates coordinates of quadrangle from transformed image ROI.
//  Context:
//  Returns:        None.
//  Parameters:
//      image       The image.
//      coeffs      The transform matrix
//                  |X|   |a11|      |a12 a13| |J|   |a14|
//                  | | = |   |*JI + |       |*| | + |   |
//                  |Y|   |a21|      |a22 a23| |I|   |a24|
//      quadr       resultant quadrangle
//  Notes:


procedure  iplGetBilinearQuad(image: pIplImage; const coeffs: TPerspectiveCoeffs;
                                  var quad: TAffineQuad);stdcall;



//  Name:           iplGetBilinearBound
//  Purpose:        calculates bounding rectangle of the transformed image ROI.
//  Context:
//  Returns:        None.
//  Parameters:
//      image    The source image.
//      coeffs      The transform matrix
//                  |X|   |a11|      |a12 a13| |J|   |a14|
//                  | | = |   |*JI + |       |*| | + |   |
//                  |Y|   |a21|      |a22 a23| |I|   |a24|
//      rect        resultant bounding rectangle
//  Notes:


procedure  iplGetBilinearBound(image: pIplImage; const coeffs: TBilinearCoeffs;
                                    var rect:TAffineRect);stdcall;



//  Name:           iplGetBilinearTransform
//  Purpose:        calculates transform matrix from vertexes of quadrangle.
//  Context:
//  Returns:        None.
//  Parameters:
//      image       The image.
//      coeffs      The resultant transform matrix
//                  |X|   |a11|      |a12 a13| |J|   |a14|
//                  | | = |   |*JI + |       |*| | + |   |
//                  |Y|   |a21|      |a22 a23| |I|   |a24|
//      quadr       quadrangle
//  Notes:


procedure  iplGetBilinearTransform(image: pIplImage; var coeffs: TBilinearCoeffs;
                                       const quad: TAffineQuad);stdcall;


//  Name:           iplGetPerspectiveQuad
//  Purpose:        calculates coordinates of quadrangle from transformed image ROI.
//  Context:
//  Returns:        None.
//  Parameters:
//      image       The image.
//      coeffs      The transform matrix
//                      a11*j + a12*i + a13
//                  x = -------------------
//                      a31*j + a32*i + 1
//
//                      a21*j + a22*i + a23
//                  y = -------------------
//                      a31*j + a32*i + 1
//      quadr       resultant quadrangle
//  Notes:


procedure  iplGetPerspectiveQuad(image: pIplImage; const coeffs: TPerspectiveCoeffs;
                                     var quad: TAffineQuad);stdcall;

//  Name:           iplGetPerspectiveBound
//  Purpose:        calculates bounding rectangle of the transformed image ROI.
//  Context:
//  Returns:        None.
//  Parameters:
//      image    The source image.
//      coeffs      The transform matrix
//                      a11*j + a12*i + a13
//                  x = -------------------
//                      a31*j + a32*i + 1
//
//                      a21*j + a22*i + a23
//                  y = -------------------
//                      a31*j + a32*i + 1
//      rect        resultant bounding rectangle
//  Notes:


procedure  iplGetPerspectiveBound(image: pIplImage; const coeffs: TPerspectiveCoeffs;
                                      var rect: TAffineRect);stdcall;




//  Name:           iplGetPerspectiveTransform
//  Purpose:        calculates transform matrix from vertexes of quadrangle.
//  Context:
//  Returns:        None.
//  Parameters:
//      image       The image.
//      coeffs      The resultant transform matrix
//                      a11*j + a12*i + a13
//                  x = -------------------
//                      a31*j + a32*i + 1
//
//                      a21*j + a22*i + a23
//                  y = -------------------
//                      a31*j + a32*i + 1
//      quadr       quadrangle
//  Notes:


procedure  iplGetPerspectiveTransform(image: pIplImage;var coeffs: TPerspectiveCoeffs;
                                          const quad: TAffineQuad);stdcall;


procedure  iplZoom(srcImage: pIplImage; dstImage: pIplImage; xDst, xSrc: Integer;
                     yDst, ySrc: Integer; interpolate: Integer);stdcall;
{
srcImage        The source image.
dstImage        The resultant image.
xDst, xSrc,
yDst, ySrc      All these should be positive integers with xDst >= xSrc and
                yDst >= ySrc. They specify the fractions xDst/xSrc and yDst/ySrc
                to expand the image in the X and Y directions.
                For example xDst = 2, xSrc = 1, yDst = 2, ySrc = 1 Floats the
                image size in each dimension to give an image 4 times larger in
                area.
interpolate     The type of interpolation to perform for resampling.
                The following are currently supported.
                IPL_INTER_NN            Nearest neighbour interpolation.
                IPL_INTER_LINEAR        Linear interpolation.
                IPL_INTER_CUBIC         Cubic convolution interpolation.
}

procedure  iplDecimate(srcImage: pIplImage; dstImage: pIplImage; xDst, xSrc: Integer;
             yDst, ySrc: Integer; interpolate: Integer);stdcall;
{
srcImage        The source image.
dstImage        The resultant image.
xDst, xSrc,
yDst, ySrc      All these should be positive integers with xDst <= xSrc
                and yDst <= ySrc. They specify the fractions xDst/xSrc
                and yDst/ySrc to shrink the image in the X and Y directions.
                For example xDst = 1, xSrc = 2, yDst = 1, ySrc = 2 halves
                the image size in each dimension to give an image 1/4 times
                smaller in area.
interpolate     The type of interpolation to perform for resampling.
                The following are currently supported.
                IPL_INTER_NN            Nearest neighbour interpolation.
                IPL_INTER_LINEAR        Linear interpolation.
                IPL_INTER_CUBIC         Cubic convolution interpolation.
                IPL_INTER_SUPER         Supersampling interpolation.
}


{
   If you use macro definitions iplDecimateFit, iplZoomFit,
   Please make sure that SRC and DST parameters aren't NULL
}

{#define iplDecimateFit(SRC,DST,INTER)\
   iplDecimate( SRC, DST, \
   DST->roi ? DST->roi->width : DST->width, \
   SRC->roi ? SRC->roi->width : SRC->width, \
   DST->roi ? DST->roi->height : DST->height, \
   SRC->roi ? SRC->roi->height : SRC->height, INTER )

#define iplZoomFit(SRC,DST,INTER)\
   iplZoom( SRC, DST, \
   DST->roi ? DST->roi->width : DST->width, \
   SRC->roi ? SRC->roi->width : SRC->width, \
   DST->roi ? DST->roi->height : DST->height, \
   SRC->roi ? SRC->roi->height : SRC->height, INTER )
}


procedure  iplMirror(srcImage: pIplImage; dstImage: pIplImage; flipAxis: Integer);stdcall;
{
srcImage        The source image.
dstImage        The resultant image.
flipAxis        Specifies the axis to mirror the image.
                Use 0 for the horizontal axis,
                    1 for a vertical axis
                   -1 for both horizontal and vertical axis.
}

const
 IPL_C  = 1;
 IPL_L1 = 2;
 IPL_L2 = 4;
 IPL_RELATIVE = 8;
 IPL_RELATIVEC = IPL_RELATIVE or IPL_C;
 IPL_RELATIVEL1 = IPL_RELATIVE or IPL_L1;
 IPL_RELATIVEL2 = IPL_RELATIVE or IPL_L2;

function iplNorm(srcImageA, srcImageB: pIplImage; normType: Integer): Double;stdcall;
{
  Name:        iplNorm
  Purpose:     Calculates C,L1,L2-norms for one IPL image, absolute and relative errors between two IPL-images.
  Context:
  Returns:     None
  Parameters:
   srcImageA - The source image, tiled or not, with maskROI or not.
   srcImageB - The source image, tiled or not, with maskROI or not, may be NULL.
   normType  - Specifies the type of norms and errors.

   If pointer of srcImageB is NULL,the next type of norms for srcImageA defined:
     normType = IPL_C  for C -norm: ||A||= max ( abs(a[i,j]) );
     normType = IPL_L1 for L1-norm: ||A||= sum ( abs(a[i,j]) )
     normType = IPL_L2 for L2-norm: ||A||= sqrt( sum (a[i,j]**2) );

   If pointer of srcImage is not NULL, the next type of absolute errors defined:
     normType = IPL_C  for C -absolute error: ||A-B||=max ( abs(a[i,j]-b[i,j]) );
     normType = IPL_L1 for L1-absolute error: ||A-B||=sum ( abs(a[i,j]-b[i,j]) );
     normType = IPL_L2 for L2-absolute error: ||A-B||=sqrt( sum (a[i,j]-b[i,j])**2 );

   If pointer of srcImage is not NULL, the next type of relative errors defined:
     normType = IPL_RELATIVEC  for C -relative error:
                ||A-B||/||B||= max(abs (a[i,j]-b[i,j]))/max(abs(a[i,j]));
     normType = IPL_RELATIVEL1 for L1-relative error:
                ||A-B||/||B||= sum(abs (a[i,j]-b[i,j]))/sum(abs(a[i,j]));
     normType = IPL_RELATIVEL2 for L2-relative error:
                ||A-B||/||B||= sqrt(sum((a[i,j]-b[i,j])**2)/sum((a[i,j])**2)));
}

type TOwnMoment = record  // spatial moment structure:
   scale: Double ;  // value to scale (m,n)th moment
   value: Double;  // spatial (m,n)th moment
   end;

  TIplMomentState = array[0..3,0..3] of TOwnMoment;

procedure iplMoments(img : pIplImage; var stt: TIplMomentState);stdcall;

function iplGetSpatialMoment(stt: TIplMomentState; m: Integer; n: Integer):Double;stdcall;
function iplGetNormalizedSpatialMoment(stt: TIplMomentState; m: Integer; n: Integer):Double;stdcall;
function iplGetCentralMoment(stt: TIplMomentState; m: Integer; n: Integer):Double;stdcall;
function iplGetNormalizedCentralMoment(stt: TIplMomentState; m: Integer; n: Integer):Double;stdcall;

function iplSpatialMoment(img: pIplImage; m: Integer; n: Integer):Double;stdcall;
function iplNormalizedSpatialMoment(img: pIplImage; m: Integer; n: Integer):Double;stdcall;
function iplCentralMoment(img: pIplImage; m: Integer; n: Integer):Double;stdcall;
function iplNormalizedCentralMoment(img: pIplImage; m: Integer; n: Integer):Double;stdcall;

{ -------------- Generation of the test images -------------------------------}


{  Name:           iplCreateImageJaehne
  Purpose:        Creates a gray (one channel) test image
  Context:        Image Processing
  Returns:        IPL image or NULL
  Parameters:
      depth       depth of the image to be created.
                  All IPL depths are possible including 32f.
                  For the 32f depth a data range is [0..1]
      width       width of the image to be created
      height      height of the image to be created
  Notes:    This test image was seen in
            B.Jaehne, Digital Image Processing, third edition, 1995

}

function iplCreateImageJaehne( depth, width, height: Integer):pIplImage;stdcall;
{
  Name:           iplCloneImage
  Purpose:        Creates a clone of an image
  Context:        Image Processing
  Returns:        IPL image or NULL
  Parameters:
      img     image to be cloned.
      The function creates a copy of an image img icluding
      the field roi by copying. The following fields of the
      created image are set by function
              ID = 0, imageId = NULL
              maskROI = NULL, tileInfo = NULL
      All IPL depths are possible including 32f.
}

function iplCloneImage( const img: pIplImage):pIplImage;stdcall;



 CONST ipldll = 'ipl.dll';
 //\\//\\//\\//\\//\\//\\//\\//\\ IPLMISC //\\//\/
function iplMalloc(Length: Integer): Pointer;stdcall;
function iplwMalloc(Length: Integer): pWord;stdcall;
function ipliMalloc(Length: Integer): pInteger;stdcall;
function iplsMalloc(Length: Integer): pSingle;stdcall;
function ipldMalloc(Length: Integer): pDouble;stdcall;

procedure iplFree(ptr: Pointer);stdcall;

const
IMG_NO_ERROR             = $000000;
IMG_HEADER_IS_NULL       = $000001;
IMG_BAD_DATA_PTR         = $000002;
IMG_BAD_IMAGE_SIZE       = $000004;
IMG_BAD_OFFSET           = $000008;
IMG_BAD_STEP             = $000010;
IMG_BAD_NUM_CHAN         = $000020;
IMG_BAD_NUM_CHAN_1U      = $000040;
IMG_BAD_DEPTH            = $000080;
IMG_BAD_ALPHA_CHAN       = $000100;
IMG_BAD_ORDER            = $000200;
IMG_BAD_ORIGIN           = $000400;
IMG_BAD_ALIGN            = $000800;
IMG_BAD_CALLBACK         = $001000;
IMG_BAD_TILE_SIZE        = $002000;
IMG_BAD_COI              = $004000;
IMG_BAD_ROI_SIZE         = $008000;

function iplBadImageHeader(hdr: pIplImage): Integer;stdcall;
{
  return 0 if valid header, else set appropriate bit if some field is
  inconsistent
}

procedure iplDeallocateHeader(image: pIplImage);stdcall;
{
  deallocate header, if image->imageData!=NULL, then first frees imageData
}

//\\//\\//\\//\\//\\//\\//\\//iplerror
type
pIPLLibVersion = ^TIPLLibVersion;
TIPLLibVersion = record
  major: Integer;                   { e.g. 1                          }
  minor: Integer;                   { e.g. 00                         }
  build: Integer;                   { e.g. 01                         }
  Name: pChar;                    { e.g. "ipl6l.lib","iplm5.dll"    }
  Version: pChar;                 { e.g. "v1.00"                    }
  InternalVersion: pChar;         { e.g. "[1.00.01, 07/25/96]"      }
  BuildDate: pChar;               { e.g. "Jun 1 96"                 }
  CallConv: pChar;                { e.g. "DLL","Microsoft","Borland"}
 end;

type IPLStatus = Integer;

 TIPLErrorCallBack = function
 (Status: IPLstatus;funcname: pChar;  const context: pChar; const filename: pChar;
  line: Integer):Integer; stdcall;

function iplGetLibVersion: pIPLLibVersion;stdcall;

const
_StsMaskImg_  = -100;

IPL_StsOk                =    0; { everithing is ok                  }
IPL_StsBackTrace         =   -1; { pseudo error for back trace       }
IPL_StsError             =   -2; { unknown /uiplecified error        }
IPL_StsInternal          =   -3; { internal error (bad state)        }
IPL_StsNoMem             =   -4; { out of memory                     }
IPL_StsBadArg            =   -5; { function arg/param is bad         }
IPL_StsBadFunc           =   -6; { unsupported function              }
IPL_StsNoConv            =   -7; { iter. didn't converge             }
IPL_StsAutoTrace         =   -8; { Tracing through ipltrace.h        }
IPL_HeaderIsNull         =   -9; { image header is NULL  }
IPL_BadImageSize         =   -10;{ Image size is invalid }
IPL_BadOffset            =   -11;{ Offset is invalid     }
IPL_BadDataPtr           =   -12;{}
IPL_BadStep              =   -13;{}
IPL_BadModelOrChSeq      =   -14;{}
IPL_BadNumChannels       =   -15;{}
IPL_BadNumChannel1U      =   -16;{}
IPL_BadDepth             =   -17;{}
IPL_BadAlphaChannel      =   -18;{}
IPL_BadOrder             =   -19;{}
IPL_BadOrigin            =   -20;{}
IPL_BadAlign             =   -21;{}
IPL_BadCallBack          =   -22;{}
IPL_BadTileSize          =   -23;{}
IPL_BadCOI               =   -24;{}
IPL_BadROISize           =   -25;{}
IPL_MaskIsTiled          =   -26;{}

IPL_MaskBadImageSize     =  IPL_BadImageSize   + _StsMaskImg_;
IPL_MaskBadOffset        =  IPL_BadOffset      + _StsMaskImg_;
IPL_MaskBadDataPtr       =  IPL_BadDataPtr     + _StsMaskImg_;
IPL_MaskBadStep          =  IPL_BadStep        + _StsMaskImg_;
IPL_MaskBadModelOrChSeq  =  IPL_BadModelOrChSeq+ _StsMaskImg_;
IPL_MaskBadNumChannels   =  IPL_BadNumChannels + _StsMaskImg_;
IPL_MaskBadDepth         =  IPL_BadDepth       + _StsMaskImg_;
IPL_MaskBadAlphaChannel  =  IPL_BadAlphaChannel+ _StsMaskImg_;
IPL_MaskBadOrder         =  IPL_BadOrder       + _StsMaskImg_;
IPL_MaskBadOrigin        =  IPL_BadOrigin      + _StsMaskImg_;
IPL_MaskBadAlign         =  IPL_BadAlign       + _StsMaskImg_;
IPL_MaskBadCOI           =  IPL_BadCOI         + _StsMaskImg_;
IPL_MaskBadROISize       =  IPL_BadROISize     + _StsMaskImg_;


function iplGetErrStatus: IPLStatus;stdcall;
procedure iplSetErrStatus(status: IPLStatus);stdcall;

const
IPL_ErrModeLeaf    = 0;           { Print error and exit program    }
IPL_ErrModeParent  = 1;           { Print error and continue        }
IPL_ErrModeSilent  = 2;           { Don't print and continue        }

function iplGetErrMode: Integer;stdcall;
procedure iplSetErrMode(mode: Integer);stdcall;

{--- iplError,iplErrorStr Declaration --------------------------------------------------}

function iplError(Status: IPLstatus; const func: pChar;
        const context: pChar; const filename:pChar; line:Integer):IPLStatus;stdcall;

function iplErrorStr(status: IPLStatus):pChar;stdcall;

{--- iplRedirectError Declaration ------------------------------------------}

function iplNulDevReport(Status: IPLstatus; const funcname: pChar;
         const context: pChar; const filename: pChar; line: Integer):IPLStatus;stdcall;

function iplStdErrReport(Status: IPLstatus; const funcname: pChar;
         const context: pChar; const filename: pChar; line: Integer):IPLStatus;stdcall;

function iplGuiBoxReport(Status: IPLstatus; const funcname: pChar;
         const context: pChar; const filename: pChar; line: Integer):IPLStatus;stdcall;

function iplRedirectError(iplErrorFunc: TIPLErrorCallBack): TIPLErrorCallBack;stdcall;

implementation


function IPL_DegToRad(deg: double):double;
begin
  Result := deg/180.0 * PI;
end;

procedure iplSetBorderMode;    external ipldll;
function iplCreateImageHeader; external ipldll;
procedure iplAllocateImage;     external ipldll;
procedure iplAllocateImageFP;   external ipldll;
procedure iplDeallocateImage;   external ipldll;
procedure iplDeallocate;        external ipldll;
function iplCreateTileInfo;     external ipldll;
procedure iplDeleteTileInfo;    external ipldll;
procedure iplSetTileInfo;       external ipldll;
function iplCreateROI;          external ipldll;
procedure iplDeleteROI;         external ipldll;
procedure iplSetROI;            external ipldll;
function iplTranslateDIB;       external ipldll;
procedure iplConvertToDIB;      external ipldll;
procedure iplConvertFromDIB;    external ipldll;
procedure iplCopy;              external ipldll;
procedure iplSet;               external ipldll;
procedure iplSetFP;             external ipldll;
procedure iplExchange;          external ipldll;
procedure iplConvert;           external ipldll;
procedure iplBitonalToGray;     external ipldll;

procedure iplAddS;              external ipldll;
procedure iplAddSFP;            external ipldll;
procedure iplSubtractS;         external ipldll;
procedure iplSubtractSFP;       external ipldll;
procedure iplMultiplyS;         external ipldll;
procedure iplMultiplySFP;       external ipldll;

procedure iplAbs;               external ipldll;
procedure iplNot;               external ipldll;
procedure iplLShiftS;           external ipldll;
procedure iplRShiftS;           external ipldll;
procedure iplSquare;            external ipldll;
procedure iplAndS;              external ipldll;
procedure iplOrS;               external ipldll;
procedure iplXorS;              external ipldll;
procedure iplAnd;               external ipldll;
procedure iplOr;                external ipldll;
procedure iplXor;               external ipldll;
procedure iplAdd;               external ipldll;
procedure iplSubtract;          external ipldll;
procedure iplMultiply;          external ipldll;
procedure iplAlphaComposite;    external ipldll;
procedure iplAlphaCompositeC;   external ipldll;
procedure iplMultiplySScale;    external ipldll;
procedure iplMultiplyScale;     external ipldll;
procedure iplPreMultiplyAlpha;  external ipldll;
procedure iplBlur;              external ipldll;

function iplCreateConvKernel;   external ipldll;
function iplCreateConvKernelFP; external ipldll;
function iplCreateConvKernelChar;        external ipldll;
procedure iplDeleteConvKernel;  external ipldll;
procedure iplDeleteConvKernelFP;external ipldll;
procedure iplGetConvKernel;     external ipldll;
procedure iplGetConvKernelFP;   external ipldll;
procedure iplGetConvKernelChar; external ipldll;
procedure iplConvolve2D;        external ipldll;
procedure iplConvolve2DFP;      external ipldll;
procedure iplConvolveSep2D;     external ipldll;
procedure iplFixedFilter;       external ipldll;
procedure iplMedianFilter;      external ipldll;
procedure iplMaxFilter;         external ipldll;
procedure iplMinFilter;         external ipldll;
procedure iplRealFft2D;         external ipldll;
procedure iplCcsFft2D;          external ipldll;
procedure iplDCT2D;             external ipldll;
procedure iplErode;             external ipldll;
procedure iplDilate;            external ipldll;
procedure iplOpen;              external ipldll;
procedure iplClose;             external ipldll;
procedure iplReduceBits;        external ipldll;
procedure iplColorToGray;       external ipldll;
procedure iplGrayToColor;       external ipldll;
procedure iplRGB2HSV;           external ipldll;
procedure iplHSV2RGB;           external ipldll;
procedure iplRGB2HLS;           external ipldll;
procedure iplHLS2RGB;           external ipldll;

procedure iplRGB2XYZ;           external ipldll;
procedure iplXYZ2RGB;           external ipldll;
procedure iplRGB2LUV;           external ipldll;
procedure iplLUV2RGB;           external ipldll;
procedure iplRGB2YUV;           external ipldll;
procedure iplYUV2RGB;           external ipldll;
procedure iplRGB2YCrCb;         external ipldll;
procedure iplYCrCb2RGB;         external ipldll;
procedure iplYCC2RGB;           external ipldll;

procedure iplThreshold;         external ipldll;
procedure iplContrastStretch;   external ipldll;
procedure iplComputeHisto;      external ipldll;
procedure iplHistoEqualize;     external ipldll;
function iplCreateColorTwist;   external ipldll;
procedure iplSetColorTwist;     external ipldll;
procedure iplDeleteColorTwist;  external ipldll;
procedure iplApplyColorTwist;   external ipldll;

procedure  iplWarpAffine;       external ipldll;
procedure  iplShear;            external ipldll;
procedure  iplRotate;           external ipldll;
procedure  iplGetAffineQuad;    external ipldll;
procedure  iplGetAffineBound;   external ipldll;
procedure  iplGetAffineTransform;external ipldll;
procedure  iplGetRotateShift;   external ipldll;
procedure  iplWarpBilinear;     external ipldll;
procedure  iplWarpBilinearQ;    external ipldll;
procedure  iplWarpPerspective;  external ipldll;
procedure  iplWarpPerspectiveQ; external ipldll;
procedure  iplGetBilinearQuad;  external ipldll;
procedure  iplGetBilinearBound; external ipldll;
procedure  iplGetBilinearTransform;      external ipldll;
procedure  iplGetPerspectiveQuad;        external ipldll;
procedure  iplGetPerspectiveBound;       external ipldll;
procedure  iplGetPerspectiveTransform;   external ipldll;

procedure iplZoom;              external ipldll;
procedure iplDecimate;          external ipldll;
procedure iplMirror;            external ipldll;

function iplNorm;               external ipldll;
procedure iplMoments;           external ipldll;

function iplGetSpatialMoment;   external ipldll;
function iplGetNormalizedSpatialMoment;  external ipldll;
function iplGetCentralMoment;   external ipldll;
function iplGetNormalizedCentralMoment;  external ipldll;

function iplSpatialMoment;      external ipldll;
function iplNormalizedSpatialMoment;     external ipldll;
function iplCentralMoment;      external ipldll;
function iplNormalizedCentralMoment;     external ipldll;

function iplCreateImageJaehne;  external ipldll;
function iplCloneImage;         external ipldll;
//||//
function iplMalloc;             external ipldll;
function iplwMalloc;            external ipldll;
function ipliMalloc;            external ipldll;
function iplsMalloc;            external ipldll;
function ipldMalloc;            external ipldll;

procedure iplFree;              external ipldll;
function iplBadImageHeader;     external ipldll;
procedure iplDeallocateHeader;  external ipldll;
//\\//\\//\\
function iplGetLibVersion;      external ipldll;
function iplGetErrStatus;       external ipldll;
procedure iplSetErrStatus;      external ipldll;
function iplGetErrMode;         external ipldll;
procedure iplSetErrMode;        external ipldll;
function iplError;              external ipldll;
function iplErrorStr;           external ipldll;
function iplNulDevReport;       external ipldll;
function iplStdErrReport;       external ipldll;
function iplGuiBoxReport;       external ipldll;
function iplRedirectError;      external ipldll;
end.
