(*
   IntegerNumber.pas
   TIntegerNumber Component
   by Glen Why
   Version 01.00.00
   No Rights Reserved


   Version History

   01.00.00 - first one
*)

unit IntegerNumber;

interface
uses
 Classes, SysUtils;

const

  IntNum_DefMin    = 0;
  IntNum_DefMax    = MaxInt;
  IntNum_DefValue  = IntNum_DefMin;

type

  TIntegerNumberOption = ( inSilentErrors, inAutoAdjust );
  TIntegerNumberOptions = set of TIntegerNumberOption;

  TIntChangeQueryEvent = procedure( Sender :TObject; NewValue :Integer;
                          Var CanChange :Boolean ) of object;

  TIntegerNumber = class( TComponent )
  private
    FValue :Integer;
    FMin, FMax :Integer;
    FOptions :TIntegerNumberOptions;
    FOnOverflow :TNotifyEvent;
    FOnUnderflow :TNotifyEvent;
    FOnChange :TNotifyEvent;
    FOnChangeQuery :TIntChangeQueryEvent;
    procedure SetMin( V :Integer );
    procedure SetMax( V :Integer );
    procedure SetValue( V :Integer );
  protected
    function AdjustValue( Value :integer ):Integer;
    procedure DoUnderflow;
    procedure DoOverflow;
    procedure Overflow; dynamic;
    procedure Underflow; dynamic;
    procedure Change; dynamic;
    function  ChangeQuery( NewValue :Integer ) :boolean; dynamic;
  public
    constructor Create( anOwner :TCOmponent ); override;
  published
    property Options :TIntegerNumberOptions
      read FOptions write FOptions;
    property Max :Integer
      read FMax write SetMax default IntNum_DefMax;
    property Min :Integer
      read FMin write SetMin default IntNum_DefMin;
    property Value :Integer
      read FValue write SetValue default IntNum_DefValue;
    property OnOverflow :TNotifyEvent
      read FOnOverflow write FOnOverflow;
    property OnUnderflow :TNotifyEvent
      read FOnUnderflow write FOnUnderflow;
    property OnChange :TNotifyEvent
      read FOnChange write FOnChange;
    property OnChangeQuery :TIntChangeQueryEvent
      read FOnChangeQuery write FOnChangeQuery;
  end;

implementation

{$I SYSUTILS.INC}

{ TIntegerNumber }


constructor TIntegerNumber.Create( anOwner :TCOmponent );
begin
 inherited Create( anOwner );
 FMax := IntNum_DefMax;
 FMin := IntNum_DefMin;
 FValue := IntNum_DefValue;
 FOptions := [];
 FOnOverflow := nil;
 FOnUnderflow := nil;
 FOnChange := nil;
 FOnChangeQuery := nil;
end;


procedure TIntegerNumber.SetMin( V :Integer );
begin
  if ( V <> FMin ) then
    begin
     if not ( csLoading in ComponentState )
      then
        if ( V > FMax ) or ( FValue < V )
         then raise ERangeError.CreateRes( SRangeError );
     FMin := V;
    end;
end;

procedure TIntegerNumber.SetMax( V :Integer );
begin
  if ( V <> FMax ) then
    begin
     if not ( csLoading in ComponentState )
      then
        if ( V < FMin ) or ( FValue > V )
         then raise ERangeError.CreateRes( SRangeError );
     FMax := V;
    end;
end;

procedure TIntegerNumber.SetValue( V :Integer );
begin
  if ( V = FValue ) then exit;
  if ( csLoading in ComponentState ) then
    begin
     FValue := V;
     exit;
    end;
  if ChangeQuery( V )
   then begin
     if ( V < FMin ) or ( V > FMax )
      then
        if ( inAutoAdjust in Options )
         then FValue := AdjustValue( V )
         else begin
           if ( V < FMin )
            then DoUnderflow
            else DoOverflow;
           exit;
         end
      else FValue :=  V;
     Change;
   end;
end;

procedure TIntegerNumber.Overflow;
begin
 if assigned( FOnOverflow ) then FOnOverflow( Self );
end;

procedure TIntegerNumber.Underflow;
begin
 if assigned( FOnUnderflow ) then FOnUnderflow( Self );
end;

procedure TIntegerNumber.Change;
begin
 if assigned( FOnChange ) then FOnChange( Self );
end;

function  TIntegerNumber.ChangeQuery( NewValue :Integer ) :boolean;
begin
  result := true;
  if assigned( FOnChangeQuery ) then FOnChangeQuery( Self, NewValue, Result );
end;

function TIntegerNumber.AdjustValue( Value :integer ):Integer;
begin
 result := value;
 if ( result < fmin )
  then result := fmin
  else if result > fmax then result := fmax;
end;

procedure TIntegerNumber.DoUnderflow;
begin
 if ( not ( inSilentErrors in Options ) ) or
 ( csDesigning in ComponentState )
  then raise ERangeError.CreateRes( SRangeError );
 Underflow;
end;

procedure TIntegerNumber.DoOverflow;
begin
 if ( not ( inSilentErrors in Options ) ) or
 ( csDesigning in ComponentState )
  then raise ERangeError.CreateRes( SRangeError );
 Overflow;
end;

end.
