{*************************************************************}
{            TImagesScroll Components for Delphi32            }
{ Version:   1.00                                             }
{ Author:    Aleksey Kuznetsov, Kiev, Ukraine                 }
{              (Xacker), ,          }
{ E-Mail:    xacker@phreaker.net                              }
{ Homepage:  http://www.angen.net/~xacker/                    }
{ Created:   January, 13, 1999                                }
{ Modified:  February, 23, 1999                               }
{ Legal:     Copyright (c) 1999 by Aleksey Xacker             }
{*************************************************************}
{   TImagesScroll (English):                                  }
{ TImagesScroll component is intended for displaying and      }
{ selecting images from TImageList component. Appearance and  }
{ purpose components is similar feature of Windows arising if }
{ necessary of a choice of icons for a file.                  }
{*************************************************************}
{   TImagesScroll (Russian):                                  }
{           }
{    TImageList.    }
{    Windows        }
{     .                        }
{*************************************************************}
{  PROPERTIES:                                                }
{ AutoHeight: Boolean  - Auto calculating Height of control   }
{ ShowFrom: Integer    - Defines an initial shown image       }
{ Images: TImageList   - The list of pictures for display     }
{ ImageIndex: Integer  - Selected image                       }
{ Notify               - Determines whether an OnSelected     }
{                        event is generated.                  }
{  EVENTS:                                                    }
{ OnSelected           - Image was selected                   }
{*************************************************************}
{ Thanks for using TImagesScroll components.                  }
{ If at occurrence of any questions concerning these          }
{ components, mail me: xacker@phreaker.net.                   }
{ For updated versions visit my H-page: www.angen.net/~xacker }
{*************************************************************}

unit ImagesScroll;

interface

uses
  Windows, Messages, Classes, Graphics, Controls, Forms, StdCtrls;

type
  TOnSelectedEvent = procedure(Sender: TObject;
                               SelectedIndex: Integer) of object;

  TheListOfImages = class(TCustomControl)
  private
    IsMouseDown: Boolean;
    OldImageIndex: Integer;

    procedure SetImageIndex(Value: Integer);

    procedure Paint; override;

    procedure WMLButtonDown(var Msg: TMessage); message wm_LButtonDown;
    procedure WMLButtonUp(var Msg: TMessage); message wm_LButtonUp;
    procedure WMMouseMove(var Msg: TMessage); message wm_MouseMove;
  protected
  public
    FShowFrom: Integer;
    FImages: TImageList;
    FImageIndex: Integer;

    property ImageIndex: Integer read FImageIndex write SetImageIndex;
  published
  end;

  TImagesScroll = class(TScrollBox)
  private
    PaintControl: TheListOfImages;

    FAutoHeight: Boolean;
    FShowFrom: Integer;
    FImages: TImageList;
    FImageIndex: Integer;
    FNotify: Boolean;
    FOnSelected: TOnSelectedEvent;

    procedure SetAutoHeight(Value: Boolean);
    procedure SetShowFrom(Value: Integer);
    procedure SetImageList(Value: TImageList);
    procedure SetImageIndex(Value: Integer);

    procedure WMHScroll(var Msg: TWMHScroll); message WM_HSCROLL;
    procedure WMSize(var Msg: TMessage); message wm_Size;
  protected
    procedure Notification(AComponent: TComponent; Operation: TOperation); override;
  public
    constructor Create(aOwner: TComponent); override;
    destructor Destroy; override;
  published
    property AutoHeight: Boolean read FAutoHeight write SetAutoHeight;
    property ShowFrom: Integer read FShowFrom write SetShowFrom;
    property Images: TImageList read FImages write SetImageList;
    property ImageIndex: Integer read FImageIndex write SetImageIndex;
    property Notify: Boolean read FNotify write FNotify;

    property OnSelected: TOnSelectedEvent read FOnSelected write FOnSelected;
   end;

procedure Register;

implementation

{ TheListOfImages - mutilated TCustomControl. Body of component. }

procedure TheListOfImages.Paint;
var
  i, x, y: Integer;
begin
  if (FImages <> nil) and (FShowFrom < FImages.Count) then
   begin
    x := FImages.Width + 2;
    y := FImages.Height + 2;
    for i := 0 to FImages.Count - FShowFrom - 1 do
     begin
      if i = OldImageIndex then
       begin
        Canvas.Brush.Color := Color;
        Canvas.Pen.Color := Color;
        Canvas.Rectangle(i * x, 0, i * x + x, y);
       end;
      if i = FImageIndex then
       with Canvas do
        begin
         Brush.Color := clHighlight;
         Pen.Color := clHighlight;
         Rectangle(i * x, 0, i * x + x, y);
        end;
      FImages.Draw(Canvas, i * x + 1, 1, i + FShowFrom);
     end;
   end;
  OldImageIndex := FImageIndex; 
end;

procedure TheListOfImages.WMLButtonDown(var Msg: TMessage);
var
  X: Word;
begin
  inherited;
  if FImages <> nil then
   begin
    X := LoWord(Msg.lParam);
    ImageIndex := X div (FImages.Width + 2);
    IsMouseDown := True;
   end;
end;

procedure TheListOfImages.WMLButtonUp(var Msg: TMessage);
begin
  inherited;
  IsMouseDown := False;
end;

procedure TheListOfImages.WMMouseMove(var Msg: TMessage);
var
  X: Integer;
begin
  inherited;
  if (FImages <> nil) and IsMouseDown then
   begin
    X := LoWord(Msg.lParam);
    if (X > Left) and (X < Width) then
     ImageIndex := X div (FImages.Width + 2);
    IsMouseDown := True;
   end;
end;

procedure TheListOfImages.SetImageIndex(Value: Integer);
begin
  if FImages <> nil then
   begin
    if Value < 0 then Value := 0;
    if Value > FImages.Count - (1 + FShowFrom) then
     Value := FImages.Count - (1 + FShowFrom);

    if FImageIndex <> Value then
     begin
      FImageIndex := Value;
      TImagesScroll(Parent).ImageIndex := Value + TImagesScroll(Parent).ShowFrom;
      Paint;
     end;
   end;  
end;

{ TImagesScroll - Main }

constructor TImagesScroll.Create(aOwner: TComponent);
begin
  inherited Create(aOwner);
  FNotify := True;
  FAutoHeight := True;
  VertScrollBar.Visible := False;
  Color := clNone;

  PaintControl := TheListOfImages.Create(Self);
  PaintControl.Parent := Self;
end;

destructor TImagesScroll.Destroy;
begin
  PaintControl.Free;
  inherited Destroy;
end;

procedure TImagesScroll.Notification(AComponent: TComponent;
  Operation: TOperation);
begin
  inherited Notification(AComponent, Operation);
  if (AComponent = FImages) and
     (Operation = opRemove) then
   FImages := nil;
end;

procedure TImagesScroll.WMHScroll(var Msg: TWMHScroll);
var
  i, x: Integer;
begin
  if (Msg.ScrollCode = sb_ThumbTrack) or (Msg.ScrollCode = sb_ThumbPosition) then
   begin
    x := FImages.Width + 2;
    i := Msg.Pos div x;
    if Msg.Pos mod x >= x div 2 then inc(i);
    HorzScrollBar.Position := i * x
   end
  else inherited
end;

procedure TImagesScroll.WMSize(var Msg: TMessage);
var
  Additional: Integer;
begin
  if (FImages <> nil) and FAutoHeight then
   begin
    if BorderStyle = bsSingle then Additional := 6
    else Additional := 2;
    Height := FImages.Height + GetSystemMetrics(sm_CYHScroll) + Additional;
   end;
  inherited
end;

procedure TImagesScroll.SetShowFrom(Value: Integer);
begin
  if (Value >= 0) and (Value <> FShowFrom) then
   begin
    FShowFrom := Value;
    PaintControl.FShowFrom := Value;

    if FImages <> nil then
     PaintControl.Width := (FImages.Count - FShowFrom) * (FImages.Width + 2);
    
    ImageIndex := 0;
    PaintControl.Refresh;
   end;
end;

procedure TImagesScroll.SetAutoHeight(Value: Boolean);
var
  Msg: TMessage;
begin
  if FAutoHeight <> Value then
   begin
    FAutoHeight := Value;
    WMSize(Msg);
   end;
end;

procedure TImagesScroll.SetImageList(Value: TImageList);
begin
  if FImages <> Value then
   begin                                                          
    FImages := Value;
    if Value <> nil then
     begin   
      PaintControl.Visible := True; 
      Value.FreeNotification(Self);
      PaintControl.Width := (FImages.Count - FShowFrom) * (FImages.Width + 2);
      PaintControl.Height := FImages.Height + 2;
      HorzScrollBar.Increment := FImages.Width + 2;
     end
    else PaintControl.Visible := False;
    PaintControl.FImages := Value;
    PaintControl.Paint;
   end;
end;

procedure TImagesScroll.SetImageIndex(Value: Integer);
var
  x: Integer;
  VisibleWidth: Integer;
begin
  if FImages <> nil then
   begin
    Value := Value - FShowFrom;
    if Value < 0 then Value := 0;
    if Value > FImages.Count - (1 + FShowFrom) then
     Value := FImages.Count - (1 + FShowFrom);
    if FImageIndex <> Value then
     begin
      FImageIndex := Value;

      PaintControl.FImageIndex := Value;
      PaintControl.Paint;

{ If selection is hiden then we must move scroll position automaticly }
      x := FImages.Width + 2;
      VisibleWidth := Width div x - 1;

      if (HorzScrollBar.Position div x > Value) or
         ((Value - VisibleWidth) * x > VisibleWidth + HorzScrollBar.Position) then
       HorzScrollBar.Position := (Value - VisibleWidth div 2) * x;

      if Assigned(FOnSelected) and Notify then FOnSelected(Self, Value + FShowFrom);
     end;
   end;
end;

procedure Register;
begin
  RegisterComponents('Xacker', [TImagesScroll]);
end;

end.
