// INT_IBC_RichEdit

{ TIB_RichEdit }

constructor TIB_RichEdit.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  inherited ReadOnly := True;
  FAutoDisplay := True;
  FIB_ColumnLink := TIB_ColumnLink.Create( Self );
  with FIB_ColumnLink do begin
    Control := Self;
    OnStateChanged := StateChanged;
    OnDataChanged := DataChanged;
    OnUpdateData := UpdateData;
  end;
  FPaintControl := TPaintControl.Create(Self, 'EDIT');
end;

destructor TIB_RichEdit.Destroy;
begin
  DataSetLink := nil;
  with FIB_ColumnLink do begin
    Control := nil;
    OnStateChanged := nil;
    OnDataChanged  := nil;
    OnUpdateData   := nil;
  end;
  FPaintControl.Free;
  inherited Destroy;
end;

procedure TIB_RichEdit.Loaded;
begin
  inherited Loaded;
  SysStateChanged;
  SysDataChanged;
end;

{------------------------------------------------------------------------------}
{  DataLink event handlers.                                                    } 
{------------------------------------------------------------------------------}

procedure TIB_RichEdit.StateChanged( Sender: TIB_DataLink;
                                     IB_DataSetLink: TIB_DataSetLink );
begin
  SysStateChanged;
end;

procedure TIB_RichEdit.DataChanged( Sender: TIB_DataLink;
                                    IB_DataSetLink: TIB_DataSetLink;
                                    IB_Column: TIB_Column );
begin
  SysDataChanged;
end;

procedure TIB_RichEdit.UpdateData( Sender: TIB_DataLink;
                                   IB_DataSetLink: TIB_DataSetLink;
                                   IB_Column: TIB_Column );
begin
  SysUpdateData;
end;

{------------------------------------------------------------------------------}
{ DataLink property coordination methods.                                      } 
{------------------------------------------------------------------------------}

function TIB_RichEdit.GetOnPrepareSQL: TIB_DataLinkEvent;
begin
  Result := FIB_ColumnLink.OnPrepareSQL;
end;

procedure TIB_RichEdit.SetOnPrepareSQL( AValue: TIB_DataLinkEvent);
begin
  FIB_ColumnLink.OnPrepareSQL := AValue;
end;

{------------------------------------------------------------------------------}

function TIB_RichEdit.GetDataSetLink: TIB_DataSetLink;
begin
  Result := FIB_ColumnLink.IB_DataSetLink;
end;

procedure TIB_RichEdit.SetDataSetLink(AValue: TIB_DataSetLink);
begin
  FIB_ColumnLink.IB_DataSetLink := AValue;
end;

{------------------------------------------------------------------------------}

function TIB_RichEdit.GetDataColumnName: string;
begin
  Result := FIB_ColumnLink.ColumnName;
end;

procedure TIB_RichEdit.SetDataColumnName(const AValue: string);
begin
  FIB_ColumnLink.ColumnName := AValue;
end;

{------------------------------------------------------------------------------}

function TIB_RichEdit.GetColumn: TIB_Column;
begin
  Result := FIB_ColumnLink.Column;
end;

{------------------------------------------------------------------------------}

function TIB_RichEdit.GetReadOnly: Boolean;
begin
  Result := FIB_ColumnLink.ControlIsReadOnly;
end;

procedure TIB_RichEdit.SetReadOnly( Value: Boolean );
begin
  if ReadOnly <> Value then begin
    FIB_ColumnLink.ControlIsReadOnly := Value;
    SysStateChanged;
  end;
end;

{------------------------------------------------------------------------------}

function TIB_RichEdit.GetPreventUpdating: Boolean;
begin
  Result := FIB_ColumnLink.ControlPreventsUpdating;
end;

procedure TIB_RichEdit.SetPreventUpdating( Value: Boolean );
begin
  if PreventUpdating <> Value then begin
    FIB_ColumnLink.ControlPreventsUpdating := Value;
    SysStateChanged;
  end;
end;

{------------------------------------------------------------------------------}

function TIB_RichEdit.GetPreventInserting: Boolean;
begin
  Result := FIB_ColumnLink.ControlPreventsInserting;
end;

procedure TIB_RichEdit.SetPreventInserting( Value: Boolean );
begin
  if PreventInserting <> Value then begin
    FIB_ColumnLink.ControlPreventsInserting := Value;
    SysStateChanged;
  end;
end;

{------------------------------------------------------------------------------}

function TIB_RichEdit.GetPreventSearching: Boolean;
begin
  Result := FIB_ColumnLink.ControlPreventsSearching;
end;

procedure TIB_RichEdit.SetPreventSearching( Value: Boolean );
begin
  if PreventSearching <> Value then begin
    FIB_ColumnLink.ControlPreventsSearching := Value;
    SysStateChanged;
  end;
end;

{------------------------------------------------------------------------------}

function TIB_RichEdit.GetSearchBuffer: string;
begin
  if FIB_ColumnLink <> nil then begin
    Result := FIB_ColumnLink.SearchBuffer;
  end else begin
    Result := '';
  end;
end;

procedure TIB_RichEdit.SetSearchBuffer( AValue: string );
begin
  if FIB_ColumnLink <> nil then begin
    FIB_ColumnLink.SearchBuffer := AValue;
  end;
end;

{------------------------------------------------------------------------------}
{ Implementation of data-aware logic.                                          } 
{------------------------------------------------------------------------------}

procedure TIB_RichEdit.SysStateChanged;
begin
  if FIB_ColumnLink <> nil then begin
    with FIB_ColumnLink do begin
      inherited Color    := Color;
      inherited ReadOnly := ReadOnly or not ( DataSetState in [ dsInserting,
                                                                dsUpdating,
                                                                dsSearching ]);
    end;
  end;
end;

procedure TIB_RichEdit.SysDataChanged;
begin
  with FIB_ColumnLink do begin
    MaxLength := ColumnMaxLength;
    FMemoLoaded := false;
    if Column <> nil then begin
      case DataSetState of
        dsActive,
        dsUpdating,
        dsInserting,
        dsDeleting: begin
          if Column is TIB_Column_Blob then begin
            if AutoDisplay then begin
              LoadMemo;
            end else begin
              Text := ColumnText;
            end;
          end else begin
            Text := ColumnText;
            FMemoLoaded := True;
          end;
        end else begin
          Text := ColumnText;
          FMemoLoaded := true;
        end;
      end;
    end else begin
      Text := ColumnText;
    end;
  end;
end;

procedure TIB_RichEdit.SysUpdateData;
begin
  FIB_ColumnLink.ColumnText := Text;
end;

procedure TIB_RichEdit.LoadMemo;
begin
  if not FMemoLoaded then begin
    try
      with Column as TIB_Column_Blob do begin
        FIgnoreChange := true;
        try
          AsStream.Position := 0;
          Lines.LoadFromStream( AsStream );
        finally
          FIgnoreChange := false;
        end;
      end;
      FMemoLoaded := True;
    except
      Lines.Text := 'Could not load memo.';
    end;
    SysStateChanged;
  end;
end;

{------------------------------------------------------------------------------}

procedure TIB_RichEdit.SetFocused(Value: Boolean);
begin
  if FFocused <> Value then begin
    FFocused := Value;
    if not (FIB_ColumnLink.Column is TIB_Column_Blob) then begin
      SysDataChanged;
    end;
  end;
end;

procedure TIB_RichEdit.SetAutoDisplay(Value: Boolean);
begin
  if FAutoDisplay <> Value then
  begin
    FAutoDisplay := Value;
    if Value then begin
      LoadMemo;
    end;
  end;
end;

//------------------------------------------------------------------------------

procedure TIB_RichEdit.WndProc(var Message: TMessage);
begin
  with Message do
    if (Msg = WM_CREATE) or
       (Msg = WM_WINDOWPOSCHANGED) or
       (Msg = CM_FONTCHANGED) then begin
      FPaintControl.DestroyHandle;
    end;
  inherited;
end;

procedure TIB_RichEdit.CMEnter(var Message: TCMEnter);
begin
  SetFocused(True);
  FIB_ColumnLink.SetFocus;
  inherited;
end;

procedure TIB_RichEdit.CMExit(var Message: TCMExit);
begin
  try
    FIB_ColumnLink.UpdateValue;
  except
    SetFocus;
    raise;
  end;
  SetFocused(False);
  inherited;
end;

procedure TIB_RichEdit.KeyDown(var Key: Word; Shift: TShiftState);
begin
  inherited KeyDown(Key, Shift);
  if FMemoLoaded then begin
    if (Key = VK_DELETE) or
      ((Key = VK_INSERT) and (ssShift in Shift)) then begin
      FIB_ColumnLink.Modify;
    end;
  end else begin
    Key := 0;
  end;
end;

procedure TIB_RichEdit.KeyPress(var Key: Char);
begin
  inherited KeyPress(Key);
  if FMemoLoaded then begin
    if (Key in [#32..#255]) and (FIB_ColumnLink.Column <> nil) and
      not FIB_ColumnLink.Column.IsValidChar(Key) then
    begin
      MessageBeep(0);
      Key := #0;
    end;
    case Key of
      ^H, ^I, ^J, ^M, ^V, ^X, #32..#255:
        FIB_ColumnLink.Modify;
      #27:
        SysDataChanged;
    end;
  end else begin
    if Key = #13 then begin
      LoadMemo;
    end;
    Key := #0;
  end;
end;

procedure TIB_RichEdit.Change;
begin
  if not FIgnoreChange then begin
    with FIB_ColumnLink do begin
      if FMemoLoaded then begin
        Modify;
      end;
      FMemoLoaded := true;
      if DataSetState in [ dsInserting, dsUpdating, dsSearching ] then begin
        ControlIsModified := true;
      end;
    end;
    inherited Change;
  end;
end;

procedure TIB_RichEdit.WMLButtonDblClk(var Message: TWMLButtonDblClk);
begin
  if not FMemoLoaded then begin
    LoadMemo;
  end else begin
    inherited;
  end;
end;

procedure TIB_RichEdit.WMCut(var Message: TMessage);
begin
  if FIB_ColumnLink.Modify then begin
    inherited;
  end;
end;

procedure TIB_RichEdit.WMPaste(var Message: TMessage);
begin
  if FIB_ColumnLink.Modify then begin
    inherited;
  end;
end;

procedure TIB_RichEdit.CMGetDataLink(var Message: TMessage);
begin
  Message.Result := Integer(FIB_ColumnLink);
end;

procedure TIB_RichEdit.WMPaint(var Message: TWMPaint);
var
  S: string;
begin
  if not (csPaintCopy in ControlState) then begin
    inherited
  end else begin
    if FIB_ColumnLink.Column <> nil then begin
      if FIB_ColumnLink.Column is TIB_Column_Blob then begin
        S := AdjustLineBreaks(FIB_ColumnLink.Column.AsString)
      end else begin
        S := FIB_ColumnLink.Column.AsString;
      end;
    end;
    SendMessage(FPaintControl.Handle, WM_SETTEXT, 0, Integer(PChar(S)));
    SendMessage(FPaintControl.Handle, WM_PAINT, Message.DC, 0);
  end;
end;


