// INT_IBC_Image

constructor TIB_Image.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  ControlStyle := ControlStyle + [csFramed, csOpaque];
  Width := 105;
  Height := 105;
  TabStop := True;
  ParentColor := False;
  FPicture := TPicture.Create;
  FPicture.OnChange := PictureChanged;
  FBorderStyle := bsSingle;
  FAutoDisplay := True;
  FCenter := True;
  FIB_ColumnLink := TIB_ColumnLink.Create( Self );
  with FIB_ColumnLink do begin
    Control := Self;
    OnDataChanged := DataChanged;
    OnUpdateData := UpdateData;
  end;
  FQuickDraw := True;
end;

destructor TIB_Image.Destroy;
begin
  DataSetLink := nil;
  with FIB_ColumnLink do begin
    Control := nil;
    OnDataChanged  := nil;
    OnUpdateData   := nil;
  end;
  Destroying;
  FPicture.Free;
  FPicture := nil;
  inherited Destroy;
end;

procedure TIB_Image.Loaded;
begin
  inherited Loaded;
  SysDataChanged;
end;

{------------------------------------------------------------------------------}

function TIB_Image.GetDataSetLink: TIB_DataSetLink;
begin
  Result := FIB_ColumnLink.IB_DataSetLink;
end;

function TIB_Image.GetColumn: TIB_Column;
begin
  Result := FIB_ColumnLink.Column;
end;

function TIB_Image.GetDataColumnName: string;
begin
  Result := FIB_ColumnLink.ColumnName;
end;

function TIB_Image.GetReadOnly: Boolean;
begin
  Result := FIB_ColumnLink.ControlIsReadOnly;
end;

function TIB_Image.GetPreventUpdating: Boolean;
begin
  Result := FIB_ColumnLink.ControlPreventsUpdating;
end;

function TIB_Image.GetPreventInserting: Boolean;
begin
  Result := FIB_ColumnLink.ControlPreventsInserting;
end;

{------------------------------------------------------------------------------}

procedure TIB_Image.SetReadOnly( Value: Boolean );
begin
  if ReadOnly <> Value then begin
    FIB_ColumnLink.ControlIsReadOnly := Value;
  end;
end;

procedure TIB_Image.SetPreventUpdating( Value: Boolean );
begin
  if PreventUpdating <> Value then begin
    FIB_ColumnLink.ControlPreventsUpdating := Value;
  end;
end;

procedure TIB_Image.SetPreventInserting( Value: Boolean );
begin
  if PreventInserting <> Value then begin
    FIB_ColumnLink.ControlPreventsInserting := Value;
  end;
end;

function TIB_Image.GetSearchBuffer: string;
begin
  if FIB_ColumnLink <> nil then begin
    Result := FIB_ColumnLink.SearchBuffer;
  end else begin
    Result := '';
  end;
end;

procedure TIB_Image.SetSearchBuffer( AValue: string );
begin
  if FIB_ColumnLink <> nil then begin
    FIB_ColumnLink.SearchBuffer := AValue;
  end;
end;

procedure TIB_Image.SetDataSetLink( AValue: TIB_DataSetLink );
begin
  FIB_ColumnLink.IB_DataSetLink := AValue;
end;

procedure TIB_Image.SetDataColumnName( const AValue: string );
begin
  FIB_ColumnLink.ColumnName := AValue;
end;

{------------------------------------------------------------------------------}

function TIB_Image.GetPalette: HPALETTE;
begin
  Result := 0;
  if FPicture.Graphic is TBitmap then
    Result := TBitmap(FPicture.Graphic).Palette;
end;

procedure TIB_Image.SetAutoDisplay( Value: Boolean );
begin
  if FAutoDisplay <> Value then begin
    FAutoDisplay := Value;
    if Value then LoadPicture;
  end;
end;

procedure TIB_Image.SetBorderStyle( Value: TBorderStyle );
begin
  if FBorderStyle <> Value then begin
    FBorderStyle := Value;
    RecreateWnd;
  end;
end;

procedure TIB_Image.SetCenter(Value: Boolean);
begin
  if FCenter <> Value then begin
    FCenter := Value;
    Invalidate;
  end;
end;

procedure TIB_Image.SetPicture(Value: TPicture);
begin
  FPicture.Assign(Value);
end;

procedure TIB_Image.SetStretch(Value: Boolean);
begin
  if FStretch <> Value then begin
    FStretch := Value;
    Invalidate;
  end;
end;

procedure TIB_Image.PictureChanged(Sender: TObject);
begin
  if FIB_ColumnLink.Modify then begin
    FIB_ColumnLink.ControlIsModified := true;
    FPictureLoaded := true;
    FBlobIDLoaded.isc_quad_high := 0;
    FBlobIDLoaded.isc_quad_low  := 0;
    Invalidate;
  end else begin
    FIB_ColumnLink.ControlIsModified := false;
    FPictureLoaded := false;
    SysDataChanged;
  end;
end;

procedure TIB_Image.LoadPicture;
begin
  if ( not FPictureLoaded ) and
     ( FIB_ColumnLink.Column is TIB_Column_Blob ) then begin
    with FIB_ColumnLink.Column as TIB_Column_Blob do begin
      if not IsLoaded then begin
        Load;
      end;
      FPicture.OnChange := nil;
      try
        AssignTo( Picture.Bitmap );
        FIB_ColumnLink.ControlIsModified := false;
        FPictureLoaded := true;
        FBlobIDLoaded := Blob_ID;
        Invalidate;
      finally
        FPicture.OnChange := PictureChanged;
      end;
    end;
  end;
end;

{------------------------------------------------------------------------------}

procedure TIB_Image.DataChanged( Sender: TIB_DataLink;
                                 IB_DataSetLink: TIB_DataSetLink;
                                 IB_Column: TIB_Column );
begin
  SysDataChanged;
end;

procedure TIB_Image.UpdateData( Sender: TIB_DataLink;
                                IB_DataSetLink: TIB_DataSetLink;
                                IB_Column: TIB_Column );
begin
  if FIB_ColumnLink.Column is TIB_Column_Blob then begin
    with TIB_Column_Blob( FIB_ColumnLink.Column ) do begin
      if Picture.Graphic is TBitmap then begin
        Assign( Picture.Graphic );
      end else begin
        Clear;
      end;
    end;
  end;
end;

{------------------------------------------------------------------------------}

procedure TIB_Image.SysDataChanged;
begin
  if FIB_ColumnLink.Column is TIB_Column_Blob then begin
    with TIB_Column_Blob( FIB_ColumnLink.Column ) do begin
      if ( Blob_ID.isc_quad_high <> FBlobIDLoaded.isc_quad_high ) or
         ( Blob_ID.isc_quad_low  <> FBlobIDLoaded.isc_quad_low  ) then begin
        FPicture.OnChange := nil;
        try
          Picture.Graphic := nil;
          Invalidate;
        finally
          FPicture.OnChange := PictureChanged;
        end;
        FPictureLoaded := false;
        if FAutoDisplay then begin
          LoadPicture;
        end else begin
          Invalidate;
        end;
      end;
    end;
  end;
end;

{------------------------------------------------------------------------------}

procedure TIB_Image.CopyToClipboard;
begin
  if Picture.Graphic <> nil then begin
    Clipboard.Assign(Picture);
  end;
end;

procedure TIB_Image.CutToClipboard;
begin
  if Picture.Graphic <> nil then begin
    if FIB_ColumnLink.Modify then begin
      CopyToClipboard;
      Picture.Graphic := nil;
    end;
  end;
end;

procedure TIB_Image.PasteFromClipboard;
begin
  if ( Clipboard.HasFormat( CF_BITMAP )) and
     ( FIB_ColumnLink.Modify ) then begin
    Picture.Bitmap.Assign( Clipboard );
  end;
end;

procedure TIB_Image.CreateParams(var Params: TCreateParams);
begin
  inherited CreateParams(Params);
  if FBorderStyle = bsSingle then
    Params.Style := Params.Style or WS_BORDER;
end;

procedure TIB_Image.KeyDown(var Key: Word; Shift: TShiftState);
begin
  inherited KeyDown(Key, Shift);
  case Key of
    VK_INSERT:
      if ssShift in Shift then PasteFromClipBoard else
        if ssCtrl in Shift then CopyToClipBoard;
    VK_DELETE:
      if ssShift in Shift then begin
        CutToClipBoard;
      end else begin
        Picture.Graphic := nil;
      end;
  end;
end;

procedure TIB_Image.KeyPress(var Key: Char);
begin
  inherited KeyPress(Key);
  case Key of
    ^X: CutToClipBoard;
    ^C: CopyToClipBoard;
    ^V: PasteFromClipBoard;
    #13: LoadPicture;
    #27: SysDataChanged;
  end;
end;

procedure TIB_Image.CMEnter(var Message: TCMEnter);
begin
  Invalidate; { Draw the focus marker }
  FIB_ColumnLink.SetFocus;
  inherited;
end;

procedure TIB_Image.CMExit(var Message: TCMExit);
begin
  Invalidate; { Erase the focus marker }
  inherited;
end;

procedure TIB_Image.CMTextChanged(var Message: TMessage);
begin
  inherited;
  if not FPictureLoaded then Invalidate;
end;

procedure TIB_Image.WMLButtonDown(var Message: TWMLButtonDown);
begin
  if TabStop and CanFocus then SetFocus;
  inherited;
end;

procedure TIB_Image.WMLButtonDblClk(var Message: TWMLButtonDblClk);
begin
  LoadPicture;
  inherited;
end;

procedure TIB_Image.WMCut(var Message: TMessage);
begin
  CutToClipboard;
end;

procedure TIB_Image.WMCopy(var Message: TMessage);
begin
  CopyToClipboard;
end;

procedure TIB_Image.WMPaste(var Message: TMessage);
begin
  PasteFromClipboard;
end;

procedure TIB_Image.Paint;
var
  W, H: Integer;
  R: TRect;
  S: string;
  DrawPict: TPicture;
begin
  with Canvas do begin
    Brush.Style := bsSolid;
    Brush.Color := Color;
    if FPictureLoaded or (csPaintCopy in ControlState) then begin
      DrawPict := TPicture.Create;
      H := 0;
      try
        if (csPaintCopy in ControlState) and
           Assigned(FIB_ColumnLink.Column) and
           (FIB_ColumnLink.Column is TIB_Column_Blob) then
        begin

//!!!          DrawPict.Assign(FIB_ColumnLink.Column);

          if DrawPict.Graphic is TBitmap then
            DrawPict.Bitmap.IgnorePalette := QuickDraw; //!!!
        end
        else
        begin
          DrawPict.Assign(Picture);
          if Focused and (DrawPict.Graphic is TBitmap) and
            (DrawPict.Bitmap.Palette <> 0) then
          begin { Control has focus, so realize the bitmap palette in foreground }
            H := SelectPalette(Handle, DrawPict.Bitmap.Palette, False);
            RealizePalette(Handle);
          end;
        end;
        if Stretch then
          if (DrawPict.Graphic = nil) or DrawPict.Graphic.Empty then
            FillRect(ClientRect)
          else
            StretchDraw(ClientRect, DrawPict.Graphic)
        else
        begin
          SetRect(R, 0, 0, DrawPict.Width, DrawPict.Height);
          if Center then OffsetRect(R, (ClientWidth - DrawPict.Width) div 2,
            (ClientHeight - DrawPict.Height) div 2);
          StretchDraw(R, DrawPict.Graphic);
          ExcludeClipRect(Handle, R.Left, R.Top, R.Right, R.Bottom);
          FillRect(ClientRect);
          SelectClipRgn(Handle, 0);
        end;
      finally
        if H <> 0 then SelectPalette(Handle, H, True);
        DrawPict.Free;
      end;
    end
    else begin
      Font := Self.Font;
      if FIB_ColumnLink.Column <> nil then
        S := FIB_ColumnLink.Column.DisplayName
      else S := Name;
      S := '(' + S + ')';
      W := TextWidth(S);
      H := TextHeight(S);
      R := ClientRect;
      TextRect(R, (R.Right - W) div 2, (R.Bottom - H) div 2, S);
    end;
    if (GetParentForm(Self).ActiveControl = Self) and
      not (csDesigning in ComponentState) and
      not (csPaintCopy in ControlState) then
    begin
      Brush.Color := clWindowFrame;
      FrameRect(ClientRect);
    end;
  end;
end;


