// INT_IBC_Edit

constructor TIB_Edit.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  inherited ReadOnly := True;
  ControlStyle := ControlStyle + [csReplicatable];
  FIB_ColumnLink := TIB_ColumnLink.Create( Self );
  with FIB_ColumnLink do begin
    Control := Self;
    OnStateChanged := StateChanged;
    OnDataChanged  := DataChanged;
    OnUpdateData   := UpdateData;
  end;
end;

destructor TIB_Edit.Destroy;
begin
  DataSetLink := nil;
  with FIB_ColumnLink do begin
    Control := nil;
    OnStateChanged := nil;
    OnDataChanged  := nil;
    OnUpdateData   := nil;
  end;
  Destroying;
  if FCanvas <> nil then begin
    FCanvas.Free;
    FCanvas := nil;
  end;
  inherited Destroy;
end;

procedure TIB_Edit.Loaded;
begin
  inherited Loaded;
  SysStateChanged;
  SysDataChanged;
end;

{------------------------------------------------------------------------------}
{  DataLink event handlers.                                                    } 
{------------------------------------------------------------------------------}

procedure TIB_Edit.StateChanged( Sender: TIB_DataLink;
                                 IB_DataSetLink: TIB_DataSetLink );
begin
  SysStateChanged;
end;

procedure TIB_Edit.DataChanged( Sender: TIB_DataLink;
                                IB_DataSetLink: TIB_DataSetLink;
                                IB_Column: TIB_Column );
begin
  SysDataChanged;
end;

procedure TIB_Edit.UpdateData( Sender: TIB_DataLink;
                               IB_DataSetLink: TIB_DataSetLink;
                               IB_Column: TIB_Column );
begin
  SysUpdateData;
end;

{------------------------------------------------------------------------------}
{ DataLink property coordination methods.                                      } 
{------------------------------------------------------------------------------}

function TIB_Edit.GetOnPrepareSQL: TIB_DataLinkEvent;
begin
  Result := FIB_ColumnLink.OnPrepareSQL;
end;

procedure TIB_Edit.SetOnPrepareSQL( AValue: TIB_DataLinkEvent);
begin
  FIB_ColumnLink.OnPrepareSQL := AValue;
end;

{------------------------------------------------------------------------------}

function TIB_Edit.GetDataSetLink: TIB_DataSetLink;
begin
  Result := FIB_ColumnLink.IB_DataSetLink;
end;

procedure TIB_Edit.SetDataSetLink(AValue: TIB_DataSetLink);
begin
  FIB_ColumnLink.IB_DataSetLink := AValue;
end;

{------------------------------------------------------------------------------}

function TIB_Edit.GetDataColumnName: string;
begin
  Result := FIB_ColumnLink.ColumnName;
end;

procedure TIB_Edit.SetDataColumnName(const AValue: string);
begin
  FIB_ColumnLink.ColumnName := AValue;
end;

{------------------------------------------------------------------------------}

function TIB_Edit.GetColumn: TIB_Column;
begin
  Result := FIB_ColumnLink.Column;
end;

{------------------------------------------------------------------------------}

function TIB_Edit.GetReadOnly: Boolean;
begin
  Result := FIB_ColumnLink.ControlIsReadOnly;
end;

procedure TIB_Edit.SetReadOnly( Value: Boolean );
begin
  if ReadOnly <> Value then begin
    FIB_ColumnLink.ControlIsReadOnly := Value;
    SysStateChanged;
  end;
end;

{------------------------------------------------------------------------------}

function TIB_Edit.GetPreventUpdating: Boolean;
begin
  Result := FIB_ColumnLink.ControlPreventsUpdating;
end;

procedure TIB_Edit.SetPreventUpdating( Value: Boolean );
begin
  if PreventUpdating <> Value then begin
    FIB_ColumnLink.ControlPreventsUpdating := Value;
    SysStateChanged;
  end;
end;

{------------------------------------------------------------------------------}

function TIB_Edit.GetPreventInserting: Boolean;
begin
  Result := FIB_ColumnLink.ControlPreventsInserting;
end;

procedure TIB_Edit.SetPreventInserting( Value: Boolean );
begin
  if PreventInserting <> Value then begin
    FIB_ColumnLink.ControlPreventsInserting := Value;
    SysStateChanged;
  end;
end;

{------------------------------------------------------------------------------}

function TIB_Edit.GetPreventSearching: Boolean;
begin
  Result := FIB_ColumnLink.ControlPreventsSearching;
end;

procedure TIB_Edit.SetPreventSearching( Value: Boolean );
begin
  if PreventSearching <> Value then begin
    FIB_ColumnLink.ControlPreventsSearching := Value;
    SysStateChanged;
  end;
end;

{------------------------------------------------------------------------------}

function TIB_Edit.GetSearchBuffer: string;
begin
  if FIB_ColumnLink <> nil then begin
    Result := FIB_ColumnLink.SearchBuffer;
  end else begin
    Result := '';
  end;
end;

procedure TIB_Edit.SetSearchBuffer( AValue: string );
begin
  if FIB_ColumnLink <> nil then begin
    FIB_ColumnLink.SearchBuffer := AValue;
  end;
end;

{------------------------------------------------------------------------------}
{ Implementation of data-aware logic.                                          } 
{------------------------------------------------------------------------------}

procedure TIB_Edit.SysDataChanged;
begin
  if FIB_ColumnLink <> nil then 
  with FIB_ColumnLink do begin
    FAlignment := ColumnAlignment;
    EditMask   := ColumnEditMask;
    MaxLength  := ColumnMaxLength;
    case CharCase of
      ecLowerCase: if Text <> LowerCase( ColumnText ) then Text := ColumnText;
      ecNormal:    if Text <>            ColumnText   then Text := ColumnText;
      ecUpperCase: if Text <> UpperCase( ColumnText ) then Text := ColumnText;
    end;
  end;
end;

procedure TIB_Edit.SysStateChanged;
begin
  with FIB_ColumnLink do begin
    inherited Color    := Color;
    inherited ReadOnly := ReadOnly or not ( DataSetState in [ dsInserting,
                                                              dsUpdating,
                                                              dsSearching ]);
  end;
end;

procedure TIB_Edit.SysUpdateData;
begin
  if FIB_ColumnLink.Column <> nil then begin
    ValidateEdit;
    FIB_ColumnLink.ColumnText := Text;
  end;
end;

{------------------------------------------------------------------------------}
{  Control specific processing                                                 } 
{------------------------------------------------------------------------------}

function TIB_Edit.EditCanModify: Boolean;
begin
  Result := FIB_ColumnLink.CanModify;
end;

procedure TIB_Edit.Reset;
begin
  SysDataChanged;
  SelectAll;
end;

procedure TIB_Edit.SetFocused( Value: Boolean );
begin
  if FFocused <> Value then begin
    FFocused := Value;
    if (FAlignment <> taLeftJustify) and not IsMasked then begin
      Invalidate;
    end;
  end;
end;

{------------------------------------------------------------------------------}

procedure TIB_Edit.Change;
begin
  with FIB_ColumnLink do begin
    if DataSetState in [ dsInserting, dsUpdating, dsSearching ] then begin
      ControlIsModified := true;
    end;
  end;
  inherited Change;
end;

procedure TIB_Edit.KeyDown(var Key: Word; Shift: TShiftState);
begin
  inherited KeyDown(Key, Shift);
  if (Key = VK_DELETE) or ((Key = VK_INSERT) and (ssShift in Shift)) then begin
    FIB_ColumnLink.Modify;
  end;
end;

procedure TIB_Edit.KeyPress(var Key: Char);
begin
  inherited KeyPress(Key);
  if (Key in [#32..#255]) and
     (FIB_ColumnLink.Column <> nil) and
    not FIB_ColumnLink.Column.IsValidChar(Key) then
  begin
    MessageBeep(0);
    Key := #0;
  end;
  case Key of
    ^H, ^V, ^X, #32..#255: FIB_ColumnLink.Modify;
    #27:
    begin
      Reset;
      Key := #0;
    end;
  end;
end;

procedure TIB_Edit.WMPaste(var Message: TMessage);
begin
  FIB_ColumnLink.Modify;
  inherited;
end;

procedure TIB_Edit.WMCut(var Message: TMessage);
begin
  FIB_ColumnLink.Modify;
  inherited;
end;

procedure TIB_Edit.CMEnter(var Message: TCMEnter);
begin
  SetFocused( true );
  FIB_ColumnLink.SetFocus;
  inherited;
end;

procedure TIB_Edit.CMExit(var Message: TCMExit);
begin
  try
    FIB_ColumnLink.UpdateValue;
  except
    SelectAll;
    SetFocus;
    raise;
  end;
  SetFocused( false );
  CheckCursor;
  DoExit;
end;

{------------------------------------------------------------------------------}

procedure TIB_Edit.WMPaint(var Message: TWMPaint);
var
  Left: Integer;
  Margins: TPoint;
  R: TRect;
  DC: HDC;
  PS: TPaintStruct;
  S: string;
begin
  if ((FAlignment = taLeftJustify) or FFocused) and
    not (csPaintCopy in ControlState) then begin
    inherited;
    Exit;
  end;
  if FCanvas = nil then begin
    FCanvas := TControlCanvas.Create;
    FCanvas.Control := Self;
  end;
  DC := Message.DC;
  if DC = 0 then DC := BeginPaint(Handle, PS);
  FCanvas.Handle := DC;
  try
    FCanvas.Font := Font;
    with FCanvas do
    begin
      R := ClientRect;
      if not (NewStyleControls and Ctl3D) and (BorderStyle = bsSingle) then
      begin
        Brush.Color := clWindowFrame;
        FrameRect(R);
        InflateRect(R, -1, -1);
      end;
      Brush.Color := Color;
      if (csPaintCopy in ControlState) and (FIB_ColumnLink.Column <> nil) then
      begin
        S := FIB_ColumnLink.Column.AsString;
        case CharCase of
          ecUpperCase: S := AnsiUpperCase(S);
          ecLowerCase: S := AnsiLowerCase(S);
        end;
      end else begin
        S := EditText;
      end;
      if PasswordChar <> #0 then FillChar(S[1], Length(S), PasswordChar);
      Margins := GetTextMargins;
      case FAlignment of
        taLeftJustify: Left := Margins.X;
        taRightJustify: Left := ClientWidth - TextWidth(S) - Margins.X - 1;
      else
        Left := (ClientWidth - TextWidth(S)) div 2;
      end;
      TextRect(R, Left, Margins.Y, S);
    end;
  finally
    FCanvas.Handle := 0;
    if Message.DC = 0 then EndPaint(Handle, PS);
  end;
end;

function TIB_Edit.GetTextMargins: TPoint;
var
  DC: HDC;
  SaveFont: HFont;
  I: Integer;
  SysMetrics, Metrics: TTextMetric;
begin
  if NewStyleControls then begin
    if BorderStyle = bsNone then begin
      I := 0
    end else begin
      if Ctl3D then begin
        I := 1
      end else begin
        I := 2;
      end;
    end;
    Result.X := SendMessage(Handle, EM_GETMARGINS, 0, 0) and $0000FFFF + I;
    Result.Y := I;
  end else begin
    if BorderStyle = bsNone then begin
      I := 0;
    end else begin
      DC := GetDC(0);
      GetTextMetrics(DC, SysMetrics);
      SaveFont := SelectObject(DC, Font.Handle);
      GetTextMetrics(DC, Metrics);
      SelectObject(DC, SaveFont);
      ReleaseDC(0, DC);
      I := SysMetrics.tmHeight;
      if I > Metrics.tmHeight then begin
        I := Metrics.tmHeight;
      end;
      I := I div 4;
    end;
    Result.X := I;
    Result.Y := I;
  end;
end;

