// INT_IBC_ComboBox

constructor TIB_ComboBox.Create(AOwner: TComponent);
begin
  inherited Create( AOwner );
  ControlStyle := ControlStyle + [csReplicatable];
  FIB_ColumnLink := TIB_ColumnLink.Create( Self );
  with FIB_ColumnLink do begin
    Control := Self;
    OnStateChanged := StateChanged;
    OnDataChanged  := DataChanged;
    OnUpdateData   := UpdateData;
  end;
  FPaintControl := TPaintControl.Create( Self, 'COMBOBOX' );
end;

destructor TIB_ComboBox.Destroy;
begin
  DataSetLink := nil;
  with FIB_ColumnLink do begin
    Control := nil;
    OnStateChanged := nil;
    OnDataChanged  := nil;
    OnUpdateData   := nil;
  end;
  Destroying;
  FPaintControl.Free;
  FPaintControl := nil;
  inherited Destroy;
end;

procedure TIB_ComboBox.Loaded;
begin
  inherited Loaded;
  SysStateChanged;
  SysDataChanged;
end;

{------------------------------------------------------------------------------}
{  DataLink event handlers.                                                    } 
{------------------------------------------------------------------------------}

procedure TIB_ComboBox.StateChanged( Sender: TIB_DataLink;
                                     IB_DataSetLink: TIB_DataSetLink );
begin
  SysStateChanged;
end;

procedure TIB_ComboBox.DataChanged( Sender: TIB_DataLink;
                                    IB_DataSetLink: TIB_DataSetLink;
                                    IB_Column: TIB_Column );
begin
  SysDataChanged;
end;

procedure TIB_ComboBox.UpdateData( Sender: TIB_DataLink;
                                   IB_DataSetLink: TIB_DataSetLink;
                                   IB_Column: TIB_Column );
begin
  SysUpdateData;
end;

{------------------------------------------------------------------------------}
{ DataLink property coordination methods.                                      } 
{------------------------------------------------------------------------------}

function TIB_ComboBox.GetOnPrepareSQL: TIB_DataLinkEvent;
begin
  Result := FIB_ColumnLink.OnPrepareSQL;
end;

procedure TIB_ComboBox.SetOnPrepareSQL( AValue: TIB_DataLinkEvent);
begin
  FIB_ColumnLink.OnPrepareSQL := AValue;
end;

{------------------------------------------------------------------------------}

function TIB_ComboBox.GetDataSetLink: TIB_DataSetLink;
begin
  Result := FIB_ColumnLink.IB_DataSetLink;
end;

procedure TIB_ComboBox.SetDataSetLink(AValue: TIB_DataSetLink);
begin
  FIB_ColumnLink.IB_DataSetLink := AValue;
end;

{------------------------------------------------------------------------------}

function TIB_ComboBox.GetDataColumnName: string;
begin
  Result := FIB_ColumnLink.ColumnName;
end;

procedure TIB_ComboBox.SetDataColumnName(const AValue: string);
begin
  FIB_ColumnLink.ColumnName := AValue;
end;

{------------------------------------------------------------------------------}

function TIB_ComboBox.GetColumn: TIB_Column;
begin
  Result := FIB_ColumnLink.Column;
end;

{------------------------------------------------------------------------------}

function TIB_ComboBox.GetReadOnly: Boolean;
begin
  Result := FIB_ColumnLink.ControlIsReadOnly;
end;

procedure TIB_ComboBox.SetReadOnly( Value: Boolean );
begin
  if ReadOnly <> Value then begin
    FIB_ColumnLink.ControlIsReadOnly := Value;
    SysStateChanged;
  end;
end;

{------------------------------------------------------------------------------}

function TIB_ComboBox.GetPreventUpdating: Boolean;
begin
  Result := FIB_ColumnLink.ControlPreventsUpdating;
end;

procedure TIB_ComboBox.SetPreventUpdating( Value: Boolean );
begin
  if PreventUpdating <> Value then begin
    FIB_ColumnLink.ControlPreventsUpdating := Value;
    SysStateChanged;
  end;
end;

{------------------------------------------------------------------------------}

function TIB_ComboBox.GetPreventInserting: Boolean;
begin
  Result := FIB_ColumnLink.ControlPreventsInserting;
end;

procedure TIB_ComboBox.SetPreventInserting( Value: Boolean );
begin
  if PreventInserting <> Value then begin
    FIB_ColumnLink.ControlPreventsInserting := Value;
    SysStateChanged;
  end;
end;

{------------------------------------------------------------------------------}

function TIB_ComboBox.GetPreventSearching: Boolean;
begin
  Result := FIB_ColumnLink.ControlPreventsSearching;
end;

procedure TIB_ComboBox.SetPreventSearching( Value: Boolean );
begin
  if PreventSearching <> Value then begin
    FIB_ColumnLink.ControlPreventsSearching := Value;
    SysStateChanged;
  end;
end;

{------------------------------------------------------------------------------}

function TIB_ComboBox.GetSearchBuffer: string;
begin
  if FIB_ColumnLink <> nil then begin
    Result := FIB_ColumnLink.SearchBuffer;
  end else begin
    Result := '';
  end;
end;

procedure TIB_ComboBox.SetSearchBuffer( AValue: string );
begin
  if FIB_ColumnLink <> nil then begin
    FIB_ColumnLink.SearchBuffer := AValue;
  end;
end;

{------------------------------------------------------------------------------}
{ Implementation of data-aware logic.                                          } 
{------------------------------------------------------------------------------}

procedure TIB_ComboBox.SysDataChanged;
begin
  with FIB_ColumnLink do begin
    MaxLength := ColumnMaxLength;
    SetComboText( ColumnText );
  end;
end;

procedure TIB_ComboBox.SysStateChanged;
begin
  SetEditReadOnly;
  inherited Color := FIB_ColumnLink.Color;
end;

procedure TIB_ComboBox.SysUpdateData;
begin
  FIB_ColumnLink.ColumnText := Text;
end;

{------------------------------------------------------------------------------}

procedure TIB_ComboBox.SetComboText( const Value: string );
var
  I: Integer;
  Redraw: Boolean;
begin
  if Value <> GetComboText then begin
    if Style <> csDropDown then begin
      Redraw := ( Style <> csSimple ) and HandleAllocated;
      if Redraw then begin
        SendMessage( Handle, WM_SETREDRAW, 0, 0 );
      end;
      try
        if Value = '' then begin
          I := -1
        end else begin
          I := Items.IndexOf( Value );
        end;
        ItemIndex := I;
      finally
        if Redraw then begin
          SendMessage( Handle, WM_SETREDRAW, 1, 0 );
          Invalidate;
        end;
      end;
      if I >= 0 then begin
        Exit;
      end;
    end;
    if Style in [csDropDown, csSimple] then begin
      Text := Value;
    end;
  end;
end;

function TIB_ComboBox.GetComboText: string;
var
  I: Integer;
begin
  if Style in [csDropDown, csSimple] then Result := Text else begin
    I := ItemIndex;
    if I < 0 then Result := '' else Result := Items[I];
  end;
end;

procedure TIB_ComboBox.Change;
begin
  if FIB_ColumnLink.Modify then begin
    inherited Change;
    FIB_ColumnLink.ControlIsModified := true;
  end;
end;

procedure TIB_ComboBox.Click;
begin
  if FIB_ColumnLink.Modify then begin
    inherited Click;
  end;
end;

{------------------------------------------------------------------------------}

procedure TIB_ComboBox.CreateWnd;
begin
  inherited CreateWnd;
  SetEditReadOnly;
end;

{------------------------------------------------------------------------------}

procedure TIB_ComboBox.KeyDown(var Key: Word; Shift: TShiftState);
begin
  inherited KeyDown(Key, Shift);
  if Key in [VK_BACK, VK_DELETE, VK_UP, VK_DOWN, 32..255] then begin
    if not FIB_ColumnLink.Modify and (Key in [VK_UP, VK_DOWN]) then Key := 0;
  end;
end;

procedure TIB_ComboBox.KeyPress(var Key: Char);
begin
  inherited KeyPress(Key);
  if (Key in [#32..#255]) and (FIB_ColumnLink.Column <> nil) and
    not FIB_ColumnLink.Column.IsValidChar(Key) then
  begin
    MessageBeep(0);
    Key := #0;
  end;
  case Key of
    ^H, ^V, ^X, #32..#255:
      FIB_ColumnLink.Modify;
    #27:
      begin
        SysDataChanged;
        SelectAll;
        Key := #0;
      end;
  end;
end;

procedure TIB_ComboBox.SetEditReadOnly;
begin
  if (Style in [csDropDown, csSimple]) and HandleAllocated then
    SendMessage(EditHandle, EM_SETREADONLY,Ord(not FIB_ColumnLink.Modifying),0);
end;

procedure TIB_ComboBox.WndProc(var Message: TMessage);
begin
  if not (csDesigning in ComponentState) then
    case Message.Msg of
      WM_COMMAND:
        if TWMCommand(Message).NotifyCode = CBN_SELCHANGE then
          if not FIB_ColumnLink.Modify then begin
            if Style <> csSimple then
              PostMessage(Handle, CB_SHOWDROPDOWN, 0, 0);
            Exit;
          end;
      CB_SHOWDROPDOWN: begin
        if Message.WParam <> 0 then begin
          FIB_ColumnLink.Modify
        end else begin
          if not FIB_ColumnLink.Modifying then begin
            SysDataChanged; {Restore text}
          end;
        end;
      end;
      WM_CREATE,
      WM_WINDOWPOSCHANGED,
      CM_FONTCHANGED:
        FPaintControl.DestroyHandle;
    end;
  inherited WndProc(Message);
end;

procedure TIB_ComboBox.ComboWndProc(var Message: TMessage; ComboWnd: HWnd;
  ComboProc: Pointer);
begin
  if not (csDesigning in ComponentState) then
    case Message.Msg of
      WM_LBUTTONDOWN:
        if (Style = csSimple) and (ComboWnd <> EditHandle) then
          if not FIB_ColumnLink.Modify then Exit;
    end;
  inherited ComboWndProc(Message, ComboWnd, ComboProc);
end;

procedure TIB_ComboBox.CMEnter(var Message: TCMEnter);
begin
  FIB_ColumnLink.SetFocus;
  inherited;
end;

procedure TIB_ComboBox.CMExit(var Message: TCMExit);
begin
  try
    FIB_ColumnLink.UpdateValue;
  except
    SelectAll;
    SetFocus;
    raise;
  end;
  inherited;
end;

procedure TIB_ComboBox.WMPaint(var Message: TWMPaint);
var
  S: string;
  R: TRect;
  P: TPoint;
  Child: HWND;
begin
  if csPaintCopy in ControlState then begin
    if FIB_ColumnLink.Column <> nil then begin
      S := FIB_ColumnLink.ColumnText
    end else begin
      S := '';
    end;
    if Style = csDropDown then begin
      SendMessage(FPaintControl.Handle, WM_SETTEXT, 0, Longint(PChar(S)));
      SendMessage(FPaintControl.Handle, WM_PAINT, Message.DC, 0);
      Child := GetWindow(FPaintControl.Handle, GW_CHILD);
      if Child <> 0 then begin
        Windows.GetClientRect(Child, R);
        Windows.MapWindowPoints(Child, FPaintControl.Handle, R.TopLeft, 2);
        GetWindowOrgEx(Message.DC, P);
        SetWindowOrgEx(Message.DC, P.X - R.Left, P.Y - R.Top, nil);
        IntersectClipRect(Message.DC, 0, 0, R.Right - R.Left, R.Bottom - R.Top);
        SendMessage(Child, WM_PAINT, Message.DC, 0);
      end;
    end else begin
      SendMessage(FPaintControl.Handle, CB_RESETCONTENT, 0, 0);
      if Items.IndexOf(S) <> -1 then begin
        SendMessage(FPaintControl.Handle, CB_ADDSTRING, 0, Longint(PChar(S)));
        SendMessage(FPaintControl.Handle, CB_SETCURSEL, 0, 0);
      end;
      SendMessage(FPaintControl.Handle, WM_PAINT, Message.DC, 0);
    end;
  end else
    inherited;
end;

procedure TIB_ComboBox.SetItems(Value: TStrings);
begin
  Items.Assign(Value);
  SysDataChanged;
end;

procedure TIB_Combobox.SetStyle(Value: TComboboxStyle);
begin
//  if (Value = csSimple) and Assigned(FIB_ColumnLink) and
//      FIB_ColumnLink.DataSetLinkFixed then
//    DBError(SNotReplicatable);
  inherited SetStyle(Value);
end;

procedure TIB_Combobox.CMGetDatalink(var Message: TMessage);
begin
  Message.Result := Integer(FIB_ColumnLink);
end;


